/*******************************************************************
 *
 *  ttfile.h                                                     1.2
 *
 *    File I/O Component (specification).
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute 
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  Changes between 1.2 and 1.1 :
 *
 *  - added macros to support re-entrant builds
 *
 *  - added the TT_Duplicate_File function to duplicate streams
 *    ( re-entrant builds only )
 *
 ******************************************************************/

/* #define TT_CONFIG_REENTRANT */

#ifndef TTFILE_H
#define TTFILE_H

#ifdef __cplusplus
extern "C" {
#endif

  void TTFile_Init();
  /* Init file component */

  void TTFile_Done();
  /* Done with file component */

  /**********************************************************************/
  /*                                                                    */
  /*  File operations.                                                  */
  /*                                                                    */
  /**********************************************************************/

#ifdef TT_CONFIG_REENTRANT
#define FILE_OPS    TT_Stream  stream,
#define FILE_OP     TT_Stream  stream
#else
#define FILE_OPS   /* void */
#define FILE_OP
#endif /* REENTRANT */

  /* The macro FILE_OPS lets us build a thread-safe or re-entrant  */
  /* implementation depending on a single configuration define     */

#ifdef TT_CONFIG_REENTRANT
#define ERROR_OPS  int* error,
#else
#define ERROR_OPS  /* void */
#endif /* REENTRANT */

  /* The macro ERROR_OPS is useful to specify the address of an error */
  /* variable in the case of a re-entrant build. The error var is     */
  /* associated to a stream by a TT_Open_File                         */

  Bool  TT_Open_File( ERROR_OPS char* filepathname, TT_Stream*  stream );
  /* open a new font file in stream 'stream' */

#ifdef TT_CONFIG_REENTRANT

  Bool  TT_Duplicate_File( TT_Stream  input_stream,
                           TT_Stream* copy,
                           int*       error );
  /* duplicate an open stream for a new instance. Useful to let several */
  /* threads accessing the same font file concurrently                  */

#endif /* REENTRANT */

  Bool  TT_Close_File( TT_Stream  stream );
  /* close the opened file in 'stream' */

#ifndef TT_CONFIG_REENTRANT        /* thread-safe implementation only */

  Bool  TT_Lock_Access( TT_Stream stream );
  /* lock access on stream 'stream' until next 'Release_Access' */

  Bool  TT_Release_Access( TT_Stream  stream );
  /* unlock access on stream 'stream' */

#endif /* !REENTRANT */

  Bool  TT_Seek_File( FILE_OPS long position );
  /* Seek file cursor to a given position */

  Bool  TT_Skip_File( FILE_OPS long distance );
  /* Skip the next 'distance' bytes in file */

  Bool  TT_Read_File( FILE_OPS void* buffer, long count );
  /* Read 'count' bytes from file into 'buffer' */

  Bool  TT_Read_At_File( FILE_OPS long position, void* buffer, long count );
  /* Read the 'count' bytes at 'position' into 'buffer' */

  Long  TT_File_Pos( FILE_OP );
  /* Return current file position */

  /**********************************************************************/
  /*                                                                    */
  /*  Frame operations.                                                 */
  /*                                                                    */
  /*  For a comprehensive explanation of frames, please refer to the    */
  /*  documentation file 'macros.txt'                                   */
  /*                                                                    */
  /**********************************************************************/

  /* Frame type declaration.*/

  typedef struct _TFileFrame
  {
    unsigned char*  address;  /* frame buffer                     */
    unsigned char*  cursor;   /* current cursor position in frame */
    int             size;     /* frame size                       */
  } TFileFrame;

#define CONSTRUCT_FRAME(_frame_)  TFileFrame _frame_ = { NULL, NULL, 0 }

  /* The macro CONSTRUCT_FRAME is used to define and init a frame  */
  /* It is important to have an default frame of { NULL, NULL, 0 } */
  /* before a call to TT_Access_Frame. Otherwise, the call will    */
  /* fail with a TT_Err_Nested_Frame_Accesses error                */

#ifdef TT_CONFIG_REENTRANT

#define FRAME_OPS   TFileFrame* frame,
#define FRAME_OP    TFileFrame* frame
#else
#define FRAME_OPS  /* void */
#define FRAME_OP   /* void */
#endif /* REENTRANT */

  /* The macro FRAME_OPS lets us build a thread-safe or re-entrant */
  /* implementation depending on a single configuration define     */

  Bool  TT_Access_Frame( FILE_OPS FRAME_OPS int size );
  /* Access the next 'size' bytes from current position */
  /* Fails if all bytes cannot be read/accessed         */

  Bool  TT_Check_And_Access_Frame( FILE_OPS FRAME_OPS int size );
  /* Access the bytes located in the next 'size' bytes of the file */
  /* Doesn't fail if less than 'size' bytes are accessible ( like  */
  /* at the end of the file )                                      */

  Bool  TT_Forget_Frame( FRAME_OP );
  /* Forget frame */

  unsigned char   Get_Byte  ( FRAME_OP );
  char            Get_Char  ( FRAME_OP );
  short           Get_Short ( FRAME_OP );
  unsigned short  Get_UShort( FRAME_OP );
  long            Get_Long  ( FRAME_OP );
  unsigned long   Get_ULong ( FRAME_OP );

#ifdef __cplusplus
}
#endif

#endif /* TTFILE_H */


/* End */
