/**
 * Driver.c - A sample device driver using the Drv16 kit.
 *
 * This file is part of the Drv16 Device Driver Development Kit.
 *
 * Copyright (c) 2013-2015 David Azarewicz david@88watts.net
 *
 * The following source code is provided to you solely for the purpose of
 * assisting you in developing your own OS/2 device drivers. You may use
 * this software in your device drivers free of charge. This Copyright
 * statement may not be removed.
 *
 * Drv16 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
#include "Dev16lib.h"

/* This is here to resolve the forward reference */
void StrategyInit(PREQPACKET prp);

/* You must have a StrategyHandler routine and it must be defined this
 * way. The kernel calls this routine whenever it wants to access the
 * driver
 *
 * An OS/2 driver should have the folowing strategy commands defined
 * as shown below, even if they do nothing.
 */
#pragma aux StrategyHandler far parm [es bx];
void StrategyHandler(PREQPACKET prp)
{
  /* This is the default status returned unless changed by a function
   * below. You must always at least have the RPDONE bit set upon
   * return even if you also have other bits set as well.
   */
  prp->usStatus = RPDONE;

  switch (prp->bCommand) {

  case STRATEGY_INIT:
    /* The STRATEGY_INIT command is issued by the kernel immediately
     * after your driver is loaded. This is so that it can initialize
     * itself and report back to the kernel if it can run or not. There
     * are some minimum things you must do in the init routine. See the
     * StrategyInit() function below. The INIT command is called at
     * ring 3. All other commands are called at ring 0.
     */
    StrategyInit(prp);
    break;

  case STRATEGY_OPEN:
    /* You only need to implement the OPEN command if you do file
     * open/close functions.
     */
    break;

  case STRATEGY_CLOSE:
    /* You only need to implement the CLOSE command if you do file
     * open/close functions.
     */
    break;

  case STRATEGY_GENIOCTL:
    /* You only need to implement the IOCTL command if you want to
     * accept IOCTL calls from applications.
     */
    prp->usStatus = RPDONE | RPERR | RPERR_BADCOMMAND;
    break;

  case STRATEGY_DEINSTALL:
    /* This command is not needed. */
    break;

  case STRATEGY_INITCOMPLETE:
    /* The INITCOMPLETE command is called after all of the drivers in
     * the system have been loaded. This is to allow your driver to
     * complete any initialization it needs to. If you need to do IDC
     * calls to other drivers, you can be sure all driver are present
     * at this time.
     *
     * You must call Drv16InitComplete() before anything else.
     */
    Drv16InitComplete();
    break;

  case STRATEGY_SAVERESTORE:
    /* you only need SAVERESTORE if you support suspend / resume. */
    break;

  default:
    /* This returns a failure status for all other commands */
    prp->usStatus = RPDONE | RPERR | RPERR_GENERAL;
  }
}

/* You can put certain parts of your code into the inittext and initdata
 * segments and then tell the kernel to discard them after your
 * initialization is done. Everything below these statements will go
 * into those segments.
 */
#pragma code_seg ("_inittext");
#pragma data_seg ("_initdata","endds");

/* This is the name of your driver. It is here so that we can use it
 * multiple times for displaying messages, etc.
 */
char cDevName[9] = "MYDRV$";

void StrategyInit(PREQPACKET prp)
{
  short Success;

  /* You must call this function to put your driver name into the device
   * driver header. The UtSetDriverName() function ensures that the driver
   * name complies with all the requirements (max 8 chars, blank padded).
   * UtSetDriverName() can be called before Drv16Init().
   */
  UtSetDriverName(cDevName);

  Success = 1;

  /* You must call the Drv16Init() function first thing in the init routine.
   * This allows all the Drv16 modules to initialize before you use them.
   */
  if (Drv16Init(prp)) Success = 0;

  if (Success)
  {
    /* You must return a status */
    prp->usStatus = RPDONE;

    /* You must tell the kernel where the end of your code and data are.
     * Everything past these locations will be discarded by the kernel
     * when you return from the init routine.
     */
    prp->init_out.usCodeEnd = (USHORT) _TextEnd;     // set end of code
    prp->init_out.usDataEnd = (USHORT) &_DataEnd;    //  and data pointers

    /* You can optionally output a message to the screen */
    cprintf("Driver %s loaded successfully.\n", cDevName);
  }
  else
  {
    /* You must return a status of at least RPDONE. You can also or in
     * an error code if you want.
     */
    prp->usStatus = RPDONE | RPERR;

    /* If you set the end locations to zero, the kernel will assume you
     * failed to innitialize.
     */
    prp->init_out.usCodeEnd = 0;     // set end of code
    prp->init_out.usDataEnd = 0;    //  and data pointers
    cprintf("Driver initialization failed. %s not loaded.\n", cDevName);
  }
}
