/*
 *      HTCMP.CMD - HyperText/2 source compare - V1.07 - C.Langanke 2001-2004
 *
 *    Syntax: htcmp sourcemask target dir [outdir]
 *
 *    This program compares specified HyperText/2 source files with
 *    files of the same name in a target directory and and creates
 *    a directory structure in an output directory, where for each
 *    section not being equal between a source and target file,
 *    difference files are created.
 *
 *    Note:
 *    - when comparing, include commands are not processed, instead
 *      each include file needs to be comared as well separately.
 *    - in order to get reasonable results, it is highly recommended
 *      to specify anchor names for all sections. If a section does
 *      not contain an anchor name, a name is generated in form of
 *      #0, #1 and so forth.
 *    - If two files to be compared do not contain anchor names, they
 *      still can be compared, if the order of the sections has not
 *      changed and no sections have been added other than at the end
 *      of the new file.
 */
/* The first comment is used as online help text */

 SIGNAL ON HALT NAME HALT

 TitleLine = STRIP(SUBSTR(SourceLine(2), 3));
 PARSE VAR TitleLine CmdName'.CMD 'Info
 Title = CmdName Info
 env       = 'OS2ENVIRONMENT';
 TRUE      = (1 = 1);
 FALSE     = (0 = 1);
 CrLf      = '0d0a'x;
 Redirection = '1>NUL 2>&1';
 '@ECHO OFF'
 rcx = SETLOCAL();

 /* some OS/2 Error codes */
 ERROR.NO_ERROR           =   0;
 ERROR.INVALID_FUNCTION   =   1;
 ERROR.FILE_NOT_FOUND     =   2;
 ERROR.PATH_NOT_FOUND     =   3;
 ERROR.ACCESS_DENIED      =   5;
 ERROR.NOT_ENOUGH_MEMORY  =   8;
 ERROR.INVALID_FORMAT     =  11;
 ERROR.INVALID_DATA       =  13;
 ERROR.NO_MORE_FILES      =  18;
 ERROR.WRITE_FAULT        =  29;
 ERROR.READ_FAULT         =  30;
 ERROR.SHARING_VIOLATION  =  32;
 ERROR.GEN_FAILURE        =  31;
 ERROR.INVALID_PARAMETER  =  87;
 ERROR.OPEN_FAILED        = 110;
 ERROR.ENVVAR_NOT_FOUND   = 204;

 GlobalVars = 'Title CmdName env TRUE FALSE CrLf Redirection ERROR.';

 /* load RexxUtil */
 CALL RxFuncAdd    'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs';
 CALL SysLoadFuncs;

 /* some defaults */
 GlobalVars        = GlobalVars 'Flag.';
 rc                = ERROR.NO_ERROR;

 Flag.fHelp        = FALSE;
 Flag.fDebug       = FALSE;
 Flag.fVerbose     = FALSE;

 SourceMask = '';
 TargetDir  = '';
 OutDir     = '';


 /* read commandline parameters */
 PARSE ARG Parms;
 DO i = 1 TO WORDS(Parms)
    ThisParm = WORD(Parms, i);
    PARSE VAR ThisParm ThisTag':'ThisValue;
    ThisTag = TRANSLATE( ThisTag);

    SELECT
       WHEN (POS( ThisTag, '/DEBUG') = 1) THEN
          Flag.fDebug = TRUE;

       WHEN (POS( ThisTag, '/VERBOSE') = 1) THEN
          Flag.fVerbose = TRUE;

       WHEN (POS( ThisTag, '/?') = 1) THEN
       DO
          Flag.fHelp = TRUE;
       END;

       OTHERWISE
       DO
          SELECT
             WHEN (SourceMask = '') THEN
                SourceMask = ThisParm;

             WHEN (TargetDir = '') THEN
                TargetDir = ThisParm;

             WHEN (OutDir = '') THEN
                OutDir = ThisParm;

             OTHERWISE
             DO
                SAY 'error: invalid parameters.'
                SAY;
                'PAUSE'
                EXIT( ERROR.INVALID_PARAMETER);
             END;
          END;
       END;
    END;
 END;

 DO 1

    /* check missing parms */
    IF (TargetDir = '') THEN
    DO
       rcx = SHowHelp();
       LEAVE;
    END;

    IF (OutDir = '') THEN
       OutDir = 'htcmp';

    /* display help */
    IF (Flag.fHelp) THEN
    DO
       rcx = SHowHelp();
       LEAVE;
    END;

    /* check source files */
    FullSourceMask = STREAM( SourceMask, 'C', 'QUERY EXISTS');
    IF (FullSourceMask = '') THEN
    DO
       SAY CmdName': error: file(s)' SourceMask 'not found.';
       rc = ERROR.FILE_NOT_FOUND;
       LEAVE;
    END;
    SourcePathLen = LASTPOS( '\', FullSourceMask) + 1;

    rc = SysFileTree( FullSourceMask, 'File.', 'FOS');
    IF (rc \= ERROR.NO_ERROR) THEN
    DO
       SAY CmdName': error in SysFileTree, rc='rc;
       LEAVE;
    END;

    IF (RIGHT( TargetDir, 1) \= '\') THEN
       TargetDir = TargetDir'\';
    IF (RIGHT( OutDir, 1) \= '\') THEN
       OutDir = OutDir'\';

    /* process all source files */
    DO i = 1 TO File.0

       /* check for target file */
       SourceFile = File.i;
       RelFile = SUBSTR( SourceFile, SourcePathLen);
       TargetFile = TargetDir''RelFile;

       IF (\FileExist( TargetFile)) THEN
       DO
          SAY '-' RelFile 'skipped !';
          ITERATE;
       END;

       /* compare files, detect changes */
       SAY '-' RelFile;
       rc = ReadHtextFile( SourceFile, 'Source.');
       rc = ReadHtextFile( TargetFile, 'Target.');

       ScanList     = Source._anchorlist;
       RemainList   = Target._anchorlist;

       ModifiedList       = '';
       ModifiedActionList = '';

       SourceIdx = 0;
       DO WHILE (ScanList \= '')
          PARSE VAR ScanList ThisPanel ScanList;
          SourceIdx = SourceIdx + 1;
          TargetIdx = WORDPOS( ThisPanel, Target._anchorlist);

          /* deleted panel */
          IF (TargetIdx = 0) THEN
          DO
             ModifiedList       = ModifiedList ThisPanel;
             ModifiedActionList = ModifiedActionList 'deleted';
             ITERATE;
          END;

          IF ((Source.SourceIdx._Title \= Target.TargetIdx._Title) |,
              (Source.SourceIdx._Content \= Target.TargetIdx._Content)) THEN
          DO
             ModifiedList = ModifiedList ThisPanel;
             ModifiedActionList = ModifiedActionList 'modified';
          END;

          /* remove from remaing list */
          RemainingIdx = WORDPOS( ThisPanel, RemainList);
          IF (RemainingIdx > 0) THEN
             RemainList = DELWORD( RemainList, RemainingIdx, 1);
       END;


       /* show result */
       DO WHILE (ModifiedList \= '')
          PARSE VAR ModifiedList ThisPanel ModifiedList;
          PARSE VAR ModifiedActionList ThisAction ModifiedActionList;
          IF (Flag.fVerbose) THEN
             SAY '-' ThisPanel':' ThisAction;
          rcx = WritePanel( OutDir, RelFile, ThisPanel);

          /* check if next remaining item comes right before next modified item */
          NextModified = WORD( ModifiedList, 1);
          NextRemain = WORD( RemainList, 1);
          NextModifiedPos = WORDPOS( NextModified, Target._anchorlist);
          NextRemainPos   = WORDPOS( NextRemain, Target._anchorlist);
          IF (NextRemainPos < NextModifiedPos) THEN
          DO
             PARSE VAR RemainList ThisPanel RemainList;
             IF (Flag.fVerbose) THEN
                SAY '-' ThisPanel': added';
             rcx = WritePanel( OutDir, RelFile, ThisPanel);
          END;
       END;
    END;

 END;

 EXIT( rc);

/* ------------------------------------------------------------------------- */

HALT:
 SAY;
 SAY 'Interrupted by user.';
 EXIT(ERROR.GEN_FAILURE);

/* ------------------------------------------------------------------------- */
ShowHelp: PROCEDURE EXPOSE (GlobalVars)

 PARSE SOURCE . . ThisFile

 SAY;
 SAY Title;
 SAY;

 /* skip header */
 DO i = 1 TO 3
    rc = LINEIN(ThisFile);
 END;

 /* show help */
 DO WHILE (ThisLine \= ' */')
    ThisLine = LINEIN(Thisfile);
    SAY SUBSTR(ThisLine, 7);
 END;

 /* close file */
 rc = LINEOUT(Thisfile);

 RETURN('');

/* ------------------------------------------------------------------------- */
FileExist: PROCEDURE
 ARG FileName

 RETURN(STREAM(Filename, 'C', 'QUERY EXISTS') > '');

/* ------------------------------------------------------------------------- */
ReadHtextFile: PROCEDURE EXPOSE (GlobalVars) Source. Target.
 PARSE ARG File, StemName;

 rc = ERROR.NO_ERROR;
 AnchorList = '';
 IF (RIGHT( StemName, 1) \= '.') THEN
    StemName = StemName'.';

 INTERPRET( 'DROP(' StemName')');
 INTERPRET( StemName'.0 = 0');
 CurrentPanel = 0;
 GenAnchor = 0;

 DO 1

    /* open file */
    IF (STREAM( File, 'C', 'OPEN READ') \= 'READY:') THEN
    DO
       SAY;
       SAY CmdName': error: cannot open' File 'for reading.';
       rc = ERROR.OPEN_FAILED;
       LEAVE;
    END;

    INTERPRET( StemName'_file = file');

    Content  = '';
    Anchor   = '';
    Title    = '';
    fpos     = 0;
    lastfpos = 0
    last2fpos = 0
    LastLine = '';
    StartPos = 0;
    DO WHILE (LINES( File))

       /* skip empty and comment lines */
       LastLine = ThisLine;
       ThisLine = LINEIN( File);
       last2fpos = lastfpos;
       lastfpos = fpos;
       fpos = fpos + LENGTH( ThisLine) + 2;

       IF (ThisLine = '') THEN ITERATE;
       IF ( LEFT( ThisLine, 2) = '..') THEN
          ITERATE;

       IF ( LEFT( ThisLine, 1) = '.') THEN
       DO
          /* check for panels */
          Command = SUBSTR( TRANSLATE( WORD( ThisLine, 1)), 2);
          IF (POS( Command, '123456789') > 0) THEN
          DO
             /* store content of previous panel */
             IF (CurrentPanel > 0) THEN
             DO

                /* check line before header, add comment line */
                IF (LEFT( LastLine, 2) = '..') THEN
                   StartPos = last2fpos;
                ELSE
                   StartPos = lastfpos;

                IF (Anchor = '') THEN
                DO
                   SAY 'warning: panel has no anchor, panel title is:' title;
                   Anchor = '#'GenAnchor;
                   GenAnchor = GenAnchor + 1;
                END;

                AnchorList = AnchorList Anchor;
                INTERPRET( StemName'CurrentPanel._content = Content');
                INTERPRET( StemName'CurrentPanel._title   = title');
                INTERPRET( StemName'CurrentPanel._end     = StartPos');
             END;

             /* check anchor of new panel */
             Anchor = '';
             Content = '';
             Title = SUBSTR( ThisLine, WORDINDEX( ThisLine, 2));
             CurrentPanel = CurrentPanel + 1;
             INTERPRET( StemName'CurrentPanel._start   = StartPos');
             ITERATE;
          END;

          IF (POS( Command, 'ANCHOR') = 1) THEN
          DO
             /* store content of previous panel */
             IF (CurrentPanel > 0) THEN
                INTERPRET( StemName'CurrentPanel._content = Content');

             /* check anchor of new panel */
             PARSE VAR ThisLine . Anchor;
             Anchor = STRIP( Anchor);

             ITERATE;
          END;


       END;

       Content = Content SPACE( ThisLine);

    END;
    rcx = STREAM( File, 'C', 'CLOSE');

    /* store content of last panel */
    INTERPRET( StemName'CurrentPanel._content = Content');

    /* save index */
    INTERPRET( StemName'0 = CurrentPanel');

    /* save anchor list */
    INTERPRET( StemName'_anchorlist = AnchorList');

 END;

 RETURN( rc);

/* ========================================================================= */
WritePanelFile: PROCEDURE EXPOSE (GlobalVars) Source. Target.
 PARSE ARG Panel, File, StartPos, PanelLen, TmpFile, TargetDir,

 rcx = STREAM( File, 'C', 'OPEN READ');
 Contents = CHARIN( File, StartPos, PanelLen);
 rcx = STREAM( File, 'C', 'CLOSE');

 rcx = SysFileDelete( TmpFile);
 rcx = LINEOUT( TmpFile, Contents);
 rcx = STREAM( TmpFile, 'C', 'CLOSE');
 'XCOPY' TmpFile TargetDir'\' Redirection;
 rcx = SysFileDelete( TmpFile);

 'REN' TargetDir'\'Panel'.*' Panel;

 RETURN( 0);

/* ========================================================================= */
WritePanel: PROCEDURE EXPOSE (GlobalVars) Source. Target.
 PARSE ARG OutDir, RelFile, Panel;

 TmpDir = VALUE('TMP',,env);
 Basename = FILESPEC( 'N', RelFile);
 fCheckBatchFile = FALSE;

 SourceIdx = WORDPOS( Panel, Source._anchorlist);
 TargetIdx = WORDPOS( Panel, Target._anchorlist);

 fWriteSourcePath = ((FileExist( Source._file)) & (WORDPOS( Panel, Source._anchorlist) > 0));
 fWriteTargetPath = ((FileExist( Target._file)) & (WORDPOS( Panel, Target._anchorlist) > 0));

 SourceOutFileTmp = TmpDir'\'Panel'.1';
 TargetOutFileTmp = TmpDir'\'Panel'.2';

 IF ((fWriteSourcePath) & (fWriteTargetPath)) THEN
 DO
    Action = 'modified';
    fCheckBatchFile = TRUE;
    FinalOutDirSource = OutDir''RelFile'\'Action'\old';
    FinalOutDirTarget = OutDir''RelFile'\'Action'\new';
 END;
 ELSE
 DO
    IF (fWriteSourcePath) THEN
       Action = 'deleted';
    ELSE
       Action = 'added';
    FinalOutDirSource = OutDir''RelFile'\'Action;
    FinalOutDirTarget = FinalOutDirSource;
 END;

 IF (fWriteSourcePath) THEN
    rcx = WritePanelFile( Panel,,
                          Source._file,,
                          Source.Sourceidx._start + 1,,
                          Source.Sourceidx._end - Source.Sourceidx._start,,
                          SourceOutFileTmp,,
                          FinalOutDirSource);

 IF (fWriteTargetPath) THEN
    rcx = WritePanelFile( Panel,,
                          Target._file,,
                          Target.Targetidx._start + 1,,
                          Target.Targetidx._end - Target.Targetidx._start,,
                          TargetOutFileTmp,,
                          FinalOutDirTarget);

 RETURN( 0);

