/************************************************************************/
/***                                                                  ***/
/*** AUTHOR            : RH                                           ***/
/***                                                                  ***/
/*** SHORT DESCRIPTION : miscellaneous helper functions for all       ***/
/***                     mtestpm modules which need them              ***/
/***                                                                  ***/
/*** PART OF           : mtestpm                                      ***/
/***                                                                  ***/
/*** COPYRIGHT         : (C) 1995 a priori Computer Solutions GmbH    ***/
/***                                                                  ***/
/************************************************************************/

#ifdef __IBMC__
#pragma strings(readonly)
#endif

/*----------------------------------------------------------------------*/
/* includes                                                             */
/*----------------------------------------------------------------------*/

#define INCL_BASE
#define INCL_PM
#include <os2.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/*--------------------------------------*/
/* Define trace levels                  */
/*--------------------------------------*/
#define   DATT_BASE_LEVEL   2000
#define   DATT_TM_LEVEL     2001
#define   TM_I              DATT_BASE_LEVEL + 1   /* Info */
#define   TM_E              DATT_BASE_LEVEL + 2   /* error */
#define   TM_H              DATT_BASE_LEVEL + 3   /* hex dump */

/*--------------------------------------*/
/* include the C-Scout definitions here */
/*--------------------------------------*/
#include "datt.h"

// mtestpm common definitions
#include "mtestpm.h"

// mtestpm resource id definitions
#include "resid.h"

/*----------------------------------------------------------------------*/
/* defines                                                              */
/*----------------------------------------------------------------------*/

#define ABOUT_TEXT "~About"

/*----------------------------------------------------------------------*/
/* external static variables                                            */
/*----------------------------------------------------------------------*/
/* anchor block handle of mtestpm.c  */
extern HAB habThis ;

/* help instance data                 */
static HWND hwndHelpInstance;
static BOOL fHelpEnabled;

/************************************************************************/
/* FUNCTION-NAME       : MiscDisplayMsg                                 */
/*                                                                      */
/* DESCRIPTION         : centers the dialog into given HWND's client    */
/*                                                                      */
/* PARAMETERS          : hwndOwner - window handle of the owner of the  */
/*                                   message box                        */
/*                       idMsg     - id of message to be retrieved from */
/*                                   resource file                      */
/*                       fsStyle   - message box style                  */
/*                       fBeep     - flag to sound alarm                */
/*                                                                      */
/* RETURN-VALUE        : none                                           */
/*                                                                      */
/************************************************************************/
VOID MiscDisplayMsg ( HWND    hwndOwner,
                      SHORT   idMsg,
                      SHORT   fsStyle,
                      BOOL    fBeep )
  {
  CHAR szText[MESSAGELEN];

  FncEntry();

  TX( !WinLoadMessage( habThis,
                       (HMODULE)NULL,
                       idMsg,
                       MESSAGELEN,
                       (PSZ)szText ) );

  if (fBeep) WinAlarm(HWND_DESKTOP, WA_ERROR);

  TX( WinMessageBox( HWND_DESKTOP,
                     hwndOwner,
                     szText,
                     (PSZ)NULL,
                     IDD_MSGBOX,
                     fsStyle ) == MBID_ERROR );

  FncExitVoid();
  }  /* MiscDisplayMsg        */

/************************************************************************/
/* FUNCTION-NAME       : MiscCenterDialog                               */
/*                                                                      */
/* DESCRIPTION         : centers the dialog into given HWND's client    */
/*                                                                      */
/* PARAMETERS          : parent hwnd, dialog hwnd                       */
/*                                                                      */
/* RETURN-VALUE        : none                                           */
/*                                                                      */
/************************************************************************/
VOID MiscCenterDialog( HWND   hwndParent,
                       HWND   hwndDialog)
  {
  SHORT  cx, cy;          /* New x and y location for dialog box. */
  SWP    swpParent;
  SWP    swpDialog;

  FncEntry();

  /*---------------------------------------------------------------*/
  /* Determine the current size (and position) of given parent     */
  /*---------------------------------------------------------------*/
  TX( !WinQueryWindowPos(hwndParent, (PSWP)&swpParent) );

  /*---------------------------------------------------------------*/
  /* Determine the current size (and position) of the dialog box.  */
  /*---------------------------------------------------------------*/
  TX( !WinQueryWindowPos(hwndDialog, (PSWP)&swpDialog) );

  /*---------------------------------------------------------------*/
  /* Calculate the correct new x and y location of the dialog box. */
  /*---------------------------------------------------------------*/
  cx = (swpParent.cx - swpDialog.cx) / 2;
  cy = (swpParent.cy - swpDialog.cy) / 2;

  /*--------------------------------------------------------------*/
  /* Set the position of the dialog box to the new x and y values.*/
  /*--------------------------------------------------------------*/
  TX( !WinSetWindowPos( hwndDialog,
                        HWND_TOP,
                        cx,
                        cy,
                        0,
                        0,
                        SWP_MOVE ) );

  FncExitVoid();

  } /* MiscCenterDialog */

/************************************************************************/
/* FUNCTION-NAME       : MiscInitHelp                                   */
/*                                                                      */
/* DESCRIPTION         : Initializes the IPF help facility              */
/*                                                                      */
/* PARAMETERS          : hwnd to associate with                         */
/*                                                                      */
/* RETURN-VALUE        : none                                           */
/*                                                                      */
/************************************************************************/
VOID MiscInitHelp(HWND hwnd)
  {
  HELPINIT helpInit;

  FncEntry();

  /*--------------------------------------------------------------*/
  /* if we return because of an error, Help will be disabled      */
  /*--------------------------------------------------------------*/
  fHelpEnabled = FALSE;

  /*--------------------------------------------------------------*/
  /* inititalize help init structure                              */
  /*--------------------------------------------------------------*/

  helpInit.cb = sizeof(HELPINIT);
  helpInit.ulReturnCode = 0L;
  helpInit.pszTutorialName = (PSZ)NULL;
  helpInit.phtHelpTable = (PHELPTABLE)MAKELONG(HELPTABLE_mtestpm,
                                               0xFFFF);
  helpInit.hmodHelpTableModule = (HMODULE)0;
  helpInit.hmodAccelActionBarModule = (HMODULE)0;
  helpInit.idAccelTable = 0;
  helpInit.idActionBar = 0;
  helpInit.pszHelpWindowTitle = (PSZ) "mtestpm Help";
  helpInit.fShowPanelId = CMIC_HIDE_PANEL_ID;
  helpInit.pszHelpLibraryName = (PSZ)"mtestpm.hlp";

  /*--------------------------------------------------------------*/
  /* creating help instance                                       */
  /*--------------------------------------------------------------*/
  hwndHelpInstance = WinCreateHelpInstance(habThis, &helpInit);

  if (!hwndHelpInstance || helpInit.ulReturnCode)
     {
     MiscDisplayMsg( hwnd,
                     IDMSG_HELPLOADERROR,
                     MB_OK | MB_ERROR,
                     TRUE );
     FncExitVoid();
     }

  /*--------------------------------------------------------------*/
  /* associate help instance with main frame                      */
  /*--------------------------------------------------------------*/
  if ( !WinAssociateHelpInstance(hwndHelpInstance, hwnd))
     {
     MiscDisplayMsg( hwnd,
                     IDMSG_HELPLOADERROR,
                     MB_OK | MB_ERROR,
                     TRUE );
     FncExitVoid();
     }

  /*--------------------------------------------------------------*/
  /* help manager is successfully initialized so set flag to TRUE */
  /*--------------------------------------------------------------*/
  fHelpEnabled = TRUE;

  FncExitVoid();

  } /* MiscInitHelp  */

/************************************************************************/
/* FUNCTION-NAME       : MiscDisplayHelp                                */
/*                                                                      */
/* DESCRIPTION         : displays the desired help panel                */
/*                                                                      */
/* PARAMETERS          : help msg HM_EXT_HELP                           */
/*                                HM_KEYS_HELP                          */
/*                                HM_HELP_INDEX                         */
/*                                HM_EXT_HELP                           */
/*                                HM_DISPLAY_HELP + panelid             */
/*                                                                      */
/* RETURN-VALUE        : none                                           */
/*                                                                      */
/************************************************************************/
VOID  MiscDisplayHelp(ULONG msg, USHORT usPanelId)
  {
  FncEntry();

  if (fHelpEnabled)
     {
     switch (msg)
        {
        case HM_EXT_HELP :
        case HM_KEYS_HELP :
        case HM_HELP_INDEX:
             if ( (BOOL)WinSendMsg(hwndHelpInstance, msg, NULL, NULL) )
                MiscDisplayMsg( HWND_DESKTOP,
                                IDMSG_HELPDISPLAYERROR,
                                MB_OK | MB_ERROR,
                                FALSE );
             break;

        case HM_DISPLAY_HELP:
             if ( (BOOL)WinSendMsg( hwndHelpInstance,
                                    HM_DISPLAY_HELP,
                                    MPFROM2SHORT(usPanelId, NULL),
                                    MPFROMSHORT(HM_RESOURCEID) ) )
                MiscDisplayMsg( HWND_DESKTOP,
                                IDMSG_HELPDISPLAYERROR,
                                MB_OK | MB_ERROR,
                                FALSE );
             break;

        default:
             break;

        }
     }

  FncExitVoid();

  } /* MiscDisplayHelp      */


/************************************************************************/
/* FUNCTION-NAME       : MiscDestroyHelp                                */
/*                                                                      */
/* DESCRIPTION         : destroys the help instance                     */
/*                                                                      */
/* PARAMETERS          : hwnd to associate with                         */
/*                                                                      */
/* RETURN-VALUE        : none                                           */
/*                                                                      */
/************************************************************************/
VOID MiscDestroyHelp()
  {
  FncEntry();

  if (hwndHelpInstance)
     WinDestroyHelpInstance(hwndHelpInstance);

  FncExitVoid();

  } /* MiscDestroyHelp    */

/************************************************************************/
/*** EOF misc.c                                                       ***/
/************************************************************************/
