## Copyright (C) 1999 Paul Kienzle
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

## Generate an Chebyshev type II filter with Rs dB of pass band ripple.
## 
## [b, a] = cheby2(n, Rs, Wc)
##    low pass filter with cutoff pi*Wc radians
##
## [b, a] = cheby2(n, Rs, Wc, 'high')
##    high pass filter with cutoff pi*Wc radians
##
## [b, a] = cheby2(n, Rs, [Wl, Wh])
##    band pass filter with edges pi*Wl and pi*Wh radians
##
## [b, a] = cheby2(n, Rs, [Wl, Wh], 'stop')
##    band reject filter with edges pi*Wl and pi*Wh radians
##
## [z, p, g] = cheby2(...)
##    return filter as zero-pole-gain rather than coefficients of the
##    numerator and denominator polynomials.
##
## [...] = cheby2(...,'s')
##     return a Laplace space filter, W can be larger than 1.
## 
## [a,b,c,d] = cheby2(...)
##  return  state-space matrices 
## 
## References: 
##
## Parks & Burrus (1987). Digital Filter Design. New York:
## John Wiley & Sons, Inc.

## Author: Paul Kienzle <pkienzle@users.sf.net>
## Modified: Doug Stewart Feb. 2003

function [a,b,c,d] = cheby2(n, Rs, W, varargin)

  if (nargin>5 || nargin<3) || (nargout>4 || nargout<2)
    usage ("[b, a] or [z, p, g] or [a,b,c,d]= cheby2 (n, Rs, W, [, 'ftype'][,'s'])");
  end

  ## interpret the input parameters
  if (!(length(n)==1 && n == round(n) && n > 0))
    error ("cheby2: filter order n must be a positive integer");
  end


  stop = 0;
  digital = 1;  
  for i=1:length(varargin)
    switch varargin{i}
    case 's', digital = 0;
    case 'z', digital = 1;
    case { 'high', 'stop' }, stop = 1;
    case { 'low',  'pass' }, stop = 0;
    otherwise,  error ("cheby2: expected [high|stop] or [s|z]");
    endswitch
  endfor

  [r, c]=size(W);
  if (!(length(W)<=2 && (r==1 || c==1)))
    error ("cheby2: frequency must be given as w0 or [w0, w1]");
  elseif (!(length(W)==1 || length(W) == 2))
    error ("cheby2: only one filter band allowed");
  elseif (length(W)==2 && !(W(1) < W(2)))
    error ("cheby2: first band edge must be smaller than second");
  endif

  if ( digital && !all(W >= 0 & W <= 1))
    error ("cheby2: critical frequencies must be in (0 1)");
  elseif ( !digital && !all(W >= 0 ))
    error ("cheby2: critical frequencies must be in (0 inf)");
  endif

  if (Rs < 0)
    error("cheby2: passband ripple must be positive decibels");
  end

  ## Prewarp to the band edges to s plane
  if digital
    T = 2;       # sampling frequency of 2 Hz
    W = 2/T*tan(pi*W/T);
  endif

  ## Generate splane poles and zeros for the chebyshev type 2 filter
  ## From: Stearns, SD; David, RA; (1988). Signal Processing Algorithms. 
  ##       New Jersey: Prentice-Hall.
  C = 1;			# default cutoff frequency
  lambda = 10^(Rs/20);
  phi = log(lambda + sqrt(lambda^2-1))/n;
  theta = pi*([1:n]-0.5)/n;
  alpha = -sinh(phi)*sin(theta);
  beta = cosh(phi)*cos(theta);
  if (rem(n,2))
    ## drop theta==pi/2 since it results in a zero at infinity
    zero = 1i*C./cos(theta([1:(n-1)/2, (n+3)/2:n]));
  else
   zero = 1i*C./cos(theta);
  endif
  pole = C./(alpha.^2+beta.^2).*(alpha-1i*beta);

  ## Compensate for amplitude at s=0
  ## Because of the vagaries of floating point computations, the
  ## prod(pole)/prod(zero) sometimes comes out as negative and
  ## with a small imaginary component even though analytically
  ## the gain will always be positive, hence the abs(real(...))
  gain = abs(real(prod(pole)/prod(zero)));

  ## splane frequency transform
  [zero, pole, gain] = sftrans(zero, pole, gain, W, stop);

  ## Use bilinear transform to convert poles to the z plane
  if digital
    [zero, pole, gain] = bilinear(zero, pole, gain, T);
  endif

  ## convert to the correct output form
  if nargout==2, 
    a = real(gain*poly(zero));
    b = real(poly(pole));
  elseif nargout==3,
    a = zero;
    b = pole;
    c = gain;
  else
    ## output ss results 
    [a, b, c, d] = zp2ss (zero, pole, gain);
  endif

endfunction
