## Copyright (C) 1996, 1997  Kurt Hornik
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## usage:  disc_pdf (X, V, P)
##
## For each element of X, compute the probability density function (PDF)
## at X of a univariate discrete distribution which assumes the values
## in V with probabilities P.

## Author:  KH <Kurt.Hornik@ci.tuwien.ac.at>
## Description:  PDF of a discrete distribution

function pdf = disc_pdf (X, V, P)
  
  if (nargin != 3)
    usage ("disc_pdf (X, V, P)");
  endif

  [r, c] = size (X);

  if (! is_vec (V))
    error ("disc_pdf:  V must be a vector");
  elseif (! is_vec (P) || (length (P) != length (V)))
    error ("disc_pdf:  P must be a vector with length (V) elements");
  elseif (! (all (P >= 0) && any (P)))
    error ("disc_pdf:  P must be a nonzero, nonnegative vector");
  endif

  n = r * c;
  m = length (V);
  X = reshape (X, n, 1);
  V = reshape (V, 1, m);
  P = reshape (P / sum (P), m, 1);

  pdf = zeros (n, 1);
  k = find (isnan (X));
  if any (k)
    pdf (k) = NaN * ones (length (k), 1);
  endif
  k = find (!isnan (X));
  if any (k)
    n = length (k);
    pdf (k) = ((X(k) * ones (1, m)) == (ones (n, 1) * V)) * P;
  endif

  pdf = reshape (pdf, r, c);

endfunction
