Program SolutionOfEquationOfHodgkinAndHuxley;
{$R-,S-,I-}
(* Filename H-HSOL06.pas, ver 0.16, 28-Oct-92.
                                Changes in
   Revised  2-Nov-92
   Revised  9-Nov-92  0.16b     Stimulation Setting
   Revised 11-Nov-92  0.16c     HardCopy
   Revised 18-Nov-92  0.16d     function dv_dt (Imemb become to be used)
                                co_n(,m,h) -> dn(,m,h)/dt
           30-Nov-92  0.16d+    n(,m,h)_power
                                Result Save functiion omitted
            4-Dec-92  0.16d++   Bug fix in real_const
   HSOL06.pas              ver 1.0  1995
                                Port to OS/2 PM
                                Target compiler is changed to Speed-Pascal/2

   HSOL06A/B/C

   This programme aims to solve the Hodgkin & Huxley's equations on given
   initial parametres and to simulate the exitation of neurones, for the
   student practice.
   Please refer the original articles for physiological and mathematical
   bases of these procedures.

   Units
        time            : msec
        voltage         : mV
        current         : micro A per squae cm
        conductance     : mmho per square cm
        capacitance     : micro Farad per square cm

   Caution : According to original difinition, the resting potential is 0,
             depolarisation signed negative.

   Versions
            The origins of "H-HSOL" program series are "Huxley" program
            series and "TEACHER" or "STUDENT" series.
            Algorhisms of these original programs are modified
            and fixed to accord to the original article. Using procedure/
            fuction type ( a Turbo Pascal alternative to procedure/func-
            tion parametre in Standard Pascal ), program become clear and
            short.

            H-HSOL01    - basic
            H-HSOL02    - fasten by global parametre call
            H-HSOL03    - graphic driver included in object file
            H-HSOL10    - test for non global version
            H-HSOL04    - channel port animation

                   vers 0.1         30-Oct-90
                                correct_params,
                                gtools1,
                                d_ionic_current,
                                calc_main,
                                init_prog_manual,
                                init_prog_auto.
                   vers 0.2          6-Nov-90
                   (= vers 1.4a )
            H-HSOL05    - TP6
                          Stimulation by Current.
            H-HSOL06    - Calculation engine is encapsulated (object membrane).
                          Menu driven user interface.
                          Text viewer is modified from that of H-HSOL11
                                                        (VText unit)
                          An old bug in initialisation is fixen
                           ( uninitialised variable dt was used in set_value)
            HSOL06      - Port to PM
            HSOL06-F    - SP/2 1.0
            HSOL06G     - SP/2 1.5
            HSOL06H     - Test for fast redraw
            HSOL06I     - Fast redraw.
            HSOL06J     - Copy traces to clipboard/file
            HSOL06K     = HSOL07 Copy traces by thread
            HSOL11

   Auther  : The Creative CAT, Physiol 1, Med., Niitaga Univ.
*)

{$R HSOL12}

uses DOS, OS2DEF, CMDEF,
     PmGpi, PmWin, BSEDOS, PmStdDlg, chnlport, hsolRID;

const
(* standard value of membrane constants *)
  Q10           =   3  ;
  stdvK         =  12.0;        (* Potassium equivalent potential *)
  stdvNa        =-115.0;        (* Sodium equivalent potential *)
  stdvL         = -10.6;        (* Leakage ions equivalent potential *)
  stdcgK        =  36.0;        (* Maximum potassium conductance *)
  stdcgNa       = 120.0;        (* Maximum sodium conductance *)
  stdgL         =   0.3;        (* Leakage conductance *)
  stdCM         =   2.0;        (* Membrane capacitance *)
  stdK          =   3.5;        (* Cable Spec. *)
  stdtemp       =   6.3;
  stdnpower     =   4  ;
  stdmpower     =   3  ;
  stdhpower     =   1  ;
  stdcodndt     =   1  ;
  stdcodmdt     =   1  ;
  stdcodhdt     =   1  ;

(* time, voltage limit *)
  maxTlimit     = 40;
  dtslow        = 0.001;
  dtnorm        = 0.005;
  dtfast        = 0.01 ; (* time step *)
  maxV          = 200;
  maxpower      = 10;

(* parametre table *)
  tableVmin     = -3000;
  tableVmax     = 2000;
  tableVfactor  = 25;

(* file names *)
  TableName     = 'HSOL.tbl';  (* table file name *)
  defSettingFile= 'HSOL.hxl';  (* default setting file name *)

(* Graph data *)
  DispNum       = 800;
  DispPortSkip  = 4;
  xofs          = 80;
  Rmargin       = 10;
  IntMagn       = 512;
  nmhMagn       = 1024;

(* stimulation *)
  MaxStim       = 10;

(* Clip Board *)
  ColumnSize = 8;
  ColumnNum  = 8;

(* PM control *)
  CalcWndClass  = 'H_HCALC';               (* Class Name *)
  ParamWndClass = 'H_HPARAMETRE';
  AppTitle      = 'Hodgkin and Huxley';    (* Application Name *)
  fSwp          = SWP_MOVE + SWP_SHOW;
  StackSize     = 8192;
  ParamStrLen   = 78;
  NumStrLen     = 18;
  WM_ShowTmRecord    = WM_USER + 1;
    (* For parametre window. showing tm record.  m1 - address of tm to display. *)
  WM_ShowAnyString   = WM_USER + 2;
    (* For parametre window. showing any string.  m1 - address of string. *)
  WM_TraceCalculated = WM_USER + 3;
    (* For calc window. notifing completion of calculation *)
  WM_TracePrepared   = WM_USER + 4;
    (* For calc window. notifing completion of Trace -> string. m1 - 0 : clip
       board, 1 : file *)
type
  realT         = extended;
  PropMode      = (action, prop);        (* action or propagated potential *)
  gates         = (nGate, mGate, hGate); (* three theoretical gates *)
  DifferenceOrder=(dif0, dif1, dif2);    (* backward difference *)
  diffs         = array [DifferenceOrder] of realT;
  DifferentialOrder=(ft, dfdt, ddfdt2);

  parametre     = array [DifferentialOrder] of diffs;
  parametres    = record
    n, m, h, v  : parametre; (* n, m, h gates, voltage *)
    gK, gNa,                 (* Potassium, Sodium conductance *)
    Iadditional,             (* Intracellular stimulation *)
    t, dt       : realT;     (* time *)
    RisingPhase : boolean    (* propagetion *)
  end;

  pwr           = 0..MaxPower;

  consts        = record (* membrane constants *)
    vK,                  (* Potassium equivalent potential *)
    vNa,                 (* Sodium equivalent potential *)
    vL,                  (* Leakage ions equivalent potential *)
    cgK,                 (* Maximum potassium conductance *)
    cgNa,                (* Maximum sodium conductance *)
    gL,                  (* Leakage conductance *)
    CM,                  (* Membrane capacitance *)
    K,
    temp,
    PhyVal,
    Codndt, Codmdt, Codhdt : realT;  (* Multiplying factor to d(gate)/dt *)
    nPower, mPower, hPower : pwr     (* Multiplying factor to gate *)
  end;

  funct1        = function(x:realT):realT;
  funct4        = function(f, v1:realT;
                           var cnst:consts; var prms:parametres):realT;
    (* procedure/function type, a Turbo Pascal alternative to
       procedure/function parametre in Standard Pascal *)

  tm            = record
    tt,
    tdt,
    tv,
    tn, tm, th,
    tgK, tgNa,
    ti        : single;
    xt,
    yv,
    yn, ym, yh,
    ygK, ygNa,
    yI        : longint
  end;

  StimPoint     = 0..MaxStim; (* 0 is for sentinel *)

  stimulusV     = record
    time,
    voltage     : realT       (* electrical stimulation *)
  end;
  stimuliVary   = array [StimPoint] of stimulusV;
  Vstim         = record
    cb          : word;
    stim        : stimuliVary;
    count,
    num         : integer
  end;
  pVstim        = ^Vstim;

  stimulusItem  = record
    time0,
    time1,
    stimval     : realT;     (* electrical stimulation *)
  end;
  stimuliary    = array [StimPoint] of stimulusItem;
  Istim         = record
    cb          : word;
    stim        : stimuliary;
    count,
    num         : integer;
    preV        : realT;
    switch      : boolean
  end;
  pIstim        = ^Istim;

  Vclamp        = record
    stim        : stimuliary;
    count,
    num         : integer;
    switch      : boolean
  end;

(* parametre tables *)
  tableV        = tableVmin..tableVmax;
  ParamTable    = array [tableV] of real;

  membrane      = object
    mode, dummy : PropMode;
    CurrentParam: parametres;                  (* main variables *)
    olddt       : realT;
    constants   : consts;                      (* membrane constants *)
    Vclamped, Iclamped : boolean;
    ClampVal    : realT;
    constructor init;
    procedure setup(initT, initV : realT;
                    initCnst : consts;
                    initMode : PropMode);
    procedure calculate;
    function  setdt : realT;
    procedure stimulationV(stimV : realT);
    procedure setVclamp(stimV : realT);
    procedure resetVclamp;
    procedure getval(var retTm : tm);
    procedure getconst(var retcgK, retcgNa:realT);
    procedure setconsts(retCnst : consts);
    procedure getconsts(var retCnst : consts);
    procedure getparams(var retPrms : parametres);
    procedure setI(initI : realT);
    destructor done
  end;

  DispArray     = array[0..DispNum] of POINTL;

  StmDlgData    = record
    cb          : word;
    text1,
    text2       : string;
    stm         : stimuliary;
    nm          : integer
  end;
  pStmDlgData   = ^StmDlgData;

  preference    = record
    cb          : word;
    MembParam   : consts;
    TLimit,
    initV       : realT;
    UseDosSleep,
    PortAnimation,
    UseTable    : boolean;
    mode        : PropMode;
    PositionLoaded : boolean;
    posx, posy,
    sizex, sizey: LONG;
    SettingFile : string
  end;
  ppref         = ^preference;

  sstring       = string[80];
  gstring       = string[10];
  ParaStr       = string[ParamStrLen];
  ppstr         = ^ParaStr;

(* dialog box data *)
  NumStr        = string[NumStrLen];
  pnstr         = ^NumStr;
  NumStrDlgDt   = record
    cb          : word;
    title,
    st          : NumStr
  end;
  pNumStrDlgDt  = ^NumStrDlgDt;

  chararray = array[0..160000] of char;
  pchary     = ^chararray;

var
(* Proper *)
  memb          : membrane;
  stimV         : Vstim;
  stimI         : Istim;
  clmpV         : Vclamp;
  falphan,
  falpham,
  falphah,
  fbetan,
  fbetam,
  fbetah        : funct1;

  TimeCourse    : array[0..DispNum] of tm;
  TmNum         : word;
  settings      : preference;

  NaChannel     : NaPortShape;
  KChannel      : KPortShape;

(* paramtre tables *)
  alphaTablen,
  alphaTablem,
  alphaTableh,
  betaTablen,
  betaTablem,
  betaTableh    : ^ParamTable;   (* Previously calculated table *)

(* PM control *)
  myhab         : HAB;
  myhmq         : HMQ;                (* Message Queue Handle *)
  hwndFrame,                          (* Frame Window Handle *)
  hwndCalcArea,                       (* Client Window Handle *)
  hwndParamArea : HWND;               (* Bottom Line *)
  ctlData       : ULONG;              (* Control Flag *)
  myqmsg        : QMSG;               (* Message Queue *)
  calchps       : HPS;                (* HPS of hwndCalcArea *)
  DrawToMain    : HMTX;
  calculating   : HEV;                (* calculating thread running *)
  idCalcThread,
  idCopyThread  : TID;
  myhaccel      : HACCEL;
  ViewMat       : MATRIXLF;           (* Viewing Matrix *)
  xMagnFIXED,
  vMagnFIXED,
  nmhMagnFIXED,
  IMagnFIXED,
  gMagnFIXED    : FIXED;              (* Magnification factor in FIXED *)

(* PM parametres *)
  AveChar,
  CharHeight    : word;
  xMagn,
  yMagn         : real;
  XtoTm         : array[0..3000] of integer;
  cxClient,
  cyClient      : integer;
  CursorPos     : integer;
  DrawCursor,
  buttonpush,
  ToCalc,
  Resized       : boolean;

(* Data for Displaying Traces *)
  vTime, nTime,
  mTime, hTime,
  gKTime, gNaTime,
  ITime         : DispArray;  (* Point arrey for GpiPolyLine *)

(* Clip Board buffer *)
  pdata         : pchary;


(* i/o or numerical routines *)

function optim(r:realT):integer;
begin
  if abs(r)<maxint then optim := round(r)
                   else optim := maxint
end;

function limited_real(f, lower, upper:realT):realT;
begin
  if      f<lower then limited_real := lower
  else if f>upper then limited_real := upper
  else                 limited_real := f
end;

function power(r:realT; n:pwr):realT;   (* r power i *)
begin                                   (* non recursion *)
  case n of
    4 : power := sqr(sqr(r));
    3 : power := r*sqr(r);
    2 : power := sqr(r);
    1 : power := r;
    0 : power := 1
  else
    power := exp(ln(r)*n)
  end
end;

function realpower(r1, r2:realT) : realT;       (* r1 power r2 *)
begin
  realpower := exp(r2*ln(r1))
end;

function errorOk(f1, f2, errlim:realT) : boolean;
begin
  errorOk := (abs(f1-f2) <= errlim*abs(f1))
end;

procedure difference(var df0, df1:diffs; difx:DifferenceOrder);
begin
  df1[difx] := df1[pred(difx)] - df0[pred(difx)]
end;

procedure estimation(var df0, df1:diffs; difx:DifferenceOrder);
(* crude estimation by backward difference *)
begin
  df1[difx] := df0[difx] + df0[succ(difx)]
end;

function summing(var f0, f1 : parametre;
                     dfx : DifferentialOrder; dt : realT) : realT;
(* to solve differental-diference equation - step (4) *)
var
  ddfx : DifferentialOrder;
begin
  ddfx    := succ(dfx);
  summing := f0[dfx, dif0] + (f0[ddfx, dif0] + f1[ddfx, dif0]
                           - (f0[ddfx, dif2]+f1[ddfx, dif2])/12) * dt/2
end;

procedure CorrectParametre(var prm0 : parametre; olddt, newdt : realT);
var
  ratio, ratio2 : realT;
  df : DifferentialOrder;
begin
  ratio  := newdt/olddt;
  ratio2 := sqr(ratio);
  for df := dfdt to ddfdt2 do begin
    prm0[df, dif1] := prm0[df, dif1] * ratio;
    prm0[df, dif2] := prm0[df, dif2] * ratio2
  end
end;

(* main routines *)

(* main equations of Hodgkin and Huxley *)

function phy(t : realT) : realT;          (* Temperature coefficient *)
begin
  phy := realpower(Q10, (t - stdtemp)/10)
end;

function IonicCurrent(vv:realT;
                        var cnsts : consts;
                        var prms  : parametres):realT;   (* step (7) *)
begin
  with cnsts, prms do
    IonicCurrent := gK*(vv-vK) + gNa*(vv-vNa) + gL*(vv-vL)
end;

(* for complete calculation *)
function dIonicCurrentdt(vv : parametre;
                           var cnsts : consts;
                           var prms  : parametres):realT;
var
  dgKdt, dgNadt, dgNadt1, dgNadt2, didt1, didt2:realT;
begin
  with cnsts, prms do begin
    dgKdt  := 4 * cgK * power(n[ft, dif0], 3) * n[dfdt, dif0];
    dgNadt := cgNa * sqr(m[ft, dif0])
                   * (2 * h[ft, dif0] * m[dfdt, dif0] + m[ft, dif0] * h[dfdt, dif0]);
    didt1  := (gK + gNa + gL) * vv[dfdt, dif0];
    didt2  := dgKdt*(vv[ft, dif0] - vK) + dgNadt * (vv[ft, dif0] - vNa)
  end;
  dIonicCurrentdt := didt1 + didt2
end;

function ddIonicCurrentdt2(var cnsts : consts;
                           var prms  : parametres) : realT;
(* only first order estimation *)
begin
  with cnsts, prms do
    ddIonicCurrentdt2 := (gK + gNa + gL) * v[ddfdt2, dif0]
end;

(* functioins for aplha, and beta *)
function alphan(vv:realT):realT;
begin
  if vv<>-10.0 then alphan := 0.01*(vv+10.0)/(exp((vv+10.0)/10.0)-1.0)
               else alphan := 0.0
end;
function betan(vv:realT):realT;
begin
  betan := 0.125*exp(vv/80.0)
end;

function alpham(vv:realT):realT;
begin
  if vv<>-25.0 then alpham := 0.1*(vv+25.0)/(exp((vv+25.0)/10.0)-1.0)
               else alpham := 0.0
end;
function betam(vv:realT):realT;
begin
  betam := 4.0*exp(vv/18.0)
end;

function alphah(vv:realT):realT;
begin
  alphah:= 0.07*exp(vv/20.0)
end;
function betah(vv:realT):realT;
begin
  betah := 1.0/(exp((vv+30.0)/10.0)+1.0)
end;

(* using loaded table *)
function talphan(vv : realT) : realT;
var v1 : tableV;
begin
  v1 := round(vv * tableVfactor);
  if (v1>=tableVmin) and (v1<=tableVmax) then talphan := alphaTablen^[v1]
                                         else talphan := alphan(vv)
end;
function tbetan(vv : realT) : realT;
var v1 : tableV;
begin
  v1 := round(vv * tableVfactor);
  if (v1>=tableVmin) and (v1<=tableVmax) then tbetan := betaTablen^[v1]
                                         else tbetan := betan(vv)
end;

function talpham(vv : realT) : realT;
var v1 : tableV;
begin
  v1 := round(vv * tableVfactor);
  if (v1>=tableVmin) and (v1<=tableVmax) then talpham := alphaTablem^[v1]
                                         else talpham := alpham(vv)
end;
function tbetam(vv : realT) : realT;
var v1 : tableV;
begin
  v1 := round(vv * tableVfactor);
  if (v1>=tableVmin) and (v1<=tableVmax) then tbetam := betaTablem^[v1]
                                         else tbetam := betam(vv)
end;

function talphah(vv : realT) : realT;
var v1 : tableV;
begin
  v1 := round(vv * tableVfactor);
  if (v1>=tableVmin) and (v1<=tableVmax) then talphah := alphaTableh^[v1]
                                         else talphah := alphah(vv)
end;
function tbetah(vv : realT) : realT;
var v1 : tableV;
begin
  v1:=round(vv * tableVfactor);
  if (v1>=tableVmin) and (v1<=tableVmax) then tbetah := betaTableh^[v1]
                                         else tbetah := betah(vv)
end;

function dndt(f, v1:realT;
                var cnsts:consts; var prms:parametres):realT;  (* step (3) *)
begin
  with cnsts do
    dndt := codndt * phyval * (falphan(v1) * (1-f) - fbetan(v1) * f)
end;

function dmdt(f, v1:realT;
                var cnsts:consts; var prms:parametres):realT;  (* step (3) *)
begin
  with cnsts do
    dmdt := codmdt * phyval * (falpham(v1) * (1-f) - fbetam(v1) * f)
end;

function dhdt(f, v1:realT;
                var cnsts:consts; var prms:parametres):realT;  (* step (3) *)
begin
  with cnsts do
    dhdt := codhdt * phyval * (falphah(v1) * (1-f) - fbetah(v1) * f)
end;

function dvdt(vv, Imemb:realT;
                var cnsts:consts; var prms:parametres):realT;  (* step (3) *)
begin
  with cnsts do
    dvdt := (Imemb-IonicCurrent(vv, cnsts, prms)) / CM
end;

constructor membrane.init;
begin
end;

procedure membrane.setup(initT, initV:realT;
                        initCnst : consts;
                        initMode : PropMode);
var
  g0,                   (* initial total conductance *)
  v0, mu, dddxdt3, ddddxdt4:realT;

  procedure initparametre(var f:parametre);
  var
    df :DifferentialOrder;
    dif:DifferenceOrder;
  begin
    for df := ft to ddfdt2 do
      for dif := dif0 to dif2 do f[df, dif] := 0
  end;

  procedure setvalue(var f:parametre;
                           v1 : realT;   (* Initial voltage *)
                           falpha, fbeta : funct1;
                           cop : realT);
  var
    alpha, beta, alphabeta : realT;     (* H-HSOL04 *)
  begin
    initparametre(f);
    alpha := falpha(v1);
    beta  := fbeta(v1);
    with constants do begin
      phyval := phy(temp);
      alpha  := alpha * phyval * cop;
      beta   := beta  * phyval * cop
    end;
    alphabeta := alpha + beta;
    f[ft    , dif0] := alpha / alphabeta;
    f[dfdt  , dif0] := (1 - f[ft, dif0]) * alpha - beta * f[ft, dif0];
    f[ddfdt2, dif0] := -alphabeta * f[dfdt, dif0];
    dddxdt3 := -alphabeta * f[ddfdt2, dif0];
    with CurrentParam do begin
      f[ft,   dif1] := f[dfdt,   dif0] * dtNorm;
      f[dfdt, dif1] := f[ddfdt2, dif0] * dtNorm;
      f[dfdt, dif2] := dddxdt3 * dtNorm
    end
  end; { of set_value }

begin { init }
  constants := initcnst;
  Vclamped  := false;
  Iclamped  := false;
  with CurrentParam, constants do begin
    initparametre(v);
    v[ft, dif0] := initV;
    t := initT;
    Iadditional:= 0;
    setvalue(n, 0, falphan, fbetan, codndt);
    setvalue(m, 0, falpham, fbetam, codmdt);
    setvalue(h, 0, falphah, fbetah, codhdt);
    gK := cgK  * power(n[ft, dif0], npower);
    gNa:= cgNa * power(m[ft, dif0], mpower)
               * power(h[ft, dif0], hpower); (* initial conductance *)
    dt := setdt;
{    settings.mode := initmode;
    if settings.mode = prop then begin
      g0 := gK + gNa + gL;                   (* resting conductance *)
      mu := (K + sqrt(K*K + 4*K*g0/CM)) / 2; (* solution of eq. 32  *)
      V0 := v[ft, dif0];
      risingphase := true;
      v[dfdt  ,dif0] := mu * v[ft, dif0];
      v[ddfdt2,dif0] := K * (v[dfdt, dif0] + g0 / CM * v[ft, dif0]);
      dddxdt3        := K * (v[ddfdt2, dif0] + g0 / CM * v[dfdt, dif0]);
      ddddxdt4       := K * (dddxdt3 + g0 / CM * v[ddfdt2, dif0]);
      v[ft  , dif1]  := v[dfdt  , dif0] * dt;
      v[dfdt, dif1]  := v[ddfdt2, dif0] * dt;
      v[dfdt, dif2]  := dddxdt3 * dt;
      v[ddfdt2, dif1]:= v[dfdt, dif2];
      v[ddfdt2, dif2]:= ddddxdt4 * dt
    end else }begin
      v[dfdt  , dif0]:=
         -IonicCurrent(v[ft, dif0], constants, CurrentParam)/CM;
      v[ddfdt2, dif0]:=
         -dIonicCurrentdt(v, constants, CurrentParam)/CM;
      v[ft,     dif1]:= v[dfdt,   dif0] * dt;
      v[dfdt,   dif1]:= v[ddfdt2, dif0] * dt
    end
  end
end;

function  membrane.setdt : realT;
   (* set appropriate dt for every time_course variable *)
const
  highVoltage = 100.0;
  lowVoltage  = 20.0;
  lowdvdt     = 10.0;
var
  vv, dv:realT;
begin
  with CurrentParam do begin
    vv := abs(v[ft, dif0]);
    dv := abs(v[dfdt, dif0])
  end;
  if       vv > highVoltage  then setdt := dtslow
  else if (vv < lowVoltage) and
          (dv < lowdvdt)     then setdt := dtfast
  else                            setdt := dtnorm
end;

procedure membrane.calculate;
(* calculate new values of every parametre, or initiate them *)
var
  vestim : realT;               (* the first extimation of v *)
  dt0    : realT;

  procedure ErrorWrite(ss:sstring);
  var s : string;
  begin
    s := 'Error in calculation of ' + ss + '.';
    WinPostMsg(hwndParamArea, WM_ShowAnyString, MPARAM(@s), 0)
  end;

  procedure solve(var f : parametre;
                      dpdt : funct4;
                      v1:realT;     (* In dn_dt etc.  - estimated v
                                       In dv_dt       - Imemb *)
                      ch:char);
(* essentially same for H-HSOL02. 6-Nov. 90 *)
  const
    errorlimit = 0.001;
    calclimit  = 1000;
  var
    alpha, beta : realT;
    estimf:parametre;
    ff : realT;
    i  : integer;
    ss : sstring;
  begin
    i := 0;
    ss:= ch;
    estimf := f;
    estimation(f[ft], estimf[ft], dif0);                (* step (2) *)
    repeat
      estimf[dfdt, dif0] :=                             (* step (3) *)
        dpdt(estimf[ft, dif0], v1, constants, CurrentParam);
      difference(f[dfdt], estimf[dfdt], dif1);
      difference(f[dfdt], estimf[dfdt], dif2);
      ff := estimf[ft, dif0];
      estimf[ft, dif0] := summing(f, estimf, ft, CurrentParam.dt);
      inc(i)                                            (* step (4) *)
    until errorOk(estimf[ft, dif0], ff, errorlimit) or (i > calclimit);
    if i > calclimit then errorwrite(ss);
    with CurrentParam do begin
      difference(f[ft]  , estimf[ft]  , dif1);
      difference(f[dfdt], estimf[dfdt], dif1);
      difference(f[dfdt], estimf[dfdt], dif2)
    end;
    f[ft, dif0] := estimf[ft, dif0];
    f[ft, dif1] := estimf[ft, dif1];
    f[dfdt]     := estimf[dfdt]
  end;

  procedure CorrectParams(olddt:realT);
  begin
    with CurrentParam do begin
      CorrectParametre(v, olddt, dt);
      CorrectParametre(n, olddt, dt);
      CorrectParametre(m, olddt, dt);
      CorrectParametre(h, olddt, dt)
    end
  end;

begin (* of calculate *)
  with CurrentParam, constants do begin
    t := t + dt;
{   vestim := limitedreal(v.x + v.dif_x * dt, vNa, vK);(* step (1) *)}
    if Vclamped then vestim := ClampVal
                else vestim := v[ft, dif0] + v[ft, dif1];
    solve(n, @dndt, vestim, 'n');          (* step (2)-(5) *)
    solve(m, @dmdt, vestim, 'm');          (* step (6) *)
    solve(h, @dhdt, vestim, 'h');          (* step (6) *)
    gK := cgK * power(n[ft, dif0], npower);       (* step (7) *)
    gNa:= cgNa* power(m[ft, dif0], mpower)
              * power(h[ft, dif0], hpower);       (* step (7) *)
{    if settings.mode = action then}
      if Vclamped then v[ft, dif0] := ClampVal
                  else solve(v, @dvdt, Iadditional/(gK+gNa+gL), 'v');
                                                  (* step (8)-(9) *)
    dt0:= dt;
    dt := setdt;
    if dt<>dt0 then CorrectParams(dt0);
    Iadditional := 0
  end
end;                            (* of calculate *)

procedure membrane.stimulationV(stimV : realT);
begin
  with CurrentParam, constants do begin
    v[ft,     dif0] := stimv;
    v[dfdt,   dif0] := -IonicCurrent(v[ft, dif0], constants, CurrentParam)/CM;
    v[ddfdt2, dif0] := -dIonicCurrentdt(v, constants, CurrentParam)/CM;
    v[dfdt,   dif1] := v[ddfdt2, dif0] * dt;
    v[dfdt,   dif2] := -ddIonicCurrentdt2(constants, CurrentParam)/CM
  end
end;

procedure membrane.setVclamp(stimV : realT);
begin
  Vclamped := true;
  ClampVal := stimV
end;

procedure membrane.resetVclamp;
begin
  Vclamped := false
end;

procedure membrane.getval(var retTm : tm);
begin
  with CurrentParam, retTm do begin
    tt  := t;
    tdt := dt;
    tv  := v[ft, dif0];
    tgK := gK;
    tgNa:= gNa;
    tn  := n[ft, dif0];
    tm  := m[ft, dif0];
    th  := h[ft, dif0]
  end;
  with constants, retTm do
    ti  := CurrentParam.gK  * (tv - vK) +
           CurrentParam.gNa * (tv - vNa) +
                        gL  * (tv - vL)
end;

procedure membrane.setI(initI : realT);
begin
  CurrentParam.Iadditional := initI
end;

procedure membrane.getconst(var retcgK, retcgNa:realT);
begin
  with constants do begin
    retcgK := cgK;
    retcgNa:= cgNa
  end
end;

procedure membrane.setconsts(retCnst : consts);
begin
  constants := retcnst
end;

procedure membrane.getconsts(var retCnst : consts);
begin
  retcnst := constants
end;

procedure membrane.getparams(var retPrms : parametres);
begin
  retprms := CurrentParam
end;

destructor membrane.done;
begin
end;

(* of class membrane *)

procedure VoltageStimulation(t, dt:realT);
var s : string;
begin
  with stimV do if count<=num then begin
    t := t - stim[count].time;
    if (t >= 0 ) and ( t < dt ) then begin
      with stim[count] do begin
        str(stim[count].voltage:6:2, s);
        s := 'Valtage'+ s + ' mV';
        WinPostMsg(hwndParamArea, WM_ShowAnyString, MPARAM(@s), 0);
        memb.stimulationV(voltage)
      end;
      inc(count)
    end
  end
end;    { VoltageStimulation }

procedure CurrentStimulation(t, v:realT);
var s, s1, s2 : string;
begin
  with stimI do if count<=num then begin
    if not switch and (stim[count].time0 <= t ) then begin
      preV  := v;
      switch:= true
    end;
  if switch then
    if (t < stim[count].time1) then
      with stim[count] do begin
        str(stimval:6:2, s);
        str(preV:6:2, s1);
        str(v:6:2, s2);
        s := 'Current'+ s + ' microA' + s1 + s2 + ' mV';
        WinPostMsg(hwndParamArea, WM_ShowAnyString, MPARAM(@s), 0);
        memb.setI(stimval)
      end
    else begin
      switch := false;
      inc(count)
    end
  end
end;    { currnt_stimulation }

procedure VoltageClamp(t : realT);
begin
  with clmpV do if count<=num then
    if not switch then begin
      if stim[count].time0 <= t then begin
        memb.setVclamp(stim[count].stimval);
        switch:= true
      end
    end else
      if t >= stim[count].time1 then begin
        memb.resetVclamp;
        switch := false;
        inc(count)
      end
end;    { currnt_stimulation }

procedure InitClampV(var st:Vclamp);
var i : integer;
begin
  with st do begin
    for i := 0 to MaxStim do
      with stim[i] do begin
        time0 := 0;
        time1 := 0;
        stimval := 0
      end;
    switch:= false;
    count := 0;
    num   := 0
  end
end;    { InitStimI }

procedure InitStimI(var st:Istim);
const
  stdamp = -30;
var i : integer;
begin
  with st do begin
    for i := 0 to MaxStim do
      with stim[i] do begin
        time0 := 0;
        time1 := 0;
        stimval := stdamp
      end;
    switch:= false;
    count := 0;
    num   := 0
  end
end;    { InitStimI }

procedure InitStimV(var st:Vstim);
var i:integer;
begin
   with st do begin
     cb := sizeof(st);
     for i := 1 to MaxStim do
       with stim[i] do begin
         time := 0;
         voltage := 0
       end;
     count := 0;
     num := 0
   end
end;    { InitStimV }

procedure SetParamDefault(var c : consts);
begin
  with c do begin
    vK   := stdvK;
    vNa  := stdvNa;
    vL   := stdvL;
    cgK  := stdcgK;
    cgNa := stdcgNa;
    gL   := stdgL;
    CM   := stdCM;
    K    := stdK;
    temp := stdtemp;
    npower := stdnpower;
    mpower := stdmpower;
    hpower := stdhpower;
    codndt := stdcodndt;
    codmdt := stdcodmdt;
    codhdt := stdcodhdt
  end
end;  { of SetParamNormal }

procedure SetUpFunctionVars(table : boolean);
var
  f : file of ParamTable;
begin
  if table then begin           (* Set to use tables *)
    assign(f, TableName);
    reset(f);
    new(alphaTablen);
    read(f, alphaTablen^);
    new(alphaTablem);
    read(f, alphaTablem^);
    new(alphaTableh);
    read(f, alphaTableh^);
    new(betaTablen);
    read(f, betaTablen^);
    new(betaTablem);
    read(f, betaTablem^);
    new(betaTableh);
    read(f, betaTableh^);
    close(f);
    falphan := @talphan;
    falpham := @talpham;
    falphah := @talphah;
    fbetan  := @tbetan;
    fbetam  := @tbetam;
    fbetah  := @tbetah
  end else begin                 (* Set not to use tables *)
    if alphaTablen <> nil then begin
      dispose(alphaTablen);
      alphaTablen := nil;
      dispose(alphaTablem);
      alphaTablem := nil;
      dispose(alphaTableh);
      alphaTableh := nil;
      dispose(betaTablen);
      betaTablen  := nil;
      dispose(betaTablem);
      betaTablem  := nil;
      dispose(betaTableh);
      betaTableh  := nil
    end;
    falphan := @alphan;
    falpham := @alpham;
    falphah := @alphah;
    fbetan  := @betan;
    fbetam  := @betam;
    fbetah  := @betah
  end
end;

procedure InitProgManual;
(* Initialise for calucultion *)
var v0 : realT;
begin
  alphaTablen := nil;
  alphaTablem := nil;
  alphaTableh := nil;
  betaTablen  := nil;
  betaTablem  := nil;
  betaTableh  := nil;
  with settings do begin
    cb     := sizeof(settings);            (* PM usage *)
    TLimit := 10;
    InitV  := 0;
    mode   := action;
    SetParamDefault(MembParam);
    UseTable := false;
    PortAnimation := true;
    UseDosSleep   := true;
    NaChannel.init;
    KChannel.init;
    SetUpFunctionVars(UseTable);
    memb.init;
    memb.setup(0, InitV, MembParam, mode);
    memb.getval(TimeCourse[0])
  end;
  InitStimV(StimV);
  InitStimI(StimI);
  InitClampV(clmpV);
  TmNum     := 0;
  DrawCursor:= false;
  settings.PositionLoaded := false;
  settings.SettingFile    := defSettingFile;
  fillchar(XtoTm, sizeof(XtoTm), 0)
end;    { InitProgManual }

procedure InitPMVar;
begin
  ctlData := FCF_TITLEBAR + FCF_SYSMENU + FCF_MENU +
             FCF_ACCELTABLE + FCF_SIZEBORDER + FCF_MINMAX +
             FCF_TASKLIST
end;

procedure LoadSettings(h : HWND; fn : string);
var t : text;
    s : consts;
    f1, f2, f3 : integer;
    psx, psy, sizx, sizy : LONG;
begin
  if fn<>'' then begin
    assign(t, fn);
    reset(t);
    if IOResult<>0 then begin
      if h<>NULLHANDLE then
        WinMessageBox(HWND_DESKTOP, h,
                      'File Not Found', '', MB_OK, 0)
    end else begin
      with s do begin
        read  (t, vK, vNa, vL);
        read  (t, cgK, cgNa, gL);
        readln(t, CM, K, temp);
        read  (t, npower, mpower, hpower);
        readln(t, codndt, codmdt, codhdt);
      end;
      readln(t, f1, f2, f3);
      readln(t, psx,  psy, sizx, sizy);
      if IOResult<>255 then begin
        with settings do begin
          MembParam     := s;
          SettingFile   := fn;
          UseTable      := f1 <> 0;
          PortAnimation := f2 <> 0;
          UseDosSleep   := f3 <> 0;
          PositionLoaded:= true;
          posx          := psx;
          posy          := psy;
          sizex         := sizx;
          sizey         := sizy
        end;
        if h<>NULLHANDLE then
          WinSetWindowPos(WinQueryWindow(h, QW_PARENT), NULLHANDLE,
                          psx, psy, sizx, sizy, SWP_SIZE + SWP_MOVE)
      end;
      close(t)
    end
  end
end;

procedure SaveSettings(h : HWND; fn : string);
var t : text;
    sw: SWP;
begin
  if fn<>'' then begin
    assign(t, fn);
    rewrite(t);
    with settings, MembParam do begin
      write  (t, vK, ' ', vNa, ' ', vL);
      write  (t, cgK, ' ', cgNa, ' ', gL);
      writeln(t, CM, ' ', K, temp);
      write  (t, npower, ' ', mpower, ' ', hpower);
      writeln(t, codndt, ' ', codmdt, ' ', codhdt);
      writeln(t, ord(UseTable), ' ', ord(PortAnimation), ' ', ord(UseDosSleep))
    end;
    WinQueryWindowPos(WinQueryWindow(h, QW_PARENT), sw);
    with sw do writeln(t, x, ' ', y, ' ', cx, ' ', cy);
    close(t)
  end
end;

procedure InitialisePS(h : HWND; var hp : HPS);
var s  : SIZEL;
    hd : HDC;
begin
  hd := WinOpenWindowDC(h);
  s.cx := 0;
  s.cy := 0;
  hp := GpiCreatePS(myhab, hd, s,
                    PU_PELS + GPIF_DEFAULT + GPIT_MICRO + GPIA_ASSOC)
                             (* creating a micro-PS *)
end;

procedure GetFontData(h : HWND; var AveChar, CharHeight : word);
var fm : FontMetrics;
    hp : HPS;
begin
  hp := WinGetPS(h);
  GpiQueryFontMetrics(hp, sizeof(fm), fm);
  with fm do begin
    AveChar    := lAveCharWidth;
    CharHeight := lEmHeight
  end;
  WinReleasePS(hp)
end;

function RealToFIXED(r : realT) : FIXED;
begin
  RealToFIXED := round(r * $10000)
end;

procedure initDisplay;
(* initialise graphic screen *)
const
  yMax = 140;
  yofs =  20;
  CurrentMagn = 0.2;
  nmhMagnY    = -100;
var ttm : tm;
    cgK, cgNa : realT;
    cx, cy : integer;

  procedure SetViewParam(ofsx, ofsy : integer);
  begin
    fillchar(ViewMat, sizeof(ViewMat), 0);
    with ViewMat do begin
      fxM11  := MAKELONG(0, 1);
      fxM22  := fxM11;
      lM31   := ofsx;
      lM32   := ofsy;
      lM33   := 1
    end;
    GpiSetDefaultViewMatrix(calchps, 9, ViewMat, TRANSFORM_REPLACE)
  end;

  procedure drawform;      (* draw scales, axes *)
  var ptl, ptl2 : POINTL;
      i : integer;
      s : string;

    procedure writestr(x, y : integer; s : string);
    var ptl : POINTL;
    begin
      ptl.x := x;
      ptl.y := round(y * yMagn) - CharHeight div 2;
      GpiCharStringAt(calchps, ptl, length(s), s[1])
    end;

    procedure writestrclr(s : string; col : ULONG);
    begin
      GpiSetColor(calchps, col);
      GpiCharString(calchps, length(s), s[1])
    end;

  begin
    str(settings.TLimit:4:1, s);
    GpiSetColor(calchps, CLR_WHITE);
    writestr(-20, 120, 'mV');
    writestr(-40, 100, '-100');
    writestr(-20, 0,   '0');
    writestr(-60, 120, 'mmho');
    writestr(-60, 100, '50');
    writestr(-20, -120,'nmh');
    writestr(-20, -100,'  1');
    writestr(cxClient - 40 - xofs, -10, s);
    with ptl do begin
      x := 0;
      y := round(100 * yMagn);
      ptl2 := ptl;
      ptl2.y := -y;
      GpiMove(calchps, ptl);
      GpiLine(calchps, ptl2);
      y := 0;
      x := 0;
      ptl2 := ptl;
      ptl2.x := round(settings.TLimit * xMagn);
      GpiMove(calchps, ptl);
      GpiLine(calchps, ptl2);
{      if settings.mode=prop then
         for i:=0 to 100 do spot_dot(i/5, current_displacement*current_magn,
                                         white);}
      x := cxClient - 270;
      y := cyClient div 2 - 30;
      GpiMove(calchps, ptl);
      writestrclr('Traces:', CLR_WHITE);
      writestrclr('V ', CLR_YELLOW);
      writestrclr('gK ', CLR_GREEN);
      writestrclr('gNa ', CLR_RED);
      writestrclr('Ii ', CLR_CYAN);
      writestrclr('n ', CLR_BLUE);
      writestrclr('m ', CLR_PINK);
      writestrclr('h' , CLR_WHITE)
    end;
    GpiSetColor(calchps, CLR_RED);
    with StimV do
      for i:=1 to num do
        writestr(round(stim[i].time * xMagn) - AveChar div 2,  100, 'v');
    GpiSetColor(calchps, CLR_PINK);
    with StimI do
      for i:=1 to num do
        writestr(round(stim[i].time0 * xMagn) - AveChar div 2, 100, 'i')
{     if settings.mode=action then writeln('Blue:n, Magenta:m, White:h')
                             else writeln('Blue:Im, Magenta:Ic, White:Ii');
}
  end; { initDisplay.drawform }

begin  { InitDisplay }
  cx    := cxClient - xofs - Rmargin;
  cy    := (cyClient - yofs) div 2;
  xMagn := cx / settings.TLimit;
  yMagn := cy / yMax;
  xMagnFIXED   := RealToFIXED(xMagn) div IntMagn;
  vMagnFIXED   := RealToFIXED(yMagn) div IntMagn;
  gMagnFIXED   := vMagnFIXED;
  IMagnFIXED   := vMagnFIXED;
  nmhMagnFIXED := RealToFIXED(yMagn * nmhMagnY) div nmhMagn;
  SetViewParam(xofs, cy + yOfs);
  drawform;
  if settings.PortAnimation then begin
    memb.getval(ttm);
    memb.getconst(cgK, cgNa);
    with ttm do begin
      SetHPS(calchps);
      NaChannel.setup(cx-210, 30-cy, 24, round(cgNa/2), 30,
                      tm, th, CLR_PINK, CLR_WHITE);
      KChannel.setup(cx-50, 30-cy, 24, round(cgK/2), 30,
                     tn, CLR_BLUE)
    end
  end
end;   { InitDisplay }

procedure SetYMagn(magny : FIXED);
begin
  with ViewMat do begin
    fxM11  := xMagnFIXED;
    fxM22  := magny
  end;
  GpiSetDefaultViewMatrix(calchps, 9, ViewMat, TRANSFORM_REPLACE)
end;

procedure ResetXYMagn;
begin
  with ViewMat do begin
    fxM11  := MAKELONG(0, 1);
    fxM22  := fxM11
  end;
  GpiSetDefaultViewMatrix(calchps, 9, ViewMat, TRANSFORM_REPLACE)
end;

function  SetDispGrid(maxT : realT) : realT;
begin
  SetDispGrid := maxT / DispNum
end;

procedure InitCalc(var ttm : tm);
var cgK, cgNa : realT;
begin { InitCalc }
  with settings do memb.setup(0, InitV, MembParam, mode);
  memb.getval(ttm);
  memb.getconst(cgK, cgNa);
  stimV.count   := 1;
  stimI.count   := 1;
  stimI.switch  := false;
  clmpV.count   := 1;
  clmpV.switch  := false;
  TmNum         := 0;
  CursorPos     := 0
end; { InitCalc }

(* Threads *)

function running(idSem : HEV) : boolean;
(* checking whether is event semaphore idSem posted.  Threads posts the
   corresponding event semaphore at the beginning of run and reset it at
   the end. *)
var l : ULONG;
begin
  DosQueryEventSem(idSem, l);
  running := (l <> 0)
end;

procedure XorDrawCursor(x : integer);
var p : POINTL;
begin
  ResetXYMagn;
  GpiSetColor(calchps, CLR_GREEN);
  GpiSetMix(calchps, BM_XOR);
  p.x := x;
  p.y := round(120 * yMagn);
  GpiMove(calchps, p);
  p.y := -p.y;
  GpiLine(calchps, p);
  GpiSetMix(calchps, BM_DEFAULT)
end;

procedure EraseDrawCursor(x : integer);
var p : POINTL;
begin
  ResetXYMagn;
  GpiSetColor(calchps, CLR_BLACK);
  p.x := x;
  p.y := round(120 * yMagn);
  GpiMove(calchps, p);
  p.y := -p.y;
  GpiLine(calchps, p)
end;

procedure DisplayTraces(t0, t1 : integer);
var dx : integer;

  procedure PolyLineWithColour(var D : DispArray; c : ULONG);
  begin
    GpiSetColor(calchps, c);
    GpiMove(calchps, D[t0]);
    GpiPolyLine(calchps, dx, D[succ(t0)])
  end;

begin
  dx := t1 - t0;
  SetYMagn(gMagnFIXED);
  PolyLineWithColour(gKTime, CLR_GREEN);
  PolyLineWithColour(gNaTime, CLR_RED);
  SetYMagn(nmhMagnFIXED);
  PolyLineWithColour(nTime, CLR_BLUE);
  PolyLineWithColour(mTime, CLR_PINK);
  PolyLineWithColour(hTime, CLR_WHITE);
  SetYMagn(IMagnFIXED);
  PolyLineWithColour(ITime, CLR_CYAN);
  SetYMagn(vMagnFIXED);
  PolyLineWithColour(vTime, CLR_YELLOW)
end;

procedure DrawTimeCourse;
var ttm : tm;
    s   : ParaStr;
begin
  if DrawCursor then EraseDrawCursor(CursorPos);
  initDisplay;
  s := 'Updating Screen. Wait.';
  WinSendMsg(hwndParamArea, WM_ShowAnyString, MPARAM(@s), 0);
  Resized := false;
  if TmNum > 0 then DisplayTraces(0, pred(TmNum));
  ttm := TimeCourse[TmNum];
  if settings.PortAnimation then
    with ttm do begin
      ResetXYMagn;
      NaChannel.draw(tm, th);
      KChannel.draw(tn)
    end;
  WinSendMsg(hwndParamArea, WM_ShowTmRecord, MPARAM(@ttm), 0);
  if DrawCursor then XorDrawCursor(CursorPos)
end;

procedure CalcMain; CDECL;
(* Main calculation and graphic display thread *)
var ttm0, ttm1           : tm;
    tm0, tm1             : integer;
    StoreGrid, StoreTime : realT;
    DispCount, DispPortCount,
    DispTime, DispPortTime : integer;
    dummy     : ULONG;

  procedure SetDispPosToTm(var ttm : tm);
  const
    CurrentDisplacement = 400;
  {  nmhMagn     = -100;}
    CurrentMagn = 0.2;
  var ymg : single;
  begin
    with ttm do begin
      xt   := round(tt * IntMagn);
      yv   := round(tv * -IntMagn);
      ygK  := round(tgK  * IntMagn);
      ygNa := round(tgNa * IntMagn);
      yI   := round(ti * CurrentMagn * IntMagn);
      yn   := round(tn * nmhMagn);
      ym   := round(tm * nmhMagn);
      yh   := round(th * nmhMagn)
    end
  end;  { CalcMain.SetDispPosToTm }

  procedure CopyTmToDisp(var t : tm; i : integer);
  begin
    with t do begin
      with vTime[i]   do begin x := xt; y := yv   end;
      with gKTime[i]  do begin x := xt; y := ygK  end;
      with gNaTime[i] do begin x := xt; y := ygNa end;
      with ITime[i]   do begin x := xt; y := yI   end;
      with nTime[i]   do begin x := xt; y := yn   end;
      with mTime[i]   do begin x := xt; y := ym   end;
      with hTime[i]   do begin x := xt; y := yh   end
    end
  end;

begin
  DosPostEventSem(calculating);
 (* This flag is used to prevent creation of another instance of this
    procedure *)
  InitCalc(ttm1);
  ToCalc        := true;
  StoreGrid     := SetDispGrid(settings.TLimit);
  StoreTime     := StoreGrid;
  DispTime      := DispNum div cxClient;
  if DispTime <1 then DispTime := 1;
  DispCount     := DispTime * DispPortSkip;
  DispPortCount := DispPortSkip;
  SetDispPosToTm(ttm1);
  CopyTmToDisp(ttm1, TmNum);
  ttm0          := ttm1;
  tm0           := 0;
  with ttm1 do
    while (tt < settings.TLimit) and (tv < maxV) and (tv > -maxV) and
          ToCalc do begin
      memb.calculate;
      memb.getval(ttm1);
      VoltageStimulation(tt, tdt);
      CurrentStimulation(tt, tv);
      VoltageClamp(tt);
      if tt > StoreTime then begin
{        if TmNum < DispNum then begin}
          SetDispPosToTm(ttm1);
          CopyTmToDisp(ttm1, TmNum);
          tm1 := TmNum;
          inc(TmNum);
          TimeCourse[TmNum] := ttm1
{        end};
        StoreTime := StoreTime + StoreGrid;
        if DispCount = 0 then begin
          if settings.UseDosSleep then DosSleep(1);
          DosRequestMutexSem(DrawToMain, SEM_INDEFINITE_WAIT);
          DisplayTraces(tm0, tm1);
          if settings.PortAnimation then begin
            ResetXYMagn;
            NaChannel.draw(tm, th);
            KChannel.draw(tn);
            DispPortCount := DispPortSkip
          end;
          DosReleaseMutexSem(DrawToMain);
          DispCount := DispTime;
          ttm0 := ttm1;
          tm0  := tm1;
          WinPostMsg(hwndParamArea, WM_ShowTmRecord, MPARAM(@ttm1), 0)
        end else
          dec(DispCount)
      end
    end;
  DosResetEventSem(calculating, dummy); (* enables next StartThread call *)
  WinPostMsg(hwndCalcArea, WM_TraceCalculated, 0, 0);
  DosExit(EXIT_THREAD, 0)    (* Destructs this thread *)
end;    { CalcMain }

procedure NumErrMsg(h : HWND);
begin
  WinMessageBox(HWND_DESKTOP, h, 'It is not a number.', '', 0, MB_OK)
end;

function NumDlgBoxProc  ( h : HWND;
                          w : ULONG;
                          m1: MPARAM;
                          m2: MPARAM) : MRESULT; CDECL;
(* ask a number with favourite title.
   This dialog box should be called
   with the last parametre of WinDlgBox of pNumStrDlgDt.

   NumStrDlgDt:
     cb, - record size
     st, - string for asked number
     title - string for title
*)

var pm : pNumStrDlgDt;
    dummy: real;
    i  : integer;
    s  : cstring[20];

begin
  case w of
    WM_INITDLG : begin
        pm := pNumStrDlgDt(m2);
        WinSetWindowPtr(h, QWL_USER, pm);
        WinSetDlgItemText(h, CM_FIRST + IDD_NUMSTRTITLE, pm^.title);
        WinSetDlgItemText(h, CM_FIRST + IDD_NUMSTRENTRY, pm^.st);
        NumDlgBoxProc := 0
      end;
    WM_COMMAND : begin
        case COMMANDMSG(@w)^.cmd - CM_FIRST of
          CM_OK     : begin
              pm := WinQueryWindowPtr(h, QWL_USER);
              i  := WinQueryDlgItemText(h, CM_FIRST + IDD_NUMSTRENTRY,
                                        ParamStrLen, s);
              pm^.st := s;
              val(pm^.st, dummy, i);
              if i = 0 then WinDismissDlg(h, 1)
                       else NumErrMsg(h)
            end;
          CM_CANCEL : WinDismissDlg(h, 0)
        else
        end;
        NumDlgBoxProc := 0
      end
  else
    NumDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
  end
end;

procedure SetNextDlgItem(h : HWND; r : realT; var id : integer);
var s  : string[20];
begin
  inc(id);
  str(r:4:2, s);
  WinSetDlgItemText(h, id, s)
end;

procedure GetNextDlgItem(h : HWND; var r : realT; var id, ioerr : integer);
var i : integer;
    s : cstring[20];
    st: string;
begin
  inc(id);
  i := WinQueryDlgItemText(h, id, 20, s);
  if i<>0 then begin
    st := s;
    val(st, r, i);
    inc(ioerr, i)
  end else
    inc(ioerr)
end;

function CondDlgBoxProc ( h : HWND;
                          w : ULONG;
                          m1: MPARAM;
                          m2: MPARAM) : MRESULT; CDECL;
(* ask a number with favourite title.
   This dialog box should be called
   with the last parametre of WinDlgBox of pNumStrDlgDt.

   NumStrDlgDt:
     cb, - record size
     st, - string for asked number
     title - string for title
*)

var pm : ppref;
    id  : integer;
    ioerr : integer;
begin
  case w of
    WM_INITDLG : begin
        pm := ppref(m2);
        WinSetWindowPtr(h, QWL_USER, pm);
        id := CM_FIRST + IDD_CONDUCTANCEDLG;
        with pm^.MembParam do begin
          SetNextDlgItem(h, cgK, id);
          SetNextDlgItem(h, cgNa, id);
          SetNextDlgItem(h, gL, id);
          SetNextDlgItem(h, vK, id);
          SetNextDlgItem(h, vNa, id);
          SetNextDlgItem(h, vL, id)
        end;
        CondDlgBoxProc := 0
      end;
    WM_COMMAND : begin
        case COMMANDMSG(@w)^.cmd - CM_FIRST of
          CM_OK     : begin
              pm := WinQueryWindowPtr(h, QWL_USER);
              id := CM_FIRST + IDD_CONDUCTANCEDLG;
              ioerr := 0;
              with pm^.MembParam do begin
                GetNextDlgItem(h, cgK, id, ioerr);
                GetNextDlgItem(h, cgNa, id, ioerr);
                GetNextDlgItem(h, gL, id, ioerr);
                GetNextDlgItem(h, vK, id, ioerr);
                GetNextDlgItem(h, vNa, id, ioerr);
                GetNextDlgItem(h, vL, id, ioerr)
              end;
              if ioerr = 0 then WinDismissDlg(h, 1)
                           else NumErrMsg(h)
            end;
          CM_CANCEL : WinDismissDlg(h, 0)
        else
        end;
        CondDlgBoxProc := 0
      end
  else
    CondDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
  end
end;

function NMHDlgBoxProc ( h : HWND;
                         w : ULONG;
                         m1: MPARAM;
                         m2: MPARAM) : MRESULT; CDECL;
(* ask a number with favourite title.
   This dialog box should be called
   with the last parametre of WinDlgBox of pNumStrDlgDt.

   NumStrDlgDt:
     cb, - record size
     st, - string for asked number
     title - string for title
*)

var pm : ppref;
    id  : integer;
    ioerr : integer;

  procedure ssss(var i : pwr);
  var rP : realT;
  begin
    rP := i;
    GetNextDlgItem(h, rP, id, ioerr);
    i := trunc(rP)
  end;

begin
  case w of
    WM_INITDLG : begin
        pm := ppref(m2);
        WinSetWindowPtr(h, QWL_USER, pm);
        id := CM_FIRST + IDD_NMHDLG;
        with pm^.MembParam do begin
          SetNextDlgItem(h, Codndt, id);
          SetNextDlgItem(h, Codmdt, id);
          SetNextDlgItem(h, Codhdt, id);
          SetNextDlgItem(h, nPower, id);
          SetNextDlgItem(h, mPower, id);
          SetNextDlgItem(h, hPower, id)
        end;
        NMHDlgBoxProc := 0
      end;
    WM_COMMAND : begin
        case COMMANDMSG(@w)^.cmd - CM_FIRST of
          CM_OK     : begin
              pm := WinQueryWindowPtr(h, QWL_USER);
              id := CM_FIRST + IDD_NMHDLG;
              ioerr := 0;
              with pm^.MembParam do begin
                GetNextDlgItem(h, Codndt, id, ioerr);
                GetNextDlgItem(h, Codmdt, id, ioerr);
                GetNextDlgItem(h, Codhdt, id, ioerr);
                ssss(nPower);
                ssss(mPower);
                ssss(hPower)
              end;
              if ioerr = 0 then
                WinDismissDlg(h, 1)
              else
                NumErrMsg(h)
            end;
          CM_CANCEL : WinDismissDlg(h, 0)
        else
        end;
        NMHDlgBoxProc := 0
      end
  else
    NMHDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
  end
end;

function MiscDlgBoxProc ( h : HWND;
                          w : ULONG;
                          m1: MPARAM;
                          m2: MPARAM) : MRESULT; CDECL;
(* ask a number with favourite title.
   This dialog box should be called
   with the last parametre of WinDlgBox of pNumStrDlgDt.

   NumStrDlgDt:
     cb, - record size
     st, - string for asked number
     title - string for title
*)

var pm : ppref;
    id  : integer;
    ioerr : integer;
{    s  : cstring[20];

  procedure ss(r : realT; var id : integer);
  var
    s  : cstring[20];
  begin
    inc(id);
    s := ToStr(r:4:2);
    WinSetDlgItemText(h, id, s)
  end;

  procedure sss(var r : realT; var id, ioerr : integer);
  var i : integer;
      s : cstring[20];
      st: string;
  begin
    inc(id);
    s := ToStr(r:4:2);
    i := WinQueryDlgItemText(h, id, 20, s);
    st:= s;
    val(st, r, i);
    if st[1] = '-' then r := -abs(r); (* why!!!!!!!!!!!!!!!! *)
    inc(ioerr, IOResult)
  end;
}
begin
  case w of
    WM_INITDLG : begin
        pm := ppref(m2);
        WinSetWindowPtr(h, QWL_USER, pm);
        id := CM_FIRST + IDD_NMHDLG;
        with pm^.MembParam do begin
          SetNextDlgItem(h, CM, id);
          SetNextDlgItem(h, temp, id);
          SetNextDlgItem(h, K, id)
        end;
        MiscDlgBoxProc := 0
      end;
    WM_COMMAND : begin
        case COMMANDMSG(@w)^.cmd - CM_FIRST of
          CM_OK     : begin
              pm := WinQueryWindowPtr(h, QWL_USER);
              id := CM_FIRST + IDD_NMHDLG;
              ioerr := 0;
              with pm^.MembParam do begin
                GetNextDlgItem(h, CM, id, ioerr);
                GetNextDlgItem(h, temp, id, ioerr);
                GetNextDlgItem(h, K, id, ioerr)
              end;
              if ioerr = 0 then WinDismissDlg(h, 1)
                           else NumErrMsg(h)
            end;
          CM_CANCEL : WinDismissDlg(h, 0)
        else
        end;
        MiscDlgBoxProc := 0
      end
  else
    MiscDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
  end
end;

function VStimDlgBoxProc (h : HWND;
                          w : ULONG;
                          m1: MPARAM;
                          m2: MPARAM) : MRESULT; CDECL;
(* ask a number with favourite title.
   This dialog box should be called
   with the last parametre of WinDlgBox of pNumStrDlgDt.

   Vstim:
     cb, - record size
     stim, - string for asked number
     count - string for title
*)
var pm : pVstim;
    r  : real;
    i  : integer;
    s  : cstring[20];
    st : string[20];
    id, ioerr : word;
begin
  VStimDlgBoxProc := 0;
  case w of
    WM_INITDLG : begin
        pm := pVstim(m2);
        WinSetWindowPtr(h, QWL_USER, pm);
        id := CM_FIRST + IDD_SETSTIMV;
        with pm^ do
          for i := 1 to num do
            with stim[i] do begin
              inc(id);
              str(time:4:2, st);
              s := st;
              WinSetDlgItemText(h, id, s);
              inc(id);
              str(voltage:5:1, st);
              s := st;
              WinSetDlgItemText(h, id, s)
            end
      end;
    WM_COMMAND : begin
        case COMMANDMSG(@w)^.cmd - CM_FIRST of
          CM_OK     : begin
              pm := WinQueryWindowPtr(h, QWL_USER);
              id := 0;
              ioerr := 0;
              i  := WinQueryDlgItemText(h, id + CM_FIRST + IDD_SETVT1, 20, s);
              while (i>0) and (id<=5) do begin
                st := s;
                val(st, r, i);
                ioerr := i;
                str(r:6:2, st);
                if ioerr<>0 then begin
                  NumErrMsg(h);
                  i := 0
                end else begin
                  with pm^ do begin
                    num := succ(id div 2);
                    with stim[num] do
                      if odd(id) then voltage := r
                                 else time    := r
                    end;
                  inc(id);
                  i  := WinQueryDlgItemText(h, id + CM_FIRST + IDD_SETVT1, 20, s)
                end
              end;
              if (ioerr=0) and not odd(id) then
                WinDismissDlg(h, 1)
              else
                WinMessageBox(HWND_DESKTOP, h, 'You must fill time and volt.', '',
                                0, MB_OK)
            end;
          CM_CANCEL : WinDismissDlg(h, 0)
        else
        end
      end
  else
    VStimDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
  end
end;

function IStimDlgBoxProc (h : HWND;
                          w : ULONG;
                          m1: MPARAM;
                          m2: MPARAM) : MRESULT; CDECL;
(* ask a number with favourite title.
   This dialog box should be called
   with the last parametre of WinDlgBox of pNumStrDlgDt.

   Vstim:
     cb, - record size
     stim, - string for asked number
     count - string for title
*)
var pm : pStmDlgData;
    r  : real;
    i  : integer;
    s  : cstring[20];
    st : string[20];
    id, ioerr : word;
begin
  IStimDlgBoxProc := 0;
  case w of
    WM_INITDLG : begin
        pm := pStmDlgData(m2);
        WinSetWindowPtr(h, QWL_USER, pm);
        with pm^ do begin
          if text1<>'' then
            WinSetDlgItemText(h, CM_FIRST + IDD_STIMISTRENGSTXT, text1);
          if text2<>'' then
            WinSetDlgItemText(h, CM_FIRST + IDD_STIMISTIMTXT, text2)
        end;
        id := CM_FIRST + IDD_SETSTIMI;
        with pm^ do
          for i := 1 to nm do
            with stm[i] do begin
              str(time0:4:2, st);
              s := st;
              inc(id);
              WinSetDlgItemText(h, id, st);
              str(time1:4:2, st);
              s := st;
              inc(id);
              WinSetDlgItemText(h, id, s);
              str(stimval:5:1, st);
              s := st;
              inc(id);
              WinSetDlgItemText(h, id, s)
            end
      end;
    WM_COMMAND : begin
        case COMMANDMSG(@w)^.cmd - CM_FIRST of
          CM_OK     : begin
              pm := WinQueryWindowPtr(h, QWL_USER);
              id := 0;
              ioerr := 0;
              i  := WinQueryDlgItemText(h, id + CM_FIRST + IDD_SETITF1, 20, s);
              while (i>0) and (id<=8) do begin
                st := s;
                val(st, r, i);
                ioerr := i;
                if ioerr<>0 then begin
                  NumErrMsg(h);
                  i := 0
                end else begin
                  with pm^ do begin
                    nm := succ(id div 3);
                    with stm[nm] do
                      case id mod 3 of
                        0 : time0 := r;
                        1 : time1 := r;
                        2 : stimval := r
                      end
                    end;
                  inc(id);
                  i  := WinQueryDlgItemText(h, id + CM_FIRST + IDD_SETITF1, 20, s)
                end
              end;
              if (ioerr=0) and not (id mod 3 = 1) then
                WinDismissDlg(h, 1)
              else
                WinMessageBox(HWND_DESKTOP, h, 'You must fill time and volt.', '',
                                0, MB_OK)
            end;
          CM_CANCEL : WinDismissDlg(h, 0)
        else
        end
      end
  else
    IStimDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
  end
end;

function PrefDlgBoxProc (h : HWND;
                          w : ULONG;
                          m1: MPARAM;
                          m2: MPARAM) : MRESULT; CDECL;
(* ask preference
   This dialog box should be called
   with the last parametre of WinDlgBox of ppref.

   preference:
     cb, - record size
     MembParam, - membrane constants
     UseTable - switch of loading previously calculated alpha/beta
     PortAnimation - switch of port animation

   In current version, PortAnimation is not changed
*)
var pm : ppref;
begin
  PrefDlgBoxProc := 0;
  case w of
    WM_INITDLG : begin
        pm := ppref(m2);
        WinSetWindowPtr(h, QWL_USER, pm);
        with pm^ do begin
          WinSendDlgItemMsg(h, CM_FIRST + IDD_SETUSETABLE, BM_SETCHECK,
                            ord(UseTable), 0);
          WinSendDlgItemMsg(h, CM_FIRST + IDD_SETPORTANIMATION, BM_SETCHECK,
                            ord(PortAnimation), 0);
          WinSendDlgItemMsg(h, CM_FIRST + IDD_SETDOSSLEEP, BM_SETCHECK,
                            ord(UseDosSleep), 0)
        end
      end;
    WM_CONTROL :
        if hi(m1) = BN_CLICKED then begin
          pm := WinQueryWindowPtr(h, QWL_USER);
          with pm^ do
            case lo(m1)-CM_FIRST of
              IDD_SETUSETABLE      : UseTable := not UseTable;
              IDD_SETPORTANIMATION : PortAnimation := not PortAnimation;
              IDD_SETDOSSLEEP      : UseDosSleep := not UseDosSleep
            else
            end
        end;
    WM_COMMAND : begin
        case COMMANDMSG(@w)^.cmd - CM_FIRST of
          CM_OK     : WinDismissDlg(h, 1);
          CM_CANCEL : WinDismissDlg(h, 0)
        else
        end
      end
  else
    PrefDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
  end
end;

function CopyDlgBoxProc (h : HWND;
                         w : ULONG;
                         m1: MPARAM;
                         m2: MPARAM) : MRESULT; CDECL;
var i : ushort;
begin
  CopyDlgBoxProc := 0;
  case w of
    WM_INITDLG : begin
        WinSetWindowUShort(h, QWL_USER, 0);
        WinSendDlgItemMsg(h, CM_FIRST + IDD_COPYTOCLIP, BM_SETCHECK, 0, 0);
        WinSendDlgItemMsg(h, CM_FIRST + IDD_COPYTOFILE, BM_SETCHECK, 0, 0)
      end;
    WM_CONTROL :
        if hi(m1) = BN_CLICKED then begin
          i := succ(ord((lo(m1)-CM_FIRST) = IDD_COPYTOFILE));
          WinSetWindowUShort(h, QWL_USER, i)
        end;
    WM_COMMAND : begin
        case COMMANDMSG(@w)^.cmd - CM_FIRST of
          CM_OK     : WinDismissDlg(h, WinQueryWindowUShort(h, QWL_USER));
          CM_CANCEL : WinDismissDlg(h, 0)
        else
        end
      end
  else
    CopyDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
  end
end;

function AboutDlgBoxProc (h : HWND;
                          w : ULONG;
                          m1: MPARAM;
                          m2: MPARAM) : MRESULT; CDECL;
begin
  AboutDlgBoxProc := WinDefDlgProc(h, w, m1, m2)
end;

function ParamWndProc(  h : HWND;
                        w : ULONG;
                        m1: MPARAM;
                        m2: MPARAM) : MRESULT; CDECL;
(* Window procedure for parametre display area.
   User WM's :
     WM_ShowTmRecord - showing tm record.  m1 - address of tm to display.
     WM_USER2 - showing any string. m1 - address of string
*)

type
  membinfo = record
    cb  : word;
    dt  : tm;
    s   : ParaStr
  end;
  ptm      = ^tm;
var
  r : RECTL;
  hp: HPS;
  minf : ^membinfo;
  s1, s2, s3, s4,
  s5, s6, s7 : string;
  t : text;
begin
  case w of
    WM_CREATE: begin
        new(minf);
        with minf^, dt do begin
          cb   := sizeof(minf^);
          tt   := 0;
          tv   := 0;
          tgK  := 0;
          tgNa := 0;
          tn   := 0;
          tm   := 0;
          th   := 0;
          s    := ''
        end;
        WinSetWindowPtr(h, QWL_USER, minf);
        ParamWndProc := 0
      end;
    WM_PAINT : begin
        minf := WinQueryWindowPtr(h, QWL_USER);
        hp   := WinBeginPaint(h, NULLHANDLE, r);
        with minf^ do
          WinDrawText(hp, length(s), s[1], r, CLR_BLUE, 0,
                      DT_LEFT + DT_ERASERECT);
        WinEndPaint(hp);
        WinReleasePS(hp);
        ParamWndProc := 0
      end;
    WM_ShowTmRecord : begin
        minf := WinQueryWindowPtr(h, QWL_USER);
        with minf^ do begin
          dt:= ptm(m1)^;
          with dt do begin
            str(tt  :4:1, s1);
            str(tv  :5:1, s2);
            str(tgK :4:1, s3);
            str(tgNa:4:1, s4);
            str(tn  :4:2, s5);
            str(tm  :4:2, s6);
            str(th  :4:2, s7)
          end;
          s := 't:' + s1 + ' V:' + s2 + ' gK:' + s3 + ' gNa:' + s4 +
              ' n:' + s5 + ' m:' + s6 + ' h:' + s7
        end;
        WinInvalidateRect(h, nil, false)
      end;
    WM_ShowAnyString : begin
        minf := WinQueryWindowPtr(h, QWL_USER);
        with minf^ do s := ppstr(m1)^;
        WinInvalidateRect(h, nil, false)
      end;
    WM_ERASEBACKGROUND : ParamWndProc := 1;
    WM_CLOSE : begin
        minf := WinQueryWindowPtr(h, QWL_USER);
        dispose(minf);
        ParamWndProc := WinDefWindowProc(h, w, m1, m2)
      end
  else
    ParamWndProc := WinDefWindowProc(h, w, m1, m2)
  end
end;

procedure CopyTracesToBuffer(pd : pchary); CDECL;
var i, m : integer;
    p    : ULONG;
    s,
    s1, s2, s3, s4,
    s5, s6, s7, s8 : string;
    lt: tm;
begin
  m := ord(pd^[0]);
  if m = 0 then p := 0
           else p := sizeof(string) + 4;
  for i := 0 to TmNum do begin
    DosSleep(0);
    with TimeCourse[i] do begin
      str(tt  :ColumnSize:2, s1);
      str(tv  :ColumnSize:2, s2);
      str(tgK :ColumnSize:2, s3);
      str(tgNa:ColumnSize:2, s4);
      str(tn  :ColumnSize:6, s5);
      str(tm  :ColumnSize:6, s6);
      str(th  :ColumnSize:6, s7);
      str(tI  :ColumnSize:2, s8)
    end;
    s := s1 + ^I + s2 + ^I + s3 + ^I + s4 + ^I +
         s5 + ^I + s6 + ^I + s7 + ^I + s8 + ^M + ^J;
    move(s[1], pd^[p], length(s));
    inc(p, length(s))
  end;
  pd^[p] := chr(0);
  WinPostMsg(hwndCalcArea, WM_TracePrepared, m, 0);
  DosExit(EXIT_THREAD, 0)
end;

function CalcWndProc(h : HWND;
                     w : ULONG;
                     m1,
                     m2: MPARAM) : MRESULT; CDECL;
(* Main window procedure *)

var rcl   : RECTL;

  procedure CalcXtoTm;
  var i, x, x0, x1 : integer;
  begin
    fillchar(XtoTm, sizeof(XtoTm), 0);
    x0 := 0;
    for i:=1 to TmNum do begin
      x1 := round(TimeCourse[i].tt * xMagn);
      for x := x0 to x1 do XtoTm[x] := i;
      x0 := succ(x1)
    end
  end;

  procedure initWin;
  var hwndF : HWND;
      x0, y0,
      x1, y1: LONG;
  begin
    InitialisePS(h, calchps);
    GetFontData(h, AveChar, CharHeight);
    hwndF := WinQueryWindow(h, QW_PARENT);
    with settings do
      if PositionLoaded then begin
        x0 := posx;
        y0 := posy;
        x1 := sizex;
        y1 := sizey
      end else begin
        x0 := 50;
        y0 := 50;
        x1 := DispNum div 2;
        y1 := 360 + CharHeight
      end;
    WinSetWindowPos(hwndF, HWND_DESKTOP,
                    x0, y0, x1, y1,
                    SWP_MOVE + SWP_SIZE + SWP_ACTIVATE);
    WinRegisterClass(myhab, ParamWndClass, @ParamWndProc,
                     CS_SIZEREDRAW,
                     sizeof(tm) + sizeof(ParaStr) + 34);
    hwndParamArea := WinCreateWindow(h, ParamWndClass, '',
                       WS_VISIBLE, 0, 0, x1, CharHeight,
                       h, HWND_TOP, 1, nil, nil);
    DosCreateEventSem(nil, calculating, 0, false);
    DosCreateMutexSem(nil, DrawToMain, 0, false)
  end; { CalcWndProc.initWin }

  procedure EnableMenuItem(MenuID : ULONG; SetEnable : boolean);
  begin
    WinSendDlgItemMsg(hwndFrame, FID_MENU, MM_SETITEMATTR,
                      MPFROM2SHORT(MenuID + CM_FIRST, 1),
                      MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED * ord(not SetEnable)))
  end;

  procedure CommandProcs(cmd : word);
  (* Treats WM_COMMAND messages *)
  var dlg     : ppref;
      pm      : pNumStrDlgDt;
      pv      : pVstim;
      changed : word;
      i       : integer;
      s       : string;

    function GetFileName(title, extention : string; save : boolean) : string;
    var fdlg  : FILEDLG;
        szTitle : cstring[10];
        szFullFile : cstring[cchmaxpath];
        i     : integer;
        s     : cstring;
    begin
      szTitle      := title + ' File';
      szFullFile   := extention;
      fillchar(fdlg, sizeof(fdlg), 0);
      with fdlg do begin
        cbsize     := sizeof(fdlg);
        fl         := FDS_HELPBUTTON + FDS_CENTER;
        if save then inc(fl, FDS_SAVEAS_DIALOG)
                else inc(fl, FDS_OPEN_DIALOG);
        pszTitle   := @szTitle;
        szFullFile := szFullFile;
        s          := '';
        i := WinFileDlg(HWND_DESKTOP, h, fdlg);
        if (i<>0) and (lReturn = DID_OK) then begin
          if papszFQFilename<>nil then begin
            s := papszFQFilename^[0]^;
            WinFreeFileDlgList(papszFQFilename)
          end else
            s := szFullFile
        end
      end;
      GetFileName := s
    end;

    procedure AskStm(var stim : stimuliary; var n : integer;
                         s1, s2 : string);
    var pc      : pStmDlgData;
        chng    : word;
    begin
      new(pc);
      with pc^ do begin
        cb    := sizeof(pc^);
        text1 := s1;
        text2 := s2;
        stm   := stim;
        nm    := n
      end;
      chng := WinDlgBox(HWND_DESKTOP, h, @IStimDlgBoxProc,
                        NULLHANDLE, IDD_SETSTIMI, pc);
      if chng<>0 then
        with pc^ do begin
          stim := stm;
          n    := nm
        end;
      dispose(pc)
    end;

    function CalcClipboardBufferSize : ULONG;
    begin
      CalcClipboardBufferSize := succ(TmNum) *
                                 ((ColumnSize + 1) * pred(ColumnNum) + ColumnSize + 2) + 1
    end;

    procedure StartCopyTracesToClipboard;
    begin
      EnableMenuItem(IDM_COPYTRACES, false);
      DosAllocSharedMem(pdata, nil, CalcClipboardBufferSize,
                        PAG_COMMIT or OBJ_GIVEABLE or PAG_WRITE or PAG_READ);
      pdata^[0] := chr(0);
      StartThread(@CopyTracesToBuffer, 32000, pdata, idCopyThread)
    end;

    procedure StartCopyTracesToFile;
    var l : ULONG;
        s : string;
    begin
      s := GetFileName('Traces', '*.DAT', true);
      if s<>'' then begin
        EnableMenuItem(IDM_COPYTRACES, false);
        l := CalcClipboardBufferSize + sizeof(string) + 4;
        getmem(pdata, l);
        move(s, pdata^[0], sizeof(string));
        move(l, pdata^[sizeof(string)], 4);
        StartThread(@CopyTracesToBuffer, 32000, pdata, idCopyThread)
      end
    end;

  begin { CalcWndProc.CommandProcs }
    new(dlg);
    dlg^    := settings;
    changed := 0;
    case cmd of
      IDM_STARTCALC, IDM_STARTCALC2 : begin
          EnableMenuItem(IDM_STARTCALC, false);
          EnableMenuItem(IDM_STARTCALC2, false);
          EnableMenuItem(IDM_COPYTRACES, false);
          if running(calculating) then
            ResumeThread(idCalcThread)
          else begin
            if DrawCursor then XorDrawCursor(CursorPos);
            DrawCursor := false;
            StartThread(@CalcMain, StackSize, nil, idCalcThread)
          end
        end;
      IDM_PAUSECALC : begin
          if running(calculating) then SuspendThread(idCalcThread);
          EnableMenuItem(IDM_STARTCALC, true);
          EnableMenuItem(IDM_STARTCALC2, true);
          EnableMenuItem(IDM_COPYTRACES, true)
        end;
      IDM_ABORTCALC :
          ToCalc := false;
      IDM_FILELOAD  :
          LoadSettings(h, GetFileName('Load', '*.HXL', false));
      IDM_FILESAVE  :
          SaveSettings(h, settings.SettingFile);
      IDM_FILESAVEAS :
          SaveSettings(h, GetFileName('Save', '*.HXL', true));
      IDM_CLEARSCREEN : begin
          WinQueryWindowRect(h, rcl);
          if DrawCursor then begin
            XorDrawCursor(CursorPos);
            DrawCursor := false
          end;
          WinFillRect(calchps, rcl, CLR_BLACK);
          initDisplay;
          WinInvalidateRect(hwndParamArea, nil, false)
        end;
      IDM_SETTMAX : begin
          new(pm);
          with pm^ do begin
            cb := sizeof(pm^);
            str(dlg^.TLimit:5:1, st);
            title := 'Maximum Time'
          end;
          changed := WinDlgBox(HWND_DESKTOP, h, @NumDlgBoxProc,
                               NULLHANDLE, IDD_NUMSTRDLG, pm);
          if changed<>0 then begin
            val(pm^.st, dlg^.TLimit, i);
            WinInvalidateRect(h, nil, false)
          end;
          dispose(pm)
        end;
      IDM_INITV : begin
          new(pm);
          with pm^ do begin
            cb := sizeof(pm^);
            str(dlg^.InitV:5:2, st);
            title := 'Initial Voltage'
          end;
          changed := WinDlgBox(HWND_DESKTOP, h, @NumDlgBoxProc,
                               NULLHANDLE, IDD_NUMSTRDLG, pm);
          if changed<>0 then with pm^, dlg^ do val(st, InitV, i);
          dispose(pm)
        end;
      IDM_SETTODEFAULT: with dlg^ do begin
          TLimit := 10;
          SetParamDefault(MembParam);
          changed := 1
        end;
      IDM_SETCONDUCTANCE :
          changed := WinDlgBox(HWND_DESKTOP, h, @CondDlgBoxProc,
                               NULLHANDLE, IDD_CONDUCTANCEDLG, dlg);
      IDM_SETNMH         :
          changed := WinDlgBox(HWND_DESKTOP, h, @NMHDlgBoxProc,
                               NULLHANDLE, IDD_NMHDLG, dlg);
      IDM_SETOTHERS    :
          changed := WinDlgBox(HWND_DESKTOP, h, @MiscDlgBoxProc,
                               NULLHANDLE, IDD_SETOTHERS, dlg);
      IDM_SETCLEARSTIM   : begin
          InitStimV(StimV);
          InitStimI(StimI);
          InitClampV(ClmpV)
        end;
      IDM_SETVOLTSTIM    : begin
          new(pv);
          pv^ := StimV;
          changed := WinDlgBox(HWND_DESKTOP, h, @VStimDlgBoxProc,
                               NULLHANDLE, IDD_SETSTIMV, pv);
          if changed<>0 then StimV := pv^;
          changed := 0;
          dispose(pv)
        end;
      IDM_SETCURRENTSTIM : begin
          AskStm(StimI.stim, StimI.num, '', '');
          changed := 0
        end;
      IDM_SETVCLAMP      : begin
          AskStm(clmpV.stim, clmpV.num, 'Clamp Voltage', 'Voltage(mV)');
          changed := 0
        end;
      IDM_SETPREF        : begin
          changed := WinDlgBox(HWND_DESKTOP, h, @PrefDlgBoxProc,
                               NULLHANDLE, IDD_SETPREF, dlg);
          if (changed<>0) and (dlg^.UseTable<>settings.UseTable) then
            SetUpFunctionVars(dlg^.UseTable)
        end;
      IDM_COPYTRACES     :
        case WinDlgBox(HWND_DESKTOP, h, @CopyDlgBoxProc,
                       NULLHANDLE, IDD_COPYTRACES, nil) of
          0 :;
          1 : begin
                EnableMenuItem(IDM_COPYTRACES, false);
                StartCopyTracesToClipboard
              end;
          2 : begin
                EnableMenuItem(IDM_COPYTRACES, false);
                StartCopyTracesToFile
              end
        end;
      IDM_ABOUT   :  WinDlgBox(HWND_DESKTOP, h, @AboutDlgBoxProc,
                               NULLHANDLE, IDD_ABOUT, nil)
    else
    end;
    if changed <> 0 then begin
      settings := dlg^;
      memb.setconsts(settings.MembParam)
    end;
    dispose(dlg)
  end;  { CalcWndProc.CommandProc }

  procedure DispPointedData(x : ULONG);
  var ttm : tm;
  begin
    if not running(calculating) then begin
      XorDrawCursor(CursorPos);
      ttm := TimeCourse[XtoTm[x]];
      if settings.PortAnimation then
        with ttm do begin
          ResetXYMagn;
          NaChannel.draw(tm, th);
          KChannel.draw(tn)
        end;
      WinPostMsg(hwndParamArea, WM_ShowTmRecord, MPARAM(@ttm), 0);
      CursorPos := x;
      XorDrawCursor(CursorPos)
    end
  end;

  procedure mouseops(x : integer);

    function ok : boolean;
    begin
      ok := not running(calculating)
            and (x > 0) and (x < cxClient - Rmargin - xofs)
    end;

  begin
    case w of
      WM_BUTTON1DOWN : begin
          buttonpush := true;
          if ok and not DrawCursor then begin
            CalcXtoTm;
            CursorPos := x;
            XorDrawCursor(x);
            DrawCursor := true
          end
        end;
      WM_BUTTON1UP   : buttonpush := false;
      WM_MOUSEMOVE   : begin
          if buttonpush and ok then DispPointedData(x);
        end
    else
    end
  end;

  procedure CopyTracesToClipboard;
  begin
    WinOpenClipbrd(myhab);
    WinEmptyClipbrd(myhab);
    WinSetClipbrdData(myhab, ULONG(pdata), CF_TEXT, CFI_POINTER);
    WinCloseClipbrd(myhab)
  end;

  procedure CopyTracesToFile;
  var f : file;
      l : ULONG;
      s : string;
  begin
    move(pdata^[0], s, sizeof(string));
    move(pdata^[sizeof(string)], l, 4);
    pdata^[pred(l)] := chr($1A);
    assign(f, s);
    rewrite(f, 1);
    blockwrite(f, pdata^[sizeof(string) + 4], l - sizeof(string) - 4);
    close(f);
    freemem(pdata, l)
  end;

begin  { CalcWndProc }
  CalcWndProc := 0;
  case w of
    WM_CREATE  : InitWin;
    WM_SIZE    : begin
        cxClient := lo(m2);
        cyClient := hi(m2);
        Resized  := true;
        if DrawCursor then begin
          XorDrawCursor(CursorPos);
          DrawCursor := false
        end;
        WinSetWindowPos(hwndParamArea, h, 0, 0,
                        cxClient, CharHeight, SWP_SIZE)
      end;
    WM_PAINT   : begin
        DosRequestMutexSem(DrawToMain, 2000);
        WinBeginPaint(h, calchps, rcl);
        WinQueryWindowRect(h, rcl);
        WinFillRect(calchps, rcl, CLR_BLACK);
        DrawTimeCourse;
        WinEndPaint(calchps);
        DosReleaseMutexSem(DrawToMain)
      end;
    WM_BUTTON1DOWN,
    WM_BUTTON1UP,
    WM_MOUSEMOVE    : begin
        mouseops(lo(m1) - xofs);
        CalcWndProc := WinDefWindowProc(h, w, m1, m2)
      end;
    WM_COMMAND : CommandProcs(COMMANDMSG(@w)^.cmd - CM_FIRST);
    WM_TraceCalculated : begin
        EnableMenuItem(IDM_STARTCALC, true);
        EnableMenuItem(IDM_STARTCALC2, true);
        EnableMenuItem(IDM_COPYTRACES, true)
      end;
    WM_TracePrepared : begin
        if m1 = 0 then CopyTracesToClipboard
                  else CopyTracesToFile;
        EnableMenuItem(IDM_COPYTRACES, true)
      end;
    WM_CLOSE   : begin
        if running(calculating) then KillThread(idCalcThread);
        GpiDestroyPS(calchps);
        CalcWndProc := WinDefWindowProc(h, w, m1, m2)
      end
    else
        CalcWndProc := WinDefWindowProc(h, w, m1, m2)
  end
end;  { CalcWndProc }


begin  { Main }
  InitProgManual;

  InitPMVar;
  if paramcount = 0 then LoadSettings(NULLHANDLE, defSettingFile)
                    else LoadSettings(NULLHANDLE, paramstr(1));

  myhab     := WinInitialize(0);
  WinRegisterClass(myhab, CalcWndClass, @CalcWndProc, CS_SIZEREDRAW, 0);
  hwndFrame := WinCreateStdWindow(HWND_DESKTOP,
                                  WS_VISIBLE,
                                  ctlData, CalcWndClass, AppTitle,
                                  0, NULLHANDLE, IDM_FILE,
                                  hwndCalcArea);
  MainDispatchLoop;

  WinDestroyMsgQueue(myhmq);
  WinTerminate(myhab)
end.



