/* REXX PROGRAM */
/* This is a sample Rexx program to modify an ascii export file. */
/* If you are preparing a file for importing into Check+ it is usually better to use a customized Command File instead
	 - see Check+ online help for Importing Ascii Files and the sample file "SampleIn.cmd". */
/* Usually the user will only need to change the CHANGE_FIELD_DATA subroutine. */
/* The general outline is:
	Load the Header Line from the Input file or the Header file
		(This also builds a template line which is used to parse the remaining data lines.)
	Write the Header Line to the output file
	For each line in the input file:
		Load the line - this also parses it to load the Rexx variables
			(For each field you are importing Check+ will add a Rexx variable with the same
			name as the field and will assign it the data from the corresponding field in the line.)
		Change the fields - modify any Rexx variables desired
		Write the modified line to the output file
			(This rebuilds the line from the Rexx variables and outputs it to the file)
	*/
/* Note: Depending on whether the data is quoted or not, the Rexx variables may or may not include the quotes in the data. */


/* MAIN PROGRAM */
"@echo off"
/* Load the Rexx variables based on the arguments the user passed in. */
ARG FileIn FileOut HeaderFile delimiter
/* Make sure the required arguments where given. */
IF FileIn = "" | FileOut = "" THEN DO
	SAY "The following arguments are expected: FILEIN FILEOUT HEADERFILE DELIMITER"
	SAY "    FILEIN and FILEOUT are required."
	SAY "    HEADERFILE will default to FILEIN if not given."
	SAY "    DELIMITER will default to ',' if not given."
	EXIT
	END
/* Set the default header file to the input file and the default delimiter to ','. */
IF HeaderFile = "" THEN HeaderFile = FileIn
IF delimiter = "" THEN delimiter = ','
/* Delete FileOut if it already exists. */
"del " FileOut "2>nul"
/* Load the Header Line and write it out to the output file. */
CALL LOAD_HEADER_LINE
CALL WRITE_HEADER_LINE
/* For each line in the input file, load the line (which loads the Rexx variables), modify the Rexx variables, and write the adjusted line out. */
DO while LINES(FileIn)
	CALL LOAD_LINE
	CALL CHANGE_FIELD_DATA
	CALL WRITE_LINE
	end
/* Close all the files. */
result = stream(FileOut, 'c', 'close')
result = stream(FileIn, 'c', 'close')
IF HeaderFile \= FileIn THEN stream(HeaderFile, 'c', 'close')
EXIT


/* SUBROUTINE */
CHANGE_FIELD_DATA:
/* Change fields as desired. */
/* This subroutine is usually the only portion that the user needs to change. */

/* The following instructions are examples of some common changes: */

/* Change the type to a type compatible with Check+.  Setting type to "" causes it to default to a Computer Check or Hand Check. */
/*
if type = '"Receive"'
	THEN type = '"Deposit"'
	ELSE type = '""'
*/

/* If the amount is negative make it positive. */
/*
IF allocationAmount < 0 THEN allocationAmount = 0 - allocationAmount
*/

/* Change dateCleared and datePrinted to a compatible format.  Check+ expects a date to indicate cleared or printed. */
/*
IF dateCleared = '"R"' THEN dateCleared = date
IF datePrinted = '"P"' THEN datePrinted = date
*/

RETURN


/* SUBROUTINE */
LOAD_HEADER_LINE:
/* Read the header line from the file. */
headerLine = LINEIN(HeaderFile)
/* Build a template from the header line which is used to load Rexx variables. */
templateLine = headerLine
/* Strip periods - field names can't have periods in them in Rexx */
IF delimiter \= '.' THEN DO
	position = pos('.', templateLine)
	DO WHILE position > 0
		templateLine = delstr(templateLine, position, 1)
		position = pos('.', templateLine)
		END
	END
IF delimiter = ' ' THEN RETURN
/* Remove all spaces from the template line */
templateLine = SPACE(templateLine, 0)
/* Strip double quotes */
position = pos('"', templateLine)
DO WHILE position > 0
	templateLine = delstr(templateLine, position, 1)
	position = pos('"', templateLine)
	END
/* Strip single quotes */
position = pos("'", templateLine)
DO WHILE position > 0
	templateLine = delstr(templateLine, position, 1)
	position = pos("'", templateLine)
	END
/* Replace all occurences of delimiter with a quoted version of the delimiter. */
delimiterString = " '"delimiter"' "
position = pos(delimiter, templateLine)
DO WHILE position > 0
	templateLine = delstr(templateLine, position, 1)
	templateLine = insert(delimiterString, templateLine, position-1)
	startPos = position + LENGTH(delimiterString)
	position = pos(delimiter, templateLine, startPos)
	END
/* SAY "Template Line is "templateLine */
RETURN


/* SUBROUTINE */
LOAD_LINE:
/* Read the line from the file. */
line = LINEIN(FileIn)
/* Parse the line based on the template line - this loads all the Rexx variables. */
command = "PARSE VAR line "templateLine
INTERPRET command
/* SAY "Line In is "line */
RETURN


/* SUBROUTINE */
WRITE_LINE:
/* Create the output line based on the template line - this builds the line from the Rexx variables. */
command = "line = "SPACE(templateLine, 0)
INTERPRET command
/* Write the line out to the file. */
rc = LINEOUT(FileOut, line)
/* SAY "Line Out is "line */
RETURN

/* SUBROUTINE */
WRITE_HEADER_LINE:
/* Write out the header line. */
rc = LINEOUT(FileOut, headerLine)
RETURN
