/*
 * KMP : K Movie Player based on FFplay
 * Copyright (c) 2007-2008 KO Myung-Hun <komh@chollian.net>
 *
 * This file is part of KMP.
 *
 * FFplay : Simple Media Player based on the ffmpeg libraries
 * Copyright (c) 2003 Fabrice Bellard
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#undef __STRICT_ANSI__  // for _beginthread() and _endthread() of SDL
#include <math.h>
#include <limits.h>
#include "libavutil/avstring.h"
#include "libavformat/avformat.h"
#include "libavformat/rtsp.h"
#include "libavdevice/avdevice.h"
#include "libswscale/swscale.h"
#include "libavcodec/audioconvert.h"
#include "libavcodec/opt.h"

#include "cmdutils.h"

#define INCL_DOS
#define INCL_WIN
#define INCL_GPI
#include <os2.h>

#include <stdio.h>
#include <stdlib.h>

#include "version.h"
#include "libavcodec/dsputil.h"
#include "libvo/cpudetect.h"
#include "libvo/fastmemcpy.h"
#include "libvo/mp_msg.h"

#include "dart.h"

#include "kmp.h"
#include "kmp_playlist.h"

void MorphToPM( void )
{
    PPIB pib;
    PTIB tib;

    DosGetInfoBlocks(&tib, &pib);

    // Change flag from VIO to PM:
    if (pib->pib_ultype==2) pib->pib_ultype = 3;
}

#undef exit

const char program_name[] = "KMP based on FFplay";
const int program_birth_year = 2007;

//#define DEBUG_SYNC

#define MAX_VIDEOQ_SIZE (5 * 256 * 1024)
#define MAX_AUDIOQ_SIZE (5 * 16 * 1024)
#define MAX_SUBTITLEQ_SIZE (5 * 16 * 1024)

/* no AV sync correction is done if below the AV sync threshold */
#define AV_SYNC_THRESHOLD 0.01
/* no AV correction is done if too big error */
#define AV_NOSYNC_THRESHOLD 10.0

/* maximum audio speed change to get correct sync */
#define SAMPLE_CORRECTION_PERCENT_MAX 10

/* we use about AUDIO_DIFF_AVG_NB A-V differences to make the average */
#define AUDIO_DIFF_AVG_NB   20

/* NOTE: the size must be big enough to compensate the hardware audio buffersize size */
#define SAMPLE_ARRAY_SIZE (2*65536)

#define COLOR_FONT      0xFFFFFF
#define COLOR_OUTLINE   0x010101
#define COLOR_SHADOW    0x010101
//#define COLOR_OUTLINE   0x0F0F0F
//#define COLOR_SHADOW    0x0F0F0F
#define COLOR_OVERLAY   0x000008

#define SUB_MARGIN_LEFT         16
#define SUB_MARGIN_RIGHT        16
#define SUB_MARGIN_BOTTOM( h )  (( h ) / 20 )

#define OSD_MARGIN_LEFT         16
#define OSD_MARGIN_RIGHT        16
#define OSD_MARGIN_TOP( h )     (( h ) / 10 )

typedef struct PacketQueue {
    AVPacketList *first_pkt, *last_pkt;
    int nb_packets;
    int size;
    int abort_request;
    Mutex mutex;
    Cond cond;
} PacketQueue;

/* VIDEO_PICTURE_QUEUE_SIZE should be 1, else shaking phenomenon occur */
#define VIDEO_PICTURE_QUEUE_SIZE 1
#define SUBPICTURE_QUEUE_SIZE 4

typedef struct VideoPicture {
    double pts;                                  ///<presentation time stamp for this picture
    AVFrame *bmp;
    YUV *convBuf;
} VideoPicture;

typedef struct SubPicture {
    double pts; /* presentation time stamp for this picture */
    AVSubtitle sub;
} SubPicture;

enum {
    AV_SYNC_AUDIO_MASTER, /* default choice */
    AV_SYNC_VIDEO_MASTER,
    AV_SYNC_EXTERNAL_CLOCK, /* synchronize to an external clock */
};

typedef struct VideoState {
    Thread parse_tid;
    Thread video_tid;
    AVInputFormat *iformat;
    int no_background;
    int abort_request;
    int paused;
    int last_paused;
    int seek_req;
    int seek_flags;
    int64_t seek_pos;
    AVFormatContext *ic;
    int dtg_active_format;

    Thread audio_tid;
    int audio_stream;

    int av_sync_type;
    double external_clock; /* external clock base */
    int64_t external_clock_time;

    double audio_clock;
    double audio_diff_cum; /* used for AV difference average computation */
    double audio_diff_avg_coef;
    double audio_diff_threshold;
    int audio_diff_avg_count;
    AVStream *audio_st;
    PacketQueue audioq;
    int audio_hw_buf_size;
    /* samples output by the codec. we reserve more space for avsync
       compensation */
    DECLARE_ALIGNED(16,uint8_t,audio_buf1[(AVCODEC_MAX_AUDIO_FRAME_SIZE * 3) / 2]);
    DECLARE_ALIGNED(16,uint8_t,audio_buf2[(AVCODEC_MAX_AUDIO_FRAME_SIZE * 3) / 2]);
    uint8_t *audio_buf;
    unsigned int audio_buf_size; /* in bytes */
    int audio_buf_index; /* in bytes */
    AVPacket audio_pkt;
    uint8_t *audio_pkt_data;
    int audio_pkt_size;
    enum SampleFormat audio_src_fmt;
    AVAudioConvert *reformat_ctx;

    int show_audio; /* if true, display audio samples */
    int16_t sample_array[SAMPLE_ARRAY_SIZE];
    int sample_array_index;
    int last_i_start;

    Thread subtitle_tid;
    int subtitle_stream;
    int subtitle_stream_changed;
    AVStream *subtitle_st;
    PacketQueue subtitleq;
    SubPicture subpq[SUBPICTURE_QUEUE_SIZE];
    int subpq_size, subpq_rindex, subpq_windex;
    Mutex subpq_mutex;
    Cond subpq_cond;

    double frame_timer;
    double frame_last_pts;
    double frame_last_delay;
    double video_clock;                          ///<pts of last decoded frame / predicted pts of next decoded frame
    int video_stream;
    AVStream *video_st;
    PacketQueue videoq;
    double video_current_pts;                    ///<current displayed pts (different from video_clock if frame fifos are used)
    int64_t video_current_pts_time;              ///<time (av_gettime) at which we updated video_current_pts - used to have running video pts
    VideoPicture pictq[VIDEO_PICTURE_QUEUE_SIZE];
    int pictq_size, pictq_rindex, pictq_windex;
    Mutex pictq_mutex;
    Cond pictq_cond;

    //    QETimer *video_timer;
    char filename[1024];
    int width, height, xleft, ytop;

    int use_sub;
    ReSampleContext *rsc;
    int paint_in_pause;
} VideoState;

static void show_help(void);
static int audio_write_get_buf_size(VideoState *is);
static double get_master_clock(VideoState *is);

#define RES_NONE    0
#define RES_48KONLY 1
#define RES_ALL     2
#define RES_AUTO    3

/* options specified by the user */
static AVInputFormat *file_iformat;
static const char *input_filename;
static int opt_screen_width = 0;
static int opt_screen_height = 0;
static int screen_width = 0;
static int screen_height = 0;
static int frame_width = 0;
static int frame_height = 0;
static enum PixelFormat frame_pix_fmt = PIX_FMT_NONE;
static int audio_disable;
static int video_disable;
static int wanted_audio_stream= 0;
static int wanted_video_stream= 0;
static int seek_by_bytes;
static int display_disable;
static int show_status;
static int av_sync_type = AV_SYNC_AUDIO_MASTER;
static int64_t start_time = AV_NOPTS_VALUE;
static int debug = 0;
static int debug_mv = 0;
static int step = 0;
static int thread_count = 1;
static int workaround_bugs = 1;
static int fast = 0;
static int genpts = 0;
static int lowres = 0;
static int idct = FF_IDCT_AUTO;
static enum AVDiscard skip_frame= AVDISCARD_DEFAULT;
static enum AVDiscard skip_idct= AVDISCARD_DEFAULT;
static enum AVDiscard skip_loop_filter= AVDISCARD_DEFAULT;
static int error_recognition = FF_ER_CAREFUL;
static int error_concealment = 3;
static int decoder_reorder_pts= 0;
static int video_driver = KVAM_AUTO;
static int volume_level = 80;
static char sub_fontnamesize[ FACESIZE + 5 ] = "24.";
static char osd_fontnamesize[ FACESIZE + 5 ] = "13.System VIO";
static int aspect_ratio = KVAR_ORIGINAL;
static int frame_drop = 0;
static int hide_mouse = 0;
static int resample = RES_NONE;
static int use_subimg = 0;
static int fix_t23 = 0;
static int no_audio_share = 0;
static long sub_color_font = COLOR_FONT;
static long sub_color_outline = COLOR_OUTLINE;
static long sub_color_shadow = COLOR_SHADOW;
static long osd_color_font = COLOR_FONT;
static long osd_color_outline = COLOR_OUTLINE;
static long osd_color_shadow = COLOR_SHADOW;
static int fix_snap = 0;
static int audio_buffer_size = 0;
static int auto_add = 0;

/* current context */
static int is_full_screen;
static VideoState *cur_stream;
static int64_t audio_callback_time;

static AVPacket flush_pkt;

#define WM_REFRESH_EVENT      ( WM_USER + 1 )

#define KMP_NAME    "K Movie Player"
#define KMP_VERSION "v0.5.1"

#define TID_SECOND      ( TID_USERMAX - 1 )
#define TID_INFORM      ( TID_USERMAX - 2 )
#define TID_MOUSE_HIDE  ( TID_USERMAX - 3 )

#define INTERVAL_SECOND     1000
#define INTERVAL_INFORM     1500
#define INTERVAL_MOUSE_HIDE 1500

static HAB  hab;
static HWND hwndFrame;
static HWND hwndSysMenu;
static HWND hwndTitleBar;
static HWND hwndMinMax;
static HWND hwndKMP;

static BOOL  fMouseShow = TRUE;
static ULONG aulAttrValue[ KVAA_LAST ] = { -1, -1, -1, -1, -1, };
static BOOL  fVideoOpened = FALSE;

static LONG  cxScreen;
static LONG  cyScreen;

static LONG  lImgWidth = 0;
static LONG  lImgHeight = 0;

static PFNWP pfnwpOldFrame = NULL;

#define TOLERANCE   3

static PPLAYLIST m_ppl = NULL;

#define PLAYLIST_PREV_FILE  1
#define PLAYLIST_NEXT_FILE  2
#define PLAYLIST_QUIT       3

static int  m_cmdPlaylist;

#define SHOW_POINTER( fShow ) \
{\
    WinShowPointer( HWND_DESKTOP, fShow );\
    fMouseShow = fShow;\
}

/* packet queue handling */
static void packet_queue_init(PacketQueue *q)
{
    memset(q, 0, sizeof(PacketQueue));
    q->mutex = CreateMutex();
    q->cond = CreateCond();
}

static void packet_queue_flush(PacketQueue *q)
{
    AVPacketList *pkt, *pkt1;

    LockMutex(q->mutex);
    for(pkt = q->first_pkt; pkt != NULL; pkt = pkt1) {
        pkt1 = pkt->next;
        av_free_packet(&pkt->pkt);
        av_freep(&pkt);
    }
    q->last_pkt = NULL;
    q->first_pkt = NULL;
    q->nb_packets = 0;
    q->size = 0;
    UnlockMutex(q->mutex);
}

static void packet_queue_end(PacketQueue *q)
{
    packet_queue_flush(q);
    DestroyMutex(q->mutex);
    DestroyCond(q->cond);
}

static int packet_queue_put(PacketQueue *q, AVPacket *pkt)
{
    AVPacketList *pkt1;

    /* duplicate the packet */
    if (pkt!=&flush_pkt && av_dup_packet(pkt) < 0)
        return -1;

    pkt1 = av_malloc(sizeof(AVPacketList));
    if (!pkt1)
        return -1;
    pkt1->pkt = *pkt;
    pkt1->next = NULL;


    LockMutex(q->mutex);

    if (!q->last_pkt)
        q->first_pkt = pkt1;
    else
        q->last_pkt->next = pkt1;
    q->last_pkt = pkt1;
    q->nb_packets++;
    q->size += pkt1->pkt.size;
    /* XXX: should duplicate packet data in DV case */
    CondSignal(q->cond);

    UnlockMutex(q->mutex);
    return 0;
}

static void packet_queue_abort(PacketQueue *q)
{
    LockMutex(q->mutex);

    q->abort_request = 1;

    CondSignal(q->cond);

    UnlockMutex(q->mutex);
}

/* return < 0 if aborted, 0 if no packet and > 0 if packet.  */
static int packet_queue_get(PacketQueue *q, AVPacket *pkt, int block)
{
    AVPacketList *pkt1;
    int ret;

    LockMutex(q->mutex);

    for(;;) {
        if (q->abort_request) {
            ret = -1;
            break;
        }

        pkt1 = q->first_pkt;
        if (pkt1) {
            q->first_pkt = pkt1->next;
            if (!q->first_pkt)
                q->last_pkt = NULL;
            q->nb_packets--;
            q->size -= pkt1->pkt.size;
            *pkt = pkt1->pkt;
            av_free(pkt1);
            ret = 1;
            break;
        } else if (!block) {
            ret = 0;
            break;
        } else {
            CondWait(q->cond, q->mutex);
        }
    }
    UnlockMutex(q->mutex);
    return ret;
}

static inline void fill_rectangle( HPS hps,
                                   int x, int y, int w, int h, int color )
{
    RECTL   rcl;

    rcl.xLeft = x;
    rcl.yBottom = y;
    rcl.xRight = x + w;
    rcl.yTop = y + h;
    WinFillRect( hps, &rcl, color );
}

#define SCALEBITS 10
#define ONE_HALF  (1 << (SCALEBITS - 1))
#define FIX(x)    ((int) ((x) * (1<<SCALEBITS) + 0.5))

#define RGB_TO_Y_CCIR(r, g, b) \
((FIX(0.29900*219.0/255.0) * (r) + FIX(0.58700*219.0/255.0) * (g) + \
  FIX(0.11400*219.0/255.0) * (b) + (ONE_HALF + (16 << SCALEBITS))) >> SCALEBITS)

#define RGB_TO_U_CCIR(r1, g1, b1, shift)\
(((- FIX(0.16874*224.0/255.0) * r1 - FIX(0.33126*224.0/255.0) * g1 +         \
     FIX(0.50000*224.0/255.0) * b1 + (ONE_HALF << shift) - 1) >> (SCALEBITS + shift)) + 128)

#define RGB_TO_V_CCIR(r1, g1, b1, shift)\
(((FIX(0.50000*224.0/255.0) * r1 - FIX(0.41869*224.0/255.0) * g1 -           \
   FIX(0.08131*224.0/255.0) * b1 + (ONE_HALF << shift) - 1) >> (SCALEBITS + shift)) + 128)

#define ALPHA_BLEND(a, oldp, newp, s)\
((((oldp << s) * (255 - (a))) + (newp * (a))) / (255 << s))

#define RGBA_IN(r, g, b, a, s)\
{\
    unsigned int v = ((const uint32_t *)(s))[0];\
    a = (v >> 24) & 0xff;\
    r = (v >> 16) & 0xff;\
    g = (v >> 8) & 0xff;\
    b = v & 0xff;\
}

#define YUVA_IN(y, u, v, a, s, pal)\
{\
    unsigned int val = ((const uint32_t *)(pal))[*(const uint8_t*)(s)];\
    a = (val >> 24) & 0xff;\
    y = (val >> 16) & 0xff;\
    u = (val >> 8) & 0xff;\
    v = val & 0xff;\
}

#define YUVA_OUT(d, y, u, v, a)\
{\
    ((uint32_t *)(d))[0] = (a << 24) | (y << 16) | (u << 8) | v;\
}


#define BPP 1

static void blend_subrect(AVPicture *dst, const AVSubtitleRect *rect, int imgw, int imgh)
{
    int wrap, wrap3, width2, skip2;
    int y, u, v, a, u1, v1, a1, w, h;
    uint8_t *lum, *cb, *cr;
    const uint8_t *p;
    const uint32_t *pal;
    int dstx, dsty, dstw, dsth;

    dstx = FFMIN(FFMAX(rect->x, 0), imgw);
    dstw = FFMIN(FFMAX(rect->w, 0), imgw - dstx);
    dsty = FFMIN(FFMAX(rect->y, 0), imgh);
    dsth = FFMIN(FFMAX(rect->h, 0), imgh - dsty);
    lum = dst->data[0] + dsty * dst->linesize[0];
    cb = dst->data[1] + (dsty >> 1) * dst->linesize[1];
    cr = dst->data[2] + (dsty >> 1) * dst->linesize[2];

    width2 = (dstw + 1) >> 1;
    skip2 = dstx >> 1;
    wrap = dst->linesize[0];
    wrap3 = rect->linesize;
    p = rect->bitmap;
    pal = rect->rgba_palette;  /* Now in YCrCb! */

    if (dsty & 1) {
        lum += dstx;
        cb += skip2;
        cr += skip2;

        if (dstx & 1) {
            YUVA_IN(y, u, v, a, p, pal);
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);
            cb[0] = ALPHA_BLEND(a >> 2, cb[0], u, 0);
            cr[0] = ALPHA_BLEND(a >> 2, cr[0], v, 0);
            cb++;
            cr++;
            lum++;
            p += BPP;
        }
        for(w = dstw - (dstx & 1); w >= 2; w -= 2) {
            YUVA_IN(y, u, v, a, p, pal);
            u1 = u;
            v1 = v;
            a1 = a;
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);

            YUVA_IN(y, u, v, a, p + BPP, pal);
            u1 += u;
            v1 += v;
            a1 += a;
            lum[1] = ALPHA_BLEND(a, lum[1], y, 0);
            cb[0] = ALPHA_BLEND(a1 >> 2, cb[0], u1, 1);
            cr[0] = ALPHA_BLEND(a1 >> 2, cr[0], v1, 1);
            cb++;
            cr++;
            p += 2 * BPP;
            lum += 2;
        }
        if (w) {
            YUVA_IN(y, u, v, a, p, pal);
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);
            cb[0] = ALPHA_BLEND(a >> 2, cb[0], u, 0);
            cr[0] = ALPHA_BLEND(a >> 2, cr[0], v, 0);
        }
        p += wrap3 + (wrap3 - dstw * BPP);
        lum += wrap + (wrap - dstw - dstx);
        cb += dst->linesize[1] - width2 - skip2;
        cr += dst->linesize[2] - width2 - skip2;
    }
    for(h = dsth - (dsty & 1); h >= 2; h -= 2) {
        lum += dstx;
        cb += skip2;
        cr += skip2;

        if (dstx & 1) {
            YUVA_IN(y, u, v, a, p, pal);
            u1 = u;
            v1 = v;
            a1 = a;
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);
            p += wrap3;
            lum += wrap;
            YUVA_IN(y, u, v, a, p, pal);
            u1 += u;
            v1 += v;
            a1 += a;
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);
            cb[0] = ALPHA_BLEND(a1 >> 2, cb[0], u1, 1);
            cr[0] = ALPHA_BLEND(a1 >> 2, cr[0], v1, 1);
            cb++;
            cr++;
            p += -wrap3 + BPP;
            lum += -wrap + 1;
        }
        for(w = dstw - (dstx & 1); w >= 2; w -= 2) {
            YUVA_IN(y, u, v, a, p, pal);
            u1 = u;
            v1 = v;
            a1 = a;
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);

            YUVA_IN(y, u, v, a, p, pal);
            u1 += u;
            v1 += v;
            a1 += a;
            lum[1] = ALPHA_BLEND(a, lum[1], y, 0);
            p += wrap3;
            lum += wrap;

            YUVA_IN(y, u, v, a, p, pal);
            u1 += u;
            v1 += v;
            a1 += a;
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);

            YUVA_IN(y, u, v, a, p, pal);
            u1 += u;
            v1 += v;
            a1 += a;
            lum[1] = ALPHA_BLEND(a, lum[1], y, 0);

            cb[0] = ALPHA_BLEND(a1 >> 2, cb[0], u1, 2);
            cr[0] = ALPHA_BLEND(a1 >> 2, cr[0], v1, 2);

            cb++;
            cr++;
            p += -wrap3 + 2 * BPP;
            lum += -wrap + 2;
        }
        if (w) {
            YUVA_IN(y, u, v, a, p, pal);
            u1 = u;
            v1 = v;
            a1 = a;
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);
            p += wrap3;
            lum += wrap;
            YUVA_IN(y, u, v, a, p, pal);
            u1 += u;
            v1 += v;
            a1 += a;
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);
            cb[0] = ALPHA_BLEND(a1 >> 2, cb[0], u1, 1);
            cr[0] = ALPHA_BLEND(a1 >> 2, cr[0], v1, 1);
            cb++;
            cr++;
            p += -wrap3 + BPP;
            lum += -wrap + 1;
        }
        p += wrap3 + (wrap3 - dstw * BPP);
        lum += wrap + (wrap - dstw - dstx);
        cb += dst->linesize[1] - width2 - skip2;
        cr += dst->linesize[2] - width2 - skip2;
    }
    /* handle odd height */
    if (h) {
        lum += dstx;
        cb += skip2;
        cr += skip2;

        if (dstx & 1) {
            YUVA_IN(y, u, v, a, p, pal);
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);
            cb[0] = ALPHA_BLEND(a >> 2, cb[0], u, 0);
            cr[0] = ALPHA_BLEND(a >> 2, cr[0], v, 0);
            cb++;
            cr++;
            lum++;
            p += BPP;
        }
        for(w = dstw - (dstx & 1); w >= 2; w -= 2) {
            YUVA_IN(y, u, v, a, p, pal);
            u1 = u;
            v1 = v;
            a1 = a;
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);

            YUVA_IN(y, u, v, a, p + BPP, pal);
            u1 += u;
            v1 += v;
            a1 += a;
            lum[1] = ALPHA_BLEND(a, lum[1], y, 0);
            cb[0] = ALPHA_BLEND(a1 >> 2, cb[0], u, 1);
            cr[0] = ALPHA_BLEND(a1 >> 2, cr[0], v, 1);
            cb++;
            cr++;
            p += 2 * BPP;
            lum += 2;
        }
        if (w) {
            YUVA_IN(y, u, v, a, p, pal);
            lum[0] = ALPHA_BLEND(a, lum[0], y, 0);
            cb[0] = ALPHA_BLEND(a >> 2, cb[0], u, 0);
            cr[0] = ALPHA_BLEND(a >> 2, cr[0], v, 0);
        }
    }
}

static void free_subpicture(SubPicture *sp)
{
    int i;

    for (i = 0; i < sp->sub.num_rects; i++)
    {
        av_free(sp->sub.rects[i].bitmap);
        av_free(sp->sub.rects[i].rgba_palette);
    }

    av_free(sp->sub.rects);

    memset(&sp->sub, 0, sizeof(AVSubtitle));
}

static void video_image_display(VideoState *is)
{
    VideoPicture *vp;
    SubPicture *sp;
    int i;
    HPS hps = NULLHANDLE; // to avoid 'uninitialized' warning
    int w, h;

    vp = &is->pictq[is->pictq_rindex];
    if (vp->bmp) {
        if (is->subtitle_st)
        {
            if (is->subpq_size > 0)
            {
                sp = &is->subpq[is->subpq_rindex];

                if (vp->pts >= sp->pts + ((float) sp->sub.start_display_time / 1000))
                {
                    for (i = 0; i < sp->sub.num_rects; i++)
                        blend_subrect(( AVPicture * )vp->bmp, &sp->sub.rects[i],
                                      lImgWidth, lImgHeight);
                }
            }
        }

        if( !use_subimg )
        {
            hps = WinGetPS( hwndKMP );

            if( is->use_sub )
                subErase( hps, COLOR_OVERLAY );

            osdErase( hps, COLOR_OVERLAY );
        }
        else if( !is->paint_in_pause )
        {
            w = lImgWidth;
            h = lImgHeight;

            if( is->use_sub )
            {
                ULONG ulRefClock = get_master_clock( is ) * 100;

                if( subInTime( ulRefClock ))
                    subDisplay( NULLHANDLE, vp->bmp->data[ 0 ], vp->bmp->linesize[ 0 ], h,
                                SUB_MARGIN_LEFT, SUB_MARGIN_BOTTOM( h ),
                                w - ( SUB_MARGIN_LEFT + SUB_MARGIN_RIGHT ),
                                h - SUB_MARGIN_BOTTOM( h ) * 2,
                                sub_color_font, sub_color_outline, sub_color_shadow, TRUE );
                else
                    subFindNext( ulRefClock );
            }

            if( osdOn())
            {
                osdDisplay( NULLHANDLE, vp->bmp->data[ 0 ], vp->bmp->linesize[ 0 ], h,
                            OSD_MARGIN_LEFT, h - OSD_MARGIN_TOP( h ),
                            w - ( OSD_MARGIN_LEFT + OSD_MARGIN_RIGHT ),
                            h - OSD_MARGIN_TOP( h ) * 2,
                            osd_color_font, osd_color_outline, osd_color_shadow, TRUE );
            }
        }

        imgDisplayYUV(( YUV * )vp->bmp );

        if( !use_subimg )
        {
            w = is->width;
            h = is->height;

            if( is->use_sub )
            {
                ULONG ulRefClock = get_master_clock( is ) * 100;

                if( subInTime( ulRefClock ))
                    subDisplay( hps, NULL, 0, 0,
                                SUB_MARGIN_LEFT, SUB_MARGIN_BOTTOM( h ),
                                w - ( SUB_MARGIN_LEFT + SUB_MARGIN_RIGHT ),
                                h - SUB_MARGIN_BOTTOM( h ) * 2,
                                sub_color_font, sub_color_outline, sub_color_shadow, TRUE );
                else
                    subFindNext( ulRefClock );
            }

            if( osdOn())
                osdDisplay( hps, NULL, 0, 0,
                            OSD_MARGIN_LEFT, h - OSD_MARGIN_TOP( h ),
                            w - ( OSD_MARGIN_LEFT + OSD_MARGIN_RIGHT ),
                            h - OSD_MARGIN_TOP( h ) * 2,
                            osd_color_font, osd_color_outline, osd_color_shadow, TRUE );

            WinReleasePS( hps );
        }
    }
}

static inline int compute_mod(int a, int b)
{
    a = a % b;
    if (a >= 0)
        return a;
    else
        return a + b;
}

#define RGB( r, g, b ) ((( r ) << 16 ) | (( g ) << 8 ) | b )

static void video_audio_display(VideoState *s)
{
    int i, i_start, x, y1, y, ys, delay, n, nb_display_channels;
    int ch, channels, h, h2, bgcolor, fgcolor;
    int16_t time_diff;
    HPS hps;

    /* compute display index : center on currently output samples */
    channels = s->audio_st->codec->channels;
    nb_display_channels = channels;
    if (!s->paused) {
        n = 2 * channels;
        delay = audio_write_get_buf_size(s);
        delay /= n;

        /* to be more precise, we take into account the time spent since
           the last buffer computation */
        if (audio_callback_time) {
            time_diff = av_gettime() - audio_callback_time;
            delay += (time_diff * s->audio_st->codec->sample_rate) / 1000000;
        }

        delay -= s->width / 2;
        if (delay < s->width)
            delay = s->width;
        i_start= x = compute_mod(s->sample_array_index - delay * channels, SAMPLE_ARRAY_SIZE);

        h= INT_MIN;
        for(i=0; i<1000; i+=channels){
            int idx= (SAMPLE_ARRAY_SIZE + x - i) % SAMPLE_ARRAY_SIZE;
            int a= s->sample_array[idx];
            int b= s->sample_array[(idx + 4*channels)%SAMPLE_ARRAY_SIZE];
            int c= s->sample_array[(idx + 5*channels)%SAMPLE_ARRAY_SIZE];
            int d= s->sample_array[(idx + 9*channels)%SAMPLE_ARRAY_SIZE];
            int score= a-d;
            if(h<score && (b^c)<0){
                h= score;
                i_start= idx;
            }
        }

        s->last_i_start = i_start;
    } else {
        i_start = s->last_i_start;
    }

    hps = WinGetPS( hwndKMP );

    osdErase( hps, COLOR_OVERLAY );

    GpiCreateLogColorTable( hps, 0, LCOLF_RGB, 0, 0, NULL );

    bgcolor = RGB( 0x00, 0x00, 0x00 );
    fill_rectangle( hps, s->xleft, s->ytop, s->width, s->height, bgcolor );

    fgcolor = RGB( 0xff, 0xff, 0xff );

    /* total height for one channel */
    h = s->height / nb_display_channels;
    /* graph height / 2 */
    h2 = (h * 9) / 20;
    for(ch = 0;ch < nb_display_channels; ch++) {
        i = i_start + ch;
        y1 = s->ytop + ch * h + (h / 2); /* position of center line */
        for(x = 0; x < s->width; x++) {
            y = (s->sample_array[i] * h2) >> 15;
            if (y < 0) {
                y = -y;
                ys = y1 - y;
            } else {
                ys = y1;
            }
            fill_rectangle( hps, s->xleft + x, ys, 1, y, fgcolor );
            i += channels;
            if (i >= SAMPLE_ARRAY_SIZE)
                i -= SAMPLE_ARRAY_SIZE;
        }
    }

    fgcolor = RGB( 0x00, 0x00, 0xff );

    for(ch = 1;ch < nb_display_channels; ch++) {
        y = s->ytop + ch * h;
        fill_rectangle( hps, s->xleft, y, s->width, 1, fgcolor );
    }

    if( osdOn())
        osdDisplay( hps, NULL, 0, 0,
                    OSD_MARGIN_LEFT, s->height - OSD_MARGIN_TOP( s->height ),
                    s->width - ( OSD_MARGIN_LEFT + OSD_MARGIN_RIGHT ),
                    s->height - OSD_MARGIN_TOP( s->height ) * 2,
                    osd_color_font, osd_color_outline, osd_color_shadow, TRUE );

    WinReleasePS( hps );
}

static int video_open(VideoState *is)
{
    //ULONG   fl = SWP_MOVE | SWP_SIZE | SWP_ZORDER | SWP_ACTIVATE | SWP_SHOW | SWP_RESTORE;
    ULONG   fl = SWP_MOVE | SWP_SIZE | SWP_ZORDER | SWP_ACTIVATE | SWP_SHOW;
    RECTL   rcl;
    int     w, h;
    int     x, y;

    // process child window
    if( is_full_screen )
    {
        // when -fs option is used without this, title bar is not highlighted
        WinSetActiveWindow( HWND_DESKTOP, hwndFrame );

        WinSetParent( hwndSysMenu, HWND_OBJECT, FALSE );
        WinSetParent( hwndTitleBar, HWND_OBJECT, FALSE );
        WinSetParent( hwndMinMax, HWND_OBJECT, FALSE );

        if( hide_mouse )
            WinStartTimer( hab, hwndKMP, TID_MOUSE_HIDE, INTERVAL_MOUSE_HIDE );
    }
    else
    {
        WinSetParent( hwndSysMenu, hwndFrame, FALSE );
        WinSetParent( hwndTitleBar, hwndFrame, FALSE );
        WinSetParent( hwndMinMax, hwndFrame, FALSE );

        if( hide_mouse && !fMouseShow )
            SHOW_POINTER( TRUE );
    }
    //WinSendMsg( hwndFrame, WM_UPDATEFRAME, ( MPARAM )( FCF_SYSMENU | FCF_TITLEBAR | FCF_MINMAX ), 0 );

    // determine width and height of window
    if( is_full_screen )
    {
        w = cxScreen;
        h = cyScreen;
    }
    else if( screen_width )
    {
        w = screen_width;
        h = screen_height;
    }
    else if( is->video_st && is->video_st->codec->width )
    {
        w = lImgWidth;
        h = lImgHeight;

        // workaround for T23 laptop with S3 Video by Franz Bakan
        if( fix_t23 )
        {
            w++;
            h++;
        }
    }
    else
    {
        w = 640;
        h = 480;
    }

    rcl.xLeft = 0;
    rcl.yBottom = 0;
    rcl.xRight = w;
    rcl.yTop = h;

    WinCalcFrameRect( hwndFrame, &rcl, FALSE );

    x = ( cxScreen - ( rcl.xRight - rcl.xLeft )) / 2;
    y = ( cyScreen - ( rcl.yTop - rcl.yBottom )) / 2;

    WinSetWindowPos( hwndFrame, HWND_TOP, x, y,
                     rcl.xRight - rcl.xLeft, rcl.yTop - rcl.yBottom,
                     fl );

    // Now, is->width and is->height is set by WM_SIZE
#if 0
    is->width = w;
    is->height = h;
#endif

    if( is->use_sub )
        subInvalidate();

    fVideoOpened = TRUE;

    return 0;
}

/* display the current picture, if any */
static void video_display(VideoState *is)
{
    if( !fVideoOpened )
        video_open( is );

    if (is->audio_st && is->show_audio)
        video_audio_display(is);
    else if (is->video_st)
        video_image_display(is);
}


static void refresh_timer_cb( void *arg )
{
    WinPostMsg( hwndKMP, WM_REFRESH_EVENT, arg, 0 );
}

/* schedule a video refresh in 'delay' ms */
static void schedule_refresh(VideoState *is, int delay)
{
    tmrAsyncCall( delay, refresh_timer_cb, is );
}

/* get the current audio clock value */
static double get_audio_clock(VideoState *is)
{
    double pts;
    int hw_buf_size, bytes_per_sec;
    pts = is->audio_clock;
    hw_buf_size = audio_write_get_buf_size(is);
    bytes_per_sec = 0;
    if (is->audio_st) {
        bytes_per_sec = is->audio_st->codec->sample_rate *
            2 * is->audio_st->codec->channels;
    }
    if (bytes_per_sec)
        pts -= (double)hw_buf_size / bytes_per_sec;
    return pts;
}

/* get the current video clock value */
static double get_video_clock(VideoState *is)
{
    double delta;
    if (is->paused) {
        delta = 0;
    } else {
        delta = (av_gettime() - is->video_current_pts_time) / 1000000.0;
    }
    return is->video_current_pts + delta;
}

/* get the current external clock value */
static double get_external_clock(VideoState *is)
{
    int64_t ti;
    ti = av_gettime();
    return is->external_clock + ((ti - is->external_clock_time) * 1e-6);
}

/* get the current master clock value */
static double get_master_clock(VideoState *is)
{
    double val;

    if (is->av_sync_type == AV_SYNC_VIDEO_MASTER) {
        if (is->video_st)
            val = get_video_clock(is);
        else
            val = get_audio_clock(is);
    } else if (is->av_sync_type == AV_SYNC_AUDIO_MASTER) {
        if (is->audio_st)
            val = get_audio_clock(is);
        else
            val = get_video_clock(is);
    } else {
        val = get_external_clock(is);
    }
    return val;
}

/* seek in the stream */
static void stream_seek(VideoState *is, int64_t pos, int rel)
{
    if (!is->seek_req) {
        is->seek_pos = pos;
        is->seek_flags = rel < 0 ? AVSEEK_FLAG_BACKWARD : 0;
        if (seek_by_bytes)
            is->seek_flags |= AVSEEK_FLAG_BYTE;
        is->seek_req = 1;
    }
}

/* pause or resume the video */
static void stream_pause(VideoState *is)
{
    is->paused = !is->paused;
    if (!is->paused) {
        is->video_current_pts = get_video_clock(is);
        is->frame_timer += (av_gettime() - is->video_current_pts_time) / 1000000.0;
    }
}

/* called to display each frame */
static void video_refresh_timer(void *opaque)
{
    VideoState *is = opaque;
    VideoPicture *vp;
    double actual_delay, delay, sync_threshold, ref_clock, diff;

    SubPicture *sp, *sp2;

    if (is->video_st) {
        if (is->pictq_size == 0) {
            /* if no picture, need to wait */
            schedule_refresh(is, 1);
        } else {
            /* dequeue the picture */
            vp = &is->pictq[is->pictq_rindex];

            /* update current video pts */
            is->video_current_pts = vp->pts;
            is->video_current_pts_time = av_gettime();

            /* compute nominal delay */
            delay = vp->pts - is->frame_last_pts;
            if (delay <= 0 || delay >= 2.0) {
                /* if incorrect delay, use previous one */
                delay = is->frame_last_delay;
            }
            is->frame_last_delay = delay;
            is->frame_last_pts = vp->pts;

            /* update delay to follow master synchronisation source */
            if (((is->av_sync_type == AV_SYNC_AUDIO_MASTER && is->audio_st) ||
                 is->av_sync_type == AV_SYNC_EXTERNAL_CLOCK)) {
                /* if video is slave, we try to correct big delays by
                   duplicating or deleting a frame */
                ref_clock = get_master_clock(is);
                diff = vp->pts - ref_clock;

                /* skip or repeat frame. We take into account the
                   delay to compute the threshold. I still don't know
                   if it is the best guess */
                sync_threshold = FFMAX(AV_SYNC_THRESHOLD, delay);
                if (fabs(diff) < AV_NOSYNC_THRESHOLD) {
                    if (diff <= -sync_threshold)
                        delay = 0;
                    else if (diff >= sync_threshold)
                        delay = 2 * delay;
                }
            }

            is->frame_timer += delay;
            /* compute the REAL delay (we need to do that to avoid
               long term errors */
            actual_delay = is->frame_timer - (av_gettime() / 1000000.0);
            if (actual_delay < 0.010) {
                /* XXX: should skip picture */
                actual_delay = 0.010;
            }
            /* launch timer for next picture */
            schedule_refresh(is, (int)(actual_delay * 1000 + 0.5));

#if defined(DEBUG_SYNC)
            printf("video: delay=%0.3f actual_delay=%0.3f pts=%0.3f A-V=%f\n",
                   delay, actual_delay, vp->pts, -diff);
#endif

            if(is->subtitle_st) {
                if (is->subtitle_stream_changed) {
                    LockMutex(is->subpq_mutex);

                    while (is->subpq_size) {
                        free_subpicture(&is->subpq[is->subpq_rindex]);

                        /* update queue size and signal for next picture */
                        if (++is->subpq_rindex == SUBPICTURE_QUEUE_SIZE)
                            is->subpq_rindex = 0;

                        is->subpq_size--;
                    }
                    is->subtitle_stream_changed = 0;

                    CondSignal(is->subpq_cond);
                    UnlockMutex(is->subpq_mutex);
                } else {
                    if (is->subpq_size > 0) {
                        sp = &is->subpq[is->subpq_rindex];

                        if (is->subpq_size > 1)
                            sp2 = &is->subpq[(is->subpq_rindex + 1) % SUBPICTURE_QUEUE_SIZE];
                        else
                            sp2 = NULL;

                        if ((is->video_current_pts > (sp->pts + ((float) sp->sub.end_display_time / 1000)))
                                || (sp2 && is->video_current_pts > (sp2->pts + ((float) sp2->sub.start_display_time / 1000))))
                        {
                            free_subpicture(sp);

                            /* update queue size and signal for next picture */
                            if (++is->subpq_rindex == SUBPICTURE_QUEUE_SIZE)
                                is->subpq_rindex = 0;

                            LockMutex(is->subpq_mutex);
                            is->subpq_size--;
                            CondSignal(is->subpq_cond);
                            UnlockMutex(is->subpq_mutex);
                        }
                    }
                }
            }

            /* display picture if frame_drop is not set */
            if( !frame_drop || ( actual_delay > 0.010 ))
                video_display(is);

            /* update queue size and signal for next picture */
            if (++is->pictq_rindex == VIDEO_PICTURE_QUEUE_SIZE)
                is->pictq_rindex = 0;

            LockMutex(is->pictq_mutex);
            is->pictq_size--;
            CondSignal(is->pictq_cond);
            UnlockMutex(is->pictq_mutex);
        }
    } else if (is->audio_st) {
        /* draw the next audio frame */

        schedule_refresh(is, 40);

        /* if only audio stream, then display the audio bars (better
           than nothing, just to test the implementation */

        /* display picture if not paused*/
        if( !is->paused )
            video_display(is);
    } else {
        schedule_refresh(is, 100);
    }
    if (show_status) {
        static int64_t last_time;
        int64_t cur_time;
        int aqsize, vqsize, sqsize;
        double av_diff;

        cur_time = av_gettime();
        if (!last_time || (cur_time - last_time) >= 500 * 1000) {
            aqsize = 0;
            vqsize = 0;
            sqsize = 0;
            if (is->audio_st)
                aqsize = is->audioq.size;
            if (is->video_st)
                vqsize = is->videoq.size;
            if (is->subtitle_st)
                sqsize = is->subtitleq.size;
            av_diff = 0;
            if (is->audio_st && is->video_st)
                av_diff = get_audio_clock(is) - get_video_clock(is);
            printf("%7.2f A-V:%7.3f aq=%5dKB vq=%5dKB sq=%5dB    \r",
                   get_master_clock(is), av_diff, aqsize / 1024, vqsize / 1024, sqsize);
            fflush(stdout);
            last_time = cur_time;
        }
    }
}

/**
 *
 * @param pts the dts of the pkt / pts of the frame and guessed if not known
 */
static int queue_picture(VideoState *is, AVFrame *src_frame, double pts)
{
    VideoPicture *vp;
    static struct SwsContext *img_convert_ctx;
    static enum PixelFormat src_pix_fmt;

if( use_subimg )
{
    /* wait until we have space to put a new picture */
    LockMutex(is->pictq_mutex);
    while (is->pictq_size >= VIDEO_PICTURE_QUEUE_SIZE &&
           !is->videoq.abort_request) {
        CondWait(is->pictq_cond, is->pictq_mutex);
    }
    UnlockMutex(is->pictq_mutex);
}

    if (is->videoq.abort_request)
        return -1;

    // Init video here, because some videos does not fill width and height
    // when video_thread() called
    if( lImgWidth == 0 || lImgHeight == 0 )
    {
        lImgWidth = is->video_st->codec->width;
        lImgHeight = is->video_st->codec->height;

        if( fix_t23 )
        {
            while( lImgWidth > cxScreen )
            {
                lImgWidth >>= 1;
                lImgHeight >>= 1;
            }

            while( lImgHeight > cyScreen )
            {
                lImgWidth >>= 1;
                lImgHeight >>= 1;
            }
        }

        if( fix_snap )
        {
            while( lImgWidth >= cxScreen )
            {
                lImgWidth >>= 1;
                lImgHeight >>= 1;
            }

            while( lImgHeight >= cyScreen )
            {
                lImgWidth >>= 1;
                lImgHeight >>= 1;
            }
        }

        // treats PIX_FMT_YUVJ420P as PIX_FMT_YUV420P
        src_pix_fmt = is->video_st->codec->pix_fmt != PIX_FMT_YUVJ420P ?
                      is->video_st->codec->pix_fmt :  PIX_FMT_YUV420P;

        if( imgInit( video_driver, hwndKMP, COLOR_OVERLAY,
                     lImgWidth, lImgHeight,
                     aspect_ratio, aulAttrValue ))
        {
            fprintf( stderr, "Video setup failed!!!\n");

            is->abort_request = 1;

            return -1;
        }
    }

    vp = &is->pictq[is->pictq_windex];

    if( vp->bmp )
    {
        av_free( vp->bmp );

        if( vp->convBuf )
            imgFreeYUV( vp->convBuf );
    }

    /* if the picture format is not PIX_FMT_YUV420P, then convert it */
    if( is->video_st->codec->width != lImgWidth ||
        is->video_st->codec->height != lImgHeight ||
        src_pix_fmt != PIX_FMT_YUV420P ||
        use_subimg )
    {
        AVFrame *conv_frame = avcodec_alloc_frame();

        vp->convBuf = imgCreateYUV();

        conv_frame->data[ 0 ] = vp->convBuf->data[ 0 ];
        conv_frame->data[ 1 ] = vp->convBuf->data[ 1 ];
        conv_frame->data[ 2 ] = vp->convBuf->data[ 2 ];

        conv_frame->linesize[ 0 ] = vp->convBuf->linesize[ 0 ];
        conv_frame->linesize[ 1 ] = vp->convBuf->linesize[ 1 ];
        conv_frame->linesize[ 2 ] = vp->convBuf->linesize[ 2 ];

        sws_flags = av_get_int(sws_opts, "sws_flags", NULL);
        img_convert_ctx = sws_getCachedContext(img_convert_ctx,
            is->video_st->codec->width, is->video_st->codec->height, src_pix_fmt,
            lImgWidth, lImgHeight, PIX_FMT_YUV420P,
            sws_flags, NULL, NULL, NULL);
        if (img_convert_ctx == NULL) {
            fprintf(stderr, "Cannot initialize the conversion context\n");
            exit(1);
        }
        sws_scale(img_convert_ctx, src_frame->data, src_frame->linesize,
                  0, is->video_st->codec->height, vp->convBuf->data, vp->convBuf->linesize );
        emms_c();

        av_free( src_frame );

        vp->bmp = conv_frame;
    }
    else
        vp->bmp = src_frame;

    vp->pts = pts;

    /* now we can update the picture count */
    if (++is->pictq_windex == VIDEO_PICTURE_QUEUE_SIZE)
        is->pictq_windex = 0;
    LockMutex(is->pictq_mutex);
    is->pictq_size++;
    UnlockMutex(is->pictq_mutex);

    return 0;
}

/**
 * compute the exact PTS for the picture if it is omitted in the stream
 * @param pts1 the dts of the pkt / pts of the frame
 */
static int output_picture2(VideoState *is, AVFrame *src_frame, double pts1)
{
    double frame_delay, pts;

    pts = pts1;

    if (pts != 0) {
        /* update video clock with pts, if present */
        is->video_clock = pts;
    } else {
        pts = is->video_clock;
    }
    /* update video clock for next frame */
    frame_delay = av_q2d(is->video_st->codec->time_base);
    /* for MPEG2, the frame can be repeated, so we update the
       clock accordingly */
    frame_delay += src_frame->repeat_pict * (frame_delay * 0.5);
    is->video_clock += frame_delay;

#if defined(DEBUG_SYNC) && 0
    {
        int ftype;
        if (src_frame->pict_type == FF_B_TYPE)
            ftype = 'B';
        else if (src_frame->pict_type == FF_I_TYPE)
            ftype = 'I';
        else
            ftype = 'P';
        printf("frame_type=%c clock=%0.3f pts=%0.3f\n",
               ftype, pts, pts1);
    }
#endif
    return queue_picture(is, src_frame, pts);
}

static void audio_thread( void *arg )
{
    dartSetVolume( MCI_SET_AUDIO_ALL, volume_level );

    dartPlay();

    while( DART.fPlaying )
        tmrDelay( 1 );
}

static void video_thread(void *arg)
{
    VideoState *is = arg;
    AVPacket pkt1, *pkt = &pkt1;
    int len1, got_picture;
    AVFrame *frame;
    double pts;

    for(;;) {
        while (is->paused && !is->videoq.abort_request) {
            tmrDelay(10);
        }
        if (packet_queue_get(&is->videoq, pkt, 1) < 0)
            break;

        if(pkt->data == flush_pkt.data){
            avcodec_flush_buffers(is->video_st->codec);
            continue;
        }

        if( !use_subimg )
        {
            /* wait until we have space to put a new picture */
            LockMutex(is->pictq_mutex);
            while (is->pictq_size >= VIDEO_PICTURE_QUEUE_SIZE &&
                   !is->videoq.abort_request) {
                CondWait(is->pictq_cond, is->pictq_mutex);
            }
            UnlockMutex(is->pictq_mutex);
        }

        /* NOTE: ipts is the PTS of the _first_ picture beginning in
           this packet, if any */
        is->video_st->codec->reordered_opaque= pkt->pts;

        frame = avcodec_alloc_frame();

        len1 = avcodec_decode_video(is->video_st->codec,
                                    frame, &got_picture,
                                    pkt->data, pkt->size);

        if(   (decoder_reorder_pts || pkt->dts == AV_NOPTS_VALUE)
           && frame->reordered_opaque != AV_NOPTS_VALUE)
            pts= frame->reordered_opaque;
        else if(pkt->dts != AV_NOPTS_VALUE)
            pts= pkt->dts;
        else
            pts= 0;
        pts *= av_q2d(is->video_st->time_base);

        av_free_packet(pkt);

//      if (len1 < 0)
//          break;
        if (got_picture) {
            if (output_picture2(is, frame, pts) < 0)
                goto the_end;
        }
        if (step)
            if (cur_stream)
                stream_pause(cur_stream);
    }
 the_end:
    imgDone();
}

static void subtitle_thread(void *arg)
{
    VideoState *is = arg;
    SubPicture *sp;
    AVPacket pkt1, *pkt = &pkt1;
    int len1, got_subtitle;
    double pts;
    int i, j;
    int r, g, b, y, u, v, a;

    for(;;) {
        while (is->paused && !is->subtitleq.abort_request) {
            tmrDelay(10);
        }
        if (packet_queue_get(&is->subtitleq, pkt, 1) < 0)
            break;

        if(pkt->data == flush_pkt.data){
            avcodec_flush_buffers(is->subtitle_st->codec);
            continue;
        }
        LockMutex(is->subpq_mutex);
        while (is->subpq_size >= SUBPICTURE_QUEUE_SIZE &&
               !is->subtitleq.abort_request) {
            CondWait(is->subpq_cond, is->subpq_mutex);
        }
        UnlockMutex(is->subpq_mutex);

        if (is->subtitleq.abort_request)
            goto the_end;

        sp = &is->subpq[is->subpq_windex];

       /* NOTE: ipts is the PTS of the _first_ picture beginning in
           this packet, if any */
        pts = 0;
        if (pkt->pts != AV_NOPTS_VALUE)
            pts = av_q2d(is->subtitle_st->time_base)*pkt->pts;

        len1 = avcodec_decode_subtitle(is->subtitle_st->codec,
                                    &sp->sub, &got_subtitle,
                                    pkt->data, pkt->size);
//            if (len1 < 0)
//                break;
        if (got_subtitle && sp->sub.format == 0) {
            sp->pts = pts;

            for (i = 0; i < sp->sub.num_rects; i++)
            {
                for (j = 0; j < sp->sub.rects[i].nb_colors; j++)
                {
                    RGBA_IN(r, g, b, a, sp->sub.rects[i].rgba_palette + j);
                    y = RGB_TO_Y_CCIR(r, g, b);
                    u = RGB_TO_U_CCIR(r, g, b, 0);
                    v = RGB_TO_V_CCIR(r, g, b, 0);
                    YUVA_OUT(sp->sub.rects[i].rgba_palette + j, y, u, v, a);
                }
            }

            /* now we can update the picture count */
            if (++is->subpq_windex == SUBPICTURE_QUEUE_SIZE)
                is->subpq_windex = 0;
            LockMutex(is->subpq_mutex);
            is->subpq_size++;
            UnlockMutex(is->subpq_mutex);
        }
        av_free_packet(pkt);
//        if (step)
//            if (cur_stream)
//                stream_pause(cur_stream);
    }
 the_end:

 return;
}

/* copy samples for viewing in editor window */
static void update_sample_display(VideoState *is, short *samples, int samples_size)
{
    int size, len, channels;

    channels = is->audio_st->codec->channels;

    size = samples_size / sizeof(short);
    while (size > 0) {
        len = SAMPLE_ARRAY_SIZE - is->sample_array_index;
        if (len > size)
            len = size;
        fast_memcpy(is->sample_array + is->sample_array_index, samples, len * sizeof(short));
        samples += len;
        is->sample_array_index += len;
        if (is->sample_array_index >= SAMPLE_ARRAY_SIZE)
            is->sample_array_index = 0;
        size -= len;
    }
}

/* return the new audio buffer size (samples can be added or deleted
   to get better sync if video or external master clock) */
static int synchronize_audio(VideoState *is, short *samples,
                             int samples_size1, double pts)
{
    int n, samples_size;
    double ref_clock;

    n = 2 * is->audio_st->codec->channels;
    samples_size = samples_size1;

    /* if not master, then we try to remove or add samples to correct the clock */
    if (((is->av_sync_type == AV_SYNC_VIDEO_MASTER && is->video_st) ||
         is->av_sync_type == AV_SYNC_EXTERNAL_CLOCK)) {
        double diff, avg_diff;
        int wanted_size, min_size, max_size, nb_samples;

        ref_clock = get_master_clock(is);
        diff = get_audio_clock(is) - ref_clock;

        if (diff < AV_NOSYNC_THRESHOLD) {
            is->audio_diff_cum = diff + is->audio_diff_avg_coef * is->audio_diff_cum;
            if (is->audio_diff_avg_count < AUDIO_DIFF_AVG_NB) {
                /* not enough measures to have a correct estimate */
                is->audio_diff_avg_count++;
            } else {
                /* estimate the A-V difference */
                avg_diff = is->audio_diff_cum * (1.0 - is->audio_diff_avg_coef);

                if (fabs(avg_diff) >= is->audio_diff_threshold) {
                    wanted_size = samples_size + ((int)(diff * is->audio_st->codec->sample_rate) * n);
                    nb_samples = samples_size / n;

                    min_size = ((nb_samples * (100 - SAMPLE_CORRECTION_PERCENT_MAX)) / 100) * n;
                    max_size = ((nb_samples * (100 + SAMPLE_CORRECTION_PERCENT_MAX)) / 100) * n;
                    if (wanted_size < min_size)
                        wanted_size = min_size;
                    else if (wanted_size > max_size)
                        wanted_size = max_size;

                    /* add or remove samples to correction the synchro */
                    if (wanted_size < samples_size) {
                        /* remove samples */
                        samples_size = wanted_size;
                    } else if (wanted_size > samples_size) {
                        uint8_t *samples_end, *q;
                        int nb;

                        /* add samples */
                        nb = (samples_size - wanted_size);
                        samples_end = (uint8_t *)samples + samples_size - n;
                        q = samples_end + n;
                        while (nb > 0) {
                            fast_memcpy(q, samples_end, n);
                            q += n;
                            nb -= n;
                        }
                        samples_size = wanted_size;
                    }
                }
#if 0
                printf("diff=%f adiff=%f sample_diff=%d apts=%0.3f vpts=%0.3f %f\n",
                       diff, avg_diff, samples_size - samples_size1,
                       is->audio_clock, is->video_clock, is->audio_diff_threshold);
#endif
            }
        } else {
            /* too big difference : may be initial PTS errors, so
               reset A-V filter */
            is->audio_diff_avg_count = 0;
            is->audio_diff_cum = 0;
        }
    }

    return samples_size;
}

/* decode one audio frame and returns its uncompressed size */
static int audio_decode_frame(VideoState *is, double *pts_ptr)
{
    AVPacket *pkt = &is->audio_pkt;
    AVCodecContext *dec= is->audio_st->codec;
    int n, len1, data_size;
    double pts;

    for(;;) {
        /* NOTE: the audio packet can contain several frames */
        while (is->audio_pkt_size > 0) {
            data_size = sizeof(is->audio_buf1);
            len1 = avcodec_decode_audio2(dec,
                                        (int16_t *)is->audio_buf1, &data_size,
                                        is->audio_pkt_data, is->audio_pkt_size);
             if (len1 < 0) {
                /* if error, we skip the frame */
                is->audio_pkt_size = 0;
                break;
            }

            is->audio_pkt_data += len1;
            is->audio_pkt_size -= len1;
            if (data_size <= 0)
                continue;

            if (dec->sample_fmt != is->audio_src_fmt) {
                if (is->reformat_ctx)
                    av_audio_convert_free(is->reformat_ctx);
                is->reformat_ctx= av_audio_convert_alloc(SAMPLE_FMT_S16, 1,
                                                         dec->sample_fmt, 1, NULL, 0);
                if (!is->reformat_ctx) {
                    fprintf(stderr, "Cannot convert %s sample format to %s sample format\n",
                        avcodec_get_sample_fmt_name(dec->sample_fmt),
                        avcodec_get_sample_fmt_name(SAMPLE_FMT_S16));
                        break;
                }
                is->audio_src_fmt= dec->sample_fmt;
            }

            if (is->reformat_ctx) {
                const void *ibuf[6]= {is->audio_buf1};
                void *obuf[6]= {is->audio_buf2};
                int istride[6]= {av_get_bits_per_sample_format(dec->sample_fmt)/8};
                int ostride[6]= {2};
                int len= data_size/istride[0];
                if (av_audio_convert(is->reformat_ctx, obuf, ostride, ibuf, istride, len)<0) {
                    printf("av_audio_convert() failed\n");
                    break;
                }
                is->audio_buf= is->audio_buf2;
                /* FIXME: existing code assume that data_size equals framesize*channels*2
                          remove this legacy cruft */
                data_size= len*2;
            }else{
                is->audio_buf= is->audio_buf1;
            }

            /* if no pts, then compute it */
            pts = is->audio_clock;
            *pts_ptr = pts;
            n = 2 * dec->channels;
            is->audio_clock += (double)data_size /
                (double)(n * dec->sample_rate);
#if defined(DEBUG_SYNC)
            {
                static double last_clock;
                printf("audio: delay=%0.3f clock=%0.3f pts=%0.3f\n",
                       is->audio_clock - last_clock,
                       is->audio_clock, pts);
                last_clock = is->audio_clock;
            }
#endif
            return data_size;
        }

        /* free the current packet */
        if (pkt->data)
            av_free_packet(pkt);

        if (is->paused || is->audioq.abort_request) {
            return -1;
        }

        /* read next packet */
        if (packet_queue_get(&is->audioq, pkt, 1) < 0)
            return -1;
        if(pkt->data == flush_pkt.data){
            avcodec_flush_buffers(dec);
            continue;
        }

        is->audio_pkt_data = pkt->data;
        is->audio_pkt_size = pkt->size;

        /* if update the audio clock with the pts */
        if (pkt->pts != AV_NOPTS_VALUE) {
            is->audio_clock = av_q2d(is->audio_st->time_base)*pkt->pts;
        }
    }
}

/* get the current audio output buffer size, in samples. With SDL, we
   cannot have a precise information */
static int audio_write_get_buf_size(VideoState *is)
{
    return is->audio_buf_size - is->audio_buf_index;
}

/* prepare a new audio buffer */
static ULONG APIENTRY dart_audio_callback( PVOID pCBData, PVOID pBuffer, ULONG len)
{
    static DECLARE_ALIGNED(16,uint8_t,res_audio_buf[(AVCODEC_MAX_AUDIO_FRAME_SIZE * 3) / 2]);

    VideoState *is = pCBData;
    uint8_t *stream = pBuffer;
    int audio_size, len1;
    double pts;

    audio_callback_time = av_gettime();

    while (len > 0) {
        if (is->audio_buf_index >= is->audio_buf_size) {
           audio_size = audio_decode_frame(is, &pts);
           if (audio_size < 0) {
                /* if error, just output silence */
               is->audio_buf_size = 1024;
               memset(is->audio_buf, 0, is->audio_buf_size);
           } else {
               if( is->rsc )
               {
                   int n;
                   int nb_out_samples;

                   n = 2 * is->audio_st->codec->channels;
                   nb_out_samples = audio_resample( is->rsc,
                                                    ( short * )res_audio_buf,
                                                    ( short * )is->audio_buf,
                                                    audio_size / n );
                   audio_size = nb_out_samples * n;
                   fast_memcpy( is->audio_buf, res_audio_buf, audio_size );
               }

               if (is->show_audio)
                   update_sample_display(is, (int16_t *)is->audio_buf, audio_size);
               audio_size = synchronize_audio(is, (int16_t *)is->audio_buf, audio_size,
                                              pts);

               is->audio_buf_size = audio_size;
           }
           is->audio_buf_index = 0;
        }
        len1 = is->audio_buf_size - is->audio_buf_index;
        if (len1 > len)
            len1 = len;
        fast_memcpy(stream, (uint8_t *)is->audio_buf + is->audio_buf_index, len1);
        len -= len1;
        stream += len1;
        is->audio_buf_index += len1;
    }

    return ( stream - ( uint8_t * )pBuffer );
}

/* open a given stream. Return 0 if OK */
static int stream_component_open(VideoState *is, int stream_index)
{
    AVFormatContext *ic = is->ic;
    AVCodecContext *enc;
    AVCodec *codec;

    if (stream_index < 0 || stream_index >= ic->nb_streams)
        return -1;
    enc = ic->streams[stream_index]->codec;

    /* prepare audio output */
    if (enc->codec_type == CODEC_TYPE_AUDIO) {
        if (enc->channels > 0) {
            enc->request_channels = FFMIN(2, enc->channels);
        } else {
            enc->request_channels = 2;
        }

        if(( resample == RES_48KONLY ) && ( enc->sample_rate == 48000 ))
        {
            printf("Resample 48KHz to 44.1KHz\n");
            is->rsc = audio_resample_init( enc->channels, enc->channels, 44100, enc->sample_rate );
            enc->sample_rate = 44100;
        }
    }

    codec = avcodec_find_decoder(enc->codec_id);
    enc->debug_mv = debug_mv;
    enc->debug = debug;
    enc->workaround_bugs = workaround_bugs;
    enc->lowres = lowres;
    if(lowres) enc->flags |= CODEC_FLAG_EMU_EDGE;
    enc->idct_algo= idct;
    if(fast) enc->flags2 |= CODEC_FLAG2_FAST;
    enc->skip_frame= skip_frame;
    enc->skip_idct= skip_idct;
    enc->skip_loop_filter= skip_loop_filter;
    enc->error_recognition= error_recognition;
    enc->error_concealment= error_concealment;

    set_context_opts(enc, avctx_opts[enc->codec_type], 0);

    if (!codec ||
        avcodec_open(enc, codec) < 0)
        return -1;

    /* prepare audio output */
    if (enc->codec_type == CODEC_TYPE_AUDIO) {
        if( dartInit( 0, BPS_16, enc->sample_rate, MCI_WAVE_FORMAT_PCM, enc->channels,
                      2, audio_buffer_size ? audio_buffer_size : ( 2 * enc->channels * 2048 ),
                      !no_audio_share, dart_audio_callback, is ))
        {
            fprintf( stderr, "Cannot intialize DART\n");
            return -1;
        }

        if( no_audio_share )
            printf("Opened audio as exclusive mode\n");

        is->audio_hw_buf_size = DART.ulBufferSize;
        is->audio_src_fmt= SAMPLE_FMT_S16;
    }

    if(thread_count>1)
        avcodec_thread_init(enc, thread_count);
    enc->thread_count= thread_count;
    ic->streams[stream_index]->discard = AVDISCARD_DEFAULT;
    switch(enc->codec_type) {
    case CODEC_TYPE_AUDIO:
        is->audio_stream = stream_index;
        is->audio_st = ic->streams[stream_index];
        is->audio_buf_size = 0;
        is->audio_buf_index = 0;

        /* init averaging filter */
        is->audio_diff_avg_coef = exp(log(0.01) / AUDIO_DIFF_AVG_NB);
        is->audio_diff_avg_count = 0;
        /* since we do not have a precise anough audio fifo fullness,
           we correct audio sync only if larger than this threshold */
        //is->audio_diff_threshold = 2.0 * 1024 / enc->sample_rate;
        is->audio_diff_threshold = 2.0 * 1024 / enc->sample_rate;

        memset(&is->audio_pkt, 0, sizeof(is->audio_pkt));
        packet_queue_init(&is->audioq);
        is->audio_tid = CreateThread( audio_thread, is );
        break;
    case CODEC_TYPE_VIDEO:
        is->video_stream = stream_index;
        is->video_st = ic->streams[stream_index];

        is->frame_last_delay = 40e-3;
        is->frame_timer = (double)av_gettime() / 1000000.0;
        is->video_current_pts_time = av_gettime();

        packet_queue_init(&is->videoq);
        is->video_tid = CreateThread(video_thread, is);
        break;
    case CODEC_TYPE_SUBTITLE:
        is->subtitle_stream = stream_index;
        is->subtitle_st = ic->streams[stream_index];
        packet_queue_init(&is->subtitleq);

        is->subtitle_tid = CreateThread(subtitle_thread, is);
        break;
    default:
        break;
    }
    return 0;
}

static void stream_component_close(VideoState *is, int stream_index)
{
    AVFormatContext *ic = is->ic;
    AVCodecContext *enc;

    if (stream_index < 0 || stream_index >= ic->nb_streams)
        return;
    enc = ic->streams[stream_index]->codec;

    switch(enc->codec_type) {
    case CODEC_TYPE_AUDIO:
        packet_queue_abort(&is->audioq);

        dartClose();

        WaitThread( is->audio_tid );

        packet_queue_end(&is->audioq);
        if (is->reformat_ctx)
            av_audio_convert_free(is->reformat_ctx);
        if( is->rsc )
            audio_resample_close( is->rsc );
        break;
    case CODEC_TYPE_VIDEO:
        packet_queue_abort(&is->videoq);

        /* note: we also signal this mutex to make sure we deblock the
           video thread in all cases */
        LockMutex(is->pictq_mutex);
        CondSignal(is->pictq_cond);
        UnlockMutex(is->pictq_mutex);

        WaitThread( is->video_tid );

        packet_queue_end(&is->videoq);
        break;
    case CODEC_TYPE_SUBTITLE:
        packet_queue_abort(&is->subtitleq);

        /* note: we also signal this mutex to make sure we deblock the
           video thread in all cases */
        LockMutex(is->subpq_mutex);
        is->subtitle_stream_changed = 1;

        CondSignal(is->subpq_cond);
        UnlockMutex(is->subpq_mutex);

        WaitThread( is->subtitle_tid );

        packet_queue_end(&is->subtitleq);
        break;
    default:
        break;
    }

    ic->streams[stream_index]->discard = AVDISCARD_ALL;
    avcodec_close(enc);
    switch(enc->codec_type) {
    case CODEC_TYPE_AUDIO:
        is->audio_st = NULL;
        is->audio_stream = -1;
        break;
    case CODEC_TYPE_VIDEO:
        is->video_st = NULL;
        is->video_stream = -1;
        break;
    case CODEC_TYPE_SUBTITLE:
        is->subtitle_st = NULL;
        is->subtitle_stream = -1;
        break;
    default:
        break;
    }
}

static void dump_stream_info(const AVFormatContext *s)
{
    if (s->track != 0)
        fprintf(stderr, "Track: %d\n", s->track);
    if (s->title[0] != '\0')
        fprintf(stderr, "Title: %s\n", s->title);
    if (s->author[0] != '\0')
        fprintf(stderr, "Author: %s\n", s->author);
    if (s->copyright[0] != '\0')
        fprintf(stderr, "Copyright: %s\n", s->copyright);
    if (s->comment[0] != '\0')
        fprintf(stderr, "Comment: %s\n", s->comment);
    if (s->album[0] != '\0')
        fprintf(stderr, "Album: %s\n", s->album);
    if (s->year != 0)
        fprintf(stderr, "Year: %d\n", s->year);
    if (s->genre[0] != '\0')
        fprintf(stderr, "Genre: %s\n", s->genre);
}

/* since we have only one decoding thread, we can use a global
   variable instead of a thread local variable */
static VideoState *global_video_state;

static int decode_interrupt_cb(void)
{
    return (global_video_state && global_video_state->abort_request);
}

/* this thread gets the stream from the disk or the network */
static void decode_thread(void *arg)
{
    VideoState *is = arg;
    AVFormatContext *ic;
    int err, i, ret, video_index, audio_index;
    AVPacket pkt1, *pkt = &pkt1;
    AVFormatParameters params, *ap = &params;
    int empty_count = 0;

    video_index = -1;
    audio_index = -1;
    is->video_stream = -1;
    is->audio_stream = -1;
    is->subtitle_stream = -1;

    global_video_state = is;
    url_set_interrupt_cb(decode_interrupt_cb);

    memset(ap, 0, sizeof(*ap));

    ap->width = frame_width;
    ap->height= frame_height;
    ap->time_base= (AVRational){1, 25};
    ap->pix_fmt = frame_pix_fmt;

    err = av_open_input_file(&ic, is->filename, is->iformat, 0, ap);
    if (err < 0) {
        print_error(is->filename, err);
        ret = -1;
        goto fail;
    }
    is->ic = ic;

    if(genpts)
        ic->flags |= AVFMT_FLAG_GENPTS;

    err = av_find_stream_info(ic);
    if (err < 0) {
        fprintf(stderr, "%s: could not find codec parameters\n", is->filename);
        ret = -1;
        goto fail;
    }
    if(ic->pb)
        ic->pb->eof_reached= 0; //FIXME hack, kmp maybe should not use url_feof() to test for the end

    /* if seeking requested, we execute it */
    if (start_time != AV_NOPTS_VALUE) {
        int64_t timestamp;

        timestamp = start_time;
        /* add the stream start time */
        if (ic->start_time != AV_NOPTS_VALUE)
            timestamp += ic->start_time;
        ret = av_seek_frame(ic, -1, timestamp, AVSEEK_FLAG_BACKWARD);
        if (ret < 0) {
            fprintf(stderr, "%s: could not seek to position %0.3f\n",
                    is->filename, (double)timestamp / AV_TIME_BASE);
        }
    }

    for(i = 0; i < ic->nb_streams; i++) {
        AVCodecContext *enc = ic->streams[i]->codec;
        ic->streams[i]->discard = AVDISCARD_ALL;
        switch(enc->codec_type) {
        case CODEC_TYPE_AUDIO:
            if ((audio_index < 0 || wanted_audio_stream-- > 0) && !audio_disable)
                audio_index = i;
            break;
        case CODEC_TYPE_VIDEO:
            if ((video_index < 0 || wanted_video_stream-- > 0) && !video_disable)
            {
                video_index = i;

                /* init subtitle here to avoid latency */
                is->use_sub = !subInit( hab, is->filename, sub_fontnamesize,
                                        1 / av_q2d( ic->streams[ i ]->time_base ), use_subimg );
            }
            break;
        default:
            break;
        }
    }
    if (show_status) {
        dump_format(ic, 0, is->filename, 0);
        dump_stream_info(ic);
    }

    /* open the streams */
    if (audio_index >= 0) {
        stream_component_open(is, audio_index);
    }

    if (video_index >= 0) {
        stream_component_open(is, video_index);
    } else {
        if (!display_disable)
            is->show_audio = 1;
    }

    if (is->video_stream < 0 && is->audio_stream < 0) {
        fprintf(stderr, "%s: could not open codecs\n", is->filename);
        ret = -1;
        goto fail;
    }

    /* add the refresh timer to draw the picture */
    schedule_refresh(is, 40);

    for(;;) {
        if (is->abort_request)
            break;

        if( !is->audioq.size && !is->videoq.size && !is->subtitleq.size )
        {
            if( ++empty_count > 10 )
                break;
        }
        else
            empty_count = 0;

        if (is->paused != is->last_paused) {
            is->last_paused = is->paused;
            if (is->paused)
                av_read_pause(ic);
            else
                av_read_play(ic);
        }
#if defined(CONFIG_RTSP_DEMUXER) || defined(CONFIG_MMSH_PROTOCOL)
        if (is->paused &&
                (!strcmp(ic->iformat->name, "rtsp") ||
                 (ic->pb && !strcmp(url_fileno(ic->pb)->prot->name, "mmsh")))) {
            /* wait 10 ms to avoid trying to get another packet */
            /* XXX: horrible */
            tmrDelay(10);
            continue;
        }
#endif
        if (is->seek_req) {
            int stream_index= -1;
            int64_t seek_target= is->seek_pos;

            if     (is->   video_stream >= 0) stream_index= is->   video_stream;
            else if(is->   audio_stream >= 0) stream_index= is->   audio_stream;
            else if(is->subtitle_stream >= 0) stream_index= is->subtitle_stream;

            if(stream_index>=0){
                seek_target= av_rescale_q(seek_target, AV_TIME_BASE_Q, ic->streams[stream_index]->time_base);
            }

            ret = av_seek_frame(is->ic, stream_index, seek_target, is->seek_flags);
            if (ret < 0) {
                fprintf(stderr, "%s: error while seeking\n", is->ic->filename);
            }else{
                if (is->audio_stream >= 0) {
                    packet_queue_flush(&is->audioq);
                    packet_queue_put(&is->audioq, &flush_pkt);
                }
                if (is->subtitle_stream >= 0) {
                    packet_queue_flush(&is->subtitleq);
                    packet_queue_put(&is->subtitleq, &flush_pkt);
                }
                if (is->video_stream >= 0) {
                    packet_queue_flush(&is->videoq);
                    packet_queue_put(&is->videoq, &flush_pkt);
                }
            }
            is->seek_req = 0;
        }

        /* if the queue are full, no need to read more */
        if (is->audioq.size > MAX_AUDIOQ_SIZE ||
            is->videoq.size > MAX_VIDEOQ_SIZE ||
            is->subtitleq.size > MAX_SUBTITLEQ_SIZE ||
            url_feof(ic->pb)) {
            /* wait 10 ms */
            tmrDelay(10);
            continue;
        }
        ret = av_read_frame(ic, pkt);
        if (ret < 0) {
            if (url_ferror(ic->pb) == 0) {
                tmrDelay(100); /* wait for user event */
                continue;
            } else
                break;
        }
        if (pkt->stream_index == is->audio_stream) {
            packet_queue_put(&is->audioq, pkt);
        } else if (pkt->stream_index == is->video_stream) {
            packet_queue_put(&is->videoq, pkt);
        } else if (pkt->stream_index == is->subtitle_stream) {
            packet_queue_put(&is->subtitleq, pkt);
        } else {
            av_free_packet(pkt);
        }
    }

    //ret = 0;

 fail:
    /* disable interrupting */
    global_video_state = NULL;

    /* close each stream */
    if (is->audio_stream >= 0)
        stream_component_close(is, is->audio_stream);
    if (is->video_stream >= 0)
        stream_component_close(is, is->video_stream);
    if (is->subtitle_stream >= 0)
        stream_component_close(is, is->subtitle_stream);
    if (is->ic) {
        av_close_input_file(is->ic);
        is->ic = NULL; /* safety */
    }
    url_set_interrupt_cb(NULL);

    if( is->use_sub )
        subDone();

    //if (ret != 0)
        WinPostMsg( hwndKMP, WM_QUIT, 0, 0 );
}

static VideoState *stream_open(const char *filename, AVInputFormat *iformat)
{
    VideoState *is;

    is = av_mallocz(sizeof(VideoState));
    if (!is)
        return NULL;
    av_strlcpy(is->filename, filename, sizeof(is->filename));
    is->iformat = iformat;
    is->ytop = 0;
    is->xleft = 0;

    /* start video display */
    is->pictq_mutex = CreateMutex();
    is->pictq_cond = CreateCond();

    is->subpq_mutex = CreateMutex();
    is->subpq_cond = CreateCond();

    tmrInit();

#if 0
    /* add the refresh timer to draw the picture */
    schedule_refresh(is, 40);
#endif

    is->av_sync_type = av_sync_type;
    is->parse_tid = CreateThread(decode_thread, is);
    if (!is->parse_tid) {
        av_free(is);
        return NULL;
    }
    return is;
}

static void stream_close(VideoState *is)
{
    VideoPicture *vp;
    int i;
    /* XXX: use a special url_shutdown call to abort parse cleanly */
    is->abort_request = 1;
    WaitThread( is->parse_tid );

    /* free all pictures */
    for(i=0;i<VIDEO_PICTURE_QUEUE_SIZE; i++) {
        vp = &is->pictq[i];
        if (vp->bmp) {
            av_free( vp->bmp );
            vp->bmp = NULL;
            if( vp->convBuf )
            {
                imgFreeYUV( vp->convBuf );
                vp->convBuf = NULL;
            }
        }
    }

    tmrDone();

    DestroyMutex(is->pictq_mutex);
    DestroyCond(is->pictq_cond);
    DestroyMutex(is->subpq_mutex);
    DestroyCond(is->subpq_cond);
}

static void stream_cycle_channel(VideoState *is, int codec_type)
{
    AVFormatContext *ic = is->ic;
    int start_index, stream_index;
    AVStream *st;

    if (codec_type == CODEC_TYPE_VIDEO)
        start_index = is->video_stream;
    else if (codec_type == CODEC_TYPE_AUDIO)
        start_index = is->audio_stream;
    else
        start_index = is->subtitle_stream;
    if (start_index < (codec_type == CODEC_TYPE_SUBTITLE ? -1 : 0))
        return;
    stream_index = start_index;
    for(;;) {
        if (++stream_index >= is->ic->nb_streams)
        {
            if (codec_type == CODEC_TYPE_SUBTITLE)
            {
                stream_index = -1;
                goto the_end;
            } else
                stream_index = 0;
        }
        if (stream_index == start_index)
            return;
        st = ic->streams[stream_index];
        if (st->codec->codec_type == codec_type) {
            /* check that parameters are OK */
            switch(codec_type) {
            case CODEC_TYPE_AUDIO:
                if (st->codec->sample_rate != 0 &&
                    st->codec->channels != 0)
                    goto the_end;
                break;
            case CODEC_TYPE_VIDEO:
            case CODEC_TYPE_SUBTITLE:
                goto the_end;
            default:
                break;
            }
        }
    }
 the_end:
    stream_component_close(is, start_index);
    stream_component_open(is, stream_index);
}

static void toggle_full_screen(void)
{
    is_full_screen = !is_full_screen;
    video_open(cur_stream);
}

static void toggle_pause(void)
{
    if (cur_stream)
        stream_pause(cur_stream);
    step = 0;

}

static void step_to_next_frame(void)
{
    if (cur_stream) {
        /* if the stream is paused unpause it, then step */
        if (cur_stream->paused)
            stream_pause(cur_stream);
    }
    step = 1;
}

static void toggle_audio_display(void)
{
    if (cur_stream) {
        cur_stream->show_audio = !cur_stream->show_audio;
        if( !cur_stream->show_audio )
            imgClearRect( NULL );
    }
}

static void opt_video( const char *arg )
{
    if( !strcmp( arg, "auto" ))
        video_driver = KVAM_AUTO;
    else if( !strcmp( arg, "snap" ))
        video_driver = KVAM_SNAP;
    else if( !strcmp( arg, "wo" ))
        video_driver = KVAM_WO;
    else if( !strcmp( arg, "dive" ))
        video_driver = KVAM_DIVE;
    else
    {
        fprintf( stderr, "invalid video driver : %s\n", arg );
        exit( 1 );
    }
}

static void opt_frame_size(const char *arg)
{
    if (av_parse_video_frame_size(&frame_width, &frame_height, arg) < 0) {
        fprintf(stderr, "Incorrect frame size\n");
        exit(1);
    }
    if ((frame_width % 2) != 0 || (frame_height % 2) != 0) {
        fprintf(stderr, "Frame size must be a multiple of 2\n");
        exit(1);
    }
}

static int opt_width(const char *opt, const char *arg)
{
    opt_screen_width = parse_number_or_die(opt, arg, OPT_INT64, 1, INT_MAX);
    return 0;
}

static int opt_height(const char *opt, const char *arg)
{
    opt_screen_height = parse_number_or_die(opt, arg, OPT_INT64, 1, INT_MAX);
    return 0;
}

static void opt_volume( const char *arg )
{
    volume_level = atoi( arg );
    if( volume_level < 0 || volume_level > 100 )
    {
        fprintf( stderr, "invalid volume level : %s\n", arg );
        exit( 1 );
    }
}

static void opt_format(const char *arg)
{
    file_iformat = av_find_input_format(arg);
    if (!file_iformat) {
        fprintf(stderr, "Unknown input format: %s\n", arg);
        exit(1);
    }
}

static void opt_subfont( const char *arg )
{
    long size;
    char *facename;

    size = strtol( arg, &facename, 0 );
    if( !size || ( *facename != '.' ))
    {
        fprintf( stderr, "Invalid font name and size: %s. Use like 24.Gulim\n", arg );
        exit( 1 );
    }

    av_strlcpy( sub_fontnamesize, arg, sizeof( sub_fontnamesize ));
}

static void opt_osdfont( const char *arg )
{
    long size;
    char *facename;

    size = strtol( arg, &facename, 0 );
    if( !size || ( *facename != '.' ))
    {
        fprintf( stderr, "Invalid font name and size: %s. Use like 24.Gulim\n", arg );
        exit( 1 );
    }

    av_strlcpy( osd_fontnamesize, arg, sizeof( osd_fontnamesize ));
}

static void opt_aspect( const char *arg )
{
    if( !strcmp( arg, "none" ))
        aspect_ratio = KVAR_NONE;
    else if( !strcmp( arg, "original" ))
        aspect_ratio = KVAR_ORIGINAL;
    else if( !strcmp( arg, "force43" ))
        aspect_ratio = KVAR_FORCE43;
    else if( !strcmp( arg, "force169" ))
        aspect_ratio = KVAR_FORCE169;
    else
    {
        fprintf( stderr, "Invalid aspect ratio : %s\n", arg );
        exit( 1 );
    }
}

static void opt_brightness( const char *arg )
{
    char *p;
    long attr = strtol( arg, &p, 0 );

    if( *p || (( attr < 0 ) || ( attr > 255 )))
    {
        fprintf( stderr, "Invalid brightness level : %s\n", arg );
        exit( 1 );
    }

    aulAttrValue[ KVAA_BRIGHTNESS ] = attr;
}

static void opt_contrast( const char *arg )
{
    char *p;
    long attr = strtol( arg, &p, 0 );

    if( *p || (( attr < 0 ) || ( attr > 255 )))
    {
        fprintf( stderr, "Invalid contrast level : %s\n", arg );
        exit( 1 );
    }

    aulAttrValue[ KVAA_CONTRAST ] = attr;
}

static void opt_saturation( const char *arg )
{
    char *p;
    long attr = strtol( arg, &p, 0 );

    if( *p || (( attr < 0 ) || ( attr > 255 )))
    {
        fprintf( stderr, "Invalid saturation level : %s\n", arg );
        exit( 1 );
    }

    aulAttrValue[ KVAA_SATURATION ] = attr;
}

static void opt_hue( const char *arg )
{
    char *p;
    long attr = strtol( arg, &p, 0 );

    if( *p || (( attr < 0 ) || ( attr > 255 )))
    {
        fprintf( stderr, "Invalid hue level : %s\n", arg );
        exit( 1 );
    }

    aulAttrValue[ KVAA_HUE ] = attr;
}

static void opt_subcolor( const char *arg )
{
    char *p;
    long color;

    color = strtol( arg, &p, 16 );
    if( color )
        sub_color_font = color;

    if( *p == ',' )
        p++;

    color = strtol( p, &p, 16 );
    if( color )
        sub_color_outline = color;

    if( *p == ',' )
        p++;

    color = strtol( p, &p, 16 );
    if( color )
        sub_color_shadow = color;
}

static void opt_osdcolor( const char *arg )
{
    char *p;
    long color;

    color = strtol( arg, &p, 16 );
    if( color )
        osd_color_font = color;

    if( *p == ',' )
        p++;

    color = strtol( p, &p, 16 );
    if( color )
        osd_color_outline = color;

    if( *p == ',' )
        p++;

    color = strtol( p, &p, 16 );
    if( color )
        osd_color_shadow = color;
}

static void opt_xext(const char *arg)
{
    plSetExcludeExts( m_ppl, arg );
}

static void opt_frame_pix_fmt(const char *arg)
{
    frame_pix_fmt = avcodec_get_pix_fmt(arg);
}

static int opt_sync(const char *opt, const char *arg)
{
    if (!strcmp(arg, "audio"))
        av_sync_type = AV_SYNC_AUDIO_MASTER;
    else if (!strcmp(arg, "video"))
        av_sync_type = AV_SYNC_VIDEO_MASTER;
    else if (!strcmp(arg, "ext"))
        av_sync_type = AV_SYNC_EXTERNAL_CLOCK;
    else {
        fprintf(stderr, "Unknown value for %s: %s\n", opt, arg);
        exit(1);
    }
    return 0;
}

static int opt_seek(const char *opt, const char *arg)
{
    start_time = parse_time_or_die(opt, arg, 1);
    return 0;
}


static int opt_debug(const char *opt, const char *arg)
{
    av_log_set_level(99);
    debug = parse_number_or_die(opt, arg, OPT_INT64, 0, INT_MAX);
    return 0;
}

static int opt_vismv(const char *opt, const char *arg)
{
    debug_mv = parse_number_or_die(opt, arg, OPT_INT64, INT_MIN, INT_MAX);
    return 0;
}

static int opt_thread_count(const char *opt, const char *arg)
{
    thread_count= parse_number_or_die(opt, arg, OPT_INT64, 0, INT_MAX);
#if !defined(HAVE_THREADS)
    fprintf(stderr, "Warning: not compiled with thread support, using thread emulation\n");
#endif
    return 0;
}

const OptionDef options[] = {
    { "h", OPT_EXIT, {(void*)show_help}, "show help" },
    { "version", OPT_EXIT, {(void*)show_version}, "show version" },
    { "L", OPT_EXIT, {(void*)show_license}, "show license" },
    { "formats", OPT_EXIT, {(void*)show_formats}, "show available formats, codecs, protocols, ..." },
    { "video", HAS_ARG, {(void*)opt_video}, "set video driver (driver=auto/snap/wo/dive)", "driver" },
    { "x", HAS_ARG | OPT_FUNC2, {(void*)opt_width}, "force displayed width", "width" },
    { "y", HAS_ARG | OPT_FUNC2, {(void*)opt_height}, "force displayed height", "height" },
    { "s", HAS_ARG | OPT_VIDEO, {(void*)opt_frame_size}, "set frame size (WxH or abbreviation)", "size" },
    { "fs", OPT_BOOL, {(void*)&is_full_screen}, "force full screen" },
    { "vol", HAS_ARG, {(void*)opt_volume}, "set initial volume level in percentage", "level" },
    { "an", OPT_BOOL, {(void*)&audio_disable}, "disable audio" },
    { "vn", OPT_BOOL, {(void*)&video_disable}, "disable video" },
    { "ast", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&wanted_audio_stream}, "", "" },
    { "vst", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&wanted_video_stream}, "", "" },
    { "ss", HAS_ARG | OPT_FUNC2, {(void*)&opt_seek}, "seek to a given position in seconds", "pos" },
    { "bytes", OPT_BOOL, {(void*)&seek_by_bytes}, "seek by bytes" },
    { "nodisp", OPT_BOOL, {(void*)&display_disable}, "disable graphical display" },
    { "f", HAS_ARG, {(void*)opt_format}, "force format", "fmt" },
    { "pix_fmt", HAS_ARG | OPT_EXPERT | OPT_VIDEO, {(void*)opt_frame_pix_fmt}, "set pixel format", "format" },
    { "subfont", HAS_ARG, {(void*)opt_subfont}, "set subtitle font name and size (xx=size, name=font name)", "xx.name" },
    { "osdfont", HAS_ARG, {(void*)opt_osdfont}, "set OSD font name and size (xx=size, name=font name)", "xx.name" },
    { "aspect", HAS_ARG, {(void*)opt_aspect}, "set aspect ratio (ratio=none, original, force43, force169)", "ratio" },
    { "framedrop", OPT_BOOL, {(void*)&frame_drop}, "frame drop if necessary" },
    { "hidemouse", OPT_BOOL, {(void*)&hide_mouse}, "hide mouse pointer on full screen mode" },
    { "brightness", HAS_ARG, {(void*)opt_brightness}, "set brightness level, except dive (level=0..255)", "level" },
    { "contrast", HAS_ARG, {(void*)opt_contrast}, "set contrast level, except dive (level=0..255)", "level" },
    { "saturation", HAS_ARG, {(void*)opt_saturation}, "set saturation level, except dive (level=0..255)", "level" },
    { "hue", HAS_ARG, {(void*)opt_hue}, "set hue level, except dive (level=0..255)", "level" },
    { "res48", OPT_BOOL, {(void*)&resample}, "resample 48KHz audio to 44.1KHz(experimental)" },
    { "subimg", OPT_BOOL, {(void*)&use_subimg}, "display subtitles/OSD to image" },
    { "subcolor", HAS_ARG, {(void*)opt_subcolor}, "set subtitle color in hexa RGB(font, outline, shadow)", "f,o,s" },
    { "osdcolor", HAS_ARG, {(void*)opt_osdcolor}, "set OSD color in hexa RGB(font, outline, shadow) ", "f,o,s" },
    { "autoadd", OPT_BOOL, {(void*)&auto_add}, "automatically add files having a similar name", "" },
    { "xext", HAS_ARG, {(void*)&opt_xext}, "exclude the following extensions on using -autoadd", "e1,e2,..." },
    { "stats", OPT_BOOL | OPT_EXPERT, {(void*)&show_status}, "show status", "" },
    { "debug", HAS_ARG | OPT_FUNC2 | OPT_EXPERT, {(void*)opt_debug}, "print specific debug info", "" },
    { "bug", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&workaround_bugs}, "workaround bugs", "" },
    { "vismv", HAS_ARG | OPT_FUNC2 | OPT_EXPERT, {(void*)opt_vismv}, "visualize motion vectors", "" },
    { "fast", OPT_BOOL | OPT_EXPERT, {(void*)&fast}, "non spec compliant optimizations", "" },
    { "genpts", OPT_BOOL | OPT_EXPERT, {(void*)&genpts}, "generate pts", "" },
    { "drp", OPT_BOOL |OPT_EXPERT, {(void*)&decoder_reorder_pts}, "let decoder reorder pts", ""},
    { "lowres", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&lowres}, "", "" },
    { "skiploop", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&skip_loop_filter}, "", "" },
    { "skipframe", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&skip_frame}, "", "" },
    { "skipidct", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&skip_idct}, "", "" },
    { "idct", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&idct}, "set idct algo",  "algo" },
    { "er", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&error_recognition}, "set error detection threshold (0-4)",  "threshold" },
    { "ec", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&error_concealment}, "set error concealment options",  "bit_mask" },
    { "sync", HAS_ARG | OPT_FUNC2 | OPT_EXPERT, {(void*)opt_sync}, "set audio-video sync. type (type=audio/video/ext)", "type" },
    { "threads", HAS_ARG | OPT_FUNC2 | OPT_EXPERT, {(void*)opt_thread_count}, "thread count", "count" },
    { "fixt23", OPT_BOOL | OPT_EXPERT, {(void*)&fix_t23}, "workaround diagonal stripes on T23 laptop with S3 Video" },
    { "noaudioshare", OPT_BOOL | OPT_EXPERT, {(void*)&no_audio_share}, "open audio as exclusive mode" },
    { "fixsnap", OPT_BOOL | OPT_EXPERT, {(void*)&fix_snap}, "workaround for a larger movie than screen size on snap mode" },
    { "audiobufsize", OPT_INT | HAS_ARG | OPT_EXPERT, {(void*)&audio_buffer_size}, "set audio buffer size in bytes for for DART", "size" },
    { "default", OPT_FUNC2 | HAS_ARG | OPT_AUDIO | OPT_VIDEO | OPT_EXPERT, {(void*)opt_default}, "generic catch all option", "" },
    { NULL, },
};

static void show_help(void)
{
    printf("usage: kmp [options] input_file\n");
    printf("\n");
    show_help_options(options, "Main options:\n",
                      OPT_EXPERT, 0);
    show_help_options(options, "\nAdvanced options:\n",
                      OPT_EXPERT, OPT_EXPERT);
    printf("\nWhile playing:\n"
           "q                   quit\n"
           "f, Enter            toggle full screen\n"
           "p, SPC              toggle pause\n"
           "m                   toggle audio mute\n"
           "a                   cycle audio channel\n"
           "v                   cycle video channel\n"
           "t                   cycle subtitle channel\n"
           "w                   show audio waves\n"
           "left/right          seek backward/forward 10 seconds\n"
           "Ctrl+left/right     seek backward/forward 1 minute\n"
           "down/up             volume down/up by 1%%\n"
           "Ctrl+down/up        volume down/up by 5%%\n"
           "mouse click         seek to position in file corresponding to fraction of width\n"
           "+/-                 grow up/down font size\n"
           "Alt-0/1/2/3         set aspec ratio none/original/force43/force169 respectively\n"
           "8/9                 brightness down/up by 5 level(except dive)\n"
           "i/o                 contrast down/up by 5 level(except dive)\n"
           "k/l                 saturation down/up by 5 level(except dive)\n"
           ",/.                 hue down/up by 5 level(except dive)\n"
           "/                   reset attributes(b/c/s/h) to default value(except dive)\n"
           "[/]                 correct subtitle sync -/+ 0.5 seconds\n"
           "PageUp/PageDown     play a previous/next file\n"
           );
}

static void opt_input_file(const char *filename)
{
    if (!strcmp(filename, "-"))
                    filename = "pipe:";
    plAddFile( m_ppl, filename, FALSE );
}

static void inform( const char *format, ... )
{
    CHAR    szInformedMsg[ 256 ];
    va_list arg_ptr;

    va_start( arg_ptr, format );
    vsnprintf( szInformedMsg, sizeof( szInformedMsg ), format, arg_ptr );
    va_end( arg_ptr );

    osdSetText( szInformedMsg );

    WinStartTimer( hab, hwndKMP, TID_INFORM, INTERVAL_INFORM );
}

#define CAS_CTRL( f )   (( f ) & KC_CTRL )
#define CAS_ALT( f )    (( f ) & KC_ALT )
#define CAS_SHIFT( f )  (( f ) & KC_SHIFT )
#define CAS_CO( f )     ( CAS_CTRL( f ) && !CAS_ALT( f ) && !CAS_SHIFT( f ))
#define CAS_AO( f )     ( !CAS_CTRL( f ) && CAS_ALT( f ) && !CAS_SHIFT( f ))
#define CAS_SO( f )     ( !CAS_CTRL( f ) && !CAS_ALT( f ) && CAS_SHIFT( f ))
#define CAS_NONE( f )   ( !CAS_CTRL( f ) && !CAS_ALT( f ) && !CAS_SHIFT( f ))

#define FKC_CHAR( f )   (( f ) & KC_CHAR )
#define FKC_SCAN( f )   (( f ) & KC_SCANCODE )
#define FKC_VIRT( f )   (( f ) & KC_VIRTUALKEY )
#define FKC_KEYUP( f )  (( f ) & KC_KEYUP )

static MRESULT EXPENTRY NewFrameWndProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
    switch( msg )
    {
        case WM_QUERYTRACKINFO :
        {
            //USHORT      ustflags = SHORT1FROMMP( mp1 );
            PTRACKINFO  pti = ( PTRACKINFO )mp2;
            RECTL       rcl;

            if( !cur_stream->video_st || is_full_screen )
                break;

            pfnwpOldFrame( hwnd, msg, mp1, mp2 );

            //if( ustflags & ( TF_LEFT | TF_RIGHT | TF_TOP | TF_BOTTOM | TF_SETPOINTERPOS ))
            {
                pti->rclBoundary.xLeft = 0;
                pti->rclBoundary.yBottom = 0;
                pti->rclBoundary.xRight = cxScreen;
                pti->rclBoundary.yTop = cyScreen;

                rcl.xLeft = 0;
                rcl.yBottom = 0;
                rcl.xRight = lImgWidth + 1;
                rcl.yTop = lImgHeight + 1;

                WinCalcFrameRect( hwnd, &rcl, FALSE );

                pti->ptlMinTrackSize.x = rcl.xRight - rcl.xLeft;
                pti->ptlMinTrackSize.y = rcl.yTop - rcl.yBottom;

                pti->ptlMaxTrackSize.x = cxScreen;
                pti->ptlMaxTrackSize.y = cyScreen;
            }

            return MRFROMLONG( TRUE );
        }

        case WM_ADJUSTWINDOWPOS :
        {
            PSWP    pswp = ( PSWP )mp1;
            RECTL   rcl;

            if( !cur_stream->video_st || is_full_screen )
                break;

            if( pswp->fl & SWP_SIZE )
            {
                rcl.xLeft = pswp->x;
                rcl.yBottom = pswp->y;
                rcl.xRight = rcl.xLeft + pswp->cx;
                rcl.yTop = rcl.yBottom + pswp->cy;

                WinCalcFrameRect( hwnd, &rcl, TRUE );

                if( rcl.xRight - rcl.xLeft <= lImgWidth )
                    rcl.xRight = rcl.xLeft + ( lImgWidth + 1 );

                if( rcl.yTop - rcl.yBottom <= lImgHeight )
                    rcl.yTop = rcl.yBottom + ( lImgHeight + 1 );

                WinCalcFrameRect( hwnd, &rcl, FALSE );

                if( rcl.xRight - rcl.xLeft > cxScreen )
                {
                    rcl.xLeft = 0;
                    rcl.xRight = cxScreen;
                }

                if( rcl.yTop - rcl.yBottom > cyScreen )
                {
                    rcl.yBottom = 0;
                    rcl.yTop = cyScreen;
                }

                pswp->fl |= SWP_MOVE;
                pswp->x = rcl.xLeft;
                pswp->y = rcl.yBottom;

                pswp->cx = rcl.xRight - rcl.xLeft;
                pswp->cy = rcl.yTop - rcl.yBottom;
            }

            break;
        }
    }

    return pfnwpOldFrame( hwnd, msg, mp1, mp2 );
}

static MRESULT EXPENTRY WndProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
    switch( msg )
    {
        case WM_REFRESH_EVENT:
            video_refresh_timer(( void * )mp1 );
            return 0;
    }

    if( !cur_stream || !cur_stream->ic )
        return WinDefWindowProc( hwnd, msg, mp1, mp2 );

    switch( msg )
    {
        case WM_CHAR :
        {
            USHORT fsFlags;
            USHORT usCh;
            USHORT usVk;

            fsFlags = SHORT1FROMMP( mp1 );
            usCh = SHORT1FROMMP( mp2 );
            usVk = SHORT2FROMMP( mp2 );

            if( FKC_KEYUP( fsFlags ))
                return MRFROMLONG( TRUE );

            if( FKC_VIRT( fsFlags ))
            {
                switch( usVk )
                {
                    case VK_ESC :
                        if( is_full_screen )
                            toggle_full_screen();
                        break;

                    case VK_NEWLINE :
                    case VK_ENTER :
                        toggle_full_screen();
                        break;

                    case VK_SPACE :
                        toggle_pause();
                        inform("%s", cur_stream->paused ? "Pause" : "Resume" );
                        break;

                    case VK_LEFT :
                    case VK_RIGHT :
                    {
                        static int64_t last_seek_time = 0;

                        int64_t cur_seek_time = av_gettime();
                        double incr, pos;
                        int tns;
                        int ns, hh, mm, ss;

                        // too fast seek cause dart to hang, so wait at least 100ms
                        if( last_seek_time + 100000 > cur_seek_time )
                            break;

                        last_seek_time = cur_seek_time;

                        if( CAS_CO( fsFlags ))
                            incr = -60.0;
                        else
                            incr = -10.0;

                        if( usVk == VK_RIGHT )
                            incr *= -1;

                        tns = cur_stream->ic->duration / AV_TIME_BASE;
                        pos = get_master_clock( cur_stream );

                        // to avoid seek error
                        if( pos + incr > tns - 3 )
                        {
                            incr = ( tns - 3 ) - pos;
                            ns = tns - 3;
                        }
                        else if( pos + incr < 0 )
                        {
                            incr = -pos;
                            ns = 0;
                        }
                        else
                            ns = pos + incr;

                        hh = ns / 3600;
                        mm = ( ns % 3600 ) / 60;
                        ss = ( ns % 60 );

                        inform( "Seek %d seconds %s / %02d:%02d:%02d (%d%%)",
                                CAS_CO( fsFlags ) ? 60 : 10,
                                usVk == VK_RIGHT ? "forward" : "backward",
                                hh, mm, ss,
                                ns * 100 / tns );

                        if (seek_by_bytes) {
                            pos = url_ftell(cur_stream->ic->pb);
                            if (cur_stream->ic->bit_rate)
                                incr *= cur_stream->ic->bit_rate / 60.0;
                            else
                                incr *= 180000.0;
                            pos += incr;
                            stream_seek(cur_stream, pos, incr);
                        } else {
                            //pos = get_master_clock(cur_stream);
                            pos += incr;
                            stream_seek(cur_stream, ( int64_t )( pos * AV_TIME_BASE ), incr );
                        }
                        break;
                    }

                    case VK_UP :
                    case VK_DOWN :
                    {
                        int   delta;

                        if( CAS_CO( fsFlags ))
                            delta = 5;
                        else
                            delta = 1;

                        if( usVk == VK_DOWN )
                            delta *= -1;

                        volume_level += delta;
                        if( volume_level > 100 )
                            volume_level = 100;
                        if( volume_level < 0 )
                            volume_level = 0;

                        dartSetVolume( MCI_SET_AUDIO_ALL, volume_level );
                        inform("Volume : %d%%", volume_level );
                        break;
                    }

                    case VK_PAGEUP :
                        m_cmdPlaylist = PLAYLIST_PREV_FILE;
                        WinPostMsg( hwnd, WM_QUIT, 0, 0 );
                        break;

                    case VK_PAGEDOWN :
                        m_cmdPlaylist = PLAYLIST_NEXT_FILE;
                        WinPostMsg( hwnd, WM_QUIT, 0, 0 );
                        break;
                }
            }
            else if( usCh ) // FKC_CHAR( fsFlags ) does not work when ALT pressed
            {
                switch( usCh )
                {
                    case 'q' :
                    case 'Q' :
                        m_cmdPlaylist = PLAYLIST_QUIT;
                        WinPostMsg( hwnd, WM_QUIT, 0, 0 );
                        break;

                    case 'f' :
                    case 'F' :
                        toggle_full_screen();
                        break;

                    case 'p' :
                    case 'P' :
                        toggle_pause();
                        inform("%s", cur_stream->paused ? "Pause" : "Resume");
                        break;

                    case 's' :
                    case 'S' :
                        step_to_next_frame();
                        inform("Step to next frame");
                        break;

                    case 'a' :
                    case 'A' :
                        stream_cycle_channel(cur_stream, CODEC_TYPE_AUDIO);
                        inform("Cycle audio channel");
                        break;

                    case 'v' :
                    case 'V' :
                        stream_cycle_channel(cur_stream, CODEC_TYPE_VIDEO);
                        inform("Cycle video channel");
                        break;

                    case 't' :
                    case 'T' :
                        stream_cycle_channel(cur_stream, CODEC_TYPE_SUBTITLE);
                        inform("Cycle subtitle channel");
                        break;

                    case 'w' :
                    case 'W' :
                        toggle_audio_display();
                        inform("%s mode", cur_stream->show_audio ? "Audio wave" : "Normal play");
                        break;

                    case 'm' :
                    case 'M' :
                    {
                        static BOOL fMute = FALSE;

                        fMute = !fMute;
                        dartSetSoundState( MCI_SET_AUDIO_ALL, !fMute );
                        inform("Mute %s", fMute ? "On" : "Off");
                        break;
                    }

                    case '+' :
                    case '=' :
                    case '-' :
                    case '_' :
                        if( cur_stream->use_sub )
                        {
                            FIXED fxPointSize;
                            CHAR  szFontName[ FACESIZE + 1 ];

                            fxPointSize = subQueryFontSize(
                                            (( usCh == '+' ) || ( usCh == '=' )) ?
                                            SQFS_NEXT : SQFS_PREV );

                            fxPointSize = subSetFontSize( fxPointSize );

                            inform("Font : %d.%s",
                                   FIXEDINT( fxPointSize ),
                                   subQueryFontName( szFontName, sizeof( szFontName )));
                        }
                        break;

                    case '0' :
                    case '1' :
                    case '2' :
                    case '3' :
                        if( CAS_AO( fsFlags ))
                        {
                            static PSZ pszAR[] = { "None",
                                                   "Original",
                                                   "Force to 4:3",
                                                   "Force to 16:9", };

                            ULONG ulRatio = KVAR_NONE;

                            switch( usCh )
                            {
                            #if 0
                                case '0' :
                                    ulRatio = KVAR_NONE;
                                    break;
                            #endif

                                case '1' :
                                    ulRatio = KVAR_ORIGINAL;
                                    break;

                                case '2' :
                                    ulRatio = KVAR_FORCE43;
                                    break;

                                case '3' :
                                    ulRatio = KVAR_FORCE169;
                                    break;
                            }

                            imgSetAspectRatio( ulRatio );

                            inform("Aspect ratio : %s", pszAR[ ulRatio ]);
                        }
                        break;

                    case '8' :
                    case '9' :
                    case 'i' :
                    case 'I' :
                    case 'o' :
                    case 'O' :
                    case 'k' :
                    case 'K' :
                    case 'l' :
                    case 'L' :
                    case ',' :
                    case '.' :
                    {
                        static PSZ pszAttr[] = { "Brightness",
                                                 "Contrast",
                                                 "Saturation",
                                                 "Hue", };

                        ULONG ulAttr;
                        ULONG ulValue;
                        int   delta;

                        switch( usCh )
                        {
                            case '8' :
                            case '9' :
                                ulAttr = KVAA_BRIGHTNESS;
                                delta = ( usCh == '8' ) ? -5 : 5;
                                break;

                            case 'i' :
                            case 'I' :
                            case 'o' :
                            case 'O' :
                                ulAttr = KVAA_CONTRAST;
                                delta = (( usCh == 'i' ) || ( usCh == 'I' )) ? -5 : 5;
                                break;

                            case 'k' :
                            case 'K' :
                            case 'l' :
                            case 'L' :
                                ulAttr = KVAA_SATURATION;
                                delta = (( usCh == 'k' ) || ( usCh == 'K' )) ? -5 : 5;
                                break;

                            case ',' :
                            case '.' :
                            default  : // to avoid warning
                                ulAttr = KVAA_HUE;
                                delta = ( usCh == ',' ) ? -5 : 5;
                                break;
                        }

                        if( imgQueryAttr( ulAttr, &ulValue ))
                            break;

                        ulValue += delta;

                        if( !imgSetAttr( ulAttr, &ulValue ))
                            inform("%s : %lu", pszAttr[ ulAttr ], ulValue );

                        break;
                    }

                    case '/' :
                        imgResetAttr();

                        inform("Reset all attributes to default");
                        break;

                    case '[' :
                    case ']' :
                        if( cur_stream->use_sub )
                        {
                            LONG lDelta = subQueryDelta();

                            lDelta += ( usCh == '[' ) ? -50 : 50;

                            subSetDelta( lDelta );

                            inform("Subtitle sync correction : %+.1fs", lDelta / 100. );
                        }
                }
            }

            return MRFROMLONG( TRUE );
        }

        case WM_MOUSEMOVE :
        case WM_BUTTON1DOWN :
        case WM_BUTTON2DOWN :
        case WM_BUTTON3DOWN :
        {
            double frac;
            int ns, hh, mm, ss;
            int tns, thh, tmm, tss;

            if( hide_mouse && is_full_screen )
            {
                if( !fMouseShow )
                    SHOW_POINTER( TRUE );

                WinStartTimer( hab, hwnd, TID_MOUSE_HIDE, INTERVAL_MOUSE_HIDE );
            }

            if( msg == WM_MOUSEMOVE )
                break; // fall through to default window procedure

            if( WinQueryFocus( HWND_DESKTOP ) != hwnd )
            {
                WinSetFocus( HWND_DESKTOP, hwnd );
                return MRFROMLONG( TRUE );
            }

            tns = cur_stream->ic->duration/1000000LL;
            thh = tns/3600;
            tmm = (tns%3600)/60;
            tss = (tns%60);
            frac = (double)SHORT1FROMMP( mp1 )/(double)cur_stream->width;
            ns = frac*tns;
            hh = ns/3600;
            mm = (ns%3600)/60;
            ss = (ns%60);
            inform("Seek to %2.0f%% (%2d:%02d:%02d) of (%2d:%02d:%02d)", frac*100,
                   hh, mm, ss, thh, tmm, tss);
            stream_seek(cur_stream, (int64_t)(cur_stream->ic->start_time+frac*cur_stream->ic->duration), 0 );
            return MRFROMLONG( TRUE );
        }

#if 0
        case WM_BUTTON1DBLCLK :
            toggle_full_screen();
            return MRFROMLONG( TRUE );
#endif

        case WM_SIZE :
            cur_stream->width  = SHORT1FROMMP( mp2 );
            cur_stream->height = SHORT2FROMMP( mp2 );

            if ( !is_full_screen ) {
                screen_width  = SHORT1FROMMP( mp2 );
                screen_height = SHORT2FROMMP( mp2 );
            }

            if( cur_stream->use_sub )
                subInvalidate();

            inform("%s mode : Size = %d X %d",
                   is_full_screen ? "Full screen" : "Windowed",
                   cur_stream->width, cur_stream->height );

            return 0;

        case WM_TIMER :
            switch( SHORT1FROMMP( mp1 ))
            {
                case TID_INFORM :
                    osdInvalidate();

                    WinStopTimer( hab, hwnd, TID_INFORM );

                    if( cur_stream->paused )
                        WinInvalidateRect( hwnd, NULL, TRUE );

                    return 0;

                case TID_SECOND :
                {
                    CHAR szPlayTime[ 256 ];

                    int ns, hh, mm, ss;
                    int tns, thh, tmm, tss;

                    if( cur_stream->paused )
                        break;

                    tns = cur_stream->ic->duration / 1000000LL;
                    thh = tns / 3600;
                    tmm = ( tns % 3600 ) / 60;
                    tss = ( tns % 60 );

                    ns = get_master_clock( cur_stream );
                    hh = ns / 3600;
                    mm = ( ns % 3600 ) / 60;
                    ss = ( ns % 60 );

                    snprintf( szPlayTime, sizeof( szPlayTime ),
                              "%s - %02d:%02d:%02d / %02d:%02d:%02d (%d%%)",
                              KMP_NAME, hh, mm, ss, thh, tmm, tss, ns * 100 / tns );

                    WinSetWindowText( hwndTitleBar, szPlayTime );

                    return 0;
                }

                case TID_MOUSE_HIDE :
                    if( fMouseShow )
                        SHOW_POINTER( FALSE );

                    WinStopTimer( hab, hwnd, TID_MOUSE_HIDE );

                    return 0;
            }
            break;  // fall through to default window procedure

        case WM_PAINT :
        {
            HPS     hps;
            RECTL   rcl;

            hps = WinBeginPaint( hwnd, NULLHANDLE, &rcl );
            // No video ?
            if( !cur_stream->video_st )
            {
                // Then just fill with BLACK
                WinFillRect( hps, &rcl, CLR_BLACK );
            }
            WinEndPaint( hps );
        }   // fall through to WM_MOVE

        case WM_MOVE :
            if( cur_stream->paused )
            {
                // to prevent subtitles/OSD from being overwritten when subimg mode
                cur_stream->paint_in_pause = TRUE;
                video_display( cur_stream );
                cur_stream->paint_in_pause = FALSE;
            }

            return 0;
    }

    return WinDefWindowProc( hwnd, msg, mp1, mp2 );
}

static void init_flags( void )
{
    mp_msg_init();

    GetCpuCaps( &gCpuCaps );
    printf("CPUflags:  MMX: %d MMX2: %d 3DNow: %d 3DNow2: %d SSE: %d SSE2: %d\n",
           gCpuCaps.hasMMX, gCpuCaps.hasMMX2,
           gCpuCaps.has3DNow, gCpuCaps.has3DNowExt,
           gCpuCaps.hasSSE, gCpuCaps.hasSSE2 );

    mm_flags = mm_support();

    sws_flags = SWS_FAST_BILINEAR | SWS_PRINT_INFO;

    printf("SwScaler CPUflags : ");
    if( mm_flags & MM_MMX )
    {
        sws_flags |= SWS_CPU_CAPS_MMX;
        printf("MMX ");
    }

    if( mm_flags & MM_MMXEXT )
    {
        sws_flags |= SWS_CPU_CAPS_MMX2;
        printf("MMX2 ");
    }

    if( mm_flags & MM_3DNOW )
    {
        sws_flags |= SWS_CPU_CAPS_3DNOW;
        printf("3DNow ");
    }
    printf("\n");
}

#define WC_KMP   "WC_KMP"

/* Called from the main */
int main( int argc, char **argv )
{
    HMQ     hmq;
    ULONG   flFrameFlags;
    QMSG    qm;
    int     i;
    BOOL    fQuit;

    _envargs( &argc, &argv, "KMPOPT");
    _response( &argc, &argv );

    MorphToPM(); // Morph the VIO application to a PM one to be able to use Win* functions

    // Make stdout and stderr unbuffered
    setbuf( stdout, NULL );
    setbuf( stderr, NULL );

    m_ppl = plCreate();

    printf("KMP " KMP_VERSION " based on FFplay " FFMPEG_VERSION ", Copyright (c) 2007-2008 KO Myung-Hun.\n");

    init_flags();

    /* register all codecs, demux and protocols */
    avcodec_register_all();
    avdevice_register_all();
    av_register_all();

    for(i=0; i<CODEC_TYPE_NB; i++){
        avctx_opts[i]= avcodec_alloc_context2(i);
    }
    avformat_opts = av_alloc_format_context();
    sws_opts = sws_getContext(16,16,0, 16,16,0, sws_flags, NULL,NULL,NULL);

    parse_options(argc, argv, options, opt_input_file);

    if( plQueryCount( m_ppl ) == 0 ) {
        show_help();
        exit(1);
    }

    if( plQueryCount( m_ppl ) == 1 && auto_add )
        plAddFileAuto( m_ppl, TOLERANCE );

    if (display_disable) {
        video_disable = 1;
    }

do
{
    fQuit = FALSE;

    m_cmdPlaylist = PLAYLIST_NEXT_FILE;

    input_filename = plQueryFileName( m_ppl );
    printf("Playing file : %s [%d/%d]\n", input_filename, plQueryIndex( m_ppl ) + 1, plQueryCount( m_ppl ));

    av_init_packet(&flush_pkt);
    flush_pkt.data= "FLUSH";

    hab = WinInitialize( 0 );
    hmq = WinCreateMsgQueue( hab, 0);

    WinRegisterClass(
        hab,
        WC_KMP,
        WndProc,
        CS_SIZEREDRAW | CS_MOVENOTIFY,
        sizeof( PVOID )
    );

    flFrameFlags = FCF_SYSMENU | FCF_TITLEBAR | FCF_MINMAX | FCF_SIZEBORDER |
                   FCF_TASKLIST;

    hwndFrame = WinCreateStdWindow(
                    HWND_DESKTOP,               // parent window handle
                    WS_VISIBLE,                 // frame window style
                    &flFrameFlags,              // window style
                    WC_KMP,                     // class name
                    KMP_NAME,                   // window title
                    0L,                         // default client style
                    NULLHANDLE,                 // resource in exe file
                    1,                          // frame window id
                    &hwndKMP                    // client window handle
                );

    if( hwndFrame != NULLHANDLE )
    {
        fVideoOpened = FALSE;

        lImgWidth = 0;
        lImgHeight = 0;

        screen_width = opt_screen_width;
        screen_height = opt_screen_height;

        hwndSysMenu = WinWindowFromID( hwndFrame, FID_SYSMENU );
        hwndTitleBar = WinWindowFromID( hwndFrame, FID_TITLEBAR );
        hwndMinMax = WinWindowFromID( hwndFrame, FID_MINMAX );

        cxScreen = WinQuerySysValue( HWND_DESKTOP, SV_CXSCREEN );
        cyScreen = WinQuerySysValue( HWND_DESKTOP, SV_CYSCREEN );

        osdInit( hab, osd_fontnamesize, use_subimg );

        cur_stream = stream_open(input_filename, file_iformat);

        if( fix_t23 )
            pfnwpOldFrame = WinSubclassWindow( hwndFrame, NewFrameWndProc );

        WinStartTimer( hab, hwndKMP, TID_SECOND, INTERVAL_SECOND );

        while( WinGetMsg( hab, &qm, NULLHANDLE, 0, 0 ))
            WinDispatchMsg( hab, &qm );

        WinStopTimer( hab, hwndKMP, TID_SECOND );

        if( fix_t23 )
            WinSubclassWindow( hwndFrame, pfnwpOldFrame );

        if (cur_stream) {
            stream_close(cur_stream);
            cur_stream = NULL;
        }
        if (show_status)
            printf("\n");

        // restore mouse pointer status
        if( hide_mouse && !fMouseShow )
            SHOW_POINTER( TRUE );

        osdDone();

        WinDestroyWindow( hwndFrame );
    }

    WinDestroyMsgQueue( hmq );
    WinTerminate( hab );

    switch( m_cmdPlaylist )
    {
        case PLAYLIST_PREV_FILE :
            fQuit = plMovePrev( m_ppl ) == NULL;
            break;

        case PLAYLIST_NEXT_FILE :
            fQuit = plMoveNext( m_ppl ) == NULL;
            break;

        case PLAYLIST_QUIT :
            fQuit = TRUE;
            break;
    }

} while( !fQuit );

    plDestroy( m_ppl );

    return 0;
}
