/*
    KMP : Image library for OS/2
    Copyright (C) 2007 by KO Myung-Hun <komh@chollian.net>

    This file is part of KMP.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Changes :
        KO Myung-Hun <komh@chollian.net> 2007/02/25
            - Added imgQueryAttr(), imgSetAttr() and imgResetAttr()
            - Changed imgInit() to initialize attribute
            - Reset attribute on imgDone()

        KO Myung-Hun <komh@chollian.net> 2007/03/03
            - Changed the prototype of imgCreateYUV() to use m_kvas instead
              of parameters
            - Allocate memory for YUV buffer more accurately in imgCreateYUV()
            - Changed imgConvert() to use m_kvas instead of pSrcBmp->w and
              pSrcBmp->h
*/

#include <os2.h>

#include <fourcc.h>

#include <stdlib.h>
#include <types.h>

#include "kva.h"
#include "yuv2rgb.h"

#include "kmp_img.h"

static KVASETUP m_kvas = { 0 };

static FOURCC m_fcc;

static APIRET imgConvertInit( VOID )
{
    KVACAPS kvac;
    int     nColorDepth = 24;

    kvaCaps( &kvac );

    switch( kvac.ulMode )
    {
        case KVAM_WO :
            m_fcc = FOURCC_Y422;

            yv12toyuy2_init();

            //nColorDepth = 24;
            break;

        case KVAM_DIVE :
        default :
            switch( kvac.fccScreen )
            {
                case FOURCC_BGR4 :
                case FOURCC_BGR3 :
                case FOURCC_LUT8 :  // maybe best T.T
                    m_fcc = FOURCC_BGR3;

                    //nColorDepth = 24;
                    break;

                case FOURCC_R565 :
                case FOURCC_R555 :
                    m_fcc = FOURCC_R565;

                    nColorDepth = 16;
                    break;
            }
            break;
    }

    yuv420_rgb_init( nColorDepth );

    return 0;
}

static VOID imgConvertDone( VOID )
{
    yuv420_rgb_close();
}

static VOID imgConvert( PVOID pDstBuffer, ULONG ulDstBPL, YUV *pSrcBmp )
{
    switch( m_fcc )
    {
        case FOURCC_Y422 :
            yv12toyuy2( pSrcBmp->data[ 0 ], pSrcBmp->data[ 1 ], pSrcBmp->data[ 2 ],
                        pDstBuffer, m_kvas.szlSrcSize.cx, m_kvas.szlSrcSize.cy,
                        pSrcBmp->linesize[ 0 ], pSrcBmp->linesize[ 1 ],
                        ulDstBPL, 2 );
            break;

        case FOURCC_BGR3 :
        case FOURCC_R565 :
            yuv2rgb( pDstBuffer, pSrcBmp->data[ 0 ], pSrcBmp->data[ 1 ], pSrcBmp->data[ 2 ],
                     m_kvas.szlSrcSize.cx, m_kvas.szlSrcSize.cy,
                     m_kvas.szlSrcSize.cx, pSrcBmp->linesize[ 0 ], pSrcBmp->linesize[ 1 ]);
            break;
    }
}

VOID imgInit( ULONG kvaMode, HWND hwnd, ULONG ulKeyColor, LONG cx, LONG cy,
              ULONG ulRatio, PULONG pulAttrValue )
{
    int i;

    kvaInit( kvaMode, hwnd, ulKeyColor );

    imgConvertInit();

    m_kvas.ulLength = sizeof( KVASETUP );
    m_kvas.szlSrcSize.cx = cx;
    m_kvas.szlSrcSize.cy = cy;
    m_kvas.rclSrcRect.xLeft = 0;
    m_kvas.rclSrcRect.yTop = 0;
    m_kvas.rclSrcRect.xRight = cx;
    m_kvas.rclSrcRect.yBottom = cy;
    m_kvas.ulRatio = ulRatio;
    m_kvas.fccSrcColor = m_fcc;

    kvaSetup( &m_kvas );

    for( i = 0; i < KVAA_LAST; i++ )
        kvaSetAttr( i, &pulAttrValue[ i ] );
}

VOID imgDone( VOID )
{
    imgConvertDone();

    imgResetAttr();

    kvaDone();
}

YUV *imgCreateYUV( VOID )
{
    YUV  *p;
    LONG lSize = m_kvas.szlSrcSize.cx * m_kvas.szlSrcSize.cy;

    p = malloc( sizeof( YUV ));
    if( !p )
        return NULL;

    p->p = malloc(( lSize * 3 ) / 2 );

    if( !p->p )
    {
        free( p );

        return NULL;
    }

    p->linesize[ 0 ] = m_kvas.szlSrcSize.cx;
    p->linesize[ 1 ] = m_kvas.szlSrcSize.cx / 2;
    p->linesize[ 2 ] = m_kvas.szlSrcSize.cx / 2;
    p->linesize[ 3 ] = 0;

    p->data[ 0 ] = p->p;
    p->data[ 1 ] = p->data[ 0 ] + lSize;
    p->data[ 2 ] = p->data[ 1 ] + lSize / 4;
    p->data[ 3 ] = NULL;

    return p;
}

VOID imgFreeYUV( YUV *p )
{
    if( p )
    {
        if( p->p )
            free( p->p );

        free( p );
    }
}

VOID imgDisplayYUV( YUV *bmp )
{
    PVOID       pBuffer;
    ULONG       ulBPL;

    if( !kvaLockBuffer( &pBuffer, &ulBPL ))
    {
        imgConvert( pBuffer, ulBPL, bmp );

        kvaUnlockBuffer();
    }
}

VOID imgClearRect( PRECTL prcl )
{
    kvaClearRect( prcl );
}

VOID imgSetAspectRatio( ULONG ulRatio )
{
    m_kvas.ulRatio = ulRatio;

    kvaSetup( &m_kvas );
}

APIRET imgQueryAttr( ULONG ulAttr, PULONG pulValue )
{
    return kvaQueryAttr( ulAttr, pulValue );
}

APIRET imgSetAttr( ULONG ulAttr, PULONG pulValue )
{
    return kvaSetAttr( ulAttr, pulValue );
}

VOID imgResetAttr( VOID )
{
    kvaResetAttr();
}

