/*
 * YV12 to YUY2, YUV420P to RGB conversion library
 * Copyright (C) 2007 by KO Myung-Hun <komh@chollian.net>
 *
 * based on WarpVision yuv2rgb
 *
 * based on mpeg2dec & MPlayer postproc & xine gamma stuff
 *
 * Copyright (C) 2000-2002 Michel Lespinasse <walken@zoy.org>
 * Copyright (C) 1999-2000 Aaron Holtzman <aholtzma@ess.engr.uvic.ca>
 *
 * 15,24 bpp and dithering from Michael Niedermayer (michaelni@gmx.at)
 * MMX/MMX2 Template stuff from Michael Niedermayer (michaelni@gmx.at) (needed for fast movntq support)
 *
 * This file is part of mpeg2dec, a free MPEG-2 video stream decoder.
 * See http://libmpeg2.sourceforge.net/ for updates.
 *
 * mpeg2dec is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * mpeg2dec is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <assert.h>

#include <types.h>

#include "dsputil.h"

#include "yuv2rgb.h"

yuv420_fun yuv2rgb;
yv12toyuy2_fun yuv2yuv;

void * table_rV[256];
void * table_gU[256];
int    table_gV[256];
void * table_bU[256];
void * t8, *t16;
int verbose = 0;
int _gamma = 0;
int entry_size = 0;
uint32_t matrix_coefficients = 6;

const int32_t Inverse_Table_6_9[8][4] = {
    {117504, 138453, 13954, 34903}, /* no sequence_display_extension */
    {117504, 138453, 13954, 34903}, /* ITU-R Rec. 709 (1990) */
    {104597, 132201, 25675, 53279}, /* unspecified */
    {104597, 132201, 25675, 53279}, /* reserved */
    {104448, 132798, 24759, 53109}, /* FCC */
    {104597, 132201, 25675, 53279}, /* ITU-R Rec. 624-4 System B, G */
    {104597, 132201, 25675, 53279}, /* SMPTE 170M */
    {117579, 136230, 16907, 35559}  /* SMPTE 240M (1987) */
};


extern void yuv2rgb_pxor( void );
extern void yuv2rgb_load( uint8_t *py, uint8_t *pu, uint8_t *pv );

extern void yuv2rgb_proc( void );
//extern void emms( void );
extern void yuv2rgb_16_mmx( uint8_t *py, uint8_t *pu, uint8_t *pv, uint8_t *image );
extern void yuv2rgb_16_mmx2( uint8_t *py, uint8_t *pu, uint8_t *pv, uint8_t *image );
extern void yuv2rgb_24_mmx( uint8_t *py, uint8_t *pu, uint8_t *pv, uint8_t *image );
extern void yuv2rgb_24_mmx2( uint8_t *py, uint8_t *pu, uint8_t *pv, uint8_t *image );

extern void yv12toyuy2_proc_mmx(uint8_t *dst, const uint8_t *ysrc,
                           const uint8_t *usrc, const uint8_t *vsrc,
                           int chromWidth);

extern void yv12toyuy2_proc_mmx2(uint8_t *dst, const uint8_t *ysrc,
                           const uint8_t *usrc, const uint8_t *vsrc,
                           int chromWidth);

#if 0
extern void yv12toyuy2_proc_3dnow(uint8_t *dst, const uint8_t *ysrc,
                           const uint8_t *usrc, const uint8_t *vsrc,
                           int chromWidth);
#endif

extern void yuv2rgb_set_gamma_mmx( int gamma );


#define RGB(i)                          \
    U = pu[i];                      \
    V = pv[i];                      \
    r = table_rV[V];                    \
    g = (void *) (((uint8_t *)table_gU[U]) + table_gV[V]);  \
    b = table_bU[U];

#define DST1(i)                 \
    Y = py_1[2*i];              \
    dst_1[2*i] = r[Y] + g[Y] + b[Y];    \
    Y = py_1[2*i+1];            \
    dst_1[2*i+1] = r[Y] + g[Y] + b[Y];

#define DST2(i)                 \
    Y = py_2[2*i];              \
    dst_2[2*i] = r[Y] + g[Y] + b[Y];    \
    Y = py_2[2*i+1];            \
    dst_2[2*i+1] = r[Y] + g[Y] + b[Y];

#define DST1RGB(i)                          \
    Y = py_1[2*i];                          \
    dst_1[6*i] = r[Y]; dst_1[6*i+1] = g[Y]; dst_1[6*i+2] = b[Y];    \
    Y = py_1[2*i+1];                        \
    dst_1[6*i+3] = r[Y]; dst_1[6*i+4] = g[Y]; dst_1[6*i+5] = b[Y];

#define DST2RGB(i)                          \
    Y = py_2[2*i];                          \
    dst_2[6*i] = r[Y]; dst_2[6*i+1] = g[Y]; dst_2[6*i+2] = b[Y];    \
    Y = py_2[2*i+1];                        \
    dst_2[6*i+3] = r[Y]; dst_2[6*i+4] = g[Y]; dst_2[6*i+5] = b[Y];

#define DST1BGR(i)                          \
    Y = py_1[2*i];                          \
    dst_1[6*i] = b[Y]; dst_1[6*i+1] = g[Y]; dst_1[6*i+2] = r[Y];    \
    Y = py_1[2*i+1];                        \
    dst_1[6*i+3] = b[Y]; dst_1[6*i+4] = g[Y]; dst_1[6*i+5] = r[Y];

#define DST2BGR(i)                          \
    Y = py_2[2*i];                          \
    dst_2[6*i] = b[Y]; dst_2[6*i+1] = g[Y]; dst_2[6*i+2] = r[Y];    \
    Y = py_2[2*i+1];                        \
    dst_2[6*i+3] = b[Y]; dst_2[6*i+4] = g[Y]; dst_2[6*i+5] = r[Y];


void yuv420_rgb16_mmx(uint8_t * image, uint8_t * py,
                      uint8_t * pu, uint8_t * pv,
                      int h_size, int v_size,
                      int rgb_stride, int y_stride, int uv_stride)
{
    int even = 1;
    int x, y;

    yuv2rgb_pxor();

    for (y = v_size; --y >= 0; ) {
    uint8_t *_image = image;
    uint8_t *_py = py;
    uint8_t *_pu = pu;
    uint8_t *_pv = pv;
        int internal_h_size = h_size;
        int aligned_h_size = (h_size + 7) &~ 7;

        if(rgb_stride >= aligned_h_size * 2) internal_h_size = aligned_h_size;

        /* load data for start of next scan line */
        yuv2rgb_load(_py, _pu, _pv);

        for (x = internal_h_size >> 3; --x >= 0; ) {
        /* this mmx assembly code deals with SINGLE scan line at a time,
               it convert 8 pixels in each iteration */

            yuv2rgb_proc();
            yuv2rgb_16_mmx(_py, _pu, _pv, _image);

        _py += 8;
        _pu += 4;
        _pv += 4;
        _image += 16;
        }

    if (!even) {
        pu += uv_stride;
        pv += uv_stride;
    }

    py += y_stride;
    image += (rgb_stride * 2);

    even = (!even);
    }

    emms();

    return;
}


void yuv420_rgb16_mmx2(uint8_t * image, uint8_t * py,
                       uint8_t * pu, uint8_t * pv,
                       int h_size, int v_size,
                       int rgb_stride, int y_stride, int uv_stride)
{
    int even = 1;
    int x, y;

    yuv2rgb_pxor();

    for (y = v_size; --y >= 0; ) {
    uint8_t *_image = image;
    uint8_t *_py = py;
    uint8_t *_pu = pu;
    uint8_t *_pv = pv;
        int internal_h_size = h_size;
        int aligned_h_size = (h_size + 7) &~ 7;

        if(rgb_stride >= aligned_h_size * 2) internal_h_size = aligned_h_size;

    /* load data for start of next scan line */
        yuv2rgb_load(_py, _pu, _pv);

        for (x = internal_h_size >> 3; --x >= 0; ) {
        /* this mmx assembly code deals with SINGLE scan line at a time,
               it convert 8 pixels in each iteration */

            yuv2rgb_proc();
            if (!_image || !_py || !_pu || !_pv) return;
            yuv2rgb_16_mmx2(_py, _pu, _pv, _image);

        _py += 8;
        _pu += 4;
        _pv += 4;
        _image += 16;
        }

    if (!even) {
        pu += uv_stride;
        pv += uv_stride;
    }

    py += y_stride;
    image += (rgb_stride * 2);

    even = (!even);
    }

    emms();

    return;
}


void yuv420_rgb24_mmx(uint8_t * image, uint8_t * py,
                      uint8_t * pu, uint8_t * pv,
                      int h_size, int v_size,
                      int rgb_stride, int y_stride, int uv_stride)
{
    int even = 1;
    int x, y;

    yuv2rgb_pxor();

    for (y = v_size; --y >= 0; ) {
    uint8_t *_image = image;
    uint8_t *_py = py;
    uint8_t *_pu = pu;
    uint8_t *_pv = pv;
    int internal_h_size = h_size;
    int aligned_h_size = (h_size + 7) &~ 7;

    if(rgb_stride >= aligned_h_size * 3) internal_h_size = aligned_h_size;

    /* load data for start of next scan line */
        yuv2rgb_load(_py, _pu, _pv);

    for (x = internal_h_size >> 3; --x >= 0; ) {
        /* this mmx assembly code deals with SINGLE scan line at a time,
               it convert 8 pixels in each iteration */

            yuv2rgb_proc();
            yuv2rgb_24_mmx(_py, _pu, _pv, _image);

        _py += 8;
        _pu += 4;
        _pv += 4;
        _image += 24;
        }

    if (!even) {
        pu += uv_stride;
        pv += uv_stride;
    }

    py += y_stride;
    image += (rgb_stride * 3);

    even = (!even);
    }

    emms();

    return;
}


void yuv420_rgb24_mmx2(uint8_t * image, uint8_t * py,
                       uint8_t * pu, uint8_t * pv,
                       int h_size, int v_size,
                       int rgb_stride, int y_stride, int uv_stride)
{
    int even = 1;
    int x, y;

    yuv2rgb_pxor();

    for (y = v_size; --y >= 0; ) {
    uint8_t *_image = image;
    uint8_t *_py = py;
    uint8_t *_pu = pu;
    uint8_t *_pv = pv;
    int internal_h_size = h_size;
    int aligned_h_size = (h_size + 7) &~ 7;

    if(rgb_stride >= aligned_h_size * 3) internal_h_size = aligned_h_size;

    /* load data for start of next scan line */
        yuv2rgb_load(_py, _pu, _pv);

    for (x = internal_h_size >> 3; --x >= 0; ) {
        /* this mmx assembly code deals with SINGLE scan line at a time,
               it convert 8 pixels in each iteration */

            yuv2rgb_proc();
            yuv2rgb_24_mmx2(_py, _pu, _pv, _image);

        _py += 8;
        _pu += 4;
        _pv += 4;
        _image += 24;
        }

    if (!even) {
        pu += uv_stride;
        pv += uv_stride;
    }

    py += y_stride;
    image += (rgb_stride * 3);

    even = (!even);
    }

    emms();

    return;
}


void __inline yuv2rgb16_c (uint8_t * py_1, uint8_t * py_2,
                           uint8_t * pu, uint8_t * pv,
                           void * _dst_1, void * _dst_2, int width)
{
    int U, V, Y;
    uint16_t * r, * g, * b;
    uint16_t * dst_1, * dst_2;

    width >>= 3;
    dst_1 = _dst_1;
    dst_2 = _dst_2;

    do {
    RGB(0);
    DST1(0);
    DST2(0);

    RGB(1);
    DST2(1);
    DST1(1);

    RGB(2);
    DST1(2);
    DST2(2);

    RGB(3);
    DST2(3);
    DST1(3);

    pu += 4;
    pv += 4;
    py_1 += 8;
    py_2 += 8;
    dst_1 += 8;
    dst_2 += 8;
    } while (--width);
}


void __inline yuv2bgr24_c(uint8_t * py_1, uint8_t * py_2,
                          uint8_t * pu, uint8_t * pv,
                          void * _dst_1, void * _dst_2, int width)
{
    int U, V, Y;
    uint8_t * r, * g, * b;
    uint8_t * dst_1, * dst_2;

    width >>= 3;
    dst_1 = _dst_1;
    dst_2 = _dst_2;

    do {
    RGB(0);
    DST1BGR(0);
    DST2BGR(0);

    RGB(1);
    DST2BGR(1);
    DST1BGR(1);

    RGB(2);
    DST1BGR(2);
    DST2BGR(2);

    RGB(3);
    DST2BGR(3);
    DST1BGR(3);

    pu += 4;
    pv += 4;
    py_1 += 8;
    py_2 += 8;
    dst_1 += 24;
    dst_2 += 24;
    } while (--width);
}


void yuv420_rgb16_c (uint8_t * image, uint8_t * py,
                     uint8_t * pu, uint8_t * pv,
                     int h_size, int v_size,
                     int rgb_stride, int y_stride, int uv_stride)
{
    int even = 1;

    do
    {
    yuv2rgb16_c(py, py + y_stride, pu, pv,
                    image, image + rgb_stride, h_size);

    if (!even) {
        pu += uv_stride;
        pv += uv_stride;
    }

    py += y_stride;
    image += (rgb_stride*2);

    even = (!even);

    } while (--v_size);
}

void yuv420_rgb24_c (uint8_t * image, uint8_t * py,
                     uint8_t * pu, uint8_t * pv,
                     int h_size, int v_size,
                     int rgb_stride, int y_stride, int uv_stride)
{
    int even = 1;

    do
    {
    yuv2bgr24_c(py, py + y_stride, pu, pv,
                    image, image + rgb_stride, h_size);

    if (!even) {
        pu += uv_stride;
        pv += uv_stride;
    }

    py += y_stride;
    image += 3 * rgb_stride;

    even = (!even);

    } while (--v_size);
}


int __inline div_round (int dividend, int divisor)
{
    if (dividend > 0)
    return (dividend + (divisor>>1)) / divisor;
    else
    return -((-dividend + (divisor>>1)) / divisor);
}

void yuv420_rgb_close(void)
{
    if (t8) av_free(t8);
    if (t16) av_free(t16);
}

void yuv420_rgb_init(int Colors)
{
    int mm_flags;
    int i;
    uint8_t table_Y[1024];
    //uint32_t * table_32 = 0;
    uint16_t * table_16 = 0;
    uint8_t * table_8 = 0;
    void *table_r = 0, *table_g = 0, *table_b = 0;

    int crv = Inverse_Table_6_9[matrix_coefficients][0];
    int cbu = Inverse_Table_6_9[matrix_coefficients][1];
    int cgu = -Inverse_Table_6_9[matrix_coefficients][2];
    int cgv = -Inverse_Table_6_9[matrix_coefficients][3];

    mm_flags = mm_support();

    for (i = 0; i < 1024; i++)
    {
    int j;

    j = (76309 * (i - 384 - 16) + 32768) >> 16;
    j = (j < 0) ? 0 : ((j > 255) ? 255 : j);

    table_Y[i] = j;
    }

    //av_log( NULL, AV_LOG_INFO, "MMX flags: %X\n", mm_flags);

    switch(Colors)
    {
        case 16:
            if (!mm_flags & MM_MMX)
            {
                av_log( NULL, AV_LOG_INFO, "No accelerated YUV420P to RGB16 transform\n");

                yuv2rgb = yuv420_rgb16_c;

                table_16 = av_malloc ((197 + 2*682 + 256 + 132) * sizeof (uint16_t));

                t16 = table_16;

                entry_size = sizeof (uint16_t);
                table_r = table_16 + 197;
                table_b = table_16 + 197 + 685;
                table_g = table_16 + 197 + 2*682;

                for (i = -197; i < 256+197; i++)
                {
                    int j = table_Y[i+384] >> 3;

                    j <<= 11;

                    ((uint16_t *)table_r)[i] = j;
                }

                for (i = -132; i < 256+132; i++)
                {
                    int j = table_Y[i+384] >> 2;

                    ((uint16_t *)table_g)[i] = j << 5;
                }

                for (i = -232; i < 256+232; i++)
                {
                    int j = table_Y[i+384] >> 3;

                    ((uint16_t *)table_b)[i] = j;
                }

            }
            else if (mm_flags & MM_MMXEXT)
            {
                av_log( NULL, AV_LOG_INFO, "Using MMXEXT for YUV420P to RGB16 transform\n");

                yuv2rgb = yuv420_rgb16_mmx2;
            }
            else if (mm_flags & MM_MMX)
            {
                av_log( NULL, AV_LOG_INFO, "Using MMX for YUV420P to RGB16 transform\n");

                yuv2rgb = yuv420_rgb16_mmx;
            }
            else
            {
                av_log( NULL, AV_LOG_INFO, "No acceleration found...\n");
                exit(1);
            }
        break;

        case 24:
        case 32:
            if (!mm_flags & MM_MMX)
            {
                av_log( NULL, AV_LOG_INFO, "No accelerated YUV420P to RGB24 transform\n");

                table_8 = av_malloc ((256 + 2*232) * sizeof (uint8_t));

                t8 = table_8;

                entry_size = sizeof (uint8_t);
                table_r = table_g = table_b = table_8 + 232;

                for (i = -232; i < 256+232; i++)
                    ((uint8_t * )table_b)[i] = table_Y[i+384];

                yuv2rgb = yuv420_rgb24_c;
            }
            else if (mm_flags & MM_MMXEXT)
            {
                av_log( NULL, AV_LOG_INFO, "Using MMXEXT for YUV420P to BGR24 transform\n");

                yuv2rgb = yuv420_rgb24_mmx2;
            }
            else if (mm_flags & MM_MMX)
            {
                av_log( NULL, AV_LOG_INFO, "Using MMX for YUV420P to BGR24 transform\n");

                yuv2rgb = yuv420_rgb24_mmx;
            }
            else
            {
                av_log( NULL, AV_LOG_INFO, "No acceleration found...\n");
                exit(1);
            }
        break;
    }

    for (i = 0; i < 256; i++)
    {
        table_rV[i] = (((uint8_t *)table_r) +
                       entry_size * div_round (crv * (i-128), 76309));
        table_gU[i] = (((uint8_t *)table_g) +
                       entry_size * div_round (cgu * (i-128), 76309));
        table_gV[i] = entry_size * div_round (cgv * (i-128), 76309);
        table_bU[i] = (((uint8_t *)table_b) +
                       entry_size * div_round (cbu * (i-128), 76309));
    }

    return;
}

void yv12toyuy2_proc_c(uint8_t *dst, const uint8_t *ysrc, const uint8_t *usrc, const uint8_t *vsrc, const int chromWidth)
{
    int i;

    for(i=0; i<chromWidth; i++)
    {
        dst[4*i+0] = ysrc[2*i+0];
        dst[4*i+1] = usrc[i];
        dst[4*i+2] = ysrc[2*i+1];
        dst[4*i+3] = vsrc[i];
    }
}

void yvu9toyuy2(const uint8_t *src1, const uint8_t *src2, const uint8_t *src3,
            uint8_t *dst,
            unsigned width, unsigned height,
            unsigned srcStride1, unsigned srcStride2,
            unsigned srcStride3, unsigned dstStride)
{
    unsigned y,x,x2,w,h;
    w=width/2; h=height;
    for(y=0;y<h;y++){
    const uint8_t* yp=src1+srcStride1*y;
    const uint8_t* up=src2+srcStride2*(y>>2);
    const uint8_t* vp=src3+srcStride3*(y>>2);
    uint8_t* d=dst+dstStride*y;
    x2=0;
    x=0;
    for(;x<w;x++,x2+=4)
    {
        d[8*x+0]=yp[x2];
        d[8*x+1]=up[x];
        d[8*x+2]=yp[x2+1];
        d[8*x+3]=vp[x];
        d[8*x+4]=yp[x2+2];
        d[8*x+5]=up[x];
        d[8*x+6]=yp[x2+3];
        d[8*x+7]=vp[x];
    }
    }
}

void yv12toyuy2(const uint8_t *ysrc, const uint8_t *usrc, const uint8_t *vsrc,
                uint8_t *dst, unsigned int width, unsigned int height,
                unsigned int lumStride, unsigned int chromStride,
                unsigned int dstStride, int vertLumPerChroma)
{
    int y;
    const int chromWidth = width >> 1;

    for(y = 0; y < height; y++)
    {
        yuv2yuv(dst, ysrc, usrc, vsrc, chromWidth);

        if ((y&(vertLumPerChroma-1)) == (vertLumPerChroma-1))
        {
            usrc += chromStride;
            vsrc += chromStride;
        }

        ysrc += lumStride;
        dst += dstStride;
    }
    if (mm_flags & MM_MMX)
      emms();
}

void yv12toyuy2_init(void)
{
    int mm_flags;

    mm_flags = mm_support();

    //av_log( NULL, AV_LOG_INFO, "MMX flags: %X\n", mm_flags);
#if 0
    if (mm_flags & MM_3DNOW)
    {
        yuv2yuv = yv12toyuy2_proc_3dnow;
        av_log( NULL, AV_LOG_INFO, "Using 3DNOW for YUV420P to YUV422 transform\n");
    } else
#endif
    if (mm_flags & MM_MMXEXT)
    {
        yuv2yuv = yv12toyuy2_proc_mmx2;
        av_log( NULL, AV_LOG_INFO, "Using MMXEXT for YUV420P to YUV422 transform\n");
    }
    else if (mm_flags & MM_MMX)
    {
        yuv2yuv = yv12toyuy2_proc_mmx;
        av_log( NULL, AV_LOG_INFO, "Using MMX for YUV420P to YUV422 transform\n");
    }
    else if (!mm_flags & MM_MMX)
    {
        yuv2yuv = yv12toyuy2_proc_c;
        av_log( NULL, AV_LOG_INFO, "No accelerated YUV420P to YUV422 transform\n");
    }
    else
    {
        av_log( NULL, AV_LOG_INFO, "No acceleration found...\n");
        exit(1);
    }
}

void yuv2rgb_set_gamma(int new_gamma)
{
    int i;

    yuv2rgb_set_gamma_mmx(new_gamma);

    for (i = 0; i < 256; i++)
    {
        table_rV[i] = (uint8_t *)table_rV[i] + (entry_size * (new_gamma - _gamma));
        table_gU[i] = (uint8_t *)table_gU[i] + (entry_size * (new_gamma - _gamma));
        table_bU[i] = (uint8_t *)table_bU[i] + (entry_size * (new_gamma - _gamma));
    }

    _gamma = new_gamma;
}
