/*
    KMP : Subtitle library for OS/2
    Copyright (C) 2007 by KO Myung-Hun <komh@chollian.net>

    This file is part of KMP.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define INCL_WIN
#define INCL_GPI
#include <os2.h>

#include <stdlib.h>
#include <string.h>

#include "subreader.h"

#include "kmp_sub.h"

static sub_data *m_sd;
static subtitle *m_cur_sub;
static BOOL     m_fErase;
static HDC      m_hdc;
static HPS      m_hps;
static HBITMAP  m_hbm;
static RECTL    m_rclCurSub;
static RECTL    m_rclInvalidSub;
static LONG     m_lHoriFontRes;
static LONG     m_lVertFontRes;
static FIXED    m_fxPointSize;
static CHAR     m_szFaceName[ FACESIZE + 1 ];

static VOID sbmDelete( VOID );

static VOID sbmInit( HAB hab, PCSZ pcszFontNameSize )
{
    HPS     hps;
    HDC     hdc;
    char   *facename;
    SIZEL   sizl;

    hps = WinGetScreenPS( HWND_DESKTOP );
    hdc = GpiQueryDevice( hps );
    DevQueryCaps( hdc, CAPS_HORIZONTAL_FONT_RES, 1, &m_lHoriFontRes );
    DevQueryCaps( hdc, CAPS_VERTICAL_FONT_RES, 1, &m_lVertFontRes );
    WinReleasePS( hps );

    m_fxPointSize = MAKEFIXED( strtol( pcszFontNameSize, &facename, 0 ), 0 );
    strcpy( m_szFaceName, facename + 1 );

    m_hdc = DevOpenDC( hab, OD_MEMORY, "*", 0L, NULL, NULLHANDLE);

    sizl.cx = 0;
    sizl.cy = 0;
    m_hps = GpiCreatePS( hab, m_hdc, &sizl,
                            PU_PELS | GPIF_DEFAULT | GPIT_MICRO | GPIA_ASSOC );
}

static VOID sbmDone( VOID )
{
    sbmDelete();

    GpiDestroyPS( m_hps );

    DevCloseDC( m_hdc );
}

#define SHADOW_DISTANCE 1

static VOID outliner( HPS hps, INT x, INT y, PSZ szMsg,
                      LONG color, LONG colorOutline, LONG colorShadow, BOOL f3d )
{
    static LONG outlineColor[] = { 0x000000, };
    INT         outlineN = sizeof( outlineColor ) / sizeof( outlineColor[ 0 ] );
    INT         i, j;
    INT         len = strlen( szMsg );
    POINTL      ptl;
    LONG        oldColor;

    oldColor = GpiQueryColor( hps );

    outlineColor[ 0 ] = colorOutline;

    if( f3d )
    {
        GpiSetColor( hps, colorShadow );
        ptl.x = x + ( outlineN + SHADOW_DISTANCE );
        ptl.y = y - ( outlineN + SHADOW_DISTANCE );
        GpiCharStringAt( hps, &ptl, len, szMsg );
    }

    for( i = outlineN; i > 0; i-- )
    {

        GpiSetColor( hps, outlineColor[ i - 1 ]);

        for( j = -i; j <= i; j++ )
        {
            ptl.x = x - i;
            ptl.y = y + j;
            GpiCharStringAt( hps, &ptl, len, szMsg );

            ptl.x = x + i;
            ptl.y = y + j;
            GpiCharStringAt( hps, &ptl, len, szMsg );
        }

        for( j = -( i - 1 ); j <= ( i - 1 ); j++ )
        {
            ptl.x = x + j;
            ptl.y = y + i;
            GpiCharStringAt( hps, &ptl, len, szMsg );

            ptl.x = x + j;
            ptl.y = y - i;
            GpiCharStringAt( hps, &ptl, len, szMsg );
        }
    }

    GpiSetColor( hps, color );
    ptl.x = x;
    ptl.y = y;
    GpiCharStringAt( hps, &ptl, len, szMsg );

    GpiSetColor( hps, oldColor );
}

static int subSplit( subtitle *sub, int split_index )
{
    char *temp;
    int  len;
    int  i;

    if( sub->lines >= SUB_MAX_TEXT )
        return -1;

    if( strlen( sub->text[ split_index ]) < 5 )
        return -1;

    for( i = sub->lines - 1; i > split_index; i-- )
        sub->text[ i + 1 ] = sub->text[ i ];

    temp = strdup( sub->text[ split_index ]);
    len = strlen( temp );
    i = len / 2;

    while( temp[ i ] && temp[ i ] != ' ' )
        i++;

    if( !temp[ i ])
    {
        while( i && temp[ i ] != ' ' )
            i--;

        // FIXME : need to check DBCS chars.
        if( i == 0 )
            i = len / 2;
    }

    free( sub->text[ split_index ]);
    sub->text[ split_index ] = malloc( i + 1 );
    memcpy( sub->text[ split_index ], temp, i );
    sub->text[ split_index ][ i ] = 0;

    split_index++;

    while( temp[ i ] && temp[ i ] == ' ' )
        i++;

    sub->text[ split_index ] = malloc( len - i + 1 );
    memcpy( sub->text[ split_index ], temp + i, len - i );
    sub->text[ split_index ][ len - i ] = 0;

    free( temp );

    sub->lines++;

    return 0;
}

static VOID sbmCreate( LONG x, LONG y, ULONG ulWidth,
                       ULONG ulColor, ULONG ulColorOutline, ULONG ulColorShadow,
                       BOOL f3d )
{
    subtitle      *  temp_sub;
    FATTRS           fat;
    SIZEF            sizf;
    BITMAPINFOHEADER bmih;
    FONTMETRICS      fm;
    ULONG            ulHeight;
    POINTL           aptl[ TXTBOX_COUNT ];
    int              aextent[ SUB_MAX_TEXT ];
    int              y1;
    int              bmp_width, bmp_height;
    int              len, extent;
    int              i;

    if( m_hbm )
        return ;

    temp_sub = malloc( sizeof( subtitle ));
    memcpy( temp_sub, m_cur_sub, sizeof( subtitle ));
    for( i = 0; i < m_cur_sub->lines; i++ )
        temp_sub->text[ i ] = strdup( m_cur_sub->text[ i ]);

    fat.usRecordLength = sizeof( FATTRS );
    fat.fsSelection = 0;
    fat.lMatch = 0;
    fat.idRegistry = 0;
    fat.usCodePage = 0;
    fat.lMaxBaselineExt = 0;
    fat.lAveCharWidth = 0;
    fat.fsType = 0;
    fat.fsFontUse = FATTR_FONTUSE_NOMIX;
    strcpy( fat.szFacename, m_szFaceName );
    GpiCreateLogFont( m_hps, NULL, 1, &fat );
    GpiSetCharSet( m_hps, 1 );

    sizf.cx = m_fxPointSize * m_lHoriFontRes / 72;
    sizf.cy = m_fxPointSize * m_lVertFontRes / 72;
    GpiSetCharBox( m_hps, &sizf );

    bmp_width = 0;

    for( i = 0; i < temp_sub->lines; i++ )
    {
        len = strlen( temp_sub->text[ i ]);
        GpiQueryTextBox( m_hps, len, temp_sub->text[ i ], TXTBOX_COUNT, aptl );
        extent = aptl[ TXTBOX_CONCAT ].x - aptl[ TXTBOX_BOTTOMLEFT ].x + 3; // 3 for outline and shadow
        if( extent > ulWidth )
        {
            if( !subSplit( temp_sub, i ))
            {
                i--;
                continue;
            }
        }
        if( bmp_width < extent )
            bmp_width = extent;

        aextent[ i ] = extent;
    }

    GpiQueryFontMetrics( m_hps, sizeof( FONTMETRICS ), &fm );
    ulHeight = fm.lMaxBaselineExt + fm.lExternalLeading + 3; // 3 for outline and shadow

    bmp_height = ulHeight * temp_sub->lines;

    memset( &bmih, 0, sizeof( BITMAPINFOHEADER ));

    bmih.cbFix = sizeof( BITMAPINFOHEADER );
    bmih.cx = bmp_width;
    bmih.cy = bmp_height;
    bmih.cPlanes = 1;
    bmih.cBitCount = 24;

    m_hbm = GpiCreateBitmap( m_hps, ( PBITMAPINFOHEADER2 )&bmih, 0, NULL, NULL );

    GpiSetBitmap( m_hps, m_hbm );

    GpiCreateLogColorTable( m_hps, 0, LCOLF_RGB, 0, 0, NULL );

    y1 = fm.lMaxDescender + ulHeight * ( temp_sub->lines - 1 ) + 2; // 2 for outline and shadow

    for( i = 0; i < temp_sub->lines; i++ )
    {                                                 // 1 for left shadow
        outliner( m_hps, ( bmp_width - aextent[ i ]) / 2 + 1, y1, temp_sub->text[ i ],
                  ulColor, ulColorOutline, ulColorShadow, f3d );

        y1 -= ulHeight;
    }

    for( i = 0; i < temp_sub->lines; i++ )
        free( temp_sub->text[ i ]);
    free( temp_sub );

    m_rclCurSub.xLeft = ( ulWidth - bmp_width ) / 2 + x;
    m_rclCurSub.xRight = m_rclCurSub.xLeft + bmp_width;
    m_rclCurSub.yBottom = y;
    m_rclCurSub.yTop = m_rclCurSub.yBottom + bmp_height;
}

static VOID sbmDelete( VOID )
{
    GpiSetBitmap( m_hps, NULLHANDLE );
    GpiDeleteBitmap( m_hbm );
    m_hbm = NULLHANDLE;
}

APIRET subInit( HAB hab, PCSZ pcszFileName, PCSZ pcszFontNameSize, float fps )
{
    char **subfilenames;

    m_sd = NULL;

    /* find subtitle file */
    subfilenames = sub_filenames( "", ( char * )pcszFileName );

    /* found subtitle file */
    if( *subfilenames )
    {
        char *name = strdup( *subfilenames );

        /* read subtitle file, use only first subtitle */
        m_sd = sub_read_file( name, fps );
        if( m_sd )
        {
            m_cur_sub = m_sd->subtitles;

            sbmInit( hab, pcszFontNameSize );
        }
        else
            free( name );
    }

    /* free subtitle filename list */
    if( subfilenames )
    {
        char **temp = subfilenames;

        while( *temp )
            free( *temp++ );

        free( subfilenames );
    }

    return ( m_sd == NULL );
}

VOID subDone( VOID )
{
    sbmDone();

    sub_free( m_sd );
}

BOOL subInTime( ULONG ulTime )
{
    return ( ulTime >= m_cur_sub->start && ulTime <= m_cur_sub->end );
}

VOID subFindNext( ULONG ulTime )
{
    subtitle *temp_sub = m_cur_sub;

    while( temp_sub != m_sd->subtitles && ulTime < temp_sub->start )
        temp_sub--;
    while( temp_sub->lines && ulTime > temp_sub->end )
        temp_sub++;

    if( m_cur_sub != temp_sub )
    {
        subInvalidate();

        m_cur_sub = temp_sub;
    }
}

VOID subDisplay( HPS hps, LONG x, LONG y, ULONG ulWidth,
                 ULONG ulColor, ULONG ulColorOutline, ULONG ulColorShadow,
                 BOOL f3d )
{
    POINTL aptl[ 3 ];

    if( !m_cur_sub->lines )
        return;

    if( !m_hbm )
        sbmCreate( x, y, ulWidth, ulColor, ulColorOutline, ulColorShadow, f3d );

    // target
    aptl[ 0 ].x = m_rclCurSub.xLeft;
    aptl[ 0 ].y = m_rclCurSub.yBottom;
    aptl[ 1 ].x = m_rclCurSub.xRight;
    aptl[ 1 ].y = m_rclCurSub.yTop;

    // source
    aptl[ 2 ].x = 0;
    aptl[ 2 ].y = 0;

    GpiBitBlt( hps, m_hps, 3, aptl, ROP_SRCPAINT, BBO_IGNORE );
}

VOID subErase( HPS hps, ULONG ulColor )
{
    if( m_fErase )
    {
        GpiCreateLogColorTable( hps, 0, LCOLF_RGB, 0, 0, NULL );
        WinFillRect( hps, &m_rclInvalidSub, ulColor );

        m_fErase = FALSE;
    }
}

FIXED subQueryFontSize( VOID )
{
    return m_fxPointSize;
}

FIXED subSetFontSize( FIXED fxPointSize )
{
    m_fxPointSize = fxPointSize;

    if( FIXEDINT( m_fxPointSize ) < 6 )
        m_fxPointSize = MAKEFIXED( 6, 0 );
    if( FIXEDINT( m_fxPointSize ) > 144 )
        m_fxPointSize = MAKEFIXED( 144, 0 );

    subInvalidate();

    return m_fxPointSize;
}

PSZ subQueryFontName( PSZ pszFontName, ULONG ulLen  )
{
    strncpy( pszFontName, m_szFaceName, ulLen );
    pszFontName[ ulLen - 1 ] = 0;

    return pszFontName;
}

VOID subSetFontName( PCSZ pszFontName )
{
    strncpy( m_szFaceName, pszFontName, sizeof( m_szFaceName ));
    m_szFaceName[ sizeof( m_szFaceName ) - 1 ] = 0;

    subInvalidate();
}


VOID subInvalidate( VOID )
{
    sbmDelete();

    m_rclInvalidSub = m_rclCurSub;
    m_fErase = TRUE;
}

