/*
    K Video Accelerator library for OS/2
    Copyright (C) 2007 by KO Myung-Hun <komh@chollian.net>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define INCL_WIN
#define INCL_GPI
#include <os2.h>

#include <stdlib.h>
#include <string.h>

#include "kva.h"
#include "kva_internal.h"
#include "kva_dive.h"
#include "kva_wo.h"

APIRET ( APIENTRY *g_pfnDone )( VOID );
APIRET ( APIENTRY *g_pfnLockBuffer )( PPVOID ppBuffer, PULONG pulBPL );
APIRET ( APIENTRY *g_pfnUnlockBuffer )( VOID );
APIRET ( APIENTRY *g_pfnSetup )( PKVASETUP pkvas );
APIRET ( APIENTRY *g_pfnCaps )( PKVACAPS pkvac );

HWND  g_hwndKVA = NULLHANDLE;
ULONG g_ulKeyColor = -1;

static ULONG    m_ulRatio = KVAR_NONE;

static BOOL     m_fKVAInited = FALSE;
static BOOL     m_fLocked = FALSE;

APIRET APIENTRY kvaInit( ULONG kvaMode, HWND hwnd, ULONG ulKeyColor )
{
    ULONG   rc;

    if( m_fKVAInited )
        return KVAE_ALREADY_INITIALIZED;

    m_fKVAInited = FALSE;
    m_fLocked = FALSE;

    m_ulRatio = KVAR_NONE;

    g_pfnDone = NULL;
    g_pfnLockBuffer = NULL;
    g_pfnUnlockBuffer = NULL;
    g_pfnSetup = NULL;

    g_hwndKVA = NULLHANDLE;
    g_ulKeyColor = -1;

    rc = KVAE_INVALID_PARAMETER;

    if( !hwnd )
        return rc;

    g_hwndKVA = hwnd;
    g_ulKeyColor = ulKeyColor;

    if( kvaMode == KVAM_WO || kvaMode == KVAM_AUTO )
    {
        rc = woInit();
        if( rc && kvaMode == KVAM_WO )
            return rc;
    }

    if( rc )
        rc = diveInit();

    if( !rc )
        m_fKVAInited = TRUE;

    return rc;
}

APIRET APIENTRY kvaDone( VOID )
{
    if( !m_fKVAInited )
        return KVAE_NOT_INITIALIZED;

    m_fKVAInited = FALSE;

    return g_pfnDone();
}

APIRET APIENTRY kvaLockBuffer( PPVOID ppBuffer, PULONG pulBPL )
{
    ULONG   rc;

    if( !m_fKVAInited )
        return KVAE_NOT_INITIALIZED;

    if( m_fLocked )
        return KVAE_ALREADY_LOCKED;

    if( ppBuffer == NULL || pulBPL == NULL )
        return KVAE_INVALID_PARAMETER;

    rc = g_pfnLockBuffer( ppBuffer, pulBPL );
    if( rc )
        return rc;

    m_fLocked = TRUE;

    return KVAE_NO_ERROR;
}

APIRET APIENTRY kvaUnlockBuffer( VOID )
{
    ULONG   rc;

    if( !m_fKVAInited )
        return KVAE_NOT_INITIALIZED;

    if( !m_fLocked )
        return KVAE_NOT_LOCKED;

    rc = g_pfnUnlockBuffer();
    if( rc )
        return rc;

    m_fLocked = FALSE;

    return 0;
}

APIRET APIENTRY kvaSetup( PKVASETUP pkvas )
{
    if( !m_fKVAInited )
        return KVAE_NOT_INITIALIZED;

    if( !pkvas )
        return KVAE_INVALID_PARAMETER;

    m_ulRatio = pkvas->ulRatio;

    return g_pfnSetup( pkvas );
}

APIRET APIENTRY kvaCaps( PKVACAPS pkvac )
{
    if( !m_fKVAInited )
        return KVAE_NOT_INITIALIZED;

    if( !pkvac )
        return KVAE_INVALID_PARAMETER;

    memset( pkvac, 0, sizeof( PKVACAPS ));

    return g_pfnCaps( pkvac );
}

APIRET APIENTRY kvaClearWindow( VOID )
{
    HPS     hps;
    RECTL   rcl;

    hps = WinGetPS( g_hwndKVA );

    WinQueryWindowRect( g_hwndKVA, &rcl );
    GpiCreateLogColorTable( hps, 0, LCOLF_RGB, 0, 0, NULL );
    WinFillRect( hps, &rcl, g_ulKeyColor );

    WinReleasePS( hps );

    return KVAE_NO_ERROR;
}


APIRET APIENTRY kvaAdjustDstRect( HWND hwnd, PRECTL prclSrc, PRECTL prclDst )
{
    ULONG   cxSrc, cySrc;
    ULONG   cxDst, cyDst;
    ULONG   cxDstNew, cyDstNew;

    WinQueryWindowRect( hwnd, prclDst );

    cxDst = prclDst->xRight;
    cyDst = prclDst->yTop;

    switch( m_ulRatio )
    {

        case KVAR_ORIGINAL :
        case KVAR_FORCE43 :
        case KVAR_FORCE169:
            switch( m_ulRatio )
            {
                case KVAR_ORIGINAL :
                    cxSrc = labs( prclSrc->xRight - prclSrc->xLeft );
                    cySrc = labs( prclSrc->yTop - prclSrc->yBottom );
                    break;

                case KVAR_FORCE43 :
                    cxSrc = 4;
                    cySrc = 3;
                    break;

                case KVAR_FORCE169 :
                default :               // to prevent from uninitializing warning
                    cxSrc = 16;
                    cySrc = 9;
                    break;
            }

            cxDstNew = cxDst;
            cyDstNew = cxDstNew * cySrc / cxSrc;
            if( cyDstNew > cyDst )
            {
                cyDstNew = cyDst;
                cxDstNew = cyDstNew * cxSrc / cySrc;
            }
            break;

        case KVAR_NONE :
        default :   // assume KVAR_NONE
            cxDstNew = cxDst;
            cyDstNew = cyDst;
            break;
    }

    prclDst->xRight = cxDstNew;
    prclDst->yTop = cyDstNew;

    WinOffsetRect( WinQueryAnchorBlock( hwnd ), prclDst,
                   ( cxDst - cxDstNew ) / 2, ( cyDst - cyDstNew ) / 2 );

    WinMapWindowPoints( hwnd, HWND_DESKTOP, ( PPOINTL )prclDst, 2 );

    return KVAE_NO_ERROR;
}

