#include "definitions.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <process.h>
#include <string.h>
#include <debug.h>

static char            *pszProgram = NULL;
static FILE            *fdRead = NULL;
static int             hWrite = 0;
static int             pid = -1;

static int playlist_program_get_lineno(void);
static char* playlist_program_get_next(char *buf, int buf_len);
static char* playlist_program_get_metadata(void);
static int playlist_program_get_timelimit(void);
static int playlist_program_reload(void);
static void playlist_program_shutdown(void);

#define FLD_FILE_NAME                0
#define FLD_METADATA                 1
#define FLD_LINE_NO                  2
#define FLD_TIME_LIMIT               3
#define FLD_LOG                      4

// Allowed reply field names.
static char  *apszReplyFields[] =
{
  "FILE-NAME",   // FLD_FILE_NAME
  "METADATA",    // FLD_METADATA
  "LINE-NO",     // FLD_LINE_NO
  "TIME-LIMIT",  // FLD_TIME_LIMIT
  "LOG"          // FLD_LOG
};

static char            *pszMetadata = NULL;
static int             iLineNo = 0;
static int             iTimeLimit = 0;

#define REPLY_FIELDS   ( sizeof(apszReplyFields) / sizeof(char *) )
#define REPLY_VALUES   4


// int rwpOpen(PRWPIPE pRWPipe, PSZ pszCommand)
//
// Creates read/write pipe handles and runs shell command pszCommand.
// Returns 0 on error or 1 if pszCommand runned.

static int rwpOpen(char *pszCommand)
{
  int        ahRead[2];
  int        ahWrite[2];
  int        hOldIn, hOldOut;
  char       *pcCMD = getenv( "COMSPEC" );

  if ( pid != -1 )
  {
    ices_log_error_output( "Playlist program already runned." );
    return 0;
  }

  if ( _pipe( &ahRead, 512, _O_TEXT ) != 0 )
  {
    ices_log_error_output( "Cannot create the pipe (read)" );
    return 0;
  }

  if ( _pipe( &ahWrite, 256, _O_BINARY ) != 0 )
  {
    ices_log_error_output( "Cannot create the pipe (write)" );
    close( ahRead[0] );
    close( ahRead[1] );
    return 0;
  }

  hOldIn = dup( STDIN_FILENO ); 
  if ( hOldIn == -1 )
  {
    ices_log_error_output( "dup(STDIN_FILENO) failed" );
    close( ahRead[0] );
    close( ahRead[1] );
    close( ahWrite[0] );
    close( ahWrite[1] );
    return 0;
  }

  hOldOut = dup( STDOUT_FILENO ); 
  if ( hOldOut == -1 )
  {
    ices_log_error_output( "dup(STDOUT_FILENO) failed" );
    close( hOldIn );
    close( ahRead[0] );
    close( ahRead[1] );
    close( ahWrite[0] );
    close( ahWrite[1] );
    return 0;
  }

  dup2( ahWrite[0], STDIN_FILENO );
  dup2( ahRead[1], STDOUT_FILENO );
  close( ahWrite[0] );
  close( ahRead[1] );
  // Execute: C:\OS2\CMD.EXE /c <command>
  pid = spawnl( P_NOWAIT, pcCMD, "", "/c", pszCommand, NULL );

  dup2( hOldIn, STDIN_FILENO );
  dup2( hOldOut, STDOUT_FILENO );
  close( hOldIn );
  close( hOldOut );

  if ( pid == -1 )
  {
    ices_log_error_output( "Cannot start playlist program" );
    close( ahRead[0] );
    close( ahWrite[1] );
    return 0;
  }

  hWrite = ahWrite[1];
  fdRead = fdopen( ahRead[0], "r" );

  return 1;
}

static int sendCmd(char *pcCmd)
{
  if ( ( write( hWrite, pcCmd, strlen( pcCmd ) ) == -1 ) ||
       ( write( hWrite, "\r\n", 2 ) == -1 ) )
    return 0;

  return 1;
}


// int rwpClose()
//
// Waits for the program and closes read/write pipe handles.

static int rwpClose()
{
  int        iStatus;
  int        iRC;

  sendCmd( "SHUTDOWN" );
  iRC = cwait( &iStatus, pid, WAIT_CHILD );

  if ( fclose( fdRead ) != 0 )
    ices_log_error_output( "Cannot close playlist program read descriptor" );

  if ( close( hWrite ) != 0 )
    ices_log_error_output( "Cannot close playlist program write handle" );

  fdRead = NULL;
  hWrite = 0;
  pid = -1;

  if ( iRC == -1 )
  {
    ices_log_error_output( "Playlist program waiting failed" );
    return -1;
  }

  return iStatus & 0xff == 0 ? iStatus >> 8 : -1;
}


int ices_playlist_program_initialize(playlist_module_t* pm)
{
  char       *pszBinDir;
  char       szProgram[_MAX_PATH];
  char       szRandomize[16];

  ices_log_debug( "Initializing program playlist handler..." );
  pszMetadata = NULL;
  iLineNo = 0;
  iTimeLimit = 0;

  if ( pm->module == NULL )
  {
    ices_log_error_output( "No playlist program defined" );
    return -1;
  }

  if ( ( pm->module[1] == ':' ) || ( strchr( pm->module, '\\' ) != NULL ) )
  {
    pszProgram = pm->module;
  }
  else
  {
    if ( _snprintf( &szProgram, sizeof(szProgram), "%s\\%s",
                    ices_util_get_bindir(), pm->module ) < 0 )
    {
      ices_log_error_output( "Cannot build full file name for playlist program" );
      return -1;
    }

    pszProgram = &szProgram;
  }

  pszProgram = debugStrDup( pszProgram );
  if ( pszProgram == NULL )
  {
    ices_log_error_output( "Not enough memory for playlist program full file name" );
    return -1;
  }

  setenv( "ICES_PLAYLIST", pm->playlist_file, 1 );
  itoa( pm->randomize, &szRandomize, 10 ); 
  setenv( "ICES_RANDOMIZE", &szRandomize, 1 );

  if ( rwpOpen( pszProgram ) == 0 )
    return -1;

  pm->get_next = playlist_program_get_next;
  pm->reload = playlist_program_reload;
  pm->get_metadata = playlist_program_get_metadata;
  pm->get_timelimit = playlist_program_get_timelimit;
  pm->get_lineno = playlist_program_get_lineno;
  pm->shutdown = playlist_program_shutdown;

  return 1;
}

static int playlist_program_reload(void)
{
  if ( pszMetadata != NULL )
  {
    debugFree( pszMetadata );
    pszMetadata = NULL;
  }
  iLineNo = 0;
  iTimeLimit = 0;

  rwpClose();

  if ( rwpOpen( pszProgram ) == 0 )
    return 0;

  return 1;
}

static void playlist_program_shutdown(void)
{
  ices_log_debug( "Send SHUTDOWN to the program %s and wait...", pszProgram );
  rwpClose();

  if ( pszProgram != NULL )
  {
    debugFree( pszProgram );
    pszProgram = NULL;
  }

  if ( pszMetadata != NULL )
  {
    debugFree( pszMetadata );
    pszMetadata = NULL;
  }
}

static char* playlist_program_get_next(char *buf, int buf_len)
{
  char       szBuf[1024];
  char       *pCh;
  char       *pcValue;
  int        iIdx;
  int        iField;

  if ( !sendCmd( "NEXT" ) )
    ices_log_error_output( "Cannot send NEXT to the playlist program" );

  if ( pszMetadata != NULL )
    debugFree( pszMetadata );
  pszMetadata = NULL;

  iLineNo = 0;
  iTimeLimit = 0;

  buf[0] = '\0';

  while( 1 )
  {
    if ( fgets( &szBuf, sizeof(szBuf), fdRead ) == NULL )
    {
      ices_log_error( "Cannot read data from playlist program" );
      return NULL;
    }

    // Skip trailing CR, LF and spaces in the received string.
    pCh = strchr( &szBuf, '\0' );
    while( ( pCh > &szBuf ) && isspace( *(pCh - 1) ) )
      pCh--;

    if ( pCh == &szBuf )
      // Empty line - end of reply.
      break;

    *pCh = '\0';

    pCh = strchr( &szBuf, ':' );
    if ( pCh == NULL )
    {
      ices_log_error_output( "Invalid reply received from playlist program" );
      continue;
    }

    pcValue = pCh + 1;
    // Skip trailing spaces in the field name.
    while( ( pCh > &szBuf ) && isspace( *(pCh - 1) ) )
      pCh--;
    *pCh = '\0';

    // Skip leading spaces in value.
    while( isspace( *pcValue ) )
      pcValue++;

    iField = -1;
    for( iIdx = 0; iIdx < REPLY_FIELDS; iIdx++ )
    {
      if ( stricmp( &szBuf, apszReplyFields[iIdx] ) == 0 )
      {
        iField = iIdx;
        break;
      }
    }

    if ( iField == -1 )
    {
      ices_log_error_output( "Unknown reply field received from playlist "
                             "program: %s", &szBuf );
      continue;
    }

    ices_log_debug( "Playlist program reply field: %s, value: %s",
                    &szBuf, pcValue );

    switch( iField )
    {
      case FLD_FILE_NAME:
        strlcpy( buf, pcValue, buf_len );
        break;

      case FLD_METADATA:
        pszMetadata = debugStrDup( pcValue );
        break;

      case FLD_LINE_NO:
        iLineNo = atoi( pcValue );
        break;

      case FLD_TIME_LIMIT:
        iTimeLimit = atoi( pcValue );
        break;

      case FLD_LOG:
        // Field "Log" - write value to the log.
        ices_log( pcValue );
        break;
    }
  }

  return buf;
}

static int playlist_program_get_lineno(void)
{
  return iLineNo;
}

static char* playlist_program_get_metadata(void)
{
  char       *pcRet = pszMetadata;

  pszMetadata = NULL;
  return pcRet;
}

static int playlist_program_get_timelimit(void)
{
  return iTimeLimit;
}
