/** Plogg, a curse(s/d) mp3/Ogg player, based on original code
 * from "jb - a curses based jukebox for mp-compressed audio"
 * (C) 1997 Kristian Wiklund (under the GPL).
 * $Id: plogg.c 1.3 2003/05/05 00:32:00 KILGORE Exp KILGORE $
 *
 * (C) 2001 Jeff Robinson (jeffnik@anecho.mb.ca), placed under the GPL
 * The Gnu General Public License as described below is available
 * in the file COPYING distributed with this package.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <os2.h>
#include <stdio.h>
#define VERSION "2003-08-13"

#define INCL_DOS

#include <curses.h>

#include <process.h>
#include <signal.h>
#include <assert.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif
// #ifdef STDC_HEADERS
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <time.h>
#include <sys/wait.h>
#include <sys/types.h>

#include "plogg.h"

/* globals:
 */

char *musiclist="list"; /* Name of our playlist */

/* Create an array to hold all our different plays and arguments */
struct filePlayer players[10];

/* Information about each song we've got loaded */
struct tune {
  char *path;
  int time, status;
} *tunes;

struct songInfo currentSong;

WINDOW *top, *middle, *bottom;
int playerCount = 0;    /* Total number of players we have available according to our .ini file */
int currentType = 0;          /* Current file-type being played */
int oneShotPlaylist = 0;        /* A non-zero value means we have been passed files to play */
int safe_refresh = 0;   /* Since ncurses isn't thread-safe, we have to keep track of our refreshes */

static const int NONE = 0;
static const int OGG = 1;
static const int MP3 = 2;

int *playlist;
int pllen = 0, plmlen = 0;
int nrsongs = 0, nrtagged = 0;
int tunenr = 0; /* the currently selected tune */
int msize = 0;  /* size of displayed tunelist */
int startedtune = -1, toptune = 0; /* the tune at the top of middle window */

#define TAGGED 1

pid_t playerpid = 0;
int starttime;

/* Prototypes */
void not_playing_title();
void drawTitle(char*);
static void playnextselected();
void choosePlayer(int);
void view_songinfo();
void playSelectedSong();
void loadPloggPlayerList();
static void load_music_list();
void get_song_struct(char*);

/** Kill the current player and reset associated variables */
static void stop_playing() {

    (void)signal(playerpid, SIG_ACK);
    (void)kill(playerpid, 9); /* yeah! BFG! */

    // Under heavy system load, sometimes it seems like we don't
    // actually launch a new player.  Should we Fix this XXX?

    playerpid = 0;
    currentType = NONE;
}

void dostuff(int i) {
    int status;
    int ec, es;

    (void)waitpid(playerpid, &status, 0);

    ec = WEXITSTATUS(status);
    es = WIFEXITED(status);

    /* this could be used to determine the play time of
     * the tune. also, check WIFEXITED, to see if user terminated
     * the player
     */

    if (es != 0 && ec == 0) {
        time_t current = time(0);
        tunes[startedtune].time = current - starttime;
    }

    playerpid = 0;

    not_playing_title();

    (void)signal((int)dostuff, SIG_ACK);
}

static void finish(int sig)
{
  FILE *f;
  int i, fd;
  char buf[1024];

  (void)endwin();

  /* if we don't do this and kill the player while
   * working on a playlist, we get the next tune played
   * when quitting
   */
  (void)signal(playerpid, SIG_IGN);

  if(playerpid) {
      (void)kill(playerpid, SIGTERM);
  }

  /* don't bother doing the save things if user can't access
   * the list!
   */

  if (access(musiclist, W_OK)) {
      exit(0);
  }

  if (oneShotPlaylist > 0) {
      exit(0);
  }

  sprintf(buf, "%s.old", musiclist);
  (void)rename(musiclist, buf);

  printf("Saving music list...");
  fd = open(musiclist,O_RDONLY);

#ifdef HAVE_LOCKF
  lockf(fd, F_LOCK, 0);
#else
#ifdef HAVE_FLOCK
  flock(fd, LOCK_EX);
#else
  flocks.l_type = F_WRLCK;
  flocks.l_len = 0;
  (void)fcntl(fd, F_SETLKW, &flocks);
#endif
#endif

  assert(f = fopen(musiclist, "w"));

  for(i=0; i < nrsongs; i++) {
      fprintf(f, "%s#%d\n", tunes[i].path, tunes[i].time);
      if(!(i%10)) {
          printf(".");
          // fsync(0);  // This call may be potentially dangerous on older OS/2 systems (even Warp 3) using HPFS?!
      }
  }

  fclose(f);

#ifdef HAVE_LOCKF
  lockf(fd, F_ULOCK, 0);
#else
#ifdef HAVE_FLOCK
  flock(fd, LOCK_UN);
#else
  flocks.l_type = F_UNLCK;
  (void)fcntl(fd, F_SETLK, &flocks);
#endif
#endif

  (void)puts("");
  (void)close(fd);
  exit(0);
}

/** Display the elapsed time of the currently playing song */
static void tunetime(int i) {
    char buf[20];

    // Calculate the playing time, if playing
    if(tunes[toptune+i].time) {
        sprintf(buf, "%2d:%02d", tunes[toptune+i].time/60,
                tunes[toptune+i].time%60);
        mvwaddstr(middle, i, 1, buf);
    } else {
        mvwaddstr(middle, i, 1, "     ");
    }

    // Show a '+' symbol next to the playing song on the playlist
    if(tunes[toptune+i].status&TAGGED) {
        mvwaddch(middle, i, 0, '+');
    } else {
        mvwaddch(middle, i, 0, ' ');
    }
}

static void refresh_screen() {

    int i;

    if (safe_refresh < 1) {
        safe_refresh++;
        wclear(top);
        wclear(middle);
        wclear(bottom);
        safe_refresh = 0;
    }

    for(i=0; i<msize; i++)
    {
        if(i+toptune >= nrsongs) {
            break;
        }

        if(i+toptune == tunenr) {
            wattron(middle, COLOR_PAIR(3));
            mvwaddnstr(middle, i, 7, tunes[i + toptune].path, COLS - 11);
            wattroff(middle, COLOR_PAIR(3));
            tunetime(i);
        } else {
            mvwaddnstr(middle, i, 7, tunes[i + toptune].path, COLS - 11);
            tunetime(i);
        }
    }

    wattron(top, A_BOLD);

    if(!playerpid) {
        not_playing_title();
    } else {
        wattron(top, COLOR_PAIR(2));
        drawTitle(tunes[startedtune].path);

        wattroff(top, COLOR_PAIR(2));
    }

    wattroff(top, A_BOLD);

    wattron(top, COLOR_PAIR(1));
    mvwaddstr(top, 0, COLS-6, " 0:00");
    wattroff(top, COLOR_PAIR(1));
    wmove(top, 1, 0);
    for(i=0;i<COLS;i++)
        waddch(top, '-');

//#ifdef LINUX_SOUND
//    mvwaddch(top, 1, volume*(COLS-1)/128, '*');
//#endif

    for(i=0;i<COLS;i++) {
        waddch(bottom, '-');
    }

    wattron(bottom, A_BOLD);
    wattron(bottom, COLOR_PAIR(1));
    mvwaddstr(bottom, 1, 0,
              "arrow keys to navigate, t - toggle selection, ~ - select/deselect all");
    mvwaddstr(bottom, 2, 0, "p - play highlighted tune, g - play selected, r - random play, q - quit playing");
    mvwaddstr(bottom, 3, 0, "v - view playlist, i - song information, ESC - exit   (v. "VERSION")");
    wattroff(bottom, COLOR_PAIR(1));
    wattroff(bottom, A_BOLD);

    if (safe_refresh < 1) {
        safe_refresh++;
        wrefresh(top);
        wrefresh(middle);
        wrefresh(bottom);
        safe_refresh = 0;
    }
}

static void draw_list() {
    int i;

    // Check to see if we need to scroll down the list
    while (tunenr - toptune >= msize) {
        toptune++;
        i = toptune + msize - 1;
        scroll(middle);

        if ((tunes[i].status & TAGGED) > 0) {
            mvwaddch(middle, msize - 1, 0, '+');
        }
        mvwaddnstr(middle, msize - 1, 7, tunes[i].path, COLS - 11);
        if (safe_refresh < 1) {
            safe_refresh++;

            wrefresh(middle);
            safe_refresh = 0;
        }
    }


    // Check if we need to scroll up the list
    while (tunenr - toptune < 0) {
        toptune--;
        wscrl(middle, -1);

        if ((tunes[toptune].status & TAGGED) > 0) {
            mvwaddch(middle, 0, 0, '+');
        }

        mvwaddnstr(middle, 0, 7, tunes[toptune].path, COLS - 11);
        if (safe_refresh < 1) {
            safe_refresh++;
            wrefresh(middle);
            safe_refresh = 0;
        }
    }

    i = tunenr - toptune;

    // We have to pay a little more attention to the centre portion
    // as the title may be too long and cause ugly wrap.  Right now
    // we'll hard-code this for a 80-character display width, but
    // in the future we should do this in a more elegant way.
    wattron(middle, COLOR_PAIR(3));

    mvwaddnstr(middle, i, 7, tunes[tunenr].path, COLS - 11);

    wattroff(middle, COLOR_PAIR(3));

    tunetime(i);

    if (safe_refresh < 1) {
        safe_refresh++;

        wrefresh(middle);
        // g_unlock();
        safe_refresh = 0;
    }
  
}

void alarmhandler(int sig) {
    int status;
    int ec;
    int diff;
    char buf[20];

    if(!playerpid) {
        return;
    }

    /* Test to see if the player is still running */
    ec = waitpid(playerpid, &status, WNOHANG);

    if (ec == -1) {
        // The player isn't there anymore, so let's make certain
        // that we stop the timer and move on.

        // We can set the time of the song, mayhaps?
        // tunes[startedtune].time = time(0) - starttime;

        // Should the be playnextselected() instead?  Sometimes it stops dead.  Fix me XXX !
        // stop_playing();
        playnextselected();
    }

    diff = time(0) - starttime;

    /* if we know how long the tune is, show remaining time as well.
     */

    wattron(top, A_BOLD);
    wattron(top, COLOR_PAIR(2));
    if(tunes[startedtune].time>0) {
        sprintf(buf, "%2d:%02d", (tunes[startedtune].time-diff)/60, (tunes[startedtune].time-diff)%60);
        mvwaddstr(top, 0, COLS-12, buf);
    }
    sprintf(buf, "%2d:%02d", diff/60, diff%60);
    mvwaddstr(top, 0, COLS-6, buf);
    wattroff(top, COLOR_PAIR(2));
    wattroff(top, A_BOLD);
    if (safe_refresh < 1) {
        safe_refresh++;
        wrefresh(top);
        safe_refresh = 0;
    }

    signal(SIGALRM, alarmhandler);
    alarm(1);
}

int selectptr = 0;

static void playnextselected();

void selplay(int sig) {
    int status;
    int ec;

    waitpid(playerpid, &status, 0);

    ec = WEXITSTATUS(status);

    playerpid = 0;

    playnextselected();


}

static void playnextselected() {
    int i;
    int intunenr;
    int ec, status;

    if(selectptr==pllen) {
        stop_playing();

        // If this is just a one-shot (list of songs from command line/drag & drop)
        if (oneShotPlaylist > 0) {
            exit(0);
        }

        refresh_screen();
        return;
    }

    intunenr  = playlist[selectptr];
    selectptr++;

    if (!playerpid) {

        // If we don't wait and make certain that the previous player stopped,
        // then the program just cycles on through the rest of the playlist.
        // There may be a better way than just using "wait()", though...
        // this could be a potential "hang"

        waitpid(playerpid, &status, 0);

        ec = WEXITSTATUS(status);
        if (ec == 0) {
            wait(playerpid);
        }
    }

    choosePlayer(intunenr);

    startedtune = intunenr;

    starttime = time(0);

    if (safe_refresh < 1) {
        safe_refresh++;
        wclear(top);
        safe_refresh = 0;
    }

    wattron(top, A_BOLD);
    wattron(top, COLOR_PAIR(2));
    drawTitle(tunes[intunenr].path);

    wattroff(top, COLOR_PAIR(2));
    wattroff(top, A_BOLD);
    mvwaddstr(top, 0, COLS-6, " 0:00");
    wmove(top, 1, 0);
    for(i=0;i<COLS;i++) {
        waddch(top, '-');
    }
    if (safe_refresh < 1) {
        safe_refresh++;
        wrefresh(top);
        safe_refresh = 0;
    }
    signal(SIGALRM, alarmhandler);
    alarm(1);
    signal((int)selplay, SIG_ACK);
}

static void make_playlist()
{
  int i;
  
  selectptr = 0;

  if(playlist && plmlen<nrtagged)
    {
      free(playlist);
      playlist = 0;
    }

  if(!playlist)
    {
      playlist = (int *)malloc(sizeof(int)*nrtagged*2);
      plmlen = nrtagged*2;
  }
  
  for(i=0;i<nrsongs;i++)
    {
      if(tunes[i].status&TAGGED)
	playlist[selectptr++] = i;
    }
  selectptr = 0;  
  pllen = nrtagged;

}

static void add_to_playlist(int i)
{
  if(plmlen<=pllen+1)
    {
      if(plmlen==0)
	plmlen=256;

      assert(playlist = (int *)realloc(playlist,sizeof(int)*plmlen*2));
      plmlen*=2;
    }
  playlist[pllen++] = i;
}

static void make_random_playlist() {
    /* Fix this XXX */

    int i, x, y, t;
    make_playlist();

    for(i=0;i<4*pllen;i++) {
        x = rand() % pllen;
        y = rand() % pllen;
        t = playlist[x];
        playlist[x] = playlist[y];
        playlist[y] = t;
    }

}

#ifdef LINUX_SOUND

static void volume_down()
{
  int i;

  mvwaddch(top, 1, volume*(COLS-1)/128, '-');
  volume--;
  if(volume<0)
    volume=0;
  mvwaddch(top, 1, volume*(COLS-1)/128, '*');
  wrefresh(top);
  i = volume | volume<<8;
  ioctl(mixer_fd, MIXER_WRITE(SOUND_MIXER_VOLUME), &i);
}

static void volume_up()
{
  int i;

  mvwaddch(top, 1, volume*(COLS-1)/128, '-');
  volume++;
  if(volume>128)
    volume=128;
  mvwaddch(top, 1, volume*(COLS-1)/128, '*');
  wrefresh(top);
  i = volume | volume<<8;
  ioctl(mixer_fd, MIXER_WRITE(SOUND_MIXER_VOLUME), &i);
}

#endif

static void view_playlist() {
    int i, sl;
    int c;
    WINDOW *w,*l;
    /* pop up a huge window above the middle and bottom ones,
     * then draw the current playlist in it. provide functions for
     * reordering, loading and saving
     */


        w = newwin(LINES-6, 0, 2, 0);
        l = newwin(4, 0, LINES-4, 0);
        if (safe_refresh < 1) {
            safe_refresh++;
            wclear(w);
            wclear(l);
            safe_refresh = 0;
        }

        for (i=0; i<COLS; i++) {
            waddch(l, '-');
        }

        mvwaddstr(l,1,0,
                  "ESC - return to main screen");

        // Now draw the playlist
        int startlist = selectptr - (msize / 2);
        if (startlist < 0) {
            startlist = 0;
        }

        if (startlist + msize > nrsongs) {
            startlist = startlist - nrsongs;
        }

        if(playlist) {
            for (i=0; i<msize; i++) {
                if(i>=pllen) {
                    break;
                }

                sl = i + startlist;

                // if(!i) {
                if (sl + 1 == selectptr) {
                    wattron(w, A_BOLD);
                }

                if(tunes[playlist[sl]].time) {
                    mvwprintw(w, i, 0, "%2d:%02d",
                              tunes[playlist[sl]].time/60,
                              tunes[playlist[sl]].time%60);
                }

                wattron(w, COLOR_PAIR(2));
                mvwaddnstr(w, i, 6, tunes[playlist[sl]].path, COLS - 11);
                wattroff(w, COLOR_PAIR(2));

                if (sl + 1 == selectptr) {
                    wattroff(w, A_BOLD);
                }
            }
        }

        if (safe_refresh < 1) {
            safe_refresh++;

            wrefresh(w);
            wrefresh(l);
            safe_refresh = 0;
        }

    while(1) {
        c = getch();

        switch(c) {
        case 27:
            // case 'Q':
            // case 'q':
            // goto getout;
            goto leavelist;
        // case 'p':
        case 'g':             // This was: case 'P':
            {
                signal(selplay, SIG_ACK);

                /* start tune only if nothing is playing, we want the
                 * user to be able to finish listening to the current song
                 */

                selectptr = 0;

                if(!playerpid) {
                    playnextselected();
                }
            }
            break;
        case 'Q':
        case 'q':
            //case 27:
            stop_playing();

            break;
        }
    }

getout:
    // Make sure our players are toast
    // if (playerpid) {
    // If we don't send a SIG_IGN, the program tries to play one more song!!
    signal(playerpid, SIG_IGN);
    kill(playerpid, 9);
    signal(playerpid, SIGTERM);
    if (safe_refresh < 1) {
        safe_refresh++;
        delwin(w);
        delwin(l);
        refresh_screen();
        safe_refresh = 0;
    }

    exit(0);


leavelist:
    // Clear the window and refresh the screen
    delwin(w);
    delwin(l);

    refresh_screen();
}

static void action(int c) {
    int i;

    switch(c) {
    case 12:
        refresh_screen();
        break;
        // case '?':
    // case 'h':
        // system(READMECMD);
        // refresh_screen();
        // break;

#ifdef LINUX_SOUND
    case 'h':
    case KEY_LEFT:
        volume_down();
        break;

    case 'l':
    case KEY_RIGHT:
        volume_up();
        break;
#endif
    case 'a':
        add_to_playlist(tunenr);
        break;

    case 'q':
        /* stop playing what we are playing if we press escape! */
        signal(SIGCHLD, SIG_IGN);
        signal(playerpid, SIG_IGN);
        stop_playing();

        not_playing_title();

        break;

    case 'i':
        view_songinfo();
        break;

    case 'v': /* view the current playlist */
        {
            view_playlist();
        }
        break;

    case '~':
        {
            if(tunes[0].status & TAGGED)
            {
                for(i=0;i<nrsongs;i++)
                    tunes[i].status&=~TAGGED;
                nrtagged=0;
            }
            else
            {
                for(i=0;i<nrsongs;i++)
                    tunes[i].status|=TAGGED;
                nrtagged=nrsongs;
            }
            refresh_screen();
            break;
        }
    case 27:
        // case 'q':
        // case 'Q':
        signal(SIGCHLD, SIG_IGN);
        signal(playerpid, SIG_IGN);
        finish(0);
        // exit(0);
        break;
    case 'j':
    case KEY_DOWN:
        i = tunenr - toptune;

        mvwaddnstr(middle, i, 7, tunes[i + toptune].path, COLS - 11);
        tunetime(i);
        tunenr++;

        if (tunenr >= nrsongs - 1) {
            tunenr = nrsongs -1;
        }

        draw_list();
        if (safe_refresh < 1) {
            safe_refresh++;
            wrefresh(middle);
            safe_refresh = 0;
        }
        break;
    case 'k':
    case KEY_UP:
        {
            i = tunenr - toptune;
                mvwaddnstr(middle, i, 7, tunes[i + toptune].path, COLS - 11);
                tunetime(i);
                tunenr--;
                if(tunenr<0) {
                    tunenr=0;
                }
                draw_list();
                if (safe_refresh < 1) {
                    safe_refresh++;
                    wrefresh(middle);
                    // g_unlock();
                    safe_refresh = 0;
                }
            break;
        }

    case KEY_NPAGE:
        {
            i = tunenr - toptune;
                mvwaddnstr(middle, i, 7, tunes[i + toptune].path, COLS - 11);
                tunetime(i);
                tunenr = tunenr + msize;
                if (tunenr>=nrsongs - 2) {
                    tunenr=nrsongs - 2;
                }
                draw_list();
                if (safe_refresh < 1) {
                    safe_refresh++;
                    wrefresh(middle);
                    safe_refresh = 0;
                }
            break;
        }

    case KEY_PPAGE:
        {
            // Move up the list one "screen-length"
            i = tunenr - toptune;
                mvwaddnstr(middle, i, 7, tunes[i + toptune].path, COLS - 11);
                tunetime(i);
                tunenr = tunenr - msize;
                if(tunenr<0) {
                    tunenr=0;
                }
                draw_list();
                if (safe_refresh < 1) {
                    safe_refresh++;
                    wrefresh(middle);
                    safe_refresh = 0;
                }
            break;
        }

    case KEY_END:
        {
            tunenr = nrsongs - 1;
            draw_list();
            wrefresh(middle);
            break;
        }

    case KEY_HOME:
        {
            tunenr = 0;
            draw_list();
            if (safe_refresh < 1) {
                safe_refresh++;
                // g_lock();
                wrefresh(middle);
                // g_unlock();
                safe_refresh = 0;
            }
            break;
        }

        // case 'n':
    case KEY_LEFT:
        {
            // A nasty hack to play the previous song
            selectptr = selectptr - 2;
            if (selectptr < 0) {
                selectptr = 0;
            }

            // Brute force.  Ugly, but effective!
            kill(playerpid, 9);
            break;
        }

    case KEY_RIGHT:
        {
            // skip to the next song
            // This is sort've a brute-force method, but seems to do the trick!
            kill(playerpid, 9);
            break;
        }

    case 'p':
        /* we must spawn an mpg123-player, if no player is running,
         * but also zap the running player, if one is running
         */

        playSelectedSong();

        break;
    case 't': /* tag the current tune */
        {
            int i = (tunes[tunenr].status & TAGGED)>0;

            i = i^1;
                wstandout(middle);
                if(i)
                {
                    tunes[tunenr].status|=TAGGED;
                    mvwaddch(middle, tunenr-toptune, 0, '+');
                    nrtagged++;
                }
                else
                {
                    tunes[tunenr].status&=~TAGGED;
                    mvwaddch(middle, tunenr-toptune, 0, ' ');
                    nrtagged--;
                }
                wstandend(middle);
                if (safe_refresh < 1) {
                    safe_refresh++;
                    wrefresh(middle);
                    safe_refresh = 0;
                }
            }
        break;
    case 'g': /* play selected music */
        {

            signal(SIGCHLD, selplay);

            /* start tune only if nothing is playing, we want the
             * user to be able to finish listening to the current song
             */

            selectptr = 0;
            make_playlist();


            if(!playerpid) {
                playnextselected();
            }

        }
        break;
    case 'r': /* randomize over tunes */

        srand( (unsigned)time( NULL ) );
        {

            signal(SIGCHLD, selplay);

            /* start tune only if nothing is playing, we want the
             * user to be able to finish listening to the current song
             */

            selectptr = 0;
            make_random_playlist();

            if(!playerpid) {
                playnextselected();
            }
        }
        break;
    }
}

static void make_ui() {
    /* create a three-window ui:
     * top line shows currently playing tune, and elapsed time.
     * if tune was played before, show remaining time as well.
     *
     * then comes a tune selection window.
     *
     * at the bottom, we want a menu window.
     * (looks sort of like the elm interface.)
     */
    top = newwin(2, 0, 0, 0);
    middle = newwin(LINES-6, 0, 2, 0);
    bottom = newwin(4, 0, LINES-4, 0);

    msize = LINES-6;

    scrollok(middle, TRUE);
}

int main(int argc, char **argv) {
    int c, i;
    char a;

    while((a=getopt(argc, argv, "hv"))>-1) {
        switch(a)
        {
	case 'h': fprintf(stderr, "USAGE: %s [-h] [-v]\n", argv[0]);
        exit(0);
        case 'v':
            {
                fprintf(stderr, "'Plogg' "VERSION" (C) 2003 Jeff Robinson (jeffnik@anecho.mb.ca>\n");
                fprintf(stderr, "Based on source code from Jukebox by K. Wiklund.\n");
                fprintf(stderr, "'Jukebox' (C) K. Wiklund 1997 <kw@dtek.chalmers.se>\n");
                exit(0);
            }
        case '?':
            exit(1);
        }
    }

    initscr();
    // Test to see if our terminal supports colours or not
    if(has_colors() == FALSE) {
        endwin();
        printf("You terminal does not support colour\n");
        exit(1);
    }

    freopen("stderr.log", "w", stderr);

    // "Initialise" colours and define our colour pairs
    start_color();
    init_color(COLOR_YELLOW, 1000, 800, 400);
    init_pair(1, COLOR_YELLOW, COLOR_BLACK);
    init_pair(2, COLOR_GREEN, COLOR_BLACK);
    init_pair(3, COLOR_BLACK, COLOR_WHITE);


    keypad(stdscr, TRUE);
    cbreak();
    noecho();

    make_ui();

    loadPloggPlayerList();      // load our list of players

    if (argc <= 1) {
        // We weren't passed any song locations, so we'll use the playlist
        load_music_list();
    } else {
        /* We have been passed a list of songs to play (this time only) */
        oneShotPlaylist = 1;
        nrsongs = argc - 1;     // Number of songs passed
        tunes = (struct tune *)malloc(sizeof(struct tune)*nrsongs);
        bzero((void *)tunes, sizeof(struct tune)*nrsongs);

        for (i = 0; i < nrsongs; i++) {
            fprintf(stderr, argv[i + 1]);
            fprintf(stderr, "**");
            tunes[i].time = 0;
            tunes[i].path = (char *)argv[i + 1];

            // Tag the songs as we go
            tunes[i].status|=TAGGED;
        }

        nrtagged=nrsongs;       // total number of tagged songs
        playSelectedSong();

    }

    /* we must set the signals AFTER the configuration is read, or else
     * we'll cause a segv when the wc -l returns
     */

    signal(finish, SIG_ACK);
    signal(dostuff, SIG_ACK);

#ifdef LINUX_SOUND
    if ((mixer_fd = open("/dev/mixer", O_RDWR)) < 0) {
        fprintf(stderr, "Error opening /dev/mixer.\n");
        exit(1);
    }

    ioctl(mixer_fd, MIXER_READ(SOUND_MIXER_VOLUME), &volume);
    volume = volume & 0xff;
#endif
    if (safe_refresh < 1) {
        safe_refresh++;

        refresh();
        refresh_screen();
        safe_refresh = 0;
    }

    for(;;) {
        c = getch();
        action(c);
    }
}

/** Choose the proper player for the selected music type */
void choosePlayer(int choice) {
    int rc, status;         // return status... for use by a few things
    int workingPlayer = -1;
    int l = strlen(tunes[choice].path);
    char testChoice[l+ 1];
    strcpy(testChoice, tunes[choice].path);

    (void)get_song_struct(tunes[choice].path);

    // We have the check and see if there is a player running already
    if (playerpid != 0) {
        // Check the status of our player
        rc = waitpid(playerpid, &status, WNOHANG);

        if (rc >= 0) {
            // kill the existing player
            kill(playerpid, 9); /* yeah! BFG! */
        }
    }

    // Parse apart any arguments
    int count;
    char workString[128];

    /* New version for multiple player support */
    for (count = 0; count < playerCount; count++) {
        if (strstr(strupr(testChoice), players[count].extension) > 0) {
            strcpy(workString, players[count].playerArgs);
            workingPlayer = count;
            break;
        }
    }

    if (workingPlayer < 0) {
        fprintf(stderr, "There is no player setup to handle %s\n", testChoice);
        return;
    }


    // Count number of tokens
    char temp_list[25][256];
    char *countTok;
    char splat[25] = {0};
    int totalTokens = 0;
    countTok = strtok (workString," -");

    while (countTok != NULL) {
        totalTokens++;
        strcpy(splat, "-");
        strcat(splat, countTok);
        strcpy(temp_list[totalTokens], splat);
        countTok = strtok(NULL, " -");
    }

    char *arg_list[totalTokens + 1];

    arg_list[0] = players[workingPlayer].playerPath;


    int j;

    for (j = 1; j <= totalTokens; j++) {
        arg_list[j] = temp_list[j];
    }

    arg_list[totalTokens + 1] = tunes[choice].path;
    arg_list[totalTokens + 2] = NULL;

    playerpid = spawnv(P_NOWAIT, arg_list[0], arg_list);

}

/** Determine how many lines are in a given file, and then put the titles in fileName */
void countLines(char *fileName) {
    FILE *fid;
    int c;
    int done = 0;
    int lastchar = 0;

    nrsongs = 0;

    // Open the file in read-only mode
    fid = fopen(fileName, "r");

    if (!fid) {
        fprintf(stderr, "I'm sorry, I can't find the playlist (called 'list').\n");
        fprintf(stderr, "Please check to make certain you have created one.");
        exit(0);
    }

    while(done == 0) {
        c = fgetc(fid);

        if(c == '\n') {
            nrsongs++;
        }

        if(c == EOF) {
            if(lastchar != '\n') {
                nrsongs++;
            }
            done = 1;
        }
        lastchar = c;
    }
    // Make certain to close the file
    fclose(fid);

    // This should return the total number of lines in our file
    return;
}

static void load_music_list() {
    FILE *in;
    char buf[1024],b2[1024];
    int cnt = 0;

    /* this is a simple format. just load a list of paths and be happy with
     * it (for now, improvements comes probably later. one idea is to
     * support some kind of detection for which cd things are on, and
     * make the program check a few possible paths etc)
     */

    /* count number of tunes first!
     */

    countLines(musiclist);

    in = fopen(musiclist, "r");

    if (!in) {
        fprintf(stderr, "No music list found.");
        exit(0);
    }

    tunes = (struct tune *)malloc(sizeof(struct tune)*nrsongs);
    bzero((void *)tunes, sizeof(struct tune)*nrsongs);

    while(!feof(in)) {
        if(!fgets(buf, 1024, in)) {
            break;
        }
        buf[strlen(buf)-1] = 0; /* lose the trailing \n */
        sscanf(buf,"%[^#]#%d", b2, &(tunes[cnt].time));
        tunes[cnt++].path = (char *)strdup(b2);
    }

    fclose(in);
}

/** Launch a player to deal with the currently selected song */
void playSelectedSong() {
    int i;

    /* If there is already a player going, we'll stop it first */
    if(playerpid) {
        signal(SIGCHLD, SIG_IGN);
        signal(playerpid, SIG_IGN);
        stop_playing();
    }

    choosePlayer(tunenr);

    startedtune = tunenr;

    starttime = time(0);
    wclear(top);
    wattron(top, COLOR_PAIR(2));
    wattron(top, A_BOLD);

    drawTitle(tunes[startedtune].path);  // Fix me XXX

    wattroff(top, A_BOLD);
    mvwaddstr(top, 0, COLS-6, " 0:00");

    wattroff(top, COLOR_PAIR(2));
    wmove(top, 1, 0);
    for(i=0; i<COLS; i++) {
        waddch(top, '-');
    }

    if (safe_refresh < 1) {
        safe_refresh++;
        wrefresh(top);
        safe_refresh = 0;
    }
    signal(SIGALRM, alarmhandler);
    alarm(1);

}

void drawTitle(char *songPath) {

    int titleLength;

    if (currentSong.songTitle != NULL) {
        titleLength = strlen(currentSong.songTitle);
    } else {
        char workingTitle[] = "NULL";
        waddnstr(top, workingTitle, COLS - 11);
        return;
    }

    if (strlen(currentSong.songArtist) > 0) {
        titleLength += strlen(currentSong.songArtist) + 3;
    } else {
        // fprintf(stderr, "We have 0-length for our song artist.\n");
    }

    safe_refresh++;

    if (titleLength < 1) {
        char workingTitle[strlen(songPath)];
        strcpy(workingTitle, songPath);
        mvwaddnstr(top, 0, 0, workingTitle, COLS - 11);
        safe_refresh = 0;
        return;
    }

    char workingTitle[titleLength];

    strcpy(workingTitle, currentSong.songTitle);
    if (strlen(currentSong.songArtist) > 0) {
        strcat(workingTitle, " - ");
        strcat(workingTitle, currentSong.songArtist);
    }
    mvwaddnstr(top, 0, 0, workingTitle, COLS - 11);
    safe_refresh = 0;
}

/** Create a dialogue box that will display information about our song */
void view_songinfo() {
    int maxLength = COLS - 10; // The longest any line can be in our dialogue
    int x;
    int calcLength = strlen(currentSong.songTitle);
    int ch;

    if (strlen(currentSong.songArtist) > calcLength) {
        calcLength = strlen(currentSong.songArtist);
    }
    if (strlen(currentSong.songAlbum) > calcLength) {
        calcLength = strlen(currentSong.songAlbum);
    }

    if (calcLength < 1) {
        fprintf(stderr, "No title available for this song... or anything else for that matter.\n");
        return;
    }

    calcLength = calcLength + 8;  // This compensates for the "label" for each row

    if (calcLength > maxLength) {
        calcLength = maxLength;
    }

    x = (int)((COLS/2) - (calcLength/2));


    WINDOW *popup_window= subwin(middle, 7, calcLength + 2, 10, x);
    if (safe_refresh < 1) {
        safe_refresh++;

        wclear(popup_window);
        safe_refresh = 0;
    }

    wattron(popup_window, A_BOLD);
    wattron(popup_window, COLOR_PAIR(2));
    mvwprintw(popup_window, 1, 1, "Title: %s", currentSong.songTitle);
    mvwprintw(popup_window, 2, 1, "Artist: %s", currentSong.songArtist);
    mvwprintw(popup_window, 3, 1, "Album: %s", currentSong.songAlbum);
    mvwprintw(popup_window, 4, 1, "Year: %s", currentSong.songYear);
    mvwprintw(popup_window, 5, 1, "ESC to exit");
    box(popup_window, 0, 0);
    overwrite(popup_window, middle);
    wattroff(popup_window, A_BOLD);
    wattroff(popup_window, COLOR_PAIR(2));
    if (safe_refresh < 1) {
        safe_refresh++;
        wrefresh(popup_window);
        safe_refresh = 0;
    }
    wgetch(popup_window);

    /* switch on the reply */
    while ((ch = getch()) != 27) {
        /* Do nothing, just wait for the ESC to exit. */
    }

    if (safe_refresh < 1) {
        safe_refresh++;
        delwin(popup_window);
        wclear(middle);

        /* re-generate the main_window contents */
        draw_list();

        wrefresh(middle);
        refresh_screen();
        safe_refresh = 0;
    }

    return;
}

/** Reset the song title display to <nothing> */
void not_playing_title() {
    int i;
    safe_refresh++;
    (void)wclear(top);

    wattron(top, A_BOLD);
    wattron(top,COLOR_PAIR(1));
    (void)waddstr(top, "<nothing is playing>");
    mvwaddstr(top, 0, COLS-6, " 0:00");
    wattroff(top,COLOR_PAIR(1));
    wattroff(top, A_BOLD);

    (void)wmove(top, 1, 0);
    for(i=0;i<COLS;i++) {
        (void)waddch(top, '-');
    }
    (void)wrefresh(top);
    safe_refresh = 0;
}

/** Get the name of the song from our passed file
 * and then populate our currentSong struct with it
 */
void get_song_struct(char *songPath) {
    // Reset our currentSong variables
    strcpy(currentSong.songTitle, "");
    strcpy(currentSong.songArtist, "");
    strcpy(currentSong.songAlbum, "");
    strcpy(currentSong.songYear, "");

    char testName[CCHMAXPATH];
    strcpy(testName, songPath);

    if (strstr(strupr(testName), ".OGG") > 0) {
        // Get Ogg-based information
        oggData(songPath, &currentSong);
    } else {
        // Get mp3-based information
        mp3Data(songPath, &currentSong);
    }
}

/* Load the list of players from our plogg.ini file */
void loadPloggPlayerList() {
    fprintf(stderr, "Loading player list...\n");
    FILE *in;
    char buf[1024], workString[1024];
    char *tExt, *tPlayer, *tFlags, *splitExt;

    int i, j, spaces;

    in = fopen("plogg.ini", "r");

    if (!in) {
        fprintf(stderr, "The plogg.ini file was not found.");
        return;      // This is a non-fatal error as we'll just use defaults
    }

    while (!feof(in)) {
        if (!fgets(buf, 1024, in)) {
            break;
        }

        /* Strip white space off the front of our line */
        i = strlen(buf);
        spaces = 0;
        for (j = 0; j < i; j++) {
            if (buf[j] != ' ') {
                break;
            } else {
                spaces++;
            }
        }

        /* Copy our buf into a workString that has been "trimmed" */
        for (j = 0; j < (i - spaces); j++) {
            workString[j] = buf[j + spaces];
        }

        workString[i - spaces] = (char)NULL;

        if (workString[0] != '#') {
            /* Parse apart the extension, player and flags */

            tExt = strtok(workString, ",");
            tPlayer = strtok(NULL, ",");
            tFlags = strtok(NULL, ",");

            /* Try and remove any trailing new-lines and such */
            if (tFlags != NULL) {
                for (j = 0; j < strlen(tFlags); j++) {
                    if (tFlags[j] == '\n' || tFlags[j] == '\r') {
                        tFlags[j] = NULL;
                        break;
                    }
                }
            }

            /* Insert our new information into our struct */
            if (tExt != NULL && tPlayer != NULL) {
                if (tFlags == NULL) {
                    tFlags = " \0";
                }

                /* Check for multiple file types and split things up here */
                splitExt = strtok(tExt, " ");
                while (splitExt != NULL) {
                    strcpy(players[playerCount].extension, strupr(splitExt));
                    strcpy(players[playerCount].playerPath, tPlayer);
                    strcpy(players[playerCount].playerArgs, tFlags);
                    playerCount++;
                    splitExt = strtok(NULL, " ");
                }
            }

        }

    }
}
