/*************************************************
*    The PMW Music Typesetter - 3rd incarnation  *
*************************************************/

/* Copyright (c) Philip Hazel, 1991 - 2002 */

/* Written by Philip Hazel, starting November 1991 */
/* This file last modified: August 2002 */

/* Store Management Routines */

#include "pmwhdr.h"

/* Store is taken from the system in blocks whose minimum
size is parameterized here. */

#define store_allocation_unit    32*1024

/* Musical data is packed into blocks whose size is
parameterized here. */

#define store_item_unit           8*1000

/* Free queue entries contain their length in the first word
and a pointer to the next entry in the second word. Allocated
blocks contain the length in the first word: the user is given
a pointer to the second word. */

#define store_block_length       0
#define store_block_next         1



/*************************************************
*                  Static data                   *
*************************************************/

static int  store_freequeue[2];
static int *store_itembuff;
static int  store_itemspace;


#ifdef trace
static FILE *tracefile = NULL;
#endif



/*************************************************
*             Trace logging                      *
*************************************************/

#ifdef trace
static void do_trace(uschar *s, int a, int b)
{
if (tracefile == NULL) tracefile = fopen("trace", "w");
fprintf(tracefile, "%s %8d = %8x %8d = %8x\n", s, a, a, b, b);
}
#endif




/*************************************************
*         Free queue sanity check                *
*************************************************/

#ifdef sanity
/* Free queue sanity check */

static void store_freequeuecheck(void)
{
int *p = (int *)(store_freequeue[store_block_next]);

while (p != NULL)
  {
  p = (int *)(p[store_block_next]);
  }
}
#endif





/*************************************************
*                 Initialise                     *
*************************************************/

/* This procedure is called when a new file is to be processed. 
All store will have been reset previously. */

void store_init(void)
{
main_storetotal = main_storestaves = 0;

/* Initialise the free queue */

store_freequeue[store_block_next] = (int)NULL;
store_freequeue[store_block_length] = 2*sizeof(int);

/* Set up the first item buffer */

store_itembuff = store_Xget(store_item_unit);
store_itemspace = store_item_unit;
}



/*************************************************
*               Get block                        *
*************************************************/

void *store_get(int bytesize)
{
int *newblock;
int  newlength;
int *previous = store_freequeue;
int *p = (int *)(previous[store_block_next]);

/* Add space for one int to hold the length, then round up to
a multiple of 2 ints so that freed blocks can hold a pointer
and a length. */

bytesize = (bytesize + 3*sizeof(int) - 1) & (-2*(int)sizeof(int));

#ifdef sanity
store_freequeuecheck();
#endif

/* Keep statistics */

main_storetotal += bytesize;

/* Search free queue for a block that is big enough */

while(p != NULL)
  {
  if (bytesize <= p[store_block_length])
    {    /* found suitable block */
    int leftover = p[store_block_length] - bytesize;
    if (leftover == 0)
      {  /* block used completely */
      previous[store_block_next] = p[store_block_next];
      }
    else
      {  /* use bottom of block */
      int *remains = p + bytesize/sizeof(int);
      remains[store_block_length] = leftover;
      previous[store_block_next] = (int)remains;
      remains[store_block_next] = p[store_block_next];
      p[store_block_length] = bytesize;
      }
    #ifdef sanity
    store_freequeuecheck();
    #endif
    #ifdef trace
    do_trace("got", bytesize, (int)p);
    #endif
    return (void *)(p + 1);    /* leave length "hidden" */
    }
  else
    {    /* try next block */
    previous = p;
    p = (int *)(p[store_block_next]);
    }
  }

/* No block long enough has been found. Get a new block. */

main_storetotal -= bytesize;  /* correction */
newlength = (bytesize > store_allocation_unit)? bytesize : store_allocation_unit;

newblock = malloc(newlength);
if (newblock == NULL) return NULL;

/* xs succeeded */

#ifdef trace
do_trace(">>>> XS", (int)newblock, 0);
#endif

newblock[store_block_length] = newlength;  /* Set block length */
main_storetotal += newlength;              /* Correction */
store_free(newblock + 1);                  /* Add to free queue */
return store_get(bytesize-sizeof(int));    /* Try again */
}



/*************************************************
*     Get store, failing if none available       *
*************************************************/

void *store_Xget(int bytesize)
{
void *yield = store_get(bytesize);
if (yield == NULL) error_moan(1, bytesize);
return yield;
}



/*************************************************
*      Copy store, failing if no store           *
*************************************************/

void *store_copy(void *p)
{
int length = ((int *)(p)-1)[store_block_length] - sizeof(int);
void *yield = store_Xget(length);
memcpy(yield, p, length);
return yield;
}


/*************************************************
*        Copy string, failing if no store        *
*************************************************/

uschar *store_copystring(uschar *s)
{
uschar *yield = store_Xget(Ustrlen(s)+1);
Ustrcpy(yield, s);
return yield;
}



/*************************************************
*         Yield address of next staff item       *
*************************************************/

/* This is used to initialize the bar index */

void *store_nextitem(void)
{
return store_itembuff;
}



/*************************************************
*               Get a staff item                 *
*************************************************/

/* Staff items come from the staff item block,
packed in together. When the block is used up, we
put in an adjustment item to the next one. */

void *store_getitem(int type)
{
void *yield;
int size = length_table[type];
if (store_itemspace < size + sizeof(b_Jumpstr))
  {
  b_Jumpstr *j = (b_Jumpstr *)store_itembuff;
  int *newbuff = store_Xget(store_item_unit);

  j->type = b_Jump;
  j->adjust = (int)newbuff - sizeof(b_Jumpstr);

  store_itembuff = newbuff;
  store_itemspace = store_item_unit;
  main_storestaves += sizeof(b_Jumpstr);
  }

yield = (void *)store_itembuff;
((bstr *)yield)->type = type;

store_itembuff = (int *)((uschar *)store_itembuff + size);
store_itemspace -= size;
main_storestaves += size;
return yield;
}



/*************************************************
*           Free a chunk of store                *
*************************************************/

/* The length is in the first word of the block, which is
before the address that the client was given. */

void store_free(void *address)
{
int *previous = store_freequeue;
int *this = (int *)(previous[store_block_next]);
int *start = ((int *)address) - 1;
int *end = start + start[store_block_length]/sizeof(int);

main_storetotal -= start[store_block_length];

#ifdef sanity
store_freequeuecheck();
#endif

#ifdef trace
do_trace("free", (int)address, 0);
#endif

/* find where to insert */

while (this != NULL)
  {
  if (start < this) break;
  previous = this;
  this = (int *)(previous[store_block_next]);
  }

/* insert */

previous[store_block_next] = (int)start;
start[store_block_next] = (int)this;

/* check for overlap with next */

if (end > this && this != NULL)
  error_moan(2, start, start[store_block_length], this);

/* check for contiguity with next */

if (end == this)
  {
  start[store_block_next] = this[store_block_next];
  start[store_block_length] += this[store_block_length];
  }

/* check for overlap/contiguity with previous */

if (previous != store_freequeue)
  {
  int *prevend = previous + previous[store_block_length]/sizeof(int);
  if (prevend > start)
    error_moan(3, previous, previous[store_block_length], start);

  if (prevend == start)
    {
    previous[store_block_next] = start[store_block_next];
    previous[store_block_length] += start[store_block_length];
    }
  }

#ifdef sanity
store_freequeuecheck();
#endif
}


/*************************************************
*       Reduce the length of a chunk of store    *
*************************************************/

void store_chop(void *address, int bytesize)
{
int freelength;
int *start = ((int *)address) - 1;
int *end;

/* Round up new length as for new blocks */

bytesize = (bytesize + 3*sizeof(int) - 1) & (-2*(int)sizeof(int));

/* Compute amount to free, and don't bother if less than 32 bytes */

freelength = start[store_block_length] - bytesize;
if (freelength < 32) return;

/* Set revised length into what remains, create a length for the
bit to be freed, and free it via the normal function. */

start[store_block_length] = bytesize;
end = start + bytesize/sizeof(int);
end[store_block_length] = freelength;
store_free(end + 1);
}

/* End of store.c */
