/*************************************************
*    The PMW Music Typesetter - 3rd incarnation  *
*************************************************/

/* Copyright (c) Philip Hazel, 1991 - 2003 */

/* Written by Philip Hazel, starting November 1991 */
/* This file last modified: October 2003 */


/* This file contains the code for generating an image of
one individual note. */


#include "pmwhdr.h"
#include "pagehdr.h"
#include "poshdr.h"
#include "outhdr.h"


/* Tables used only in this module */

static uschar common[] = {
  49, 50, 52, 54, 56, 58,       /* stems down */
  49, 50, 51, 53, 55, 57};      /* stems up */

static uschar *reststrings[] = {
  US"*", US"+", US",", US"-", US".", US"z.w{{y.",
  US"zzx.w{{y.w{{y.", US"zzzx.w{{y.w{{y.w{{y." };

static uschar *tailstrings[] = {
  US"",  US"",       US"",            US"", 
  US"H", US"<xv<v|", US"<xv<xv<v|v|", US"<xv<xv<xv<v|v|v|",
  US"",  US"",       US"",            US"", 
  US"E", US";v|;xv", US";v|;v|;xvxv", US";v|;v|;v|;xvxvxv" };

static uschar headchars[] = {
  '1', '2', 'M', 'L', 'L', 'L', 'L', 'L',    /* normal */
  'n', 'n', 'n', 'n', 'n', 'n', 'n', 'n',    /* cross */
  'm', 'm', 'm', 'l', 'l', 'l', 'l', 'l',    /* harmonic */
   0,   0,   0,   0,   0,   0,   0,   0,     /* none */ 
  '1', '2', 'M', 'L', 'L', 'L', 'L', 'L',    /* only = normal */
  178, 178, 178, 178, 178, 178, 178, 178     /* direct */ 
};   

/* These next tables are accent-specific adjustments, in the order
> v tp down up */

static uschar *accabovestrings[] = { US"U", US"Y", US"W", US"\234", US"e", US"g" };
static uschar *accbelowstrings[] = { US"U", US"Z", US"X", US"\234", US"f", US"h" };

static int accaboveadjusts[] = { -6000, -1000, -2000, -1000, -1000, -1000 };
static int accbelowadjusts[] = { -2000,  3000,     0,  2000,  1000,  1000 };

/* Further per-accidental adjustments for accents */

                                 /* -   ##     $      $$     %      # */
static int accaccaboveadjusts[] = { 0, 0000, -3000, -3000, -3000, -3000 };
static int accaccbelowadjusts[] = { 0, 0000,  0000,  0000,  3000,  3000 };

/* Dot position adjustments for rests */

static int restdotadjusts[] = { -20, 0, 0, -25, -20, -10, -10, 0 };

/* Table of dynamic numbers in order of printing outside dynamics */

static uschar dyn_list[] = { dyn_gt, dyn_wedge, dyn_tp, dyn_vline,
  dyn_down, dyn_up };


/* These tables are for the straightforward ornaments. Those with blank
strings are not straightforward, and have individual code. */

static char *ornament_strings[] = {
/* ferm  tr  trsh trfl trnat trem1 trem2 trem3 */
    "",  "",  "",  "",  "",   "",   "",   "",
/* mord  dmord  imord dimord turn iturn arp arpu arpd spread */
   "O",   "P",   "Q",  "R",   "S", "i", "",   "", "",   "",
/* dsharp   dsharprb      dsharpsb */
   "&",  "~v\215v&~v\216", "~v\213v&~v\214",
/* flat      flatrb        flatsb */
   "\'", "~\215|\'~\216", "~\213|\'~\214",
/* dflat     dflatrb               dflatsb */
   "\'\'", "~\215|\'\'~\216", "~\213|\'\'~\214",
/* nat      natrb       natsb */
   "(", "\215(\216", "\213(\214",
/* sharp sharprb sharpsb */
   "%", "\215%\216", "\213%\214" };

static int ornament_xadjusts[] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0,
  1, -2, -2,  /* dsharps */
  2, -1, -1,  /* flats */
  0, -3, -3,  /* dflats */
  2, -1, -1,  /* naturals */
  1, -1, -1}; /* sharps */

static int ornament_yaadjusts[] = {
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, -2000, -2000, -1000, -1000, 0, 0, 0, 0,
  1000, 2000, 2000,  /* dsharp */
  1000, 1000, 1000,  /* flat */
  1000, 1000, 1000,  /* dflat */
  3000, 3000, 3000,  /* natural */
  3000, 3000, 3000}; /* sharp */


static int ornament_ybadjusts[] = {
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 2000, 2000, 1000, 1000, 0, 0, 0, 0,
  0,     -3000, -3000,  /* dsharp */
  -3000, -4000, -4000,  /* flat */
  -3000, -4000, -4000,  /* dflat */
  -2000, -2000, -2000,  /* natural */
  -2000, -2000, -2000}; /* sharp */





/*************************************************
*         Actually print one note                *
*************************************************/

static BOOL show_note(int x)
{
uschar buff[100];
uschar *p = buff;
BOOL ledgered = FALSE;
BOOL inverted = (n_flags & nf_invert) != 0;
int top = P_6L;
int bot = P_0L;
int fontsize = (n_fontsize*main_stavemagn)/1000;
int y = out_ystave - (n_pitch - 130)*main_stavemagn - n_pcorrection;
int yy;

DEBUG(("show_note() start\n"));

/* Set up for coupled notes */

if ((n_flags & nf_coupleU) != 0)
  {
  top += out_upgap;
  bot += out_upgap;
  }
else if ((n_flags & nf_coupleD) != 0)
  {
  top -= out_downgap;
  bot -= out_downgap;
  }

/* No anti-aliasing for the main parts of a note */

dev_setalias(FALSE);

/* First deal with ledger lines if required. We repeat the code
for above & below, as there seems no tidy way of combining it
owing to the requirement for <= or >= comparisons. We can optimize
into a single music-font string if the size is standard. The existence
of breves makes this messy! */

if (n_pitch <= bot && out_stavelines >= 5 && bar_cont->noteheadstyle != nh_none)
  {
  int breve_right = 0;
  int xx = x;

  if (n_notetype == breve)
    {
    xx -= ((((curmovt->breveledgerextra - 2300)*main_stavemagn)/1000) *
      n_fontsize)/10000;
    breve_right = mac_muldiv(2*curmovt->breveledgerextra, fontsize, 10000);
    }

  ledgered = TRUE;
  yy = out_ystave - (bot - 130)*main_stavemagn - n_pcorrection;
  if (n_fontsize == 10000 && !inverted)
    {
    uschar *p = buff;
    while (n_pitch <= bot) { *p++ = '='; *p++ = 'w'; bot -= 4; }
    *(--p) = 0;  /* removes redundant last move */
    dev_musstring(buff, fontsize, xx, yy);
    if (n_notetype == breve)
      dev_musstring(buff, fontsize, xx + breve_right, yy);
    }
  else  /* Have to position each line separately */
    {
    int yyy = yy;
    strcpy(CS buff, inverted? (n_upflag? "=}=" : "=yy{=") : "=");
    while (yy <= y)
      {
      dev_musstring((yy == y && !n_upflag && inverted)? buff+1 : buff,
        fontsize, xx, yy);
      yy += 4*main_stavemagn;
      }
    if (n_notetype == breve)
      {
      yy = yyy;
      xx += breve_right;
      while (yy <= y)
        {
        dev_musstring((yy == y && !n_upflag && inverted)? buff+1 : buff,
          fontsize, xx, yy);
        yy += 4*main_stavemagn;
        }
      }
    }
  }

else if (n_pitch >= top && out_stavelines >= 5 && bar_cont->noteheadstyle != nh_none)
  {
  int breve_right = 0;
  int xx = x;

  if (n_notetype == breve)
    {
    xx -= ((((curmovt->breveledgerextra - 2300)*main_stavemagn)/1000) *
      n_fontsize)/10000;
    breve_right = mac_muldiv(2*curmovt->breveledgerextra, fontsize, 10000);
    }

  ledgered = TRUE;
  yy = out_ystave - (top - 130)*main_stavemagn - n_pcorrection;
  if (n_fontsize == 10000 && !inverted)
    {
    uschar *p = buff;
    while (n_pitch >= top) { *p++ = '='; *p++ = 'x'; top += 4; }
    *(--p) = 0;  /* removes redundant last move */
    dev_musstring(buff, fontsize, xx, yy);
    if (n_notetype == breve)
      dev_musstring(buff, fontsize, xx + breve_right, yy);
    }
  else
    {
    int yyy = yy;
    strcpy(CS buff, inverted? (n_upflag? "=}=" : "=yy{=") : "=");

    while (yy >= y)
      {
      dev_musstring((yy == y && n_upflag && inverted)? buff+1 : buff,
        fontsize, xx, yy);
      yy -= 4*main_stavemagn;
      }

    if (n_notetype == breve)
      {
      yy = yyy;
      xx += breve_right;
      while (yy >= y)
        {
        dev_musstring((yy == y && n_upflag && inverted)? buff+1 : buff,
          fontsize, xx, yy);
        yy -= 4*main_stavemagn;
        }
      }
    }
  }


/* Optimize the common case where there is a complete character
available in the music font. */

if (n_notetype < dsquaver && n_stemlength == 0 &&
  bar_cont->noteheadstyle == nh_normal &&
    (n_flags & (nf_invert|nf_stem)) == nf_stem)
      {
      if ((n_flags & nf_appogg) != 0) *p++ = n_upflag? 129 : 130;
      *p++ = common[n_notetype + n_upflag*6];
      *p = 0;
      dev_musstring(buff, fontsize, x, y);
      dev_setalias(TRUE); 
      return ledgered;
      }


/* Deal with rarer cases, first dealing with stems & tails. We impose
a minimum stemlength at this point. */

if (n_stemlength < -8000) n_stemlength = -8000;

if ((n_flags & nf_stem) != 0)
  {
  int direction = n_upflag? -1 : +1;
  int font10 = fontsize;   /* 10pt at font scale */
  int font2  = font10/5;   /* 2pt at font scale */
  int font1  = font2/2;    /* 1pt at font scale */

  if ((n_flags & nf_appogg) != 0) *p++ = n_upflag? 129 : 130;

  yy = y + (direction*n_stemlength*main_stavemagn)/1000;
  p += sprintf(CS p, "%s", tailstrings[n_notetype + n_upflag*8]);

  /* Notes with stems up */

  if (n_upflag)
    {
    if (yy <= y)    /* stem is lengthened */
      {
      int stemch = (bar_cont->noteheadstyle == nh_cross)? 'o' : 'J';
      int z = yy;
      while (z <= y)
        {
        p += sprintf(CS p, "%cww|", stemch);
        z += font10;
        }
      p -= 3;
      *p = 0;
      dev_musstring(buff, fontsize, x, yy);
      p = buff;
      if (z < y + font10) *p++ = stemch;
      if (bar_cont->noteheadstyle == nh_harmonic) *p++ = 'q';
      }

    else            /* stem is shortened */
      {
      int z = yy - font10 - font2;
      p += sprintf(CS p, "xxx");
      while (z <= y)
        {
        p += sprintf(CS p, "q|");
        z += font2;
        }
      *(--p) = 0;
      dev_musstring(buff, fontsize, x, yy);
      p = buff;
      if (z > y) *p++ = 'q';
      }
    }


  /* Notes with stems down */

  else
    {
    if (yy >= y)    /* stem is lengthened */
      {
      int stemch = (bar_cont->noteheadstyle == nh_cross)? 'p' : 'K';
      int z = yy;
      while (z >= y)
        {
        p += sprintf(CS p, "%cxx~", stemch);
        z -= font10;
        }
      p -= 3;
      *p = 0;
      dev_musstring(buff, fontsize, x, yy);
      p = buff;
      if (z > y - font10) *p++ = stemch;
      if (bar_cont->noteheadstyle == nh_harmonic) *p++ = 'r';
      }

    else            /* stem is shortened */
      {
      int z = yy + font10 + font2;
      p += sprintf(CS p, "www");
      while (z >= y)
        {
        p += sprintf(CS p, "r~v");
        z -= font1;
        }
      *(--p) = 0;
      dev_musstring(buff, fontsize, x, yy);
      p = buff;
      if (z < y) *p++ = 'r';
      }
    }
  }


/* Now add the note head and print the whole thing */

if (bar_cont->noteheadstyle != nh_none)
  {
  if (inverted)
    {
    if (n_upflag)
      {
      if (n_notetype == breve)
        p += sprintf(CS p, "}}}}{{{{z"); 
      else *p++ = 125;
      }
    else
      {
      if (n_notetype == breve)
        p += sprintf(CS p, "{yyyyyyyyyyyy}"); 
      else
        {       
        *p++ = 123;
        *p++ = 121;
        *p++ = 121;
        } 
      }
    }
  *p++ = headchars[n_notetype + 8*(bar_cont->noteheadstyle)];
  }

*p = 0;
dev_musstring(buff, fontsize, x, y);

/* Reset anti-aliasing, and return whether ledger lines or not */

dev_setalias(TRUE);
DEBUG(("show_note() end\n"));
return ledgered;
}




/*************************************************
*         Actually print one rest                *
*************************************************/

static void show_rest(int x, int notetype)
{
int fontsize = (n_fontsize*main_stavemagn)/1000;
int yoffset = n_restlevel;

/* Rests longer than a crotchet have to have ledger lines when they
are printed off the stave. Also move a semibreve rest down on a
1-line stave and up on a 3-line stave. We must also adjust the position 
of breve and semibreve rests for cue sizes. Most rests (and the ledger)
are printed non-anti-aliased. */

dev_setalias(FALSE);

if (notetype <= minim)
  {
  int loffset = 0;
  yoffset += 8000;

  switch (notetype)
    {
    case -1:     /* long rest */
    yoffset -= 2000;
    break;

    case breve:
    yoffset += n_pcorrection;
    /* Fall through */

    case minim:
    if (yoffset > 16000 || yoffset < 0) loffset = -2000;
    break;

    case semibreve:
    if (out_stavelines == 1) yoffset -= 4000;
      else if (out_stavelines == 3) yoffset += 4000;
    yoffset += 2*n_pcorrection;
    if (yoffset > 12000 || yoffset < -4000) loffset = 2000;
    break;
    }

  if (loffset)
    dev_musstring(US"=", fontsize, x - (10*main_stavemagn)/10, out_ystave -
      ((yoffset + loffset)*main_stavemagn)/1000);
  }

else yoffset += 4000 + n_pcorrection;

if (notetype >= 0)
  {
  /* We'd like to anti-alias rests shorter than a minim, but for those shorter
  than a quaver, the joining up gets messed up. The best we can do at present 
  is to overprint them again, non aliased. */  
   
  if (notetype >= crotchet) dev_setalias(TRUE);       
   
  dev_musstring(reststrings[n_notetype], fontsize,
    x, out_ystave - (yoffset*main_stavemagn)/1000);
    
  if (main_state == state_displaying && notetype > quaver) 
    {
    dev_setalias(FALSE);       
    dev_musstring(reststrings[n_notetype], fontsize,
      x, out_ystave - (yoffset*main_stavemagn)/1000);
    }
  }   

else dev_muschar(x, out_ystave - (yoffset*main_stavemagn)/1000,
  mc_longrest, fontsize);
  
dev_setalias(TRUE); 
}




/*************************************************
*    Generate one note/rest + dots & accents     *
*************************************************/

/* The relevant data is all in global variables */

void out_shownote(void)
{
int fontsize = (n_fontsize*main_stavemagn)/1000;
int p_staccato = 0;
int x = n_x + n_cueadjust;
int yyabove, yybelow;
int acc_level, acc_upflag;
BOOL ledgered = FALSE;

DEBUG(("out_shownote() start\n"));

/* If the note is invisible, skip printing; just show accents, etc. */

if ((bar_cont->flags & cf_notes) != 0 && (n_flags & nf_hidden) == 0)
  {
  /* If printing a breve, move left to align with semibreve position */

  if (n_pitch != 0 && n_notetype == breve) x -= (23*main_stavemagn)/10;

  /* First, any accidental is set, prior to the given position */

  if (n_acc)
    dev_muschar(x - mac_muldiv(n_accleft, n_fontsize, 10000),
      out_ystave - (n_pitch - 130)*main_stavemagn - n_pcorrection,
        out_acctable[n_acc + ((n_flags & nf_accrbra)? 6 :
          (n_flags & nf_accsbra)? 12 : 0)],
            fontsize);

  /* Now we can output the note or rest. */

  if (n_pitch == 0)
    {
    int notetype;
    if (out_manyrest == 1) notetype = n_notetype; else
      {
      notetype = -1;
      n_flags &= ~(nf_dot+nf_plus);   /* Kill dots for many bar rest */
      }
    show_rest(x, notetype);
    }
  else ledgered = show_note(x);


  /* Deal with horizontal dots/plus - fudge for quavers and breves */

  if ((n_flags & (nf_dot+nf_plus)) != 0)
    {
    int dotpos = 84;
    int dotlevel;

    if (n_pitch == 0)
      {
      dotlevel = mac_muldiv(L_3S + n_restlevel, main_stavemagn, 1000);
      /* if (n_notetype == breve || n_notetype == crotchet) dotpos -= 20; */
      dotpos += restdotadjusts[n_notetype];
      }

    else
      {
      int dotpitch = n_pitch | 2;  /* force into space above */

      if ((n_flags & nf_lowdot) != 0 && (n_pitch & 2) == 0) dotpitch -= 4;
      if ((n_flags & nf_highdot) != 0 && (n_pitch & 2) != 0) dotpitch += 4;

      dotlevel = (dotpitch - 130)*main_stavemagn;
      if (n_notetype == breve) dotpos += 50;
      }

    if ((n_flags & nf_dotright) != 0) dotpos += 60;
      else if (n_upflag && n_notetype >= quaver && n_pitch != 0) dotpos += 16;

    dotpos = (dotpos*main_stavemagn)/10 + n_dotxadjust;

    /* For cue notes there are two choices: either to scale the position
    according to the cue size, or to align the dots with full-sized notes
    that may be above or below (alignment by centre of dot). */  
    
    if ((n_flags & nf_cuesize) != 0)
      { 
      if ((n_flags & nf_cuealign) != 0)
        dotpos += mac_muldiv(640 - mac_muldiv(640, n_fontsize, 10000),
          main_stavemagn, 1000) - n_cueadjust;
      else
        dotpos = mac_muldiv(dotpos, n_fontsize, 10000);
      } 

    /* Output the dot(s) */
    
    dev_musstring(((n_flags & nf_plus) == 0)? US"?" : US"\207", fontsize,
      x + dotpos, out_ystave - dotlevel - n_pcorrection);

    if ((n_flags & nf_dot2) != 0)
      dev_musstring(US"?", fontsize, x + dotpos + (35*main_stavemagn)/10,
        out_ystave - dotlevel - n_pcorrection);
    }
  }


/* If there are no dynamics and no ornaments, there's nothing more to do */

if ((n_acflags & af_dynamics) == 0 && n_ornament == NULL) return;



/* Now set up a level and up flag for expression marks - normally these are
from the standard note values, but they are different if the accents are
flagged for printing on the same side of the note as the stem. For chords it is
arranged that the accents come with the appropriate note head. */

acc_level = (n_pitch - 130)*main_stavemagn;

if ((n_acflags & af_opposite) == 0)
  {
  acc_upflag = n_upflag;
  }
else
  {
  acc_upflag = !n_upflag;
  if ((n_flags & nf_stem) != 0)
    acc_level +=
      n_upfactor * mac_muldiv((12000+n_stemlength), main_stavemagn, 1000);
  }



/* Staccato, ring, & bar go inside the staff. They are allowed together -
the staccato is nearest to the note. We don't need to compensate for ties
as the ties themselves are moved in this case. */

if ((n_acflags & af_dyninside) != 0)
  {
  int adjust = 4*main_stavemagn;
  int p = acc_level;

  if (acc_upflag)
    {
    adjust = -adjust;
    p -= 8*main_stavemagn;

    /* Accent at notehead; ensure not on line; not for 0 or 1-line staves */

    if (out_stavelines >= 2)
      { 
      if (acc_upflag == n_upflag)
        {
        if (((n_pitch & 2) == 0) && n_pitch != P_1L &&
          (!ledgered || n_pitch >
            (P_5L - ((n_flags & nf_coupleD)? out_downgap : 0))))
              p -= 2*main_stavemagn;
        }
      
      /* Accent at stem end; ensure not on line; rather assumes stemlength
      adjustments will be in whole points... */
      
      else
        {
        int pp = p/main_stavemagn;
        if (pp >= (-6) && pp <= 10 && (pp & 2) != 0)
          p -= 2*main_stavemagn;
        }
      } 
    }
  else  /* !acc_upflag */ 
    {
    /* Accent at notehead; ensure not on line; not for 0 or 1-line staves */
    if (out_stavelines >= 2)
      {  
      if (acc_upflag == n_upflag)
        {
        if (((n_pitch & 2) == 0) && n_pitch != P_5L &&
          (!ledgered || n_pitch <
            (P_1L + ((n_flags & nf_coupleU)? out_upgap : 0))))
              p += 2*main_stavemagn;
        }
      
      /* Accent at stem end; ensure not on line (rather assumes stemlength
      adjustments will be in whole points... */
      
      else
        {
        int pp = p/main_stavemagn;
        if (pp >= (-6) && pp <= 10 && (pp & 2) != 0)
          p += 2*main_stavemagn;
        }
      } 
    }

  if (out_beaming && (acc_upflag != n_upflag)) p += n_upfactor * 1000;

  if ((n_acflags & af_staccato) != 0)
    {
    dev_musstring(US">", fontsize, x + out_dynmovex[dyn_staccato],
      out_ystave - p - out_dynmovey[dyn_staccato]);
    p += adjust;
    p_staccato++;
    out_dynmovex[dyn_staccato] = out_dynmovey[dyn_staccato] = 0;
    }

  /* The ring character prints 4 points lower than the other two, and a
  bit further away from the notehead when clear of stave or ledger lines. */

  if ((n_acflags & af_ring) != 0)
    {
    int yy = 0;
    if ((n_flags & nf_couple) == 0)
      {
      if (acc_upflag)
        {
        if (n_upflag && n_pitch <= P_1S) yy = main_stavemagn;
        }
      else
        {
        if (!n_upflag && n_pitch >= P_4S) yy = -main_stavemagn;
        }
      }

    dev_musstring(US"\206", fontsize, x + out_dynmovex[dyn_ring],
      out_ystave - p - 4*main_stavemagn + yy - out_dynmovey[dyn_ring]);
    p += adjust + yy;
    p_staccato++;
    out_dynmovex[dyn_ring] = out_dynmovey[dyn_ring] = 0;
    }

  if ((n_acflags & af_bar) != 0)
    {
    dev_musstring(US"T", fontsize, x + out_dynmovex[dyn_bar],
      out_ystave - p - out_dynmovey[dyn_bar]);
    p_staccato++;
    out_dynmovex[dyn_bar] = out_dynmovey[dyn_bar] = 0;
    }
  }


/* Set up y values for things that print above or below the stave;
there is a different set for the accents and the ornaments, but we
compute the basic values here for both of them. Set the stemlength
to the value for the first note of a chord, which will be the bit
that sticks out. */

n_stemlength = n_orig_stemlength;
yybelow = misc_ybound(TRUE, n_nexttie, TRUE, FALSE);
yyabove = misc_ybound(FALSE, n_nexttie, TRUE, FALSE);


/* We can common up the code for the accents and bowing marks into
a loop. This is dependent on the order of the flags in the word,
and the bowing marks must come last, because the above/below
control is different. All these items print above or below the
stave. */

if ((n_acflags & af_dynoutside) != 0)
  {
  int i;
  int yextra = 0;
  int upflag = acc_upflag;
  int f = n_acflags << 3;        /* first bit to top position */

  /* The order is: > v V ' down up. We assume that only one of the first
  four will exist, possibly with one of the last two. */

  for (i = 0; i < 6; i++)
    {
    int xadjust = 0;
    int ayybelow = yybelow;
    int ayyabove = yyabove;

    if (i == 0) xadjust = main_stavemagn; else if (i == 4)
      {
      int newupflag = (bar_cont->flags & cf_bowingabove) == 0;
      if (newupflag != upflag) yextra = 0;
      upflag = newupflag;
      }

    /* Further adjustments when accents are not on the stem side and there
    are no "inside" accidentals. Effectively we cancel some of the space
    put in for accidental signs. We can't take this out of the loop, because
    the up/down bows may be different to other accents. */

    if (upflag == n_upflag && (n_acflags & af_dyninside) == 0)
      {
      ayybelow += accaccbelowadjusts[n_acc];
      ayyabove += accaccaboveadjusts[n_acc];
      }

    ayybelow = ((ayybelow <= out_stavebottom)? ayybelow :
      out_stavebottom) - 8000;
    ayyabove = ((ayyabove > out_stavetop)? ayyabove :
      out_stavetop) + 3000;

    /* Print the accent if its bit is set. */

    if (f < 0)
      {
      int d = dyn_list[i]; /* standard accent number */
      int y;
      uschar *s;

      if (upflag)
        {
        s = accbelowstrings[i];
        y = ayybelow + accbelowadjusts[i] - yextra;
        }
      else
        {
        s = accabovestrings[i];
        y = ayyabove + accaboveadjusts[i] + yextra;
        }

      dev_musstring(s, fontsize, n_x + xadjust + out_dynmovex[d],
        out_ystave - mac_muldiv(y, main_stavemagn, 1000) - out_dynmovey[d]);

      if (i < 4) yextra = 4000;
      out_dynmovex[d] = out_dynmovey[d] = 0;
      }

    f = f << 1;
    }
  }



/* Deal with ornaments. There are only *very* rarely more than one,
so don't bother about the recomputation that then happens. */

if (n_ornament != NULL)
  {
  yybelow = ((yybelow <= out_stavebottom)? yybelow : out_stavebottom) - 8000;
  yyabove = ((yyabove > out_stavetop)? yyabove : out_stavetop) + 3000;

  while (n_ornament != NULL)
    {
    uschar s[100];
    uschar *p = s;
    BOOL below = (n_acflags & af_opposite) != 0;
    int size = fontsize;
    int ornament = n_ornament->ornament;
    int x = n_ornament->x + n_x;
    int y = n_ornament->y;

    /* Above/below accidentals are special */

    if (ornament >= or_dsharp)
      {
      size = ((curmovt->fontsizes)->fontsize_vertacc * main_stavemagn)/1000;
      if (ornament >= or_accbelow)
        {
        below = TRUE;
        y += yybelow + 8000 - (8*size)/10;
        ornament -= or_accbelow - or_dsharp;
        }
      else 
        {
        below = FALSE;
        y += yyabove;
        }
      }
     
    /* Adjust y for other ornaments */
       
    else y += below? yybelow : yyabove;

    /* Particular code for each ornament */

    switch (ornament)
      {
      case or_trsh:
      *p++ = '%';
      goto TR;

      case or_trfl:
      *p++ = '\'';
      goto TR;

      case or_trnat:
      *p++ = '(';

      case or_tr:
      TR:
        {
        int tsize = (curmovt->fontsizes)->fontsize_trill;
        int asize = (6*tsize)/10;
        int *matrix = (curmovt->fontsizes)->fontmatrix_trill;
        if (matrix != NULL) memcpy(font_transform, matrix, 4*sizeof(int));
        out_string(curmovt->trillstring, font_it, tsize, x,
          out_ystave - mac_muldiv(y, main_stavemagn, 1000), 0);
        font_reset();
        size = asize;
        if (below) y -= asize;
          else if (n_ornament->ornament == or_trfl) y += (8*tsize)/10;
            else y += tsize;
        x += 2*main_stavemagn;
        }
      break;


      case or_ferm:
      if (n_pitch == 0) x -= main_stavemagn;
      *p++ = below? '/' : ')';
      break;

      case or_arp:
      case or_arpu:
      case or_arpd:
        {
        int h = (n_maxpitch - n_minpitch + 2)/4;
        if ((n_minpitch & 2) != 0 && (n_maxpitch & 2) != 0) h++;

        if (ornament == or_arpd)
          {
          *p++ = 165;
          h--;
          }
        do *p++ = 145; while (--h >= 0);
        if (ornament == or_arpu) p[-1] = 164;

        y = n_minpitch - 130;
        if ((y & 2) == 0) y -= 2;
          else if ((n_maxpitch & 2) != 0) y--;
        y = y*1000 + n_ornament->y;

        if (n_maxaccleft) x -= n_maxaccleft;
          else if (n_invertleft) x -= (55*main_stavemagn)/10;
        }
      break;

      case or_spread:
        {
        int co; 
        int y0 = n_minpitch - 128;
        int y1 = n_maxpitch - 128;

        if (0 <= y0 && y0 <= 16 && (y0 & 3) == 0) y0--;
        if (0 <= y1 && y1 <= 16 && (y1 & 3) == 0) y1++;
        
        co = (4000 * (y1 - y0))/14;
        if (co > 4000) co = 4000; 

        y0 = mac_muldiv(y0 * 1000 + n_ornament->y, main_stavemagn, 1000);
        y1 = mac_muldiv(y1 * 1000 + n_ornament->y, main_stavemagn, 1000);

        x -= 5000;
        if (n_maxaccleft) x -= n_maxaccleft;
          else if (n_invertleft) x -= 4*main_stavemagn;

        out_slur(x, y0, x, y1, 0, co, 0, 1000);
        }
      break;


      /* Tremolos are handled with their own code */

      case or_trem3:
      *p++ = 146;

      case or_trem2:
      *p++ = 146;

      case or_trem1:
      *p++ = 146;

      y = n_pitch - 124;
      if (n_notetype >= minim)
        {
        x += (n_upfactor*255*main_stavemagn)/100;
        if (!out_psoutput && !n_upflag) x -= (45*main_stavemagn)/100;
        }

      if (n_upflag)
        {
        int yy = (n_ornament->ornament == or_trem3)? 4000 : 2000; 
        y = (y + ((n_ornament->ornament == or_trem1)? 4 : 2))* 1000;
        if (out_beaming && n_stemlength >= yy) y += (n_stemlength - yy)/2;
        }
      else
        {
        int yy = 2000; 
        switch (n_ornament->ornament)
          {
          case or_trem3: y -= 18; yy = 4000; break;
          case or_trem2: y -= 14; break;
          case or_trem1: y -= (y & 2)? 10 : 12; break;
          }
        y = y*1000;
        if (out_beaming && n_stemlength >= yy) y -= (n_stemlength - yy)/2;
        }
      break;

      /* Handle those cases that have no complications, but just
      require setting a string and a position. This includes
      accidentals printed above and below. */

      default:
      p += sprintf(CS p, "%s", ornament_strings[ornament]);
      x += ornament_xadjusts[ornament] * main_stavemagn;
      y += below? ornament_ybadjusts[ornament] :
        ornament_yaadjusts[ornament];
      break;
      }

    /* Output the string which has been set up (if any) */

    if (p > s)
      {
      *p = 0;
      dev_musstring(s, size, x, out_ystave -
        mac_muldiv(y, main_stavemagn, 1000));
      }

    /* Move on to next ornament, if any */

    while ((++n_ornament)->type == b_Jump)
      n_ornament = (b_ornamentstr *)
        ((b_Jumpstr *)(((b_Jumpstr *)n_ornament)->adjust) + 1);

    if (n_ornament->type != b_ornament) break;
    }
  }

DEBUG(("out_shownote() end\n"));
}

/* End of shownote.c */
