/*************************************************
*    The PMW Music Typesetter - 3rd incarnation  *
*************************************************/

/* Copyright (c) Philip Hazel, 1991 - 2004 */

/* Written by Philip Hazel, starting November 1991 */
/* This file last modified: February 2004 */


/* This file contains code for supporting the "draw" directives
in PMW. It contains both the code for reading in the definition
of a draw funtion, and the code for executing it. */


#include "pmwhdr.h"
#include "readhdr.h"
#include "outhdr.h"


#define magic 0x74657874


/* Structure for describing a text item */

typedef struct {
  int   ident;
  int   rotate;
  uschar *text;
  short int flags;
  short int size;
} drawtextstr;




/* Types of entry in draw items; note that the table of
required data on the stack (stack_rqd) in must be kept in
step with this. */

enum {
  dr_accleft, dr_add, dr_and,
  dr_barnumber, dr_bra,
  dr_copy, dr_currentgray, dr_currentlinewidth, dr_currentpoint, dr_curveto, dr_cvs,
  dr_def, dr_div, dr_draw, dr_dup,
  dr_end, dr_eq, dr_exch, dr_exit,
  dr_false, dr_fill, dr_fillretain, dr_fontsize,
  dr_gaptype, dr_ge, dr_gt,
  dr_headbottom, dr_headleft, dr_headright, dr_headtop,
  dr_if, dr_ifelse,
  dr_jump,
  dr_ket,
  dr_le, dr_leftbarx, dr_linebottom, dr_linegapx, dr_linegapy,
  dr_linelength, dr_lineto, dr_linetop, dr_loop, dr_lt,
  dr_magnification, dr_moveto, dr_mul,
  dr_ne, dr_neg, dr_not, dr_number,
  dr_or, dr_origin, dr_originx, dr_originy,
  dr_pagelength, dr_pagenumber, dr_pop, dr_pstack,
  dr_rcurveto, dr_repeat, dr_rlineto, dr_rmoveto, dr_roll,
  dr_setgray, dr_setlinewidth, dr_show, dr_stavesize,
  dr_stavespace, dr_stavestart, dr_stembottom, dr_stemtop,
  dr_stringwidth, dr_stroke, dr_sub, dr_systemdepth,
  dr_text, dr_topleft, dr_translate, dr_true,
  dr_varname, dr_varref,
  dr_xor
};

/* Number of entries required to be on the stack for each operator */

static uschar stack_rqd[] = {
  0, 2, 2,           /* accleft, add, and */
  0, 0,              /* barnumber, bra */
  1, 0, 0, 0, 6, 2,  /* copy, currentgray, currentlinewidth, currentpoint, curveto, cvs */
  2, 2, 0, 1,        /* def, div, draw, dup */
  0, 2, 2, 0,        /* end, eq, exch, exit */
  0, 0, 0, 1,        /* false, fill, fillretain, fontsize */
  0, 2, 2,           /* gaptype, ge, gt */
  0, 0, 0, 0,        /* headbottom, headleft, headright, headtop */
  2, 3,              /* if, ifelse */
  0,                 /* jump */
  0,                 /* ket */
  2, 0, 0, 0, 0,     /* le, leftbarx, linebottom, linegapx, linegapy */
  0, 2, 0, 1, 2,     /* linelength, lineto, linetop, loop, lt */
  0, 2, 2,           /* magnification, moveto, mul */
  2, 1, 1, 0,        /* ne, neg, not, number */
  2, 0, 0, 0,        /* or, origin, originx, originy */
  0, 0, 1, 0,        /* pagelength, pagenumber, pop, pstack */
  6, 2, 2, 2, 2,     /* rcurveto, repeat, rlineto, rmoveto, roll */
  1, 1, 1, 0,        /* setgray, setlinewidth, show, stavesize */
  0, 0, 0, 0,        /* stavespace, stavestart, stembottom, stemtop */
  1, 0, 2, 0,        /* stringwidth, stroke, sub, systemdepth */
  0, 0, 2, 0,        /* text, topleft, translate, true */
  0, 0,              /* varname, varref */
  2                  /* xor */
};




/*************************************************
*                Variables                       *
*************************************************/

static BOOL currentpoint;
static int gray;
static int level;
static int xp, yp, cp;
static int draw_stack[draw_stacksize];
static int draw_variables[max_draw_variable + 1];



/*************************************************
*          Table of operators/variables etc      *
*************************************************/

typedef struct {
  char *name;
  int  value;
} draw_op;

/* This table is used for compiling operators, and also for finding a name to 
print in an error message. The names that don't correspond to operator names
are enclosed in <> so they don't affect the first usage. */

static draw_op draw_operators[] = {
  { "<number>",     dr_number },
  { "<text>",       dr_text },
  { "<name>",       dr_varname },
  { "<reference>",  dr_varref },
  { "accleft",      dr_accleft },
  { "add",          dr_add },
  { "and",          dr_and }, 
  { "barnumber",    dr_barnumber },
  { "copy",         dr_copy }, 
  { "currentgray",  dr_currentgray },
  { "currentlinewidth", dr_currentlinewidth },
  { "currentpoint", dr_currentpoint },
  { "curveto",      dr_curveto },
  { "cvs",          dr_cvs },
  { "def",          dr_def },
  { "div",          dr_div },
  { "dup",          dr_dup },
  { "eq",           dr_eq }, 
  { "exch",         dr_exch },
  { "exit",         dr_exit }, 
  { "false",        dr_false }, 
  { "fill",         dr_fill },
  { "fillretain",   dr_fillretain },
  { "fontsize",     dr_fontsize },
  { "gaptype",      dr_gaptype },
  { "ge",           dr_ge },
  { "gt",           dr_gt },  
  { "headbottom",   dr_headbottom },
  { "headleft",     dr_headleft },
  { "headright",    dr_headright },
  { "headtop",      dr_headtop },
  { "if",           dr_if },
  { "ifelse",       dr_ifelse},
  { "le",           dr_le }, 
  { "leftbarx",     dr_leftbarx },
  { "linebottom",   dr_linebottom },
  { "linegapx",     dr_linegapx },
  { "linegapy",     dr_linegapy },
  { "linelength",   dr_linelength },
  { "lineto",       dr_lineto },
  { "linetop",      dr_linetop },
  { "loop",         dr_loop }, 
  { "lt",           dr_lt }, 
  { "magnification",dr_magnification},
  { "moveto",       dr_moveto },
  { "mul",          dr_mul },
  { "ne",           dr_ne }, 
  { "neg",          dr_neg },
  { "not",          dr_not },
  { "or",           dr_or },  
  { "origin",       dr_origin },
  { "originx",      dr_originx },
  { "originy",      dr_originy },
  { "pagelength",   dr_pagelength },
  { "pagenumber",   dr_pagenumber },
  { "pop",          dr_pop },
  { "pstack",       dr_pstack },
  { "rcurveto",     dr_rcurveto },
  { "repeat",       dr_repeat }, 
  { "rlineto",      dr_rlineto },
  { "rmoveto",      dr_rmoveto },
  { "roll",         dr_roll },
  { "setgray",      dr_setgray },
  { "setlinewidth", dr_setlinewidth },
  { "show",         dr_show },
  { "stavesize",    dr_stavesize },
  { "stavespace",   dr_stavespace },
  { "stavestart",   dr_stavestart },
  { "stembottom",   dr_stembottom },
  { "stemtop",      dr_stemtop },
  { "stringwidth",  dr_stringwidth },
  { "stroke",       dr_stroke },
  { "sub",          dr_sub },
  { "systemdepth",  dr_systemdepth },
  { "topleft",      dr_topleft },
  { "translate",    dr_translate },
  { "true",         dr_true },
  { "xor",          dr_xor }  
};

static int draw_operator_count = sizeof(draw_operators)/sizeof(draw_op);



/*************************************************
*        Read a draw text string                 *
*************************************************/

/* This is also called when reading arguments for draw calls. It reads
the string and parameters into a store block, and yields its address,
cast to an int. */

int read_draw_text(void)
{
drawtextstr *textptr = store_Xget(sizeof(drawtextstr));
textptr->ident = magic;  /* magic number to identify */
textptr->text = string_check(string_read());

textptr->flags = 0;
textptr->size = 0;
textptr->rotate = 0;

while (read_ch == '/')
  {
  int size;
  next_ch();
  switch (read_ch)
    {
    case 'b':
    if (Ustrncmp(read_chptr, "ox", 2) == 0)
      {
      next_ch();
      next_ch();
      next_ch();
      textptr->flags |= text_box;
      }
    else error_moan(10, "/box, /c, /e, /ring, /rot, or /s");
    break;

    case 'c':
    textptr->flags &= ~text_endalign;
    textptr->flags |= text_centre;
    next_ch();
    break;

    case 'e':
    textptr->flags &= ~text_centre;
    textptr->flags |= text_endalign;
    next_ch();
    break;

    case 'n':
    next_ch();
    if (read_ch == 'c') textptr->flags &= ~text_centre;
    else if (read_ch == 'e') textptr->flags &= ~text_endalign;
    else error_moan(37, "/nc or /ne");
    next_ch();
    break;

    case 's':
    next_ch();
    read_expect_integer(&size, FALSE, FALSE);
    if (--size < 0 || size >= MaxTextFont)
      { error_moan(39, MaxTextFont); size = 0; }
    textptr->size = size;
    break;

    case 'r':
    if (Ustrncmp(read_chptr, "ot", 2) == 0)
      {
      int rotate;
      next_ch();
      next_ch();
      next_ch();
      read_expect_integer(&rotate, TRUE, TRUE);
      textptr->rotate = rotate;
      break;
      }
    else if (Ustrncmp(read_chptr, "ing", 3) == 0)
      {
      next_ch();
      next_ch();
      next_ch();
      next_ch();
      textptr->flags |= text_ring;
      break;
      }
    /* Else fall through */

    default:
    error_moan(10, "/box, /c, /e, /ring, /rot, or /s");
    break;
    }
  }

return (int)textptr;
}


/*************************************************
*                  Read a Draw function          *
*************************************************/

void read_draw(void)
{
tree_node *node = store_Xget(sizeof(tree_node));
int *ptr, left;
int bracount = 0;
uschar word[80];

read_word(word);
if (word[0] == 0) { error_moan(10, "name"); return; }

node->name = store_copystring(word);
node->data = store_Xget(draw_blocksize);

ptr = (int *)node->data;
left = draw_blocksize/sizeof(int);


/* Loop to read the contents of the draw function */

for (;;)
  {
  int type = -1;
  int value;

  sigch();

  /* Deal with numerical values */

  if (isdigit(read_ch) || read_ch == '-' || read_ch == '+')
    {
    type = dr_number;
    read_expect_integer(&value, TRUE, TRUE);
    }

  /* Deal with strings */

  else if (read_ch == '\"')
    {
    type = dr_text;
    value = read_draw_text();
    }

  /* Deal with brackets */

  else if (read_ch == '{') { next_ch(); bracount++; type = dr_bra; }
  else if (read_ch == '}')
    {
    next_ch();
    type = dr_ket;
    if (--bracount < 0) error_moan(118, node->name);
    }

  /* Deal with variable names */

  else if (read_ch == '/')
    {
    tree_node *node;
    type = dr_varname;
    next_ch();
    read_word(word);
    if (word[0] == 0) error_moan(37, "Variable name"); else
      {
      node = Tree_Search(draw_variable_tree, word);
      if (node == NULL)
        {
        if (draw_nextvariable > max_draw_variable)
          {
          error_moan(100, max_draw_variable + 1);
          draw_nextvariable--;
          }
        value = draw_nextvariable++;
        node = store_Xget(sizeof(tree_node));
        node->name = store_copystring(word);
        node->data = (uschar *)value;
        Tree_InsertNode(&draw_variable_tree, node);
        }
      else value = (int)(node->data);
      }
    }

  /* Else it must be a word */

  else if (isalpha(read_ch))
    {
    read_word(word);
    if (Ustrcmp(word, "enddraw") == 0) { *ptr = dr_end; break; }

    /* Deal with "subroutine" call */

    if (Ustrcmp(word, "draw") == 0)
      {
      tree_node *node;
      type = dr_draw;
      read_word(word);
      node = Tree_Search(draw_tree, word);
      if (node == NULL) error_moan(70, word); else value = (int)node;
      }

    /* Deal with normal operators and variables */

    else
      {
      draw_op *first = draw_operators;
      draw_op *last = first + draw_operator_count;

      while (last > first)
        {
        int c;
        draw_op *middle = first + (last - first)/2;
        c = Ustrcmp(middle->name, word);
        if (c == 0) { type = middle->value; break; }
        if (c > 0) last = middle; else first = middle + 1;
        }
      }

    /* If haven't matched a standard variable or operator, try
    for a user variable */

    if (type < 0)
      {
      tree_node *node = Tree_Search(draw_variable_tree, word);
      if (node != NULL)
        {
        type = dr_varref;
        value = (int)(node->data);
        }
      }

    /* Grumble if unrecognized word */

    if (type < 0) error_moan(69, word);
    }

  /* Grumble if unrecognized input */

  else { error_moan(10, "number, string, name, or curly bracket"); break; }

  /* Extend to new block if necessary */

  if (left < 4)
    {
    *ptr++ = dr_jump;
    *ptr = (int)store_Xget(draw_blocksize);
    ptr = (int *)(*ptr);
    left = draw_blocksize/sizeof(int);
    }

  /* Add this item to the "program" */

  *ptr++ = type;
  left--;
  if (type == dr_number || type == dr_text || type == dr_draw ||
      type == dr_varname || type == dr_varref)
    {
    *ptr++ = value;
    left--;
    }
  }

/* Insert into tree; give error if duplicate */

if (!Tree_InsertNode(&draw_tree, node)) error_moan(14, node->name);
}




/*************************************************
*     Generate an error while drawing            *
*************************************************/

static void draw_error(int n, char *s, tree_node *t)
{
uschar buff[80];
if (out_stave < 0)
  Ustrcpy(buff, "in a heading or footing");
else
  format_sprintf(buff, "in bar %b of stave %d%M", out_bar, out_stave);
error_moan(n, s, t->name, buff);
}



/*************************************************
*           Scale for a grace/cue note           *
*************************************************/

static int cuegrace_scale(int value)
{
/* Not grace note */
if (n_length != 0)
  {
  return ((n_flags & nf_cuesize) == 0)? value :
    mac_muldiv(value, (curmovt->fontsizes)->fontsize_cue, 10000);
  }

/* Grace note */
else
  {
  int size = ((n_flags & nf_cuesize) != 0)?
    (curmovt->fontsizes)->fontsize_cuegrace : (curmovt->fontsizes)->fontsize_grace;
  return mac_muldiv(value, size, 10000);
  }
}



/*************************************************
*   Set up an overdraw saved block for graphic   *
*************************************************/

static void setup_overdraw(int thickness, int *x, int *y, int *c)
{
int *pp;
overdrawstr *last = out_overdraw;
overdrawstr *new = store_Xget(sizeof(overdrawstr) + sizeof(int)*(3*cp));
if (last == NULL) out_overdraw = new; else
  {
  while (last->next != NULL) last = last->next;
  last->next = new;
  }
new->next = NULL;
new->texttype = FALSE;
new->d.g.gray = gray;
new->d.g.linewidth = thickness;
new->d.g.ystave = out_ystave;
new->d.g.count = cp;
pp = &(new->d.g.data[0]);
memcpy(pp, x, cp*sizeof(int));
memcpy(pp+cp, y, cp*sizeof(int));
memcpy(pp+cp+cp, c, cp*sizeof(int));
}



/*************************************************
*       Check validity of program pointer        *
*************************************************/

static BOOL check_ptr(int *p, tree_node *t)
{
int *pp = (int *)t->data;
while (pp != p && *pp != dr_end)
  {
  int type = *pp++;
  if (type == dr_jump) pp = (int *)(*pp);
  if (type == dr_draw || type == dr_varref || type == dr_number ||
    type == dr_text || type == dr_varname) pp++;
  }
return p == pp;
}



/*************************************************
*   Interpret a draw function - recursive call   *
*************************************************/

static BOOL sub_draw(tree_node *t, int *p, int *x, int *y, int *c, BOOL overflag)
{
if (p == NULL) p = (int *)t->data;
if (level > 20) draw_error(84, " ", t);

while (*p != dr_end)
  {
  int errornumber = 0; 
  if (out_drawstackptr > draw_stacksize - 4) errornumber = 121;
    else if (out_drawstackptr < stack_rqd[*p]) errornumber = 71;
  
  if (errornumber != 0)
    {
    int i;
    uschar *s = US"???";
    for (i = 0; i < draw_operator_count; i++)
      if (draw_operators[i].value == *p)
        { s = US draw_operators[i].name; break; }
    draw_error(errornumber, CS s, t);
    }

  switch (*p++)
    {
    case dr_jump:
    p = (int *)(*p);
    break;

    case dr_bra:
      {
      int count = 1;
      draw_stack[out_drawstackptr++] = (int)p;
      while (*p != dr_end && (*p != dr_ket || --count > 0))
        {
        int type = *p++;
        if (type == dr_jump) p = (int *)(*p);
        else if (type == dr_bra) count++;
        else if (type == dr_draw || type == dr_varref || type == dr_number || 
          type == dr_text || type == dr_varname) p++;
        }
      if (*p != dr_ket) draw_error(116, " ", t); else p++;
      }
    break;

    case dr_ket:
    return TRUE;

    case dr_if:
      {
      int *pp = (int *)draw_stack[--out_drawstackptr];
      if (!check_ptr(pp, t)) { draw_error(117, " ", t); return FALSE; }
      if (draw_stack[--out_drawstackptr] != 0 &&
        !sub_draw(t, pp, x, y, c, overflag)) return FALSE;
      }
    break;

    case dr_ifelse:
      {
      int *pp2 = (int *)draw_stack[--out_drawstackptr];
      int *pp1 = (int *)draw_stack[--out_drawstackptr];
      if (!check_ptr(pp1, t) || !check_ptr(pp2, t)) { draw_error(117, " ", t); return FALSE; }
       if (!sub_draw(t, (draw_stack[--out_drawstackptr] != 0)? pp1 : pp2, x, y, c, overflag))
         return FALSE;
      }
    break;

    case dr_number:
    case dr_text:
    case dr_varname:
    draw_stack[out_drawstackptr++] = *p++;
    break;

    case dr_varref:
    draw_stack[out_drawstackptr++] = draw_variables[*p++];
    break;

    case dr_accleft:
    draw_stack[out_drawstackptr++] = n_maxaccleft;
    break;

    case dr_add:
    draw_stack[out_drawstackptr-2] += draw_stack[out_drawstackptr-1];
    out_drawstackptr--;
    break;
    
    case dr_and:
      {
      int a1 = draw_stack[--out_drawstackptr] / 1000;
      int a2 = draw_stack[--out_drawstackptr] / 1000;
      draw_stack[out_drawstackptr++] = (a1 & a2)*1000;
      }
    break;    

    case dr_barnumber:
      {
      int a = 0, b = 0;
      uschar buff[20];
      format_movt = curmovt;
      format_sprintf(buff, "%b", out_bar);
      sscanf(CS buff, "%d.%d", &a, &b);
      if (b < 10) b *= 100; else if (b < 100) b *= 10;
      draw_stack[out_drawstackptr++] = a * 1000 + b;
      }
    break;
    
    case dr_copy:
      {
      int count = draw_stack[--out_drawstackptr] / 1000;
      if (out_drawstackptr < count) { draw_error(71, "copy", t); break; }
      memcpy(draw_stack + out_drawstackptr, draw_stack + out_drawstackptr - count, 
        count * sizeof(int));       
      out_drawstackptr += count;
      } 
    break;  

    case dr_currentgray:
    draw_stack[out_drawstackptr++] = gray;
    break;

    case dr_currentlinewidth:
    draw_stack[out_drawstackptr++] = draw_thickness;
    break;

    case dr_currentpoint:
    if (cp <= 0) draw_error(72, "currentpoint", t);
    draw_stack[out_drawstackptr++] = x[xp-1] - draw_ox;
    draw_stack[out_drawstackptr++] = y[yp-1] - draw_oy;
    break;

    case dr_curveto:
      {
      int i;
      if (!currentpoint) draw_error(72, "curveto", t);
      for (i = 6; i >= 2; i -= 2)
        {
        x[xp++] = draw_ox + draw_stack[out_drawstackptr - i];
        y[yp++] = draw_oy + draw_stack[out_drawstackptr - i + 1];
        }
      out_drawstackptr -= 6;
      c[cp++] = path_curve;
      }
    break;

    case dr_cvs:
      {
      drawtextstr *d = (drawtextstr *)draw_stack[--out_drawstackptr];
      int n = draw_stack[--out_drawstackptr];
      if ((int)d < 0x8000 || d->ident != magic) draw_error(82, "cvs", t);
      format_sprintf(d->text, "%f", n);
      draw_stack[out_drawstackptr++] = (int)d;
      }
    break;

    case dr_def:
      {
      int n = draw_stack[out_drawstackptr-2];
      if (n < 0 || n > max_draw_variable)
        draw_error(101, "", t);
      else draw_variables[n] = draw_stack[out_drawstackptr-1];
      }
    out_drawstackptr -= 2;
    break;

    case dr_div:
    if (draw_stack[out_drawstackptr-1] == 0)
      {
      draw_error(95, "", t);
      }
    else
     {
     draw_stack[out_drawstackptr-2] =
        mac_muldiv(draw_stack[out_drawstackptr-2], 1000,
          draw_stack[out_drawstackptr-1]);
     out_drawstackptr--;
     }
    break;

    case dr_draw:
    level++;
    (void) sub_draw((tree_node *)(*p++), NULL, x, y, c, overflag);
    level--;
    break;
    
    case dr_dup:
    draw_stack[out_drawstackptr] = draw_stack[out_drawstackptr-1];
    out_drawstackptr++;
    break;

    case dr_eq:
    draw_stack[out_drawstackptr-2] =
      (draw_stack[out_drawstackptr-2] == draw_stack[out_drawstackptr-1])? 1000 : 0;  
    out_drawstackptr--;
    break;    

    case dr_exch:
      {
      int temp = draw_stack[out_drawstackptr-1];
      draw_stack[out_drawstackptr-1] = draw_stack[out_drawstackptr-2];
      draw_stack[out_drawstackptr-2] = temp;
      }
    break;

    case dr_exit:
    return FALSE;
      
    case dr_fill:
    if (!currentpoint) draw_error(72, "fill", t);
    c[cp++] = path_end;
    if (overflag) setup_overdraw(-1, x, y, c); else dev_path(x, y, c, -1);
    cp = xp = yp = 0;
    currentpoint = FALSE;
    break;

    case dr_fillretain:
    if (!currentpoint) draw_error(72, "fillretain", t);
    c[cp++] = path_end;
    if (overflag) setup_overdraw(-1, x, y, c); else dev_path(x, y, c, -1);
    break;

    case dr_fontsize:
      {
      drawtextstr *d = (drawtextstr *)draw_stack[--out_drawstackptr];
      if ((int)d < 0x8000 || d->ident != magic) draw_error(82, "fontsize", t);
      draw_stack[out_drawstackptr++] = ((curmovt->fontsizes)->fontsize_text)[d->size];
      }
    break;

    case dr_false:
    draw_stack[out_drawstackptr++] = 0;
    break;   
     
    case dr_gaptype:
    draw_stack[out_drawstackptr++] = draw_gap;
    break;

    case dr_ge:
    draw_stack[out_drawstackptr-2] =
      (draw_stack[out_drawstackptr-2] >= draw_stack[out_drawstackptr-1])? 1000 : 0;  
    out_drawstackptr--;
    break;    

    case dr_gt:
    draw_stack[out_drawstackptr-2] =
      (draw_stack[out_drawstackptr-2] > draw_stack[out_drawstackptr-1])? 1000 : 0;  
    out_drawstackptr--;
    break;    

    case dr_headbottom:
    draw_stack[out_drawstackptr++] =
      (n_minpitch - 128)*main_stavemagn - cuegrace_scale(2*main_stavemagn);
    break;

    case dr_headleft:
    draw_stack[out_drawstackptr++] = n_invertleft? cuegrace_scale(6*main_stavemagn) : 0;
    break;

    case dr_headright:
    draw_stack[out_drawstackptr++] = cuegrace_scale((n_invertright? 12 : 6)*main_stavemagn);
    break;

    case dr_headtop:
    draw_stack[out_drawstackptr++] =
      (n_maxpitch - 128)*main_stavemagn + cuegrace_scale(2*main_stavemagn);
    break;

    case dr_le:
    draw_stack[out_drawstackptr-2] =
      (draw_stack[out_drawstackptr-2] <= draw_stack[out_drawstackptr-1])? 1000 : 0;  
    out_drawstackptr--;
    break;    

    case dr_leftbarx:
    draw_stack[out_drawstackptr] = out_lastbarlinex - draw_ox;
    if (out_startlinebar) draw_stack[out_drawstackptr] -= 6000;
    out_drawstackptr++;
    break;

    case dr_linebottom:
    draw_stack[out_drawstackptr++] = (n_minpitch & 2)? 2*main_stavemagn : 0;
    break;

    case dr_linegapx:
    draw_stack[out_drawstackptr++] = draw_lgx;
    break;

    case dr_linegapy:
    draw_stack[out_drawstackptr++] = draw_lgy;
    break;

    case dr_linelength:
    draw_stack[out_drawstackptr++] = curmovt->linelength;
    break;

    case dr_lineto:
    if (!currentpoint) draw_error(72, "lineto", t);
    y[yp++] = draw_oy + draw_stack[--out_drawstackptr];
    x[xp++] = draw_ox + draw_stack[--out_drawstackptr];
    c[cp++] = path_line;
    break;

    case dr_linetop:
    draw_stack[out_drawstackptr++] = (n_maxpitch & 2)? 2*main_stavemagn : 0;
    break;
    
    case dr_loop:
      { 
      int count = 1000; 
      int *pp = (int *)draw_stack[--out_drawstackptr];
      if (!check_ptr(pp, t)) { draw_error(117, " ", t); return FALSE; }
      while (count-- > 0 && sub_draw(t, pp, x, y, c, overflag));
      } 
    break;  

    case dr_lt:
    draw_stack[out_drawstackptr-2] =
      (draw_stack[out_drawstackptr-2] < draw_stack[out_drawstackptr-1])? 1000 : 0;  
    out_drawstackptr--;
    break;    

    case dr_magnification:
    draw_stack[out_drawstackptr++] = main_magnification;
    break;

    case dr_moveto:
    y[yp++] = draw_oy + draw_stack[--out_drawstackptr];
    x[xp++] = draw_ox + draw_stack[--out_drawstackptr];
    c[cp++] = path_move;
    currentpoint = TRUE;
    break;

    case dr_mul:
    draw_stack[out_drawstackptr-2] =
      mac_muldiv(draw_stack[out_drawstackptr-1],
        draw_stack[out_drawstackptr-2], 1000);
    out_drawstackptr--;
    break;

    case dr_ne:
    draw_stack[out_drawstackptr-2] =
      (draw_stack[out_drawstackptr-2] != draw_stack[out_drawstackptr-1])? 1000 : 0;  
    out_drawstackptr--;
    break;    

    case dr_neg:
    draw_stack[out_drawstackptr-1] = -draw_stack[out_drawstackptr-1];
    break;
    
    case dr_not:
    draw_stack[out_drawstackptr-1] = (~(draw_stack[out_drawstackptr-1]/1000))*1000;   
    break; 

    case dr_or:
      {
      int a1 = draw_stack[--out_drawstackptr] / 1000;
      int a2 = draw_stack[--out_drawstackptr] / 1000;
      draw_stack[out_drawstackptr++] = (a1 | a2)*1000;
      }
    break;    

    case dr_origin:
    case dr_originx:
    draw_stack[out_drawstackptr++] = draw_ox;
    break;

    case dr_originy:
    draw_stack[out_drawstackptr++] = draw_oy;
    break;

    case dr_pagelength:
    draw_stack[out_drawstackptr++] = main_pagelength;
    break;

    case dr_pagenumber:
    draw_stack[out_drawstackptr++] = curpage->number * 1000;
    break;

    case dr_topleft:
    draw_stack[out_drawstackptr++] = - draw_ox;
    draw_stack[out_drawstackptr++] =   out_ystave - draw_oy;
    break;

    case dr_pop:
    out_drawstackptr--;
    break;

    case dr_pstack:
      {
      int i;
      info_printf("Draw stack: ");
      for (i = 0; i < out_drawstackptr; i++)
        info_printf("%f ", draw_stack[i]);
      info_printf("\n");
      }
    break;

    case dr_rcurveto:
      {
      int i;
      if (!currentpoint) draw_error(72, "rcurveto", t);
      for (i = 0; i <= 2; i++)
        {
        x[xp+i] = x[xp-1] + draw_stack[out_drawstackptr + 2*i - 6];
        y[yp+i] = y[yp-1] + draw_stack[out_drawstackptr + 2*i - 5];
        }
      xp += 3;
      yp += 3;
      out_drawstackptr -= 6;
      c[cp++] = path_curve;
      }
    break;

    case dr_repeat:
      { 
      int *pp = (int *)draw_stack[--out_drawstackptr];
      int count = draw_stack[--out_drawstackptr] / 1000; 
      if (!check_ptr(pp, t)) { draw_error(117, " ", t); return FALSE; }
      while (count-- > 0 && sub_draw(t, pp, x, y, c, overflag));
      } 
    break;  

    case dr_rlineto:
    if (!currentpoint) draw_error(72, "rlineto", t);
    y[yp] = y[yp-1] + draw_stack[--out_drawstackptr];
    yp++;
    x[xp] = x[xp-1] + draw_stack[--out_drawstackptr];
    xp++;
    c[cp++] = path_line;
    break;

    case dr_rmoveto:
    if (!currentpoint) draw_error(72, "rmoveto", t);
    y[yp] = y[yp-1] + draw_stack[--out_drawstackptr];
    yp++;
    x[xp] = x[xp-1] + draw_stack[--out_drawstackptr];
    xp++;
    c[cp++] = path_move;
    break;

    case dr_roll:
      {
      int i, j;
      int amount = (draw_stack[--out_drawstackptr])/1000;
      int count = (draw_stack[--out_drawstackptr])/1000;

      if (out_drawstackptr < count)
        draw_error(71, "roll", t);

      if (amount > 0) for (i = 0; i < amount; i++)
        {
        int temp = draw_stack[out_drawstackptr - 1];
        for (j = 1; j < count; j++)
          draw_stack[out_drawstackptr-j] = draw_stack[out_drawstackptr-j-1];
        draw_stack[out_drawstackptr - count] = temp;
        }

      else for (i = 0; i < -amount; i++)
        {
        int temp = draw_stack[out_drawstackptr - count];
        for (j = count; j > 1; j--)
          draw_stack[out_drawstackptr-j] = draw_stack[out_drawstackptr-j+1];
        draw_stack[out_drawstackptr - 1] = temp;
        }
      }
    break;

    case dr_setgray:
    gray = draw_stack[--out_drawstackptr];
    dev_setgray(gray);
    break;

    case dr_setlinewidth:
    draw_thickness = mac_muldiv(draw_stack[--out_drawstackptr],
      main_stavemagn, 1000);
    break;

    case dr_stringwidth:
    case dr_show:
      {
      int *matrix;
      int xx, yy, width, flags, unscaled_fontsize, fontsize;
      int boxring = 0;

      drawtextstr *d = (drawtextstr *)draw_stack[--out_drawstackptr];
      if ((int)d < 0x8000 || d->ident != magic) draw_error(82, "show", t);
      unscaled_fontsize = ((curmovt->fontsizes)->fontsize_text)[d->size];
      fontsize = mac_muldiv(main_stavemagn, unscaled_fontsize, 1000);
      matrix = ((curmovt->fontsizes)->fontmatrix_text)[d->size];
      if (matrix != NULL) memcpy(font_transform, matrix, 4*sizeof(int));
      if (d->rotate) font_rotate(d->rotate);

      width = string_width(d->text, font_rm, fontsize);

      /* If stringwidth, just return values */

      if (p[-1] == dr_stringwidth)
        {
        draw_stack[out_drawstackptr++] = width;
        draw_stack[out_drawstackptr++] = font_stringheight;
        }

      /* Else carry on to do the showing */

      else
        {
        if (!currentpoint) draw_error(72, "show", t);
        xx = x[xp-1];
        yy = y[yp-1];

        flags = d->flags;
        boxring = flags & (text_box | text_ring);

        if ((flags & text_centre) != 0)
          {
          xx -= width/2;
          yy -= font_stringheight/2;
          }
        else if ((flags & text_endalign) != 0)
          {
          xx -= width;
          yy -= font_stringheight;
          }
        else
          {
          y[yp++] = yy + font_stringheight;
          x[xp++] = xx + width;
          c[cp++] = path_move;
          }

        if (overflag)
          {
          overdrawstr *last = out_overdraw;
          overdrawstr *new = store_Xget(sizeof(overdrawstr));
          if (last == NULL) out_overdraw = new; else
            {
            while (last->next != NULL) last = last->next;
            last->next = new;
            }
          new->next = NULL;
          new->texttype = TRUE;
          new->d.t.text = d->text;
          new->d.t.fontsize = fontsize;
          new->d.t.boxring = boxring;
          new->d.t.xx = xx;
          new->d.t.yy = out_ystave - yy;
          memcpy(new->d.t.matrix, font_transform, 4*sizeof(int));
          }
        else out_string(d->text, font_rm, fontsize, xx, out_ystave - yy, boxring);
        }
      font_reset();
      }
    break;

    case dr_stavesize:
    draw_stack[out_drawstackptr++] = main_stavemagn;
    break;

    case dr_stavestart:
    draw_stack[out_drawstackptr++] = (out_stave < 0)? 0 :
      (out_sysblock->startxposition + out_sysblock->xjustify - draw_ox);
    break;

    case dr_stavespace:
    draw_stack[out_drawstackptr++] = out_sysblock->stavespacing[out_stave];
    break;

    case dr_stembottom:
    draw_stack[out_drawstackptr] = (n_minpitch - 130)*1000;
    if ((n_flags & (nf_stem | nf_stemup)) == nf_stem)
      draw_stack[out_drawstackptr] -= cuegrace_scale(12000 + n_stemlength);
    draw_stack[out_drawstackptr] = mac_muldiv(draw_stack[out_drawstackptr],
      main_stavemagn, 1000);
    out_drawstackptr++;
    break;

    case dr_stemtop:
    draw_stack[out_drawstackptr] = (n_maxpitch - 126)*1000;
    if ((n_flags & (nf_stem | nf_stemup)) == (nf_stem | nf_stemup))
      draw_stack[out_drawstackptr] += cuegrace_scale(12000+ n_stemlength);
    draw_stack[out_drawstackptr] = mac_muldiv(draw_stack[out_drawstackptr],
      main_stavemagn, 1000);
    out_drawstackptr++;
    break;

    /*** Code common with dr_show
    case dr_stringwidth:
    ***/

    case dr_stroke:
    if (!currentpoint) draw_error(72, "stroke", t);
    c[cp++] = path_end;
    if (overflag) setup_overdraw(draw_thickness, x, y, c);
      else dev_path(x, y, c, draw_thickness);
    cp = xp = yp = 0;
    currentpoint = FALSE;
    break;

    case dr_systemdepth:
    draw_stack[out_drawstackptr++] = out_sysblock->systemdepth;
    break;

    case dr_sub:
    draw_stack[out_drawstackptr-2] -= draw_stack[out_drawstackptr-1];
    out_drawstackptr--;
    break;

    case dr_translate:
    draw_oy += draw_stack[--out_drawstackptr];
    draw_ox += draw_stack[--out_drawstackptr];
    break;
    
    case dr_true:
    draw_stack[out_drawstackptr++] = 1000;
    break;   
     
    case dr_xor:
      {
      int a1 = draw_stack[--out_drawstackptr] / 1000;
      int a2 = draw_stack[--out_drawstackptr] / 1000;
      draw_stack[out_drawstackptr++] = (a1 ^ a2)*1000;
      }
    break;    
    }
  }
  
return TRUE; 
}




/*************************************************
*    Prime stack and interpret a draw function   *
*************************************************/

/* This is the external interface to the drawing action. */

void out_dodraw(tree_node *t, int *args, BOOL overflag)
{
int x[100];
int y[100];
int c[100];

if (args != NULL)
  {
  int i;
  for (i = 1; i <= args[0]; i++) draw_stack[out_drawstackptr++] = args[i];
  }

xp = yp = cp = level = gray = 0;
currentpoint = FALSE;
dev_setgray(0);
(void) sub_draw(t, NULL, x, y, c, overflag);
dev_setgray(0);
}

/* End of setdraw.c */
