/*************************************************
*    The PMW Music Typesetter - 3rd incarnation  *
*************************************************/

/* Copyright (c) Philip Hazel, 1991 - 2005 */

/* Written by Philip Hazel, starting November 1991 */
/* This file last modified: September 2005 */


/* This file contains initializing code, including the
main program, which is the entry point to PMW. */


#include "pmwhdr.h"
#include "outhdr.h"


static int dsb_bar = -1;
static int dsb_movement = -1;
static int dsb_stave = -1;


/* If you increase the number of keys, make sure that the keyoffset field
in rdargs.c is big enough. */

static char *arg_pattern =
  ","
  "a4ona3/s,"
  "a4sideways/s,"
  "a5ona4/s,"
  "c/k/n,"
  "debug/s,"
  "dsb/k,"
  "duplex/s,"
  "eps/s,"
  "F/k,"
  "f/k,"
  "H/k,"
  "-help=help/s,"
  "includefont/s,"
  "MP/k,"
  "MV/k,"
  "manualfeed/s,"
  "midi/k,"
  "mb=midibars/k,"
  "mm=midimovement/k/n,"
  "nr=norepeats/s,"
  "nw=nowidechars/s,"
  "o/k,"
  "p/k,"
  "pamphlet/s,"
  "printadjust/k/2,"
  "printgutter/k,"
  "printscale/k,"
  "printside/k/n,"
  "reverse/s,"
  "s/k,"
  "t/k/n,"
  "tumble/s,"
  "V/s,"
  "v/s";


enum {
  arg_aa_input,    /* The only unkeyed possibility */
  arg_a4ona3,
  arg_a4sideways,
  arg_a5ona4,
  arg_c,
  arg_debug,
  arg_dsb,
  arg_duplex,
  arg_eps,
  arg_F,
  arg_f,
  arg_H,
  arg_help,
  arg_includefont,
  arg_MP,
  arg_MV,
  arg_manualfeed,
  arg_midi,
  arg_midibars,
  arg_midimovement,
  arg_norepeats,
  arg_nowidechars,
  arg_o,
  arg_p,
  arg_pamphlet,
  arg_printadjustx,
  arg_printadjusty,
  arg_printgutter,
  arg_printscale,
  arg_printside,
  arg_reverse,
  arg_s,
  arg_t,
  arg_tumble,
  arg_V,
  arg_v
};


/*************************************************
*          Given help on command syntax          *
*************************************************/

static void givehelp(void)
{
printf("\nPMW version %s\n", version_string);

printf("\n            OPTIONS\n\n");
printf("-a4ona3               print A4 images 2-up on A3\n");
printf("-a5ona4               print A5 images 2-up on A4\n");
printf("-a4sideways           assume A4 paper fed sideways\n");
printf("-c <number>           set number of copies\n");
printf("-debug                write debugging info to stderr\n");
printf("-dsb <m>,<s>,<b>      write internal debugging bar data\n");
printf("-duplex               set duplex printing in the PostScript\n");
printf("-eps                  output encapsulated PostScript\n");
printf("-F <directory>        specify fontmetrics directory\n");
printf("-f <name>             specify format name\n");
printf("-H <file>             specify PostScript header file\n");
printf("-help                 output this information\n");
printf("-includefont          include PMW font in the output\n");
printf("-MP <file>            specify MIDIperc file\n");
printf("-MV <file>            specify MIDIvoices file\n");
printf("-manualfeed           set manualfeed in the PostScript\n");
printf("-mb <range>           synonym for -midibars\n");
printf("-midi <file>          specify MIDI output file\n");
printf("-midibars <range>     limit MIDI output to given bar range\n");
printf("-midimovement <n>     specifies movement for MIDI output\n");
printf("-mm <n>               synonym for -midimovement\n");
#ifndef NO_PMWRC
printf("-norc                 don't read .pmwrc (must be first option)\n");
#endif
printf("-norepeats            do not play repeats in MIDI output\n");
printf("-nowidechars          don't use 100-point stave chars\n");
printf("-nr                   synonym for -norepeats\n");
printf("-nw                   synonym for -nowidechars\n");
printf("-o <file>             specify output file ('-' for stdout)\n");
printf("-p <list>             select pages\n");
printf("-pamphlet             print pages in pamphlet order\n");
printf("-printadjust <x> <y>  move on page by (x,y)\n");
printf("-printgutter <x>      move recto/verso pages by x/-x\n");
printf("-printscale <n>       scale the image by n\n");
printf("-printside <n>        print only odd or even sides\n");
printf("-reverse              output pages in reverse order\n");
printf("-s <list>             select staves\n");
printf("-t <number>           set transposition\n");
printf("-tumble               set tumble for duplex printing\n");
printf("-V                    output PMW version number\n");
printf("-v                    output verification information\n");

printf("\nDefault output is <input>.ps when a file name is given.\n");
printf("Default output is stdout if no file name is given.\n");

printf("\n          EXAMPLES\n\n");
printf("pmw myscore\n");
printf("pmw -s 1,2-4 -p 3,6-10,11 -f small -c 2 k491.pmw\n");
printf("pmw -pamphlet -a5ona4 scorefile\n");
printf("pmw -s 1 -midi zz.mid -mm 2 -mb 10-20 sonata\n");
}



/*************************************************
*           Print routine for info display       *
*************************************************/

/* This could just be replaced by fprintf() to stderr nowadays, but we keep the
separate function just in case in the future we want do so something else with
all this output. */

void info_printf(char *format, ...)
{
uschar buff[256];
va_list ap;
va_start(ap, format);
format_vsprintf(buff, format, ap);
fprintf(stderr, "%s", CS buff);
va_end(ap);
}



/*************************************************
*        Display information about music         *
*************************************************/

/* This function is called after pagination if the -v option is present */

void display_info(void)
{
pagestr *p = main_pageanchor;
int movt;
int laststave = -1;
int toppitch[max_stave+1];
int botpitch[max_stave+1];
int totalpitch[max_stave+1];
int notecount[max_stave+1];

info_printf("Data store used = ");
if (main_storetotal < 10000) info_printf("%d", main_storetotal);
  else info_printf("%dK", main_storetotal/1024);
info_printf(" (stave data ");
if (main_storestaves < 10000) info_printf("%d", main_storestaves);
  else info_printf("%dK", main_storestaves/1024);
info_printf(")\n");

/* Display information about the staves in each movement */

for (movt = 1; movt <= main_lastmovement; movt++)
  {
  int stave;
  movtstr *m = movement[movt];

  info_printf("\nMOVEMENT %d\n\n", movt);

  for (stave = 0; stave <= m->laststave; stave++)
    {
    stavestr *s = (m->stavetable)[stave];
    if (s == NULL) continue;               /* skips stave 0 if not there */

    info_printf("Stave %2d: ", stave);

    if (m->totalnocount == 0)
      info_printf("%d bar%s", s->lastbar, (s->lastbar == 1)? "":"s");
    else info_printf("%d(+%d) bars",
      s->lastbar - m->totalnocount, m->totalnocount);

    if (stave > laststave)
      {
      laststave = stave;
      toppitch[stave] = -1;
      botpitch[stave] = 9999;
      notecount[stave] = totalpitch[stave] = 0;
      }

    if (s->notecount > 0)
      {
      info_printf(";%s range  %p to %p average %p",
        (s->lastbar == 1)? " ":"",
          s->botpitch, s->toppitch, s->totalpitch/s->notecount);

      if (s->toppitch > toppitch[stave]) toppitch[stave] = s->toppitch;
      if (s->botpitch < botpitch[stave]) botpitch[stave] = s->botpitch;
      totalpitch[stave] += s->totalpitch;
      notecount[stave] += s->notecount;
      }

    info_printf("\n");
    }
  }


/* If there is more than one movement, display overall information for
each stave. */

if (main_lastmovement > 1)
  {
  int stave;
  info_printf("\nOVERALL\n\n");
  for (stave = 1; stave <= laststave; stave++)
    {
    info_printf("Stave %2d: ", stave);
    if (notecount[stave] > 0)
      info_printf("range  %p to %p average %p",
        botpitch[stave], toppitch[stave], totalpitch[stave]/notecount[stave]);
    info_printf("\n");
    }
  }


/* Now display information about the page layout */

if (p != NULL) info_printf("\nPAGE LAYOUT\n\n");

while (p != NULL)
  {
  int count = 14;
  sysblock *s = p->sysblocks;
  info_printf("Page %d bars: ", p->number);

  while (s != NULL)
    {
    if (s->type == sh_system)
      {
      format_movt = s->movt;
      if (count > 65)
        {
        info_printf("\n ");
        count = 1;
        }
      info_printf("%b-%b%s ", s->barstart, s->barend,
        (s->flags & sysblock_stretch)? "":"*");
      count += 6;
      if (s->overrun < 30)
        {
        info_printf("(%d) ", s->overrun);
        count += 5;
        }
      }
    s = s->next;
    }

  info_printf("\n  Space left on page = %f", p->spaceleft);
  if (p->overrun > 0 && p->overrun < 100000)
    info_printf(" Overrun = %f", p->overrun);
  info_printf("\n");

  p = p->next;
  }
}



/*************************************************
*         Convert string to stave map            *
*************************************************/

/* An error is detected by the string's not having been used up. For some
reason gcc gives a weird warning if ss is used directly with strtol, even with
the right casts. That's why I use another variable of type char *. */

void init_strtomap(uschar *ss, uschar **endptr, int *map)
{
long int i;
char *sss = (char *)ss;
*endptr = ss;
mac_initstave(map, 0);
while (isdigit(*sss))
  {
  long int s = strtol(sss, &sss, 0);
  long int t = s;
  if (*sss == '-')
    {
    sss++;
    t = strtol(sss, &sss, 0);
    }
  if (t < s || t > max_stave) return;
  for (i = s; i <= t; i++) mac_setstave(map, i);
  while (*sss == ',' || *sss == ' ') sss++;
  }
*endptr = US sss;
}



/*************************************************
*             Decode command line                *
*************************************************/

static void decode_command(int argc, char **argv)
{
arg_result results[64];
int rc = rdargs(argc, argv, arg_pattern, results);

if (rc != 0) error_moan(0, results[0].data.text, results[1].data.text);

/* Deal with "V" and "help" */

if (results[arg_V].data.text != NULL)
  {
  printf("PMW version %s\n", version_string);
  printf("Copyright (c) Philip Hazel 2004\n");
  exit(EXIT_SUCCESS);
  }

if (results[arg_help].data.text != NULL)
  {
  givehelp();
  exit(EXIT_SUCCESS);
  }

/* Deal with verifying and debugging */

if (results[arg_v].data.text != NULL) verify = TRUE;

if (results[arg_debug].data.text != NULL)
  {
  debug_file = stderr;
  debugging = TRUE;
  debug_printf("PMW run started\n");
  }

if (results[arg_dsb].data.text != NULL)
  {
  debug_file = stderr;
  sscanf(results[arg_dsb].data.text, "%d,%d,%d", &dsb_movement, &dsb_stave,
    &dsb_bar);
  }

/* Deal with "from" & "o" */

if (results[arg_aa_input].data.text != NULL)
  arg_from_name = US results[arg_aa_input].data.text;

if (results[arg_o].data.text != NULL)
  arg_to_name = US results[arg_o].data.text;

/* Deal with overriding fontmetrics and psheader, MIDIperc, and MIDIvoices
files */

if (results[arg_F].data.text != NULL)
  font_metrics = US results[arg_F].data.text;

if (results[arg_H].data.text != NULL)
  ps_header = US results[arg_H].data.text;

if (results[arg_MP].data.text != NULL)
  midi_perc = US results[arg_MP].data.text;

if (results[arg_MV].data.text != NULL)
  midi_voices = US results[arg_MV].data.text;

/* Deal with MIDI output */

if (results[arg_midi].data.text != NULL)
  midi_filename = US results[arg_midi].data.text;

if (results[arg_midibars].data.text != NULL)
  {
  uschar *endptr;
  play_startbar = Ustrtoul(results[arg_midibars].data.text, &endptr, 10);
  if (*endptr == 0) play_endbar = play_startbar; else
    {
    if (*endptr++ != '-') error_moan(123);   /* Hard error */
    play_endbar = Ustrtoul(endptr, &endptr, 10);
    if (*endptr != 0) error_moan(123);         /* Hard error */
    }
  }

if (results[arg_midimovement].presence != arg_present_not)
  play_movt_number = results[arg_midimovement].data.number;

/* Some BOOL options */

if (results[arg_norepeats].data.text != NULL) play_repeats = FALSE;
if (results[arg_nowidechars].data.text != NULL) stave_use_widechars = FALSE;

/* Deal with stave selection */

if (results[arg_s].data.text != NULL)
  {
  uschar *endptr;
  init_strtomap(US results[arg_s].data.text, &endptr, main_staves);
  mac_setstave(main_staves, 0);
  if (*endptr != 0) error_moan(74);  /* Hard error */
  }

/* Deal with page selection */

if (results[arg_p].data.text != NULL)
  {
  uschar *s = US results[arg_p].data.text;
  stave_list *p;
  stave_list **pp = &output_pagelist;

  while (*s)
    {
    int first, last, n;
    int count = sscanf(CS s, "%d%n-%d%n", &first, &n, &last, &n);
    if (count == 0) error_moan(75); else   /* Hard error */
      {
      p = malloc(sizeof(stave_list));
      p->first = first;

      if (count == 1) p->last = first; else
        {
        if (first > last) error_moan(76);  /* Hard error */
          else p->last = last;
        }

      p->next = NULL;
      *pp = p;
      pp = &(p->next);
      }

    s += n;
    if (*s == ',') s++;
    }
  }

/* Deal with transposition */

if (results[arg_t].presence != arg_present_not)
  main_transpose = results[arg_t].data.number;

/* Deal with format */

if (results[arg_f].data.text != NULL)
  {
  int i;
  uschar *f = US results[arg_f].data.text;
  main_format = malloc(sizeof(f) + 1);
  for (i = 0; i <= Ustrlen(f); i++)
    main_format[i] = tolower(f[i]);
  }

/* Deal with copies */

if (results[arg_c].presence != arg_present_not)
  output_copies = results[arg_c].data.number;

/* Deal with a number of printing configuration options */

if (results[arg_reverse].data.text != NULL) print_reverse = TRUE;
if (results[arg_a4sideways].data.text != NULL) print_pagefeed = pc_a4sideways;
if (results[arg_a4ona3].data.text != NULL) print_imposition = pc_a4ona3;
if (results[arg_a5ona4].data.text != NULL) print_imposition = pc_a5ona4;

if (results[arg_includefont].data.text != NULL) output_includefont = TRUE;
if (results[arg_manualfeed].data.text != NULL) output_manualfeed = TRUE;
if (results[arg_duplex].data.text != NULL) output_duplex = TRUE;
if (results[arg_tumble].data.text != NULL) output_tumble = TRUE;

if (results[arg_pamphlet].data.text != NULL) print_pamphlet = TRUE;
if (results[arg_eps].data.text != NULL) print_imposition = pc_EPS;

if (results[arg_printadjustx].data.text != NULL)
  {
  float d;
  sscanf(results[arg_printadjustx].data.text, "%g", &d);
  print_image_xadjust = (int)(1000.0 * d);
  }

if (results[arg_printadjusty].data.text != NULL)
  {
  float d;
  sscanf(results[arg_printadjusty].data.text, "%g", &d);
  print_image_yadjust = (int)(1000.0 * d);
  }

if (results[arg_printgutter].data.text != NULL)
  {
  float d;
  sscanf(results[arg_printgutter].data.text, "%g", &d);
  print_gutter = (int)(1000.0 * d);
  }

if (results[arg_printscale].data.text != NULL)
  {
  float d;
  sscanf(results[arg_printscale].data.text, "%g", &d);
  print_magnification = (int)(1000.0 * d);
  if (print_magnification == 0) { printf("Bad printscale value\n"); exit(EXIT_FAILURE); }
  }

if (results[arg_printside].presence != arg_present_not)
  {
  int n = results[arg_printside].data.number;
  if (n == 1) print_side2 = FALSE;
    else if (n == 2) print_side1 = FALSE;
      else { printf("Printside must specify 1 or 2\n"); exit(EXIT_FAILURE); }
  }
}



/*************************************************
*                Entry Point                     *
*************************************************/

/* This comment is historical, left here for nostalgia purposes only:

------------------------------------------------------------------------------
PMW can be run in an windowing environment, or as a command-line program.
The windowing environments are of course system-specific, but have the
characteristic that they are event-driven. Thus in such an environment
we hand over control to a system routine, and only get it back (if at all)
when the program is finishing.

For the moment, we are thinking only of the RISC OS environment, but
writing as flexibly as possible, so that future ports are easier.
------------------------------------------------------------------------------

The future has arrived. This port of PMW for UNix-like systems runs only as a
single command, reading one input and producing one output. However, the code
has been minimally disturbed, which is why the style is a bit weird for such a
program. */

int main(int argc, char **argv)
{
int newargc;
char **newargv;
uschar to_buffer[256];
verify = FALSE;
ps_file = stdout;

sys_init1();                   /* Early system initialization */
version_init();                /* Set up for messages */

font_List = malloc(max_fonts * sizeof(fontstr));
font_table = malloc(font_tablen * sizeof(int));

/* Call the second local initialization function, and then decode the (possibly
modified) command line. */

newargc = sys_init2(argc, argv, &newargv, arg_pattern);
decode_command(newargc, newargv);

/* Set up default fonts */

font_init();

/* Initialize MIDI data */

read_midi_translation(&midi_voicenames, midi_voices);
read_midi_translation(&midi_percnames, midi_perc);

/* We could have a third system initializer after argument decoding, but
haven't seen the need yet. */

read_init();                   /* One-off read initialization */

/* One-time initialization */

beam_stemadjusts = malloc(max_beamsize);
movement = malloc((main_max_movements+1) * sizeof(movtstr *));

/* Initialization done */

main_initialized = TRUE;

/* In this version of PMW, the only facility provided is for the generation of
PostScript. It does not drive a screen directly. */

if (verify)
  {
  fprintf(stderr, "PMW version %s\n", version_string);
  main_shownlogo = TRUE;
  }

/* Sort out the input file, and possibly the name of the output file. If no
input file is supplied, read stdin. Output is to stdout (default) unless
overridden by -o (arg_to_name will be set). */

if (arg_from_name == NULL)
  {
  input_file = stdin;
  }
else
  {
  main_filename = arg_from_name;
  input_file = Ufopen(main_filename, "r");
  if (input_file == NULL)
    {
    error_moan(4, main_filename, strerror(errno));
    return rc_disaster;
    }
  if (arg_to_name == NULL)
    {
    uschar *p;
    arg_to_name = to_buffer;
    Ustrcpy(arg_to_name, arg_from_name);
    p = arg_to_name + Ustrlen(arg_to_name);
    while (p > arg_to_name && p[-1] != '.' && p[-1] != '/') p--;
    if (p > arg_to_name && p[-1] == '.') p[-1] = 0;
    Ustrcat(arg_to_name, ".ps");
    }
  }

/* Read the file */

if (verify) fprintf(stderr, "Reading ...\n");
read_start();
while (main_state == state_reading) read_go();
if (main_rc > rc_warning) return main_rc;

/* Output debugging if requested */

if (dsb_movement > 0) debug_showbar(dsb_movement, dsb_stave, dsb_bar);

/* Do the typesetting */

if (verify) fprintf(stderr, "Paginating ...\n");
main_state = state_paginating;
paginate_start();
while (main_state == state_paginating) paginate_go();
if (main_rc > rc_warning) return main_rc;

/* Show pagination information if verifying */

if (verify) display_info();

/* If a file name other than "-" is set for the output, open it. Otherwise
we'll be writing to stdout. */

if (arg_to_name != NULL && Ustrcmp(arg_to_name, "-") != 0)
  {
  if (verify)
    fprintf(stderr, "\nWriting PostScript file \"%s\" ...\n", arg_to_name);
  ps_file = Ufopen(arg_to_name, "w");
  if (ps_file == NULL)
    {
    error_moan(4, arg_to_name, strerror(errno));
    return main_rc;
    }
  }
else if (verify) fprintf(stderr, "\nWriting Postscript to stdout ...\n");

/* Set up for printing, and go for it */

main_state = state_printing;
print_lastpage = main_lastpage;
if (print_pamphlet) print_lastpage = (print_lastpage + 3) & (-4);

/* This diagram shows the computed values and the positions where the origin
can go in each case. In practice we take the upper value where there are two
possibilities.

 ------------ Sideways -------------   |   ------------ Upright -----------
 ----- 1-up -----   ----- 2-up -----   |   ---- 1-up ----    ---- 2-up ----
  Port     Land      Port     Land     |    Port     Land     Port     Land
 x------  -------   -------  ---x---   |   -----    x----    x----    -----
 |  0  |  |  4  |   |  2  |  |  6  |   |   |   |    |   |    |   |    |   |
 ------x  x------   x------  ---x---   |   | 1 |    | 5 |    | 3 |    x 7 |
                                       |   |   |    |   |    |   |    |   |
                                       |   x----    -----    ----x    -----
*/

print_pageorigin =
  ((print_pagefeed == pc_a4sideways)? 0 : 1) +
  ((print_imposition == -1)? 0 : 2) +
  (opt_landscape? 4 : 0);

ps_init();
ps_go();
if (ps_file != stdout) fclose(ps_file);

/* Write MIDI output if required */

if (midi_filename != NULL) midi_write();

return main_rc;
}

/* End of main.c */
