
/* Copyright (c) 1995, 1996, 1997, 1999, 2000, 2001, 2002, 2004, 2005,
 * 2006, 2007, 2008  by Arkkra Enterprises */
/* All rights reserved */

/* functions to support Mup macros */

/* When a macro is defined, its text is copied to a file. When a macro is
 * invoked, information about the current input file is pushed on a stack,
 * and the macro text is read from the macro file. This file also has the
 * code to handle include files. They are handled similarly. Info about the
 * current file is pushed on a stack, and text is read from the included
 * file. */

#include <string.h>
#include <fcntl.h>
#include "globals.h"

#ifdef VMS
#define unlink delete
#endif

/* size of macro name hash table. should be prime, big enough to not have too
 * many collisions, small enough to not use too much memory. */
#define MTSIZE	(79)

/* how many bytes to allocate at a time when collecting macro arguments */
#define MAC_ARG_SZ	(512)

/* We may need just a plain copy of the text passed as a macro parameter,
 * or a quoted copy, or both. These flags say which we need.
 * This is currently only actually implemented in a limited form:
 * we always generate an unquoted copy, and if there is any chance any
 * parameter in a macro will be needed in quoted form, we make quoted
 * copies of all parameters for that macro. This is because to really
 * figure out unequivicably which we need would be complicated enough
 * that we'd probably wind up spending more than we'd save,
 * so just doing this half-hearted optimization seems like the best tradeoff.
 */
#define QS_UNQUOTED	(0x1)
#define QS_QUOTED	(0x2)

/* A macro without parameters is defined without a following parenthesis,
 * and is different than a macro with zero parameters. This value must not be
 * a valid value for number of parameters, so must be negative. */
#define WITHOUT_PARAMS	(-1)

/* information about a macro parameter */
struct MAC_PARAM {
	char	*param_name;	/* name of the parameter */
	int	quote_style;	/* bitmap of QS_* values */
	struct MAC_PARAM *next;	/* for linked list */
};

/* info about a macro */
struct MACRO {
	char	*macname;	/* name of macro */
	char	*filename;	/* file in which macro was defined */	
	int	lineno;		/* line in file where macro definition began */
	int	lineoffset;	/* how many lines we are into macro */
	long	offset;		/* offset in macro temporary file where the
				 * text of the macro is stored for later use */
	long	quoted_offset;	/* offset into macro temp file where the
				 * quoted version is stored, if any.
				 * Only used on macro parameters. */
	struct MACRO 	*next;	/* for hash collision list */
	int	recursion;	/* incremented each time the macro is called,
				 * and decremented on completion. If this gets
				 * above 1 we are in trouble and ufatal */
	int	num_params;	/* We need to distinguish macros
				 * with empty parameter lists
				 * (e.g., define X() @) from macros
				 * with no parameters (e.g., define X @),
				 * so we represent the latter by
				 * WITHOUT_PARAMS. */
	struct	MAC_PARAM *parameters_p;	/* list of macro parameter
				 * names. Null if this macro doesn't
				 * have parameters */
};

/* macro information hash table */
static struct MACRO *Mactable[MTSIZE];

/* temporary file for saving text of macros. Need separate handles for reading
 * and writing */
static FILE *Mactmp_read = (FILE *) 0;
static FILE *Mactmp_write = (FILE *) 0;
#ifdef UNIX_LIKE_FILES
static char Macfile[] = "mupmacXXXXXX";		/* name of temp file */
#else
/* As last resort, we use a temp file name of 11 character length,
 * so make sure we have enough room for that. Usually L_tmpnam is
 * already longer than that, but better safe than core dump.
 */
#if L_tmpnam < 12
#undef L_tmpnam
#define L_tmpnam 12
#endif
static char Macfile[L_tmpnam];		/* name of temp file */
#endif

/* Some OSs require us to open files in binary mode. Most implementations
 * of fopen these days accept the 'b' suffix, even when they don't care
 * about it, but to be safe, we only add on systems that appear to have
 * binary mode defined. */
#ifdef O_BINARY
static char *Read_mode = "rb";
#ifndef UNIX_LIKE_FILES
static char *Write_mode = "wb";
static char *Append_mode = "ab";
#endif
#else
static char *Read_mode = "r";
#ifndef UNIX_LIKE_FILES
static char *Write_mode = "w";
static char *Append_mode = "a";
#endif
#endif

/* maximum number of files on file stack */
#ifndef FOPEN_MAX
#ifdef _NFILE
#define FOPEN_MAX	_NFILE
#else
#define FOPEN_MAX	(20)
#endif
#endif
/* The -5 is to account for stdin, stdout, stderr, and the 2 tmp file handles.
 * The +20 is to allow for nested macros. They can take 2 stack slots per call
 * since argument expansion also uses a slot, but they don't take extra
 * file descriptors, since all the macros are kept in one file. */
#define MAXFSTK	(FOPEN_MAX - 5 + 20)


struct FILESTACK {
	FILE	*file;
	char	*filename;
	long	fileoffset;		/* fseek position in file */
	int	lineno;			/* where we are in the file */
	int	lineno_inc;		/* saved Lineno_increment */
	struct MACRO	*mac_p;		/* if pushing because of a macro call,
					 * this will point to info about the
					 * macro, otherwise 0. */
};

static struct FILESTACK Filestack[MAXFSTK];

static int Fstkptr = -1;		/* stack pointer for Filestack */
static char quote_designator[] = "`";

extern int unlink();			/* to remove temp file */
extern char *tmpnam();		/* generate temp file name */

/* static function declarations */
static void pushfile P((FILE *file, char *filename, int lineno,
		struct MACRO *mac_p));
static char *path_combiner P((char *prefix));
static int is_absolute_path P((char *filename));
static struct MACRO *findMacro P((char *macname));
static int hashmac P(( char *macname));
static struct MACRO *setup_macro P((char *macname, int has_params));
static void free_parameters P((struct MAC_PARAM *param_p, char *macname,
		int values_only));
static char *dupstring P((char *string));
static char *mkmacparm_name P((char *macname, char *param_name));
static struct MACRO *resolve_mac_name P((char *macname));
static int has_quote_designator P((char *macname));




/* add macro name to hash table and arrange to save its text in temp file */

void
define_macro(macname)

char *macname;		/* name of macro to be defined */

{
	int has_params = NO;
	char *mac_name;		/* copy of macro name, because what gets
				 * passed in can get overwritten by parameter
				 * names */
	struct MACRO *mac_p;
	struct MAC_PARAM *parm_p;

	debug (4, "define_macro macname=%s\n", macname);

	/* there might be some leading white space in front of macro name,
	 * if so, ignore that */
	while (*macname == ' ' || *macname == '\t') {
		macname++;
	}

	/* if ends with a ( this macro has parameters */
	if (macname[strlen(macname) - 1] == '(') {
		/* this is a macro with parameters */
		macname[strlen(macname) - 1] = '\0';
		has_params = YES;
	}

	mac_p = setup_macro(macname, has_params);
	mac_name = mac_p->macname;

	if (has_params == YES) {
		get_parameters(mac_name);
	}

	/* copy the macro text to the macro temp file */
	if (save_macro(Mactmp_write) == YES) {
		/* There may have been a reference to a parameter to be
		 * quoted. So remind ourseleves to create quoted copies
		 * of parameters when this macro is called later.
		 * This is a half-hearted optimization:
		 * we will sometimes create extra copies unnecessarily,
		 * but much of the time we will get the benefit of a
		 * full optimization with only a fraction of the work. */
		for (parm_p = mac_p->parameters_p;
					parm_p != (struct MAC_PARAM *) 0;
					parm_p = parm_p->next) {
			parm_p->quote_style |= QS_QUOTED;
		}
	}
		

	/* terminate the macro with a NULL, so that when lex hits it when
	 * the macro is called, it will think it hit EOF */
	putc('\0', Mactmp_write);

	/* make sure macro has really been written and not still buffered */
	(void) fflush(Mactmp_write);

#ifndef UNIX_LIKE_FILES
	/* on non-unix system, multiple opens on the same file may not work,
	 * so open and close each time */
	fclose(Mactmp_write);
#endif
}


/* save macro info in hash table and make sure macro temporary file is
 * set up and ready to use */

static struct MACRO *
setup_macro(macname, has_params)

char *macname;		/* name of macro being defined */
int has_params;		/* YES or NO */

{
	struct MACRO *mac_p;	/* info about current macro */
	int h;			/* hash number */
	static int have_mac_file = NO;
#ifdef UNIX_LIKE_FILES
	int filedesc;		/* of tmp file for storing macros */
#endif


	/* if macro has not been defined before, add to hash table */
	if ((mac_p = findMacro(macname)) == (struct MACRO *) 0) {

		MALLOC(MACRO, mac_p, 1);
		h = hashmac(macname);
		mac_p->next = Mactable[h];
		Mactable[h] = mac_p;

		mac_p->macname = dupstring(macname);
		mac_p->recursion = 0;
	}
	else {
		l_warning(Curr_filename, yylineno,
					"macro '%s' redefined", macname);
		free_parameters(mac_p->parameters_p, macname, NO);
	}
	mac_p->parameters_p = (struct MAC_PARAM *) 0;
	/* We don't know how many parameters there will be yet,
	 * only whether there are some. If there will be some,
	 * we mark how many we have so far (zero). Note that it
	 * is possible to have a macro with zero parameters (define X())
	 * which is different than one without parameters (define X).
	 */
	mac_p->num_params = (has_params == YES ? 0 : WITHOUT_PARAMS);

	/* save current filename and line number so they can be used to give
	 * useful error messages when the macro is called */
	if (Fstkptr >= 0 && Filestack[Fstkptr].mac_p != (struct MACRO *) 0) {
		/* if current expanding a macro, get file/line info relative
		 * to the macro */
		mac_p->filename = Filestack[Fstkptr].mac_p->filename;
		mac_p->lineno = Filestack[Fstkptr].mac_p->lineno +
				Filestack[Fstkptr].mac_p->lineoffset;
	}
	else {
		mac_p->filename = Curr_filename;
		mac_p->lineno = yylineno;
	}

	/* if we don't yet have a temp file to store macro info, open one */
	if (have_mac_file == NO) {
		/* We need separate read/write file pointers in case of defining
		 * one macro inside another, so can't easily use tmpfile().
		 * The most straightforward way to do this is to use tmpnam.
		 * But recent versions of gcc complain loudly that tmpnam
		 * is dangerous--you should use mkstemp instead. While it's
		 * true tmpnam could have problems under certain conditions,
		 * Mup's use of it hardly qualifies as "dangerous." Yes,
		 * perhaps if you tried really hard by running lots and
		 * lots of instances of Mup simultaneously, and they all
		 * used macros, maybe you could get one of them to fail once
		 * in a while. But some systems may not support mkstemp,
		 * since it's more from BSD than System V, and it's unclear
		 * how it ought to function on a system that has DOS-style
		 * (8.3 character) filenames if you gave it a longer string
		 * than would be a legal file name.
		 * And, interestingly, the Solaris manual pages say
		 * that you should NOT use mkstemp, in direct conflict with
		 * gcc's opinion. They say to use tmpfile, which we can't do
		 * because that just gives a single file pointer for writing,
		 * but we want one for writing and one for reading.
		 * So it's very unclear what to do. For unix-ish systems
		 * we'll go with mkstemp, and for others use tmpnam.
		 * We get a temp file name, open it twice, once for writing,
		 * once for reading, then remove the file. This will make it a
 		 * hidden file that will disappear when the 2 file pointers
		 * are closed.
		 */
#ifdef UNIX_LIKE_FILES
		if ((filedesc = mkstemp(Macfile)) < 0) {
			ufatal("can't create temporary file for macro storage (possibly you don't have write permissions on directory/file?)");
		}
		if ((Mactmp_write = fdopen(filedesc, "w")) == (FILE *) 0) {
			pfatal("can't fdopen temporary file for macro storage");
		}
		if ((Mactmp_read = fopen(Macfile, "r")) == (FILE *) 0) {
			pfatal("can't open temporary file for reading macro");
		}

		/* arrange for file to vanish */
		(void) unlink(Macfile);
#else
		/* On non-UNIX systems, unlinking an open file may not have
		 * the desired effect, and some systems don't properly
		 * handle both a read and write FILE * on the same file.
		 * So for those we just create it here and have to open
		 * and close the file each time. */
		(void) tmpnam(Macfile);
		if ((Mactmp_write = fopen(Macfile, Write_mode)) == (FILE *) 0) {
			/* If tmpnam isn't implemented or fails,
			 * try a hard-coded temp name and hope
			 * for the best... */
			(void) strcpy(Macfile, "MupMacF.tmp");
			if ((Mactmp_write = fopen(Macfile, Write_mode))
							== (FILE *) 0) {
				ufatal("can't open temporary file for macro storage (possibly you don't have write permissions on directory/file?)");
			}
		}
#endif
		have_mac_file = YES;
	}
#ifndef UNIX_LIKE_FILES
	else {
		if ((Mactmp_write = fopen(Macfile, Append_mode)) == (FILE *) 0) {
			pfatal("can't open temporary file for macro storage (maybe out of memory?)");
		}
	}
#endif
	/* make sure we're at the end. Now that we have separate read/write
	 * file pointers, this should be unnecessary. However, on one non-UNIX
	 * system is seemed the opening for append didn't really work,
	 * requiring this fseek to actually go to the end. And in a previous
	 * variation of this code, this fseek was needed for the UNIX way too.
	 * In any case, it doesn't hurt to be sure we are where we want to be. */
	if (fseek(Mactmp_write, 0L, SEEK_END) != 0) {
		pfatal("fseek failed in setup_macro");
	}

	/* keep track of where this macro will begin in the temp file */
	mac_p->offset = ftell(Mactmp_write);

	return(mac_p);
}


#ifndef UNIX_LIKE_FILES
/* on non-unix systems, unlinking an open file may produce undesired results,
 * so clean up the macro temp file after parsing is done. Unfortunately,
 * this leave a higher possibility of leaving an orphan temp file. */

void
mac_cleanup()

{
	if (Macfile[0] != '\0') {
		unlink(Macfile);
	}
	Macfile[0] =  '\0';
}
#endif


/* look up macro name in hash table and return info about it, or null if
 * not defined */

static struct MACRO *
findMacro(macname)

char *macname;		/* which macro to look up */

{
	struct MACRO *mac_p;	/* pointer to info about macro */


	/* search hash table and collision chain off the table for match
	 * of macro name */
	for (mac_p = Mactable[ hashmac(macname) ]; mac_p != (struct MACRO *) 0;
						mac_p = mac_p->next) {

		if (strcmp(mac_p->macname, macname) == 0) {
			/* found it! */
			return(mac_p);
		}
	}

	/* macro not defined */
	return( (struct MACRO *) 0);
}


/* remove a macro definition. We just remove its entry from the hash table.
 * Its text will still remain in the macro temp file, because it seems like
 * too much trouble to do storage management on a temp file. */
/* Note that if asked to undef a macro that isn't defined, it silently
 * does nothing. */

void
undef_macro(macname)

char *macname;		/* which macro to undefine */

{
	struct MACRO *mac_p;	/* to walk though list of info about macros */
	struct MACRO **mac_p_p;	/* to keep track of delete place */


	/* there might be some leading white space in front of macro name,
	 * if so, ignore that */
	while (*macname == ' ' || *macname == '\t') {
		macname++;
	}

	/* keep track of where to delete from linked list */
	for (mac_p_p = &(Mactable[ hashmac(macname) ]);
					*mac_p_p != (struct MACRO *) 0;
					mac_p_p = &((*mac_p_p)->next)) {
		mac_p = *mac_p_p;
		if (strcmp(mac_p->macname, macname) == 0) {
			
			/* found it--delete this entry from list */
			*mac_p_p = mac_p->next;
			FREE(mac_p->macname);
			free_parameters(mac_p->parameters_p, macname, NO);
			FREE(mac_p);
			return;
		}
	}
}


/* generate hash number from macro name */

static int
hashmac(macname)

char *macname;

{
	int h;

	/* add up characters of name and take sum modulo hash table size */
	for (h = 0; *macname != '\0'; macname++) {
		h += *macname;
	}
	return(h % MTSIZE);
}


/* when macro is called, arrange to read its text from macro file */

void
call_macro(macname)

char *macname;		/* which macro to call */

{
	struct MACRO *mac_p;	/* info about the macro */


	debug(4, "call_macro macname=%s\n", macname);

	if ((mac_p = resolve_mac_name(macname)) == (struct MACRO *) 0) {
		l_yyerror(Curr_filename, yylineno,
				"macro '%s' not defined", macname);
		return;
	}

	/* If macro has parameters, remove any previous arguments
	 * and gather the arguments for this call. */
	if (mac_p->num_params >= 0) {
		free_parameters(mac_p->parameters_p, macname, YES);
		if (get_mac_arguments(macname, mac_p->num_params) == NO) {
			/* something was wrong with argument. Don't bother
			 * trying to expand, because we'll probably just
			 * just lots more error messages */
			return;
		}
	}

#ifndef UNIX_LIKE_FILES
	if ((Mactmp_read = fopen(Macfile, Read_mode)) == (FILE *) 0) {
		pfatal("can't open macro file for reading (maybe out of memory?)");
	}
#endif

	/* save old yyin value and make macro definition the input */
	pushfile(Mactmp_read, mac_p->filename, mac_p->lineno, mac_p);
	
	/* go to where macro definition begins */
	if (fseek(Mactmp_read, (has_quote_designator(macname)
			? mac_p->quoted_offset : mac_p->offset),
			SEEK_SET) != 0) {
		pfatal("fseek failed in call_macro");
	}
}


/* save info about current yyin and set yyin to specified file */

static void
pushfile(file, filename, lineno, mac_p)

FILE *file;		/* replace current file with this file */
char *filename;		/* name of new file to use */
int lineno;		/* current linenumber in new file */
struct MACRO *mac_p;	/* if switching to macro temp file because of a
			 * macro call, this is information about macro.
			 * Otherwise, it will be null. */

{
	debug(4, "pushfile file=%s", filename);

	/* do error checks */
	if (++Fstkptr >= MAXFSTK) {
		l_ufatal(filename, lineno,
			"too many nested files or macros (%d levels maximum)\n",
			MAXFSTK);
	}
	
	if (mac_p != (struct MACRO *) 0) {
		if ( ++(mac_p->recursion) != 1) {
			l_ufatal(Curr_filename, yylineno,
				"macro '%s' called recursively",
				mac_p->macname);
		}
	}

	/* save current info */
	Filestack[Fstkptr].file = yyin;
	Filestack[Fstkptr].fileoffset = ftell(yyin);
	Filestack[Fstkptr].filename = Curr_filename;
	Filestack[Fstkptr].lineno = yylineno;
	Filestack[Fstkptr].lineno_inc = Lineno_increment;
	Lineno_increment = 0;
	Filestack[Fstkptr].mac_p = mac_p;

	/* arrange to use the new file */
	new_lexbuff(file);

	/* if we are now expanding a macro, we don't change the input line
	 * number.  If doing an include, then we do. */
	if (Filestack[Fstkptr].mac_p == (struct MACRO *) 0) {
		yylineno = lineno;
		Curr_filename = filename;
	}
	else {
		Filestack[Fstkptr].mac_p->lineoffset = 0;
	}
}


/* if there are any files on the Filestack, go back to the previous one
 * on the stack. Return 1 if something was popped, 0 if stack was empty */

int
popfile()

{
	debug(4, "popfile");

	/* if nothing on file stack, nothing to do  except return 0 */
	if (Fstkptr < 0) {
		return(0);
	}

	if (Filestack[Fstkptr].mac_p != (struct MACRO *) 0) {
		/* returning from macro call */
		Filestack[Fstkptr].mac_p->recursion = 0;
#ifndef UNIX_LIKE_FILES
		(void) fclose(yyin);
#endif
	}
	else {
		/* this is an include rather than a macro file, so close it */
		(void) fclose(yyin);
	}

	/* set things back to the previous file */
	yyin = Filestack[Fstkptr].file;
	Curr_filename = Filestack[Fstkptr].filename;
	if (Filestack[Fstkptr].fileoffset >= 0) {
		if (fseek(yyin, Filestack[Fstkptr].fileoffset, SEEK_SET) != 0) {
			pfatal("fseek failed in popfile");
		}
	}
	yylineno = Filestack[Fstkptr].lineno;
	Lineno_increment = Filestack[Fstkptr].lineno_inc;

	/* go back to previous file */
	del_lexbuff();

	/* decrement stackpointer */
	Fstkptr--;

	return(1);
}


/* return 1 if we are NOT currently expanding a macro, 0 if we are.
 * This backwards logic is used because when we ARE doing a macro, we
 * should NOT muck with yylineno, and vice versa. If in a macro, adjust
 * the line offset within the macro by the specified amount. */

int
not_in_mac(inc_dec)

int inc_dec;	/* how much to increment/decrement the in-macro line offset */
{
	if (Fstkptr >= 0 && Filestack[Fstkptr].mac_p != (struct MACRO *) 0) {
		/* we are in a macro */
		Filestack[Fstkptr].mac_p->lineoffset += inc_dec;
		return(0);
	}
	else {
		return(1);
	}
}


/* if an error occurs while expanding a macro, give additional help */

void
mac_error()

{
	struct MACRO *mac_p;

	if (Fstkptr >= 0 && (mac_p = Filestack[Fstkptr].mac_p)
						!= (struct MACRO *) 0) {
		(void) fprintf(stderr, "note: previous error found while expanding macro %s'%s' from %s: line %d:\n",
				(strchr(mac_p->macname, '(') ? "parameter " : ""),
				mac_p->macname,
				mac_p->filename,
				mac_p->lineno + mac_p->lineoffset);
		print_offending_line(mac_p->filename, mac_p->lineno + mac_p->lineoffset);
	}
}


/* process an included file */

void
includefile(fname)

char *fname;	/* name of file to include */

{
	FILE *file;		/* the included file */
	char *fnamecopy;


	/* attempt to open file. Give message if fail */
	/* Note that if we find the file somewhere up the MUPPATH
	 * rather than directly, fname will be updated to contain the
	 * actual path of the file we are using, so we can tell that
	 * to the user. That way if they have several files by the same
	 * name but in different directories, and we're picking up a different
	 * one than they had intended, we can give them a clue of what's
	 * going on... */
	if ((file = find_file(&fname)) == (FILE *) 0) {

		l_ufatal(Curr_filename, yylineno,
				"can't open include file '%s'", fname);
	}

	/* need to make copy of file name */
	fnamecopy = dupstring(fname);

	/* arrange to connect yyin to the included file, save info, etc */
	pushfile(file, fnamecopy, 1, (struct MACRO *) 0);
}


/* Find a file to be included. First look using file name as is.
 * If not found, check if is absolute path name, and if so, give up.
 * Otherwise, try prepending each component of $MUPPATH in turn,
 * and trying that as a path. If a file is found, return that.
 * If we reach the end of the list, give up. 
 * Giving up means returning 0. If we find it somewhere up the MUPPATH
 * rather than directly, update filename to point to the actual path used.
 */

FILE *
find_file(filename_p)

char **filename_p;

{
	char *filename;
	FILE *file;
	char *fullpath;
	char *combiner;			/* what goes between path components */
	char *envmuppath;		/* from getenv("MUPPATH") */
	char *muppath;			/* value of $MUPPATH */
	char *prefix;			/* component of $MUPPATH,
					 * to prepend to filename */
	char *path_separator = "\0";	/* between components in $MUPPATH */


	/* first try name just as it is. */
	filename = *filename_p;
	if ((file = fopen(filename, Read_mode)) != (FILE *) 0) {
		return(file);
	}

	/* If it's an absolute path, we have to give up */
	if (is_absolute_path(filename)) {
		return ((FILE *) 0);
	}

	/* See if user set a $MUPPATH for where to look for includes */
	if ((envmuppath = getenv("MUPPATH")) != (char *) 0) {

#ifdef UNIX_LIKE_PATH_RULES
		path_separator = ":";
#endif
#ifdef DOS_LIKE_PATH_RULES
		path_separator = ";";
#endif
		if (*path_separator == '\0') {
			/* If user went to the trouble of setting $MUPPATH,
			 * they probably think it will work, so we better
			 * let them know it doesn't. Hopefully they will
			 * tell us what path rules their operating system
			 * uses, so we support it in the next release. */
			warning("MUPPATH facility not implemented for this operating system");
			return (FILE *) 0;
		}

		/* make a copy, so strtok can overwrite the separator */
		MALLOCA(char, muppath, strlen(envmuppath) + 1);
		(void) strcpy(muppath, envmuppath);

		/* walk through $MUPPATH */
		for (prefix = strtok(muppath, path_separator);
				prefix != (char *) 0;
				prefix = strtok(0, path_separator)) {
	
			combiner = path_combiner(prefix);

			/* get enough space for the full name */
			MALLOCA(char, fullpath, strlen(filename) +
				strlen(prefix) + 1 + strlen(combiner));

			/* create full path */
			sprintf(fullpath, "%s%s%s", prefix, combiner, filename);

			/* See if this file exists */
			debug(4, "checking '%s' for include, using MUPPATH",
					fullpath);
			if ((file = fopen(fullpath, Read_mode)) != (FILE *) 0) {
				*filename_p = fullpath;
				FREE(muppath);
				return(file);
			}

			/* no file here, no need to save this path */
			FREE(fullpath);
		}
		FREE(muppath);
	}

	return (FILE *) 0;
}


/* return true if given filename is an absolute path name */

static int
is_absolute_path(filename)

char *filename;

{

#ifdef UNIX_LIKE_PATH_RULES
	/* For Unix, a pathname is absolute if it starts with a slash */
	return (*filename == '/');
#endif

#ifdef DOS_LIKE_PATH_RULES
	/* If second character is a colon, then absolute */
	if (*filename != '\0' && *(filename + 1) == ':') {
		return(1);
	}
	else {
		return(0);
	}
#endif

#if ! defined(UNIX_LIKE_PATH_RULES) && ! defined(DOS_LIKE_PATH_RULES)
	/* Not implemented for this operating system. We'll pretend
	 * it's not, which will make it fall through and fail later. */
	return(0);
#endif
}


/* What to use to glue together a prefix and relative path to get a
 * full path. */

static char *
path_combiner(prefix)

char *prefix;

{

#ifdef UNIX_LIKE_PATH_RULES
	/* Unix separator is slash. If there was already a slash at the
	 * end of the prefix, no problem: multiples slashes are like one */
	return ("/");
#endif

#ifdef DOS_LIKE_PATH_RULES
	/* Use backslash, unless prefix ended with slash or backslash,
	 * in which case we don't need anything. */
	char last_ch;
	last_ch = prefix[strlen(prefix) - 1];
	return ((last_ch == '\\' || last_ch == '/') ? "" : "\\");
#endif

#if ! defined(UNIX_LIKE_PATH_RULES) && ! defined(DOS_LIKE_PATH_RULES)
	/* Shouldn't be here. Unimplemented for this operating system */
	return ("");
#endif
}


/* return YES if macro is currently defined, NO if it isn't */

int
is_defined(macname, paramtoo)

char *macname;
int paramtoo;	/* if YES, also look for macro parameter by this name,
		 * otherwise just macro */

{
	if (paramtoo) {
		return(resolve_mac_name(macname) == 0 ? NO : YES);
	}
	else {
		return(findMacro(macname) == 0 ? NO : YES);
	}
}


/* when the -D option is used on the command line, save the macro definition. */

void
cmdline_macro(macdef)

char *macdef;	/* MACRO=definition */

{
	char *def;


	/* separate out the macro name */
	if (*macdef == '_') {
		if (Mupmate == YES) {
			l_yyerror(0, -1, "Run > Set Options > Define Macros: macro name cannot start with underscore.");
		}
		else {
			l_yyerror(0, -1, "argument for %cD is invalid: macro name cannot start with underscore", Optch);
		}
		return;
	}
	for (def = macdef; *def != '\0'; def++) {
		if ( ! isupper(*def) && ! isdigit(*def) && *def != '_') {
			break;
		}
	}

	/* make sure has form XXX=definition */
	if (def == macdef) {
		if (Mupmate == YES) {
			l_yyerror(0, -1, "Run > Set Options > Define Macros: macro name only allowed to contain upper case letters, numbers and underscores.");
		}
		else {
			l_yyerror(0, -1, "argument for %cD is missing or wrong format", Optch);
		}
		return;
	}

	if (*def == '=') {
		*def++ = '\0';
	}
	else if (*def != '\0') {
		if (Mupmate == YES) {
			l_yyerror(0, -1, "Run > Set Options > Define Macros: macro name is invalid or missing '=' on macro definition.");
		}
		else {
			l_yyerror(0, -1, "argument for %cD had invalid name or missing '=' on macro definition", Optch);
		}
		return;
	}

	Curr_filename = "Command line argument";
	/* command line macros can never have parameters */
	(void) setup_macro(macdef, NO);

	/* copy the macro to the macro temp file */
	do {
		putc(*def, Mactmp_write);
	} while ( *def++ != '\0');
	(void) fflush(Mactmp_write);
#ifndef UNIX_LIKE_FILES
	fclose(Mactmp_write);
#endif
}


/* recursively free a list of macro parameters */

static void
free_parameters(param_p, macname, values_only)

struct MAC_PARAM *param_p;	/* what to free */
char *macname;			/* name of macro having this parameter */
int values_only;		/* if YES, just get rid of current
				 * argument values, otherwise dispose of
				 * the entire parameters list */

{
	char *mp_name;		/* internal name of macro parameter */


	if (param_p == (struct MAC_PARAM *) 0) {
		/* end of list */
		return;
	}

	/* recurse */
	free_parameters(param_p->next, macname, values_only);

	/* need to undef the internal name */
	mp_name = mkmacparm_name(macname, param_p->param_name);
	undef_macro(mp_name);
	FREE(mp_name);

	if (values_only == NO) {

		/* release space */
		if (param_p->param_name != (char *) 0) {
			FREE(param_p->param_name);
		}
		FREE(param_p);
	}
}


/* given a macro name and a parameter name, add the parameter name to the
 * list of parameters for the macro */

void
add_parameter(macname, param_name)

char *macname;
char *param_name;		/* name of parameter to add */

{
	struct MACRO *macinfo_p;	/* which macro to add to */
	struct MAC_PARAM *new_p;	/* new parameter */
	struct MAC_PARAM *param_p;	/* to walk through parameter list */


	/* get space to store info about the parameter */
	MALLOC(MAC_PARAM, new_p, 1);

	/* get the macro information to know where to attach */
	if ((macinfo_p = findMacro(macname)) == (struct MACRO *) 0) {
		pfatal("add_parameter unable to find macro %s", macname);
	}

	/* if this is first parameter, link directly to macro, otherwise
	 * to the end of the parameter linked list */
	if (macinfo_p->parameters_p == (struct MAC_PARAM *) 0) {
		macinfo_p->parameters_p = new_p;
	}
	else {
		/* walk down current parameter list */
		for (param_p = macinfo_p->parameters_p;
				param_p != (struct MAC_PARAM *) 0;
				param_p = param_p->next) {

			/* check for duplicate name */
			if (strcmp(param_name, param_p->param_name) == 0) {
				l_yyerror(Curr_filename, yylineno,
					"duplicate parameter name %s for macro %s",
					param_name, macname);
			}

			/* link onto end of list */
			if (param_p->next == (struct MAC_PARAM *) 0) {
				param_p->next = new_p;
				break;
			}
		}
	}

	/* fill in the info */
	new_p->param_name = dupstring(param_name);
	/* assume non-quoted for now */
	new_p->quote_style = QS_UNQUOTED;
	new_p->next = (struct MAC_PARAM *) 0;

	(macinfo_p->num_params)++;
}


/* given a string to duplicate, allocate space for a copy and return pointer
 * to the copy. Caller is responsible for freeing if it needs to be freed */

static char *
dupstring(string)

char *string;	/* what to duplicate */

{
	char *newstr;	/* the duplicate */

	/* get space and copy the old to new */
	MALLOCA(char, newstr, strlen(string) + 1);
	(void) strcpy(newstr, string);

	return(newstr);
}


/* save the value of a macro argument by making a macro out of it */

void
set_parm_value(macname, argbuff, argnum)

char *macname;	/* name of macro */
char *argbuff;	/* value of argument */
int argnum;	/* which argument. 1 for the first, 2 for second, etc */

{
	static struct MAC_PARAM *param_p;	/* keep track of current
			 * parameter. The first time we are called for a
			 * given macro, we look up the macro and get its
			 * first parameter. After that, we just follow
			 * the linked list of parameters */
	struct MACRO *mac_p;	/* info about macro */
	struct MACRO *argmac_p;	/* info about the parameter */
	char *mp_name;	/* pointer to malloc-ed space containing internal
			 * name of MACRO(PARAMETER) */
	char *p;	/* to copy value to macro temp file */

	if (argnum == 1) {
		/* this is the first argument, so we have to look up the
		 * macro */
		if ((mac_p = findMacro(macname)) == (struct MACRO *) 0) {
			pfatal("set_parm_value can't find macro");
		}

		/* point to head of parameters list */
		param_p = mac_p->parameters_p;
	}
	else {
		/* just advance to the next parameter */
		if (param_p != (struct MAC_PARAM *) 0) {
			param_p = param_p->next;
		}
		if (param_p == (struct MAC_PARAM *) 0) {
			/* no next parameter. Error msg is printed elsewhere,
			 * so just clean up and return */
			FREE(argbuff);
			return;
		}
	}

	/* if argbuff is null, there is no argument, which really means the
	 * argument is the null string. */
	if (argbuff == (char *) 0) {
		argbuff = dupstring("");
	}

	/* now associate the value with the parameter */
	mp_name = mkmacparm_name(macname, param_p->param_name);
	/* A parameter cannot itself have parameters */
	argmac_p = setup_macro(mp_name, NO);

	/* copy the macro to the macro temp file */
	if (param_p->quote_style & QS_UNQUOTED) {
		fprintf(Mactmp_write, "%s", argbuff);
		putc('\0', Mactmp_write);
	}

	if (param_p->quote_style & QS_QUOTED) {
		short in_string;	/* are we within double quotes? */
		short escaped;		/* did we just see a backslash? */

		/* Remember where we are stashing this quoted copy */
		argmac_p->quoted_offset = ftell(Mactmp_write);

		/* Add the initial quote */
		putc('"', Mactmp_write);

		/* Have to copy a character at a time, because we
		 * need to backslash any embedded quotes. This follows
		 * rules like the ANSI C preprocessor, except we only have
		 * to worry about strings, not character constants, because
		 * Mup doesn't have character constants. Also, we don't
		 * squeeze not-in-string white space runs to a single space,
		 * because this keeps the code simpler and there's no
		 * particular benefit in squeezing, other than perhaps
		 * saving a few bytes in the macro temp file. This code
		 * is similar to the gcc implementation of cpp. */
		in_string = escaped = NO;
		for (p = argbuff; *p != '\0'; p++) {
			if (escaped == YES) {
				escaped = NO;
			}
			else {
				if (*p == '\\') {
					escaped = YES;
				}
				if (in_string == YES) {
					if (*p == '"') {
						/* reached end of string */
						in_string = NO;
					}
				}
				else if (*p == '"') {
					/* starting a string */
					in_string = YES;
				}
			}

			/* Escape quotes always; escape backslashes
			 * when they are inside strings. */
			if (*p == '"' || (in_string == YES && *p == '\\')) {
				putc('\\', Mactmp_write);
			}
			putc(*p, Mactmp_write);
		}

		/* Add the final quote */
		putc('"', Mactmp_write);
		putc('\0', Mactmp_write);
	}
	(void) fflush(Mactmp_write);
#ifndef UNIX_LIKE_FILES
	fclose(Mactmp_write);
#endif

	/* temp space no longer needed */
	FREE(mp_name);
	FREE(argbuff);
}


/* make an internal macro name for a macro parameter name. The internal
 * name is MACRO(PARAMETER). Space for name is malloc-ed, caller must free */

static char *
mkmacparm_name(macname, param_name)

char *macname;
char *param_name;

{
	char *internal_name;

	/* add 3 for the 2 parentheses and the null */
	MALLOCA(char, internal_name,
			strlen(macname) + strlen(param_name) + 3);

	(void) sprintf(internal_name, "%s(%s)", macname, param_name);
	return(internal_name);
}


/* add a character to the current macro argument buffer. */

char *
add2argbuff(argbuff, c)

char *argbuff;		/* the argument buffer so far */
int c;			/* a character to add to the buffer */

{
	static int offset;	/* where in argbuff to put character */
	static int length;	/* how many characters we have allocated */


	if (argbuff == (char *) 0) {
		/* first time we were called, so malloc some space */
		MALLOCA(char, argbuff, MAC_ARG_SZ);
		offset = 0;
		length = MAC_ARG_SZ;
	}
	else if (offset == length - 1) {
		/* need more space */
		length += MAC_ARG_SZ;
		REALLOCA(char, argbuff, length);
	}

	/* put character in the buffer and null terminate it */
	argbuff[offset++] = (char) c;
	argbuff[offset] = '\0';

	return(argbuff);
}


/* given a macro name that might be either a macro parameter or a regular
 * macro, return pointer to the proper MACRO struct. We do this by
 * searching up the stack of macros being expanded. If the name matches
 * that of a macro parameter, use that, otherwise treat as a normal macro.
 * Return null if can't resolve */

static struct MACRO *
resolve_mac_name(macname)

char *macname;

{
	struct MACRO *mac_p;	/* macro that matches the macname */
	int i;			/* index through file stack */
	char *mp_name;		/* macro parameter internal name */
	int quoted;		/* if has quote designator */
	char *basename;		/* macname not counting quote designator */


	if ((quoted = has_quote_designator(macname)) == YES) {
		basename = macname + strlen(quote_designator);
	}
	else {
		basename = macname;
	} 
		
	/* first go up the stack of macro calls, seeing if the macro
	 * name matches the parameter name of any macro. If so, that's
	 * the correct macro to use. Failing that, we look for the macro
	 * name just as is. If that fails too, we have to admit defeat
	 * and return null. */
	for (i = 0; i <= Fstkptr; i++) {
		if (Filestack[i].mac_p != (struct MACRO *) 0) {
			/* we are expanding a macro. See if that macro
			 * has a parameter by the name we are trying to
			 * resolve. If so, that's what we want */
			mp_name = mkmacparm_name(Filestack[i].mac_p->macname,
					basename);
			mac_p = findMacro(mp_name);
			FREE(mp_name);
			if (mac_p != 0) {
				/* Eureka! We found it */
				return(mac_p);
			}
		}
	}

	/* well, guess it wasn't a macro parameter. Try treating as
	 * just an ordinary macro name */
	/* Not allowed to be a special quoted macro; you can only do that
	 * to a macro parameter */
	if (quoted == YES) {
		l_yyerror(Curr_filename, yylineno,
			"cannot use '%s' on a macro, only on a macro parameter",
			quote_designator);
		return ((struct MACRO *) 0);
	}

	return(findMacro(macname));
}


/* Return YES if macro name reference includes designator denoting
 * it is to be quoted (like usage of # in ANSI C preprocessor) */

static int
has_quote_designator(macname)

char *macname;

{
	return (strncmp(macname, quote_designator, strlen(quote_designator))
							== 0 ? YES : NO );
}


/* For "preprocessor" option, similar to the C compiler option to
 * just run the macro preprocessor, instead of the usual yacc-generated
 * yyparse(), we have this function that simply writes tokens out.
 */

void
preproc()
{
	while (yylex() != 0) {
		/* In strings, the backslashes before any embedded quotes
		 * will have been swallowed, so we have to recreate them. */
		if (yytext[0] == '"') {
			char *t;
			putchar('"');
			for (t = yytext + 1; *t != '\0'; t++) {
				if (*t == '"' && *(t+1) != '\0') {
					putchar('\\');
				}
				putchar(*t);
			}
		}
		else {
			printf("%s", yytext);
		}
	}
}
