/***********************************************************************\
 *                               XComp/2                               *
 *              Copyright (C) by Stangl Roman, 1999, 2001              *
 * This Code may be freely distributed, provided the Copyright isn't   *
 * removed, under the conditions indicated in the documentation.       *
 *                                                                     *
 * XComp.cpp    XComp/2 main code.                                     *
 *                                                                     *
\***********************************************************************/

/***********************************************************************\
 *                                                                     *
 * Dualthreaded compare design:                                        *
 *                                                                     *
 * Using the /MP commandline option performs a dualthreaded comparism, *
 * that is thread 1 is responsible reading the source file and thread  *
 * 2 is responsible reading the target file. Thread 1 then does the    *
 * actual comparism and descending to the next file.                   *
 *                                                                     *
 * Thread 1:                        Thread 2:                          *
 *                                                                     *
 * Initially Thread 1 has created   Initially Thread 2 gets SEM_2      *
 * all semaphores and owns SEM_1    immediately after thread creation  *
 * and SEM_3                                                           *
 *                                                                     *
 * :loop                            :loop                              *
 *                                  Thread 2 waits for SEM_1           *
 * Thread 1 releases SEM_1 ------->                                    *
 * Thread 2 reads source file       Thread 2 now owns SEM_1            *
 * Thread 1 waits for SEM_2         Thread 2 reads target file         *
 *                         <------- Thread 2 releases SEM_2            *
 * Thread 1 now owns SEM_2          Thread 2 waits for SEM_3           *
 * Thread 1 releases SEM_3 ------->                                    *
 * Thread 1 waits for SEM_1         Thread 2 now owns SEM_3            *
 *                         <------- Thread 2 releases SEM_1            *
 * Thread 1 now owns SEM_1          Thread 2 waits for SEM_2           *
 * Thread 1 releases SEM_2 ------->                                    *
 * Thread 1 waits for SEM_3         Thread 2 owns SEM_2                *
 *                         <------- Thread 2 releases SEM_3            *
 * Thread 1 now owns SEM_3                                             *
 * goto loop                        goto loop                          *
 *                                                                     *
 * The logic used here may look like overly complex and one may notice *
 * that 2 semaphores may be enought, but field tests show that it      *
 * won't work that way. The reason I believe is that if 1 thread       *
 * executes a DosReleaseMutexSem(), DosRequestMutexSem() while another *
 * thread waits with DosRequestMutexSem(), the first thread may        *
 * immediately release and receive ownership without the 2'nd thread   *
 * running (and thus having a chance to receive ownership) at all.     *
 * (Assume here a single semaphore). Define the DEBUG macro to see     *
 * the synchronization in action.                                      *
 *                                                                     *
 * I thought that releasing a semaphore while another thread is        *
 * blocked on the same semaphore causes the current thread to preempt  *
 * and the other thread to execute. This does not seem to be true      *
 * (nor sufficient documented in the toolkit).                         *
 *                                                                     *
\***********************************************************************/
#pragma strings(readonly)

#include    <XComp.hpp>

            XCOMP::XCOMP(int argc, char *argv[]) : iCompareBufferSize(0),
                                                   iSourcePathRootLen(0),
                                                   iTargetPathRootLen(0),
                                                   hfileSourceFile(0),
                                                   hfileTargetFile(0),
                                                   ulBytesReadSourceFile(0),
                                                   ulBytesReadTargetFile(0),
                                                   pbSourceData(0),
                                                   pbTargetData(0),
                                                   iCountProblemNone(0),
                                                   iCountProblemFindFiles(0),
                                                   iCountProblemFindDirs(0),
                                                   iCountProblemOpen(0),
                                                   iCountProblemCompare(0),
                                                   iSourceTotalRead(0),
                                                   iTargetTotalRead(0),
                                                   iSourceTotalMS(0),
                                                   iTargetTotalMS(0),
                                                   iTotalMS(0),
                                                   puProfilingSource(0),
                                                   puProfilingTarget(0),
                                                   puProfilingTotal(0),
                                                   apiretRcSource(0),
                                                   apiretRcTarget(0),
                                                   hmtxThread1(0),
                                                   hmtxThread2(0),
                                                   hmtxThread3(0),
                                                   iStatusFlag(0)
{
    TIB    *ptib=0;
    PIB    *ppib=0;
    int     iArgument;
    APIRET  apiretRc;

    memset(acExecutableFile, '\0', sizeof(acExecutableFile));
    memset(acLogFile, '\0', sizeof(acLogFile));
                                        /* Get fully qualified path to this executable */
    DosGetInfoBlocks(&ptib, &ppib);
    if(ppib!=0)
        DosQueryModuleName(ppib->pib_hmte, sizeof(acExecutableFile), acExecutableFile);
    if(strlen(acExecutableFile)==0)
        throw(EXCP(ERR_PIB, "Invalid process information block"));
                                        /* Check commandline options */
    cout << endl;
    if(argc<3)
        throw(EXCP(ERR_ARGUMENTS, "Too few commandline arguments specified"));
    if(strlen(argv[1])>CCHMAXPATH)
        throw(EXCP(ERR_SOURCEPATH, "Source path too long"));
    if(strlen(argv[2])>CCHMAXPATH)
        throw(EXCP(ERR_TARGETPATH, "Target path too long"));
    strcpy((char *)acSourcePath, argv[1]);
    strcpy((char *)acTargetPath, argv[2]);
                                        /* Check for additional commandline arguments */
    for(iArgument=3; iArgument<argc; iArgument++)
        {
        strupr(argv[iArgument]);
                                        /* Verbose */
        if((strstr(argv[iArgument], "-VERBOSE")) ||
            (strstr(argv[iArgument], "/VERBOSE")))
            iStatusFlag|=XCOMP_STATUS_VERBOSE;
                                        /* More verbose */
        if((strstr(argv[iArgument], "--VERBOSE")) ||
            (strstr(argv[iArgument], "//VERBOSE")))
            iStatusFlag|=XCOMP_STATUS_VERBOSE2;
                                        /* Line number information for non-fatal messages */
        if((strstr(argv[iArgument], "-LINE")) ||
            (strstr(argv[iArgument], "/LINE")))
            iStatusFlag|=XCOMP_STATUS_LINENUMBER;
                                        /* Multithread option */
        if((strstr(argv[iArgument], "-MP")) ||
            (strstr(argv[iArgument], "/MP")))
            {
            iStatusFlag|=XCOMP_STATUS_MP;
            if(DosCreateMutexSem(NULL, &hmtxThread1, 0, TRUE)!=NO_ERROR)
                iStatusFlag&=(~XCOMP_STATUS_MP);
            if(DosCreateMutexSem(NULL, &hmtxThread2, 0, FALSE)!=NO_ERROR)
                iStatusFlag&=(~XCOMP_STATUS_MP);
            if(DosCreateMutexSem(NULL, &hmtxThread3, 0, TRUE)!=NO_ERROR)
                iStatusFlag&=(~XCOMP_STATUS_MP);
            if(iStatusFlag & XCOMP_STATUS_MP)
                _beginthread((void(_Optlink *)(void *))xcompThread, 0, 65536, (void *)this);
            }
                                        /* Log option */
        if((strstr(argv[iArgument], "-LOG")) ||
            (strstr(argv[iArgument], "/LOG")))
            {
            iStatusFlag|=XCOMP_STATUS_LOG;
            if((argv[iArgument][4]==':') && (argv[iArgument][5]!='\0'))
                {
                strcpy(acLogFile, &argv[iArgument][5]);
                }
            else
                {
                if(getenv("XCOMP"))
                    {
                    strcpy(acLogFile, getenv("XCOMP"));
                    }
                else
                    {
                    strcpy(acLogFile, acExecutableFile);
                    strcpy(strrchr(acLogFile, '.'), ".Log");
                    }
                }
            ofstreamLogFile.open(acLogFile, ios::out);
            if(!ofstreamLogFile)
                {
                EXCP   *pexcpError=new EXCP(ERR_LOGFILEOPEN, "", __LINE__);
            
                *pexcpError << "Opening logfile " << acLogFile << " failed";
                throw(*pexcpError);
                }
            }
                                        /* No pause after miscompares option */
        if((strstr(argv[iArgument], "-/!C")) ||
            (strstr(argv[iArgument], "/!C")))
            iStatusFlag|=XCOMP_STATUS_NOPAUSEERRCOMP;
                                        /* No pause after destination file not found option */
        if((strstr(argv[iArgument], "-/!F")) ||
            (strstr(argv[iArgument], "/!F")))
            iStatusFlag|=XCOMP_STATUS_NOPAUSEERRFIND;
                                        /* Small buffer */
        if((strstr(argv[iArgument], "-TINY")) ||
            (strstr(argv[iArgument], "/TINY")))
            iStatusFlag|=XCOMP_STATUS_SMALLBUFFER;
        }
}

            XCOMP::~XCOMP(void)
{
                                        /* Report performace data */
    iTotalMS=puProfilingTotal->elapsedMilliSec();
    if((iSourceTotalMS!=0) && (iTargetTotalMS!=0) && (iTotalMS!=0))
        {
                                        /* To calculate the throughput we take the number of bytes
                                           read multiply that by 1000 and divide it by the time in
                                           milliseconds resulting in bytes/second. We then divide that
                                           by 1024 (>>10) to calculate kB out of bytes */
        cout << MSG_THROUGHPUT ": Throughput Source " << ((iSourceTotalRead>>10)*1000/iSourceTotalMS) << "kB/s, Destination " <<
            ((iTargetTotalRead>>10)*1000/iTargetTotalMS) << "kB/s, Total " <<
            (((iSourceTotalRead+iTargetTotalRead)>>10)*1000/iTotalMS) << "kB/s" << endl;
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << endl;
            ofstreamLogFile << MSG_THROUGHPUT ": Throughput Source " << ((iSourceTotalRead>>10)*1000/iSourceTotalMS) << "kB/s, Destination " <<
                ((iTargetTotalRead>>10)*1000/iTargetTotalMS) << "kB/s, Total " <<
                (((iSourceTotalRead+iTargetTotalRead)>>10)*1000/iTotalMS) << "kB/s" << endl;
            }
        } 
    if(puProfilingSource!=0)
        delete puProfilingSource;
    if(puProfilingTarget!=0)
        delete puProfilingTarget;
    if(puProfilingTotal!=0)
        delete puProfilingTotal;
    if(iCountProblemOpen!=0)
        {
        cout << MSG_PROBLEMOPEN ": " << iCountProblemOpen << " file(s) could not be opened. This may be caused by non existing or" << endl;
        cout << "locked files." << endl;
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << MSG_PROBLEMOPEN ": " << iCountProblemOpen << " file(s) could not be opened. This may be caused by non existing or" << endl;
            ofstreamLogFile << "locked files." << endl;
            }
        }
    if(iCountProblemCompare!=0)
        {
        cout << MSG_PROBLEMCOMPARE ": " << iCountProblemCompare << " file(s) did not match. You don't have an exact copy of the source" << endl;
        cout << "files with the target files. Not matching files are listed above. Please rerun\nthe copy operation!" << endl;
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << MSG_PROBLEMCOMPARE ": " << iCountProblemCompare << " file(s) did not match. You don't have an exact copy of the source" << endl;
            ofstreamLogFile << "files with the target files. Not matching files are listed above. Please rerun\nthe copy operation!" << endl;
            }
        }
    if(iCountProblemFindFiles!=0)
        {
        cout << MSG_PROBLEMFINDFILES ": " << iCountProblemFindFiles << " directory(ies) could not be opened. This may be caused by non" << endl;
        cout << "accessible directories due to ACLs (e.g. on HPFS386), an incorrect/outdated" << endl;
        cout << "file system driver (e.g CD-ROM) or file system corruption." << endl;
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << MSG_PROBLEMFINDFILES ": " << iCountProblemFindFiles << " directory(ies) could not be opened. This may be caused by non" << endl;
            ofstreamLogFile << "accessible directories due to ACLs (e.g. on HPFS386), an incorrect/outdated" << endl;
            ofstreamLogFile << "file system driver (e.g CD-ROM) or file system corruption." << endl;
            }
        }
    if(iCountProblemFindDirs!=0)
        {
        cout << MSG_PROBLEMFINDDIRS ": " << iCountProblemFindDirs << " subdirectory(ies) could not be descended into. This may be" << endl;
        cout << "caused by non accessible directories due to ACLs (e.g. on HPFS386), an" << endl;
        cout << "incorrect/outdated file system driver (e.g CD-ROM) or file system corruption." << endl;
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << MSG_PROBLEMFINDDIRS ": " << iCountProblemFindDirs << " subdirectory(ies) could not be descended into. This may be" << endl;
            ofstreamLogFile << "caused by non accessible directories due to ACLs (e.g. on HPFS386), an" << endl;
            ofstreamLogFile << "incorrect/outdated file system driver (e.g CD-ROM) or file system corruption." << endl;
            }
        }
    cout << MSG_COMPARETOTAL ": " << (iCountProblemNone-iCountProblemOpen-iCountProblemCompare) << " file(s) compared "
        "successfully, "<< iCountProblemNone << " file(s) compared totally." << endl;
    if(iStatusFlag & XCOMP_STATUS_LOG)
        {
        ofstreamLogFile << MSG_COMPARETOTAL ": " << (iCountProblemNone-iCountProblemOpen-iCountProblemCompare) << " file(s) compared "
            "successfully, "<< iCountProblemNone << " file(s) compared totally." << endl;
        }
    if(hmtxThread1!=NULLHANDLE)
        DosCloseMutexSem(hmtxThread1);
    if(hmtxThread2!=NULLHANDLE)
        DosCloseMutexSem(hmtxThread2);
    if(hmtxThread3!=NULLHANDLE)
        DosCloseMutexSem(hmtxThread3);
    if(iStatusFlag & XCOMP_STATUS_LOG)
        ofstreamLogFile.close();
}

void        XCOMP::process(void)
{
    ULONG   ulMemorySize;

                                        /* Allocate timers */
    puProfilingSource=new UProfiling();
    puProfilingTarget=new UProfiling();
    puProfilingTotal=new UProfiling();
                                        /* Get available memory size */
    iCompareBufferSize=XCOMPDATASIZE;
    if(DosQuerySysInfo(QSV_TOTPHYSMEM, QSV_TOTPHYSMEM, &ulMemorySize, sizeof(ulMemorySize))==NO_ERROR)
        {
                                        /* As the returned total physical memory for e.g. a 128MB
                                           SIMM is less than 128MB due to the memory hole between
                                           640kB and 1MB, just add 1MB (as a rule of thumb) */
        ulMemorySize+=(1<<20);
                                        /* Starting with 16MB, double buffer size whenever
                                           RAM doubles */
        if(ulMemorySize>=(1<<24))
            iCompareBufferSize<<=1;
        if(ulMemorySize>=(1<<25))
            iCompareBufferSize<<=1;
        if(ulMemorySize>=(1<<26))
            iCompareBufferSize<<=1;
        if(ulMemorySize>=(1<<27))
            iCompareBufferSize<<=1;
        if(ulMemorySize>=(1<<28))
            iCompareBufferSize<<=1;
        if(ulMemorySize>=(1<<29))
            iCompareBufferSize<<=1;
        if(ulMemorySize>=(1<<30))
            iCompareBufferSize<<=1;
        if(ulMemorySize>=(1<<31))
            iCompareBufferSize<<=1;
        }
                                        /* When requested, just use 2 64kB buffers */
    if(iStatusFlag & XCOMP_STATUS_SMALLBUFFER)
        iCompareBufferSize=(1<<16);
    pbSourceData=new BYTE[iCompareBufferSize];
    pbTargetData=new BYTE[iCompareBufferSize];
    iCountProblemFindFiles=iCountProblemFindDirs=iCountProblemOpen=iCountProblemCompare=0;
    verifySourcePath();
    verifyTargetPath();
    cout << "Comparing files qualified by " << acSourceFiles << endl;
    cout << "  at  " << acSourcePath << endl;
    cout << "with  " << acTargetPath << endl;
    cout << "using " << iCompareBufferSize << " bytes buffer size\n" << endl;
    if(iStatusFlag & XCOMP_STATUS_LOG)
        {
        time_t      timetCurrent;
        struct tm  *ptmCurrent;
        char        acTimeCurrent[81];

        ofstreamLogFile << "XCOMP/2 - The recursive file compare utility for OS/2, V2.40\n" << endl;
        ofstreamLogFile << "          (C) Roman Stangl 03, 2001 (Roman_Stangl@at.ibm.com)" << endl;
        ofstreamLogFile << "          http://www.geocities.com/SiliconValley/Pines/7885/\n" << endl;
        time(&timetCurrent);
        ptmCurrent=localtime(&timetCurrent);
        strftime(acTimeCurrent, sizeof(acTimeCurrent)-1,
            "Execution from %a. %d.%b.%Y %H:%M", ptmCurrent);
        ofstreamLogFile << acTimeCurrent << endl << endl; 
        ofstreamLogFile << "Comparing files qualified by " << acSourceFiles << endl;
        ofstreamLogFile << "  at  " << acSourcePath << endl;
        ofstreamLogFile << "with  " << acTargetPath << endl;
        ofstreamLogFile << "using " << iCompareBufferSize << " bytes buffer size\n" << endl;
        }
                                        /* Do the matching */
    puProfilingTotal->start();
    searchFiles(acSourcePath, acSourceFiles, acTargetPath);
    puProfilingTotal->stop();
                                        /* Inform thread to shutdown */
    iStatusFlag|=XCOMP_STATUS_SHUTDOWN;
    delete pbSourceData;
    delete pbTargetData;
    cout << "                                                                               " << endl;
}

void        XCOMP::usage(void)
{
    cout << "XCOMP/2 - The recursive file compare utility for OS/2, V2.40\n" << endl;
    cout << "          (C) Roman Stangl 03, 2001 (Roman_Stangl@at.ibm.com)" << endl;
    cout << "          http://www.geocities.com/SiliconValley/Pines/7885/\n" << endl;
    cout << "Use the XCOMP command to selectively compare groups of files located in two" << endl;
    cout << "directories, including all subdirectories.\n" << endl;
    cout << "Syntax:" << endl;
    cout << "  XCOMP [drive:\\|\\\\server\\][path\\]filename(s) [drive:\\|\\\\server\\]path" << endl;
    cout << "        [/MP] [/LOG[:[drive:\\|\\\\server\\][path\\]filename]] [/!C] [/!F]" << endl;
    cout << "        [/TINY] [/LINE]" << endl;
    cout << "Where:" << endl;
    cout << "  [drive:\\|\\\\server\\][path\\]filename(s)" << endl;
    cout << "                 Specifies the location and name of the source file(s) to" << endl;
    cout << "                 compare from. You may specify a fully qualified path or" << endl;
    cout << "                 UNC path." << endl;
    cout << "  [drive:\\|\\\\server\\]path" << endl;
    cout << "                 Specifies the location of the destination path to compare" << endl;
    cout << "                 with. You may specify a fully qualified path or UNC path." << endl;
    cout << "  [/MP]" << endl;
    cout << "                 Specifies that 1 thread reads the source and 1 thread reads" << endl;
    cout << "                 the target file. This improves througput when comparing from 2" << endl;
    cout << "                 different physical drives (e.g. CD-ROM and Harddisk)." << endl;
    cout << "  [/LOG[:[drive:\\|\\\\server\\][path\\]filename]]" << endl;
    cout << "                 Specifies that XCOMP/2 logs all problems into a file specified" << endl;
    cout << "                 either by this parameter, or by the XCOMP environment variable" << endl;
    cout << "                 or into XCOMP.LOG (put into the directory XCOMP/2 was" << endl;
    cout << "                 installed into) otherwise." << endl;
    cout << "  [/!C]" << endl;
    cout << "                 By default, XCOMP/2 pauses at all mismatches. Specifying this" << endl;
    cout << "                 option allows XCOMP/2 just display the mismatch and continue" << endl;
    cout << "                 the comparism without a pause (e.g. useful when using the /LOG" << endl;
    cout << "                 option or output redirection)." << endl;
    cout << "  [/!F]" << endl;
    cout << "                 By default, XCOMP/2 pauses for files in the source location" << endl;
    cout << "                 that can't be found at the target location. Specifying this" << endl;
    cout << "                 this option allows XCOMP/2 just display the miss and continue" << endl;
    cout << "                 the comparism without a pause (e.g. useful when using the" << endl;
    cout << "                 /LOG option or output redirection)." << endl;
    cout << "  [/TINY]" << endl;
    cout << "                 2 64kB buffers are used instead of a percentage of total RAM." << endl;
    cout << "  [/LINE]" << endl;
    cout << "                 Display line number information for messages (useful for e.g." << endl;
    cout << "                 debugging)" << endl;
    cout << "Returns:" << endl;
    cout << "  0              Successful completion" << endl;
    cout << "  1              Files could not be opened to compare (possibly 0-length,"  << endl;
    cout << "                 locked or not existing)" << endl;
    cout << "  2              Directories could not be opened to search for files (possibly" << endl;
    cout << "                 access right or file system problems)" << endl;
    cout << "  3              Directories could not be opened to search for directories" << endl;
    cout << "                 possibly access right or file system problems)" << endl;
    cout << "  4              A mismatch between at least 1 file was detected" << endl;
    cout << "  100+           Fatal, unrecoverable exceptions\n" << endl;
}

void        XCOMP::verifySourcePath(void)
{
    char    acSourcePathCopy[CCHMAXPATH];
    ULONG   ulDiskNum;
    ULONG   ulTemp;
    int     iUNCFilename;
    char   *pcFilename;
    APIRET  apiretRc;

                                        /* If just a drive letter was specified, add the
                                           current directory of that drive (note that
                                           DosQueryCurrentDir() does not return the
                                           drive letter) */
    if((acSourcePath[1]==':') && (acSourcePath[2]=='\0'))
        {
        ulDiskNum=(ULONG)acSourcePath[0];
        if((ulDiskNum>='a') && (ulDiskNum<='z'))
            ulDiskNum-=((ULONG)'a'-1);
        if((ulDiskNum>='A') && (ulDiskNum<='Z'))
            ulDiskNum-=((ULONG)'A'-1);
        ulTemp=sizeof(acSourcePathCopy);
        apiretRc=DosQueryCurrentDir(ulDiskNum, acSourcePathCopy, &ulTemp);
        if(apiretRc==NO_ERROR)
            {
            if(strlen(acSourcePathCopy))
                {
                strcat(acSourcePath, "\\");
                strcat(acSourcePath, acSourcePathCopy);
                }
            strcat(acSourcePath, "\\*");
            }
        }
                                        /* Check for trailing backslash, which we replace
                                           (DosQueryPathInfo() doesn't like them) */
    iSourcePathRootLen=strlen(acSourcePath);
    if(acSourcePath[iSourcePathRootLen-1]=='\\')
        strcat(acSourcePath, "*");
                                        /* If the user has specified "\", ... everything
                                           that follows must be remembered, as DosQueryPathInfo()
                                           just expands "." and ".." while keeping the rest 
                                           Q:\Test\* -> *
                                           .\* -> *
                                           \\Server\Alias\* -> *
                                           . -> n/a 
                                           .. -> n/a */
    pcFilename=strrchr(acSourcePath, '\\');
    if(pcFilename==strchr(acSourcePath, '\\')+1)
        pcFilename=0;
    if(pcFilename==0)
        pcFilename=strrchr(acSourcePath, '.');
    if(pcFilename==0)
        pcFilename=strrchr(acSourcePath, ':');
    if(pcFilename!=0)
        {
        pcFilename++;
        if(*pcFilename=='\0')
            pcFilename=0;
        }
                                        /* Now let's OS/2 try to expand the path to a fully
                                           qualified name. We then compare the expanded to the
                                           original qualified path name to guess what the user
                                           might have meant, e.g when residing at Q:\Test:
                                           . -> Q:\Test\*
                                           .\* -> Q:\Test\*
                                           Q:* -> Q:\Test\*
                                           ..\Test\* -> Q:\Test\*  
                                           Q:\Test\* -> Q:\Test\*
                                           Q:\Test\ -> Q:\Test\* */
    apiretRc=DosQueryPathInfo(acSourcePath, FIL_QUERYFULLNAME, acSourcePathCopy, sizeof(acSourcePathCopy));
    if(apiretRc!=NO_ERROR)
        {
        EXCP   *pexcpError=new EXCP(ERR_DOSQUERYSOURCEPATH, "", __LINE__);
    
        *pexcpError << "Error SYS" << apiretRc << " returned by DosQueryPathInfo()";
        throw(*pexcpError);
        }
    if(strcmpi(acSourcePath, acSourcePathCopy))
        {
        if(pcFilename==0)
            {
            pcFilename=strstr(acSourcePathCopy, acSourcePath);
            if((pcFilename!=0) && (!strcmpi(pcFilename, acSourcePath)))
                strcpy(acSourcePath, acSourcePathCopy);
            else
                {
                strcpy(acSourcePath, acSourcePathCopy);
                if(acSourcePath[strlen(acSourcePath)-1]=='\\')                    
                    strcat(acSourcePath, "*");
                else
                    strcat(acSourcePath, "\\*");
                }
            }
        else
            strcpy(acSourcePath, acSourcePathCopy);
        }
                                        /* Test for an UNC-name */
    iUNCFilename=FALSE;
    if(!strncmp(acSourcePath, "\\\\", sizeof("\\\\")-1))
        iUNCFilename=TRUE;
                                        /* Test for a backslash, as everything that follows must
                                           be the filename(s) we want to compare from */
    pcFilename=strrchr(acSourcePath, '\\');
    if(pcFilename!=0)
        {
                                        /* If we found the backslash, construct path and filename(s)
                                           info, but take care for UNC filenames */
        if((iUNCFilename==TRUE) && (pcFilename<=(acSourcePath+1)))
            {
            throw(EXCP(ERR_UNCSOURCEPATH, "Invalid UNC name specified for source path"));
            }
        pcFilename++;
        if(*pcFilename!='\0')
            strcpy(acSourceFiles, pcFilename);
        *pcFilename='\0';
        iSourcePathRootLen=strlen(acSourcePath);
        }
    else
        {
        ULONG   ulDriveNum;
        ULONG   ulDriveMap;
        ULONG   ulLength;

                                        /* We need the current drive anyway */
        apiretRc=DosQueryCurrentDisk(&ulDriveNum, &ulDriveMap);
        if(apiretRc!=NO_ERROR)
            {
            EXCP   *pexcpError=new EXCP(ERR_CURRENTDISK, "", __LINE__);
        
            *pexcpError << "Error SYS" << apiretRc << " returned by DosQueryCurrentDisk()";
            throw(*pexcpError);
            }
                                        /* If we didn't find a backslash, we may either have:
                                             - just the filename(s)
                                             - just the drive letter */
        pcFilename=strchr(acSourcePath, ':');
        if(pcFilename==acSourcePath)
            {
            throw(EXCP(ERR_SOURCEDRIVE, "Drive letter not specified for source path"));
            }
        if(pcFilename==0)
            {
                                        /* We didn't find a drive letter, so let's use the
                                           current drive and directory */
            strcpy(acSourceFiles, acSourcePath);
            strcpy(acSourcePath, "C:\\");
            ulLength=CCHMAXPATH-strlen(acSourcePath);
            acSourcePath[0]=(char)((ulDriveNum-1)+(ULONG)'a');
            apiretRc=DosQueryCurrentDir(ulDriveNum, &acSourcePath[strlen(acSourcePath)], &ulLength);
            if(apiretRc!=NO_ERROR)
                {
                EXCP   *pexcpError=new EXCP(ERR_CURRENTDIR, "", __LINE__);
            
                *pexcpError << "Error SYS" << apiretRc << " returned by DosQueryCurrentDir()";
                throw(*pexcpError);
                }
            }
        else if(*pcFilename==':')
            {
                                        /* We found a drive letter, which has to be enhanced with
                                           the current directory on that drive */
            strcpy(acSourceFiles, pcFilename+1);
            strcpy(pcFilename+1, "\\");
            ulLength=CCHMAXPATH-strlen(acSourcePath);
            ulDriveNum=((ULONG)*(pcFilename-1) | 0x00000020)-(ULONG)'a';
            ulDriveNum++;
            apiretRc=DosQueryCurrentDir(ulDriveNum, &acSourcePath[strlen(acSourcePath)], &ulLength);
            if(apiretRc!=NO_ERROR)
                {
                EXCP   *pexcpError=new EXCP(ERR_CURRENTDIR1, "", __LINE__);
            
                *pexcpError << "Error SYS" << apiretRc << " returned by DosQueryCurrentDir()";
                throw(*pexcpError);
                }
            }
        else
            {
                                        /* As we didn't find any drive letter, the user must have
                                           specified the filename(s) he's looking for, so get the
                                           current directory and the current drive */
            strcpy(acSourceFiles, acSourcePath);
            acSourcePath[0]=(char)((ulDriveNum-1)+(ULONG)'a');
            strcpy(&acSourcePath[1], ":\\");
            ulLength=CCHMAXPATH-strlen(acSourcePath);
            apiretRc=DosQueryCurrentDir(ulDriveNum, &acSourcePath[strlen(acSourcePath)], &ulLength);
            if(apiretRc!=NO_ERROR)
                {
                EXCP   *pexcpError=new EXCP(ERR_CURRENTDIR, "", __LINE__);
            
                *pexcpError << "Error SYS" << apiretRc << " returned by DosQueryCurrentDir()";
                throw(*pexcpError);
                }
            }
        iSourcePathRootLen=strlen(acSourcePath);
        if(acSourcePath[iSourcePathRootLen-1]!='\\')
            {
            strcat(acSourcePath, "\\");
            iSourcePathRootLen++;
            }
        }
                                        /* Check for trailing backslash, if nothing follows then
                                           just assume user wanted to specify "*" */
    if(!strlen(acSourceFiles))
        strcpy(acSourceFiles, "*");
}

void        XCOMP::verifyTargetPath(void)
{
    char    acTargetPathCopy[CCHMAXPATH];
    ULONG   ulDiskNum;
    ULONG   ulTemp;
    int     iUNCFilename;
    char   *pcPath;
    APIRET  apiretRc;

                                        /* If just a drive letter was specified, add the
                                           current directory of that drive (note that
                                           DosQueryCurrentDir() does not return the
                                           drive letter) */
    if((acTargetPath[1]==':') && (acTargetPath[2]=='\0'))
        {
        ulDiskNum=(ULONG)acTargetPath[0];
        if((ulDiskNum>='a') && (ulDiskNum<='z'))
            ulDiskNum-=((ULONG)'a'-1);
        if((ulDiskNum>='A') && (ulDiskNum<='Z'))
            ulDiskNum-=((ULONG)'A'-1);
        ulTemp=sizeof(acTargetPathCopy);
        apiretRc=DosQueryCurrentDir(ulDiskNum, acTargetPathCopy, &ulTemp);
        if(apiretRc==NO_ERROR)
            {
            strcat(acTargetPath, "\\");
            strcat(acTargetPath, acTargetPathCopy);
            }
        }
                                        /* Check for leading backslash, which we remove */
    iTargetPathRootLen=strlen(acTargetPath);
    if(acTargetPath[iTargetPathRootLen-1]=='\\')
        acTargetPath[iTargetPathRootLen-1]='\0';
                                        /* Now let's OS/2 try to expand the path to a fully
                                           qualified name (if possible, that is we ignore an
                                           error here as we'll find it later), and add a trailing
                                           backslash */
    apiretRc=DosQueryPathInfo(acTargetPath, FIL_QUERYFULLNAME, acTargetPath, sizeof(acTargetPath));
    iTargetPathRootLen=strlen(acTargetPath);
    if(acTargetPath[iTargetPathRootLen-1]!='\\')
        strcat(acTargetPath, "\\");
                                        /* Test for an UNC-name */
    iUNCFilename=FALSE;
    if(!strncmp(acTargetPath, "\\\\", sizeof("\\\\")-1))
        iUNCFilename=TRUE;
                                        /* Test for a backslash, as everything that follows must
                                           be the path we want to compare to */
    pcPath=strrchr(acTargetPath, '\\');
    if(pcPath!=0)
        {
                                        /* If we found the backslash, construct path info, but 
                                           take care for UNC filenames */
        if(iUNCFilename==TRUE)
            {
            if(pcPath<=(acTargetPath+1))
                {
                throw(EXCP(ERR_UNCTARGETPATH, "Invalid UNC name specified for target path"));
                }
            if((*(pcPath+1)!='\0') || (strchr(acTargetPath+2, '\\')>=pcPath))
                {
                throw(EXCP(ERR_UNCTARGETPATH1, "Invalid UNC name specified for target path"));
                }
            }
        iTargetPathRootLen=strlen(acTargetPath);
        }
    else
        {
        ULONG   ulDriveNum;
        ULONG   ulDriveMap;
        ULONG   ulLength;

                                        /* If we didn't find a backslash, we may either have:
                                             - just the drive letter */
        pcPath=strchr(acTargetPath, ':');
        if(pcPath==acTargetPath)
            {
            throw(EXCP(ERR_TARGETDRIVE, "Drive letter not specified for target path"));
            }
        if(pcPath!=NULL)
            {
                                        /* We found a drive letter, which has to be enhanced with
                                           the current directory on that drive */
            strcpy(pcPath+1, "\\");
            ulLength=CCHMAXPATH-strlen(acTargetPath);
            ulDriveNum=((ULONG)*(pcPath-1) | 0x00000020) - (ULONG)'a';
            ulDriveNum++;
            apiretRc=DosQueryCurrentDir(ulDriveNum, &acTargetPath[strlen(acTargetPath)], &ulLength);
            if(apiretRc!=NO_ERROR)
                {
                EXCP   *pexcpError=new EXCP(ERR_CURRENTDIR4, "", __LINE__);
            
                *pexcpError << "Error SYS" << apiretRc << " returned by DosQueryCurrentDir()";
                throw(*pexcpError);
                }
            }
        else
            {
            throw(EXCP(ERR_TARGETPATHINVALID, "Invalid target path specified"));
            }
        }
    iTargetPathRootLen=strlen(acTargetPath);
    if(acTargetPath[iTargetPathRootLen-1]!='\\')
        {
        strcat(acTargetPath, "\\");
        iTargetPathRootLen++;
        }
}

void        XCOMP::searchFiles(char *pcSourcePath, char *pcSourceFiles, char *pcTargetPath)
{
    char            acSourceFiles[CCHMAXPATH];
    char            acTargetFiles[CCHMAXPATH];
    char            acSourceDirectory[CCHMAXPATH];
    HDIR            hdirSourceFiles;
    FILEFINDBUF3    ffb3SourceFiles;    
    ULONG           ulffb3Length;
    ULONG           ulFindCount;
    int             iKey;
    APIRET          apiretRc;
    char           *pcSourceDirectories;
    char           *pcTargetDirectories;

    strcpy(acSourceFiles, pcSourcePath);
    strcat(acSourceFiles, pcSourceFiles);
                                        /* Say what we're going to compare */
    strcpy(acSourceDirectory, &acSourceFiles[iSourcePathRootLen]);
    pcSourceDirectories=strrchr(acSourceDirectory, '\\');
    if(pcSourceDirectories==0)
        {
        acSourceDirectory[0]='\\';
        acSourceDirectory[1]='\0';
        }
    else
        *(pcSourceDirectories+1)='\0';
    cout << acSourceDirectory << endl;
    cout.flush();
                                        /* Open files enumeration */
    hdirSourceFiles=HDIR_CREATE;
    ulffb3Length=sizeof(ffb3SourceFiles);
    ulFindCount=1;
    apiretRc=DosFindFirst(acSourceFiles, 
        &hdirSourceFiles, 
        FILE_ARCHIVED | FILE_SYSTEM | FILE_HIDDEN | FILE_READONLY,
        &ffb3SourceFiles,
        ulffb3Length,
        &ulFindCount,
        FIL_STANDARD);
    if((apiretRc==ERROR_SEEK) || (apiretRc==ERROR_PATH_NOT_FOUND))
        {
        iCountProblemFindFiles++;
        if(iStatusFlag & XCOMP_STATUS_NOPAUSEERRCOMP)
            {
            cout << MSG_DOSFINDFIRSTSOURCE ": Error SYS" << apiretRc << " returned by DosFindFirst()";
            if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                cout << " at Line " << (int)__LINE__;
            cout << endl;
            if(iStatusFlag & XCOMP_STATUS_LOG)
                {
                ofstreamLogFile << MSG_DOSFINDFIRSTSOURCE ": Error SYS" << apiretRc << " returned by DosFindFirst() at Line " << (int)__LINE__ << endl;
                }
            DosBeep(800, 200);
            }
        else
            {
            cout << MSG_DOSFINDFIRSTSOURCE ": Error SYS" << apiretRc << " returned by DosFindFirst()";
            if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                cout << " at Line " << (int)__LINE__;
            cout << endl;
            cout << "          ==> Press Enter to continue ";
            cout.flush();
            DosBeep(800, 200);
            iKey=_getch();
            if((iKey==0) || (iKey==0xE0))
                iKey=_getch();
            cout << endl;
            if(iStatusFlag & XCOMP_STATUS_LOG)
                {
                ofstreamLogFile << MSG_DOSFINDFIRSTSOURCE ": Error SYS" << apiretRc << " returned by DosFindFirst() at Line " << (int)__LINE__ << endl;
                }
            }
        }
    else if((apiretRc!=NO_ERROR) && (apiretRc!=ERROR_NO_MORE_FILES))
        {
        EXCP   *pexcpError=new EXCP(ERR_DOSFINDFIRSTSOURCE, "", __LINE__);
    
        *pexcpError << "Error SYS" << apiretRc << " returned by DosFindFirst()";
        throw(*pexcpError);
        }
    if(iStatusFlag & XCOMP_STATUS_VERBOSE2)
        {
        if(apiretRc==NO_ERROR)
            cerr << endl << "  Line " << __LINE__ << ": DosFindFirst() found source file " << ffb3SourceFiles.achName << endl;
        }
                                        /* Enumerate files */
    if(apiretRc==ERROR_PATH_NOT_FOUND)
        return;
    while(apiretRc!=ERROR_NO_MORE_FILES)
        {
        compareFiles(pcSourcePath, ffb3SourceFiles.achName, pcTargetPath);
        ulFindCount=1;
        memset(&ffb3SourceFiles, 0, ulffb3Length);
        apiretRc=DosFindNext(hdirSourceFiles,
            &ffb3SourceFiles,
            ulffb3Length,
            &ulFindCount);
        if((apiretRc==ERROR_SEEK) || (apiretRc==ERROR_PATH_NOT_FOUND))
            {
            iCountProblemFindFiles++;
            if(iStatusFlag & XCOMP_STATUS_NOPAUSEERRCOMP)
                {
                cout << MSG_DOSFINDNEXTSOURCE ": Error SYS" << apiretRc << " returned by DosFindNext()";
                if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                    cout << " at Line " << (int)__LINE__;
                cout << endl;
                if(iStatusFlag & XCOMP_STATUS_LOG)
                    {
                    ofstreamLogFile << MSG_DOSFINDNEXTSOURCE ": Error SYS" << apiretRc << " returned by DosFindNext() at Line " << (int)__LINE__ << endl;
                    }
                DosBeep(800, 200);
                }
            else
                {
                cout << MSG_DOSFINDNEXTSOURCE ": Error SYS" << apiretRc << " returned by DosFindNext()";
                if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                    cout << " at Line " << (int)__LINE__;
                cout << endl;
                cout << "          ==> Press Enter to continue ";
                cout.flush();
                DosBeep(800, 200);
                iKey=_getch();
                if((iKey==0) || (iKey==0xE0))
                    iKey=_getch();
                cout << endl;
                if(iStatusFlag & XCOMP_STATUS_LOG)
                    {
                    ofstreamLogFile << MSG_DOSFINDNEXTSOURCE ": Error SYS" << apiretRc << " returned by DosFindNext() at Line " << (int)__LINE__ << endl;
                    }
                }
            }
        else if((apiretRc!=NO_ERROR) && (apiretRc!=ERROR_NO_MORE_FILES))
            {
            EXCP   *pexcpError=new EXCP(ERR_DOSFINDNEXTSOURCE, "", __LINE__);
        
            *pexcpError << "Error SYS" << apiretRc << " returned by DosFindNext()";
            throw(*pexcpError);
            }
        if(iStatusFlag & XCOMP_STATUS_VERBOSE2)
            {
            if(apiretRc==NO_ERROR)
                cerr << "  Line " << __LINE__ << ": DosFindNext() found source file " << ffb3SourceFiles.achName << endl;
            }
        }
                                        /* Close file enumeration, if not a single file
                                           was found DosFindFirst() returns a handle which
                                           DosFindClose() finds invalid (I'm not sure if this
                                           isn't a bug) */    
    apiretRc=DosFindClose(hdirSourceFiles);
    if((apiretRc!=NO_ERROR) && (apiretRc!=ERROR_INVALID_HANDLE))
        {
        EXCP   *pexcpError=new EXCP(ERR_DOSFINDCLOSESOURCE, "", __LINE__);
    
        *pexcpError << "Error SYS" << apiretRc << " returned by DosFindClose()";
        throw(*pexcpError);
        }
                                        /* Open directories enumeration */
    strcpy(acSourceFiles, pcSourcePath);
    pcSourceDirectories=strchr(acSourceFiles, '\0');
    strcpy(acTargetFiles, pcTargetPath);
    pcTargetDirectories=strchr(acTargetFiles, '\0');
    strcat(acSourceFiles, "*");
    hdirSourceFiles=HDIR_CREATE;
    ulffb3Length=sizeof(ffb3SourceFiles);
    ulFindCount=1;
    apiretRc=DosFindFirst(acSourceFiles, 
        &hdirSourceFiles, 
        MUST_HAVE_DIRECTORY | FILE_ARCHIVED | FILE_SYSTEM | FILE_HIDDEN | FILE_READONLY,
        &ffb3SourceFiles,
        ulffb3Length,
        &ulFindCount,
        FIL_STANDARD);
    if((apiretRc==ERROR_SEEK) || (apiretRc==ERROR_PATH_NOT_FOUND))
        {
        iCountProblemFindDirs++;
        if(iStatusFlag & XCOMP_STATUS_NOPAUSEERRCOMP)
            {
            cout << MSG_DOSFINDFIRSTDIR ": Error SYS" << apiretRc << " returned by DosFindFirst()";
            if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                cout << " at Line " << (int)__LINE__;
            cout << endl;
            if(iStatusFlag & XCOMP_STATUS_LOG)
                {
                ofstreamLogFile << MSG_DOSFINDFIRSTDIR ": Error SYS" << apiretRc << " returned by DosFindFirst() at Line " << (int)__LINE__ << endl;
                }
            DosBeep(800, 200);
            }
        else
            {
            cout << MSG_DOSFINDFIRSTDIR ": Error SYS" << apiretRc << " returned by DosFindFirst()";
            if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                cout << " at Line " << (int)__LINE__;
            cout << endl;
            cout << "          ==> Press Enter to continue ";
            cout.flush();
            DosBeep(800, 200);
            iKey=_getch();
            if((iKey==0) || (iKey==0xE0))
                iKey=_getch();
            cout << endl;
            if(iStatusFlag & XCOMP_STATUS_LOG)
                {
                ofstreamLogFile << MSG_DOSFINDFIRSTDIR ": Error SYS" << apiretRc << " returned by DosFindFirst() at Line " << (int)__LINE__ << endl;
                }
            }
        }
    else if((apiretRc!=NO_ERROR) && (apiretRc!=ERROR_NO_MORE_FILES))
        {
        EXCP   *pexcpError=new EXCP(ERR_DOSFINDFIRSTDIR, "", __LINE__);
    
        *pexcpError << "Error SYS" << apiretRc << " returned by DosFindFirst()";
        throw(*pexcpError);
        }
    if(iStatusFlag & XCOMP_STATUS_VERBOSE2)
        {
        if(apiretRc==NO_ERROR)
            cerr << "  Line " << __LINE__ << ": DosFindFirst() found source dir " << ffb3SourceFiles.achName << endl;
        }
                                        /* Enumerate directories, but skip . and .. */
    while((apiretRc!=ERROR_NO_MORE_FILES) && (apiretRc!=ERROR_PATH_NOT_FOUND))
        {
        if((strcmp(ffb3SourceFiles.achName, ".")) && (strcmp(ffb3SourceFiles.achName, "..")))
            {
            strcpy(pcSourceDirectories, ffb3SourceFiles.achName);
            strcat(pcSourceDirectories, "\\");
            strcpy(pcTargetDirectories, ffb3SourceFiles.achName);
            strcat(pcTargetDirectories, "\\");
            searchFiles(acSourceFiles, pcSourceFiles, acTargetFiles);
            }
        ulFindCount=1;
        memset(&ffb3SourceFiles, 0, ulffb3Length);
        apiretRc=DosFindNext(hdirSourceFiles,
            &ffb3SourceFiles,
            ulffb3Length,
            &ulFindCount);
        if((apiretRc==ERROR_SEEK) || (apiretRc==ERROR_PATH_NOT_FOUND))
            {
            iCountProblemFindDirs++;
            if(iStatusFlag & XCOMP_STATUS_NOPAUSEERRCOMP)
                {
                cout << MSG_DOSFINDNEXTDIR ": Error SYS" << apiretRc << " returned by DosFindNext()";
                if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                    cout << " at Line " << (int)__LINE__;
                cout << endl;
                if(iStatusFlag & XCOMP_STATUS_LOG)
                    {
                    ofstreamLogFile << MSG_DOSFINDNEXTDIR ": Error SYS" << apiretRc << " returned by DosFindNext() at Line " << (int)__LINE__ << endl;
                    }
                DosBeep(800, 200);
                }
            else
                {
                cout << MSG_DOSFINDNEXTDIR ": Error SYS" << apiretRc << " returned by DosFindNext()";
                if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                    cout << " at Line " << (int)__LINE__;
                cout << endl;
                cout << "          ==> Press Enter to continue ";
                cout.flush();
                DosBeep(800, 200);
                iKey=_getch();
                if((iKey==0) || (iKey==0xE0))
                    iKey=_getch();
                cout << endl;
                if(iStatusFlag & XCOMP_STATUS_LOG)
                    {
                    ofstreamLogFile << MSG_DOSFINDNEXTDIR ": Error SYS" << apiretRc << " returned by DosFindNext() at Line " << (int)__LINE__ << endl;
                    }
                }
            }
        else if((apiretRc!=NO_ERROR) && (apiretRc!=ERROR_NO_MORE_FILES))
            {
            EXCP   *pexcpError=new EXCP(ERR_DOSFINDNEXTDIR, "", __LINE__);
        
            *pexcpError << "Error SYS" << apiretRc << " returned by DosFindNext()";
            throw(*pexcpError);
            }
        if(iStatusFlag & XCOMP_STATUS_VERBOSE2)
            {
            if(apiretRc==NO_ERROR)
                cerr << "  Line " << __LINE__ << ": DosFindNext() found source dir " << ffb3SourceFiles.achName << endl;
            }
        }
                                        /* Close file enumeration */    
    apiretRc=DosFindClose(hdirSourceFiles);
    if((apiretRc!=NO_ERROR) && (apiretRc!=ERROR_INVALID_HANDLE))
        {
        EXCP   *pexcpError=new EXCP(ERR_DOSFINDCLOSETARGET, "", __LINE__);
    
        *pexcpError << "Error SYS" << apiretRc << " returned by DosFindClose()";
        throw(*pexcpError);
        }
}

int         XCOMP::compareFiles(char *pcSourcePath, char *pcSourceFile, char *pcTargetPath)
{
    char            acSourceFile[CCHMAXPATH];
    char            acTargetFile[CCHMAXPATH];
    ULONG           ulActionSourceFile;
    ULONG           ulActionTargetFile;
    ULONG           ulErrorFound;
    FILESTATUS3     filestatus3SourceFile;
    FILESTATUS3     filestatus3TargetFile;
    ULONG           ulRetryCount;
    ULONG           ulBytesReadAccumulated;
    ULONG           ulBytesReadExpected;
    ULONG           ulPercentageCompleted;
    ULONG           ulPercentageUsed=FALSE;
    ULONG           ulFilePosSourceFile;
    ULONG           ulFilePosTargetFile;
    ULONG           ulBlockCurrent;
    ULONG           ulBlockFinished;
    int             iKey;
    APIRET          apiretRc;

    strcpy(acSourceFile, pcSourcePath);    
    strcat(acSourceFile, pcSourceFile);    
    strcpy(acTargetFile, pcTargetPath);    
    strcat(acTargetFile, pcSourceFile);   
    iCountProblemNone++;
    ulRetryCount=0;
    ulBlockFinished=0;
                                        /* Say what we're going to compare */
    cout << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << "\r";
    cout.flush();
RetryReadingFromBeginning:
    ulBlockCurrent=0;
    ulErrorFound=FALSE;
    if(iStatusFlag & XCOMP_STATUS_VERBOSE)
        {
        cout << endl;
        cerr << "  Line " << __LINE__ << ": Opening source file " << acSourceFile << endl;
        }
    apiretRcSource=DosOpen(acSourceFile,
        &hfileSourceFile,
        &ulActionSourceFile,
        0,
        FILE_NORMAL,
        OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
        OPEN_FLAGS_FAIL_ON_ERROR | OPEN_FLAGS_SEQUENTIAL | OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY,
        0);
    if(apiretRcSource!=NO_ERROR)
        {
        ulErrorFound=TRUE;
        iCountProblemOpen++;
        cout << endl << MSG_DOSOPENSOURCE ": Error SYS" << apiretRcSource << " opening source file returned by DosOpen()";
        if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
            cout << " at Line " << (int)__LINE__;
        cout << endl;
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
            ofstreamLogFile << endl << MSG_DOSOPENSOURCE ": Error SYS" << apiretRcSource << " opening source file returned by DosOpen() at Line " << (int)__LINE__ << endl;
            }
        DosClose(hfileSourceFile);
        return(apiretRcSource);
        }
    if(iStatusFlag & XCOMP_STATUS_VERBOSE)
        {
        cerr << "  Line " << __LINE__ << ": Opening target file " << acTargetFile << endl;
        }
    apiretRcTarget=DosOpen(acTargetFile,
        &hfileTargetFile,
        &ulActionTargetFile,
        0,
        FILE_NORMAL,
        OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
        OPEN_FLAGS_FAIL_ON_ERROR | OPEN_FLAGS_SEQUENTIAL | OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY,
        0);
    if(apiretRcTarget!=NO_ERROR)
        {
        ulErrorFound=TRUE;
        iCountProblemOpen++;
        if((apiretRcTarget==ERROR_OPEN_FAILED) ||
            (apiretRcTarget==ERROR_INVALID_NAME) ||
            (apiretRcTarget==ERROR_FILENAME_EXCED_RANGE))
            {
            cout << endl << MSG_DOSOPENTARGET ": Error SYS" << apiretRcTarget << " target file not found by DosOpen()";
            if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                cout << " at Line " << (int)__LINE__;
            cout << endl;
            }
        else
            {
            cout << endl << MSG_DOSOPENTARGET ": Error SYS" << apiretRcTarget << " opening target file returned by DosOpen()";
            if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                cout << " at Line " << (int)__LINE__;
            cout << endl;
            }
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
            if((apiretRcTarget==ERROR_OPEN_FAILED) ||
                (apiretRcTarget==ERROR_INVALID_NAME) ||
                (apiretRcTarget==ERROR_FILENAME_EXCED_RANGE))
                {
                ofstreamLogFile << MSG_DOSOPENTARGET ": Error SYS" << apiretRcTarget << " target file not found by DosOpen() at Line " << (int)__LINE__ << endl;
                }
            else
                {
                ofstreamLogFile << MSG_DOSOPENTARGET ": Error SYS" << apiretRcTarget << " opening target file returned by DosOpen() at Line " << (int)__LINE__ << endl;
                }
            }
        if(!(iStatusFlag & XCOMP_STATUS_NOPAUSEERRFIND))
            {
            cout << "          ==> Press Enter to continue ";
            cout.flush();
            DosBeep(800, 200);
            iKey=_getch();
            if((iKey==0) || (iKey==0xE0))
                iKey=_getch();
            cout << endl;
            }
        DosClose(hfileSourceFile);
        DosClose(hfileTargetFile);
        return(apiretRcTarget);
        }
    else
        {
                                        /* Query soure file size */
        apiretRcSource=DosQueryFileInfo(hfileSourceFile,
            FIL_STANDARD,
            &filestatus3SourceFile,
            sizeof(filestatus3SourceFile));
        if(iStatusFlag & XCOMP_STATUS_VERBOSE)
            {
            if(apiretRcSource!=NO_ERROR)
                {
                cerr << "  Line " << __LINE__ << "Error SYS" << apiretRcSource << " during source DosQueryFileInfo()  at Line " << (int)__LINE__ << endl;
                }
            else if(iStatusFlag & XCOMP_STATUS_VERBOSE2)
                {
                cerr << "  Line " << __LINE__ << ": DosQueryFileInfo() found source " << filestatus3SourceFile.cbFile << " bytes in size" << endl;
                }
            apiretRcTarget=DosQueryFileInfo(hfileTargetFile,
                FIL_STANDARD,
                &filestatus3TargetFile,
                sizeof(filestatus3TargetFile));
            if(apiretRcTarget!=NO_ERROR)
                {
                cerr << "  Line " << __LINE__ << "Error SYS" << apiretRcTarget << " during target DosQueryFileInfo()  at Line " << (int)__LINE__ << endl;
                }
            else if(iStatusFlag & XCOMP_STATUS_VERBOSE2)
                {
                cerr << "  Line " << __LINE__ << ": DosQueryFileInfo() found target " << filestatus3TargetFile.cbFile << " bytes in size" << endl;
                }
            }
                                        /* Initialize */
        ulBytesReadAccumulated=0;
        ulBytesReadExpected=0;
        ulPercentageCompleted=0;
                                        /* Calculate one percent of the file's size (if it
                                           has at least 100 bytes) */
        if(filestatus3SourceFile.cbFile>=100);
            ulBytesReadExpected=(filestatus3SourceFile.cbFile/100);
                                        /* If file is larger than buffer, start displaying a 
                                           percentage base progress indicator, else just
                                           display the filename */
        if(filestatus3SourceFile.cbFile>iCompareBufferSize)
            {
            ulPercentageUsed=TRUE;
            cout << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << " (" << ulPercentageCompleted << "% done)";
            }
        else
            cout << &pcSourcePath[iSourcePathRootLen] << pcSourceFile;
        if(ulRetryCount>0)
            cout << " Retry " << ulRetryCount;
        cout << "  \r";
        cout.flush();
                                        /* Loop until all blocks of the files have been
                                           compared */
        while (TRUE)
            {
            ulBytesReadSourceFile=ulBytesReadTargetFile=0;
                                        /* Consistency check by comparing the file pointer
                                           position of both files */
            apiretRc=DosSetFilePtr(hfileSourceFile, 0L, FILE_CURRENT, &ulFilePosSourceFile);
            if(apiretRc!=NO_ERROR)
                {
                EXCP   *pexcpError=new EXCP(ERR_DOSFILEPTRSOURCE, "", __LINE__);
            
                *pexcpError << "Error SYS" << apiretRc << " returned by DosSetFilePtr()";
                throw(*pexcpError);
                }
            apiretRc=DosSetFilePtr(hfileTargetFile, 0L, FILE_CURRENT, &ulFilePosTargetFile);
            if(apiretRc!=NO_ERROR)
                {
                EXCP   *pexcpError=new EXCP(ERR_DOSFILEPTRTARGET, "", __LINE__);
            
                *pexcpError << "Error SYS" << apiretRc << " returned by DosSetFilePtr()";
                throw(*pexcpError);
                }
            if(ulFilePosSourceFile!=ulFilePosTargetFile)
                {
                ulErrorFound=TRUE;
                iCountProblemCompare++;
                if(iStatusFlag & XCOMP_STATUS_NOPAUSEERRCOMP)
                    {
                    cout << endl << MSG_INTERNALERROR ": Internal error (" << ulFilePosSourceFile << ":" << ulFilePosTargetFile <<")";
                    if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                        cout << " at Line " << (int)__LINE__;
                    cout << endl;
                    if(iStatusFlag & XCOMP_STATUS_LOG)
                        {
                        ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
                        ofstreamLogFile << MSG_INTERNALERROR ": Internal error (" << ulFilePosSourceFile << ":" << ulFilePosTargetFile <<") at Line " << (int)__LINE__ << endl;
                        }
                    DosBeep(800, 200);
                    }
                else
                    {
                    cout << endl << MSG_INTERNALERROR ": Internal error (" << ulFilePosSourceFile << ":" << ulFilePosTargetFile <<")";
                    if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                        cout << " at Line " << (int)__LINE__;
                    cout << endl;
                    cout << "          ==> Press Enter to continue ";
                    cout.flush();
                    DosBeep(800, 200);
                    iKey=_getch();
                    if((iKey==0) || (iKey==0xE0))
                        iKey=_getch();
                    cout << endl;
                    if(iStatusFlag & XCOMP_STATUS_LOG)
                        {
                        ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
                        ofstreamLogFile << MSG_INTERNALERROR ": Internal error (" << ulFilePosSourceFile << ":" << ulFilePosTargetFile <<") at Line " << (int)__LINE__ << endl;
                        }
                    }
                break;
                }
                                        /* Read iCompareBufferSize sized blocks of data from both
                                           the source and target file and compare them. If
                                           iCompareBufferSize bytes couldn't be read anymore, we
                                           have read the last block and can exit the loop */
            if(iStatusFlag & XCOMP_STATUS_MP)
                {
#ifdef  DEBUG
                {
                ULONG   ulTemp;

                DosWrite(1, "\r\n", sizeof("\r\n")-1, &ulTemp);
                DosWrite(1, "0...", sizeof("0...")-1, &ulTemp);
                DosResetBuffer(1);
                }
#endif  /* DEBUG */
                                        /* If we're running multithreaded, release sempahore
                                           to let thread read target file while we read the
                                           source file and wait until the thread has finished */
                apiretRc=DosReleaseMutexSem(hmtxThread1);
                apiretRcSource=readSourceFile();
                if(iStatusFlag & XCOMP_STATUS_VERBOSE2)
                    DosBeep(500, 10);
#ifdef  DEBUG
                {
                ULONG   ulTemp;

                DosWrite(1, "1...", sizeof("1...")-1, &ulTemp);
                DosResetBuffer(1);
                }
#endif  /* DEBUG */
                do  {
                    apiretRc=DosRequestMutexSem(hmtxThread2, 1000);
                    } while(apiretRc!=NO_ERROR);
#ifdef  DEBUG
                {
                ULONG   ulTemp;

                DosWrite(1, "2...", sizeof("2...")-1, &ulTemp);
                DosResetBuffer(1);
                }
#endif  /* DEBUG */
                apiretRc=DosReleaseMutexSem(hmtxThread3);
                do  {
                    apiretRc=DosRequestMutexSem(hmtxThread1, 1000);
                    } while(apiretRc!=NO_ERROR);
#ifdef  DEBUG
                {
                ULONG   ulTemp;

                DosWrite(1, "3...", sizeof("3...")-1, &ulTemp);
                DosResetBuffer(1);
                }
#endif  /* DEBUG */
                apiretRc=DosReleaseMutexSem(hmtxThread2);
                do  {
                    apiretRc=DosRequestMutexSem(hmtxThread3, 1000);
                    } while(apiretRc!=NO_ERROR);
#ifdef  DEBUG
                {
                ULONG   ulTemp;

                DosWrite(1, "4...", sizeof("4...")-1, &ulTemp);
                DosWrite(1, "\r\n", sizeof("\r\n")-1, &ulTemp);
                DosResetBuffer(1);
                }
#endif  /* DEBUG */
                }
            else
                {
                                        /* If we're running singlethreaded, read source and then
                                           target file */
                apiretRcSource=readSourceFile();
                apiretRcTarget=readTargetFile();
                }
            if(apiretRcSource!=NO_ERROR)
                {
                ulErrorFound=TRUE;
                cout << endl << MSG_READSOURCE ": Error SYS" << apiretRcSource << " reading source file returned by DosRead()";
                if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                    cout << " at Line " << (int)__LINE__;
                cout << endl;
                if(iStatusFlag & XCOMP_STATUS_LOG)
                    {
                    ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
                    ofstreamLogFile << MSG_READSOURCE ": Error SYS" << apiretRcSource << " reading source file returned by DosRead() at Line " << (int)__LINE__ << endl;
                    }
                break;
                }
            else
                {
                iSourceTotalRead+=ulBytesReadSourceFile;
                iSourceTotalMS+=puProfilingSource->elapsedMilliSec();
                }
            if(apiretRcTarget!=NO_ERROR)
                {
                ulErrorFound=TRUE;
                cout << endl << MSG_READTARGET ": Error SYS" << apiretRcTarget << " reading target file returned by DosRead()";
                if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                    cout << " at Line " << (int)__LINE__;
                cout << endl;
                if(iStatusFlag & XCOMP_STATUS_LOG)
                    {
                    ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
                    ofstreamLogFile << MSG_READTARGET ": Error SYS" << apiretRcTarget << " reading target file returned by DosRead() at Line " << (int)__LINE__ << endl;
                    }
                break;
                }
            else
                {
                iTargetTotalRead+=ulBytesReadTargetFile;
                iTargetTotalMS+=puProfilingTarget->elapsedMilliSec();
                }
            if(ulBytesReadSourceFile!=ulBytesReadTargetFile)
                {
#ifdef  DEBUG
                if(ulRetryCount<=RETRY_COUNT)
                    {
                    cout << endl << MSG_RETRYCOUNT ": Retry number " << ulRetryCount << endl;
                    cout << "          ==> Press Enter to continue ";
                    cout.flush();
                    DosBeep(800, 200);
                    iKey=_getch();
                    if((iKey==0) || (iKey==0xE0))
                        iKey=_getch();
                    cout << endl;
                    }
#endif  /* DEBUG */
                ulErrorFound=TRUE;
                if(ulRetryCount<RETRY_COUNT)
                    break;
                iCountProblemCompare++;
                if(iStatusFlag & XCOMP_STATUS_NOPAUSEERRCOMP)
                    {
                    cout << endl << MSG_COMPARESTATS ": Source (" << ulBytesReadSourceFile << ") and target (" << ulBytesReadTargetFile << ") sizes differ";
                    if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                        cout << " at Line " << (int)__LINE__;
                    cout << endl;
                    if(iStatusFlag & XCOMP_STATUS_LOG)
                        {
                        ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
                        ofstreamLogFile << MSG_COMPARESTATS ": Source (" << ulBytesReadSourceFile << ") and target (" << ulBytesReadTargetFile << ") sizes differ at Line " << (int)__LINE__ << endl;
                        }
                    DosBeep(800, 200);
                    }
                else
                    {
                    cout << endl << MSG_COMPARESTATS ": Source (" << ulBytesReadSourceFile << ") and target (" << ulBytesReadTargetFile << ") sizes differ";
                    if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                        cout << " at Line " << (int)__LINE__;
                    cout << endl;
                    cout << "          ==> Press Enter to continue ";
                    cout.flush();
                    DosBeep(800, 200);
                    iKey=_getch();
                    if((iKey==0) || (iKey==0xE0))
                        iKey=_getch();
                    cout << endl;
                    if(iStatusFlag & XCOMP_STATUS_LOG)
                        {
                        ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
                        ofstreamLogFile << MSG_COMPARESTATS ": Source (" << ulBytesReadSourceFile << ") and target (" << ulBytesReadTargetFile << ") sizes differ at Line " << (int)__LINE__ << endl;
                        }
                    }
                break;
                }
                                        /* Compare what we have read, and in
                                           case of an error, cause a retry again */
            if(memcmp(pbSourceData, pbTargetData, ulBytesReadSourceFile))
                {
#ifdef  DEBUG
                if(ulRetryCount<=RETRY_COUNT)
                    {
                    cout << endl << MSG_RETRYEXCEEDED ": Retry number " << ulRetryCount << endl;
                    cout << "          ==> Press Enter to continue ";
                    cout.flush();
                    DosBeep(800, 200);
                    iKey=_getch();
                    if((iKey==0) || (iKey==0xE0))
                        iKey=_getch();
                    cout << endl;
                    }
#endif  /* DEBUG */
                ulErrorFound=TRUE;
                if(ulRetryCount<RETRY_COUNT)
                    break;
                                        /* If we retried and still haven't found
                                           a match, then we have an unrecoverable
                                           miscompare */
                iCountProblemCompare++;
                if(iStatusFlag & XCOMP_STATUS_NOPAUSEERRCOMP)
                    {
                    cout << endl << MSG_DIFFERENCE ": Contents of source and target file differs";
                    if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                        cout << " at Line " << (int)__LINE__;
                    cout << endl;
                    if(iStatusFlag & XCOMP_STATUS_LOG)
                        {
                        ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
                        ofstreamLogFile << MSG_DIFFERENCE ": Contents of source and target file differs at Line " << (int)__LINE__ << endl;
                        }
                    DosBeep(800, 200);
                    }
                else
                    {
                    cout << endl << MSG_DIFFERENCE ": Contents of source and target file differs";
                    if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
                        cout << " at Line " << (int)__LINE__;
                    cout << endl;
                    cout << "          ==> Press Enter to continue ";
                    cout.flush();
                    DosBeep(800, 200);
                    iKey=_getch();
                    if((iKey==0) || (iKey==0xE0))
                        iKey=_getch();
                    cout << endl;
                    if(iStatusFlag & XCOMP_STATUS_LOG)
                        {
                        ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
                        ofstreamLogFile << MSG_DIFFERENCE ": Contents of source and target file differs at Line " << (int)__LINE__ << endl;
                        }
                    }
                break;
                }
            if(ulBytesReadSourceFile!=iCompareBufferSize)
                break;
            ulBytesReadAccumulated+=ulBytesReadSourceFile;
            if(ulBytesReadExpected!=0)
                {
                ulPercentageCompleted=ulBytesReadAccumulated/ulBytesReadExpected;
                if(ulPercentageCompleted<100)
                    cout << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << " (" << ulPercentageCompleted << "% done)";
                if(ulRetryCount>0)
                    cout << " Retry " << ulRetryCount;
                cout << "  \r";
                cout.flush();
                }
                                        /* We've successfully read the last block
                                           so reset retry counter so that next block
                                           starts again with the first retry.
                                           However, if a previous block that we've already
                                           read successfully will now fail, that fail will
                                           increment the retry counter (and thus may 
                                           overflow the retry counter then), as we assume
                                           anything we have already read successful must
                                           still be readable */
            if(ulBlockCurrent==ulBlockFinished)
                ulRetryCount=0;
                                        /* We count the blocks successfully read
                                           (even when a retries were required) */
            if(ulBlockCurrent==ulBlockFinished)
                ulBlockFinished++;
            ulBlockCurrent++;
            }
                                        /* Blank out completion percentage and retry count */
        if(ulErrorFound==FALSE) 
            {
            if(!((iStatusFlag & XCOMP_STATUS_VERBOSE) && (ulPercentageUsed==FALSE)))
                cout << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << "                    " << endl;
            cout.flush();
                                        /* We're done, thus set retry count to
                                           allow exiting this file */
            ulRetryCount=RETRY_COUNT;
            }
        }
    apiretRcSource=DosClose(hfileSourceFile);
    if(apiretRcSource!=NO_ERROR)
        {
        cout << MSG_DOSCLOSESOURCE ": Error SYS" << apiretRcSource << " closing source file returned by DosClose()";
        if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
            cout << " at Line " << (int)__LINE__;
        cout << endl;
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
            ofstreamLogFile << MSG_DOSCLOSESOURCE ": Error SYS" << apiretRcSource << " closing source file returned by DosClose() at Line " << (int)__LINE__ << endl;
            }
        }
    apiretRcTarget=DosClose(hfileTargetFile);
    if(apiretRcTarget!=NO_ERROR)
        {
        cout << MSG_DOSCLOSETARGET ": Error SYS" << apiretRcTarget << " closing target file returned by DosClose()";
        if(iStatusFlag & XCOMP_STATUS_LINENUMBER)
            cout << " at Line " << (int)__LINE__;
        cout << endl;
        if(iStatusFlag & XCOMP_STATUS_LOG)
            {
            ofstreamLogFile << &pcSourcePath[iSourcePathRootLen] << pcSourceFile << endl;
            ofstreamLogFile << MSG_DOSCLOSETARGET ": Error SYS" << apiretRcTarget << " closing target file returned by DosClose() at Line " << (int)__LINE__ << endl;
            }
        }
                                        /* Check for retry required */
    if(ulRetryCount<RETRY_COUNT)
        {
        ulRetryCount++;
        goto RetryReadingFromBeginning;
        }    
    return(NO_ERROR);
}

APIRET      XCOMP::readSourceFile(void)
{
    APIRET          apiretRc=NO_ERROR;

                                        /* Read a block from the source file and
                                           profile that */
    puProfilingSource->start();
    apiretRc=DosRead(hfileSourceFile,
        pbSourceData,
        iCompareBufferSize,
        &ulBytesReadSourceFile);
    puProfilingSource->stop();
    return(apiretRc);
}

APIRET      XCOMP::readTargetFile(void)
{
    APIRET          apiretRc=NO_ERROR;

                                        /* Read a block from the target file and
                                           profile that */
    puProfilingTarget->start();
    apiretRc=DosRead(hfileTargetFile,
        pbTargetData,
        iCompareBufferSize,
        &ulBytesReadTargetFile);
    puProfilingTarget->stop();
    return(apiretRc);
}


void        XCOMP::processThread(void)
{
    APIRET          apiretRc;

                                        /* Request our semaphore */
    do  {
        apiretRc=DosRequestMutexSem(hmtxThread2, 1000);
        } while((apiretRc!=NO_ERROR) && !(iStatusFlag & XCOMP_STATUS_SHUTDOWN));
                                        /* Keep thread running as long as XCOMP is
                                           running */
    while(!(iStatusFlag & XCOMP_STATUS_SHUTDOWN))
        {
                                        /* Wait until the semaphore is posted or 1
                                           timeout. In the first case we can read the
                                           target file, in the second we just continue
                                           the loop */  
        apiretRc=DosRequestMutexSem(hmtxThread1, 1000);
        if(apiretRc==NO_ERROR)
            {
#ifdef  DEBUG
            {
            ULONG   ulTemp;

            DosWrite(1, "A...", sizeof("A...")-1, &ulTemp);
            DosResetBuffer(1);
            }
#endif  /* DEBUG */
                                        /* If semaphore was posted, the source file is
                                           just going to be read and we are going to read
                                           the target file. Once we finish, release the
                                           semaphore so that the source and target buffers
                                           can be compared */
            apiretRcTarget=readTargetFile();
            if(iStatusFlag & XCOMP_STATUS_VERBOSE2)
                DosBeep(500, 10);
            apiretRc=DosReleaseMutexSem(hmtxThread2);
            do  {
                apiretRc=DosRequestMutexSem(hmtxThread3, 1000);
                } while((apiretRc!=NO_ERROR) && !(iStatusFlag & XCOMP_STATUS_SHUTDOWN));
#ifdef  DEBUG
            {
            ULONG   ulTemp;

            DosWrite(1, "B...", sizeof("B...")-1, &ulTemp);
            DosResetBuffer(1);
            }
#endif  /* DEBUG */
            apiretRc=DosReleaseMutexSem(hmtxThread1);
            do  {
                apiretRc=DosRequestMutexSem(hmtxThread2, 1000);
                } while((apiretRc!=NO_ERROR) && !(iStatusFlag & XCOMP_STATUS_SHUTDOWN));
#ifdef  DEBUG
            {
            ULONG   ulTemp;

            DosWrite(1, "C...", sizeof("C...")-1, &ulTemp);
            DosResetBuffer(1);
            }
#endif  /* DEBUG */
            apiretRc=DosReleaseMutexSem(hmtxThread3);
            }
        else
            {
            continue;
            }
        }
}

int         XCOMP::getProblemFindFilesCount(void)
{
    return(iCountProblemFindFiles);
}

int         XCOMP::getProblemFindDirsCount(void)
{
    return(iCountProblemFindDirs);
}

int         XCOMP::getProblemOpenCount(void)
{
    return(iCountProblemOpen);
}

int         XCOMP::getProblemCompareCount(void)
{
    return(iCountProblemCompare);
}


void        _Optlink xcompThread(XCOMP *pXCOMP)
{
    pXCOMP->processThread();
    _endthread();
}

int     main(int argc, char *argv[])
{
    int     iReturnCode=0;
    XCOMP  *pxcompInstance=0;

    try
        {
        pxcompInstance=new XCOMP(argc, argv);
        try
            {
            pxcompInstance->process();
            }
        catch(EXCP& rexcpError)
            {
            rexcpError.print();
            DosBeep(800, 200);
            cout << endl;
            pxcompInstance->usage();
            iReturnCode=rexcpError.getError();
            }
                                        /* If no fatal exception happened, set return code
                                           depending on criticalness of recoverable errors */
        if((iReturnCode==0) && (pxcompInstance->getProblemOpenCount()))
            iReturnCode=1;
        if(pxcompInstance->getProblemFindFilesCount())
            iReturnCode=2;
        if(pxcompInstance->getProblemFindDirsCount())
            iReturnCode=3;
        if(pxcompInstance->getProblemCompareCount())
            iReturnCode=4;
        }
    catch(EXCP& rexcpError)
        {
        pxcompInstance->usage();
        DosBeep(800, 200);
        rexcpError.print();
        iReturnCode=rexcpError.getError();
        }
    delete pxcompInstance;
                                        /* Inform user we're done as it makes sense
                                           to run XCOMP/2 in the background */
    DosBeep(800, 200);
    DosSleep(25);
    DosBeep(1200, 200);
    DosSleep(25);
    DosBeep(800, 200);
    DosSleep(25);
    return(iReturnCode);
}

