Unit TextSearchQuery;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

// Encapsulates a parsed search query.

uses
  Classes, SysUtils;

Type
  ESearchSyntaxError = class( Exception )
  end;

  TSearchTermCombineMethod = ( cmOr, cmAnd, cmNot );

  TSearchTerm = class
    Text: string;
    CombineMethod: TSearchTermCombineMethod;
  end;

  TTextSearchQuery = class
  protected
    Terms: TList;
    function GetTerm( Index: longint ): TSearchTerm;
    function GetTermCount: longint;
  public
    constructor Create( SearchString: string );
    destructor Destroy; override;

    property Term[ Index: longint ]: TSearchTerm read GetTerm;
    property TermCount: longint read GetTermCount;
  end;

Implementation

uses
  ACLStringUtility, ACLUtility, Dialogs;

constructor TTextSearchQuery.Create( SearchString: string );
var
  SearchWord: string;
  RemainingSearchString: string;
  CombineMethod: TSearchTermCombineMethod;
  Term: TSearchTerm;
begin
  Terms := TList.Create;
  try
    RemainingSearchString := Uppercase( SearchString );
    while RemainingSearchString <> '' do
    begin
      SearchWord := ExtractNextValue( RemainingSearchString, ' ' );

      // Check for modifiers + (word must be matched)
      // and - (word must not be matched)
      case SearchWord[ 1 ] of
       '+':
         CombineMethod := cmAnd;
       '-':
         CombineMethod := cmNot;
       else
         CombineMethod := cmOr;
      end;
      if CombineMethod <> cmOr then
      begin
        // delete + or -
        if Length( SearchWord ) = 1 then
          raise ESearchSyntaxError.Create( 'No search word given after "'
                                           + SearchWord + '" before "'
                                           + RemainingSearchString
                                           + '"' );
        Delete( SearchWord, 1, 1 );
      end;

      Term := TSearchTerm.Create;
      Term.Text := SearchWord;
      Term.CombineMethod := CombineMethod;
      Terms.Add( Term );
    end;
  except
    Destroy; // clean up
    raise; // reraise exception
  end;
end;

destructor TTextSearchQuery.Destroy;
begin
  DestroyListObjects( Terms );
  Terms.Destroy;
end;

function TTextSearchQuery.GetTerm( index: longint ): TSearchTerm;
begin
  Result := Terms[ Index ];
end;

function TTextSearchQuery.GetTermCount: longint;
begin
  Result := Terms.Count;
end;

Initialization
End.
