Unit SettingsUnit;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

// Defines settings (options) in a record and contains functions
// for loading and saving them to ini file.

Uses
  Graphics, Forms,Classes;

Const
  ContentsBackgroundColorIndex          = 0;
  ContentsTextColorIndex                = 1;
  ContentsLinesColorIndex               = 2;
  IndexBackgroundColorIndex             = 3;
  IndexTextColorIndex                   = 4;
  SearchBackgroundColorIndex            = 5;
  SearchTextColorIndex                  = 6;
  NotesBackgroundColorIndex             = 7;
  NotesTextColorIndex                   = 8;
  TopicBackgroundColorIndex             = 9;

  clLightYellow = $ffffc0;
  clLightBlue = $e0e0ff;
  clLightCyan = $c0ffff;
  clLightGreen = $e0ffe0;

  DefaultColors: array[ 0..9 ] of TColor
   = ( clLightCyan,
       clBlack,
       clBlue,
       clLightGreen,
       clBlack,
       clLightBlue,
       clBlack,
       clWhite,
       clBlack,
       clWhite );

Type
  TIndexStyle = ( isFileOnly, isAlphabetical, isFull );

  TSettings= record
    MRUList: TStringList;

    ShowHints: boolean;

    // Frame split fractions (0.2..0.8)
    ViewSplit: real;

    // COlours
    Colors: array[ 0..9 ] of TColor;

    NormalFont: TFont;
    FixedFont: TFont;

    IndexStyle: TIndexStyle;

    SmoothScrolling: boolean;
    UseOriginalDialogs: boolean;

    BackgroundImageFilename: string;
  end;

  Procedure WriteWindowPos( X, Y, width, height: longint;
                            Maximised: boolean );
  Procedure ReadWindowPos( Var X, Y, width, height: longint;
                           Var Maximised: boolean );
  Procedure LoadSettings;
  Procedure SaveSettings;

Var
  Settings: TSettings;

Implementation

Uses
  SysUtils, IniFiles, Dos, Forms,
  ACLFileUtility, ACLUtility;

Const
  IniFileName = 'NewView.ini';
  GeneralSection = 'General';
  FontsSection = 'Fonts';
  ColoursSection = 'Colours';
  WindowSection = 'Window';
  MRUSection = 'RecentFiles';

  DefaultWidth = 620;
  DefaultHeight = 460;

Function IniFilePath: string;
Var
  Dir: string;
  Name, Ext: string;
  UserIniFile: string;
Begin
 // Put it in the same place as the application
  UserIniFile:= GetApplicationDir;
  // Note: do NOT combine these two steps. Caused an obscure bug!
  FSplit( UserIniFile, Dir, Name, Ext );
  Result:= AddSlash( Dir ) + IniFileName;
End;

Procedure LoadSettings;
Var
  IniFile: TAsciiIniFile;
  i: longint;
  MRUCount: longint;
  AFile: string;
  IndexStyleString: string;
Begin
  IniFile:= TAsciiIniFile.Create( IniFilePath );
  with IniFile do
  begin
    with Settings do
    begin
      ShowHints:= ReadBool( GeneralSection, 'ShowHints', true );

      // Read split points, as units of 0.1%
      ViewSplit:= ReadInteger( GeneralSection, 'ViewSplit', 800 ) / 1000;

      //Colours
      for i:= 0 to High( Colors ) do
        Colors[ i ]:= ReadInteger( ColoursSection,
                                   'Color' + IntToStr( i ),
                                   DefaultColors[ i ] );
      MRUCount:= ReadInteger( MRUSection,
                              'Count',
                              0 );
      for i:= 0 to MRUCount -1 do
      begin
        AFile:= ReadString( MRUSection,
                            'File' + IntToStr( i ),
                            '' );
        if AFile <> '' then
          MRUList.Add( AFile );
      end;

      NormalFont:= Screen.GetFontFromPointSize(
        ReadString( FontsSection, 'NormalFont', 'Helv' ),
        ReadInteger( FontsSection, 'NormalFontSize', 8 ) );
      FixedFont:= Screen.GetFontFromPointSize(
        ReadString( FontsSection, 'FixedFont', 'Courier' ),
        ReadInteger( FontsSection, 'FixedFontSize', 8 ) );

      IndexStyleString:= ReadString( GeneralSection, 'IndexStyle', 'Full' );
      if IndexStyleString = 'FileOnly' then
        IndexStyle:= isFileOnly
      else if IndexStyleString = 'Alphabetical' then
        IndexStyle:= isAlphabetical
      else
        IndexStyle:= isFull;

      SmoothScrolling:= ReadBool( GeneralSection, 'SmoothScrolling', true );

      UseOriginalDialogs:= ReadBool( GeneralSection, 'UseOriginalDialogs', false );

      BackgroundImageFilename := ReadString( GeneralSection, 'ToolbarBackground', '' );
    end;
  end;
  IniFile.Destroy;
End;

Procedure SaveSettings;
Var
  IniFile: TAsciiIniFile;
  i: longint;
  IndexStyleString: string;
Begin
  IniFile:= TAsciiIniFile.Create( IniFilePath );
  with IniFile do
  begin
    with Settings do
    begin
      // Viewer
      WriteBool( GeneralSection, 'ShowHints', ShowHints );

      // Write split points, as units of 0.1%
      WriteInteger( GeneralSection, 'ViewSplit', Round( ViewSplit * 1000 ) );

      // COlours
      for i:= 0 to High( Colors ) do
        WriteInteger( ColoursSection, 'Color' + IntToStr( i ), Colors[ i ] );

      WriteInteger( MRUSection,
                    'Count',
                    MRUList.Count );
      for i:= 0 to MRUList.Count -1 do
        WriteString( MRUSection,
                     'File' + IntToStr( i ),
                     MRUList[ i ] );

      WriteString( FontsSection, 'NormalFont', NormalFont.FaceName );
      WriteInteger( FontsSection, 'NormalFontSize', NormalFont.PointSize );
      WriteString( FontsSection, 'FixedFont', FixedFont.FaceName );
      WriteInteger( FontsSection, 'FixedFontSize', FixedFont.PointSize );

      case IndexStyle of
        isFileOnly:
          IndexStyleString:= 'FileOnly';
        isAlphabetical:
          IndexStyleString:= 'Alphabetical';
        isFull:
          IndexStyleString:= 'Full';
      end;

      WriteString( GeneralSection, 'IndexStyle', IndexStyleString );

      WriteBool( GeneralSection, 'SmoothScrolling', SmoothScrolling );
      WriteBool( GeneralSection, 'UseOriginalDialogs', UseOriginalDialogs );

      WriteString( GeneralSection, 'ToolbarBackground', BackgroundImageFilename );
    end;
  end;

  IniFile.Destroy;

End;

Procedure WriteWindowPos( X, Y, width, height: longint;
                          Maximised: boolean );
Var
  IniFile: TAsciiIniFile;
Begin
  IniFile:= TAsciiIniFile.Create( IniFilePath );
  IniFile.WriteInteger( WindowSection,
                        'X',
                        X );
  IniFile.WriteInteger( WindowSection,
                        'Y',
                        Y );
  IniFile.WriteInteger( WindowSection,
                        'Width',
                        Width );
  IniFile.WriteInteger( WindowSection,
                        'Height',
                        Height );
  IniFile.WriteBool( WindowSection,
                     'Maximised',
                     Maximised );
  IniFile.Destroy;
End;

Procedure ReadWindowPos( Var X, Y, width, height: longint;
                         Var Maximised: boolean );
Var
  IniFile: TAsciiIniFile;
Begin
  IniFile:= TAsciiIniFile.Create( IniFilePath );
  X:=IniFile.ReadInteger( WindowSection,
                          'X',
                          ( Screen.Width-DefaultWidth ) div 2 );
  Y:=IniFile.ReadInteger( WindowSection,
                          'Y',
                          ( Screen.Height-DefaultHeight ) div 2 );
  Width:=IniFile.ReadInteger( WindowSection,
                              'Width',
                              DefaultWidth );
  Height:=IniFile.ReadInteger( WindowSection,
                               'Height',
                               DefaultHeight );
  Maximised:=IniFile.ReadBool( WindowSection,
                               'Maximised',
                               false );
  IniFile.Destroy;
End;

Initialization
  Settings.NormalFont:= Screen.GetFontFromPointSize( 'Helv',
                                                     8 );
  Settings.FixedFont:= Screen.GetFontFromPointSize( 'Courier',
                                                     8 );
End.
