Unit MainForm;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

Uses
  Classes, Forms, Graphics, ComCtrls, ExtCtrls, SplitBar, TabCtrls,
  CustomOutline, Outline2, Dialogs, StdCtrls, Buttons, RichTextView, coolbar2,
  HelpFile, HelpTopic, HelpWindow, CustomMemo, Messages, HT, WindowMessages,
  CustomListBox, SysUtils;

Type

  THelpNote = class
    Text: PChar;
    Topic: TTopic;
    InsertPoint: longint;

    // calculated
    InsertText: PChar;
    InsertLength: longword;

    constructor Create;
    destructor Destroy; override;
  end;

  // Navigate point: a saved history position specifiying
  // a set of windows that were displayed.
  TNavigatePoint = class
    ContentsTopic: TTopic;
    Windows: TList;
    constructor Create;
    destructor Destroy; Override;
  end;

  // A window containing a topic.
  THelpWindow = class( TControl )
    View: TRichTextView;
    CloseButton: TSpeedButton; // not working to well right now.

    Images: TImageList;

    Topic: TTopic; // the topic being displayed
    Group: longint; // IPF group number

    ChildWindows: TList;

    Rect: THelpWindowRect;

    procedure SetupComponent; override;
    procedure Resize; override;
    destructor Destroy; override;
  end;

  // A help window definition as saved in a navigate point
  TSavedHelpWindow = class
    Topic: TTopic;
    Group: longint;

    ChildWindows: TList;
    Parent: TSavedHelpWindow;

    Rect: THelpWindowRect;

    TopCharIndex: longint; // used when saving for navigation

    constructor Create( Window: THelpWindow );
    destructor Destroy; override;
  end;

  TMainForm = Class (TForm)
    VSplitBar: TSplitBar;
    Notebook: TNoteBook;
    IndexSearchEdit: TEdit;
    SearchTextEdit: TEdit;
    SearchResultsListBox: TListBox;
    NotesListBox: TListBox;
    TabSet: TTabSet;
    CoolBar: TCoolBar2;
    ViewPopupMenu: TPopupMenu;
    SelectAllPMI: TMenuItem;
    CopyPMI: TMenuItem;
    AddNoteButton: TBitBtn;
    ContentsOutline: TOutline2;
    IndexListBox: TCustomListBox;
    ViewCollapseAllMI: TMenuItem;
    MenuItem7: TMenuItem;
    SystemOpenDialog: TSystemOpenDialog;
    LoadHelpFileTask: TStdTask;
    HelperThread: THelperThread;
    SearchButton: TButton;
    MenuItem1: TMenuItem;
    ViewIndexMI: TMenuItem;
    ViewContentsMI: TMenuItem;
    ViewSearchMI: TMenuItem;
    ViewNotesMI: TMenuItem;
    ViewRefreshMI: TMenuItem;
    MenuItem11: TMenuItem;
    DisplayPanel: TPanel;
    ButtonImages: TImageList;
    BookmarksMenu: TMenuItem;
    EditNoteButton: TBitBtn;
    GotoNoteButton: TBitBtn;
    FileCloseMI: TMenuItem;
    StatusLabel: TLabel;
    ProgressBar: TProgressBar;
    MainMenu: TMainMenu;
    FileMenu: TMenuItem;
    OpenMI: TMenuItem;
    FileSaveAsMI: TMenuItem;
    PrintMI: TMenuItem;
    FileInformationMI: TMenuItem;
    MenuItem3: TMenuItem;
    LibrariesMI: TMenuItem;
    OptionsMI: TMenuItem;
    DebugMI: TMenuItem;
    MenuItem4: TMenuItem;
    ExitMI: TMenuItem;
    EditMenu: TMenuItem;
    SelectAllMI: TMenuItem;
    CopyMI: TMenuItem;
    MenuItem5: TMenuItem;
    FindMI: TMenuItem;
    FindNextMI: TMenuItem;
    NavigateMenu: TMenuItem;
    NavigateBackMI: TMenuItem;
    NavigateForwardMI: TMenuItem;
    NavigateHistoryMI: TMenuItem;
    MenuItem6: TMenuItem;
    NavigateNextMI: TMenuItem;
    NavigatePreviousMI: TMenuItem;
    ToolsMenu: TMenuItem;
    AddBookmarkMI: TMenuItem;
    GlobalSearchMI: TMenuItem;
    HelpMenu: TMenuItem;
    HelpMI: TMenuItem;
    HelpAboutMI: TMenuItem;
    DeleteNoteButton: TBitBtn;
    AddNoteMI: TMenuItem;
    OpenDialog: TOpenDialog;

    Procedure CopyPMIOnClick (Sender: TObject);
    Procedure SelectAllPMIOnClick (Sender: TObject);
    Procedure AddNoteButtonOnClick (Sender: TObject);
    Procedure SearchTextEditOnChange (Sender: TObject);
    Procedure IndexListBoxOnClick (Sender: TObject);
    Procedure ViewCollapseAllMIOnClick (Sender: TObject);
    Procedure NotesListBoxOnDblClick (Sender: TObject);
    Procedure LoadHelpFileTaskOnSchedule (Task: TBasicTask;
      TaskItem: TTaskItem);
    Procedure LoadHelpFileTaskOnExecute (Parameters: TObject;
      Context: TStdTaskItem);
    Procedure HelpMIOnClick (Sender: TObject);
    Procedure NotebookOnPageChanged (Sender: TObject);
    Procedure ViewRefreshMIOnClick (Sender: TObject);
    Procedure ViewNotesMIOnClick (Sender: TObject);
    Procedure ViewSearchMIOnClick (Sender: TObject);
    Procedure ViewIndexMIOnClick (Sender: TObject);
    Procedure ViewContentsMIOnClick (Sender: TObject);
    Procedure DisplayPanelOnSetupShow (Sender: TObject);
    Procedure MainFormOnCloseQuery (Sender: TObject; Var CanClose: Boolean);
    Procedure GlobalSearchMIOnClick (Sender: TObject);
    Procedure GotoNoteButtonOnClick (Sender: TObject);
    Procedure EditNoteButtonOnClick (Sender: TObject);
    Procedure NotesListBoxOnItemFocus (Sender: TObject; Index: LongInt);
    Procedure DeleteNoteButtonOnClick (Sender: TObject);
    Procedure AddBookmarkMIOnClick (Sender: TObject);
    Procedure AddNoteMIOnClick (Sender: TObject);
    Procedure FileCloseMIOnClick (Sender: TObject);
    Procedure CoolBarOnSectionResize (HeaderControl: THeaderControl;
      section: THeaderSection);
    Procedure CoolBarOnSectionClick (HeaderControl: THeaderControl;
      section: THeaderSection);
    Procedure MainFormOnDestroy (Sender: TObject);
    Procedure MainFormOnSetupShow (Sender: TObject);
    Procedure MainFormOnCreate (Sender: TObject);
    Procedure MainFormOnShow (Sender: TObject);
    Procedure ContentsOutlineOnItemFocus (Node: TNode);
    Procedure FindNextMIOnClick (Sender: TObject);
    Procedure FindMIOnClick (Sender: TObject);
    Procedure IndexSearchEditOnScan (Sender: TObject; Var KeyCode: TKeyCode);
    Procedure IndexSearchEditOnChange (Sender: TObject);
    Procedure FileInformationMIOnClick (Sender: TObject);
    Procedure SearchResultsListBoxOnItemFocus (Sender: TObject;
      Index: LongInt);
    Procedure SearchTextEditOnScan (Sender: TObject; Var KeyCode: TKeyCode);
    Procedure SearchButtonOnClick (Sender: TObject);
    Procedure FileSaveAsMIOnClick (Sender: TObject);
    Procedure OptionsMIOnClick (Sender: TObject);
    Procedure IndexListBoxOnItemFocus (Sender: TObject; Index: LongInt);
    Procedure TabSetOnChange (Sender: TObject; NewTab: LongInt;
      Var AllowChange: Boolean);
    Procedure NotebookOnSetupShow (Sender: TObject);
    Procedure NavigateBackMIOnClick (Sender: TObject);
    Procedure NavigatePreviousMIOnClick (Sender: TObject);
    Procedure NavigateNextMIOnClick (Sender: TObject);
    Procedure CopyMIOnClick (Sender: TObject);
    Procedure SelectAllMIOnClick (Sender: TObject);
    Procedure DebugMIOnClick (Sender: TObject);
    Procedure HelpAboutMIOnClick (Sender: TObject);
    Procedure OnOverLink ( Sender: TRichTextView; LinkString: String);
    Procedure OnNotOverLink ( Sender: TRichTextView; LinkString: String);
    Procedure OnClickLink ( Sender: TRichTextView; LinkString: String);
    Procedure OnWindowCloseClick( Sender: TObject);
    Procedure BackButtonOnClick (Sender: TObject);
    Procedure RTViewOnSetupShow (Sender: TObject);
    Procedure OpenMIOnClick (Sender: TObject);
    Procedure ExitMIOnClick (Sender: TObject);

    Procedure MainFormOnResize (Sender: TObject);
    Procedure VSplitBarOnChange (NewSplit: LongInt);
  Protected
    // Handle our own WM_OPENED message
    Procedure WMOpened( Var Msg: TMessage ); Message WM_OPENED;
    Procedure WMFollowLink( Var Msg: TMessage ); Message WM_FOLLOWLINK;

  Protected
    Procedure OnHint( Sender: TObject );

  Protected
    // Startup and loading functions
    Procedure CheckEnvironmentVars;
    Procedure ParseParameters;
    Procedure ShowUsage;

    Procedure CreateMRUMenuItems;
    Procedure MRUItemClick( Sender: TObject );

    // GIven a "filename" which may include a path find
    // it it in various paths and extensions
    Function FindHelpFile( FileName: string ): string;

    Procedure OnHelpFileLoadProgress( n, outof: integer;
                                      message: string );

    Function TranslateEnvironmentVar( Filenames: string ): string;
    Function OpenFile( FileNames: string ): boolean;
    Procedure CloseFile;

    // Returns nil if file is not open
    Function FindOpenHelpFile( FileName: string ): THelpFile;


    Procedure SaveNavigatePoint;
    Procedure SaveWindows( SourceList: TList;
                           DestList: TList;
                           Parent: TSavedHelpWindow );

    Procedure UpdateCurrentNavigatePoint;

    Procedure NavigateToPoint( NavPoint: TNavigatePoint );
    Procedure DisplayWindows( WindowList: TList;
                              Parent: THelpWindow );
    Procedure CloseWindows;

    Procedure NavigateBack;
    Procedure NavigateForward;
    Procedure NavigatePreviousInContents;
    Procedure NavigateNextInContents;

    Procedure EnableControls;
    Procedure EnableSearchButton;

    // Used in loading contents
    Procedure AddChildNodes( HelpFile: THelpFile;
                             ParentNode: TNode;
                             Level: longint;
                             Var TopicIndex: longint );
    Procedure LoadContents;
    Procedure LoadIndex;

    procedure ApplySettings;

    // Note manipulations
    procedure AddNote;
    Procedure EditNote( NoteIndex: longint );
    procedure DeleteNote( NoteIndex: longint );
    Procedure SaveNotes( HelpFile: THelpFile );
    Procedure LoadNotes( HelpFile: THelpFile );
    Procedure GotoCurrentNote;

    Procedure InsertNotesIntoTopic( Window: THelpWindow );
    function FindOriginalNoteCharIndex( NoteCharIndex: longword;
                                        Topic: TTopic ): longword;
    function FindActualNoteCharIndex( NoteCharIndex: longword;
                                      MaxNoteIndex: longword;
                                      Topic: TTopic ): longword;
    procedure RefreshNoteInsertInfo( NoteIndex: longword );
    procedure ClearNotes;

    Procedure EnableNotesControls;
    Procedure UpdateNotesDisplay;

    procedure FileOpen;
    procedure FilePrint;
    procedure AddBookmark;

    Procedure SetStatus( Text: String );
    Procedure ResetProgress;

    // Major display topic function.
    procedure DisplayTopic( Topic: TTopic );

    // Called when viewing topics from global search
    Procedure OnViewGlobalSearchTopic( FileName: string;
                                       TopicIndex: longint );

    Procedure SetLayout;
    function OpenWindow( Topic: TTopic;
                         Group: longint;
                         Parent: THelpWindow;
                         Rect: THelpWindowRect ): THelpWindow;
    Procedure LayoutWindows( WindowList: TList );
    Procedure SetWindowLayout( Window: THelpWindow );
    Procedure RefreshWindows( WindowList: TList );

    Procedure DisplayTopicInWindow( Window: THelpWindow );
    Procedure FollowLink( Link: THelpLink;
                          SourceWindow: THelpWindow );

    Function FindWindowFromView( View: TRichTextView; WindowList: TList ): THelpWindow;
    Function FindWindowFromGroup( Group: longint; WindowList: TList ): THelpWindow;
    Function GetActiveWindow: THelpWindow;

    Files: TList; // current open help files.
    MRUMenuItems: TList; // most recently used file list

    // Current topic has the vague meaning that it was the last
    // topic selected by the user... (?)
    CurrentTopic: TTopic;

    Windows: TList; // top level help windows

    PageHistory: TStringList; // history
    CurrentHistoryIndex: longint; // where we are in history

    Navigating: boolean; // true while going to a particular history point

    DisplayedIndex: TStringList; // duplicate of index listbox,
                                 // for fast case insensitive searching
    InIndexSearch: boolean; // true while searching index

    FindText: string; // last text found (Ctrl-F)

    Notes: TList; // Notes in current files.
    NotesChangedSinceLoad: boolean; // So we know when to save.

    // while loading... so owe can display progress
    LoadingFilenameList: TStringList;
    LoadingFileIndex: integer;
    LoadingTotalSize: longint;
    LoadingSizeSoFar: longint;

    Debug: boolean;
    DoProfiling: boolean;

    CloseButtonBitmap: TBitmap;

    Procedure OnException( Sender: TObject;
                           E: Exception );
  End;

Var
  MainForm: TMainForm;

Implementation

uses
  BseDos, OS2Def, BseErr, BseTIB,
  PMWin, Dos,
  ACLStringUtility, ACLFileUtility, ACLFileIOUtility, ACLPCharUtility, ACLUtility,
  ACLProfile, ACLDialogs, ACLConstants,
  FileInformationForm, OptionsForm, AboutBox, NoteForm, GlobalSearchForm,
  SettingsUnit, Version, FileDialogForm,
  TextSearchQuery, SearchUnit;

{$R Images}

const
  // Coolbar button indexes
  ciOpen = 0;
  ciBack = 1;
  ciForward = 2;
  ciPrint = 3;
  ciAddNote = 4;
  ciAddBookmark = 5;
  ciPrevious = 6;
  ciNext = 7;

  // Page indexes.
  piContents = 0;
  piIndex = 1;
  piSearch = 2;
  piNotes = 3;

  HelpPathEnvironmentVar = 'HELP';
  BookshelfEnvironmentVar = 'BOOKSHELF';

  _MAX_PATH = 260;

var
  StartMem: longword;
  LastMem: longword;

// --- TNavigatePoint ---------------------------------

constructor TNavigatePoint.Create;
begin
  inherited Create;
  ContentsTopic:= nil;
  Windows:= TList.Create;
end;

destructor TNavigatePoint.Destroy;
begin
  inherited Destroy;
  DestroyListObjects( Windows );
  Windows.Destroy;
end;

// --- THelpNote ---------------------------------

constructor THelpNote.Create;
begin
  Text:= nil;
  InsertText:= nil;
  Topic:= nil;
  InsertPoint := -1;
end;

destructor THelpNote.Destroy;
begin
  if Text <> nil then
    StrDispose( Text );
  if InsertText <> nil then
    StrDispose( InsertText);
end;

// --- THelpWindow ---------------------------------

procedure THelpWindow.SetupComponent;
begin
  inherited SetupComponent;

  Color := clWhite;//clBlue;
  View:= TRichTextView.Create( Self );
  View.UseDefaultMenu:= false;
  View.Parent := self;

  Images:= TImageList.Create( Self );

  ChildWindows:= TList.Create;

  CloseButton:= TSpeedButton.Create( self );
  CloseButton.Parent := self;
  CloseButton.Glyph := MainForm.CloseButtonBitmap;
end;

destructor THelpWindow.Destroy;
begin
  DestroyListObjects( ChildWindows );
  ChildWindows.Destroy;
  inherited Destroy;
end;

procedure THelpWindow.Resize;
var
  ButtonW, ButtonH: longint;
begin
  inherited Resize;
  ButtonW := View.ScrollBarWidth;// CloseButton.Glyph.Width + 4;
  ButtonH := View.ScrollBarWidth;// CloseButton.Glyph.Height + 4;

  CloseButton.Hide;
  CloseButton.SetWindowPos( Width - ButtonW,
                            Height - ButtonH,
                            ButtonW,
                            ButtonH );
  View.SetWindowPos( 0, 0,
                     Width,
                     Height ); // -ButtonH );
end;

// --- TSavedHelpWindow ---------------------------------

constructor TSavedHelpWindow.Create;
begin
  inherited Create;
  ChildWindows:= TList.Create;
end;

destructor TSavedHelpWindow.Destroy;
begin
  DestroyListObjects( ChildWindows );
  ChildWindows.Destroy;
  inherited Destroy;
end;

// --- TMainForm implementation ---

Procedure TMainForm.CopyPMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
Begin
  Window := GetActiveWindow;
  if Window = nil then
    exit;
  Window.View.CopySelectionToClipboard;
End;

Procedure TMainForm.SelectAllPMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
Begin
  Window := GetActiveWindow;
  if Window = nil then
    exit;
  Window.View.SelectAll;
End;

Procedure TMainForm.AddNoteButtonOnClick (Sender: TObject);
Begin
  AddNote;
End;

Procedure TMainForm.EnableSearchButton;
var
  CanSearch: boolean;
begin
  CanSearch := false;
  if Files.Count > 0 then
    if trim( SearchTextEdit.Text ) > '' then
      CanSearch := true;
  SearchButton.Enabled := CanSearch;
end;

Procedure TMainForm.SearchTextEditOnChange (Sender: TObject);
Begin
  EnableSearchButton;
End;

Procedure TMainForm.OnHint( Sender: TObject );
begin
  SetStatus( Application.Hint );
end;

Procedure TMainForm.IndexListBoxOnClick (Sender: TObject);
Begin

End;

Procedure TMainForm.ViewCollapseAllMIOnClick (Sender: TObject);
Begin
  ContentsOutline.CollapseAll;
End;

Procedure TMainForm.NotesListBoxOnDblClick (Sender: TObject);
Begin
  GotoCurrentNote;
End;

Procedure TMainForm.LoadHelpFileTaskOnSchedule (Task: TBasicTask;
  TaskItem: TTaskItem);
Begin
End;

Procedure TMainForm.HelpMIOnClick (Sender: TObject);
Begin
End;

Procedure TMainForm.NotebookOnPageChanged (Sender: TObject);
Begin
  if Notebook.PageIndex = piSearch then
  begin
    SearchButton.Focus;
    SearchTextEdit.Focus;
  end;
End;

Procedure TMainForm.ViewRefreshMIOnClick (Sender: TObject);
Begin
  RefreshWindows( Windows );
End;

Procedure TMainForm.ViewNotesMIOnClick (Sender: TObject);
Begin
  TabSet.TabIndex:= piNotes;
End;

Procedure TMainForm.ViewSearchMIOnClick (Sender: TObject);
Begin
  TabSet.TabIndex:= piSearch;
End;

Procedure TMainForm.ViewIndexMIOnClick (Sender: TObject);
Begin
  TabSet.TabIndex:= piIndex;
End;

Procedure TMainForm.ViewContentsMIOnClick (Sender: TObject);
Begin
  Tabset.TabIndex:= piContents;
End;

Procedure TMainForm.DisplayPanelOnSetupShow (Sender: TObject);
Begin
End;

function TMainForm.OpenWindow( Topic: TTopic;
                               Group: longint;
                               Parent: THelpWindow;
                               Rect: THelpWindowRect ): THelpWindow;
var
  Window: THelpWindow;
begin
  if ( Group <> -1 ) and (Parent = nil ) then
  begin
    Window:= FindWindowFromGroup( Group, Windows );

    if Window <> nil then
    begin
      Window.Topic:= Topic;
      DisplayTopicInWindow( Window );
      exit;
    end;
  end;
  // else -group -1 means don't search existing windows.

  // not found
  Window:= THelpWindow.Create( nil );

  if Parent = nil then
  begin
    Windows.Add( Window );
    Window.Parent := DisplayPanel;
  end
  else
  begin
    Parent.ChildWindows.Add( Window );
    Window.Parent := Parent;
  end;

  Window.Topic:= Topic;
  Window.Group:= Group;

  Window.View.PopupMenu := ViewPopupMenu;

  Window.View.Images:= Window.Images;

  Window.View.OnClickLink:= OnClickLink;
  Window.View.OnOverLink:= OnOverLink;
  Window.View.OnNotOverLink:= OnNotOverLink;

  Window.CloseButton.OnClick := OnWindowCloseClick;

  Window.Rect := THelpWindowRect.Create;
  Topic.GetContentsWindowRect( Window.Rect );
  if Rect <> nil then
  begin
    if Rect.Left <> -1 then
      Window.Rect.Left:= Rect.Left;
    if Rect.Bottom <> -1 then
      Window.Rect.Bottom:= Rect.Bottom;
    if Rect.Width <> -1 then
      Window.Rect.Width:= Rect.Width;
    if Rect.Height <> -1 then
      Window.Rect.Height:= Rect.Height;
  end;

  SetWindowLayout( Window );

  DisplayTopicInWindow( Window );

  Result:= Window;
end;

Function TMainForm.FindWindowFromView( View: TRichTextView;
                                       WindowList: TList ): THelpWindow;
var
  WindowIndex: longint;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Result:= WindowList[ WindowIndex ];
    if Result.View = View then
      exit;
    Result:= FindWindowFromView( View, Result.ChildWindows );
    if Result <> nil then
      exit;
  end;
  Result:= nil;
end;

Function TMainForm.FindWindowFromGroup( Group: longint; WindowList: TList ): THelpWindow;
var
  WindowIndex: longint;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Result:= WindowList[ WindowIndex ];
    if Result.Group = Group then
      exit;
    Result:= FindWindowFromGroup( Group, Result.ChildWindows );
    if Result <> nil then
      exit;
  end;
  Result:= nil;
end;

Procedure TMainForm.SetWindowLayout( Window: THelpWindow );
var
  X, Y: longint;
  W, H: longint;
begin
  // Setup the rich text control
  Window.View.Color:= Settings.Colors[ TopicBackgroundColorIndex ];
  Window.View.BasicLeftMargin:= 10;
  Window.View.BasicRightMargin:= 10;
  Window.View.NormalFont:= Settings.NormalFont;
  Window.View.FixedFont:= Settings.FixedFont;
  Window.View.SmoothScroll:= Settings.SmoothScrolling;

  // Position the window correctly.

  W:= Window.Rect.Width * Window.Parent.Width div 100;
  H:= Window.Rect.Height * Window.Parent.Height div 100;

  if Window.Rect.Left = XYPosCenter then
    X:= Window.Parent.Width div 2 - W div 2
  else if Window.Rect.Left = XPosRight then
    X:= Window.Parent.Width - W
  else
    X:= Window.Rect.Left * Window.Parent.Width div 100;

  if Window.Rect.Bottom = XYPosCenter then
    Y:= Window.Parent.Height div 2 - H div 2
  else if Window.Rect.Bottom = YPosTop then
    Y:= Window.Parent.Height - H
  else
    Y:= Window.Rect.Bottom * Window.Parent.Height div 100;

  Window.SetWindowPos( X, Y, W, H );
end;

Procedure TMainForm.RefreshWindows( WindowList: TList );
var
  WindowIndex: longint;
  Window: THelpWindow;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Window:= WindowList[ WindowIndex ];
    DisplayTopicInWindow( Window );
    RefreshWindows( Window.ChildWindows );
  end;
end;

Procedure TMainForm.FollowLink( Link: THelpLink;
                                SourceWindow: THelpWindow );
var
  LinkedTopic: TTopic;
  ParentWindow: THelpWindow;
  HelpFile: THelpFile;
begin
  HelpFile:= Link.HelpFile as THelpFile;
  if not Between( Link.TopicIndex, 0, HelpFile.TopicCount - 1 ) then
  begin
    ShowMessage( 'Cannot follow link to nonexistent topic #'
                 + IntToStr( Link.TopicIndex ) );
    exit;
  end;
  LinkedTopic:= HelpFile.Topics[ Link.TopicIndex ];
  ParentWindow:= nil;
  if Link.Split then
    ParentWindow:= SourceWindow;
  if Link.GroupIndex <> -1 then
    OpenWindow( LinkedTopic, Link.GroupIndex, ParentWindow, Link.Rect )
  else if Link.ViewPort then
    OpenWindow( LinkedTopic, -1, ParentWindow, Link.Rect )
  else
    OpenWindow( LinkedTopic, LinkedTopic.ContentsGroupIndex, ParentWindow, Link.Rect );
end;

Procedure TMainForm.DisplayTopicInWindow( Window: THelpWindow );
var
  text: PChar;
  ImageIndices: TList;
  LinkIndex: longint;
  Link: THelpLink;
  HelpFile: THelpFile;
Begin
  Window.View.Clear;
  ImageIndices:= TList.Create;

  HelpFile := TopicFile( Window.Topic );
  Window.Topic.GetText( HelpFile.HighlightWords,
                        Debug,
                        Text,
                        ImageIndices );

  HelpFile.GetImages( ImageIndices, Window.Images );
  Window.View.AddText( Text );
  StrDispose( Text );

  InsertNotesIntoTopic( Window );

  for LinkIndex:= 0 to Window.Topic.Links.Count - 1 do
  begin
    Link:= Window.Topic.Links[ LinkIndex ];
    if Link.Automatic then
      FollowLink( Link, Window );
  end;

  ImageIndices.Destroy;
End;

Procedure TMainForm.MainFormOnCloseQuery (Sender: TObject;
  Var CanClose: Boolean);
Begin
  CloseFile;
End;

procedure TMainForm.DisplayTopic( Topic: TTopic );
begin
  if Navigating then
    exit;

  UpdateCurrentNavigatePoint;

  CloseWindows;

  CurrentTopic:= Topic;

  OpenWindow( CurrentTopic, CurrentTopic.ContentsGroupIndex, nil, nil );
  SetStatus( 'Opened topic #'
             + IntToStr( Topic.Index ) );

  Navigating:= true;

  if ContentsOutline.SelectedNode = nil then
    ContentsOutline.SetSelectedObject( Topic )
  else if ContentsOutline.SelectedNode.Data <> Topic then
    ContentsOutline.SetSelectedObject( Topic );

  SaveNavigatePoint;

  Navigating:= false;
  /// find in index...
  // find in search results...

  EnableControls;

end;

Procedure TMainForm.GlobalSearchMIOnClick (Sender: TObject);
Begin
  GlobalSearchForm.ViewTopicCallback:= OnViewGlobalSearchTopic;
  GlobalSearchForm.Show;
  WinSetOwner( GlobalSearchForm.Frame.Handle, Frame.Handle );
End;

Procedure TMainForm.OnViewGlobalSearchTopic( FileName: string;
                                             TopicIndex: longint );
var
  HelpFile: THelpFile;
begin
  HelpFile:= FindOpenHelpFile( FileName );

  if HelpFile = nil then
    if OpenFile( FIlename ) then
      HelpFile:= Files[ 0 ];

  if HelpFile <> nil then
    if TopicIndex <> -1 then
      DisplayTopic( HelpFile.Topics[ TopicIndex ] );

end;

Procedure TMainForm.GotoNoteButtonOnClick (Sender: TObject);
begin
  GotoCurrentNote;
end;

Procedure TMainForm.EditNoteButtonOnClick (Sender: TObject);
Begin
  if NotesListBox.ItemIndex = -1 then
    exit;
  EditNote( NotesListBox.ItemIndex );
End;

Procedure TMainForm.NotesListBoxOnItemFocus (Sender: TObject; Index: LongInt);
var
  Note: THelpNote;
Begin
  Note:= NotesListBox.Items.Objects[ NotesListBox.ItemIndex ] as THelpNote;
  EnableNotesControls;
End;

Procedure TMainForm.DeleteNoteButtonOnClick (Sender: TObject);
Begin
  if NotesListBox.ItemIndex = -1 then
    exit;
  DeleteNote( NotesListBox.ItemIndex );
End;

Procedure TMainForm.AddBookmarkMIOnClick (Sender: TObject);
Begin
  AddBookmark;
End;

Procedure TMainForm.AddNoteMIOnClick (Sender: TObject);
Begin
  AddNote;
End;

Procedure TMainForm.FileCloseMIOnClick (Sender: TObject);
Begin
  CloseFile;
End;

Procedure TMainForm.SetStatus( Text: String );
begin
  StatusLabel.Caption:= Text;
  StatusLabel.Refresh;
end;

Procedure TMainForm.ResetProgress;
begin
  ProgressBar.Position:= 0;
  ProgressBar.Hide;
end;

Procedure TMainForm.CoolBarOnSectionResize (HeaderControl: THeaderControl;
  section: THeaderSection);
Begin

End;

Procedure TMainForm.CoolBarOnSectionClick (HeaderControl: THeaderControl;
  section: THeaderSection);
Begin
  case Section.Index of
    ciOpen:
      FileOpen;
    ciBack:
      NavigateBack;
    ciForward:
      NavigateForward;
    ciPrint:
      FilePrint;
    ciAddNote:
      AddNote;
    ciAddBookmark:
      AddBookmark;
    ciPrevious:
      NavigatePreviousInContents;
    ciNext:
      NavigateNextInContents;
  end;

End;

// ---------------- Notes ----------------------

function TMainForm.FindOriginalNoteCharIndex( NoteCharIndex: longword;
                                              Topic: TTopic ): longword;
var
  NoteIndex: longint;
  Note: THelpNote;
begin
  NoteIndex:= 0;
  Result:= NoteCharIndex;
  for NoteIndex:= 0 to Notes.Count - 1 do
  begin
    Note:= Notes[ NoteIndex ];
    if Note.Topic = Topic then
      if Note.InsertPoint < NoteCharIndex then
        dec( Result, Note.InsertLength );
  end;
end;

function TMainForm.FindActualNoteCharIndex( NoteCharIndex: longword;
                                            MaxNoteIndex: longword;
                                            Topic: TTopic ): longword;
var
  NoteIndex: longint;
  Note: THelpNote;
begin
  NoteIndex:= 0;
  Result:= NoteCharIndex;
  for NoteIndex:= 0 to MaxNoteIndex - 1 do
  begin
    Note:= Notes[ NoteIndex ];
    if Note.Topic = Topic then
      if Note.InsertPoint < NoteCharIndex then
        inc( Result, Note.InsertLength );
  end;
end;

procedure TMainForm.RefreshNoteInsertInfo( NoteIndex: longword );
var
  Note: THelpNote;
begin
  Note:= Notes[ NoteIndex ];

  if Note.Topic = nil then
    exit;
  with Note do
  begin
    if InsertText <> nil then
      StrDispose( InsertText );
    InsertText:= StrAlloc( StrLen( Text ) + 100 );
    StrPCopy( InsertText,
              '<red><link note'
              + IntToStr( NoteIndex )
              + '>' );
    StrCat( InsertText, Text );
    StrCat( InsertText, '<black></link>' );
    InsertLength:= StrLen( InsertText );
  end;
end;

procedure TMainForm.ClearNotes;
var
  Note: THelpNote;
begin
  while Notes.Count > 0 do
  begin
    Note:= Notes[ 0 ];
    Note.Destroy;
    Notes.Delete( 0 );
  end;

end;

procedure TMainForm.AddNote;
var
  Note: THelpNote;
  Window: THelpWindow;
  Topic: TTopic;
begin
  Window:= GetActiveWindow;
  if Window = nil then
  begin
    ShowMessage( 'Before adding a note, position the cursor where you want the note to be placed.' );
    exit;
  end;
  if Window.View.CursorIndex <> -1 then
  begin
    if Window.View.LinkFromIndex( Window.View.CursorIndex )  <> '' then
    begin
      ShowMessage( 'You can''t add a note within a link or another note' );
      exit;
    end;
  end;

  NoteForm.DeleteNoteButton.Enabled:= false;
  NoteForm.Text:= '';
  if NoteForm.ShowModal <> mrOK then
    exit;

  Note:= THelpNote.Create;
  Note.Text:= NoteForm.Text;

  TrimEndLines( Note.Text );
  if Window.View.CursorIndex <> -1 then
    Note.InsertPoint:= FindOriginalNoteCharIndex( Window.View.CursorIndex, Window.Topic )
  else
    Note.InsertPoint := 0;
  Note.Topic:= Window.Topic;

  Notes.Add( Note );
  RefreshNoteInsertInfo( Notes.Count - 1 );

  DisplayTopicInWindow( Window );

  UpdateNotesDisplay;
  NotesChangedSinceLoad:= true;
end;

procedure TMainForm.DeleteNote( NoteIndex: longint );
var
  Note: THelpNote;
begin
  Note:= Notes[ NoteIndex ];
  Notes.Delete( NoteIndex );

  RefreshWindows( Windows );

  Note.Destroy;

  UpdateNotesDisplay;
  NotesChangedSinceLoad:= true;
end;

Procedure TMainForm.EditNote( NoteIndex: longint );
var
  Note: THelpNote;
begin
  Note:= Notes[ NoteIndex ];

  NoteForm.Text:= Note.Text;

  NoteForm.DeleteNoteButton.Enabled:= true;

  if NoteForm.ShowModal = mrCancel then
    exit;

  if NoteForm.ModalResult = cmDiscard then
  begin
    DeleteNote( NoteIndex );
    exit;
  end;

  if Note.Text <> nil then
    StrDispose( Note.Text );

  Note.Text:= NoteForm.Text;
  TrimEndLines( Note.Text );
  RefreshNoteInsertInfo( NoteIndex );

  RefreshWindows( Windows );

  UpdateNotesDisplay;
  NotesChangedSinceLoad:= true;

end;

Procedure TMainForm.GotoCurrentNote;
var
  Note: THelpNote;
Begin
  if NotesListBox.ItemIndex = -1 then
    exit;
  Note:= NotesListBox.Items.Objects[ NotesListBox.ItemIndex ] as THelpNote;
  DisplayTopic( Note.Topic );
End;

// ---------------- Bookmarks ----------------------

procedure TMainForm.AddBookmark;
begin
end;

// --------------------------------------------------

procedure TMainForm.FilePrint;
begin
end;

Procedure TMainForm.MainFormOnDestroy (Sender: TObject);
var
  Maximized: boolean;
Begin
  Maximized:= WindowState = wsMaximized;
  if Maximized then
    WindowState:= wsNormal;

  WriteWindowPos( Left, Bottom, Width, Height,
                  Maximized );
  SaveSettings;
  MRUMenuItems.Destroy;
  PageHistory.Destroy;
  ClearNotes;
  Notes.Destroy;
  Files.Destroy;
  DisplayedIndex.Destroy;
  Windows.Destroy;
  CloseButtonBitmap.Destroy;
End;

Procedure TMainForm.MainFormOnSetupShow (Sender: TObject);
Begin
  ProfileEvent( 'OnSetupShow' );
  TabSet.TabIndex:= 0;
  Notebook.PageIndex:= 0;
End;

// This is all very jolly... but... I think I have to call it from
// ExcptHandler in system.pas before the stack gets unwound for the
// except clause...Doh!
PROCEDURE WriteCallStack( Var F: TextFile;
                          const e: exception );

VAR
  EBP,EIP,TargetEIP: LONGWORD;
  StackList: record
    OldEBP: LONGWORD;
    RetEIP: LONGWORD;
  end;
  CallArray: array[1..5] of BYTE;
  pThreadInfo: PTIB;
  pProcessInfo: PPIB;
Begin
  DosGetInfoBlocks( pThreadInfo, pProcessInfo );

  IF pThreadInfo^.tib_pstack = nil THEN
    exit;

  WriteLn( F, IntToHex( e.contextrecord.ctx_regEIP, 8 ) );

         {???????????????????????????????}
         {if the pointer at ESP is a call we use ESP instead of EBP}
  MemCopy( pointer( e.contextrecord.ctx_regESP ),
           Addr( StackList ),
           sizeof( StackList ));

  TargetEIP := e.contextrecord.ctx_regEIP;
  IF StackList.OldEBP = e.contextrecord.ctx_regEBP THEN
  BEGIN
    EBP := StackList.RetEIP; //After push EBP
    dec(TargetEIP);
  END
  ELSE
    EBP := StackList.OldEBP;
  MemCopy( pointer( EBP - 5 ),
           Addr( CallArray ),
           5 );
  IF CallArray[1] = $E8 THEN
  BEGIN
    MemCopy( Addr( CallArray[ 2 ] ),
             Addr( EIP ),
             sizeof( EIP ) );

    IF EBP+EIP <= TargetEIP THEN
      WriteLn( F, IntToHex( EBP-5, 8 ) );
  END;

  EBP := e.ContextRecord.ctx_regEBP;
  WHILE EBP <> longword( pThreadInfo^.tib_pstack ) DO
  BEGIN
    MemCopy( pointer( EBP ),
             Addr( StackList ),
             sizeof( StackList ) );
    WriteLn( F, IntToHex( StackList.RetEIP-5, 8 ) );
    EBP := StackList.OldEBP;
  END;
END;

Procedure TMainForm.OnException( Sender: TObject;
                                 E: Exception );
var
  TheText : string;
  F: TextFile;
begin

  AssignFile( F, 'crash.log' );
  if FileExists( 'crash.log' ) then
    Append( F )
  else
    Rewrite( F );

  WriteLn( F, 'NewView ' + GetAppVersion + ' crash log' );
  WriteLn( F, FormatDateTime( 'd mmmm yyyy, hh:mm:ss', now ) );
  WriteLn( F, 'Exception type: ' + E.ClassName );
  WriteLn( F, 'Description: ' + E.Message );
  WriteLn( F, 'Location: $'
               + IntToHex( longword( E.ExcptAddr ), 8 ) );

//  WriteCallStack( F, E );
  WriteLn( F, '---- End of report ----' );
  WriteLn( F, '' );
  System.Close( F );

  TheText := 'This application has crashed. ' + EndLine + EndLine
          + E.Message + EndLine + EndLine
          + 'Close application? '
          + EndLine
          + '(Details logged to crash.log)';
  if DoYesNoDlg( 'Application error - Close?',
                 TheText ) then
    Application.Terminate;
end;

Procedure TMainForm.MainFormOnCreate (Sender: TObject);
var
  DefaultFont: TFont;
Begin
  ContentsOutline.SmoothScroll := false;

  CloseButtonBitmap := TBitmap.Create;
  CloseButtonBitmap.LoadFromResourceName( 'CloseBtnBitmap' );

  DefaultFont := Screen.GetFontFromPointSize( 'WarpSans', 9 );
  if DefaultFont = nil then
    DefaultFont := Screen.GetFontFromPointSize( 'Helv', 8 );

  // Does not work prior to FP4.
  Application.Font:= DefaultFont;

  Application.OnException := OnException;

  ProfileEvent( 'Starting NewView: MainFormOnCreate' );

  Application.OnHint := OnHint;

  ProfileEvent( 'Set font/hint' );

  StartMem:= MemAvailBytes;

  DisplayedIndex:= TStringList.Create;

  Files:= TList.Create;
  Notes:= TList.Create;
  Windows:= TList.Create;

  Navigating:= false;
  InIndexSearch:= false;

  PageHistory:= TStringList.Create;
  CurrentHistoryIndex:= -1;

  CloseFile;

  Settings.MRUList:= TStringList.Create;

  ProfileEvent( 'Loading settings' );

  LoadSettings;

  ProfileEvent( 'Applying settings' );

  ApplySettings;

  ProfileEvent( 'Creating MRU list' );

  MRUMenuItems:= TLIst.Create;

  CreateMRUMenuItems;

  ProfileEvent( 'OnCreate done' );

End;

Procedure TMainForm.ApplySettings;
var
  ToolbarBitmap: TBitmap;
begin
  ContentsOutline.Color:= Settings.Colors[ ContentsBackgroundColorIndex ];
  ContentsOutline.PenColor:= Settings.Colors[ ContentsTextColorIndex ];
  ContentsOutline.TreeLineColor:= Settings.Colors[ ContentsLinesColorIndex ];
  IndexListBox.Color:= Settings.Colors[ IndexBackgroundColorIndex ];
  IndexListBox.PenColor:= Settings.Colors[ IndexTextColorIndex ];
  SearchResultsListBox.Color:= Settings.Colors[ SearchBackgroundColorIndex ];
  SearchResultsListBox.PenColor:= Settings.Colors[ SearchTextColorIndex ];
  NotesListBox.Color:= Settings.Colors[ NotesBackgroundColorIndex ];
  NotesListBox.PenColor:= Settings.Colors[ NotesTextColorIndex ];

  Coolbar.BackgroundBitmap := nil;
  if FileExists( Settings.BackgroundImageFilename ) then
  begin
    ToolbarBitmap:= TBitmap.Create;
    try
      ToolbarBitmap.LoadFromFIle( Settings.BackgroundImageFilename );
      Coolbar.BackgroundBitmap := ToolbarBitmap;
    except
    end;
    ToolbarBitmap.Destroy;
  end;
  LayoutWindows( Windows );
end;

Procedure TMainForm.MainFormOnShow (Sender: TObject);
var
  AnIcon: TICon;
Begin
  ProfileEvent( 'MainFormOnShow' );

  AnIcon:= TICon.Create;
  AnIcon.LoadFromResourceName( 'AppIcon' );
  Icon:= AnIcon;

  ProfileEvent( 'Post WM_OPENED' );

  PostMsg( Handle, WM_OPENED, 0, 0 );
End;

Procedure TMainForm.ContentsOutlineOnItemFocus (Node: TNode);
var
  Topic: TTopic;
Begin
  Topic:= Node.Data as TTopic;
  DisplayTopic( Topic );
End;

Procedure TMainForm.CheckEnvironmentVars;
var
  HelpOK: boolean;
  BookshelfOK: boolean;
begin
  HelpOK := GetEnv( HelpPathEnvironmentVar ) <> '';
  BookshelfOK := GetEnv( BookshelfEnvironmentVar ) <> '';
  if HelpOK and BookshelfOK then
    exit;

  if BookshelfOK then
    MessageBox( 'The '
                + HelpPathEnvironmentVar
                + ' environment variable, used to find help files, '
                + ' is not defined. '
                + 'You may have difficulties launching help.',
                mtWarning,
                [ mbOK ] )
  else if HelpOK then
    MessageBox( 'The ' + EndLine
                + BookshelfEnvironmentVar + EndLine
                + ' environment variable, used to find help files, ' + EndLine
                + ' is not defined. ' + EndLine
                + 'You may have difficulties launching help.',
                mtWarning,
                [ mbOK ] )
  else
    MessageBox( 'The environment variables, '
                + HelpPathEnvironmentVar
                + ' and '
                + BookshelfEnvironmentVar
                + ' used for finding help files, are not defined. '
                + 'You may have difficulties launching help.',
                mtWarning,
                [ mbOK ] )

end;

Procedure TMainForm.WMOpened( Var Msg: TMessage );
var
  X, Y, W, H: longint;
  Maximize: boolean;
begin
  SetLayout;

  ApplySettings;

  EnableControls;

  ReadWindowPos( X, Y, W, H, Maximize );
  WindowState:= wsNormal;
  SetWindowPos( X, Y, W, H );
  if Maximize then
    WindowState:= wsMaximized;

  Application.ProcessMessages;

  CheckEnvironmentVars;

  ParseParameters;
end;

Procedure TMainForm.ParseParameters;
var
  NeedShowUsage: boolean;
  ParamIndex: longint;
  Param: string;
  Filename: string;
  Topic: string;
begin
  ProfileEvent( 'ParseParameters' );
  Filename:= '';
  Topic:= '';
  NeedShowUsage:= false;

  for ParamIndex:= 1 to ParamCount do
  begin
    Param:= UpperCase( ParamStr( ParamIndex ) );
    if ( Param = '/?' )
       or ( Param = '-?' )
       or ( Param = '?' )
       or ( Param = '/H' )
       or ( Param = '-H' ) then
    begin
      NeedShowUsage:= true
    end
    else if ( Param = '/PROFILE' ) then
    begin
      DoProfiling:= true
    end
    else
    begin
      if Filename = '' then
        Filename:= Param
      else if Topic = '' then
        Topic:= Param
      else
        NeedShowUsage:= true;
    end;
  end;

  if NeedShowUsage then
    ShowUsage
  else
  begin
    if Filename > '' then
    begin
      ProfileEvent( 'Call OpenFile' );

      OpenFile( Filename );
      if Topic <> '' then
      begin
        ProfileEvent( 'Do search for topic' );
        Tabset.TabIndex:= piSearch;
        SearchTextEdit.Text:= Topic;
        SearchButton.Click;
      end;
    end;
  end;

end;

Procedure TMainForm.WMFollowLink( Var Msg: TMessage );
var
  HelpFile: THelpFile;
  NewTopic: TTopic;
  Link: THelpLink;
  Window: THelpWindow;
begin
  Link:= THelpLink( Msg.Param1 );
  Window:= THelpWindow( Msg.Param2 );
  HelpFile := TopicFile( Window.Topic );
  NewTopic:= HelpFile.Topics[ Link.TopicIndex ];

  if not NewTopic.ShowInContents then
  begin
    // topic does not appear in contents
    // assume it is a link to a different window...
    UpdateCurrentNavigatePoint;
    FollowLink( Link, Window );
    SaveNavigatePoint;
  end
  else
  begin
    DisplayTopic( NewTopic );
  end;

end;

Procedure TMainForm.MainFormOnResize (Sender: TObject);
Begin
  if not Visible then
    exit;
  if Handle = 0 then
    exit;

  SetLayout;

End;

Procedure TMainForm.VSplitBarOnChange (NewSplit: LongInt);
Begin
  SetLayout;
End;

Procedure TMainForm.SetLayout;
var
  x: longint;
Begin
  Notebook.YStretch:= ysFrame;
  Tabset.YAlign:= yaTop;
//  RTView.YStretch:= ysFrame;
  DisplayPanel.YStretch:= ysFrame;

  Notebook.Left:=0;
  Notebook.Width:= VSplitBar.Left;
  Tabset.Left:= 0;
  Tabset.Width:= VSplitBar.Left;
//  RTView.Left:= VSplitBar.Left + VSplitBar.Width;
//  RTView.Width:= ClientWidth - RTView.Left;
x:= VSplitBar.Left + VSplitBar.Width;
  DisplayPanel.Left:= VSplitBar.Left + VSplitBar.Width;
  DisplayPanel.Width:= ClientWidth - DisplayPanel.Left;
//  NotesLabel.Left:= RTView.Left;
//  NotesMemo.Left:= NotesLabel.Left + NotesLabel.Width;
//  NotesMemo.Width:= ClientWidth - NotesMemo.Left;
//  NotesMemo.YAlign:= yaTop;
//  NotesLabel.YAlign:= yaTop;
  ProgressBar.Left:= ClientWidth div 2;
  ProgressBar.Width:= ClientWidth - ProgressBar.Left;
  StatusLabel.Left:= 0;
  StatusLabel.Width:= ClientWidth div 2;
  VSplitBar.YStretch:= ysFrame;

  LayoutWindows( Windows );
End;

Procedure TMainForm.LayoutWindows( WindowList: TList );
var
  Window: THelpWindow;
  WindowIndex: longint;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    Window:= WindowList[ WindowIndex ];
    SetWindowLayout( Window );
    LayoutWindows( Window.ChildWindows );
  end;
end;

Procedure TMainForm.FindNextMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
begin
  Window:= GetActiveWindow;
  if Window = nil then
  begin
    ShowMessage( 'Click in a window first' );
    exit;
  end;
  if Window.View.Find( foFromCurrent, FindText ) = -1 then
    Beep( 1000, 100 );
End;

Procedure TMainForm.FindMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
begin
  Window:= GetActiveWindow;
  if Window = nil then
  begin
    ShowMessage( 'Click in a window first' );
    exit;
  end;
  if InputQuery( 'Find', 'Enter text to find', FindText ) then
    if Window.View.Find( foFromStart, FindText ) = -1 then
      Beep( 1000, 100 );

End;

Procedure TMainForm.IndexSearchEditOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  case KeyCode of
    kbCUp:
    begin
      if IndexListBox.ItemIndex > 0 then
        IndexListBox.ItemIndex:= IndexListBox.ItemIndex - 1;
      KeyCode:= kbNull;
    end;

    kbCDown:
    begin
      if IndexListBox.ItemIndex < IndexListBox.Items.Count - 1 then
        IndexListBox.ItemIndex:= IndexListBox.ItemIndex + 1;
      KeyCode:= kbNull;
    end;

    kb_VK + VK_NEWLINE:
    begin
//      RTView.Focus;
    end;
  end;

{  if ( KeyCode = kbBkSp )
     and ( IndexSearchEdit.SelLength > 0 ) then
  begin
    if IndexSearchEdit.SelStart > 0 then
    begin
      IndexSearchEdit.SelStart:= IndexSearchEdit.SelStart - 1;
      IndexSearchEdit.SelLength:= IndexSearchEdit.SelLength + 1;
    end;

    WinSendMsg( IndexSearchEdit.Handle,
                EM_CLEAR,
                0, 0 );
    KeyCode:= kbNull;
  end;}

End;

Procedure TMainForm.IndexSearchEditOnChange (Sender: TObject);
var
  MatchIndex: longint;
  IndexINdex: longint;
  SearchText: string;
Begin
  if InIndexSearch then
    exit;

  MatchIndex:= -1;
  SearchText:= trim( IndexSearchEdit.Text );
  for IndexIndex:= 0 to DisplayedIndex.Count - 1 do
  begin
    if StrStarts( SearchText, DisplayedIndex[ IndexIndex ] ) then //IndexEntry ) then
    begin
      MatchIndex:= IndexIndex;
      break;
    end;
  end;

  if MatchIndex = -1 then
    exit;

  InIndexSearch:= true;

  if IndexListBox.ItemIndex <> MatchIndex then
    IndexListBox.ItemIndex:= MatchIndex;

  InIndexSearch:= false;
End;

Procedure TMainForm.FileInformationMIOnClick (Sender: TObject);
var
  FileIndex: longint;
  HelpFile: THelpFile;

  TotalTopicCount: longint;
  TotalIndexCount: longint;

Begin
  TotalTopicCount:= 0;
  TotalIndexCount:= 0;

  for FileIndex:= 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];

    inc( TotalTopicCount, HelpFile.TopicCount );
    inc( TotalIndexCount, HelpFile.Index.Count );
  end;

  with FileInformationForm.InformationMemo do
  begin
    BeginUpdate;
    Lines.Clear;
    Lines.Add( 'Title: ' + THelpFile( Files[ 0 ] ).Title );
    for FileIndex:= 0 to Files.Count - 1 do
    begin
      HelpFile:= Files[ FileIndex ];
      Lines.Add( 'Filename: ' + HelpFile.FileName );
    end;

    Lines.Add( '' );
    Lines.Add( 'Topic Count: '
               + IntToStr( TotalTopicCount ) );
    Lines.Add( 'Index Count: '
               + IntToStr( TotalIndexCount ) );

    EndUpdate;
  end;
  FileInformationForm.ShowModal;
End;

Procedure TMainForm.SearchResultsListBoxOnItemFocus (Sender: TObject;
  Index: LongInt);
var
  Topic: TTopic;
Begin
  if SearchResultsListBox.Items.Objects[ Index ] = nil then
    exit;
  Topic:= SearchResultsListBox.Items.Objects[ Index ] as TTopic;
  DisplayTopic( Topic );
End;

Procedure TMainForm.SearchTextEditOnScan (Sender: TObject;
  Var KeyCode: TKeyCode);
Begin
  case KeyCode of
    kbCUp:
    begin
      if SearchResultsListBox.ItemIndex > 0 then
        SearchResultsListBox.ItemIndex:= SearchResultsListBox.ItemIndex - 1;
      KeyCode:= kbNull;
    end;

    kbCDown:
    begin
      if SearchResultsListBox.ItemIndex < SearchResultsListBox.Items.Count - 1 then
        SearchResultsListBox.ItemIndex:= SearchResultsListBox.ItemIndex + 1;
      KeyCode:= kbNull;
    end;
  end;
End;

Procedure TMainForm.SearchButtonOnClick (Sender: TObject);
var
  SearchText: string;
  SearchResults: TList;
  FileIndex: longint;
  HelpFile: THelpFile;
  TopicIndex: longint;
  Topic: TTopic;
  Query: TTextSearchQuery;
Begin
  SearchText:= trim( SearchTextEdit.Text );
  if SearchText = '' then
    exit;

  try
    Query := TTextSearchQuery.Create( SearchText );
  except
    on e: ESearchSyntaxError do
    begin
      ShowMessage( 'Error in search syntax: '
                   + e.Message );
      exit;
    end;
  end;

  Cursor:= crHourGlass;

  SearchResults:= TList.Create;

  // Search open help file
  for FileIndex:= 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    SearchHelpFile( HelpFile,
                    Query,
                    SearchResults,
                    HelpFile.HighlightWords );
  end;

  // Sort results across all files by relevance
  SearchResults.Sort( TopicRelevanceCompare );

  // Load topics into search results list.
  SearchResultsListBox.BeginUpdate;

  SearchResultsListBox.Clear;
  for TopicIndex := 0 to SearchResults.Count - 1 do
  begin
    Topic := SearchResults[ TopicIndex ];
    SearchResultsListBox.Items.AddObject( Topic.Title
                                          + ' ['
                                          + IntToStr( Topic.SearchRelevance )
                                          + ']',
                                          Topic );
  end;

  SearchResultsListBox.ItemIndex := -1;
  SearchResultsListBox.EndUpdate;
  Application.ProcessMessages; // make sure list gets displayed

  Query.Destroy;
  SearchResults.Destroy;

  if SearchResultsListBox.Items.Count > 0 then
  begin
    SearchResultsListBox.ItemIndex:= 0;
  end
  else
    SearchResultsListBox.Items.Add( '(No matches found for "' + SearchText + '")' );

  Cursor:= crDefault;
End;

Procedure TMainForm.FileSaveAsMIOnClick (Sender: TObject);
var
  F: File;
  EntryText: PChar;
  Window: THelpWindow;
  Filename: string;
Begin
  Window:= GetActiveWindow;
  if Window = nil then
  begin
    ShowMessage( 'Before saving, click in the window you want to save.' );
    exit;
  end;
  FileDialogForm.AllowMultipleFiles:= false;
  FileDialogForm.Caption:= 'Save Topic';
  FileDialogForm.FilterComboBox.Filter := 'All files (*.*)|*.*';
  if FileDialogForm.ShowModal = mrOK then
  begin
    Filename:= FileDialogForm.Filenames[ 0 ];
    if FileExists( Filename ) then
      if not DoConfirmDlg( 'Save',
                           'Replace existing file '
                           + Filename
                           + '?' ) then
        exit;
    System.Assign( F, Filename );
    Rewrite( F );
    if IOResult <> 0 then
    begin
      ShowMessage( 'Could not write to file ' + Filename );
      exit;
    end;
    EntryText:= StrAlloc( Window.View.TextEnd + 1 );
    Window.View.CopyToBuffer( EntryText );
    System.BlockWrite( F, EntryText^, strlen( EntryText ) );
    StrDispose( EntryText );
    System.Close( F );
  end;
End;

Procedure TMainForm.OptionsMIOnClick (Sender: TObject);
Begin
//  OptionsForm.MyParent := self;
  if OptionsForm.ShowModal = mrOK then
    ApplySettings;
End;

Procedure TMainForm.ShowUsage;
begin
  ShowMessage( 'Usage: NewView [filename [topic]]' );
end;

Procedure TMainForm.IndexListBoxOnItemFocus (Sender: TObject; Index: LongInt);
var
  Topic: TTopic;
Begin
  Topic:= DisplayedIndex.Objects[ Index ] as TTopic;
  DisplayTopic( Topic );
End;

Procedure TMainForm.TabSetOnChange (Sender: TObject; NewTab: LongInt;
  Var AllowChange: Boolean);
Begin
  NoteBook.PageIndex:= NewTab;
End;

Procedure TMainForm.NotebookOnSetupShow (Sender: TObject);
Begin
  SearchTextEdit.yAlign:= yaTop;
  SearchTextEdit.xStretch:= xsFrame;
  SearchButton.yAlign:= yaTop;
  SearchButton.xAlign:= xaRight;
  SearchResultsListBox.yStretch:= ysFrame;
  IndexListBox.yStretch:= ysFrame;
  IndexListBox.xStretch:= xsFrame;
  NotesListBox.xStretch:= xsFrame;
  NotesListBox.yStretch:= ysFrame;
End;

Procedure TMainForm.EnableControls;
var
  BackEnabled: boolean;
  ForwardEnabled: boolean;
  FileOpen: boolean;
begin
  BackEnabled:= CurrentHistoryIndex > 0;
  ForwardEnabled:= CurrentHistoryIndex < PageHistory.Count - 1;
  FileOpen:= Files.Count > 0;//_HelpFile <> nil;

  Coolbar.Sections[ ciBack ].Disabled:= not BackEnabled;
  NavigateBackMI.Enabled:= BackEnabled;
  Coolbar.Sections[ ciForward ].Disabled:= not ForwardEnabled;
  NavigateForwardMI.Enabled:= ForwardEnabled;

  FileSaveAsMI.Enabled:= FileOpen;

  Coolbar.Sections[ ciPrint ].Disabled:= not FileOpen;
  PrintMI.Enabled:= FileOpen;
  FileInformationMI.Enabled:= FileOpen;

  Coolbar.Sections[ ciAddBookmark ].Disabled:= not FileOpen;
  AddBookmarkMI.Enabled:= FileOpen;
  Coolbar.Sections[ ciAddNote ].Disabled:= not FileOpen;
  AddNoteMI.Enabled:= FileOpen;

  FileCloseMI.Enabled:= FileOpen;

  EnableSearchButton;
  EnableNotesControls;
end;

Procedure TMainForm.NavigateBackMIOnClick (Sender: TObject);
begin
  NavigateBack;
end;

Procedure TMainForm.SaveNavigatePoint;
var
  NavPoint: TNavigatePoint;
begin
  // delete rest of history.
  while CurrentHistoryIndex < PageHistory.Count - 1 do
  begin
    NavPoint:= PageHistory.Objects[ CurrentHistoryIndex + 1 ] as TNavigatePoint;
    NavPoint.Destroy;
    PageHistory.Delete( CurrentHistoryIndex + 1 );
  end;

  NavPoint:= TNavigatePoint.Create;
  SaveWindows( Windows, NavPoint.Windows, nil );

  if ContentsOutline.SelectedNode <> nil then
    NavPoint.ContentsTopic:= ContentsOutline.SelectedNode.Data as TTopic
  else
    NavPoint.ContentsTopic:= nil;

  if CurrentTopic <> nil then
    PageHistory.AddObject( CurrentTopic.Title, NavPoint )
  else
    PageHistory.AddObject( '(Blank)', NavPoint );
  inc( CurrentHistoryIndex );

end;

Procedure TMainForm.UpdateCurrentNavigatePoint;
var
  NavPoint: TNavigatePoint;
begin
  if CurrentHistoryIndex = -1 then
    exit;

  NavPoint:= PageHistory.Objects[ CurrentHistoryIndex ] as TNavigatePoint;

  DestroyListObjects( NavPoint.Windows );
  NavPoint.Windows.Clear;

  SaveWindows( Windows, NavPoint.Windows, nil );
end;

Procedure TMainForm.SaveWindows( SourceList: TList;
                                 DestList: TList;
                                 Parent: TSavedHelpWindow );

var
  WindowIndex: longint;
  Window: THelpWindow;
  WindowCopy: TSavedHelpWindow;
begin
  // limit storage to only what's need since list will be static.
  DestList.Capacity:= SourceList.Count;
  for WindowIndex:= 0 to SourceList.Count - 1 do
  begin
    Window:= SourceList[ WindowIndex ];
    WindowCopy:= TSavedHelpWindow.Create( Window );
    WindowCopy.Parent:= Parent;
    WindowCopy.Rect:= Window.Rect;
    WindowCopy.Topic:= Window.Topic;
    WindowCopy.Group:= Window.Group;
    WindowCopy.TopCharIndex:= Window.View.TopCharIndex;
    SaveWindows( Window.ChildWindows, WindowCopy.ChildWindows, WindowCopy );
    DestList.Add( WindowCopy );
  end;
end;

Procedure TMainForm.DisplayWindows( WindowList: TList;
                                    Parent: THelpWindow );
var
  WindowIndex: longint;
  WindowCopy: TSavedHelpWindow;
  NewWindow: THelpWindow;
begin
  for WindowIndex:= 0 to WindowList.Count - 1 do
  begin
    WindowCopy:= WindowList[ WindowIndex ];
    NewWindow:= OpenWindow( WindowCopy.Topic,
                            WindowCopy.Group,
                            Parent,
                            WindowCopy.Rect );
    NewWindow.View.TopCharIndex := WindowCopy.TopCharIndex;
    DisplayWindows( WindowCopy.ChildWindows, NewWindow );
  end;
end;

Procedure TMainForm.NavigateToPoint( NavPoint: TNavigatePoint );
begin
  Navigating:= true;
  CloseWindows;
  DisplayWindows( NavPoint.Windows, nil );
  ContentsOutline.SetSelectedObject( NavPoint.ContentsTopic );

  EnableControls;
  Navigating:= false;
end;

Procedure TMainForm.NavigateForward;
var
  NavPoint: TNavigatePoint;
Begin
  if CurrentHistoryIndex < PageHistory.Count - 1 then
  begin
    UpdateCurrentNavigatePoint;
    inc( CurrentHistoryIndex );
    NavPoint:= PageHistory.Objects[ CurrentHistoryIndex ] as TNavigatePoint;
    NavigateToPoint( NavPoint );
  end;
End;

Procedure TMainForm.NavigateBack;
var
  NavPoint: TNavigatePoint;
Begin
  if CurrentHistoryIndex > 0 then
  begin
    UpdateCurrentNavigatePoint;
    dec( CurrentHistoryIndex );
    NavPoint:= PageHistory.Objects[ CurrentHistoryIndex ] as TNavigatePoint;
    NavigateToPoint( NavPoint );
  end;
End;

Procedure TMainForm.NavigatePreviousInContents;
begin
  ContentsOutline.GotoNextNodeUp;
end;

Procedure TMainForm.NavigateNextInContents;
begin
  ContentsOutline.GotoNextNodeDown;
end;

Procedure TMainForm.InsertNotesIntoTopic( Window: THelpWindow );
var
  NoteIndex: longint;
  Note: THelpNote;
  ActualInsertPoint: longword;
begin
  NoteIndex:= 0;
  for NoteIndex:= 0 to Notes.Count - 1 do
  begin
    Note:= Notes[ NoteIndex ];
    if Note.Topic = Window.Topic then
    begin
         // Adjust insert point for any notes we have already inserted.
      ActualInsertPoint:= FindActualNoteCharIndex( Note.InsertPoint,
                                                   NoteIndex,
                                                   Window.Topic );
      Window.View.InsertText( ActualInsertPoint, Note.InsertText );
    end;
  end;
end;

Procedure TMainForm.NavigatePreviousMIOnClick (Sender: TObject);
Begin
  NavigatePreviousInContents;
End;

Procedure TMainForm.NavigateNextMIOnClick (Sender: TObject);
Begin
  NavigateNextInContents;
End;

Function TMainForm.GetActiveWindow: THelpWindow;
var
  View: TRichTextView;
  FirstWindow: THelpWindow;
begin
  Result:= nil;
  if ActiveControl is TRichTextView then
  begin
    View:= ActiveControl as TRichTextView;
    Result:= FindWindowFromView( View, Windows );
  end
  else if Windows.Count = 1 then
  begin
    FirstWindow:= Windows[ 0 ];
    if FirstWindow.ChildWindows.Count = 0 then
      Result:= FirstWindow;
  end;
end;

Procedure TMainForm.CopyMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
begin
  Window:= GetActiveWindow;
  if Window = nil then
  begin
    ShowMessage( 'Select some text first' );
    exit;
  end;
  Window.View.CopySelectionToClipboard;
End;

Procedure TMainForm.SelectAllMIOnClick (Sender: TObject);
var
  Window: THelpWindow;
begin
  Window:= GetActiveWindow;
  if Window = nil then
  begin
    ShowMessage( 'Click in a text window first' );
    exit;
  end;
  Window.View.SelectAll;
End;

Procedure TMainForm.DebugMIOnClick (Sender: TObject);
Begin
  Debug:= not Debug;
  DebugMI.Checked:= Debug;
  RefreshWindows( Windows );
End;

Procedure TMainForm.HelpAboutMIOnClick (Sender: TObject);
Begin
  AboutBox.ShowModal;
End;

Procedure TMainForm.OnOverLink ( Sender: TRichTextView; LinkString: String);
var
  Link: THelpLink;
  LinkIndex: longint;
  Window: THelpWindow;
Begin
  if StrLeft( LinkString, 4 ) = 'note' then
    SetStatus( 'Click to edit note' )
  else
  begin
    Window:= FindWindowFromView( Sender, Windows );
    LinkIndex:= StrToInt( LinkString );
    Link:= Window.Topic.Links[ LinkIndex ];

     // Note: assumption that links are only to the same file!
    SetStatus( 'Link to "'
               + trim( TopicFile( Window.Topic ).Topics[ Link.TopicIndex ].Title )
               + '"' );
  end;
End;

Procedure TMainForm.OnNotOverLink ( Sender: TRichTextView; LinkString: String);
Begin
  SetStatus( '' );
end;

Procedure TMainForm.OnClickLink ( Sender: TRichTextView; LinkString: String);
var
  Link: THelpLink;
  LinkIndex: longint;
  Window: THelpWindow;
  NoteIndex: longint;
Begin
  if StrLeft( LinkString, 4 ) = 'note' then
  begin
    NoteIndex := StrToInt( StrRightFrom( LinkString, 5 ) );
    NotesListBox.ItemIndex := NoteIndex;
    EditNote( NoteIndex );
  end
  else
  begin
    Window:= FindWindowFromView( Sender, Windows );
    LinkIndex:= StrToInt( LinkString );
    Link:= Window.Topic.Links[ LinkIndex ];

    PostMsg( Self.Handle,
             WM_FOLLOWLINK,
             longint( Link ),
             longint( Window ) );

  end;
End;

Procedure TMainForm.OnWindowCloseClick( Sender: TObject);
var
  Window: THelpWindow;
  ParentWindow: THelpWindow;
  WindowIndex: longint;
Begin
  UpdateCurrentNavigatePoint;
  Window := THelpWindow( TControl( Sender ).Parent );
  if Window.Parent = DisplayPanel then
  begin
    WindowIndex := Windows.IndexOf( Window );
    Windows.Delete( WindowIndex );
  end
  else
  begin
    ParentWindow := Window.Parent as THelpWindow;
    WindowIndex := ParentWindow.ChildWindows.IndexOf( Window );
    ParentWindow.ChildWindows.Delete( WindowIndex );
  end;
  Window.Destroy;
  SaveNavigatePoint;
  EnableControls;
End;

Procedure TMainForm.BackButtonOnClick (Sender: TObject);
Begin
  NavigateBack;
End;

Procedure TMainForm.RTViewOnSetupShow (Sender: TObject);
Begin
End;

Procedure TMainForm.ExitMIOnClick (Sender: TObject);
Begin
  Close;
End;

Procedure TMainForm.CreateMRUMenuItems;
var
  MenuItem: TMenuItem;
  i: integer;
  FileName: string;
  FileNames: string;
  MRUText: string;
begin
  for i:= 0 to MRUMenuItems.Count - 1 do
  begin
    MenuItem:= MRUMenuItems[ i ];
    MenuItem.Destroy;
  end;

  MRUMenuItems.Clear;
  if Settings.MRUList.Count > 0 then
  begin
    MenuItem:= TMenuItem.Create( self );
    MenuItem.Caption:= '-';
    FileMenu.Add( MenuItem );
    MRUMenuItems.Add( MenuItem );
  end;

  for i:= 0 to Settings.MRUList.Count -1 do
  begin
    MenuItem:= TMenuItem.Create( self );

    FileNames:= Settings.MRUList[ i ];
    MRUText:= '';
    while FileNames <> '' do
    begin
      FileName:= ExtractNextValue( FileNames, '+' );
      AddToListString( MRUText, ExtractFileName( FileName ), '+' );
    end;

    MenuItem.Caption:= '~'
                       + IntToStr( i + 1 )
                       + '. '
                       + MRUText;
    MenuItem.OnClick:= MRUItemClick;
    MenuItem.Tag:= i;
    FileMenu.Add( MenuItem );
    MRUMenuItems.Add( MenuItem );
  end;
end;

procedure TMainForm.MRUItemClick( Sender: TObject );
var
  Tag: longint;
  MenuItem: TMenuItem;
  FileNames: string;
begin
  MenuItem:= Sender as TMenuItem;
  Tag:= MenuItem.Tag;
  FileNames:= Settings.MRUList[ Tag ];
  OpenFile( FileNames );
end;

{Procedure SkipChildTopics( HelpFile: THelpFile;
                           Level: longint;
                           Var TopicIndex: longint )
begin
  while TopicIndex < HelpFile.TopicCount do
  begin
    Topic:= HelpFile.Topics[ TopicIndex ];
    if Topic.ContentsLevel < Level then
      break;
    assert( not Topic.ShowInContents, 'Non-hidden topic under hidden topic!' );
  end;
end;
}
Procedure TMainForm.AddChildNodes( HelpFile: THelpFile;
                                   ParentNode: TNode;
                                   Level: longint;
                                   Var TopicIndex: longint );
var
  Topic: TTopic;
  Node: TNode;
begin
//  ProfileEvent( 'AddChildNodes from ' + IntToStr( TopicIndex ) );
  assert( ParentNode <> nil );
  Node := nil;
  while TopicIndex < HelpFile.TopicCount do
  begin
    Topic:= HelpFile.Topics[ TopicIndex ];
//    ProfileEvent( 'Topic #' + IntToStr( TopicIndex )
//                  + ' level ' + IntToStr( Topic.ContentsLevel )
//                  + ' title ' + Topic.Title );

    if Topic.ShowInContents then
    begin
      if Topic.ContentsLevel < Level then
        break;

      if Topic.ContentsLevel = Level then
      begin
        Node:= ParentNode.AddChild( Topic.Title,
                                    Topic );
        inc( TopicIndex );
      end
      else
      begin
        assert( Node <> nil );
        AddChildNodes( HelpFile,
                       Node,
                       Topic.ContentsLevel,
                       TopicIndex );
        Node := nil;
      end
    end
    else
    begin
//      ProfileEvent( '  Skipping hidden' );
      inc( TopicIndex );
    end;
  end;

//  ProfileEvent( 'Exit AddChildNodes at ' + IntToStr( TopicIndex ) );

end;

Procedure TMainForm.LoadContents;
var
  TopicIndex: longint;
  Topic: TTopic;
  Node: TNode;
  FileIndex: longint;
  HelpFile: THelpFile;
begin
  ContentsOutline.BeginUpdate;
  ProfileEvent( 'Load contents outline' );

  ContentsOutline.Clear;

  ProfileEvent( 'Loop files' );

  Node := nil;

  for FileIndex:= 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    ProfileEvent( 'File ' + IntToStr( FileIndex ) );
    TopicIndex:= 0;
    while TopicIndex < HelpFile.TopicCount do
    begin
      Topic:= HelpFile.Topics[ TopicIndex ];
//      ProfileEvent( 'Topic #' + IntToStr( TopicIndex )
//                    + ' level ' + IntToStr( Topic.ContentsLevel )
//                    + ' title ' + Topic.Title );
      assert( Topic.ContentsLevel >= 0,
              'Topic contents level is ' + IntToStr( Topic.ContentsLevel ) );
      if Topic.ShowInContents then
      begin
        if Topic.ContentsLevel = 1 then
        begin
          Node:= ContentsOutline.AddChild( Topic.Title,
                                           Topic );
          inc( TopicIndex );
        end
        else
        begin
          // subnodes
          assert( Node <> nil, 'No level 1 topic for subnodes!' );
          AddChildNodes( HelpFile,
                         Node,
                         Topic.ContentsLevel,
                         TopicIndex );
          Node := nil;
        end;
      end
      else
      begin
//        ProfileEvent( '  Skipping hidden' );
        inc( TopicIndex );
      end;
    end;
  end;
  ProfileEvent( '  EndUpdate' );
  ContentsOutline.EndUpdate;
  ProfileEvent( '  Contents loaded' );

end;

Procedure TMainForm.SaveNotes( HelpFile: THelpFile );
var
  NotesFileName: string;
  TopicIndex: longword;
  Note: THelpNote;
  NoteIndex: longint;

  NotesFile: HFile;
  OpenAction: ULong;
  rc: APIRET;
  CName: Cstring;
  FileNoteCount: integer;

begin
  if not NotesChangedSinceLoad then
    exit;

  NotesFileName:= ChangeFileExt( HelpFile.FileName, '.nte' );

  FileNoteCount:= 0;
  for  NoteIndex:= 0 to Notes.Count - 1 do
  begin
    Note:= Notes[ NoteIndex ];

    if Note.Topic.HelpFile = HelpFile then
      inc( FileNoteCount );
  end;

  if FileNoteCount = 0 then
  begin
    if FileExists( NotesFileName ) then
      DeleteFile( NotesFileName );
    exit;
  end;

  CName:= NotesFileName;
  rc:= DosOpen( CName,
                NotesFile,
                OpenAction,
                0, // file size
                0, // attrs
                OPEN_ACTION_CREATE_IF_NEW + OPEN_ACTION_REPLACE_IF_EXISTS,
                OPEN_SHARE_DENYREADWRITE + OPEN_ACCESS_WRITEONLY,
                nil ); // no eas
  if rc<> 0 then
  begin
    case rc of
      ERROR_ACCESS_DENIED:
        ShowMessage( 'Could not access notes file for writing: ' + NotesFileName );

      ERROR_SHARING_VIOLATION:
        ShowMessage( 'Notes file in use by another program: ' + NotesFileName );

      else
        ShowMessage( 'Could not write to notes file: ' + NotesFileName );
    end;
    exit;
  end;

  for  NoteIndex:= 0 to Notes.Count - 1 do
  begin
    Note:= Notes[ NoteIndex ];

    if Note.Topic.HelpFile <> HelpFile then
      continue;

    TopicIndex:= HelpFile.IndexOfTopic( Note.Topic );

    MyWriteLn( NotesFile,
               IntToStr( TopicIndex ));
    MyWriteLn( NotesFile,
               IntToStr( Note.InsertPoint ) );

    MyWrite( NotesFile,
             Note.Text,
             StrLen( Note.Text ) );

    MyWriteLn( NotesFile,
               '' );
    MyWriteLn( NotesFile,
               '#ENDNOTE#' );

  end;

  DosClose( NotesFile );
end;

Procedure TMainForm.LoadNotes( HelpFile: THelpFile );
var
  NotesFileName: string;
  TopicIndex: longint;
  InsertPoint: longint;
  Note: THelpNote;

  NotesFile: HFile;
  OpenAction: ULong;
  rc: APIRET;
  CName: Cstring;

  Buffer: PChar;
  NotEOF: boolean;
  NoteText: PChar;

begin
  NotesFileName:= ChangeFileExt( HelpFile.FileName, '.nte' );

  if not FileExists( NotesFileName ) then
    // no notes
    exit;

  CName:= NotesFileName;
  rc:= DosOpen( CName,
                NotesFile,
                OpenAction,
                0, // file size - irrelevant, not creating,
                0, // attrs - ''
                OPEN_ACTION_OPEN_IF_EXISTS,
                OPEN_SHARE_DENYREADWRITE + OPEN_ACCESS_READONLY,
                nil ); // no eas
  if rc<> 0 then
  begin
    case rc of
      ERROR_ACCESS_DENIED:
        ShowMessage( 'Could not open notes file: ' + NotesFileName );

      ERROR_SHARING_VIOLATION:
        ShowMessage( 'Notes file in use by another program: ' + NotesFileName );

      else
        ShowMessage( 'Could not write to notes file: ' + NotesFileName );
    end;
    exit;
  end;

  Buffer:= StrAlloc( 256 );
  NoteText:= StrAlloc( 256 );

  NotEOF:= true;

  while NotEOF do
  begin
    // Read contents index
    NotEOF:= MyReadParagraph( NotesFile, Buffer );
    if not NotEOF then
      continue;
    try
      TopicIndex:= StrToInt( StrPas( Buffer ) );
    except
      TopicIndex:= -1;
    end;

    // Read insert point
    NotEOF:= MyReadParagraph( NotesFile, Buffer );
    if not NotEOF then
      continue;
    try
      InsertPoint:= StrToInt( StrPas( Buffer ) );
    except
      InsertPoint:= -1;
    end;

    StrCopy( NoteText, '' );

    while NotEOF do
    begin
      NotEOF:= MyReadParagraph( NotesFile, Buffer );
      if StrIComp( Buffer, '#ENDNOTE#' ) = 0 then
      begin
        // found end of note
        if ( TopicIndex >= 0 )
           and ( InsertPoint >= 0 ) then
        begin
          Note:= THelpNote.Create;
          Note.Topic:= HelpFile.Topics[ TopicIndex ];
          Note.InsertPoint:= InsertPoint;
          Note.Text:= StrNew( NoteText );
          Notes.Add( Note );
          RefreshNoteInsertInfo( Notes.Count - 1 );
        end;
        break;
      end;
      if NoteText[ 0 ] <> #0 then
        AddAndResize( NoteText, #13 + #10 );
      AddAndResize( NoteText, Buffer );
    end;

  end;
  DosClose( NotesFile );

  UpdateNotesDisplay;
  NotesChangedSinceLoad:= false;

end;

Procedure TMainForm.UpdateNotesDisplay;
var
  NoteIndex: longint;
  Note: THelpNote;
  NoteTitle: string;
begin
  NotesListBox.Clear;
  for NoteIndex:= 0 to Notes.Count - 1 do
  begin
    Note:= Notes[ NoteIndex ];

    if Note.Topic > nil then
      NoteTitle := Note.Topic.Title
    else
      NoteTitle := StrLeft( StrPas( Note.Text ), 100 );
    NotesListBox.Items.AddObject( NoteTitle,
                                  Note );
  end;
  EnableNotesControls;
end;

Procedure TMainForm.EnableNotesControls;
var
  NoteSelected: boolean;
begin
  NoteSelected:= NotesListBox.ItemIndex <> -1;
  EditNoteButton.Enabled:= NoteSelected;
  GotoNoteButton.Enabled:= NoteSelected;
  DeleteNoteButton.Enabled:= NoteSelected;
  AddNoteButton.Enabled := Files.Count > 0;
end;

Procedure TMainForm.CloseFile;
var
  FileIndex: longint;
  HelpFile: THelpFile;
  M1: longint;
begin
  Caption:= 'New View - No file';
  CloseWindows;
  ContentsOutline.SelectedNode:= Nil;

  M1:= MemAvail;

  ContentsOutline.Clear;

  ProfileEvent( 'Free contents: ' + IntToStr( MemAvail - M1 ) );
  M1:= MemAvail;

  DisplayedIndex.Clear;
  IndexListBox.Clear;
  ProfileEvent( 'Clear index ' + IntToStr( MemAvail - M1 ) );
  M1:= MemAvail;

  NotesListBox.Clear;
  SearchResultsListBox.Clear;

  ProfileEvent( 'Notes, search etc ' + IntToStr( MemAvail - M1 ) );
  M1:= MemAvail;

  // First save notes. It's important we do this first
  // since we scan all notes each time to find the ones
  // belonging to this file.
  for FileIndex := 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    SaveNotes( HelpFile );
  end;

  // Now destroy help files
  for FileIndex := 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    HelpFile.Free;
  end;

  Files.Clear;

  ProfileEvent( 'Destroy helpfiles ' + IntToStr( MemAvail - M1 ) );
  M1:= MemAvail;


  ClearNotes;

  EnableControls;

end;

Function TMainForm.FindOpenHelpFile( FileName: string ): THelpFile;
var
  FileIndex: longint;
begin
  for FileIndex:= 0 to Files.Count - 1 do
  begin
    Result:= Files[ FileIndex ];
    if StringsSame( Result.Filename, FileName ) then
      // found
      exit;
  end;
  Result:= nil;
end;

// This rather horrendous looking bit of code simply:

// Gets the contents from each file
// Sorts it alphabetically.
// Merges all the sorted contents and indexes together,
// alphabetically.
procedure TMainForm.LoadIndex;
var
  HelpFile: THelpFile;
  TextCompareResult: integer;

  FileIndex: longint;
  TopicIndex: longint;

  Contents: TStringList;
  Lists: TList; // of stringlists...
  ListNextIndex: array[ 0..100 ] of longint;
  Topic: TTopic;

  ListIndex: longint;

  ListEntry: string;
  LowestEntry: string;
  LowestEntryListIndex: longint;
  LowestEntryList: TStringList;
  LowestEntryFile: THelpFile;

  LowestEntryIndexInList: longint;
  List: TStringList;

  CreatedContentsLists: TList;

  LastEntry: string;
begin
  ProfileEvent( 'Create index' );
  ProfileEvent( '  Get/sort lists' );

  ProgressBar.Position:= 70;
  SetStatus( 'Building index... ' );

  Lists:= TList.Create;
  CreatedContentsLists:= TList.Create;

  for FileIndex:= 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    ProgressBar.Position:= 70 + 10 * FileIndex div Files.Count;

    if Settings.IndexStyle in [ isAlphabetical, isFull ] then
    begin
      Contents:= TStringList.Create;
      for TopicIndex:= 0 to HelpFile.TopicCount - 1 do
      begin
        Topic:= HelpFile.Topics[ TopicIndex ];
        if Topic.ShowInContents then
          Contents.AddObject( Topic.Title,
                              Topic );
      end;
      Contents.Sort;
      Lists.Add( Contents );
      CreatedContentsLists.Add( Contents );
    end;
    if Settings.IndexStyle in [ isFileOnly, isFull ] then
      Lists.Add( HelpFile.Index );
  end;

  // Initialise index for each helpfile index
  for ListIndex:= 0 to Lists.Count - 1 do
    ListNextIndex[ ListIndex ]:= 0;

  DisplayedIndex.Clear;
  ProgressBar.Position:= 80;

  ProfileEvent( '  Merge lists' );

  while true do
  begin
    LowestEntry:= '';
    LowestEntryListIndex:= -1;

    // Find alphabetically lowest (remaining) topic

    for ListIndex:= 0 to Lists.Count - 1 do
    begin
      List:= Lists[ ListIndex ];
      if ListNextIndex[ ListIndex ] < List.Count then
      begin
        // list is not yet finished, get next entry

        ListEntry:= List[ ListNextIndex[ ListIndex ] ];

        if LowestEntry <> '' then
          TextCompareResult:= CompareText( ListEntry, LowestEntry )
        else
          TextCompareResult:= -1;
        if TextCompareResult < 0 then
        begin
          // this index entry comes before the lowest one so far
          LowestEntry:= ListEntry;
          LowestEntryListIndex:= ListIndex;
        end;
      end;
    end;

    if LowestEntryListIndex = -1 then
      // we're out
      break;

    LowestEntryList:= Lists[ LowestEntryListIndex ];
    LowestEntryFile:= Files[ LowestEntryListIndex div 2 ]; //since the list contains contents then index for each file
    LowestEntryIndexInList:= ListNextIndex[ LowestEntryListIndex ];

    Topic:= LowestEntryList.Objects[ LowestEntryIndexInList ] as TTopic;

    if LowestEntry <> LastEntry then
      DisplayedIndex.AddObject( LowestEntry,
                                Topic );
    LastEntry:= LowestEntry;

    inc( ListNextIndex[ LowestEntryListIndex ] );

    if LowestEntryListIndex mod 2 = 1 then
    begin
      // found in one of indexes. Check for subsequent indented strings
      List:= Lists[ LowestEntryListIndex ];
      while longint( ListNextIndex[ LowestEntryListIndex ] ) < List.Count do
      begin
        ListEntry:= List[ ListNextIndex[ LowestEntryListIndex ] ];
        if Length( ListEntry ) > 0 then
          if ListEntry[ 1 ] = ' ' then
          begin
            // found one,
            Topic:= LowestEntryList.Objects[ LowestEntryIndexInList ] as TTopic;

            DisplayedIndex.AddObject( ListEntry,
                                      Topic );
            inc( ListNextIndex[ LowestEntryListIndex ] );
          end
          else
            break
        else
          break;
      end;
    end;
  end;

  ProgressBar.Position:= 95;
  ProfileEvent( '  Display index' );

  // Now display the final index list
//  IndexListBox.Items.Assign( DisplayedIndex );
  IndexListBox.Items.Assign( DisplayedIndex );

  ProfileEvent( '  Tidy up' );

  Lists.Destroy;

  // destroy sorted contents lists
  for ListIndex:= 0 to CreatedContentsLists.Count - 1 do
  begin
    List:= CreatedContentsLists[ ListIndex ];
    List.Destroy;
  end;
  CreatedContentsLists.Destroy;
  ProfileEvent( '  Done' );
end;

// Given a filename, which may or may not contain a path or extension,
// finds the actual file. This can involve searching
// the help and bookshelf paths.
Function TMainForm.FindHelpFile( FileName: string ): string;
var
  AlternativeFileName: string;
begin
  Result:= '';

  AlternativeFileName:= '';
  if ExtractFileExt( Filename ) = '' then
  begin
    Filename:= ChangeFileExt( Filename, '.inf' );
    AlternativeFileName:= ChangeFileExt( Filename, '.hlp' );
  end;

  if ExtractFilePath( FileName ) <> '' then
  begin
    // Path specified; just see if it exists
    if FileExists( Filename ) then
      Result:= Filename
    else if FileExists( AlternativeFilename ) then
      Result:= AlternativeFilename;

  end
  else
  begin
    // Path not specified; Search help paths
    if not SearchPath( HelpPathEnvironmentVar,
                       FileName,
                       Result ) then
    begin
      if not SearchPath( BookshelfEnvironmentVar,
                         FileName,
                         Result ) then
      begin
        // Didn't find as specified or as .inf, try .hlp
        if AlternativeFilename <> '' then
        begin
          if not SearchPath( HelpPathEnvironmentVar,
                             AlternativeFileName,
                             Result ) then
            if not SearchPath( BookshelfEnvironmentVar,
                               AlternativeFileName,
                               Result ) then
              Result:= '';
        end;
      end;
    end;
  end;
end;

Procedure TMainForm.OnHelpFileLoadProgress( n, outof: integer;
                                            message: string );
var
  ThisFileSize: longint;
  ProgressOnFiles: longint;
  RelativeSizeOfFile: double;
  Filename: string;
  ProgressOnThisFile: longint;

begin
  Filename:= LoadingFilenameList[ LoadingFileIndex ];

  ThisFileSize:= longint( LoadingFilenameList.Objects[ LoadingFileIndex ] );

  ProgressOnFiles:= round( 100 * LoadingSizeSoFar / LoadingTotalSize );
  RelativeSizeOfFile:= ThisFileSize / LoadingTotalSize;

  ProgressOnThisFile := round( 100 * n / outof * RelativeSizeOfFile );

  ProgressBar.Position:= ( ProgressOnFiles
                           + ProgressOnThisFile ) div 2;
  SetStatus( 'Loading file ' + ExtractFileName( Filename ) + ': ' + message );
  ProgressBar.Show;
end;

Procedure TMainForm.LoadHelpFileTaskOnExecute (Parameters: TObject;
  Context: TStdTaskItem);
begin
end;

Function TMainForm.TranslateEnvironmentVar( Filenames: string ): string;
var
  EnvironmentVarValue: string;
begin
  EnvironmentVarValue:= GetEnv( Uppercase( FileNames ) );
  if DosError = 0 then
    // environment var exists - use it's value
    Result:= EnvironmentVarValue
  else
    Result:= FileNames;
end;

Function TMainForm.OpenFile( FileNames: string ): boolean;
var
  HelpFiles: TList;
  HelpFile: THelpFile;
  FileIndex: longint;
  FileName: string;
  FullFilePath: string;
  PreviousMRUIndex: longint;
  RemainingFileNames: string;
  FileSize: longint;
//  ThreadFiles: TStringList;
begin
  ProfileEvent( 'OpenFile' );

  PageHistory.Clear;
  CurrentHistoryIndex:=-1;
  if DoProfiling then
    StartProfile( 'Profile' );
  Cursor:= crHourGlass;

  HelpFiles:= TList.Create;

  ProfileEvent( 'Translate environment vars' );

  RemainingFileNames:= TranslateEnvironmentVar( Filenames );

  LoadingFilenameList:= TStringList.Create;
  LoadingTotalSize:= 0;

  while RemainingFileNames <> '' do
  begin
    FileName:= ExtractNextValue( RemainingFileNames, '+' );
    ProfileEvent( 'File: ' + FileName );

    FullFilePath:= FindHelpFile( Filename );
    if FullFilePath <> '' then
    begin
      Filename:= FullFilePath;
      FileSize:= GetFileSize( Filename );
      inc( LoadingTotalSize, FileSize );
    end;
    ProfileEvent( '  Full path: ' + FullFilePath );

    LoadingFilenameList.AddObject( Filename, TObject( FileSize ) );
  end;

{  ThreadFiles:= TStringList.Create;
  ThreadFiles.Assign( LoadingFilenameList );
  LoadHelpFileTask.Schedule( ThreadFiles );}

  LoadingSizeSoFar:= 0;
  for FileIndex:= 0 to LoadingFilenameList.Count - 1 do
  begin
    Filename:= LoadingFilenameList[ FileIndex ];
    ProfileEvent( '  Loading: ' + Filename );
    try
      LoadingFileIndex:= FileIndex;
      HelpFile:= THelpFile.Create( FileName,
                                   OnHelpFileLoadProgress );
      inc( LoadingSizeSoFar, longint( LoadingFilenameList.Objects[ FileIndex ] ) );
      HelpFiles.Add( HelpFile );
    except
      on E: Exception do
      begin
        if E is EHelpFileException then
          ShowMessage( 'Could not open '
                       + FileName
                       + ': '
                       + E.Message )
        else
          ShowMessage( 'An error occurred loading '
                       + FileName
                       + '. It may be an damaged help file '
                       + 'or there may be a bug in this program.'
                       + #10 + #10
                       + E.Message );

        Cursor:= crDefault;
        Result:= false;
        while HelpFiles.Count > 0 do
        begin
          HelpFile:= HelpFiles[ 0 ];
          HelpFile.Destroy;
          HelpFiles.Delete( 0 );
        end;
        LoadingFilenameList.Destroy;
        HelpFiles.Destroy;
        ResetProgress;
        exit;
      end
    end;
  end;
{end;


Procedure TMainForm.LoadHelpFileTaskOnExecute (Parameters: TObject;
  Context: TStdTaskItem);
var
  FileIndex: longint;
  Filename: string;
  HelpFile: THelpFile;
  HelpFiles: TList;
Begin
  HelpFiles:= TList.Create;
  for FileIndex:= 0 to LoadingFilenameList.Count - 1 do
  begin
    Filename:= LoadingFilenameList[ FileIndex ];
    try
      LoadingFileIndex:= FileIndex;
      HelpFile:= THelpFile.Create( FileName,
                                   OnHelpFileLoadProgress );
      inc( LoadingSizeSoFar, longint( LoadingFilenameList.Objects[ FileIndex ] ) );
      HelpFiles.Add( HelpFile );
    except
      on E: EHelpFileException do
      begin
        ShowMessage( 'Could not open '
                     + FileName
                     +': '
                     + E.Message );
        Cursor:= crDefault;
        Result:= false;
        while HelpFiles.Count > 0 do
        begin
          HelpFile:= HelpFiles[ 0 ];
          HelpFile.Destroy;
          HelpFiles.Delete( 0 );
        end;
        LoadingFilenameList.Destroy;
        HelpFiles.Destroy;
        ResetProgress;
        exit;
      end;
    end;
  end;
End;

procedure TMainForm.OnFileLoaded;
begin}

  // Now that we have successfully loaded the new help file(s)
  // close the existing one.
  CloseFile;

  AssignList( HelpFiles, Files );

  ProgressBar.Position:= 50;
  SetStatus( 'Displaying... ' );

  LoadingFilenameList.Destroy;
  HelpFiles.Destroy;

  Result:= true;

  Caption:= 'NewView - ' + THelpFile( Files[ 0 ] ).Title;

  PreviousMRUIndex:= FindStringInList( FileNames, Settings.MRUList );
  if PreviousMRUIndex > -1 then
    // is already in list, remove
    Settings.MRUList.Delete( PreviousMRUIndex );
  Settings.MRUList.Insert( 0, FileNames );

  if Settings.MRUList.Count > 9 then
    Settings.MRUList.Delete( 9 );
  CreateMRUMenuItems;

  ProgressBar.Position:= 51;
  SetStatus( 'Loading notes... ' );

  for FileIndex:= 0 to Files.Count - 1 do
  begin
    HelpFile:= Files[ FileIndex ];
    LoadNotes( HelpFile );
  end;

  ProgressBar.Position:= 55;
  SetStatus( 'Display contents... ' );

  LoadContents;

  if ContentsOutline.ChildCount = 1 then
  begin
    ProfileEvent( '  Expand first node' );
    // Contents has only one top level node... expand it
    ContentsOutline.Children[ 0 ].Expand;
  end;

  ProgressBar.Position:= 55;
  SetStatus( 'Display first topic... ' );

  ProfileEvent( '  Display first topic' );

  if ContentsOutline.ChildCount > 0 then
    ContentsOutline.SelectedNode:= ContentsOutline.Children[ 0 ];

  LoadIndex;

  SearchResultsListBox.Clear;

  Cursor:= crDefault;

  ProgressBar.Position:= 100;
  SetStatus( 'Done' );

  DosSleep( 50 );
  ResetProgress;

  StopProfile;

  EnableControls;

  ContentsOutline.Focus;
end;

Procedure TMainForm.OpenMIOnClick (Sender: TObject);
Begin
  FileOpen;
end;

procedure TMainForm.FileOpen;
begin
  if Settings.UseOriginalDialogs then
  begin
    if SystemOpenDialog.Execute then
      OpenFile( SystemOpenDialog.FileName );
  end
  else
  begin
    FileDialogForm.AllowMultipleFiles:= true;
    FileDialogForm.Caption:= 'Open Help File';
    FileDialogForm.FilterComboBox.Filter := 'Help Files (*.hlp, *.inf)|*.inf;*.hlp | All Files (*.*)|*.*';

    if FileDialogForm.ShowModal = mrOK then
      OpenFile( ListToString( FileDialogForm.FileNames, '+' ) );
  end;
End;

Procedure TMainForm.CloseWindows;
Begin
  DestroyListObjects( Windows );
  Windows.Clear;
end;

Initialization
  RegisterClasses ([TMainForm, TSplitBar,
    TNoteBook,
    TEdit, TListBox, TLabel,
    TTabSet, TRichTextView, TCoolBar2, TProgressBar, TMainMenu, TMenuItem,
    TBitBtn, TOpenDialog, TImageList, TPanel, TButton, THelperThread, TStdTask,
    TSystemOpenDialog, TOutline2, TCustomListBox, TPopupMenu, TSpeedButton]);
End.
