Unit GlobalSearchForm;

// NewView - a new OS/2 Help Viewer
// Copyright 2001 Aaron Lawrence (aaronl at consultant dot com)
// This software is released under the Gnu Public License - see readme.txt

Interface

// Global search uses a thread (THelperThread,
// from Erik Huelsmann) to load helpfiles, search them, and
// send the results to be displayed.

Uses
  Classes, Forms, Graphics, Grids, Buttons, StdCtrls, ComCtrls, HT, WindowMessages,
  MultiColumnListBox, Outline2, HelpFile;

Type
  TViewTopicCallback = procedure( Filename: string;
                                  TopicIndex: longint ) of object;

  TGlobalSearchForm = Class (TForm)
    ViewTopicButton: TBitBtn;
    CancelButton: TBitBtn;
    SearchTextEdit: TEdit;
    SearchButton: TBitBtn;
    SearchTypeLabel1: TLabel;
    ResultsLabel: TLabel;
    ProgressBar: TProgressBar;
    SearchTask: TStdTask;
    ProgressLabel: TLabel;
    ResultsOutline: TOutline2;
    HelperThread: THelperThread;
    Procedure ResultsOutlineOnItemDblClick (Node: TNode);
    Procedure ViewTopicButtonOnClick (Sender: TObject);
    Procedure GlobalSearchFormOnCreate (Sender: TObject);
    Procedure SearchTaskOnFinish (Sender: TStdTaskItem);
    Procedure GlobalSearchFormOnCloseQuery (Sender: TObject;
      Var CanClose: Boolean);
    Procedure SearchTaskOnProgress (Sender: TStdTaskItem; I: LongInt;
      Const S: String);
    Procedure SearchTaskOnExecute (Parameters: TObject; Context: TStdTaskItem);
    Procedure SearchButtonOnClick (Sender: TObject);
    Procedure GlobalSearchFormOnShow (Sender: TObject);
  Protected
    Procedure OnFileProgress( n, outof: integer;
                              Message: string );
//    Procedure OnFoundMatch( Match: string );
    Procedure OnFoundMatch( Result: TObject );
    Procedure SetProgressLabel( S: String );
    // Handle our own WM_OPENED message
    Procedure WMOpened( Var Msg: TMessage ); Message WM_OPENED;
    Procedure ClearResults;
    Procedure ViewTopic;

  public
    // set by caller
    ViewTopicCallback: TViewTopicCallback;
  End;

Var
  GlobalSearchForm: TGlobalSearchForm;

Implementation

uses
  SysUtils, ACLFileUtility, BseDos, CUstomHeaderControl, HelpTopic,
  ACLProfile, TextSearchQuery, Dialogs,
  SearchUnit;

type
  // Returned by thread. (Note - helpfiles and topics are destroyed dynamically.)
  TSearchResult = class
    Filename: string;
    FileTitle: string;
    MatchingTopics: TList;
    constructor Create;
    destructor Destroy; override;
  end;

  // Used to store filenames in outline
  THelpFileInfo = class
    FileName: string;
  end;

constructor TSearchResult.Create;
begin
  inherited Create;
  MatchingTopics:= TList.Create;
end;

destructor TSearchResult.Destroy;
begin
  inherited Destroy;
  MatchingTopics.Destroy;
end;

Procedure TGlobalSearchForm.ResultsOutlineOnItemDblClick (Node: TNode);
Begin
  ViewTopic;
End;

Procedure TGlobalSearchForm.ViewTopicButtonOnClick (Sender: TObject);
begin
  ViewTopic;
end;

Procedure TGlobalSearchForm.ViewTopic;
var
  Node: TNode;
  HelpFileInfo: THelpFileInfo;
  TopicIndex: longint;
Begin
  Node:= ResultsOutline.SelectedNode;
  if Node = nil then
    exit;
  case Node.Level of
    0:
    begin
      // file node
      HelpFileInfo:= Node.Data as THelpFileInfo;
      TopicIndex:= -1;
    end;

    1:
    begin
      // topic node
      HelpFileInfo:= Node.Parent.Data as THelpFileInfo;
      TopicIndex:= longint( Node.Data );
    end;

    else
      assert( false, 'Invalid node level in ViewTopic!: ' + IntToStr( Node.Level ) );
  end;

  ViewTopicCallback( HelpFileInfo.FileName, TopicIndex );
End;

Procedure TGlobalSearchForm.GlobalSearchFormOnCreate (Sender: TObject);
Begin
End;

Procedure TGlobalSearchForm.SearchTaskOnFinish (Sender: TStdTaskItem);
Begin
  if ResultsOutline.ChildCount > 0 then
  begin
    ResultsOutline.SelectedNode:= ResultsOutline.Children[ 0 ];
    ResultsOutline.Focus;
    SetProgressLabel( '' );
  end
  else
    SetProgressLabel( '(No results found)' );
End;

Procedure TGlobalSearchForm.GlobalSearchFormOnCloseQuery (Sender: TObject;
  Var CanClose: Boolean);
Begin
End;

Procedure TGlobalSearchForm.SetProgressLabel( S: String );
begin
  ProgressLabel.Text:= S;
  ProgressLabel.Refresh;
end;

Procedure TGlobalSearchForm.SearchTaskOnProgress (Sender: TStdTaskItem;
  I: LongInt; Const S: String);
Begin
  ProgressBar.Position:= I;
  SetProgressLabel( S );
End;

Procedure TGlobalSearchForm.OnFileProgress( n, outof: integer;
                                            Message: string );
Begin
  // Ignore. I just need this because Sibyl won't accept passing
  // nil as a procedure parameter.
end;

Procedure TGlobalSearchForm.SearchTaskOnExecute (Parameters: TObject;
  Context: TStdTaskItem);
var
  Files: TStringList;
  FileIndex: longint;
  Filename: string;
  HelpFile: THelpFile;
  ResultIndex: longint;
  SearchResult: TSearchResult;
  Topic: TTopic;
  Query: TTextSearchQuery;
Begin
//  StartProfile( 'searchprofile' );

  Query := Parameters as TTextSearchQuery;
  Context.Progress( 0, 'Finding files...' );
  Files:= TStringList.Create;
  GetFilesForPath( 'HELP', '*.inf;*.hlp', Files );
  Context.Progress( 2, 'Searching files...' );

  for FileIndex:= 0 to Files.Count - 1 do
  begin
    Filename:= Files[ FileIndex ];
    Context.Progress( 5 + FileIndex * 95 div Files.Count,
                      'Searching '
                      + Filename
                      + ' ('
                      + IntToStr( FileIndex + 1 )
                      + ' of '
                      + IntToStr( Files.Count )
                      + ')...' );

    try
      HelpFile:= THelpFile.Create( FileName, OnFileProgress );

      // Create a searchresult object to send back to main thread.
      SearchResult:= TSearchResult.Create;
      SearchResult.Filename:= HelpFile.Filename;
      SearchResult.FileTitle:= HelpFile.Title;
      SearchResult.MatchingTopics:= TList.Create;

      SearchHelpFile( HelpFile,
                      Query,
                      SearchResult.MatchingTopics,
                      nil // don't care about words matched
                      );

      if SearchResult.MatchingTopics.Count > 0 then
      begin
        ProfileEvent( '  Sort results' );
        SearchResult.MatchingTopics.Sort( TopicRelevanceCompare );
        ProfileEvent( '  Display results' );

        Context.SendData( SearchResult, OnFoundMatch );
      end
      else
        SearchResult.Destroy;

      ProfileEvent( 'Unload helpfile' );
      HelpFile.Destroy;

    except
      on E: EHelpFileException do
      begin
//        Context.SendStringMessage( '(Could not open ' + Filename + ')', OnFoundMatch );
      end;
    end;

  end;
  Context.Progress( 100, 'Done' );
  Query.Destroy;
  Files.Destroy;
End;

Procedure TGlobalSearchForm.ClearResults;
var
  FileIndex: longint;
begin
  for FileIndex:= 0 to ResultsOutline.ChildCount - 1 do
    ResultsOutline.Children[ FileIndex ].Data.Free;
  ResultsOutline.Clear;
end;

Procedure TGlobalSearchForm.SearchButtonOnClick (Sender: TObject);
var
  SearchText: string;
  Query: TTextSearchQuery;
Begin
  SearchText:= trim( SearchTextEdit.Text );
  if SearchText = '' then
    exit;

  try
    Query := TTextSearchQuery.Create( SearchText );
  except
    on e: ESearchSyntaxError do
    begin
      ShowMessage( 'Error in search syntax: '
                   + e.Message );
      exit;
    end;
  end;

  ClearResults;
  StartProfile( 'SearchProfile' );

  SearchTask.Schedule( Query );
End;

Procedure TGlobalSearchForm.GlobalSearchFormOnShow (Sender: TObject);
Begin
  SetProgressLabel( '' );
  PostMsg( Handle, WM_OPENED, 0, 0 );
End;

Procedure TGlobalSearchForm.OnFoundMatch( Result: TObject );
var
  SearchResult: TSearchResult;
  Topic: TTopic;
  HelpFileInfo: THelpFileInfo;
  FileNode: TNode;
  TopicIndex: longint;
begin
  SearchResult:= Result as TSearchResult;

  HelpFileInfo:= THelpFileInfo.Create;
  HelpFileInfo.FileName:= SearchResult.FileName;
  FileNode:= ResultsOutline.AddChild( SearchResult.FileTitle
                                      + ' ('
                                      + SearchResult.FileName
                                      + ')',
                                      HelpFileInfo );
  for TopicIndex := 0 to SearchResult.MatchingTopics.Count - 1 do
  begin
    Topic:= SearchResult.MatchingTopics[ TopicIndex ];
    FileNode.AddChild( Topic.Title,
                       TObject( Topic.Index ) );
  end;
end;

Procedure TGlobalSearchForm.WMOpened( Var Msg: TMessage );
begin
  SearchTextEdit.XStretch:= xsFrame;
  ProgressLabel.XStretch:= xsFrame;
//  ResultsListBox.XStretch:= xsFrame;
//  ResultsListBox.YStretch:= ysFrame;
  ResultsOutline.XStretch:= xsFrame;
  ResultsOutline.YStretch:= ysFrame;
  ProgressLabel.YAlign:= yaTop;
  SearchTextEdit.YAlign:= yaTop;
  SearchTextEdit.Focus;
end;

Initialization
  RegisterClasses ([TGlobalSearchForm, TBitBtn, TEdit, TLabel,
    TProgressBar, TStdTask, TOutline2, THelperThread]);
End.
