/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, I1.HR                           *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Apr 93                                                   *
*  Last Update : Apr 93                                                   *
*                                                                         *
*  This Module is part of the DALIB / UNILIB                              *
*                                                                         *
*  Module      : overlap1.c                                               *
*                                                                         *
*  Function    : fill out overlapping areas for distributed arrays        *
*                                                                         *
*   source  : is a source array                                           *
*   target  : is the target, conform with source, but has an overlap      *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "system.h"

     /*********************************************************
     *                                                        *
     *  Exchange the overlap area                             *
     *                                                        *
     *   l1      N1                              r1           *
     *  -----------------------------------------------       *
     *  |T    | Sxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx |     |       *
     *  -----------------------------------------------       *
     *                                                        *
     *********************************************************/

void dalib_communicate_overlap (target, source, N1, l1, r1)

unsigned char *target, *source;
int N1, l1, r1;

{ int left, right;

#ifdef DEBUG
  printf ("%d comm overlap, %d (%d - %d)\n", pcb.i, N1, l1, r1);
#endif

  /* make sure that overlap area is not too big */

  if (N1 < l1)
   { printf ("left overlap area (size = %d) too big, only %d bytes\n", l1, N1);
     exit (-1);
   }
  if (N1 < r1)
   { printf ("right overlap area (size = %d) too big, only %d bytes\n",r1,N1);
     exit (-1);
   }

  left  = pcb.i - 1;
  if (left == 0) left = pcb.p;
  right = pcb.i + 1;
  if (right == pcb.p +1) right = 1;

  if (target+l1 != source)
     dalib_memcpy (target+l1, source, N1);

  asend (pcb.i, left, source, r1);
  areceive (pcb.i, right, target + l1 + N1, r1);

  asend (pcb.i, right, source + N1 - l1, l1);
  areceive (pcb.i, left, target, l1);

}  /* dalib_communicate_overlap */

     /*********************************************************
     *                                                        *
     *  Overlapping of one-dimensional arrays                 *
     *                                                        *
     *   with communication                                   *
     *                                                        *
     *                                                        *
     *   l1      N1                              r1           *
     *  -----------------------------------------------       *
     *  |T    | Sxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx |     |       *
     *  -----------------------------------------------       *
     *                                                        *
     *********************************************************/

void dalib_coverlap1__ (target, source, N1, l1, r1, size)

unsigned char *target, *source;
int *N1, *l1, *r1, *size;

/*   source is (N1), target is (N1+l1+r1), size is number of bytes */

{ int bytes, myN;

  bytes     = *size;
  myN       = dalib_local_size (*N1);
  dalib_communicate_overlap (target, source, myN*bytes, *l1*bytes, *r1*bytes);

}  /* dalib_coverlap1 */

     /*********************************************************
     *                                                        *
     *  Overlapping of two-dimensional arrays                 *
     *                                                        *
     *   with communication                                   *
     *                                                        *
     *      l2      N2                              r2        *
     *     -----------------------------------------------    *
     *     |T      ,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,       |    *
     * l1  |       ,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,       |    *
     *     |     | Sxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx |     |    *
     * N1  |     | xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx |     |    *
     *     |     | xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx |     |    *
     *     |       ,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,       |    *
     * r1  |       ,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,       |    *
     *     -----------------------------------------------    *
     *                                                        *
     *********************************************************/

void dalib_coverlap2__ (target, source, N1, l1, r1, N2, l2, r2, size)

unsigned char *target, *source;
int *N1, *l1, *r1, *N2, *l2, *r2, *size;

{ int bytes, myN;

  bytes     = *size;
  myN       = dalib_local_size (*N2);
  dalib_overlap_seg2 (0, target, source, *N1*bytes, *l1*bytes, *r1*bytes,
                                      myN, *l2, *r2);

  /* last dimension will be overlapped by communication */

  bytes *= (*N1 + *l1 + *r1);
  dalib_communicate_overlap (target, target + *l2 * bytes,
                             myN * bytes, *l2 * bytes, *r2 * bytes);

} /* dalib_coverlap2_ */

     /*********************************************************
     *                                                        *
     *  Overlapping of three-dimensional arrays               *
     *                                                        *
     *********************************************************/

void dalib_coverlap3__ (target, source, N1, l1, r1, N2, l2, r2, N3, l3, r3, size)

unsigned char *target, *source;
int *N1, *l1, *r1, *N2, *l2, *r2, *N3, *l3, *r3, *size;

{ int bytes, myN;

  bytes     = *size;
  myN       = dalib_local_size (*N3);

  dalib_overlap_seg3 (0, target, source, *N1*bytes, *l1*bytes, *r1*bytes,
                                      *N2, *l2, *r2, myN, *l3, *r3);

  /* last dimension will be overlapped by communication */

  bytes *= (*N1 + *l1 + *r1) * (*N2 + *l2 + *r2);
  dalib_communicate_overlap (target, target + *l3 * bytes,
                             myN * bytes, *l3 * bytes, *r3 * bytes);

} /* dalib_coverlap3_ */

     /*********************************************************
     *                                                        *
     *  Overlapping of four-dimensional arrays                *
     *                                                        *
     *********************************************************/

void dalib_coverlap4__ (target, source, N1, l1, r1, N2, l2, r2,
                                      N3, l3, r3, N4, l4, r4, size)

unsigned char *target, *source;
int *N1, *l1, *r1, *N2, *l2, *r2, *N3, *l3, *r3, *N4, *l4, *r4, *size;

{ int bytes, myN;

  bytes     = *size;
  myN       = dalib_local_size (*N4);

  dalib_overlap_seg4 (0, target, source, *N1*bytes, *l1*bytes, *r1*bytes,
                      *N2, *l2, *r2, *N3, *l3, *r3, myN, *l4, *r4);

  /* last dimension will be overlapped by communication */

  bytes *= (*N1 + *l1 + *r1) * (*N2 + *l2 + *r2) * (*N3 + *l3 + *r3);
  dalib_communicate_overlap (target, target + *l4 * bytes,
                             myN * bytes, *l4 * bytes, *r4 * bytes);


} /* dalib_coverlap4_ */

