/*
 * $Id: redirect.c,v 1.7.52.3 2003/12/15 03:13:47 squidadm Exp $
 *
 * DEBUG: section 61    Redirector
 * AUTHOR: Duane Wessels
 */
#include "squidDeclaration.h"

#include "squid.h"

typedef struct {
    void *data;
    char *orig_url;
    struct in_addr client_addr;
    const char *client_ident;
    const char *method_s;
    RH *handler;
} redirectStateData;

static HLPCB redirectHandleReply;
static void redirectStateFree(redirectStateData * r);
static helper *redirectors = NULL;
static OBJH redirectStats;
static int n_bypassed = 0;
CBDATA_TYPE(redirectStateData);

static void
redirectHandleReply(void *data, char *reply)
{
    redirectStateData *r = (redirectStateData *)data;
    int valid;
    char *t;
    debug(61, 5) ("redirectHandleRead: {%s}\n", reply ? reply : "<NULL>");
    if (reply) {
       t = strchr(reply, ' ');
       if (t)
           *t = '\0';
       if (*reply == '\0')
           reply = NULL;
    }
    valid = cbdataValid(r->data);
    cbdataUnlock(r->data);
    if (valid)
       r->handler(r->data, reply);
    redirectStateFree(r);
}

static void
redirectStateFree(redirectStateData * r)
{
    safe_free(r->orig_url);
    cbdataFree(r,redirectStateData);
}

static void
redirectStats(StoreEntry * sentry)
{
    storeAppendPrintf(sentry, "Redirector Statistics:\n");
    helperStats(sentry, redirectors);
    if (Config.onoff.redirector_bypass)
       storeAppendPrintf(sentry, "\nNumber of requests bypassed "
           "because all redirectors were busy: %d\n", n_bypassed);
}

/**** PUBLIC FUNCTIONS ****/

void
redirectStart(clientHttpRequest * http, RH * handler, void *data)
{
    ConnStateData *conn = http->conn;
    redirectStateData *r = NULL;
    const char *fqdn;
    char buf[8192];
    assert(http);
    assert(handler);
    debug(61, 5) ("redirectStart: '%s'\n", http->uri);
/*
    if (Config.Program.redirect == NULL) {
       handler(data, NULL);
       return;
    }
    if (Config.accessList.redirector) {
       aclCheck_t ch;
       memset(&ch, '\0', sizeof(ch));
       ch.src_addr = http->conn->peer.sin_addr;
       ch.my_addr = http->conn->me.sin_addr;
       ch.my_port = ntohs(http->conn->me.sin_port);
       ch.request = http->request;
       if (!aclCheckFast(Config.accessList.redirector, &ch)) {
           / * denied -- bypass redirector * /
           handler(data, NULL);
           return;
       }
    }
*/
    if (Config.onoff.redirector_bypass && redirectors->stats.queue_size) {
       /* Skip redirector if there is one request queued */
       n_bypassed++;
       handler(data, NULL);
       return;
    }
    r = cbdataAlloc(redirectStateData);
    r->orig_url = xstrdup(http->uri);
    r->client_addr = conn->log_addr;
    if (http->request->auth_user_request)
       r->client_ident = authenticateUserRequestUsername(http->request->auth_user_request);
    else if (conn->rfc931[0]) {
       r->client_ident = conn->rfc931;
    } else {
       r->client_ident = dash_str;
    }
    r->method_s = RequestMethodStr[http->request->method];
    r->handler = handler;
    r->data = data;
    cbdataLock(r->data);
    if ((fqdn = fqdncache_gethostbyaddr(r->client_addr, 0)) == NULL)
       fqdn = dash_str;
    snprintf(buf, 8192, "%s %s/%s %s %s\n",
       r->orig_url,
       inet_ntoa(r->client_addr),
       fqdn,
       r->client_ident[0] ? rfc1738_escape(r->client_ident) : dash_str,
       r->method_s);
    helperSubmit(redirectors, buf, redirectHandleReply, r);
}

void
redirectInit(void)
{
    static int init = 0;
    if (!Config.Program.redirect)
       return;
    if (redirectors == NULL)
       redirectors = helperCreate("redirector");
    redirectors->cmdline = Config.Program.redirect;
    redirectors->n_to_start = Config.redirectChildren;
    redirectors->ipc_type = IPC_TCP_SOCKET;
    helperOpenServers(redirectors);
    if (!init) {
       cachemgrRegister("redirector",
           "URL Redirector Stats",
           redirectStats, 0, 1);
       init = 1;
       CBDATA_INIT_TYPE(redirectStateData);
    }
}

void
redirectShutdown(void)
{
    if (!redirectors)
       return;
    helperShutdown(redirectors);
    if (!shutting_down)
       return;
    helperFree(redirectors);
    redirectors = NULL;
}
