
/*
 * confload.c
 * (C) 2000 Antonino Iannella, Stellar-X Pty Ltd
 * Changes (C) 2002 Sergey I. Yevtushenko
 * Released under GPL, see COPYING-2.0 for details.
 *
 * These routines load the msntauth configuration file.
 * It stores the servers to query, sets the denied and
 * allowed user files, and provides the
 * authenticating function.
 */

#define BUFFER_LEN  4096

#define INCL_DOS
#define INCL_DOSPROCESS
#define INCL_DOSMEMMGR
#define INCL_DOSERRORS

#include <os2.h>

#define PURE_32 1

#include <netcons.h>
#include <neterr.h>
#include <access.h>
#include <dcdb.h>
#include <server.h>
#include <wksta.h>

#include "lsos2auth.h"

extern char Denyuserpath[MAXPATHLEN];
extern char Allowuserpath[MAXPATHLEN];

typedef struct
{
    char domain[LSHOSTLEN];     /* Domain name */
    char group[LSHOSTLEN];      /* Group name to check */
    char dc[LSHOSTLEN];         /* Detected Domain Controller name */
}
DomainTuple;

DomainTuple Domains[MAXDOMAINS];
int DomainsCount = 0;

static char* pBuffer = 0;

/* Declarations */

static void ProcessLine(char *);
static void AddDomain(char *, char *);

/*
 * Opens and reads the configuration file.
 * Returns 0 on success, or 1 for error.
 */

int OpenConfigFile(void)
{
	FILE *ConfigFile;
	char Confbuf[2049];			/* Line reading buffer */
    char cFile[MAXPATHLEN + 1] = {0};
    char* cEnv = 0;

	/* Initialise defaults */

    DomainsCount = 0;
    memset(Domains, '\0', sizeof(Domains));
    memset(Denyuserpath, '\0', MAXPATHLEN);
    memset(Allowuserpath, '\0', MAXPATHLEN);

    do
    {
        cEnv = getenv("LSOS2AUTH");

        if(cEnv)
        {
            sprintf(cFile, "%s%s", cEnv, CONFIGFILE);
            break;
        }

        cEnv = getenv("ETC");

        if(cEnv)
        {
            sprintf(cFile, "%s%s", cEnv, CONFIGFILE);
            break;
        }
    }
    while(0);

    if(!cFile[0])
    {
        syslog(LOG_ERR, "OpenConfigFile: Environment variables LSOS2AUTH "
                        "and ETC are undefined.");
        return 1;
    }

    /* Open file */

    ConfigFile = fopen(cFile, "r");

    if (!ConfigFile)
    {
        syslog(LOG_ERR, "OpenConfigFile: Failed to open %s.", cFile);
		syslog(LOG_ERR, "%s", strerror(errno));
		return 1;
    }
	/* Read in, one line at a time */
	while (!feof(ConfigFile))
    {
		Confbuf[0] = '\0';
		if (NULL == fgets(Confbuf, 2048, ConfigFile))
			break;
        Confbuf[2048] = '\0';
		ProcessLine(Confbuf);
    }

    fclose(ConfigFile);

	/*
	 * Check that at least one server is being queried. Report error if not.
	 * Denied and allowed user files are hardcoded, so it's fine if they're
	 * not set in the confugration file.
	 */
    if(!DomainsCount)
    {
        syslog(LOG_ERR,
               "OpenConfigFile: No domains defined in %s. At least one is needed.",
               cFile);
		return 1;
    }
    return 0;
}

/* Parses a configuration file line. */

static void ProcessLine(char *Linebuf)
{
	char *Directive;
	char *Param1;
	char *Param2;
	char *Param3;

	/* Ignore empty lines */
	if (strlen(Linebuf) == 0)
		return;

	/* Break up on whitespaces */
	if ((Directive = strtok(Linebuf, " \t\n")) == NULL)
		return;

	/* Check for a comment line. If found, stop . */
	if (Directive[0] == '#')
		return;

    /* Check for 'domain' line. Check for 2 parameters. */
    if (stricmp(Directive, "domain") == 0)
    {
		Param1 = strtok(NULL, " \t\n");
		if (NULL == Param1)
        {
            syslog(LOG_ERR, "ProcessLine: 'domain' missing NAME parameter.");
			return;
        }

        Param2 = strtok(NULL, " \t\n");
		if (NULL == Param2)
        {
            syslog(LOG_ERR, "ProcessLine: 'domain' missing GROUP parameter.");
			return;
        }

        AddDomain(Param1, Param2);
		return;
    }

    /* Check for denyusers line */
    if (stricmp(Directive, "denyusers") == 0)
    {
		Param1 = strtok(NULL, " \t\n");

		if (NULL == Param1)
        {
			syslog(LOG_ERR,
				   "ProcessLine: A 'denyusers' line needs a filename parameter.");
			return;
        }
		memset(Denyuserpath, '\0', MAXPATHLEN);
		strncpy(Denyuserpath, Param1, MAXPATHLEN - 1);
		return;
    }
	/* Check for allowusers line */
    if (stricmp(Directive, "allowusers") == 0)
    {
		Param1 = strtok(NULL, " \t\n");

		if (NULL == Param1)
        {
			syslog(LOG_ERR,
				   "ProcessLine: An 'allowusers' line needs a filename parameter.");
			return;
        }
		memset(Allowuserpath, '\0', MAXPATHLEN);
		strncpy(Allowuserpath, Param1, MAXPATHLEN - 1);
		return;
    }
	/* Reports error for unknown line */
	syslog(LOG_ERR, "ProcessLine: Ignoring '%s' line.", Directive);
}

int GetDCName(char* domain, char* server, int sz)
{
    return Net32GetDCName(NULL, domain, server, sz);
}

void AddDomain(char *Name, char *Group)
{
    if (DomainsCount == MAXDOMAINS)
    {
        syslog(LOG_ERR, "AddDomain: Ignoring '%s' domain line; "
               "too many domains.", Name);
		return;
    }

    //Domains[DomainCount].domain

    if(GetDCName(Name, Domains[DomainsCount].dc, LSHOSTLEN))
    {
        syslog(LOG_ERR, "AddDomain: Ignoring '%s' domain line; "
               "unable to locate DC for this domain.", Name);
        return;
    }

    strncpy(Domains[DomainsCount].domain, Name , LSHOSTLEN - 1);
    strncpy(Domains[DomainsCount].group , Group, LSHOSTLEN - 1);

    DomainsCount++;
}

#ifndef LOG_AUTHPRIV
#define LOG_AUTHPRIV LOG_AUTH
#endif

int QueryDomain(char *username, char *password, char* server, char* group)
{
    //Checks are done in this order because checking of group
    //works much faster

    int rc = 0;
    ULONG ulTotal = 0;
    ULONG ulAvail = 0;
    struct group_info_0 *pGroups = 0;
    struct user_logon_req_1*  pReq = 0;
    struct user_logon_info_1* pRet = 0;

    if(!pBuffer)
    {
        rc = DosAllocMem((PPVOID)&pBuffer, BUFFER_LEN,
                         PAG_WRITE | PAG_READ | PAG_COMMIT);

        if(rc)
        {
            syslog(LOG_ERR, "QueryDomain: Not enough memory.");
            return rc;
        }
    }

    rc = Net32UserGetGroups(server, username, 0, pBuffer, BUFFER_LEN,
                            &ulAvail, &ulTotal, 0);

    if(rc)
    {
        syslog(LOG_AUTH | LOG_INFO, "QueryDomain: Net32UserGetGroups rc = %d for user '%s'",
               rc, username);
        return rc;
    }

    //Verify user's group

    pGroups = (struct group_info_0 *)pBuffer;

    rc = 0;

    while(ulAvail--)
    {
        if(!stricmp(pGroups->grpi0_name, group))
        {
            rc = 1;
            break;
        }

        pGroups++;
    }

    if(!rc)
    {
        syslog(LOG_AUTH | LOG_INFO,
               "QueryDomain: user '%s' does not belong to specified group",
               username);

        return 5; /* access denied */
    }

    //Verify if user can be logged in
    ulAvail = 0;

    pReq = (struct user_logon_req_1*)pBuffer;
    pRet = (struct user_logon_info_1*)pBuffer;

    strncpy(pReq->usrreq1_name, username, UNLEN);
    strncpy(pReq->usrreq1_password, password, SESSION_PWLEN);
    strupr(pReq->usrreq1_name);
    strupr(pReq->usrreq1_password);

    rc = Net32UserValidate2(server, 1, pBuffer, BUFFER_LEN,
                            0, &ulAvail, 0);

    if(rc)
    {
        syslog(LOG_AUTH | LOG_INFO,
               "QueryDomain: user '%s' can't be logged in",
               username);
    }

    return rc;
}

/*
 * Cycles through all domains to query.
 * Returns 0 if domain could authenticate the user.
 * Returns 1 if user can't be authenticated or user does not
 * belong to specified group.
 */

int QueryDomains(char *username, char *password)
{
    int i;

    for (i = 0; i < DomainsCount; i++)
    {
        int rc;

        rc = QueryDomain(username, password, Domains[i].dc, Domains[i].group);

        if(!rc)
			return 0;
    }
	return 1;
}

