(**************************************************************************)
(*                                                                        *)
(*  Web server input stream handler                                       *)
(*  Copyright (C) 2019   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE NetStream;

        (********************************************************)
        (*                                                      *)
        (*          Handler for I/O through either a TLS        *)
        (*          channel or directly through a socket        *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            2 April 2018                    *)
        (*  Last edited:        21 July 2019                    *)
        (*  Status:             Working if you don't use TLS    *)
        (*                                                      *)
        (********************************************************)


IMPORT Strings;

FROM SYSTEM IMPORT ADR, CARD8, LOC;

<* IF SUPPORTTLS THEN *>
    FROM TLS IMPORT
        (* type *)  TLSsession,
        (* proc *)  OpenTLSsession, CloseTLSsession, TLSsend,
                    TLSrecv, TLSrecvLine;
<* END *>

IMPORT SockStream;

FROM SockStream IMPORT
    (* type *)  SSstate,
    (* proc *)  OpenSockStream, PutToSocket;

FROM TransLog IMPORT
    (* type *)  TransactionLogID;

FROM Inet2Misc IMPORT
    (* proc *)  WaitForSocketOut, Synch;

FROM Sockets IMPORT
    (* type *)  Socket,
    (* proc *)  recv, send;

FROM Watchdog IMPORT
    (* type *)  WatchdogID,
    (* proc *)  KickWatchdog;

FROM LowLevel IMPORT
    (* proc *)  Copy, EVAL;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST
    BufferSize = 512;
    Nul = CHR(0);  CR = CHR(13);  LF = CHR(10);

TYPE
    NStream = POINTER TO RECORD
                             TLS: BOOLEAN;
                             <* IF SUPPORTTLS THEN *>
                                 TLShandle: TLSsession;
                             <* END *>
                             SStream: SSstate;
                         END (*RECORD*);

(************************************************************************)
(*                              OPEN/CLOSE                              *)
(************************************************************************)

PROCEDURE OpenNetStream (S: Socket;  IsServer: BOOLEAN;
                        logID: TransactionLogID;  TLS: BOOLEAN): NStream;

    (* Opens a new NStream object that handles socket data. *)

    VAR result: NStream;

    BEGIN
        NEW (result);
        result^.TLS := TLS;
        IF result^.TLS THEN
            <* IF SUPPORTTLS THEN *>
                result^.TLShandle := OpenTLSsession(S, IsServer, logID);
            <* END *>
            result^.SStream := NIL;
        ELSE
            <* IF SUPPORTTLS THEN *>
                result^.TLShandle := NIL;
            <* END *>
            result^.SStream := OpenSockStream (S);
        END (*IF*);
        RETURN result;
    END OpenNetStream;

(************************************************************************)

PROCEDURE CloseNetStream (VAR (*INOUT*) stream: NStream);

    (* Deletes the stream. *)

    BEGIN
        IF stream <> NIL THEN
            IF stream^.TLS THEN
                <* IF SUPPORTTLS THEN *>
                    CloseTLSsession (stream^.TLShandle);
                <* END *>
            ELSE
                SockStream.CloseSockStream (stream^.SStream);
            END (*IF*);
            DISPOSE (stream);
        END (*IF*);
    END CloseNetStream;

(************************************************************************)
(*                            SYNCHRONISATION                           *)
(************************************************************************)

(*
PROCEDURE DoSynch (stream: NStream);

    (* Forces any pending output to be sent. *)

    BEGIN
        Synch (stream^.sock);
    END DoSynch;
*)

(************************************************************************)
(*                                INPUT                                 *)
(************************************************************************)

(*
PROCEDURE OldGetLine (stream: NStream;  VAR (*OUT*) line: ARRAY OF CHAR): BOOLEAN;

    (* Gets one line from the specified stream.  Returns a FALSE        *)
    (* result, with no data, at end of stream.                          *)

    VAR length: CARDINAL;

    (********************************************************************)

    PROCEDURE StoreChar (ch: CHAR);

        (* Appends ch to line, except where this would cause overflow.  *)

        BEGIN
            IF length < HIGH(line) THEN
                line[length] := ch;  INC(length);
            END (*IF*);
        END StoreChar;

    (********************************************************************)

    VAR ch: CHAR;  FoundCR: BOOLEAN;

    BEGIN
        length := 0;  FoundCR := FALSE;

        LOOP
            WITH stream^ DO
                IF bufpos >= buflength THEN
                    buflength := recv (sock, buffer, BufferSize, 0);
                    IF (buflength = MAX(CARDINAL)) OR (buflength = 0) THEN
                        RETURN FALSE;
                    END (*IF*);
                    bufpos := 0;
                END (*IF*);
                ch := buffer[bufpos];  INC(bufpos);
            END (*WITH*);

            (* Our main job here is to find the line terminator.  *)

            IF FoundCR THEN
                IF ch = CR THEN
                    StoreChar(CR);
                ELSE
                    FoundCR := FALSE;
                    IF ch = LF THEN
                        StoreChar (Nul);
                        RETURN TRUE;
                    ELSE
                        StoreChar(ch);
                    END (*IF*);
                END (*IF*);
            ELSIF ch = CR THEN
                FoundCR := TRUE;
            ELSE
                IF ch = LF THEN
                    StoreChar (Nul);
                    RETURN TRUE;
                ELSE
                    StoreChar (ch);
                END (*IF*);
            END (*IF*);

        END (*LOOP*);

    END OldGetLine;
*)

(************************************************************************)

PROCEDURE GetLine (stream: NStream;  VAR (*OUT*) line: ARRAY OF CHAR): BOOLEAN;

    (* Gets one line from the specified stream.  Returns a FALSE        *)
    (* result, with no data, at end of stream.                          *)

    VAR actual: CARDINAL;

    BEGIN
        IF stream^.TLS THEN
            <* IF SUPPORTTLS THEN *>
                TLSrecvLine (stream^.TLShandle, line, HIGH(line)+1, actual);
            <* ELSE *>
                actual := 0;
            <* END *>
            RETURN actual > 0;
        ELSE
            RETURN SockStream.GetLine (stream^.SStream, line);
        END (*IF*);
    END GetLine;

(************************************************************************)

PROCEDURE GetBytes (stream: NStream;  VAR (*OUT*) result: ARRAY OF CHAR;
                        wanted: CARDINAL;  VAR (*OUT*) actual: CARDINAL);

    (* Gets at most wanted chars into result,  Returns actual as the    *)
    (* number of characters found.                                      *)

    BEGIN
        IF stream^.TLS THEN
            <* IF SUPPORTTLS THEN *>
            TLSrecv (stream^.TLShandle, result, wanted, actual);
            <* ELSE *>
                actual := 0;
            <* END *>
        ELSE
            actual := SockStream.PutToSocket (stream^.SStream, result, wanted);
        END (*IF*);
    END GetBytes;

(************************************************************************)
(*                                  OUTPUT                              *)
(************************************************************************)

PROCEDURE PutBytes (stream: NStream;  VAR (*IN*) message: ARRAY OF LOC;
                                                length: CARDINAL): CARDINAL;

    (* Sends a message of "length" bytes.  The returned value is the number *)
    (* of bytes sent, or MAX(CARDINAL) if there was an error.               *)

    BEGIN
        IF stream^.TLS THEN
            <* IF SUPPORTTLS THEN *>
                RETURN TLSsend (stream^.TLShandle, message, length);
            <* END *>
        ELSE
            RETURN PutToSocket (stream^.SStream, message, length);
        END (*IF*);
    END PutBytes;

(************************************************************************)

PROCEDURE PutEOL (stream: NStream);

    (* Sends CRLF.  *)

    TYPE TwoByte = ARRAY [0..1] OF CHAR;
    CONST EOL = TwoByte {CR, LF};
    VAR CRLF: TwoByte;

    BEGIN
        CRLF := EOL;
        EVAL (PutBytes (stream, CRLF, 2));
    END PutEOL;

(************************************************************************)

END NetStream.

