
/***********************************************************************
Web Client Test

This is a diagnostic tool used to analyze headers generated by various
web servers.  It writes a comprehensive log file, named "WCTEST.LOG" to
disk, showing all events and the times at which they occurred, with
hexadecimal and ASCII dump of all data sent and received.

Author:  Patrick J. Gleason
Date:  16-Dec-2006
Project:  Bulletron Internet Operating System, version 12.0
Compiler:  IBM VisualAge C++, version 3.0
Web Site:  http://www.idkcomp.com
E-Mail:  ora68@nehemiah.idkcomp.com
Phone: (315) 475-5598

***********************************************************************/
/*
   Operating System
*/
#include <OS2DEF.H>                     // Defines OS/2 data types.
#define INCL_DOSPROCESS                 // Include declaration for DosSleep(), thread and process control.
#include <BSEDOS.H>                     // Defines base operating system APIs.
#define INCL_DOSERRORS                  // Used by "BSEERR.H".
#include <BSEERR.H>                     // Defines OS/2 API result codes; needed by ASIPORTS.H for OS/2.
/*
   Sockets
*/
#include <types.h>                      // needed by "utils.h"
#include <sys\socket.h>                 // Define "AF_INET".
#include <netdb.h>                      // Define "hostent" and "servent" structures.
#include <nerrno.h>                     // Define error codes for sockets API.
#include <utils.h>                      // Define ntohs().
#include <netinet\in.h>
/*
   C Language
*/
#undef min
#include <stdlib.h>                     // Define the Standard C Library.
#include <signal.h>                     // Define signal() and related constants.
#include <stdio.h>                      // Define C Language standard input and output.
#include <time.h>                       // Define time() and ctime().
#include <errno.h>                      // Define C run time library errors.
/*
   Macros
*/
#define movmem(f, t, c) memcpy(t, f, (unsigned) c)
#define setmem(src, len, chr) memset(src, (int) chr, (unsigned) len)
#define clrmem(dst) memset(dst, 0, sizeof(dst))
#define TRUE 1
#define FALSE 0
/*
   Global Data
*/
FILE *logfile=NULL;                     // handle for the program execution log
unsigned int timer=0;                   // number of seconds remaining until monitor() aborts I/O on "s"
int s=-1;                               // socket number, or -1 if not connected
char inbuf[4096]={0};                   // socket input ring buffer
int iptr=0;                             // ring buffer insertion pointer
int eptr=0;                             // ring buffer extraction pointer
int running=TRUE;                       // stops all loops if FALSE
TID montid;                             // thread ID of Monitor Thread
/*
   Function Prototypes
*/
int main(int argc, char *argv[]);       // the whole program
void xrtrim(char *bufr);                // remove trailing white space
void logerr(char *msg);                 // record a message in the program log
void coredump(unsigned char *buf, int len);  // DEBUG dump
void ccisr(int x);                      // CTRL/C Interrupt Service Routine
void monitor(void *x);                  // asynchronous coroutine to time out socket I/O
int connect_socket(char *host, int port);  // connect to the server
int soputs(char *str, int opt);         // send a null-terminated ASCII string to the server
int sogets(char *buf, int size);        // read a line feed terminated string from the server
int sogetc(void);                       // read one character from the server
void transact(char *argv[]);            // retrieve a web page
/*
   Entry Point
*/
int main(int argc, char *argv[]){
   int test;                            // result of sock_init()
   APIRET rc;                           // result of operating system calls
   char bufr[256];                      // build messages here
                                        //
   if(argc<4){                          // If the user did not supply enough parameters,
      printf("Usage:\n\n");             //  then show him what to do.
      printf("   WCTEST host.domain GET url\n");
      exit(EXIT_FAILURE);               //
   }                                    //
/*
   Create the log file.
*/
   logfile=fopen("WCTEST.LOG", "w");    // Create a log file
   if(!logfile){                        // If the log file could not be created, then
      printf("*** ERROR %d creating WCTEST.LOG.\n", errno);  //  display an error message on the screen,
      exit(EXIT_FAILURE);               //  and return failure to the operating system.
   }                                    //
/*
   Trap program termination.
*/
   signal(SIGABRT, ccisr);              // abnormal termination signal sent by the abort function
   signal(SIGBREAK, ccisr);             // Ctrl-Break signal
   signal(SIGINT, ccisr);               // Ctrl-C signal
   signal(SIGTERM, ccisr);              // program termination signal sent by the user
/*
   Initialize the sockets interface.
*/
   test=sock_init();                    // Try to initialize the sockets interface.
   if(test){                            // If sockets initialization failed, then
      logerr("*** ERROR - INET.SYS driver not running.");
      fclose(logfile);                  // Save the log file.
      exit(EXIT_FAILURE);               // Kill the process and report failure.
   }                                    //
/*
   Start the Monitor Thread.
*/
   montid=_beginthread(monitor, NULL, 32678, NULL);
/*
   Get the web page.
*/
   transact(argv);                      // Standard "goto" work-around.
/*
   Shut Down
*/
   running=FALSE;                       // Terminate all loops, letting any coroutines terminate.
   rc=DosWaitThread(&montid, DCWW_WAIT);  // Wait for the Monitor Thread to terminate; might take a second.
   if(rc!=NO_ERROR){                    // If there was any problem with that, then
      sprintf(bufr, "*** ERROR %d in main() waiting for Monitor Thread to terminate.", rc);
      logerr(bufr);                     //  record it in the execution log.
   }                                    //
   logerr("Normal Program Termination");
   fclose(logfile);                     // Close and save the log file.
   logfile=NULL;                        // Remember that the log file is not open.
   exit(EXIT_SUCCESS);                  // Call the operating system to abort the current process.
   return EXIT_SUCCESS;                 // This should never execute, but some compilers need it.
}

/***********************************************************************
Remove Trailing Control Characters and Spaces

The main purpose of this is to get rid of carriage returns and line
feeds at the end of a message so that the screen doesn't get messed up
if wierd stuff comes in.  As a benign side effect, simplification of
the program makes it get rid of trailing spaces and all other control
characters, which can't be displayed anyways.


Input Arguments:

   str                  specifies the memory address of the null-
                        terminated string of ASCII characters to be
                        trimmed.


Output Arguments:

   str                  The string is modified in place.


Constants Used:

   none


Global Data Used:

   none


Object Data Used:

   n/a


Returns:

   none


***********************************************************************/
void xrtrim(char *bufr){
   int i;                               // general purpose counter
                                        //
   for(i=strlen(bufr); i>0; i--){       // Do this for each character from the end to the beginning.
      if(bufr[i-1]>=' '){               // If this is not a control character, then
         break;                         //  stop.
      }                                 //
      bufr[i-1]='\0';                   // Replace trailing control characters with nulls.
   }                                    //
   return;                              // Done.
}                                       //

/***********************************************************************
Record an Error Message

This subroutine records an error or progress message in the program
execution log and to the screen.  A newline is added here, so none
should be placed at the end of the message string.


Input Arguments:

   msg                  specifies the memory address of a null-
                        terminated ASCII string to be recorded in the
                        log.


Output Arguments:

   none


Constants Used:

   none


Global Data Used:

   logfile              (read only) is the handle of the program
                        execution log file; NULL if the execution log
                        is not open.


Object Data Used:

   n/a


Returns:

   none


***********************************************************************/
void logerr(char *msg){
   time_t t0;                           // the time now, in seconds since the beginning of 1970
   char timestr[32];                    // a null-terminated ASCII version of the time
   char temp[256];                      // truncated output for the screen.
                                        //
   t0=time(NULL);                       // Get the current system time.
   strcpy(timestr, ctime(&t0));         // Let the run time library convert it to an ASCII string.
   xrtrim(timestr);                     // Remove any trailing control characters and spaces.
   if(logfile){                         // If there is a program log file, then
      fprintf(logfile, "%s %s\n", timestr, msg);  //  write the message in the log.
   }                                    //
   clrmem(temp);                        // Make sure that the truncated message is null-terminated.
   strncpy(temp, msg, sizeof(temp)-1);  // Make a copy of the message, avoiding buffer overflow.
   xrtrim(temp);                        // Remove any trailing carriage returns, line feeds and white space.
   temp[53]=0;                          // Make it fit in 78 columns with the time stamp included.
   printf("%s %s\n", timestr, temp);    // In any case, write the message to the screen.
   return;                              // Done.
}                                       //

/***********************************************************************
DEBUG Style Dump

We want to be able to study all of the data going back and forth
between the client and the server, in hexadecimal as well as printable
ASCII.


Input Arguments:

   buf                  specifies the memory address of the buffer to
                        be dumped.

   len                  specifies the number of bytes to dump


Output Arguments:

   none


Constants Used:

   none


Global Data Used:

   logfile              is the handle for the program execution log
                        file; NULL if not open.


Object Data Used:

   n/a


Returns:

   none


***********************************************************************/
void coredump(unsigned char *buf, int len){
   unsigned char c;                     // current charcter
   int i;                               // read cursor
   int j;                               // line cursor
                                        //
   if(!logfile){                        // If there is no program execution log, then
      return;                           //  just skip this.
   }                                    //
   fprintf(logfile, "-------------------------------------------------------------------------\n");
   for(i=0; i<len; i+=16){              // Write 16 bytes to a line.
                                        //
      fprintf(logfile, "%06X  ", i);    // Write the offset and two spaces.
                                        //
      for(j=0; j<7; j++){               // Write the first 7 bytes with a space after.
         if((i+j)<len){                 // If the current byte exists, then
            fprintf(logfile, "%02X ", buf[i+j]);  //  write it in hex with a space after.
         } else {                       // If the pointer is beyond the end of the buffer, then
            fprintf(logfile, "   ");    //  write three spaces to keep columns aligned.
         }                              //
      }                                 //
                                        //
      if((i+j)<len){                    // If the current byte exists, then write the eighth byte with a hyphen after.
         if((i+j+1)<len){               // If this is the final byte, though,
            fprintf(logfile, "%02X-", buf[i+j]);
         } else {                       //  then don't include the hyphen.
            fprintf(logfile, "%02X ", buf[i+j]);  //  write the eighth byte with a hyphen after.
         }                              //
      } else {                          // If the pointer is beyond the end of the buffer, then
         fprintf(logfile, "   ");       //  write three spaces to keep columns aligned.
      }                                 //
                                        //
      for(j=8; j<16; j++){              // Write the last 8 bytes with a space after.
         if((i+j)<len){                 // If the current byte exists, then
            fprintf(logfile, "%02X ", buf[i+j]);  //  write the eighth byte with a hyphen after.
         } else {                       // If the pointer is beyond the end of the buffer, then
            fprintf(logfile, "   ");    //  write three spaces to keep columns aligned.
         }                              //
      }                                 //
                                        //
      fprintf(logfile, " ");            // Two spaces are needed to separate the hex from the ASCII.
      for(j=0; j<16; j++){              // Do this for the next 16 bytes.
         if((i+j)<len){                 // If the current byte exists, then
            c=buf[i+j];                 //  take a closer look at it.
            if((c>=' ')&&(c<128)){      // If it represents a printable ASCII code, then
               fprintf(logfile, "%c", buf[i+j]);  // print its character.
            } else {                    // If it's not printable, then
               fprintf(logfile, ".");   //  replace it with a period.
            }                           //
         } else {                       // If beyond the end of the buffer, then
            fprintf(logfile, " ");      //  print a space to keep columns aligned.
         }                              //
      }                                 //
                                        //
      fprintf(logfile, "  |\n");        // Finish the line with the traditional C newline.
   }                                    //
   fprintf(logfile, "-------------------------------------------------------------------------\n\n");
   return;                              // Done.
}                                       //

/***********************************************************************
CTRL/C Interrupt Service Routine

This subroutine is called by the C run time library if the operating
system requests that the program terminate for any reason.  It makes
sure that the proggram execution log file is closed and saved.  This
is required and defined by "signal.h".


Input Arguments:

   x                    specified the reason for program termination,
                        defined in "signal.h" as:

                        SIGILL     1  illegal instruction - invalid function image
                        SIGSEGV    2  invalid access to memory
                        SIGFPE     3  floating point exception
                        SIGTERM    4  OS/2 SIGTERM (killprocess) signal
                        SIGABRT    5  abort() signal
                        SIGINT     6  OS/2 SIGINTR signal
                        SIGUSR1    7  user exception in range 0xa0000000 - 0xbfffffff
                        SIGUSR2    8  user exception in range 0xc0000000 - 0xdfffffff
                        SIGUSR3    9  user exception in range 0xe0000000 - 0xffffffff
                        SIGBREAK  10  OS/2 Ctrl-Break sequence


Output Arguments:

   none


Constants Used:

   EXIT_FAILURE         is defined as 8 in "stdlib.h".


Global Data Used:

   logfile              is the handle for the proggram execution log
                        file; NULL if the log file is not open.

   running              is normally TRUE, but set FALSE when the program
                        needs to terminate.  This ends any loops in
                        coroutines.


Object Data Used:

   n/a


Returns:

   none


***********************************************************************/
void ccisr(int x){
   running=FALSE;                       // Stop all loops in the program.
   logerr("CTRL/C typed.");             // Record what happened.
   logerr("Program Shutdown");          // Record session statistics.
   if(logfile){                         // If the program execution log is open, then
      fclose(logfile);                  //  close and save the log file,
      logfile=NULL;                     //  and remember that there is no log file now.
   }                                    //
   exit(EXIT_FAILURE);                  // Call the operating system to terminate the current process and indicate failure.
   return;                              // This should never execute.
}                                       //

/***********************************************************************
Monitor Thread

This asynchronous thread times out I/O operations.


Input Arguments:

   x                    not used; required by _begin_thread().


Output Arguments:

   none


Constants Used:

   none


Global Data Used:

   running              is normanlly TRUE, but set FALSE if the main
                        thread decides to terminate the process.

   timer                is the number of seconds remaining until the
                        current socket I/O operation times out; zero
                        any time no I/O operation in progress.

   s                    is the socket number; -1 if no connection


Object Data Used:

   n/a


Returns:

   none


***********************************************************************/
void monitor(void *x){
   char bufr[256];                      // build messages here
                                        //
   while(running){                      // Do this until the process needs to terminate.
      if(timer>0){                      // If an I/O operation is being timed out, then
         timer--;                       // decrement the timer.
         sprintf(bufr, "%d seconds remaining for socket %d.", timer, s);
         logerr(bufr);                  // Record is happening in the log.
         if(timer==0){                  // If the timer goes to zero, then
            so_cancel(s);               //  cancel the socket I/O. */
            sprintf(bufr, "Timed out operation on socket %d.", s);
            logerr(bufr);               // Record what happened in the log.
         }                              //
      }                                 //
      DosSleep(1000);                   // Wait a second.
   }                                    //
   return;                              // The thread will terminate now.
}                                       //

/***********************************************************************
Connect to a Host

Input Arguments:

   host                 specifies the memory address of a null-
                        terminated ASCII string with the name of the
                        remote server.  This may be dotted decimal.

   port                 specifies the port number to be used.  To
                        use whatever "etc/services" specifies for
                        HTTP, set this to zero.


Output Arguments:

   none


Constants Used:

   AF_INET              is defined as 2 in "sys/socket.h".

   SOCK_STREAM          is defined as 1 in "sys/socket.h".

   SOCEADDRNOTAVAIL     is defined as 10049 in "nerrno.h" - Can't assign
                        requested address.

   SOCENETUNREACH       is defined as 10051 in "nerrno.h" - Network is
                        unreachable (routing problem).

   SOCEHOSTUNREACH      is defined as 10065 in "nerrno.h" - No route to
                        host.

   SOCEAFNOSUPPORT      is defined as 10047 in "nerrno.h" - Address
                        family not supported by protocol family.

   SOCEALREADY          is defined as 10037 in "nerrno.h" - Operation
                        already in progress.

   SOCENOTSOCK          is defined as 10038 in "nerrno.h" - Socket
                        operation on non-socket.

   SOCECONNREFUSED      is defined as 10061 in "nerrno.h" - Connection
                        refused.

   SOCEFAULT            is defined as 10014 in "nerrno.h" - Bad memory
                        address.

   SOCEISCONN           is defined as 10056 in "nerrno.h" - Socket is
                        already connected.

   SOCETIMEDOUT         is defined as 10060 in "nerrno.h" - Connection
                        timed out.

   SOCENOBUFS           is defined as 10055 in "nerrno.h" - No buffer
                        space available.

   SOCEOPNOTSUPP        is defined as 10045 in "nerrno.h" - Operation
                        not supported on socket.


Global Data Used:

   timer                if greater than zero, then the asynchronous
                        monitor() thread decrements at one second
                        intervals and aborts the current operation
                        on socket "s" if the count goes to zero.

   s                    receives the socket number, or -1 if connection
                        fails for any reason.


Object Data Used:

   n/a


Returns:

   TRUE                 if successfully connected

   FALSE                if not


***********************************************************************/
int connect_socket(char *host, int port){
   int test;                            // result of connect()
   unsigned long int iaddr;             // result of inet_addr()
   struct hostent *hp;                  // result of gethostbyname()
   struct servent *sp;                  // result of getservbyname()
   struct sockaddr_in addr;             // addressing scheme, network address and port of the client
   char bufr[256];                      // build messages here
                                        //
   sprintf(bufr, "Attempting to connect with \"%s\" using port %d.", host, port);
   logerr(bufr);                        // Record starting the connection attempt.
   clrmem(inbuf);                       // Clear the input ring buffer
   iptr=eptr=0;                         //  and reset its controls.
/*
   Get the internet address of the news server.
*/
   iaddr=inet_addr(host);               // Try to interpret the host name as a dotted decimal address.
   if(iaddr==-1){                       // If that didn't work, then
      hp=gethostbyname(host);           //  try the domain name service.
      if(!hp){                          // If the DNS couldn't figure it out, then
         sprintf(bufr, "*** ERROR in connect_socket() - no DNS record for \"%s\".", host);
         logerr(bufr);                  //  record an error message,
         return FALSE;                  //  and abort the operation.
      }                                 // If DNS lookup worked, then
      memmove((char*)(&iaddr), hp->h_addr, hp->h_length);  // recover the result.
   }                                    //
   setmem(&addr, 0, sizeof(struct sockaddr_in)); // Initialize the network address structure.
   addr.sin_family=AF_INET;             // The addressing family will be "internet", as opposed to "DecNET", "AppleTalk", "Vines", "Netware", etc.
   addr.sin_addr.s_addr=iaddr;          // Fill in the IP address of the server.
   sprintf(bufr, "Host name \"%s\" translated to \"%d.%d.%d.%d\".", host,
           ((unsigned char *)(&iaddr))[0],
           ((unsigned char *)(&iaddr))[1],
           ((unsigned char *)(&iaddr))[2],
           ((unsigned char *)(&iaddr))[3]);
   logerr(bufr);                        // Record translating the host name successfully.
/*
   Determine which port to use.
*/
   if(port){                            // If a port was specified, then
      addr.sin_port=htons(port);        //  convert it to network byte order.
   } else {                             // Specify port 0 to use whatever "etc/services" defines for HTTP.
      sp=getservbyname("www-http", "tcp");  // Look up HTTP in the "etc/services" file.
      if(!sp){                          // If HTTP is not defined in "etc/services", then
         sprintf(bufr, "*** ERROR %d in connect_socket() determining port for HTTP.", sock_errno());
         logerr(bufr);                  //  record the error in the program execution log,
         return FALSE;                  //  and abort the operation.
      }                                 //
      addr.sin_port=sp->s_port;         // Fill in the port part of the network address structure.
   }                                    //
/*
   Get a socket.
*/
   s=socket(AF_INET, SOCK_STREAM, 0);   // Ask the operating system for a socket.
   if(s==-1){                           // If it could not give one, then
      sprintf(bufr, "*** ERROR %d in connect_socket() attempting to allocate a socket.", sock_errno());
      logerr(bufr);                     //  record an error message,
      return FALSE;                     //  and abort the operation. */
   }                                    //
   sprintf(bufr, "Socket number %d allocated.", s);
   logerr(bufr);                        // Record successfully allocating a socket.
/*
   Connect the socket to the host.
*/
   timer=30;                            // Tell monitor() to abort this if it takes longer than 30 seconds.
   test=connect(s, (struct sockaddr *)(&addr), sizeof(struct sockaddr_in));  // Ask the operating system to connect the socket to the network address.
   timer=0;                             // Tell monitor() to stop counting.
   if(!test){                           // If a connection was obtained, then
      sprintf(bufr, "Connected to \"%s\" port %d.", host, ntohs(addr.sin_port));
      logerr(bufr);                     //  record how long it took,
      return TRUE;                      //  and return the socket number.
   }                                    // If the connection could not be obtained, then
   test=sock_errno();                   //   interpret the problem.
   switch(test){                        //
   case SOCEADDRNOTAVAIL:
   case SOCENETUNREACH:                 // Network Unreachable
   case SOCEHOSTUNREACH:                // Host Unreachable
      sprintf(bufr, "*** ERROR in connect_socket() - \"%s\" is unreachable at this time.", host);
      break;
   case SOCEAFNOSUPPORT:
      sprintf(bufr, "*** ERROR in connect_socket() - \"%s\" does not support internet addressing.", host);
      break;
   case SOCEALREADY:
      sprintf(bufr, "*** ERROR in connect_socket() - Socket %d in use by another process.", s);
      break;
   case SOCENOTSOCK:
      sprintf(bufr, "*** ERROR in connect_socket() - Invalid socket number: %d.", s);
      break;
   case SOCECONNREFUSED:
      sprintf(bufr, "*** ERROR in connect_socket() - \"%s\" is not accepting connections at this time.", host);
      break;
   case SOCEFAULT:
      sprintf(bufr, "*** ERROR in connect_socket() - Invalid sockaddr_in structure, address %08X.", &addr);
      break;
   case SOCEISCONN:
      sprintf(bufr, "*** ERROR in connect_socket() - Socket %d is already connected.", s);
      break;
   case SOCETIMEDOUT:
      sprintf(bufr, "*** ERROR in connect_socket() - connection timed out.");
      break;
   case SOCENOBUFS:
      sprintf(bufr, "*** ERROR in connect_socket() - no buffer space available.");
      break;
   case SOCEOPNOTSUPP:
      sprintf(bufr, "*** ERROR in connect_socket() - socket %d does not support connect().", s);
      break;
   default:
      sprintf(bufr, "*** ERROR %d in connect_socket() connecting to \"%s\" using port %d.", test, host, port);
      break;
   }
   logerr(bufr);                        // Record the error in the program execution log.
   soclose(s);                          // Close the connection.
   s=-1;                                // Tell everyone that the socket is closed.
   return FALSE;                        // Abort the current operation.
}                                       //

/***********************************************************************
Send a String to the Server

This sends an arbitrary, null-terminated, stringg of ASCII characters
to a remote server connected via socket "s".  All text sent is recorded
in the program log.  The connection is closed and "s" is set to -1 if
any error occurs.


Input Arguments:

   str                  specifies the memory address of a null-
                        terminated string of ASCII characters to be
                        sent to the server.

   opt                  TRUE means that the HTTP request is being sent;
                        FALSE means that an HTTP header is being sent.


Output Arguments:

   none


Constants Used:

   none


Global Data Used:

   timer                if greater than zero, then the asynchronous
                        monitor() thread decrements at one second
                        intervals and aborts the current operation
                        on socket "s" if the count goes to zero.

   s                    receives the socket number, or -1 if connection
                        fails for any reason.


Object Data Used:

   n/a


Returns:

   TRUE                 if successfully connected

   FALSE                if not


***********************************************************************/
int soputs(char *str, int opt){
   int len;                             // number of characters in "str"
   int test;                            // result of send()
   char *p;                             // read cursor for "str"
   char temp[128];                      // truncated line for screen output
   char bufr[256];                      // build messages here
                                        //
   len=strlen(str);                     // Count the number of characters to be sent.
   p=str;                               // Initialize a read cursor for "str".
   while(len){                          // Do this until all data has been sent.
      timer=30;                         // Tell monitor() to abort this if it takes longer than 30 seconds.
      test=send(s, p, len, 0);          // Try to send everything that remains.
      timer=0;                          // Tell monitor() to stop counting.
      if(test<0){                       // If send() failed, then record the error.
         test=sock_errno();             // Get details about the error.
         sprintf(bufr, "*** ERROR %d in soputs() sending %d bytes to socket %d.",
                 test, len, s);
         logerr(bufr);                  // Record the error in the program log.
         soclose(s);                    // Close the connection.
         s=-1;                          // Remember that the connection is closed.
         return FALSE;                  // Return failure.
      }                                 //
      p+=test;                          // Advance the read cursor by the number of characters sent.
      len-=test;                        // Subtract the number of characters sent from the number remaining.
   }                                    //
   clrmem(temp);                        // Make sure that the text in "temp" is null-terminated.
   strncpy(temp, str, sizeof(temp)-1);  // Make a copy of the output, preventing overflow.
   xrtrim(temp);                        // Remove and trailing carriage returns, line feeds and white space from the copy.
   temp[70]=0;                          // Truncate the copy at 70 characters to leave room for "CHDR> ", a carriage return and a line feed on an 80 column line.
   if(opt){                             // Distinguish the request from
      sprintf(bufr, "CREQ> %s", temp);  //  the headers in the log.
   } else {                             //
      sprintf(bufr, "CHDR> %s", temp);  //
   }                                    //
   logerr(bufr);                        // Display the header on the screen.
   coredump(str, strlen(str));          // Record everything in the log.
   return TRUE;                         // Return success.
}                                       //

/***********************************************************************
Receive ASCII Line

This function reads socket "s" into a specified buffer until a line feed
is stored or the buffer is full (should not happen).


Input Arguments:

   size                 specifies the number of bytes of memory
                        available at "buf", to avoid overflow.


Output Arguments:

   buf                  specifies the memory address of a buffer into
                        which the incoming line of text can be placed.


Constants Used:

   none


Global Data Used:

   none


Object Data Used:

   n/a


Returns:

   -1                   if the connection with the server is lost while
                        receiving, or

   the number of bytes received.


***********************************************************************/
int sogets(char *buf, int size){
   int i;                               // write cursor for "buf"
   int test;                            // result of sogetc()
                                        //
   setmem(buf, size, 0);                // Initialize the return buffer to all nulls so that the received text can be distinguished if an error aborts the process.
   i=0;                                 // Initialize the return buffer pointer to the first character.
   if(s==-1){                           // If the connection was lost in a prior call, then
      return -1;                        //  return failure.
   }                                    //
   do {                                 // Do this until a line feed is received.
      test=sogetc();                    // Get the next character from the input stream.
      if(test==-1){                     // If the connection was lost, but
         if(i>0){                       //  some data was received, then
            break;                      //  return that data
         }                              //  and let the next call detect
         return -1;                     //  loss of connection.
      }                                 //
      if(i>=size-1){                    // If the buffer is full, then
         break;                         // stop.
      }                                 //
      buf[i++]=test;                    // Store the received character.
   } while(test!='\n');                 // Stop if a line feed was just stored.
   return i;                            // Return the number of characters received.
}                                       //

/***********************************************************************
Get the next Character from the Socket


Input Arguments:

   none


Output Arguments:

   none


Constants Used:

   SOCECONNRESET        is defined as 10054 in "nerrno.h" - Connection
                        reset by peer.


Global Data Used:

   timer                if greater than zero, then the asynchronous
                        monitor() thread decrements at one second
                        intervals and aborts the current operation
                        on socket "s" if the count goes to zero.

   s                    receives the socket number, or -1 if connection
                        fails for any reason.

   inbuf                is a large ring buffer used to maximize the
                        efficiency of operating system calls

   iptr                 is the input buffer insertion pointer, which
                        is the index of the next slot where a character
                        can be deposited.

   eptr                 is the index of the oldest character in "inbuf".
                        When "eptr" reaches "iptr", "inbuf" is empty.
                        When "iptr" reaches "eptr"-1, "inbuf is full.


Object Data Used:

   n/a


Returns:

   -1                   if there is nothing to read and the connection
                        is closed; otherwise,

   the ASCII code of the next character received.


***********************************************************************/
int sogetc(void){
   int test;                            // result of recv()
   char bufr[256];                      // build messages here
                                        //
   if(eptr<iptr){                       // If the input buffer is not empty, then
      return(inbuf[eptr++]);            //  extract the next character and return it.
   }                                    // If the input buffer is empty, then
   iptr=eptr=0;                         //  reset the insertion and extraction pointers.
   if(s==-1){                           // If the connection was lost in a prior call, then
      return -1;                        //  return failure.
   }                                    //
   timer=30;                            // Tell monitor() to kill this operation if it takes more than 30 seconds.
   test=recv(s, inbuf, sizeof(inbuf), 0);  // Try to receive as much as the input buffer can hold.
   timer=0;                             // Tell monitor() to stop counting the timeout.
   if(test>0){                          // If any data was received, then
      iptr+=test;                       //  move the ring buffer insertion pointer to the next available slot.
      return inbuf[eptr++];             // Extract the oldest character and return it.
   }                                    //
   if(test==0){                         // If nothing was received, then assume that the server died.
      logerr("Zero bytes received in sogetc().");
      soclose(s);                       // Close the connection.
      s=-1;                             // Remember that there is no connection.
      return -1;                        // Return failure.
   }                                    // If "test" is less than zero, then an error occurred.
   test=sock_errno();                   // Get details about the error.
   switch(test){                        // Decide what to do.
   case SOCECONNRESET:                  // If the server closed the connection, then
      logerr("Connection closed by server.");
      break;                            //  this is expected at the end of a web page.
   default:                             // If anything else happened, then
      sprintf(bufr, "*** ERROR %d in sogetc() reading socket %d.", test, s);
      logerr(bufr);                     //  record the error in the program execution log.
      break;                            //
   }                                    // For all errors,
   soclose(s);                          //  close the connection,
   s=-1;                                //  remember that the connection is closed,
   return -1;                           //  and return failure.
}                                       //

/***********************************************************************
HTTP Transaction

This is the classic work-around for the "no goto" problem of many modern
programming languages.  It encapsulates a segment of code that would be
in the main() thread, with numerous "goto"s to the clean up code by
using "return" statements as unconditional "goto"s.

It established a TCP connection with the server, builds an HTTP request,
sends the request, and reads everything that the server sends back.  The
transaction can conclude when "Content-length" is satisfied or the
server closes the connection.

Input Arguments:

   argv                 is the array of memory addresses of ASCII
                        strings parsed from the command line by the
                        command line interpretter.  These would be:

      argv[0]           program name

      argv[1]           web server name

      argv[2]           HTTP method ("GET", "POST", etc.)

      argv[3]           file specification and arguments


Output Arguments:

   none


Constants Used:

   none


Global Data Used:

   none


Object Data Used:

   n/a


Returns:

   none


***********************************************************************/
void transact(char *argv[]){
   int i;                               // general loop counter
   int len;                             // number of characters received
   int test;                            // result of many functions
   int chunked;                         // TRUE if body chenked, FALSE if not
   int chunk_size;                      // number of bytes in the current chunk
   int content_length;                  // value of the "Content-Length: " header
   int namelen;                         // number of bytes in "localhost"
   char *p;                             // result of strstr()
   char *mem;                           // address of dynamically allocated memory for receiving the document body
   struct sockaddr_in localhost;        // used to get our IP address for the "Host: " header
   char temp[128];                      // the longest piece of the received line that will not wrap
   char bigbuf[2048];                   // receive headers here
   char bufr[256];                      // build output here
/*
   1. Establish the connection.
*/
   test=connect_socket(argv[1], 0);     // Establish a connection with the specified server.
   if(!test){                           // If the connection could not be established, then
      return;                           //  quit.
   }                                    //
/*
   2. Send the request headers.
*/
   sprintf(bufr, "%s %s HTTP/1.1\r\n", argv[2], argv[3]);  // Format the HTTP request.
   test=soputs(bufr, TRUE);             // Send the request to the server.
   if(!test){                           // If the connection was lost, then
      return;                           //  quit.
   }                                    //
                                        //
   sprintf(bufr, "Host: %s\r\n", argv[1]);  // Specify the host we want.
   test=soputs(bufr, FALSE);            // Send the header to the server.
   if(!test){                           // If the connection was lost, then
      return;                           //  quit.
   }                                    //
                                        //
   test=soputs("User-Agent: WCTEST/1.2\r\n", FALSE);  // Tell the server what client is being used.
   if(!test){                           // If the connection was lost, then
      return;                           //  quit.
   }                                    //
                                        //
   test=soputs("Accept: */*\r\n", FALSE);  // Tell the server that we will accept anything.
   if(!test){                           // If the connection was lost, then
      return;                           //  quit.
   }                                    //
                                        //
   test=soputs("Accept-Language: en-us,en;q=0.5\r\n", FALSE);  // Tell the server what language we want.
   if(!test){                           // If the connection was lost, then
      return;                           //  quit.
   }                                    //
                                        //
   test=soputs("Accept-Charset: ISO-8859-1,utf-8;q=0.7,*;q=0.7\r\n", FALSE);  // Tell the server what character set we want.
   if(!test){                           // If the connection was lost, then
      return;                           //  quit.
   }                                    //
                                        //
   test=soputs("Connection: close\r\n", FALSE);  // Tell the server to close the connection at the end of the page.
   if(!test){                           // If the connection was lost, then
      return;                           //  quit.
   }                                    //
                                        //
   test=soputs("\r\n", FALSE);          // Terminate the HTTP headers.
   if(!test){                           // If the connection was lost, then
      return;                           //  quit.
   }                                    //
/*
   4. Receive the HTTP response.
*/
   len=sogets(bigbuf, sizeof(bigbuf));  // Receive a header line from the server.
   if(len==-1){                         // If the connection is lost, then
      return;                           //  stop early.
   }                                    //
   clrmem(temp);                        // Make sure that the text in "temp" is null-terminated.
   strncpy(temp, bigbuf, sizeof(temp)-1);  // Make a copy of the input, preventing overflow.
   xrtrim(temp);                        // Remove and trailing carriage returns, line feeds and white space from the copy.
   temp[70]=0;                          // Truncate the copy at 70 characters to leave room for "SHDR> ", a carriage return and a line feed on an 80 column line.
   sprintf(bufr, "SRSP> %s", temp);
   logerr(bufr);                        // Display the header on the screen.
   coredump(bigbuf, len);               // Record everything in the log.
/*
   5. Receive headers from the server.
*/
   chunked=FALSE;                       // Assume that the body of the document is not chunked.
   content_length=-1;                   // Assume that the content length is unspecified.
   do {                                 // Do this until the server closes the connection.
      len=sogets(bigbuf, sizeof(bigbuf));  // Receive a header line from the server.
      if(len==-1){                      // If the connection is lost, then
         break;                         //  stop early.
      }                                 //
      clrmem(temp);                     // Make sure that the text in "temp" is null-terminated.
      strncpy(temp, bigbuf, sizeof(temp)-1);  // Make a copy of the input, preventing overflow.
      xrtrim(temp);                     // Remove and trailing carriage returns, line feeds and white space from the copy.
      temp[70]=0;                       // Truncate the copy at 70 characters to leave room for "SHDR> ", a carriage return and a line feed on an 80 column line.
      sprintf(bufr, "SHDR> %s", temp);
      logerr(bufr);                     // Display the header on the screen.
      coredump(bigbuf, len);            // Record everything in the log.
                                        //
      test=strnicmp(bigbuf, "content-length: ", 16);  // Check for a "Content-Length: " header.
      if(test==0){                      // If this is a "Content-Length: " header, then
         content_length=atoi(bigbuf+16);  //  remember the number of bytes in the body.
         sprintf(bufr, "Content Length = %d", content_length);
         logerr(bufr);                  // Record interpreting the content length.
      }                                 //
                                        //
      test=strnicmp(bigbuf, "transfer-encoding: ", 19);  // Check for a "Transfer-Encoding: " header.
      if(test==0){                      // If this is a "Transfer-Encoding: " header, then look for the word "chunked".
         clrmem(temp);                  // Make sure that the copy of the header line is null-terminated.
         strncpy(temp, bigbuf+19, sizeof(temp)-1);  // Make a copy that can be converted to all lower case for case insensitive searching.
         strlwr(temp);                  // Convert the copy to all lower case.
         p=strstr(temp, "chunked");     // Search for the word "chunked".
         if(p){                         // If "chunked" was found, then
            chunked=TRUE;               //  remember that so the end of the body can be detected.
         }                              //
      }                                 //
   } while(temp[0]);                    // Stop when a blank line is received.
/*
   6.  Receive the document body.
*/
   if(chunked){                         // If the body is chunked, then interpret chunk sizes and receive chunks.
      while(running){                   // Do this until the process ends.
         len=sogets(bigbuf, sizeof(bigbuf));  // Receive one line of ASCII text from the server.
         if(len==-1){                   // If the connection was lost while receiving, then
            break;                      //  stop.
         }                              //
         test=sscanf(bigbuf, "%x", &chunk_size);  // Convert the chunk size from hexadecimal to integer.
         if(test!=1){                   // If sscanf() couldn't find a number, then
            chunk_size=0;               //  the number is zero.
         }                              //
         sprintf(bufr, "Chunk Size = %d.", chunk_size);
         logerr(bufr);                  // Record the chunk sizes in the log.
         coredump(bigbuf, len);         // Show what is being interpreted as chunk size.
         if(chunk_size==0){             // A chunk size of zero marks
            break;                      //  the end of the transmission.
         }                              //
         mem=calloc(chunk_size, 1);     // Allocate as much memory as will be needed to hold this chunk.
         if(!mem){                      // If the required memory can not be allocated, then
            sprintf(bufr, "*** ERROR %d trying to allocate %d bytes of memory in transact().", errno, chunk_size);
            logerr(bufr);               //  record the error in the log,
            soclose(s);                 //  close the connection.
            s=-1;                       //  remember that the connection is no longer open,
            return;                     //  and give up.
         }                              //
         for(i=0; i<chunk_size; i++){   // Do this for the number of bytes in the chunk.
            test=sogetc();              // Receive one byte.
            if(test==-1){               // If the connection was lost, then
               return;                  //  give up.
            }                           //
            mem[i]=test;                // Append this byte to the received chunk.
         }                              //
         coredump(mem, i);              // Dump the entire chunk to the log.
         free(mem);                     // Free the buffer used to receive the chunk.
      }                                 //
      soclose(s);                       // Close the connection with the server.
      s=-1;                             // Remember that the connection is closed.
      return;                           // Done.
   }                                    // If not chunked, then use HTTP 1.0 methods.
                                        //
   if(content_length!=-1){              // If the content length was specified, then this is probably binary data.
      mem=calloc(content_length, 1);    // Allocate as much memory as will be needed to hold the entire body.
      if(!mem){                         // If the required memory can not be allocated, then
         sprintf(bufr, "*** ERROR %d trying to allocate %d bytes of memory in transact().", errno, chunk_size);
         logerr(bufr);                  //  record the error in the log,
         soclose(s);                    //  close the connection.
         s=-1;                          //  remember that the connection is no longer open,
         return;                        //  and give up.
      }                                 //
      for(i=0; i<content_length; i++){  // Do this for the number of bytes in the body.
         test=sogetc();                 // Receive one byte.
         if(test==-1){                  // If the connection was lost, then
            return;                     //  give up.
         }                              //
         mem[i]=test;                   // Append this byte to the received chunk.
      }                                 //
      coredump(mem, i);                 // Dump the entire body to the log.
      free(mem);                        // Free the buffer used to receive the body.
      soclose(s);                       // Close the connection with the server.
      s=-1;                             // Remember that the connection is closed.
      return;                           // Done.
   }                                    // If the content length was not specified, then assume that the server will close when done.
                                        //
   while(running){                      // Do this until the server closes the connection.
      len=sogets(bigbuf, sizeof(bigbuf));  // Receive a line of the body from the server.
      if(len==-1){                      // If the connection is lost, then
         break;                         //  stop early.
      }                                 //
      clrmem(temp);                     // Make sure that the text in "temp" is null-terminated.
      strncpy(temp, bigbuf, sizeof(temp)-1);  // Make a copy of the input, preventing overflow.
      xrtrim(temp);                     // Remove and trailing carriage returns, line feeds and white space from the copy.
      temp[70]=0;                       // Truncate the copy at 70 characters to leave room for "SBDY> ", a carriage return and a line feed on an 80 column line.
      sprintf(bufr, "SBDY> %s", temp);
      logerr(bufr);                     // Display the body on the screen.
      coredump(bigbuf, len);            // Record everything in the log.
   }                                    // Stop when the server closes the connection.
   return;                              // Done.
}                                       //

