/*
 * network.c - PM VNC Viewer, Networking Supports
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#define INCL_DOS
#define INCL_PM
#include <os2.h>

#include "pmvncdef.h"

HAB     habNetwork = NULLHANDLE ;
HMQ     hmqNetwork = NULLHANDLE ;

/*
 * VNC Server and Session Options
 */

#define VNCPORTBASE     (5900)

static  int     ServerSock ;

/*
 * Notifications to Window Thread
 */

void    netFail(PUCHAR msg)
{
    WinPostMsg(hwndClient, WM_VNC_FAIL, NULL, MPFROMP(msg)) ;
}

void    netNotify(int ev, MPARAM mp1, MPARAM mp2)
{
    WinPostMsg(hwndClient, ev, mp1, mp2) ;
}

/*
 * thread control variables
 */

static  BOOL    stopThread = FALSE ;
static  BOOL    doneThread = FALSE ;

/*
 * netSend/Recv - network I/O with exact length
 */

BOOL    netSend(PUCHAR buf, int len)
{
    int     cnt, num ;
    
    for (cnt = 0 ; cnt < len ;  ) {
        if (stopThread) {
	    return FALSE ;
	}
	if ((num = send(ServerSock, (buf + cnt), (len - cnt), 0)) <= 0) {
	    return FALSE ;
	}
	cnt += num ;
    }
    return TRUE ;
}

BOOL    netRecv(PUCHAR buf, int len)
{
    int     cnt, num ;
    
    for (cnt = 0 ; cnt < len ;  ) {
        if (stopThread) {
	    return FALSE ;
	}
	if ((num = recv(ServerSock, (buf + cnt), (len - cnt), 0)) <= 0) {
	    return FALSE ;
	}
	cnt += num ;
    }
    return TRUE ;
}

void    netDump(PUCHAR buf, int len)
{
    int     cnt ;
    
    for (cnt = 0 ; cnt < len ; cnt++) {
        printf("%02x ", (buf[cnt] & 0xff)) ;
	if ((cnt % 16) == 15) {
	    printf("\n") ;
	}
    }
    printf("\n") ;
}

/*
 * netThread - networking thread
 */

static  void    netThread(void *arg)
{
    UCHAR   host[256] ;
    UCHAR   mesg[256] ;
    ULONG   port      ;
    ULONG   ipaddr    ;
    struct  sockaddr_in server ;
    struct  hostent     *hp    ;
    
    if (sscanf(SessServerName, "%[^:]:%d", host, &port) != 2) {
        netFail("bad server spec.") ;
        doneThread = TRUE ;
	return ;
    }
    if (port < 100) {
        port += VNCPORTBASE ;
    }

    TRACE("Connect to Server %s:%d\n", host, port) ;

    if (isdigit(host[0])) {
        ipaddr = inet_addr(host) ;
    } else if ((hp = gethostbyname(host)) != NULL) {
        ipaddr = *((ULONG *) hp->h_addr) ;
    } else {
        netFail("no such host") ;
	doneThread = TRUE ;
	return ;
    }
   
    if ((ServerSock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        netFail("failed to create socket") ;
	doneThread = TRUE ;
	return ;
    }
    
    memset(&server, 0, sizeof(server)) ;
    server.sin_family      = AF_INET ;
    server.sin_addr.s_addr = ipaddr  ;
    server.sin_port        = htons((USHORT) port) ;

    if (connect(ServerSock, (struct sockaddr *) &server, sizeof(server)) == -1) {
        sprintf(mesg, "failed to connect (%d)", errno) ;
        netFail(mesg) ;
	doneThread = TRUE ;
	return ;
    }
    
    TRACE("Connected\n") ;
    
    habNetwork = WinInitialize(0) ;
    hmqNetwork = WinCreateMsgQueue(habNetwork, 0) ;
    
    if (protoConnInit() != TRUE) {
	WinDestroyMsgQueue(hmqNetwork) ;
	WinTerminate(habNetwork) ;
        doneThread = TRUE ;
	return ;
    }
    if (protoSendFmtEnc() != TRUE) {
	WinDestroyMsgQueue(hmqNetwork) ;
	WinTerminate(habNetwork) ;
        doneThread = TRUE ;
        return ;
    }
    if (protoSendRequest(FALSE, NULL) != TRUE) {
	WinDestroyMsgQueue(hmqNetwork) ;
	WinTerminate(habNetwork) ;
        doneThread = TRUE ;
        return ;
    }

    while (stopThread == FALSE) {
        if (protoDispatch() != TRUE) {
	    break ;
	}
    }

    (*VncCtx->rectDone) () ;

    WinDestroyMsgQueue(hmqNetwork) ;
    WinTerminate(habNetwork) ;
    doneThread = TRUE ;
    
    return ;
}

/*
 * netStartup - startup networking thread
 */

#define STKSIZ  (1024 * 64)     /* 32KB caused stack overflow */

BOOL    netStartup(HAB hab)
{
    if (_beginthread(netThread, NULL, STKSIZ, NULL) < 0) {
        netFail("failed to start network thread") ;
        return FALSE ;
    }
    return TRUE ;
}

/*
 * netFinish - terminate networking thread
 */

void    netFinish(HAB hab)
{
    int     retry ;
    
    stopThread = TRUE ;
    close(ServerSock) ;
    
    for (retry = 100 ; doneThread == FALSE && retry > 0 ; retry--) {
        DosSleep(50) ;
    }
    return ;
}
