/************************************************************************/
/*                                                                      */
/*              Script to archive the FtpServer log files               */
/*                                                                      */
/*   Depending on what Setup options you have chosen, FtpServer writes  */
/*   up to three log files, whose default names are FTPTRANS.LOG,       */
/*   FTPUSERS.LOG, and COMMON.LOG.  This script moves those three       */
/*   files, with renaming, to the 'LOGS' subdirectory.  It will not     */
/*   attempt to move nonexistent files, so it will work even if you     */
/*   have enabled only a subset of the available logging options.       */
/*   It will also work if you have changed the log file names.          */
/*                                                                      */
/*   Note, however, that the destination directory for the archived     */
/*   files is hard-coded.  If you want to use a different directory,    */
/*   you will have to modify line 73 of this script.  If you want the   */
/*   three log files to go to three different directories, or some      */
/*   similar variant, you will have to modify the code.                 */
/*                                                                      */
/*   Programmer:      P. Moylan                                         */
/*   Last modified:   31 March 2019                                     */
/*                                                                      */
/*   Installation:                                                      */
/*      Put this file anywhere you like, but execute it from the        */
/*      FtpServer main directory, i.e. the directory that contains      */
/*      FTPD.INI or FTPD.TNI.                                           */
/*                                                                      */
/*   Usage:    MoveLog        (with no parameters)                      */
/*                                                                      */
/*   This, however, is for a single execution.  To make this script     */
/*   really useful you should arrange to run it periodically, using     */
/*   any of the cron-like utilities for OS/2.  My own approach is to    */
/*   create a program object for this file, and use the 'schedule'      */
/*   feature of DragText to run it on the first day of every month.     */
/*                                                                      */
/************************************************************************/

call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
call SysLoadFuncs
CALL CheckPrerequisites SelectTNI INIget

IF SelectTNI("FTPD") THEN extension = "TNI"
ELSE extension = "INI"
INIfile = "FTPD."extension
call MoveLogFile INIfile, 'TransLogName', 'FTPTRANS.LOG', 'T'
call MoveLogFile INIfile, 'UserLogName', 'FTPUSERS.LOG', 'U'
call MoveLogFile INIfile, 'CommonLogName', 'COMMON.LOG', 'C'
EXIT 0

/****************************************************************/

MoveLogFile: PROCEDURE

    /* Move one log file.  */

    PARSE ARG INIfile, keyname, default, prefix

    fname = INIget(INIfile, '$SYS', keyname)
    IF fname = '' THEN fname = default
    CALL MoveFile fname, prefix
    RETURN

/****************************************************************/

MoveFile: PROCEDURE

    /* Move one file.  */

    PARSE ARG srcfile, prefix

    IF STREAM(srcfile, 'C', 'QUERY EXISTS') \= '' THEN DO
        dstfile = prefix||DATE('S')||'.log'
        '@RENAME 'srcfile' 'dstfile
        '@MOVE 'dstfile 'logs >nul'
    END /* do */
    RETURN

/****************************************************************/
/*                      CHECKING PREREQUISITES                  */
/****************************************************************/

CheckPrerequisites: PROCEDURE

    /* The argument is a space-separated list of prerequisite   */
    /* functions, for example                                   */
    /*      CALL CheckPrerequisites rxu SelectTNI INIget        */
    /* where (at least in this version) each list item is       */
    /* either 'rxu' or a function from my TNItools package.     */
    /* If any is missing then we exit with an error message.    */

    PARSE UPPER ARG funclist
    funclist = STRIP(funclist)
    needrxu = 0
    needtools = 0
    DO WHILE funclist \= ''
        PARSE VAR funclist func funclist
        funclist = STRIP(funclist)
        IF func = 'RXU' THEN DO

            /* Initialise RXU if not already available, fail if */
            /* the RxFuncAdd operation fails.  We must          */
            /* RxFuncQuery RxuTerm because RxuTerm does not     */
            /* deregister RxuInit.  The RxFuncDrop is needed    */
            /* because RxFuncAdd seems to report failure if the */
            /* function is already registered.                  */

            IF RxFuncQuery('RxuTerm') THEN DO
                CALL RxFuncDrop('RxuInit')
                CALL RxFuncAdd 'RxuInit','RXU','RxuInit'
                IF result THEN DO
                    SAY 'Cannot load RXU'
                    needrxu = 1
                END
                ELSE CALL RxuInit
            END
        END
        ELSE DO
            func = func||'.CMD'
            IF SysSearchPath('PATH', func) = '' THEN DO
                SAY 'ERROR: 'func' must be in your PATH'
                needtools = 1
            END
        END
    END
    IF needrxu THEN SAY 'You can find RXU1a.zip at Hobbes'
    IF needtools THEN SAY 'Please install the TNItools package'
    IF needrxu | needtools THEN EXIT 1
    RETURN

/****************************************************************/

