/****************************************************************/
/*      Add a limited-use account to the FtpServer database     */
/*                                                              */
/*      FtpServer itself will remove the account after it       */
/*                 has been used 'limit' times                  */
/*                                                              */
/*      Author:       Peter Moylan (peter@pmoylan.org)          */
/*      Started:      1 January 2002                            */
/*      Last revised: 30 March 2019                             */
/*                                                              */
/*  Usage:                                                      */
/*         limiteduse username password limit                   */
/*                                                              */
/*  Installation:                                               */
/*         Put this file anywhere you like, but run it from the */
/*          the directory containing FTPD.INI or FTPD.TNI       */
/*         Edit the value of the PathString variable, in        */
/*         line 32 of this script, to reflect the directory     */
/*         you really want the user to use.                     */
/*                                                              */
/*         Remark: if the PathString is really complicated,     */
/*         you might want to develop a solution that uses       */
/*         LoadPRM rather than this script.                     */
/*                                                              */
/****************************************************************/

/****************************************************************/
/*                      EXAMPLE PATHSTRING                      */
/*  You will probably have to modify this for your own needs.   */
/****************************************************************/

PathString = '"/"="C:\Download"'

/****************************************************************/

CALL RxFuncAdd SysLoadFuncs, rexxutil, sysloadfuncs
CALL SysLoadFuncs
CALL CheckPrerequisites SelectTNI INIget INIput

PARSE ARG username password limit
IF username = '' THEN
    DO
        SAY "Usage: limiteduse username password limit"
        SAY "       (The limit is optional and defaults to 1)"
        EXIT 1
    END

IF limit = "" THEN limit = 1

ini = INIFileFor(username)

/* Check for duplicated username. */

IF INIget(ini, username) \= '' THEN
    DO
        SAY "Username "username" already exists"
        SAY "Try a different name"
        EXIT 1
    END

SAY "Updating "ini

/* Create the INI file entries for this new user. */

CALL INIput ini, username, 'Category', '03'X
CALL INIput ini, username, 'Notes', 'Limited-use account'
CALL INIput ini, username, 'Password', password
CALL INIput ini, username, 'RealName', ''
CALL INIput ini, username, 'LoginLimit', Card4(limit)
CALL INIput ini, username, 'SpeedLimit', 'FFFFFFFF'X
CALL INIput ini, username, 'UserLimit', '01000000'X
CALL INIput ini, username, 'Volume', PathString

SAY "Done"
EXIT 0

/****************************************************************/
/*  Procedure to encode an unsigned integer as a four-byte      */
/*  little-endian binary value.  This is elementary in most     */
/*  programming languages, but complicated in Rexx because Rexx */
/*  does not have numeric data types, only strings.             */
/****************************************************************/

Card4:  PROCEDURE

    PARSE ARG val
    result = ""
    DO 4
        byte = val // 256
        val = val % 256
        result = result || D2C(byte,1)
    END
    RETURN result

/****************************************************************/
/*  Procedure to find the INI or TNI file that contains the     */
/*  data for one user.  In many installations this will turn    */
/*  out to be FTPD.INI or FTPD.TNI, but if the option to use    */
/*  multiple INI files is activated then the file name is based */
/*  on a hash coding of the username.  The file is created if   */
/*  it doesn't already exist.                                   */
/****************************************************************/

IniFileFor:  PROCEDURE

    PARSE UPPER ARG username
    IF SelectTNI("FTPD") THEN extension = "TNI"
    ELSE extension = "INI"
    HashMax = INIget('FTPD.'extension, '$SYS', 'HashMax')
    IF HashMax = '' THEN HashMax = 0
    ELSE HashMax = C2D(REVERSE(HashMax))
    IF HashMax = 0 THEN code = ''
    ELSE DO
        code = 0
        DO WHILE username <> ''
            ch = LEFT(username,1)
            username = RIGHT(username, LENGTH(username)-1)
            code = (16*code + C2D(ch)) // HashMax
        END
        code = TRANSLATE(FORMAT(code,4), '0', ' ')
    END
    file = 'FTPD'code'.'extension

    /* Create the file if it doesn't already exist. */

    IF STREAM(file, 'C', 'QUERY EXISTS') = '' THEN DO
        CALL STREAM file, 'C', 'OPEN WRITE'
        CALL STREAM file, 'C', 'CLOSE'
    END

    RETURN file

/****************************************************************/
/*                      CHECKING PREREQUISITES                  */
/****************************************************************/

CheckPrerequisites: PROCEDURE

    /* The argument is a space-separated list of prerequisite   */
    /* functions, for example                                   */
    /*      CALL CheckPrerequisites rxu SelectTNI INIget        */
    /* where (at least in this version) each list item is       */
    /* either 'rxu' or a function from my TNItools package.     */
    /* If any is missing then we exit with an error message.    */

    PARSE UPPER ARG funclist
    funclist = STRIP(funclist)
    needrxu = 0
    needtools = 0
    DO WHILE funclist \= ''
        PARSE VAR funclist func funclist
        funclist = STRIP(funclist)
        IF func = 'RXU' THEN DO

            /* Initialise RXU if not already available, fail if */
            /* the RxFuncAdd operation fails.  We must          */
            /* RxFuncQuery RxuTerm because RxuTerm does not     */
            /* deregister RxuInit.  The RxFuncDrop is needed    */
            /* because RxFuncAdd seems to report failure if the */
            /* function is already registered.                  */

            IF RxFuncQuery('RxuTerm') THEN DO
                CALL RxFuncDrop('RxuInit')
                CALL RxFuncAdd 'RxuInit','RXU','RxuInit'
                IF result THEN DO
                    SAY 'Cannot load RXU'
                    needrxu = 1
                END
                ELSE CALL RxuInit
            END
        END
        ELSE DO
            func = func||'.CMD'
            IF SysSearchPath('PATH', func) = '' THEN DO
                SAY 'ERROR: 'func' must be in your PATH'
                needtools = 1
            END
        END
    END
    IF needrxu THEN SAY 'You can find RXU1a.zip at Hobbes'
    IF needtools THEN SAY 'Please install the TNItools package'
    IF needrxu | needtools THEN EXIT 1
    RETURN

/****************************************************************/

