/***************************************************************************/
/*
    usbcalls.c

    The original code is USBCALLS
        Copyright (c) 2001,2002 Markus Montkowski
        Released under the Aladdin Free Public License
        (Version 9, September 18, 2000)
        See LICENSE for details

    Modifications & additions to the original code are
        Copyright (c) 2006 Richard L Walsh
        Released March 15, 2006

*/
/***************************************************************************/
/*
    This version of usbcalls has been restructured to permit building
    it either as a dll or as a static library.  It also permits you to
    exclude functionality you don't need.  However, if you build a dll
    but don't include everything, please do *not* name it "usbcalls"!

    A set of #defines in usbcalls.h controls whether to build a dynamic
    or a static library and which features to exclude.  By default, all
    features are included when building a dll.

        USB_BUILD_STATIC    build a static library instead of a dll
        USB_NO_BULK         exclude Bulk Transfer Handling
        USB_NO_NOTIFY       exclude Device Add/Remove Notification
        USB_NO_INTERRUPT    exclude Interrupt Handling
        USB_NO_ISO          exclude Isochronous Transfer Handling
        USB_NO_REXX         exclude REXX interface

    Code to query devices, open & close them, and to send control
    messages to them are always included.

    To support use as a static library, two functions have been
    added to initialize the driver and to terminate its use:
        UsbDriverInit() & UsbDriverTerm()
    In the dll version, these calls are no-ops - _DLL_InitTerm()
    performs these functions automatically.

*/
/***************************************************************************/

#define INCL_DOSERRORS
#define INCL_DOSSEMAPHORES
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#include <OS2.h>
#include <stdio.h>
#include <string.h>
#include "usbcalls.h"

#ifdef __cplusplus
  extern "C" {
#endif

/***************************************************************************/
// Common to all
/***************************************************************************/

static APIRET UsbDriverInitInternal( void);
static APIRET UsbDriverTermInternal( void);
static BOOL   IsBadWritePointer( PVOID pBase, ULONG ulSize);

#define  IOCAT_USBRES            0x000000A0  // USB Resource device control
#define  IOCTLF_NUMDEVICE        0x00000031  // Get Number of pluged in Devices
#define  IOCTLF_GETINFO          0x00000032  // Get Info About a device
#define  IOCTLF_AQUIREDEVICE     0x00000033
#define  IOCTLF_RELEASEDEVICE    0x00000034
#define  IOCTLF_GETSTRING        0x00000035
#define  IOCTLF_SENDCONTROLURB   0x00000036
#define  IOCTLF_SENDBULKURB      0x00000037  // Send
#define  IOCTLF_START_IRQ_PROC   0x00000038  // Start IRQ polling in a buffer
#define  IOCTLF_GETDEVINFO       0x00000039  // Get information about device
#define  IOCTLF_STOP_IRQ_PROC    0x0000003A  // Stop IRQ Polling
#define  IOCTLF_START_ISO_PROC   0x0000003B  // Start ISO buffering in a Ringbuffer
#define  IOCTLF_STOP_ISO_PROC    0x0000003C  // Stop ISO buffering
#define  IOCTLF_CANCEL_IORB      0x0000003D  // Abort an IO;
#define  IOCTLF_REG_STATUSSEM    0x00000041  // Register Semaphore for general Statuschange
#define  IOCTLF_DEREG_STATUSSEM  0x00000042  // Deregister Semaphore
#define  IOCTLF_REG_DEVICESEM    0x00000043  // Register Semaphore for a vendor&deviceID
#define  IOCTLF_DEREG_DEVICESEM  0x00000044  // Deregister Semaphore

typedef struct
{
  USHORT    usVendorID;
  USHORT    usProductID;
  USHORT    usBCDDevice;
  USHORT    usDeviceNumber; // Get the usDeviceNumber device in the system fi. if 2 aquire the 2nd device
                            // 0 means first not aquired device.
} AQUIREDEV, *PAQUIREDEV;

typedef struct
{
  ULONG     ulHandle;
  UCHAR     bRequestType;
  UCHAR     bRequest;
  USHORT    wValue;
  USHORT    wIndex;
  USHORT    wLength;
  ULONG     ulTimeout;      // in milliseconds
} USBCALLS_CTRL_REQ, *PUSBCALLS_CTRL_REQ;

HFILE   g_hUSBDrv   = 0;
BOOL    g_fInit     = 0;
char    szDevice[]  = "USBRESM$";

/***************************************************************************/
// Bulk Transfer Handling
/***************************************************************************/

#ifndef USB_NO_BULK

typedef struct
{
  ULONG     ulDevHandle;
  ULONG     ulEventDone;
  UCHAR     ucEndpoint;
  UCHAR     ucInterface;
  USHORT    usDataProcessed;
  USHORT    usDataRemain;
  USHORT    usStatus;
} USBCALLS_BULK_REQ, *PUSBCALLS_BULK_REQ;

#endif // USB_NO_BULK

/***************************************************************************/
// Device Add/Remove Notification
/***************************************************************************/

#ifndef USB_NO_NOTIFY

static APIRET UsbNotifyInit( void);
static void   UsbNotifyTerm( void);

#define NOTIFY_FREE         0
#define NOTIFY_CHANGE       1
#define NOTIFY_DEVICE       2
#define MAX_NOTIFICATIONS   256

#define DEV_SEM_ADD         0x00000001
#define DEV_SEM_REMOVE      0x00000002
#define DEV_SEM_MASK        0x00000003
#define DEV_SEM_VENDORID    0x00000004
#define DEV_SEM_PRODUCTID   0x00000008
#define DEV_SEM_BCDDEVICE   0x00000010

typedef struct
{
  HEV       hDeviceAdded;
  HEV       hDeviceRemoved;
  USHORT    usFlags;
  USHORT    usVendor;
  USHORT    usProduct;
  USHORT    usBCDDevice;
} NOTIFYENTRY, *PNOTIFYENTRY;

typedef struct{
  ULONG     ulSize;
  ULONG     ulCaps;
  ULONG     ulSemDeviceAdd;
  ULONG     ulSemDeviceRemove;
} STATUSEVENTSET, * PSTATUSEVENTSET;

typedef struct{
  ULONG     ulSize;
  ULONG     ulCaps;
  ULONG     ulSemDeviceAdd;
  ULONG     ulSemDeviceRemove;
  USHORT    usVendorID;
  USHORT    usProductID;
  USHORT    usBCDDevice;
  USHORT    usReserved;
} DEVEVENTSET, * PDEVEVENTSET;

ULONG   g_ulFreeNotifys = 0;
HMTX    g_hSemNotifytable = 0;
NOTIFYENTRY g_Notifications[MAX_NOTIFICATIONS];

#endif // USB_NO_NOTIFY

/***************************************************************************/
// Interrupt Handling
/***************************************************************************/

#ifndef USB_NO_INTERRUPT

typedef struct
{
  ULONG     ulDevHandle;
  ULONG     ulIrqEvent;
  UCHAR     ucEndpoint;
  UCHAR     ucInterface;
}USBCALLS_IRQ_START, *PUSBCALLS_IRQ_START;

#endif // USB_NO_INTERRUPT

/***************************************************************************/
// ISO Handling
/***************************************************************************/

#ifndef USB_NO_ISO

static APIRET UsbIsoInit( void);
static void   UsbIsoTerm( void);

#define ISO_DIRMASK 0x80

typedef struct
{
  ULONG     ulDevHandle;
  UCHAR     ucEndpoint;
  UCHAR     ucInterface;
} USBCALLS_ISO_START, *PUSBCALLS_ISO_START;

typedef struct
{
  ULONG     hSemAccess;        // Syncronise access to the Pos values
  ULONG     hDevice;
  USHORT    usPosWrite;
  USHORT    usPosRead;
  USHORT    usBufSize;
  UCHAR     ucEndpoint;
  UCHAR     ucInterface;
  UCHAR     ucBuffer[16*1023];
} ISORINGBUFFER, *PISORINGBUFFER;

ULONG   g_ulNumIsoRingBuffers = 0;
HMTX    g_hSemRingBuffers = 0;
PISORINGBUFFER g_pIsoRingBuffers = 0;

#endif // USB_NO_ISO

/***************************************************************************/
// Common code
/***************************************************************************/

// for the static-lib version, this must be called by the app,
// either directly or indirectly (e.g. by libusb->usb_os_init);
// for the dll version, this is a no-op if called by an app -
// _DLL_InitTerm will call UsbDriverInitInternal() directly <RW>

APIRET APIENTRY UsbDriverInit( void)
{
#ifdef USB_BUILD_STATIC
  return UsbDriverInitInternal();
#else
  return 0;
#endif
}

/***************************************************************************/

static APIRET UsbDriverInitInternal( void)
{
  ULONG     ulAction;
  APIRET    rc;

  if (g_fInit)
    return 0;

do {
  rc = DosOpen( szDevice,
                &g_hUSBDrv,
                &ulAction,
                0,
                FILE_NORMAL,
                OPEN_ACTION_OPEN_IF_EXISTS,
                OPEN_ACCESS_READWRITE |
                OPEN_FLAGS_NOINHERIT |
                OPEN_SHARE_DENYNONE,
                0 );

  if (rc)
    break;

#ifndef USB_NO_NOTIFY
  rc = UsbNotifyInit();
  if (rc)
    break;
#endif

#ifndef USB_NO_ISO
  rc = UsbIsoInit();
  if (rc)
    break;
#endif

  g_fInit = TRUE;

} while (0);

  if (rc)
    UsbDriverTermInternal();

  return rc;
}

/***************************************************************************/

// for the static-lib version, this must be called by the app,
// either directly or indirectly (e.g. by libusb->usb_os_term);
// for the dll version, this is a no-op if called by an app -
// _DLL_InitTerm will call UsbDriverTermInternal() directly <RW>

APIRET APIENTRY UsbDriverTerm( void)
{
#ifdef USB_BUILD_STATIC
  return UsbDriverTermInternal();
#else
  return 0;
#endif
}

/***************************************************************************/

static APIRET UsbDriverTermInternal( void)
{
  APIRET    rc = 0;

  g_fInit = FALSE;

#ifndef USB_NO_NOTIFY
  UsbNotifyTerm();
#endif

#ifndef USB_NO_ISO
  UsbIsoTerm();
#endif

  if (g_hUSBDrv) {
    rc = DosClose(g_hUSBDrv);
    g_hUSBDrv = 0;
  }

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbQueryNumberDevices( ULONG *pulNumDev)
{
  APIRET    rc;
  ULONG     ulLength;

  if (!g_fInit)
    return USB_NOT_INIT;

  if (IsBadWritePointer( pulNumDev, sizeof(ULONG)))
    return ERROR_INVALID_PARAMETER;

  ulLength = sizeof(ULONG);
  *pulNumDev = 0;
  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_NUMDEVICE,
                    NULL, 0, NULL,
                    pulNumDev, ulLength, &ulLength);

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbQueryDeviceReport( ULONG ulDevNumber,
                                      ULONG *pulBufLen,
                                      CHAR *pData)
{
  APIRET    rc;
  ULONG     ulParmLen;

  if (!g_fInit)
    return USB_NOT_INIT;

  if (IsBadWritePointer( pulBufLen, sizeof(ULONG)))
    return ERROR_INVALID_PARAMETER;

  if (pData != NULL && IsBadWritePointer( pData, *pulBufLen))
    return ERROR_INVALID_PARAMETER;

  if (pData == NULL)
   *pulBufLen = 0;
  ulParmLen = sizeof(ulDevNumber);
  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_GETINFO,
                    (PVOID)&ulDevNumber, ulParmLen, &ulParmLen,
                    pData, *pulBufLen, pulBufLen);

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbOpen( PUSBHANDLE pHandle,
                         USHORT usVendor,
                         USHORT usProduct,
                         USHORT usBCDDevice,
                         USHORT usEnumDevice)
{
  APIRET    rc;
  AQUIREDEV Aquire;
  ULONG     ulParmLen, ulDataLen;

  if (!g_fInit)
    return USB_NOT_INIT;

  if (IsBadWritePointer(pHandle,sizeof(USBHANDLE)))
    return ERROR_INVALID_PARAMETER;

  Aquire.usVendorID     = usVendor;
  Aquire.usProductID    = usProduct;
  Aquire.usBCDDevice    = usBCDDevice;
  Aquire.usDeviceNumber = usEnumDevice;
  ulParmLen = sizeof(Aquire);
  ulDataLen = sizeof(USBHANDLE);
  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_AQUIREDEVICE,
                    &Aquire, ulParmLen, &ulParmLen,
                    pHandle, ulDataLen, &ulDataLen);

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbClose( USBHANDLE Handle)
{
  APIRET rc;
  ULONG ulDataLen,ulParmLen;

  if (!g_fInit)
    return USB_NOT_INIT;

  ulParmLen = sizeof(USBHANDLE);
  ulDataLen = 0;

  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_RELEASEDEVICE,
                    (PVOID)&Handle, ulParmLen, &ulParmLen,
                    NULL, ulDataLen, &ulDataLen);

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbCtrlMessage( USBHANDLE Handle,
                                UCHAR  ucRequestType,
                                UCHAR  ucRequest,
                                USHORT usValue,
                                USHORT usIndex,
                                USHORT usLength,
                                UCHAR  *pData,
                                ULONG  ulTimeout)
{
  APIRET rc;
  USBCALLS_CTRL_REQ CtrlRequest;
  ULONG   ulParmLen, ulDataLen;

  if (!g_fInit)
    return USB_NOT_INIT;

  ulParmLen = sizeof(USBCALLS_CTRL_REQ);
  CtrlRequest.ulHandle     = Handle;
  CtrlRequest.bRequestType = ucRequestType;
  CtrlRequest.bRequest     = ucRequest;
  CtrlRequest.wValue       = usValue;
  CtrlRequest.wIndex       = usIndex;
  CtrlRequest.wLength      = usLength;
  CtrlRequest.ulTimeout    = ulTimeout;
  ulDataLen = usLength;

  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_SENDCONTROLURB,
                    (PVOID)&CtrlRequest, ulParmLen, &ulParmLen,
                    ulDataLen>0?(PVOID)pData:NULL,
                    ulDataLen,
                    ulDataLen>0?&ulDataLen:NULL);

  return rc;
}

/***************************************************************************/

static BOOL IsBadWritePointer( PVOID pBase, ULONG ulSize)
{
  APIRET    rc;
  ULONG     ulFlags;

  rc = DosQueryMem( pBase, &ulSize, &ulFlags);

  return (rc != 0 ||
          (ulFlags & (PAG_WRITE | PAG_COMMIT)) != (PAG_WRITE | PAG_COMMIT));
}

/***************************************************************************/
// Bulk Transfer Handling
/***************************************************************************/

#ifndef USB_NO_BULK

// This is a modified version of the original code that works around
// two problems: (1) if the transfer buffer is the last page in a
// segment (i.e. at address 0xF000), nothing gets transferred;
// (2) a timeout may occur even if the transfer completed succesfully.
// Unfortunately, the biggest problem of all remains:  IOCTLF_CANCEL_IORB
// doesn't work.  If the request times-out, the app's stack segment will
// remain locked & can't be freed without a reboot! <RW>

APIRET APIENTRY UsbBulkRead( USBHANDLE Handle,
                             UCHAR  Endpoint,
                             UCHAR  Interface,
                             ULONG  *ulNumBytes,
                             UCHAR  *pData,
                             ULONG  ulTimeout)
{
  UCHAR     ucTempBuf[0x3000];
  APIRET    rc;
  BOOL      fAligned;
  ULONG     ulParmLen, ulDataLen;
  ULONG     ulToProcess, ulTotalProcessed;
  ULONG     ulTempEventCount;
  PUCHAR    pucTempBuf = (PUCHAR)(((ULONG)&ucTempBuf+0x1000) & 0xFFFFF000);
  PUCHAR    pucBuffer;
  USBCALLS_BULK_REQ BulkRequest;

  if (!g_fInit)
    return USB_NOT_INIT;

  // 10 01 2003 - KIEWITZ -> Still @@ToDo Add Endpoint check based on descriptors
  // We currently only allow Endpoint-addresses 80h->8Fh here
  if ((Endpoint<0x80) || (Endpoint>0x8F))
     return USB_ERROR_INVALID_ENDPOINT;

  if (*ulNumBytes == 0)
     return 0;

  rc = DosCreateEventSem( NULL,
                         (PHEV)&BulkRequest.ulEventDone,
                         DC_SEM_SHARED,
                         FALSE);
  if (rc)
     return rc;

  BulkRequest.ulDevHandle = Handle;
  BulkRequest.ucEndpoint  = Endpoint;
  BulkRequest.ucInterface = Interface;
  ulParmLen               = sizeof(USBCALLS_BULK_REQ);
  ulToProcess             = *ulNumBytes;
  ulTotalProcessed        = 0;

  // transfers will fail if the temp buffer is the last page in a segment
  if (((ULONG)pucTempBuf & 0xF000) == 0xF000)
    pucTempBuf += 0x1000;

  do 
  {
    BulkRequest.usDataProcessed = 0;
    BulkRequest.usDataRemain = 0;
    BulkRequest.usStatus = 0;

    // Copy as much as 4k
    ulDataLen = (ulToProcess > 0x1000) ? 0x1000 : ulToProcess;

    // if the caller's buffer is page-aligned & not the last page
    // in a segment, use it - otherwise, use our temp buffer
    fAligned = (((ULONG)pData & 0x0FFF) == 0 && ((ULONG)pData & 0xF000) != 0xF000);
    if (fAligned)
      pucBuffer = pData;
    else
      pucBuffer = pucTempBuf;

    rc = DosDevIOCtl( g_hUSBDrv,
                      IOCAT_USBRES, IOCTLF_SENDBULKURB,
                      (PVOID)&BulkRequest, ulParmLen, &ulParmLen,
                      pucBuffer, ulDataLen, &ulDataLen);
    if (rc) 
      break;

    // We made the request, now wait for it to finish...
    rc = DosWaitEventSem( (HEV)BulkRequest.ulEventDone,
                          (ulTimeout ? ulTimeout : 0xFFFFFFFF));

    if (rc == ERROR_TIMEOUT) 
    {
      // some timeouts are spurious, so confirm this one is valid
      if (BulkRequest.usDataProcessed == 0 || BulkRequest.usStatus != 0)
      {
        printf( "USBCALLS: read timeout - processed= %x  remain= %x  status= %x\n",
                 BulkRequest.usDataProcessed, BulkRequest.usDataRemain, BulkRequest.usStatus);

        // this should abort Bulk-Reading but it actually does nothing
        DosDevIOCtl( g_hUSBDrv,
                     IOCAT_USBRES, IOCTLF_CANCEL_IORB,
                     (PVOID)&BulkRequest, ulParmLen, &ulParmLen,
                     NULL, 0, NULL);

        break;
      }
      printf( "USBCALLS: read timeout ignored - processed= %x  remain= %x  status= %x\n",
               BulkRequest.usDataProcessed, BulkRequest.usDataRemain, BulkRequest.usStatus);
    }
    // Reset semamorph for next block (if available)
    DosResetEventSem( (HEV)BulkRequest.ulEventDone, &ulTempEventCount);

    if (!fAligned)
       memcpy( pData, pucBuffer, BulkRequest.usDataProcessed);

    // Adjust count and source pointer
    ulToProcess      -= ulDataLen;
    pData            += ulDataLen;
    ulTotalProcessed += BulkRequest.usDataProcessed;

    if (BulkRequest.usDataProcessed != ulDataLen) 
    {
      // Transfered less than we wanted? so something is wrong, abort
      rc = USB_ERROR_LESSTRANSFERED; 
      break;
    }

  } while (ulToProcess > 0);

  DosCloseEventSem( (HEV)BulkRequest.ulEventDone);
  *ulNumBytes = ulTotalProcessed;

  return rc;
}

/***************************************************************************/

// This code should be subject to the same problems as UsbBulkRead but
// I haven't encountered them.  See the comments above for details. <RW>

APIRET APIENTRY UsbBulkWrite( USBHANDLE Handle,
                              UCHAR  Endpoint,
                              UCHAR  Interface,
                              ULONG  ulNumBytes,
                              UCHAR  *pData,
                              ULONG  ulTimeout)
{
  UCHAR     ucTempBuf[0x3000];
  APIRET    rc;
  BOOL      fAligned;
  ULONG     ulParmLen, ulDataLen;
  ULONG     ulTempEventCount;
  PUCHAR    pucTempBuf = (PUCHAR)(((ULONG)&ucTempBuf+0x1000) & 0xFFFFF000);
  PUCHAR    pucBuffer;
  USBCALLS_BULK_REQ BulkRequest;

  if (!g_fInit)
    return USB_NOT_INIT;

  // 10 01 2003 - KIEWITZ -> Still @@ToDo Add Endpoint check based on descriptors
  // We currently only allow Endpoint-addresses 00h->0Fh here
  if (Endpoint > 0x0F)
     return USB_ERROR_INVALID_ENDPOINT;

  if (ulNumBytes == 0)
     return 0;

  rc = DosCreateEventSem( NULL, (PHEV)&BulkRequest.ulEventDone,
                          DC_SEM_SHARED, FALSE);
  if (rc)
     return rc;

  BulkRequest.ulDevHandle = Handle;
  BulkRequest.ucEndpoint  = Endpoint;
  BulkRequest.ucInterface = Interface;
  ulParmLen = sizeof(USBCALLS_BULK_REQ);

  // transfers will fail if the temp buffer is the last page in a segment
  if (((ULONG)pucTempBuf & 0xF000) == 0xF000)
    pucTempBuf += 0x1000;

  do
  {
    // Copy as much as 4k
    ulDataLen = (ulNumBytes > 0x1000) ? 0x1000 : ulNumBytes;

    fAligned = (((ULONG)pData & 0x0FFF) == 0 && ((ULONG)pData & 0xF000) != 0xF000);
    if (fAligned)
      pucBuffer = pData;
    else
    {
      pucBuffer = pucTempBuf;
      memcpy( pucBuffer, pData, ulDataLen);
    }

    rc = DosDevIOCtl( g_hUSBDrv,
                      IOCAT_USBRES, IOCTLF_SENDBULKURB,
                      (PVOID)&BulkRequest, ulParmLen, &ulParmLen,
                      pucBuffer, ulDataLen, &ulDataLen );
    if (rc)
      break;

    // We made the request, now wait for it to finish...
    rc = DosWaitEventSem( (HEV)BulkRequest.ulEventDone,
                          (ulTimeout ? ulTimeout : 0xFFFFFFFF));

    if (rc == ERROR_TIMEOUT) 
    {
      // some timeouts are spurious, so confirm this one is valid
      if (BulkRequest.usDataProcessed == 0 || BulkRequest.usStatus != 0)
      {
        printf( "USBCALLS: write timeout - processed= %x  remain= %x  status= %x\n",
                 BulkRequest.usDataProcessed, BulkRequest.usDataRemain, BulkRequest.usStatus);

        // this should abort Bulk-Writing but it actually does nothing
        DosDevIOCtl( g_hUSBDrv,
                     IOCAT_USBRES, IOCTLF_CANCEL_IORB,
                     (PVOID)&BulkRequest, ulParmLen, &ulParmLen,
                     NULL, 0, NULL);

        break;
      }
      printf( "USBCALLS: write timeout ignored - processed= %x  remain= %x  status= %x\n",
                BulkRequest.usDataProcessed, BulkRequest.usDataRemain, BulkRequest.usStatus);
    }

    // Reset semamorph for next block (if available)
    DosResetEventSem( (HEV)BulkRequest.ulEventDone, &ulTempEventCount);

    // Adjust count and source pointer
    ulNumBytes -= ulDataLen;
    pData      += ulDataLen;

  } while (ulNumBytes > 0);

  DosCloseEventSem( (HEV)BulkRequest.ulEventDone);

  return rc;
}

#endif // USB_NO_BULK

/***************************************************************************/
// Device Add/Remove Notification
/***************************************************************************/

#ifndef USB_NO_NOTIFY

static APIRET UsbNotifyInit( void)
{
  memset( g_Notifications, 0, sizeof(g_Notifications));
  return (DosCreateMutexSem( NULL, &g_hSemNotifytable, DC_SEM_SHARED, FALSE));
}

/***************************************************************************/

static void UsbNotifyTerm( void)
{

#if 0
  // @@ ToDo deregister all Events on unload
  if (g_fInit)
  {
    int i;
    for (i = 0; i < MAX_NOTIFICATIONS; i++)
      if (g_Notifications[i].usFlags != NOTIFY_FREE)
        ; 
  }
#endif

  if (g_hSemNotifytable)
    DosCloseMutexSem( g_hSemNotifytable);

  g_hSemNotifytable = 0;

  return;
}

/***************************************************************************/

APIRET APIENTRY UsbRegisterChangeNotification( PUSBNOTIFY pNotifyID,
                                               HEV hDeviceAdded,
                                               HEV hDeviceRemoved)
{
  APIRET    rc;
  int       i;
  ULONG     ulSize;
  STATUSEVENTSET EventSet;

  if (!g_fInit)
    return USB_NOT_INIT;

  if (IsBadWritePointer( pNotifyID, sizeof(ULONG)) ||
      (hDeviceAdded == 0 && hDeviceRemoved == 0))
    return ERROR_INVALID_PARAMETER;

  ulSize = sizeof(EventSet);
  EventSet.ulSize = ulSize;

  if (hDeviceAdded)
  {
    ULONG ulCnt;
    rc = DosQueryEventSem(hDeviceAdded,&ulCnt);
    if (rc)
      return rc;
    EventSet.ulCaps         = DEV_SEM_ADD;
    EventSet.ulSemDeviceAdd = hDeviceAdded;
  }

  if (hDeviceRemoved)
  {
    ULONG ulCnt;
    rc = DosQueryEventSem( hDeviceRemoved, &ulCnt);
    if (rc)
      return rc;
    EventSet.ulCaps            |= DEV_SEM_REMOVE;
    EventSet.ulSemDeviceRemove = hDeviceRemoved;
  }

  rc = DosRequestMutexSem( g_hSemNotifytable, SEM_INDEFINITE_WAIT);
  if (rc)
    return rc;

  for (i = 0; i < MAX_NOTIFICATIONS; i++)
  {
    if (g_Notifications[i].usFlags == NOTIFY_FREE)
    {
      g_Notifications[i].usFlags        = NOTIFY_CHANGE;
      g_Notifications[i].hDeviceAdded   = hDeviceAdded;
      g_Notifications[i].hDeviceRemoved = hDeviceRemoved;
      g_Notifications[i].usVendor       = 0;
      g_Notifications[i].usProduct      = 0;
      g_Notifications[i].usBCDDevice    = 0;
      break;
    }
  }
  DosReleaseMutexSem( g_hSemNotifytable);
  if (i == MAX_NOTIFICATIONS)
    return USB_ERROR_NO_MORE_NOTIFICATIONS;

  // @@ToDo come up with a better way to generate IDs
  *pNotifyID = (USBNOTIFY)(&g_Notifications[i]);
  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_REG_STATUSSEM,
                    NULL, 0, NULL,
                    &EventSet,ulSize, &ulSize);
  if (rc)
  {
    g_Notifications[i].usFlags = NOTIFY_FREE;
    *pNotifyID = 0;
  }

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbRegisterDeviceNotification( PUSBNOTIFY pNotifyID,
                                               HEV hDeviceAdded,
                                               HEV hDeviceRemoved,
                                               USHORT usVendor,
                                               USHORT usProduct,
                                               USHORT usBCDVersion)
{
  APIRET    rc;
  ULONG     ulCnt;
  ULONG     ulSize;
  int       i;
  DEVEVENTSET EventSet;

  if (!g_fInit)
    return USB_NOT_INIT;

  if (IsBadWritePointer( pNotifyID, sizeof(ULONG)) ||
      hDeviceAdded == 0 || hDeviceRemoved == 0 ||
      usVendor == 0     || usVendor == 0xFFFF  ||
      usProduct == 0    || usProduct == 0xFFFF )
    return ERROR_INVALID_PARAMETER;

  rc = DosQueryEventSem( hDeviceAdded, &ulCnt);
  if (rc)
    return rc;
  rc = DosQueryEventSem( hDeviceRemoved, &ulCnt);
  if (rc)
    return rc;

  ulSize = sizeof(EventSet);
  EventSet.ulSize            = ulSize;
  EventSet.ulCaps            = DEV_SEM_ADD | DEV_SEM_REMOVE |
                               DEV_SEM_VENDORID | DEV_SEM_PRODUCTID |
                               DEV_SEM_BCDDEVICE ;
  EventSet.ulSemDeviceAdd    = hDeviceAdded;
  EventSet.ulSemDeviceRemove = hDeviceRemoved;
  EventSet.usVendorID        = usVendor;
  EventSet.usProductID       = usProduct;
  EventSet.usBCDDevice       = usBCDVersion;
  EventSet.usReserved        = 0;

  rc = DosRequestMutexSem( g_hSemNotifytable, SEM_INDEFINITE_WAIT);
  if (rc)
    return rc;

  for (i = 0; i < MAX_NOTIFICATIONS; i++)
  {
    if (g_Notifications[i].usFlags == NOTIFY_FREE)
    {
      g_Notifications[i].usFlags        = NOTIFY_DEVICE;
      g_Notifications[i].hDeviceAdded   = hDeviceAdded;
      g_Notifications[i].hDeviceRemoved = hDeviceRemoved;
      g_Notifications[i].usVendor       = usVendor;
      g_Notifications[i].usProduct      = usProduct;
      g_Notifications[i].usBCDDevice    = usBCDVersion;
      break;
    }
  }
  DosReleaseMutexSem( g_hSemNotifytable);
  if (i == MAX_NOTIFICATIONS)
    return USB_ERROR_NO_MORE_NOTIFICATIONS;

  // @@ToDo come up with a better way to generate IDs
  *pNotifyID = (USBNOTIFY)(&g_Notifications[i]);
  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_REG_DEVICESEM,
                    NULL, 0, NULL,
                    &EventSet,ulSize, &ulSize);
  if (rc)
  {
    g_Notifications[i].usFlags = NOTIFY_FREE;
    *pNotifyID = 0;
  }

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbDeregisterNotification( USBNOTIFY NotifyID)
{
  APIRET    rc;
  USBNOTIFY MinID;
  USBNOTIFY MaxID;
  ULONG     Index;
  ULONG     ulFunction;
  ULONG     ulSize;
  DEVEVENTSET EventSet;

  if (!g_fInit)
    return USB_NOT_INIT;

  MinID = (USBNOTIFY)(&g_Notifications[0]);
  MaxID = (USBNOTIFY)(&g_Notifications[MAX_NOTIFICATIONS-1]);

  if (NotifyID < MinID || NotifyID > MaxID)
    return ERROR_INVALID_PARAMETER;

  Index = NotifyID - MinID;

  if (Index % sizeof(NOTIFYENTRY))
    return ERROR_INVALID_PARAMETER;

  Index /= sizeof(NOTIFYENTRY);

  rc = DosRequestMutexSem( g_hSemNotifytable, SEM_INDEFINITE_WAIT);

  switch (g_Notifications[Index].usFlags)
  {
    case NOTIFY_FREE:
      DosReleaseMutexSem( g_hSemNotifytable);
      return ERROR_INVALID_PARAMETER;
    case NOTIFY_CHANGE:
      ulFunction = IOCTLF_DEREG_STATUSSEM;
      ulSize = sizeof(STATUSEVENTSET);
      EventSet.ulSize            = ulSize;
      EventSet.ulCaps            = DEV_SEM_ADD | DEV_SEM_REMOVE;
      EventSet.ulSemDeviceAdd    = g_Notifications[Index].hDeviceAdded;
      EventSet.ulSemDeviceRemove = g_Notifications[Index].hDeviceRemoved;
      break;
    case NOTIFY_DEVICE:
      ulFunction = IOCTLF_DEREG_DEVICESEM;
      ulSize = sizeof(DEVEVENTSET);
      EventSet.ulSize            = ulSize;
      EventSet.ulCaps            = DEV_SEM_ADD | DEV_SEM_REMOVE |
                                   DEV_SEM_VENDORID | DEV_SEM_PRODUCTID |
                                   DEV_SEM_BCDDEVICE ;
      EventSet.ulSemDeviceAdd    = g_Notifications[Index].hDeviceAdded;
      EventSet.ulSemDeviceRemove = g_Notifications[Index].hDeviceRemoved;
      EventSet.usVendorID        = g_Notifications[Index].usVendor;
      EventSet.usProductID       = g_Notifications[Index].usProduct;
      EventSet.usBCDDevice       = g_Notifications[Index].usBCDDevice;
      EventSet.usReserved        = 0;
      break;
    default:
      DosReleaseMutexSem( g_hSemNotifytable);
      return ERROR_GEN_FAILURE;
  }

  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, ulFunction,
                    NULL, 0, NULL,
                    &EventSet,ulSize, &ulSize);
  if (rc == 0)
    memset( &g_Notifications[Index], 0, sizeof(g_Notifications[Index]));

  DosReleaseMutexSem(g_hSemNotifytable);

  return rc;
}

#endif // USB_NO_NOTIFY

/***************************************************************************/
// Interrupt Handling
/***************************************************************************/

#ifndef USB_NO_INTERRUPT

APIRET APIENTRY UsbIrqStart( USBHANDLE Handle,
                             UCHAR  Endpoint,
                             UCHAR  Interface,
                             USHORT ulNumBytes,
                             UCHAR  *pData,
                             PHEV   pHevModified)
{
  APIRET    rc;
  HEV       hEvent;
  ULONG     ulParmLen;
  ULONG     ulDataLen;
  USBCALLS_IRQ_START IrqStart;

  if (!g_fInit)
    return USB_NOT_INIT;

  if (ulNumBytes == 0 || IsBadWritePointer( pData, ulNumBytes))
    return ERROR_INVALID_PARAMETER;

  rc = DosCreateEventSem( NULL,
                          &hEvent,
                          DC_SEM_SHARED,
                          FALSE);
  if (rc)
    return rc;

  IrqStart.ulDevHandle = Handle;
  IrqStart.ulIrqEvent  = hEvent;
  IrqStart.ucEndpoint  = Endpoint;
  IrqStart.ucInterface = Interface;
  ulParmLen = sizeof(IrqStart);
  ulDataLen = ulNumBytes;

  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_START_IRQ_PROC,
                    (PVOID)&IrqStart, ulParmLen, &ulParmLen,
                    pData, ulDataLen, &ulDataLen);
  if (rc)
    DosCloseEventSem(hEvent);
  else
    *pHevModified = hEvent;

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbIrqStop( USBHANDLE Handle, HEV HevModified)
{
  APIRET    rc;
  ULONG     ulParmLen;
  ULONG     ulDataLen;

  if (!g_fInit)
    return USB_NOT_INIT;

  ulParmLen = sizeof(Handle);
  ulDataLen = sizeof(HevModified);
  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_STOP_IRQ_PROC,
                    (PVOID)&Handle, ulParmLen, &ulParmLen,
                    &HevModified, ulDataLen, &ulDataLen);
  if (!rc)
    DosCloseEventSem(HevModified);

  return rc;
}

#endif // USB_NO_INTERRUPT

/***************************************************************************/
// ISO Handling
/***************************************************************************/

#ifndef USB_NO_ISO

static APIRET UsbIsoInit( void)
{
  APIRET    rc;
  int       i;

  rc = DosCreateMutexSem( NULL, &g_hSemRingBuffers, DC_SEM_SHARED, FALSE);
  if (rc)
    return rc;

  g_ulNumIsoRingBuffers = 8;
  rc = DosAllocMem( (PPVOID)&g_pIsoRingBuffers,
                    g_ulNumIsoRingBuffers * sizeof(ISORINGBUFFER),
                    PAG_WRITE | PAG_COMMIT | OBJ_TILE);
  if (rc)
  {
    DosCloseMutexSem(g_hSemRingBuffers);
    g_hSemRingBuffers = 0;
    g_ulNumIsoRingBuffers = 0;
  }
  else
    for (i = 0; i < g_ulNumIsoRingBuffers; i++)
      g_pIsoRingBuffers[i].usBufSize = 16*1023;

  return rc;
}

/***************************************************************************/

static void UsbIsoTerm( void)
{

  if (g_pIsoRingBuffers)
    DosFreeMem( g_pIsoRingBuffers);

  if (g_hSemRingBuffers)
    DosCloseMutexSem( g_hSemRingBuffers);

  g_pIsoRingBuffers = 0;
  g_hSemRingBuffers = 0;

  return;
}

/***************************************************************************/

APIRET APIENTRY UsbIsoStart( USBHANDLE Handle,
                             UCHAR  Endpoint,
                             UCHAR  Interface,
                             ISOHANDLE *phIso)
{
  APIRET    rc;
  ULONG     ulParmLen;
  ULONG     ulDataLen;
  ULONG     i;
  PISORINGBUFFER     pIter = g_pIsoRingBuffers;
  USBCALLS_ISO_START IsoStart;

  if (!g_fInit)
    return USB_NOT_INIT;

  rc = DosRequestMutexSem( g_hSemRingBuffers, SEM_INDEFINITE_WAIT);
  if (rc)
    return rc;

  for (i = 0; i < g_ulNumIsoRingBuffers; i++, pIter++)
    if (pIter->hDevice == 0)
    {
      pIter->hDevice = Handle;
      break;
    }

  DosReleaseMutexSem( g_hSemRingBuffers);

  if (i == g_ulNumIsoRingBuffers)
    return USB_ERROR_OUTOF_RESOURCES;

  IsoStart.ulDevHandle = Handle;
  IsoStart.ucEndpoint  = Endpoint;
  IsoStart.ucInterface = Interface;
  ulParmLen = sizeof(IsoStart);
  ulDataLen = sizeof(ISORINGBUFFER);

  rc = DosDevIOCtl( g_hUSBDrv,
                    IOCAT_USBRES, IOCTLF_STOP_IRQ_PROC,
                    (PVOID)&IsoStart, ulParmLen, &ulParmLen,
                    pIter, ulDataLen, &ulDataLen);
  if (rc)
  {
    pIter->hDevice = 0;
    *phIso = 0;
  }
  else
  {
    pIter->ucEndpoint  = Endpoint;
    pIter->ucInterface = Interface;
  }

  return rc;
}

/***************************************************************************/

APIRET IsInvalidIsoHandle( const ISOHANDLE hIso)
{
  ULONG i;
  PISORINGBUFFER pIter = g_pIsoRingBuffers;

  for (i = 0; i < g_ulNumIsoRingBuffers; i++, pIter++)
    if (pIter == (PISORINGBUFFER)hIso && pIter->hDevice)
      return 0;

  return ERROR_INVALID_PARAMETER;
}

/***************************************************************************/

// does this _do_ anything? <RW>

APIRET APIENTRY UsbIsoStop( ISOHANDLE hIso)
{
  APIRET rc = 0;

  if (!g_fInit)
    return USB_NOT_INIT;

//  rc = DosDevIOCtl( g_hUSBDrv,

  return rc;
}

/***************************************************************************/

// does this _do_ anything? <RW>

APIRET APIENTRY UsbIsoDequeue( ISOHANDLE hIso,
                               UCHAR * pBuffer,
                               ULONG ulNumBytes)
{
  APIRET rc;
  PISORINGBUFFER pRB = (PISORINGBUFFER)hIso;

  rc = IsInvalidIsoHandle( hIso);
  if (rc == 0)
    if (!(pRB->ucEndpoint & ISO_DIRMASK))
      rc = ERROR_INVALID_PARAMETER;

  return rc;
}

/***************************************************************************/

// does this _do_ anything? <RW>

APIRET APIENTRY UsbIsoPeekQueue( ISOHANDLE hIso,
                                 UCHAR * pByte,
                                 ULONG ulOffset)
{
  APIRET rc;
  PISORINGBUFFER pRB = (PISORINGBUFFER)hIso;

  rc = IsInvalidIsoHandle(hIso);
  if (rc == 0)
    if (!(pRB->ucEndpoint & ISO_DIRMASK))
      rc = ERROR_INVALID_PARAMETER;

  return rc;
}

/***************************************************************************/

// does this _do_ anything? <RW>

APIRET APIENTRY UsbIsoEnqueue( ISOHANDLE hIso,
                               const UCHAR * pBuffer,
                               ULONG ulNumBytes)
{
  APIRET rc;
  PISORINGBUFFER pRB = (PISORINGBUFFER)hIso;

  rc = IsInvalidIsoHandle( hIso);
  if (rc == 0)
    if (pRB->ucEndpoint & ISO_DIRMASK)
      rc = ERROR_INVALID_PARAMETER;

  return rc;
}

/***************************************************************************/

APIRET APIENTRY UsbIsoGetLength( ISOHANDLE hIso,
                                 ULONG *pulLength)
{
  APIRET rc;
  USHORT ri;
  USHORT wi;
  PISORINGBUFFER pRB = (PISORINGBUFFER) hIso;

  rc = IsInvalidIsoHandle( hIso);
  if (rc)
    return rc;

  wi = pRB->usPosWrite;
  ri = pRB->usPosRead;

  if (ri == wi)
    *pulLength = 0;
  else
    if (ri < wi)
      *pulLength = wi - ri;
    else
      *pulLength = wi + (pRB->usBufSize - ri);

  return 0;
}

#endif // USB_NO_ISO

/***************************************************************************/
// DLL-only code
/***************************************************************************/

#ifndef USB_BUILD_STATIC

ULONG _System _DLL_InitTerm( ULONG modhandle, ULONG flag)
{
  ULONG ulRtn = 0;

  if (flag == 0)
  {
    if (UsbDriverInitInternal() == 0)
      ulRtn = 1;
  }
  else
  if (flag == 1)
  {
    if (UsbDriverTermInternal() == 0)
      ulRtn = 1;
  }

  return ulRtn;
}

#endif // ifndef USB_BUILD_STATIC

/***************************************************************************/
// not used
/***************************************************************************/

#if 0

typedef struct
{
  UCHAR  bRequestType;
  UCHAR  bRequest;
  USHORT wValue;
  USHORT wIndex;
  USHORT wLength;
  ULONG  ulTimeout; /* in milliseconds */
}SETUPPACKET, *PSETUPPACKET;

BOOL IsBadReadPointer( PVOID pBase, ULONG ulSize);

/***************************************************************************/

static BOOL IsBadReadPointer( PVOID pBase, ULONG ulSize)
{
  APIRET    rc;
  ULONG     ulFlags;

  rc = DosQueryMem(pBase, &ulSize, &ulFlags);

  return (rc != 0 ||
          (ulFlags & (PAG_READ | PAG_COMMIT)) != (PAG_READ | PAG_COMMIT));
}

#endif // not used

/***************************************************************************/

#ifdef __cplusplus
}  // extern "C"
#endif

/***************************************************************************/

