/***************************************************************************/
/* ptpro.c
 *
 * Copyright (C) 2001-2005 Mariusz Woloszyn <emsi@ipartners.pl>
 *
 *  This file is part of ptpro.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */
/***************************************************************************/
/*
 *  PTPro for OS/2 is based on ptpcam, part of the libptp2 package.
 *  Because it has been extensively rewritten and restructured, its
 *  name has been changed to avoid confusion with the original.
 *  All changes and new code are
 *      Copyright (C) 2006 Richard L Walsh <rich@e-vertise.com>
 */
/***************************************************************************/

#include "ptp.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "usb.h"
#include "ptpro.h"

/***************************************************************************/

#pragma pack(1)

/***************************************************************************/

short   verbose = 0;
int     ptpro_usb_timeout = USB_TIMEOUT;
char    underline_buf[64];

/***************************************************************************/
/***************************************************************************/
/*
    top-level functions
        main & help
        show info about the camera(s)
        get / set camera properties
        show file info
        get / delete files
        reset camera / clear stalls

    internal functions
        init / close camera & usb structures
        custom functions for libptp

*/
/***************************************************************************/
/***************************************************************************/
/*
    main & help

    main
        parse_options
        range_from_arg
    usage
    help

*/
/***************************************************************************/

int main( int argc, char ** argv)
{
    int     action = 0;
    OPTS    opts;

    action = parse_options( &opts, argc, argv);
    if (!action)
        return 0;

do {
    PTPParams   params;
    PTP_USB     ptp_usb;

    usb_set_debug( verbose);

    if (init_usb( &ptp_usb, &params) < 0)
        break;

    if (action == ACT_LIST_DEVICES) {
        list_devices( &ptp_usb, &params, &opts);
        break;
    }

    ptp_usb.dev = find_device( opts.busn, opts.devn, (opts.flags & OPT_FORCE));
    if (ptp_usb.dev == NULL)
        break;

    if (action == ACT_DEVICE_RESET) {
        reset_device( &ptp_usb, &params);
        break;
    }

    if (open_camera( &ptp_usb, &params) < 0)
        break;

    switch (action) {
        case ACT_LIST_FILES:
            list_files( &params);
            break;
        case ACT_HANDLE_INFO:
            show_handle_info( &params, &opts);
            break;
        case ACT_GET_FILE:
            get_files( &params, &opts);
            break;
        case ACT_DELETE_FILE:
            delete_files( &params, &opts);
            break;
        case ACT_SHOW_INFO:
            show_info( &params);
            break;
        case ACT_STORAGE_INFO:
            show_storage_info( &params);
            break;
        case ACT_LIST_OPERATIONS:
            list_operations( &params);
            break;
        case ACT_LIST_PROPERTIES:
            list_properties( &params);
            break;
        case ACT_GETSET_PROPERTY:
            getset_property( &params, &opts);
            break;
    }

    close_camera( &ptp_usb, &params);

} while (0);

    usb_term();

    return 0;
}

/***************************************************************************/

int parse_options( OPTS * pOpts, int argc, char ** argv)

{
    int ret = 0;
    int action = 0;
    int option_index = 0;
    int opt = 0;
    int fOK = 1;

    static struct option loptions[] = {
        {"list-files",0,0,'l'},
        {"file-info",1,0,'i'},
        {"save-file",1,0,'f'},
        {"save-thumb",1,0,'t'},
        {"erase-file",1,0,'e'},
        {"replace",0,0,'r'},
        {"device-list",0,0,'d'},
        {"camera-info",0,0,'c'},
        {"dev",1,0,0},
        {"bus",1,0,0},
        {"storage-info",0,0,'s'},
        {"operation-list",0,0,'o'},
        {"property-list",0,0,'p'},
        {"getset-property",1,0,'g'},
        {"val",1,0,0},
        {"reset",0,0,'R'},
        {"force",0,0,'F'},
        {"verbose",2,0,'v'},
        {"help",0,0,'h'},
        {0,0,0,0}
    };

    if (argc == 1) {
        usage();
        return 0;
    }

    memset( pOpts, 0, sizeof(OPTS));
    opterr = 0;

    while (fOK) {
        opt = getopt_long( argc, argv, "li:f:t:e:rdcsopg:RFv::h",
                           loptions, &option_index);

        switch (opt) {

        // special cases
        case -1:
            if (optind+1 == argc && ret == ACT_GET_FILE && !pOpts->value)
                pOpts->value = strdup( argv[optind]);
            fOK = 0;
            break;
        case '?':
            ptpro_error( "Option '-%c' is unknown or missing its arguments", optopt);
            ret = fOK = 0;
            break;
        case 'h':
            help();
            ret = fOK = 0;
            break;

        // modifiers
        case 0:
            if (!(strcmp( "val", loptions[option_index].name)))
                pOpts->value = strdup( optarg);
            else
            if (!(strcmp( "bus", loptions[option_index].name)))
                pOpts->busn = strtol( optarg, NULL, 10);
            else
            if (!(strcmp( "dev", loptions[option_index].name)))
                pOpts->devn = strtol( optarg, NULL, 10);
            break;
        case 'r':
            pOpts->flags |= OPT_REPLACE;
            break;
        case 'F':
            pOpts->flags |= OPT_FORCE;
            break;
        case 'v':
            if (optarg) 
                verbose = strtol( optarg, NULL, 10);
            else
                verbose = 1;
            break;

        // actions
        case 'l':
            action = ACT_LIST_FILES;
            break;
        case 'i':
            if (range_from_arg( optarg, &pOpts->first, &pOpts->last))
                action = ACT_HANDLE_INFO;
            else
                ret = fOK = 0;
            break;
        case 'f':
            if (range_from_arg( optarg, &pOpts->first, &pOpts->last))
                action = ACT_GET_FILE;
            else
                ret = fOK = 0;
            break;
        case 't':
            if (range_from_arg( optarg, &pOpts->first, &pOpts->last)) {
                action = ACT_GET_FILE;
                pOpts->flags |= OPT_THUMBNAIL;
            }
            else
                ret = fOK = 0;
            break;
        case 'e':
            if (range_from_arg( optarg, &pOpts->first, &pOpts->last))
                action = ACT_DELETE_FILE;
            else
                ret = fOK = 0;
            break;
        case 'c':
            action = ACT_SHOW_INFO;
            break;
        case 'd':
            action = ACT_LIST_DEVICES;
            break;
        case 's':
            action = ACT_STORAGE_INFO;
            break;
        case 'o':
            action = ACT_LIST_OPERATIONS;
            break;
        case 'p':
            action = ACT_LIST_PROPERTIES;
            break;
        case 'g':
            action = ACT_GETSET_PROPERTY;
            pOpts->property = strtol( optarg, NULL, 16);
            break;
        case 'R':
            action = ACT_DEVICE_RESET;
            break;
        default:
            ptpro_error( "Getopt returned character code 0x%X", opt);
            break;
        } // end switch

        // prevent multiple commands
        if (action) {
            if (ret) {
                ptpro_error( "Option '-%c' conflicts with previous option", opt);
                ret = fOK = 0;
            }
            else {
                ret = action;
                action = 0;
            }
        }
    } // end while

    return ret;
}

/***************************************************************************/

int range_from_arg( char* arg, uint32_t* first, uint32_t* last)
{
    int     rtn = 0;
    char *  ptr;

do {
    *first = strtoul( arg, &ptr, 10);

    if (*first == 0) {
        if (arg[0] == '*')
            ptr = &arg[1];
        else
            if (arg[0] != '0')
                break;
    }

    if (*ptr == 0) {
        if (arg[0] == '*')
            *last = ~0;
        else
            *last = *first;
        rtn = 1;
        break;
    }

    if (*ptr != '-')
        break;

    ptr++;
    if (ptr[0] == '*' && ptr[1] == 0)
        *last = ~0;
    else
        *last = strtoul( ptr, 0, 10);

    if (*last > 0 && *last >= *first)
        rtn = 1;

} while (0);

    if (!rtn)
        ptpro_error( "Invalid or malformed range of handles: '%s'", arg);

    return rtn;
}

/***************************************************************************/

void usage()
{
    printf( "USAGE: ptpro [Options][Filename] (use '-h' to list options)\n\n");
}

/***************************************************************************/

void help()
{
    printf( "USAGE: ptpro [Options][Filename]\n");
    printf( "Options:\n"
    "  -l, --list-files             List all files & their handles\n"
    "  -i, --file-info=Handles      Show extended file info\n"
    "  -f, --save-file=Handles      Save file(s)\n"
    "  -t, --save-thumb=Handles     Save thumbnail(s)\n"
    "  -e, --erase-file=Handles     Erase file(s) from camera\n"
    "  -r  --replace                Replace existing file when saving to disk\n"
    "  -d, --device-list            List all USB PTP devices\n"
    "  -c, --camera-info            Show camera info\n"
    "      --dev=Device-Number      Specify which USB PTP device to use\n"
    "  -s, --storage-info           Show storage (memory) info\n"
    "  -o, --operation-list         List supported operations\n"
    "  -p, --property-list          List all PTP device properties\n"
    "  -g, --getset-property=Number Get or set property value\n"
    "      --val=Value              New property value (use with -g to set value)\n"
    "  -R, --reset                  Reset the device\n"
    "  -F, --force                  Talk to non PTP devices\n"
    "  -v, --verbose                Show error details (use -v2 for debug info)\n"
    "  -h, --help                   Print this help message\n"
    "Handles:  [First-Last] or [First] or [*] (all files)\n"
    "Filename: [Path\\][Name][.Ext] - '*' copies the original Name or Ext\n"
    "          '#' inserts a sequence number in Name (e.g. cats#0123.*)\n");
}

/***************************************************************************/

// a less-than-compelling routine that creates an underline
// the length of the camera model name

char *  underline( const char *str)
{
    int i;

    i = strlen( str);
    if (i >= sizeof(underline_buf))
        i = sizeof(underline_buf - 1);
    memset( underline_buf, '=', i);
    underline_buf[i] = 0;

    return underline_buf;
}

/***************************************************************************/
/***************************************************************************/
/*
    show info about the camera(s)

    list_devices
    showinfo
    list_operations
    show_storage_info

*/
/***************************************************************************/

void list_devices( PTP_USB *ptp_usb, PTPParams *params, OPTS *pOpts)
{
    int               force = (pOpts->flags & OPT_FORCE);
    int               found = 0;
    struct usb_bus    *bus;
    struct usb_device *dev;

    bus = usb_get_busses();

    for (; bus; bus = bus->next)
        for (dev = bus->devices; dev; dev = dev->next) {

            if ((dev->config->interface->altsetting->bInterfaceClass ==
                 USB_CLASS_PTP) || force)
                if (dev->descriptor.bDeviceClass != USB_CLASS_HUB)
                {
                    ptp_usb->dev = dev;
                    if (open_camera( ptp_usb, params) < 0)
                        continue;

                    if (!found){
                        printf( "\n Device list\n ===========\n");
                        printf( " Bus  Device  VendorID  ProductID  Device\n");
                        found = 1;
                    }

                    printf( " %2s     %2s     0x%04X    0x%04X    %s\n",
                            bus->dirname, dev->filename, dev->descriptor.idVendor,
                            dev->descriptor.idProduct, params->deviceinfo.Model);

                    close_camera( ptp_usb, params);
                    memset( ptp_usb, 0, sizeof(PTP_USB));
                }
        }

    if (!found)
        printf("\nFound no PTP devices\n");

    return;
}

/***************************************************************************/

void show_info( PTPParams *params)
{

    printf("\n Camera information  %s\n",   params->deviceinfo.Model);
    printf(" ==================  %s\n",     underline(params->deviceinfo.Model));
    printf("      Manufacturer:  %s\n",     params->deviceinfo.Manufacturer);
    printf("      SerialNumber:  %s\n",     params->deviceinfo.SerialNumber);
    printf("     DeviceVersion:  %s\n",     params->deviceinfo.DeviceVersion);
    printf("       ExtensionID:  0x%08X\n", params->deviceinfo.VendorExtensionID);
    printf("  ExtensionVersion:  0x%04X\n", params->deviceinfo.VendorExtensionVersion);
    printf("    ExtDescription:  %s\n",     params->deviceinfo.VendorExtensionDesc);

    if (params->deviceinfo.CaptureFormats_len == 0)
        printf( "    CaptureFormats:  [unknown]\n");
    else {
        uint32_t ctr;
        uint16_t fmt;
        for (ctr = 0; ctr < params->deviceinfo.CaptureFormats_len; ctr++) {
            fmt = params->deviceinfo.CaptureFormats[ctr];
            printf( "%s0x%04X  %s\n",
                    (ctr ? "                     " : "    CaptureFormats:  "),
                    fmt, ptp_get_codename( fmt, PTP_OBJECT_FORMAT));
        }
    }

    if (params->deviceinfo.ImageFormats_len == 0)
        printf( "       FileFormats:  [unknown]\n");
    else {
        uint32_t ctr;
        uint16_t fmt;
        for (ctr = 0; ctr < params->deviceinfo.ImageFormats_len; ctr++) {
            fmt = params->deviceinfo.ImageFormats[ctr];
            printf( "%s0x%04X  %s\n",
                    (ctr ? "                     " : "       FileFormats:  "),
                    fmt, ptp_get_codename( fmt, PTP_OBJECT_FORMAT));
        }
    }

    return;
}

/***************************************************************************/

void list_operations( PTPParams *params)
{
    int             i;
    const char*     name;

    printf( "\n Operations  %s\n", params->deviceinfo.Model);
    printf( " ==========  %s\n", underline( params->deviceinfo.Model));
    for (i = 0; i < params->deviceinfo.OperationsSupported_len; i++) {
        name = ptp_get_operation_name( params,
                            params->deviceinfo.OperationsSupported[i]);
        printf( "    0x%04X:  %s\n",
                params->deviceinfo.OperationsSupported[i],
                (name ? name : "[unknown]"));
    }
}

/***************************************************************************/

void show_storage_info( PTPParams *params)
{
    PTPStorageIDs   sids;
    PTPStorageInfo  sin ;
    int             i;

do {
    memset( &sids, 0, sizeof(sids));

    if (ptp_getstorageids ( params, &sids) != PTP_RC_OK) {
        ptpro_error( "Could not get storage IDs.");
        break;
    }
    if (sids.n == 0) {
        ptpro_error( "No storage IDs returned.");
        break;
    }

    printf ( "\n Storage information  %s\n", params->deviceinfo.Model);
      printf ( " ===================  %s\n", underline(params->deviceinfo.Model));

    for (i = 0; i < sids.n; i++) {

        memset( &sin, 0, sizeof(sin));
        if (ptp_getstorageinfo ( params, (sids.Storage)[i], &sin) != PTP_RC_OK) {
            ptpro_error( "Could not get storage info for id %d", (sids.Storage)[i]);
            continue;
        }

        printf ( "          StorageID:  0x%04X\n", (sids.Storage)[i]);
        printf ( "        StorageType:  0x%04X\t%s\n", sin.StorageType,
                                ptp_get_codename( sin.StorageType, PTP_STORAGE));
        printf ( "     FilesystemType:  0x%04X\t%s\n", sin.FilesystemType,
                                ptp_get_codename( sin.FilesystemType, PTP_FILESYSTEM));
        printf ( "   AccessCapability:  0x%04X\t%s\n", sin.AccessCapability,
                                ptp_get_codename( sin.AccessCapability, PTP_STORAGE_ACCESS));
        printf ( "      MaxCapability:  %lld\n", sin.MaxCapability);
        printf ( "   FreeSpaceInBytes:  %lld\n", sin.FreeSpaceInBytes);
        printf ( "  FreeSpaceInImages:  0x%04X\n", sin.FreeSpaceInImages);
        printf ( " StorageDescription:  %s\n", (sin.StorageDescription ?
                                sin.StorageDescription : "[none]"));
        printf ( "        VolumeLabel:  %s\n\n", (sin.VolumeLabel ?
                                sin.VolumeLabel : "[none]"));

        if (sin.StorageDescription)
            free( sin.StorageDescription);
        if (sin.VolumeLabel)
            free( sin.VolumeLabel);
    }

    if (sids.Storage)
        free( sids.Storage);

} while (0);

    return;
}

/***************************************************************************/
/***************************************************************************/
/*
    get / set camera properties

    list_properties
    getset_property
        print_propval
        set_property

*/
/***************************************************************************/

void list_properties( PTPParams *params)
{
    const char*     name;
    int             i;

    printf( "\n Properties  %s\n", params->deviceinfo.Model);
    printf( " ==========  %s\n", underline( params->deviceinfo.Model));

    for (i = 0; i < params->deviceinfo.DevicePropertiesSupported_len; i++){
        name = ptp_get_property_name( params,
                        params->deviceinfo.DevicePropertiesSupported[i]);
        printf( "    0x%04X:  %s\n",
                params->deviceinfo.DevicePropertiesSupported[i],
                (name ? name : "[unknown]"));
    }
}

/***************************************************************************/

void getset_property( PTPParams *params, OPTS *pOpts)
{
    PTPDevicePropDesc dpd;

    if (!ptp_property_issupported( params, pOpts->property)) {
        ptpro_error( "The device does not support property 0x%04X", pOpts->property);
        return;
    }

    memset( &dpd, 0, sizeof(dpd));
    if (ptp_getdevicepropdesc( params, pOpts->property, &dpd) != PTP_RC_OK) {
        ptpro_error( "Could not get device property description for 0x%04X", pOpts->property);
        return;
    }

    printf( "\n Get/Set Property  %s\n", params->deviceinfo.Model);
    printf(   " ================  %s\n", underline( params->deviceinfo.Model));
    printf(   "        Property:  0x%04X\n", pOpts->property);
    printf(   "     Description:  %s\n", ptp_get_property_name( params, pOpts->property));
    printf(   "        DataType:  0x%04X\n", dpd.DataType);

    printf(   "   %sValue:  ", (pOpts->value ? "Previous" : " Current"));
    if (dpd.FormFlag == PTP_DPFF_Enumeration)
        PRINT_PROPVAL_DEC( dpd.CurrentValue);
    else 
        PRINT_PROPVAL_HEX( dpd.CurrentValue);
    printf("\n");

    if (pOpts->value) {
        uint16_t ret;
        ret = set_property( params, pOpts->property, pOpts->value, dpd.DataType);
        if (ret == PTP_RC_OK)
            printf( "        NewValue:  %s\n", pOpts->value);
        else
            ptpro_error( "Unable to set property value - rc= %X", ret);

        ptp_free_devicepropdesc( &dpd);
        return;
    }

    printf( "    DefaultValue:  ");
    if (dpd.FormFlag == PTP_DPFF_Enumeration)
        PRINT_PROPVAL_DEC( dpd.FactoryDefaultValue);
    else 
        PRINT_PROPVAL_HEX( dpd.FactoryDefaultValue);
    printf("\n");

    if (dpd.GetSet == PTP_DPGS_Get)
        printf( "      Read/Write:  read only\n");
    else {
        printf( "      Read/Write:  read & write\n");
        if (dpd.FormFlag == PTP_DPFF_Enumeration &&
            dpd.FORM.Enum.NumberOfValues) {
            int i = 0;
            printf( "     ValueFormat:  enumeration\n");
            printf( "   AllowedValues:  ");
            PRINT_PROPVAL_HEX( dpd.FORM.Enum.SupportedValue[i]);
            printf("\n");
            for( i++; i < dpd.FORM.Enum.NumberOfValues; i++){
                printf( "                   ");
                PRINT_PROPVAL_HEX( dpd.FORM.Enum.SupportedValue[i]);
                printf("\n");
            }
        }
        else
        if (dpd.FormFlag == PTP_DPFF_Range) {
            printf( "     ValueFormat:  range\n");
            printf( "   AllowedValues:  ");
            PRINT_PROPVAL_DEC( dpd.FORM.Range.MinimumValue);
            printf( " - ");
            PRINT_PROPVAL_DEC( dpd.FORM.Range.MaximumValue);
            printf( "; step size: ");
            PRINT_PROPVAL_DEC( dpd.FORM.Range.StepSize);
            printf( "\n");
        }
    }

    ptp_free_devicepropdesc( &dpd);

    return;
}

/***************************************************************************/

short print_propval( uint16_t datatype, void* value, short hex)
{
    short   ret = 0;

    switch (datatype) {
        case PTP_DTC_INT8:
            printf("%hhi",*(char*)value);
            break;
        case PTP_DTC_UINT8:
            printf("%hhu",*(unsigned char*)value);
            break;
        case PTP_DTC_INT16:
            printf("%hi",*(int16_t*)value);
            break;
        case PTP_DTC_UINT16:
            if (hex==PTPCAM_PRINT_HEX)
                printf("0x%04hX (%hi)",*(uint16_t*)value, *(uint16_t*)value);
            else
                printf("%hi",*(uint16_t*)value);
            break;
        case PTP_DTC_INT32:
            printf("%i",*(int32_t*)value);
            break;
        case PTP_DTC_UINT32:
            if (hex==PTPCAM_PRINT_HEX)
                printf("0x%08X (%i)",*(uint32_t*)value, *(uint32_t*)value);
            else
                printf("%i",*(uint32_t*)value);
            break;
        case PTP_DTC_STR:
            printf("\"%s\"",(char *)value);
            ret = -1;
            break;
        default:
            ret = -1;
            break;
    }

    return ret;
}

/***************************************************************************/

uint16_t set_property( PTPParams* params, uint16_t property,
                       char* value, uint16_t datatype)
{
    uint16_t    ret;
    void* val = NULL;

    switch(datatype) {
    case PTP_DTC_INT8:
        val=malloc(sizeof(int8_t));
        *(int8_t*)val=(int8_t)strtol(value,NULL,0);
        break;
    case PTP_DTC_UINT8:
        val=malloc(sizeof(uint8_t));
        *(uint8_t*)val=(uint8_t)strtol(value,NULL,0);
        break;
    case PTP_DTC_INT16:
        val=malloc(sizeof(int16_t));
        *(int16_t*)val=(int16_t)strtol(value,NULL,0);
        break;
    case PTP_DTC_UINT16:
        val=malloc(sizeof(uint16_t));
        *(uint16_t*)val=(uint16_t)strtol(value,NULL,0);
        break;
    case PTP_DTC_INT32:
        val=malloc(sizeof(int32_t));
        *(int32_t*)val=(int32_t)strtol(value,NULL,0);
        break;
    case PTP_DTC_UINT32:
        val=malloc(sizeof(uint32_t));
        *(uint32_t*)val=(uint32_t)strtol(value,NULL,0);
        break;
    case PTP_DTC_STR:
        val=(void *)value;
        break;
    }

    ret = ptp_setdevicepropvalue( params, property, val, datatype);
    if (val && datatype != PTP_DTC_STR)
        free(val);

    return ret;
}

/***************************************************************************/
/***************************************************************************/
/*
    show file info

    list_files
    show_handle_info
        validate_range

*/
/***************************************************************************/

void list_files( PTPParams *params)
{
    int             i;
    PTPObjectInfo   oi;
    struct tm      *ptm;

    if (ptp_getobjecthandles( params, 0xffffffff, 0, 0,
                              &params->handles) != PTP_RC_OK) {
        ptpro_error( "Could not get object handles.");
        return;
    }

    printf( "\n File list  %s\n", params->deviceinfo.Model);
    printf(   " =========  %s\n", underline( params->deviceinfo.Model));
    printf("\n Handle\t     Size     Date     Time   Name\n");

    for (i = 0; i < params->handles.n; i++) {
        if (ptp_getobjectinfo( params, params->handles.Handler[i],
                               &oi) != PTP_RC_OK) {
            ptpro_error( "Could not get object info for handle %d.",
                          params->handles.Handler[i]);
            break;
        }

        if (oi.ObjectFormat == PTP_OFC_Association)
            continue;

        ptm = localtime( &oi.CaptureDate);
        printf( " % 4d\t% 9d  %04d-%02d-%02d  %02d:%02d  %s\n",
                params->handles.Handler[i], oi.ObjectCompressedSize,
                ptm->tm_year+1900, ptm->tm_mon+1, ptm->tm_mday,
                ptm->tm_hour, ptm->tm_min, oi.Filename);
    }

    return;
}

/***************************************************************************/

void show_handle_info( PTPParams *params, OPTS *pOpts)
{
    uint32_t       *pHandle;
    uint32_t       *pEnd;
    struct tm       *ptm;
    PTPObjectInfo   oi;

    if (ptp_getobjecthandles( params, 0xffffffff, 0, 0,
                              &params->handles) != PTP_RC_OK) {
        ptpro_error( "Could not get object handles.");
        return;
    }

    if (!validate_range( params, pOpts, &pHandle, &pEnd))
        return;

    printf ( "\n     File information  %s\n", params->deviceinfo.Model);
    printf (   "     ================  %s\n", underline( params->deviceinfo.Model));

    for (; pHandle <= pEnd; pHandle++) {
        memset( &oi, 0, sizeof(oi));
        if (ptp_getobjectinfo( params, *pHandle, &oi) != PTP_RC_OK) {
            ptpro_error( "Could not get object info for handle %d", *pHandle);
            continue;
        }

        printf ( "              Handle:  %04d\n", *pHandle);
        printf ( "            Filename:  %s\n", (oi.Filename ? oi.Filename : "[none]"));

        if (oi.CaptureDate == 0)
            ptm = 0;
        else
            ptm = localtime( &oi.CaptureDate);
        if (!ptm)
            printf ( "         CaptureDate:  [unknown]\n");
        else
            printf ( "         CaptureDate:  %04d-%02d-%02d  %02d:%02d:%02d\n",
                     ptm->tm_year+1900, ptm->tm_mon+1, ptm->tm_mday,
                     ptm->tm_hour, ptm->tm_min, ptm->tm_sec);

        if (oi.ModificationDate == 0)
            ptm = 0;
        else
            ptm = localtime( &oi.ModificationDate);
        if (!ptm)
            printf ( "    ModificationDate:  [unknown]\n");
        else
            printf ( "    ModificationDate:  %04d-%02d-%02d  %02d:%02d:%02d\n",
                     ptm->tm_year+1900, ptm->tm_mon+1, ptm->tm_mday,
                     ptm->tm_hour, ptm->tm_min, ptm->tm_sec);

        printf ( "            Keywords:  %s\n", (oi.Keywords ? oi.Keywords : "[none]"));
        printf ( "    ProtectionStatus:  0x%04X\t%s\n", oi.ProtectionStatus, ptp_get_codename( oi.ProtectionStatus, PTP_PROTECTION));
        printf ( "ObjectCompressedSize:  %d\n", oi.ObjectCompressedSize);
        printf ( "        ObjectFormat:  0x%04X\t%s\n", oi.ObjectFormat, ptp_get_codename( oi.ObjectFormat, PTP_OBJECT_FORMAT));
        printf ( "       ImagePixWidth:  %d\n", oi.ImagePixWidth);
        printf ( "      ImagePixHeight:  %d\n", oi.ImagePixHeight);
        printf ( "       ImageBitDepth:  %d\n", oi.ImageBitDepth);
        printf ( " ThumbCompressedSize:  %d\n", oi.ThumbCompressedSize);
        printf ( "         ThumbFormat:  0x%04X\t%s\n", oi.ThumbFormat, ptp_get_codename( oi.ThumbFormat, PTP_OBJECT_FORMAT));
        printf ( "       ThumbPixWidth:  %d\n", oi.ThumbPixWidth);
        printf ( "      ThumbPixHeight:  %d\n", oi.ThumbPixHeight);
        printf ( "        ParentObject:  %04d\n", oi.ParentObject);
        printf ( "     AssociationType:  0x%04X\t%s\n", oi.AssociationType, ptp_get_codename( oi.AssociationType, PTP_ASSOCIATION));
        printf ( "     AssociationDesc:  0x%04X\n", oi.AssociationDesc);
        printf ( "      SequenceNumber:  0x%04X\n", oi.SequenceNumber);
        printf ( "           StorageID:  0x%04X\n\n", oi.StorageID);
    }

    return;
}

/***************************************************************************/

int validate_range( PTPParams *params, OPTS *pOpts,
                    uint32_t **ppFirst, uint32_t **ppLast)
{
    int fErr;

    *ppFirst = params->handles.Handler;
    *ppLast = &(*ppFirst)[params->handles.n - 1];

    // 0 means "start at the first handle"
    if (pOpts->first == 0)
        pOpts->first = **ppFirst;

    // -1 means "end at the last handle"
    if (pOpts->last == ~0)
        pOpts->last = **ppLast;

    // are we completely out of range?
    if (pOpts->first > **ppLast || pOpts->last < **ppFirst) {
        ptpro_error( "Invalid range specified - must be from %d thru %d",
                      **ppFirst, **ppLast);
        return 0;
    }

    // are we partially out of range?
    fErr = 0;
    if (pOpts->first < **ppFirst) {
        pOpts->first = **ppFirst;
        fErr = 1;
    }
    if (pOpts->last > **ppLast) {
        pOpts->last = **ppLast;
        fErr = 1;
    }
    if (fErr)
        ptpro_error( "Invalid range specified - changing it to %d thru %d",
                      pOpts->first, pOpts->last);

    // adjust pointers to beginning & end of range
    while (**ppFirst < pOpts->first)
        (*ppFirst)++;
    while (**ppLast > pOpts->last)
        (*ppLast)--;

    return 1;
}

/***************************************************************************/
/***************************************************************************/
/*
    get / delete files

    delete_files
    get_files
        parse_filename
        make_filename

*/
/***************************************************************************/

void delete_files( PTPParams *params, OPTS *pOpts)
{
    int             ret;
    uint32_t       *pHandle;
    uint32_t       *pEnd;
    PTPObjectInfo   oi;

    if (ptp_getobjecthandles( params, 0xffffffff, 0, 0,
                              &params->handles) != PTP_RC_OK) {
        ptpro_error( "Could not get object handles.");
        return;
    }

    if (!validate_range( params, pOpts, &pHandle, &pEnd))
        return;

    printf ( "\n Erase files  %s\n", params->deviceinfo.Model);
    printf (   " ===========  %s\n", underline( params->deviceinfo.Model));

    for (; pHandle <= pEnd; pHandle++) {
        memset( &oi, 0, sizeof(PTPObjectInfo));
        ret = ptp_getobjectinfo( params, *pHandle, &oi);
        if (ret != PTP_RC_OK) {
            ptpro_error( "Could not get info for handle %d", *pHandle);
            if (ret == PTP_ERROR_IO)
                clear_stall( (PTP_USB*)params->data);
            continue;
        }

        if (oi.ObjectFormat == PTP_OFC_Association) {
            printf( " Handle %d is not a file - skipping...\n", *pHandle);
            continue;
        }

        ret = ptp_deleteobject( params, *pHandle, 0);
        if (ret == PTP_RC_OK)
            printf(" Erasing %3d  (%s)... done\n", *pHandle, oi.Filename);
        else {
            ptpro_error( "Could not delete handle %d - rc= %X",
                          *pHandle, ret);
            if (ret == PTP_ERROR_IO)
                clear_stall( (PTP_USB*)params->data);
        }
    }

    return;
}

/***************************************************************************/

void get_files( PTPParams *params, OPTS *pOpts)
{
    int             ret;
    int             fName;
    char           *pName;
    uint32_t       *pHandle;
    uint32_t       *pEnd;
    PTPObjectInfo   oi;
    FNSTRUCT        fns;
    char            path[260];

    if (ptp_getobjecthandles( params, 0xffffffff, 0, 0,
                              &params->handles) != PTP_RC_OK) {
        ptpro_error( "Could not get object handles.");
        return;
    }

    fName = parse_filename( pOpts->value, &fns);
    if (fName)
        pName = path;
    else
        pName = 0;

    if (!validate_range( params, pOpts, &pHandle, &pEnd))
        return;

    if (pOpts->flags & OPT_THUMBNAIL) {
        printf ( "\n Save thumbnails  %s\n", params->deviceinfo.Model);
        printf (   " ===============  %s\n", underline( params->deviceinfo.Model));
    }
    else {
        printf ( "\n Save files  %s\n", params->deviceinfo.Model);
        printf (   " ==========  %s\n", underline( params->deviceinfo.Model));
    }

    for (; pHandle <= pEnd; pHandle++) {
        memset( &oi, 0, sizeof(PTPObjectInfo));
        ret = ptp_getobjectinfo( params, *pHandle, &oi);
        if (ret != PTP_RC_OK) {
            ptpro_error( "Could not get info for handle %d", *pHandle);
            if (ret == PTP_ERROR_IO)
                clear_stall( (PTP_USB*)params->data);
            continue;
        }

        if (oi.ObjectFormat == PTP_OFC_Association) {
            printf( " Handle %d is not a file - skipping...\n", *pHandle);
            continue;
        }

        if (fName)
            make_filename( pName, oi.Filename, &fns);
        ret = os2_get_file( params, &oi, *pHandle, pName,
                           (pOpts->flags & OPT_REPLACE),
                           (pOpts->flags & OPT_THUMBNAIL));

        if (ret == PTP_ERROR_IO)
            clear_stall( (PTP_USB*)params->data);
    }

    return;
}

/***************************************************************************/

int parse_filename( char *filename, FNSTRUCT *fns)
{
    char *  pBksl;
    char *  pDot;
    char *  ptr;
    char *  pOut;
    uint    cnt;
    uint    posCtr;

    if (!filename || (filename[0] == '*' && filename[1] == 0) ||
        strcmp( filename, "*.*") == 0)
        return 0;

do {
    memset( fns, 0, sizeof(FNSTRUCT));

    pBksl = strrchr( filename, '\\');
    if (!pBksl)
        pBksl = filename;
    else
        pBksl++;

    pDot = strrchr( pBksl, '.');
    if (pDot) {
        if (pDot == pBksl)
            pDot = 0;
        else
            *pDot++ = 0;
    }

    pOut = fns->format;

    cnt = pBksl - filename;
    if (cnt) {
        memcpy( pOut, filename, cnt);
        pOut += cnt;
    }

    posCtr = 0;
    ptr = pBksl;
    while (*ptr) {

        if (*ptr == '*') {
            ptr++;
            if (fns->namePos == 0) {
                fns->namePos = ++posCtr;
                *pOut++ = '%';
                *pOut++ = 's';
            }
            continue;
        }

        if (*ptr == '#' && fns->nbrPos == 0) {
            ptr++;
            fns->nbrPos = ++posCtr;

            fns->seqNbr = atol( ptr);
            if (fns->seqNbr == 0)
                fns->seqNbr = 1;

            cnt = 0;
            while (*ptr >= '0' && *ptr <= '9' ) {
                cnt++;
                ptr++;
            }
            if (cnt == 0)
                cnt++;
            pOut += sprintf( pOut, "%%0%dd", cnt);
            continue;
        }

        *pOut++ = *ptr++;
    }

    if (ptr == pBksl) {
        strcpy( pOut, "%s.%s");
        fns->namePos = ++posCtr;
        fns->extPos = ++posCtr;
        break;
    }

    if (!pDot || *pDot == 0) {
        fns->extPos = ++posCtr;
        strcpy( pOut, ".%s");
        break;
    }

    *pOut++ = '.';
    while (*pDot) {
        if (*pDot == '*') {
            pDot++;
            if (fns->extPos == 0) {
                fns->extPos = ++posCtr;
                *pOut++ = '%';
                *pOut++ = 's';
            }
        }
        else
            *pOut++ = *pDot++;
    }
    *pOut = 0;

} while (0);

    return 1;
}

/***************************************************************************/

void make_filename( char* pOut, char* pIn, FNSTRUCT *fns)
{
    char *  pExt;
    char *  x1 = "";
    char *  x2 = "";
    char *  x3 = "";

    pExt = strrchr( pIn, '.');
    if (pExt)
        *pExt = 0;

    if (fns->namePos == 1)
        x1 = pIn;
    else
    if (fns->namePos == 2)
        x2 = pIn;

    if (fns->nbrPos == 1)
        x1 = (char*)fns->seqNbr++;
    else
    if (fns->nbrPos == 2)
        x2 = (char*)fns->seqNbr++;

    if (pExt) {
        if (fns->extPos == 1)
            x1 = pExt+1;
        else
        if (fns->extPos == 2)
            x2 = pExt+1;
        else
        if (fns->extPos == 3)
            x3 = pExt+1;
    }

    sprintf( pOut, fns->format, x1, x2, x3);

    if (pExt)
        *pExt = '.';

    return;
}

/***************************************************************************/
/*
    reset camera / clear stalls

    reset_device
        usb_ptp_get_device_status
        usb_ptp_device_reset
    clear_stall
        usb_get_endpoint_status
        usb_clear_stall_feature
*/
/***************************************************************************/

void reset_device( PTP_USB *ptp_usb, PTPParams *params)
{
    uint16_t        devstatus[2] = {0,0};
    int             ret;

    if (init_ptp_usb( params, ptp_usb) < 0)
        return;
    
    // get device status (devices like that regardless of its result)
    ret = usb_ptp_get_device_status( ptp_usb, devstatus);
    ptpro_debug( "first get_device_status returned %x", ret);

    // check endpoint statuses & clear any stalls
    clear_stall( ptp_usb);

    // get device status (now there should be some results)
    ret = usb_ptp_get_device_status( ptp_usb, devstatus);
    if (ret < 0) 
        ptpro_error( "Unable to get device status");
    else    {
        if (devstatus[1] == PTP_RC_OK) 
            printf( "Device status OK\n");
        else
            ptpro_error( "Device status 0x%04X", devstatus[1]);
    }
    
    // finally reset the device (that clears previously opened sessions)
    ret = usb_ptp_device_reset( ptp_usb);
    if (ret < 0)
        ptpro_error( "Unable to reset device");

    // get device status (devices likes that regardless of its result)
    ret = usb_ptp_get_device_status( ptp_usb, devstatus);
    ptpro_debug( "second get_device_status returned %x", ret);

    close_usb( ptp_usb);
}

/***************************************************************************/

int usb_ptp_get_device_status( PTP_USB* ptp_usb, uint16_t* devstatus)
{
    return (usb_control_msg( ptp_usb->handle,
                             USB_DP_DTH|USB_TYPE_CLASS|USB_RECIP_INTERFACE,
                             USB_REQ_GET_DEVICE_STATUS, 0, 0,
                             (char *)devstatus, 4, USB_TIMEOUT));
}

/***************************************************************************/

int usb_ptp_device_reset( PTP_USB* ptp_usb)
{
    return (usb_control_msg( ptp_usb->handle,
                             USB_TYPE_CLASS|USB_RECIP_INTERFACE,
                             USB_REQ_DEVICE_RESET, 0, 0,
                             NULL, 0, USB_TIMEOUT));
}

/***************************************************************************/

void clear_stall( PTP_USB* ptp_usb)
{
    uint16_t status;
    int      ret;

    // check the in endpoint status
    if (ptp_usb->inep) {
        status = 0;
        ret = usb_get_endpoint_status( ptp_usb, ptp_usb->inep, &status);
        if (ret >= 0 && status)
            ret = usb_clear_stall_feature( ptp_usb, ptp_usb->inep);
        if (ret < 0)
            ptpro_debug( "Unable to reset input pipe.");
    }
    
    // check the out endpoint status
    if (ptp_usb->outep) {
        status = 0;
        ret = usb_get_endpoint_status( ptp_usb, ptp_usb->outep, &status);
        if (ret >= 0 && status)
            ret = usb_clear_stall_feature( ptp_usb, ptp_usb->outep);
        if (ret < 0)
            ptpro_debug( "Unable to reset output pipe.");
    }
    
    // check the interrupt endpoint status
    if (ptp_usb->intep) {
        status = 0;
        ret = usb_get_endpoint_status( ptp_usb, ptp_usb->intep, &status);
        if (ret >= 0 && status)
            ret = usb_clear_stall_feature( ptp_usb, ptp_usb->intep);
        if (ret < 0)
            ptpro_debug( "Unable to reset interrupt pipe.");
    }
}

/***************************************************************************/

int usb_get_endpoint_status( PTP_USB* ptp_usb, int ep, uint16_t* status)
{
     return (usb_control_msg( ptp_usb->handle, USB_DP_DTH|USB_RECIP_ENDPOINT,
                              0, 0, ep,
                              (char *)status, 2, USB_TIMEOUT));
}

/***************************************************************************/

int usb_clear_stall_feature( PTP_USB* ptp_usb, int ep)
{
    return (usb_control_msg( ptp_usb->handle, USB_RECIP_ENDPOINT,
                             USB_REQ_CLEAR_FEATURE, USB_FEATURE_HALT, ep,
                             NULL, 0, USB_TIMEOUT));
}

/***************************************************************************/
/***************************************************************************/
/*
    init / close camera & usb structures

    init_usb
    find_device
    open_camera
    init_ptp_usb
        find_endpoints
    close_camera
    close_usb

*/
/***************************************************************************/

int init_usb( PTP_USB *ptp_usb, PTPParams *params)
{
    if (usb_init() < 0 || usb_find_busses() < 0 || usb_find_devices() < 0) {
        ptpro_error( "Unable to init USB subsystem");
        return -1;
    }

    memset( params, 0, sizeof(PTPParams));
    memset( ptp_usb, 0, sizeof(PTP_USB));

    params->byteorder       = PTP_DL_LE;
    params->read_func       = ptp_read_func;
    params->write_func      = ptp_write_func;
    params->check_int_func  = ptp_check_int;
    params->check_int_fast_func=ptp_check_int;

    params->sendreq_func    = ptp_usb_sendreq;
    params->senddata_func   = ptp_usb_senddata;
    params->getresp_func    = ptp_usb_getresp;
    params->getdata_func    = ptp_usb_getdata;

    params->error_func      = ptp_error_func;
    params->debug_func      = ptp_debug_func;

    params->data            = ptp_usb;

    return 0;
}

/***************************************************************************/
/*
   find_device() returns the pointer to a usb_device structure matching
   given busn, devicen numbers. If any or both of arguments are 0 then the
   first matching PTP device structure is returned. 
*/

struct usb_device* find_device( int busn, int devn, short force)
{
    struct usb_bus      *bus;
    struct usb_device   *dev;

    bus = usb_get_busses();
    for (; bus; bus = bus->next)
        for (dev = bus->devices; dev; dev = dev->next)
            if ((dev->config->interface->altsetting->bInterfaceClass ==
                                                    USB_CLASS_PTP) || force)
                if (dev->descriptor.bDeviceClass != USB_CLASS_HUB)
                {
                    int curbusn, curdevn;
                    curbusn = strtol( bus->dirname, NULL, 10);
                    curdevn = strtol( dev->filename, NULL, 10);

                    if (devn == 0) {
                        if (busn == 0)
                            return dev;
                        if (curbusn == busn)
                            return dev;
                    }
                    else {
                        if ((busn == 0) && (curdevn == devn))
                            return dev;
                        if ((curbusn == busn) && (curdevn == devn))
                            return dev;
                    }
                }

    ptpro_error( "Could not find any PTP device%s",
            ((busn || devn) ? " matching given bus/device numbers." : "."));

    return NULL;
}

/***************************************************************************/

int open_camera( PTP_USB *ptp_usb, PTPParams *params)
{
    if (init_ptp_usb( params, ptp_usb) < 0)
        return -1;

    // my Canon A610 has to be set to config 1 - I don't know
    // if this holds true for other makes/models, so try opening
    // with config 0 and if that fails, try config 1 (my Canon
    // remains set to config 1 until it's turned off)
    if (ptp_opensession( params, 1) != PTP_RC_OK) {
        ptpro_debug( "Unable to open session using configuration 0 - trying configuration 1");
        usb_set_configuration( ptp_usb->handle, 1);
        if (ptp_opensession( params, 1) != PTP_RC_OK) {
            ptpro_error( "Could not open session.");
            close_usb( ptp_usb);
            return -1;
        }
    }

    if (ptp_getdeviceinfo( params, &params->deviceinfo) != PTP_RC_OK) {
        ptpro_error( "Could not get device info");
        close_usb( ptp_usb);
        return -1;
    }

    return 0;
}

/***************************************************************************/

int init_ptp_usb( PTPParams* params, PTP_USB* ptp_usb)
{
    ptpro_debug( "Initializing device %s", ptp_usb->dev->filename);
    find_endpoints( ptp_usb);

    ptp_usb->handle = usb_open( ptp_usb->dev);
    if (!ptp_usb->handle) {
        ptpro_error( "Unable to open device.");
        return -1;
    }
    if (verbose)
        ptpro_debug( "device handle= %X", *(int*)ptp_usb->handle);

    usb_claim_interface( ptp_usb->handle,
                         ptp_usb->dev->config->interface->altsetting->bInterfaceNumber);

    return 0;
}

/***************************************************************************/

void find_endpoints( PTP_USB *ptp_usb)
{
    int i, n;
    struct usb_endpoint_descriptor *ep;

    ep = ptp_usb->dev->config->interface->altsetting->endpoint;
    n = ptp_usb->dev->config->interface->altsetting->bNumEndpoints;

    for (i = 0; i < n; i++) {
        if (ep[i].bmAttributes == USB_ENDPOINT_TYPE_BULK) {
            if ((ep[i].bEndpointAddress & USB_ENDPOINT_DIR_MASK) ==
                                          USB_ENDPOINT_DIR_MASK) {
                ptp_usb->inep = ep[i].bEndpointAddress;
                if (verbose)
                    ptpro_debug( "Found inep: 0x%02X", ptp_usb->inep);
            }
            else {
                ptp_usb->outep = ep[i].bEndpointAddress;
                if (verbose)
                    ptpro_debug( "Found outep: 0x%02X", ptp_usb->outep);
            }
            continue;
        }

        if (ep[i].bmAttributes == USB_ENDPOINT_TYPE_INTERRUPT &&
            (ep[i].bEndpointAddress & USB_ENDPOINT_DIR_MASK) ==
                                      USB_ENDPOINT_DIR_MASK) {
            ptp_usb->intep = ep[i].bEndpointAddress;
            if (verbose)
                ptpro_debug( "Found intep: 0x%02X", ptp_usb->intep);
        }
    }
}

/***************************************************************************/

void close_camera( PTP_USB *ptp_usb, PTPParams *params)
{
    if (ptp_usb->handle) {
        if (ptp_closesession( params) != PTP_RC_OK)
            ptpro_error( "Could not close session.");
        close_usb( ptp_usb);
    }
}

/***************************************************************************/

void close_usb( PTP_USB* ptp_usb)
{
    if (ptp_usb->handle) {
        clear_stall( ptp_usb);
        usb_release_interface( ptp_usb->handle,
                               ptp_usb->dev->config->interface->altsetting->bInterfaceNumber);
        usb_close( ptp_usb->handle);
    }
}

/***************************************************************************/
/***************************************************************************/
/*
    custom functions for libptp

    ptp_read_func
    ptp_write_func
    ptp_check_int
    ptp_debug_func
    ptp_error_func
    ptpro_error
    ptpro_debug

*/
/***************************************************************************/

static short ptp_read_func( unsigned char *bytes, unsigned int size, void *data)
{
    int     result;
    PTP_USB *ptp_usb = (PTP_USB *)data;

    result = usb_bulk_read( ptp_usb->handle, ptp_usb->inep, (char*)bytes,
                            size, USB_TIMEOUT);

    // I'm not sure that this condition will ever occur
    if (result == 0 && size != 0)
        result = usb_bulk_read( ptp_usb->handle, ptp_usb->inep, (char*)bytes,
                                size, USB_TIMEOUT);

    return (result > 0 ? PTP_RC_OK : PTP_ERROR_IO);
}

/***************************************************************************/

static short ptp_write_func( unsigned char *bytes, unsigned int size, void *data)
{
    int     result;
    PTP_USB *ptp_usb=(PTP_USB *)data;

    result = usb_bulk_write( ptp_usb->handle, ptp_usb->outep, (char*)bytes,
                             size, USB_TIMEOUT);

    return (result > 0 ? PTP_RC_OK : PTP_ERROR_IO);
}

/***************************************************************************/

static short ptp_check_int( unsigned char *bytes, unsigned int size, void *data)
{
    int     result;
    PTP_USB *ptp_usb = (PTP_USB *)data;

    result = usb_bulk_read( ptp_usb->handle, ptp_usb->intep, (char*)bytes,
                            size, USB_TIMEOUT);
    if (result == 0 && size != 0)
        result = usb_bulk_read( ptp_usb->handle, ptp_usb->intep, (char*)bytes,
                                size, USB_TIMEOUT);

    return (result >= 0 ? PTP_RC_OK : PTP_ERROR_IO);
}

/***************************************************************************/

void ptp_debug_func( void *data, const char *format, va_list args)
{
    if (!verbose)
        return;
    vfprintf( stderr, format, args);
    fprintf( stderr,"\n");
    fflush( stderr);
}

/***************************************************************************/

void ptp_error_func( void *data, const char *format, va_list args)
{
    if (!verbose)
        return;
    vfprintf( stderr, format, args);
    fprintf( stderr,"\n");
    fflush( stderr);
}

/***************************************************************************/

void ptpro_error( const char* format, ...)
{
    va_list args;
    char    buf[256];

    strcpy( buf, "ERROR: ");
    strcat( buf, format);
    strcat( buf, "\n");

    va_start( args, format);
    vfprintf( stderr, buf, args);
    fflush( stderr);
    va_end (args);
}

/***************************************************************************/

void ptpro_debug( const char* format, ...)
{
    va_list args;
    char    buf[256];

    if (verbose < 2)
        return;

    strcpy( buf, "PTPRO: ");
    strcat( buf, format);
    strcat( buf, "\n");

    va_start( args, format);
    vfprintf( stderr, buf, args);
    fflush( stderr);
    va_end (args);
}

/***************************************************************************/

#pragma pack()

/***************************************************************************/

