/***************************************************************************/
/* ptp.c
 *
 * Copyright (C) 2001-2005 Mariusz Woloszyn <emsi@ipartners.pl>
 *
 *  This file is part of libptp2.
 *
 *  libptp2 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  libptp2 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libptp2; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
/***************************************************************************/
/*
 *  This version of libptp2 for OS/2 has been partially rewritten
 *  and extensively restructured.  All changes and new code are
 *      Copyright (C) 2006 Richard L Walsh <rich@e-vertise.com>
 */
/***************************************************************************/

#include "ptp.h"
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

/***************************************************************************/

/* Transaction data phase description */
#define PTP_DP_NODATA           0x0000    /* no data phase */
#define PTP_DP_SENDDATA         0x0001    /* sending data */
#define PTP_DP_GETDATA          0x0002    /* receiving data */
#define PTP_DP_DATA_MASK        0x00ff    /* data phase mask */

/* Number of PTP Request phase parameters */
#define PTP_RQ_PARAM0           0x0000    /* zero parameters */
#define PTP_RQ_PARAM1           0x0100    /* one parameter */
#define PTP_RQ_PARAM2           0x0200    /* two parameters */
#define PTP_RQ_PARAM3           0x0300    /* three parameters */
#define PTP_RQ_PARAM4           0x0400    /* four parameters */
#define PTP_RQ_PARAM5           0x0500    /* five parameters */

/* PTP Events wait for or check mode */
#define PTP_EVENT_CHECK         0x0000    /* waits for */
#define PTP_EVENT_CHECK_FAST    0x0001    /* checks */

#define PTP_CNT_INIT(cnt)       {memset(&cnt,0,sizeof(cnt));}

/***************************************************************************/

/* Pack / unpack functions */
#include "ptp-pack.c"

#pragma pack(1)

/***************************************************************************/
/**
 * Major PTP functions
 *
 * all ptp_ functions should take integer parameters in host byte order!
 **/
/***************************************************************************/
/**
 * ptp_getdeviceinfo:
 * params:    PTPParams*
 *
 * Gets device info dataset and fills deviceinfo structure.
 *
 * Return values: Some PTP_RC_* code.
 **/

uint16_t ptp_getdeviceinfo( PTPParams* params, PTPDeviceInfo* deviceinfo)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           di = NULL;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetDeviceInfo;
    ptp.Nparam  = 0;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &di);
    if (ret == PTP_RC_OK)
        ptp_unpack_DI( params, di, deviceinfo);

    free(di);
    return ret;
}

/***************************************************************************/
/**
 * ptp_opensession:
 * params:  PTPParams*
 *          session            - session number 
 *
 * Establishes a new session.
 *
 * Return values: Some PTP_RC_* code.
 **/

uint16_t ptp_opensession( PTPParams* params, uint32_t session)
{
    uint16_t        ret;
    PTPContainer    ptp;

    /* SessonID & TransactionID should always
       be set to 0 for OpenSession request! */
    params->session_id     = 0;
    params->transaction_id = 0;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_OpenSession;
    ptp.Param1  = session;
    ptp.Nparam  = 1;
    ret = ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);

    /* now set the global session id to current session number */
    params->session_id = session;

    return ret;
}

/***************************************************************************/
/**
 * ptp_closesession:
 * params:    PTPParams*
 *
 * Closes session.
 *
 * Return values: Some PTP_RC_* code.
 **/

uint16_t ptp_closesession( PTPParams* params)
{
    PTPContainer    ptp;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CloseSession;
    ptp.Nparam  = 0;

    return ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_getststorageids:
 * params:    PTPParams*
 *
 * Gets array of StorageiDs and fills the storageids structure.
 *
 * Return values: Some PTP_RC_* code.
 **/

uint16_t ptp_getstorageids( PTPParams* params, PTPStorageIDs* storageids)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           sids = NULL;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetStorageIDs;
    ptp.Nparam  = 0;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &sids);
    if (ret == PTP_RC_OK)
        ptp_unpack_SIDs( params, sids, storageids);

    free( sids);
    return ret;
}

/***************************************************************************/
/**
 * ptp_getststorageinfo:
 * params:  PTPParams*
 *          storageid       - StorageID
 *
 * Gets StorageInfo dataset of desired storage and fills storageinfo
 * structure.
 *
 * Return values: Some PTP_RC_* code.
 **/

uint16_t ptp_getstorageinfo( PTPParams* params, uint32_t storageid,
                             PTPStorageInfo* storageinfo)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           si = NULL;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetStorageInfo;
    ptp.Param1  = storageid;
    ptp.Nparam  = 1;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &si);
    if (ret == PTP_RC_OK)
        ptp_unpack_SI( params, si, storageinfo);

    free( si);
    return ret;
}

/***************************************************************************/
/**
 * ptp_getobjecthandles:
 * params:  PTPParams*
 *          storage             - StorageID
 *          objectformatcode    - ObjectFormatCode (optional)
 *          associationOH       - ObjectHandle of Association for which a
 *                                list of children is desired (optional)
 *          objecthandles       - pointer to structure
 *
 * Fills objecthandles with structure returned by device.
 *
 * Return values: Some PTP_RC_* code.
 **/

uint16_t ptp_getobjecthandles( PTPParams* params, uint32_t storage,
                               uint32_t objectformatcode,
                               uint32_t associationOH,
                               PTPObjectHandles* objecthandles)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           oh = NULL;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetObjectHandles;
    ptp.Param1  = storage;
    ptp.Param2  = objectformatcode;
    ptp.Param3  = associationOH;
    ptp.Nparam  = 3;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &oh);
    if (ret == PTP_RC_OK)
        ptp_unpack_OH( params, oh, objecthandles);

    free( oh);
    return ret;
}

/***************************************************************************/

uint16_t ptp_getobjectinfo( PTPParams* params, uint32_t handle,
                            PTPObjectInfo* objectinfo)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           oi = NULL;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetObjectInfo;
    ptp.Param1  = handle;
    ptp.Nparam  = 1;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &oi);
    if (ret == PTP_RC_OK)
        ptp_unpack_OI( params, oi, objectinfo);

    free( oi);
    return ret;
}

/***************************************************************************/

uint16_t ptp_getobject( PTPParams* params, uint32_t handle, char** object)
{
    PTPContainer ptp;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetObject;
    ptp.Param1  = handle;
    ptp.Nparam  = 1;

    return ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, object);
}

/***************************************************************************/

uint16_t ptp_getthumb( PTPParams* params, uint32_t handle,  char** object)
{
    PTPContainer ptp;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetThumb;
    ptp.Param1  = handle;
    ptp.Nparam  = 1;

    return ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, object);
}

/***************************************************************************/
/**
 * ptp_deleteobject:
 * params:  PTPParams*
 *          handle          - object handle
 *          ofc             - object format code (optional)
 * 
 * Deletes desired objects.
 *
 * Return values: Some PTP_RC_* code.
 **/

uint16_t ptp_deleteobject( PTPParams* params, uint32_t handle, uint32_t ofc)
{
    PTPContainer ptp;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_DeleteObject;
    ptp.Param1  = handle;
    ptp.Param2  = ofc;
    ptp.Nparam  = 2;

    return ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/

uint16_t ptp_getdevicepropdesc( PTPParams* params, uint16_t propcode, 
                                PTPDevicePropDesc* devicepropertydesc)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           dpd = NULL;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetDevicePropDesc;
    ptp.Param1  = propcode;
    ptp.Nparam  = 1;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &dpd);
    if (ret == PTP_RC_OK)
        ptp_unpack_DPD( params, dpd, devicepropertydesc);

    free( dpd);
    return ret;
}

/***************************************************************************/

uint16_t ptp_getdevicepropvalue( PTPParams* params, uint16_t propcode,
                                 void** value, uint16_t datatype)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           dpv = NULL;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_GetDevicePropValue;
    ptp.Param1  = propcode;
    ptp.Nparam  = 1;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &dpv);
    if (ret == PTP_RC_OK)
        ptp_unpack_DPV( params, dpv, value, datatype);

    free( dpv);
    return ret;
}

/***************************************************************************/

uint16_t ptp_setdevicepropvalue( PTPParams* params, uint16_t propcode,
                                 void* value, uint16_t datatype)
{
    uint16_t        ret;
    PTPContainer    ptp;
    uint32_t        size;
    char*           dpv = NULL;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_SetDevicePropValue;
    ptp.Param1  = propcode;
    ptp.Nparam  = 1;
    size        = ptp_pack_DPV( params, value, &dpv, datatype);

    ret = ptp_transaction( params, &ptp, PTP_DP_SENDDATA, size, &dpv);

    free(dpv);
    return ret;
}

/***************************************************************************/

int ptp_property_issupported( PTPParams* params, uint16_t property)
{
    int i = 0;

    for (; i < params->deviceinfo.DevicePropertiesSupported_len; i++)
        if (params->deviceinfo.DevicePropertiesSupported[i] == property)
            return 1;

    return 0;
}

/***************************************************************************/

void ptp_free_devicepropdesc( PTPDevicePropDesc* dpd)
{
    uint16_t i;

    free( dpd->FactoryDefaultValue);
    free( dpd->CurrentValue);

    switch (dpd->FormFlag) {

        case PTP_DPFF_Range:
            free( dpd->FORM.Range.MinimumValue);
            free( dpd->FORM.Range.MaximumValue);
            free( dpd->FORM.Range.StepSize);
            break;

        case PTP_DPFF_Enumeration:
            for (i = 0; i < dpd->FORM.Enum.NumberOfValues; i++)
                free( dpd->FORM.Enum.SupportedValue[i]);
            free(dpd->FORM.Enum.SupportedValue);
    }
}

/***************************************************************************/
/* transaction handling */
/***************************************************************************/

/**
 * ptp_transaction:
 * params:  PTPParams*      params
 *          PTPContainer*   ptp     - general ptp container
 *          uint16_t        flags   - lower 8 bits - data phase description
 *          unsigned int    sendlen - senddata phase data length
 *          char**          data    - send or receive data buffer pointer
 *
 * Performs PTP transaction. ptp is a PTPContainer with appropriate fields
 * filled in (i.e. operation code and parameters). It's up to caller to do
 * so.
 * The flags decide whether the transaction has a data phase and what is its
 * direction (send or receive). 
 * If transaction is sending data the sendlen should contain its length in
 * bytes, otherwise it's ignored.
 * The data should contain an address of a pointer to data going to be sent
 * or is filled with such a pointer address if data are received depending
 * on dataphase direction (send or receive) or is being ignored (no
 * dataphase).
 * The memory for a pointer should be preserved by the caller, if data are
 * being retreived the appropriate amount of memory is being allocated
 * (the caller should handle that!).
 *
 * Return values: Some PTP_RC_* code.
 * Upon success PTPContainer* ptp contains PTP Response Phase container with
 * all fields filled in.
 **/

uint16_t ptp_transaction( PTPParams* params, PTPContainer* ptp, 
                          uint16_t flags, unsigned int sendlen, char** data)
{
    uint16_t ret;

    if (params == NULL || ptp == NULL)
        return PTP_ERROR_BADPARAM;
    
    ptp->Transaction_ID = params->transaction_id++;
    ptp->SessionID = params->session_id;

    /* send request */
    ret = params->sendreq_func( params, ptp);
    if (ret != PTP_RC_OK)
        return ret;

    /* is there a dataphase? */
    switch (flags & PTP_DP_DATA_MASK) {

        case PTP_DP_SENDDATA:
            ret = params->senddata_func( params, ptp, *data, sendlen);
            if (ret != PTP_RC_OK)
                return ret;
            break;

        case PTP_DP_GETDATA:
            ret = params->getdata_func( params, ptp, (unsigned char**)data);
            if (ret != PTP_RC_OK)
                return ret;
            break;

        case PTP_DP_NODATA:
            break;

        default:
            return PTP_ERROR_BADPARAM;
    }

    /* get response */
    return (params->getresp_func( params, ptp));
}

/***************************************************************************/
//  replacable send / receive functions
/***************************************************************************/

uint16_t ptp_usb_sendreq( PTPParams* params, PTPContainer* req)
{
    uint16_t    ret;
    PTPUSBBulkContainer usbreq;

    /* build appropriate USB container */
    usbreq.length   = htod32( PTP_USB_BULK_REQ_LEN_X( req->Nparam));
    usbreq.type     = htod16( PTP_USB_CONTAINER_COMMAND);
    usbreq.code     = htod16( req->Code);
    usbreq.trans_id = htod32( req->Transaction_ID);
    usbreq.payload.params.param1 = htod32( req->Param1);
    usbreq.payload.params.param2 = htod32( req->Param2);
    usbreq.payload.params.param3 = htod32( req->Param3);
    usbreq.payload.params.param4 = htod32( req->Param4);
    usbreq.payload.params.param5 = htod32( req->Param5);

    /* send it to responder */
    ret = params->write_func( (unsigned char *)&usbreq,
                              PTP_USB_BULK_REQ_LEN_X( req->Nparam),
                              params->data);

    if (ret != PTP_RC_OK) {
        ptp_error( params, "LIBPTP: error sending request - code= 0x%04x  rc= 0x%04x",
                   req->Code, ret);
        ret = PTP_ERROR_IO;
    }

    return ret;
}

/***************************************************************************/

uint16_t ptp_usb_senddata( PTPParams* params, PTPContainer* ptp,
                           unsigned char *data, unsigned int size)
{
    uint16_t ret;
    PTPUSBBulkContainer usbdata;

do {
    /* build appropriate USB container */
    usbdata.length  = htod32( PTP_USB_BULK_HDR_LEN+size);
    usbdata.type    = htod16( PTP_USB_CONTAINER_DATA);
    usbdata.code    = htod16( ptp->Code);
    usbdata.trans_id= htod32( ptp->Transaction_ID);
    memcpy( usbdata.payload.data, data,
        (size < PTP_USB_BULK_PAYLOAD_LEN) ? size : PTP_USB_BULK_PAYLOAD_LEN);

    /* send first part of data */
    ret = params->write_func( (unsigned char *)&usbdata, PTP_USB_BULK_HDR_LEN+
        ((size < PTP_USB_BULK_PAYLOAD_LEN) ? size : PTP_USB_BULK_PAYLOAD_LEN),
                              params->data);
    if (ret != PTP_RC_OK)
        break;

    if (size <= PTP_USB_BULK_PAYLOAD_LEN)
        break;

    /* if everything OK send the rest */
    ret = params->write_func( data+PTP_USB_BULK_PAYLOAD_LEN,
                              size-PTP_USB_BULK_PAYLOAD_LEN, params->data);

} while (0);

    if (ret != PTP_RC_OK) {
        ptp_error( params, "LIBPTP: error sending data - code= 0x%04x  rc= 0x%04x",
                   ptp->Code, ret);
        ret = PTP_ERROR_IO;
    }

    return ret;
}

/***************************************************************************/

uint16_t ptp_usb_getdata( PTPParams* params, PTPContainer* ptp,
                          unsigned char **data)
{
    uint16_t ret;
    PTPUSBBulkContainer usbdata;
    uint32_t len;

    PTP_CNT_INIT(usbdata);

do {
    /* read first(?) part of data */
    ret = params->read_func( (unsigned char *)&usbdata,
                             sizeof(usbdata), params->data);
    if (ret != PTP_RC_OK)
        break;

    if (dtoh16( usbdata.type) != PTP_USB_CONTAINER_DATA) {
        ret = PTP_ERROR_DATA_EXPECTED;
        break;
    }

    if (dtoh16( usbdata.code) != ptp->Code) {
        ret = dtoh16( usbdata.code);
        if (ret != PTP_RC_OK)
            break;
    }

    /* evaluate data length */
    /* allocate memory for data if not allocated already */
    len = dtoh32( usbdata.length) - PTP_USB_BULK_HDR_LEN;
    if (*data == NULL)
        *data = calloc( len, 1);

    /* copy first part of data to 'data' */
    memcpy( *data, usbdata.payload.data,
            ((len > PTP_USB_BULK_PAYLOAD_LEN) ?
             PTP_USB_BULK_PAYLOAD_LEN : len));

    /* is that all of data? */
    if (len + PTP_USB_BULK_HDR_LEN <= sizeof(usbdata))
        break;

    /* if not finally read the rest of it */
    ret = params->read_func( ((unsigned char *)(*data))+
                             PTP_USB_BULK_PAYLOAD_LEN,
                             len-PTP_USB_BULK_PAYLOAD_LEN,
                             params->data);

} while (0);

    if (ret != PTP_RC_OK) {
        ptp_error( params, "LIBPTP: error getting data - code= 0x%04x  rc= 0x%04x",
                   ptp->Code, ret);
        ret = PTP_ERROR_IO;
    }

    return ret;
}

/***************************************************************************/

uint16_t ptp_usb_getresp( PTPParams* params, PTPContainer* resp)
{
    uint16_t ret;
    PTPUSBBulkContainer usbresp;

    PTP_CNT_INIT(usbresp);

do {
    // read response, it should never be longer than sizeof(usbresp)
    ret = params->read_func( (unsigned char *)&usbresp,
                             sizeof(usbresp), params->data);

    if (ret != PTP_RC_OK)
        break;

    if (dtoh16( usbresp.type) != PTP_USB_CONTAINER_RESPONSE) {
        ret = PTP_ERROR_RESP_EXPECTED;
        break;
    }

    if (dtoh16( usbresp.code) != resp->Code) {
        ret = dtoh16( usbresp.code);
        if (ret != PTP_RC_OK)
            break;
    }

    // build an appropriate PTPContainer
    resp->Code = dtoh16( usbresp.code);
    resp->SessionID = params->session_id;
    resp->Transaction_ID = dtoh32( usbresp.trans_id);
    resp->Param1 = dtoh32( usbresp.payload.params.param1);
    resp->Param2 = dtoh32( usbresp.payload.params.param2);
    resp->Param3 = dtoh32( usbresp.payload.params.param3);
    resp->Param4 = dtoh32( usbresp.payload.params.param4);
    resp->Param5 = dtoh32( usbresp.payload.params.param5);

} while (0);

    if (ret != PTP_RC_OK) {
        ptp_error( params, "LIBPTP: error getting response - code= 0x%04x  rc= 0x%04x",
                  resp->Code, ret);
        ret = PTP_ERROR_IO;
    }

    return ret;
}

/***************************************************************************/
//  reolacable Error & Debug display functions
/***************************************************************************/

void ptp_debug( PTPParams *params, const char *format, ...)
{  
    va_list args;

    va_start( args, format);
    if (params->debug_func != NULL)
        params->debug_func( params->data, format, args);
    else {
        vfprintf( stderr, format, args);
        fprintf( stderr, "\n");
        fflush( stderr);
    }
    va_end (args);
}  

/***************************************************************************/

void ptp_error( PTPParams *params, const char *format, ...)
{  
    va_list args;

    va_start( args, format);
    if (params->error_func != NULL)
        params->error_func( params->data, format, args);
    else {
        vfprintf( stderr, format, args);
        fprintf( stderr, "\n");
        fflush( stderr);
    }
    va_end (args);
}

/***************************************************************************/
/***************************************************************************/

//  The following are commented-out because they aren't used
//  and ilink isn't smart enough to exclude them from the exe.

#if 0

/***************************************************************************/
//  Major functions
/***************************************************************************/
/**
 * ptp_sendobjectinfo:
 * params:  PTPParams*
 *          store           - destination StorageID on Responder
 *          parenthandle    - Parent ObjectHandle on responder
 *          handle          - see Return values
 *          objectinfo      - ObjectInfo that is to be sent
 * 
 * Sends ObjectInfo of file that is to be sent via SendFileObject.
 *
 * Return values: Some PTP_RC_* code.
 * Upon success : store         - Responder StorageID in which
 *                                object will be stored
 *                parenthandle  - Responder Parent ObjectHandle
 *                                in which the object will be stored
 *                handle        - Responder's reserved ObjectHandle
 *                                for the incoming object
 **/

uint16_t ptp_sendobjectinfo( PTPParams* params, uint32_t* store, 
                             uint32_t* parenthandle, uint32_t* handle,
                             PTPObjectInfo* objectinfo)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           oidata = NULL;
    uint32_t        size;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_SendObjectInfo;
    ptp.Param1  = *store;
    ptp.Param2  = *parenthandle;
    ptp.Nparam  = 2;
    
    size = ptp_pack_OI( params, objectinfo, &oidata);
    ret = ptp_transaction( params, &ptp, PTP_DP_SENDDATA, size, &oidata); 

    *store = ptp.Param1;
    *parenthandle = ptp.Param2;
    *handle = ptp.Param3; 

    free( oidata);
    return ret;
}

/***************************************************************************/
/**
 * ptp_sendobject:
 * params:  PTPParams*
 *          object          - contains the object that is to be sent
 *          size            - object size
 *        
 * Sends object to Responder.
 *
 * Return values: Some PTP_RC_* code.
 *
 */

uint16_t ptp_sendobject( PTPParams* params, char* object, uint32_t size)
{
    PTPContainer ptp;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_SendObject;
    ptp.Nparam  = 0;

    return ptp_transaction( params, &ptp, PTP_DP_SENDDATA, size, &object);
}

/***************************************************************************/
/**
 * ptp_initiatecapture:
 * params:  PTPParams*
 *          storageid       - destination StorageID on Responder
 *          ofc             - object format code
 * 
 * Causes device to initiate the capture of one or more new data objects
 * according to its current device properties, storing the data into store
 * indicated by storageid. If storageid is 0x00000000, the object(s) will
 * be stored in a store that is determined by the capturing device.
 * The capturing of new data objects is an asynchronous operation.
 *
 * Return values: Some PTP_RC_* code.
 **/

uint16_t ptp_initiatecapture( PTPParams* params, uint32_t storageid,
                              uint32_t ofc)
{
    PTPContainer ptp;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_InitiateCapture;
    ptp.Param1  = storageid;
    ptp.Param2  = ofc;
    ptp.Nparam  = 2;

    return ptp_transaction(params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/

int ptp_operation_issupported( PTPParams* params, uint16_t operation)
{
    int i = 0;

    for (; i < params->deviceinfo.OperationsSupported_len; i++)
        if (params->deviceinfo.OperationsSupported[i] == operation)
            return 1;

    return 0;
}

/***************************************************************************/
//  Event handling functions
/***************************************************************************/

uint16_t ptp_usb_event_check( PTPParams* params, PTPContainer* event)
{
    return ptp_usb_event( params, event, PTP_EVENT_CHECK_FAST);
}

/***************************************************************************/

uint16_t ptp_usb_event_wait( PTPParams* params, PTPContainer* event)
{
    return ptp_usb_event( params, event, PTP_EVENT_CHECK);
}

/***************************************************************************/

uint16_t ptp_usb_event( PTPParams* params, PTPContainer* event, int wait)
{
    uint16_t ret;
    PTPUSBEventContainer usbevent;

    if (params == NULL || event==NULL)
        return PTP_ERROR_BADPARAM;

    PTP_CNT_INIT( usbevent);
    
    switch (wait) {
        case PTP_EVENT_CHECK:
            ret = params->check_int_func( (unsigned char*)&usbevent,
                                          sizeof(usbevent), params->data);
            break;
        case PTP_EVENT_CHECK_FAST:
            ret = params->check_int_fast_func( (unsigned char*) &usbevent,
                                               sizeof(usbevent), params->data);
            break;
        default:
            ret=PTP_ERROR_BADPARAM;
    }

    if (ret != PTP_RC_OK) {
        ptp_error( params, "LIBPTP: error reading event - rc= 0x%04x", ret);
        ret = PTP_ERROR_IO;
        return ret;
    } 

    /* if we read anything over interrupt endpoint it must be an event */
    /* build an appropriate PTPContainer */
    event->Code      = dtoh16( usbevent.code);
    event->SessionID = params->session_id;
    event->Transaction_ID = dtoh32( usbevent.trans_id);
    event->Param1    = dtoh32( usbevent.param1);
    event->Param2    = dtoh32( usbevent.param2);
    event->Param3    = dtoh32( usbevent.param3);

    return PTP_RC_OK;
}

/***************************************************************************/
//  Kodak Extensions
/***************************************************************************/
/**
 * ptp_ek_sendfileobjectinfo:
 * params:  PTPParams*
 *          store           - destination StorageID on Responder
 *          parenthandle    - Parent ObjectHandle on responder
 *          handle          - see Return values
 *          objectinfo      - ObjectInfo that is to be sent
 * 
 * Sends ObjectInfo of file that is to be sent via SendFileObject.
 *
 * Return values: Some PTP_RC_* code.
 * Upon success: store          - Responder StorageID in which
 *                                object will be stored
 *               parenthandle   - Responder Parent ObjectHandle
 *                                in which the object will be stored
 *               handle         - Responder's reserved ObjectHandle
 *                                for the incoming object
 **/

uint16_t ptp_ek_sendfileobjectinfo( PTPParams* params, uint32_t* store, 
                                    uint32_t* parenthandle, uint32_t* handle,
                                    PTPObjectInfo* objectinfo)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           oidata=NULL;
    uint32_t        size;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_EK_SendFileObjectInfo;
    ptp.Param1  = *store;
    ptp.Param2  = *parenthandle;
    ptp.Nparam  = 2;
    
    size = ptp_pack_OI( params, objectinfo, &oidata);
    ret = ptp_transaction( params, &ptp, PTP_DP_SENDDATA, size, &oidata); 

    *store        = ptp.Param1;
    *parenthandle = ptp.Param2;
    *handle       = ptp.Param3; 

    free(oidata);
    return ret;
}

/***************************************************************************/
/**
 * ptp_ek_sendfileobject:
 * params:  PTPParams*
 *          object          - contains the object that is to be sent
 *          size            - object size
 *        
 * Sends object to Responder.
 *
 * Return values: Some PTP_RC_* code.
 *
 */

uint16_t ptp_ek_sendfileobject( PTPParams* params, char* object, uint32_t size)
{
    PTPContainer ptp;

    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_EK_SendFileObject;
    ptp.Nparam  = 0;

    return ptp_transaction( params, &ptp, PTP_DP_SENDDATA, size, &object);
}

/***************************************************************************/
//  Canon PTP extensions support
//  (C) Nikolai Kopanygin 2003
/***************************************************************************/
/**
 * ptp_canon_getobjectsize:
 * params:  PTPParams*
 *          uint32_t handle     - ObjectHandle
 *          uint32_t p2         - Yet unknown parameter, value 0 works.
 * 
 * Gets form the responder the size of the specified object.
 *
 * Return values: Some PTP_RC_* code.
 * Upon success : uint32_t* size    - The object size
 *                uint32_t rp2      - Yet unknown parameter
 *
 **/

uint16_t ptp_canon_getobjectsize( PTPParams* params, uint32_t handle,
                                  uint32_t p2, uint32_t* size, uint32_t* rp2) 
{
    uint16_t ret;
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_GetObjectSize;
    ptp.Param1  = handle;
    ptp.Param2  = p2;
    ptp.Nparam  = 2;

    ret = ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);

    *size = ptp.Param1;
    *rp2  = ptp.Param2;

    return ret;
}

/***************************************************************************/
/**
 * ptp_canon_startshootingmode:
 * params:    PTPParams*
 * 
 * Starts shooting session. It emits a StorageInfoChanged
 * event via the interrupt pipe and pushes the StorageInfoChanged
 * and CANON_CameraModeChange events onto the event stack
 * (see operation PTP_OC_CANON_CheckEvent).
 *
 * Return values: Some PTP_RC_* code.
 *
 **/

uint16_t ptp_canon_startshootingmode( PTPParams* params)
{
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_StartShootingMode;
    ptp.Nparam  = 0;

    return ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_canon_endshootingmode:
 * params:    PTPParams*
 * 
 * This operation is observed after pressing the Disconnect 
 * button on the Remote Capture app. It emits a StorageInfoChanged 
 * event via the interrupt pipe and pushes the StorageInfoChanged
 * and CANON_CameraModeChange events onto the event stack
 * (see operation PTP_OC_CANON_CheckEvent).
 *
 * Return values: Some PTP_RC_* code.
 *
 **/
uint16_t ptp_canon_endshootingmode( PTPParams* params)
{
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_EndShootingMode;
    ptp.Nparam  = 0;

    return ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_canon_viewfinderon:
 * params:    PTPParams*
 * 
 * Prior to start reading viewfinder images, one  must call this operation.
 * Supposedly, this operation affects the value of the CANON_ViewfinderMode
 * property.
 *
 * Return values: Some PTP_RC_* code.
 *
 **/

uint16_t ptp_canon_viewfinderon( PTPParams* params)
{
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_ViewfinderOn;
    ptp.Nparam  = 0;

    return ptp_transaction(params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_canon_viewfinderoff:
 * params:    PTPParams*
 * 
 * Before changing the shooting mode, or when one doesn't need to read
 * viewfinder images any more, one must call this operation.
 * Supposedly, this operation affects the value of the CANON_ViewfinderMode
 * property.
 *
 * Return values: Some PTP_RC_* code.
 *
 **/
uint16_t ptp_canon_viewfinderoff( PTPParams* params)
{
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_ViewfinderOff;
    ptp.Nparam  = 0;

    return ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_canon_reflectchanges:
 * params:  PTPParams*
 *          uint32_t p1     - Yet unknown parameter, value 7 works
 * 
 * Make viewfinder reflect changes.
 * There is a button for this operation in the Remote Capture app.
 * What it does exactly I don't know. This operation is followed
 * by the CANON_GetChanges(?) operation in the log.
 *
 * Return values: Some PTP_RC_* code.
 *
 **/

uint16_t ptp_canon_reflectchanges( PTPParams* params, uint32_t p1)
{
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_ReflectChanges;
    ptp.Param1  = p1;
    ptp.Nparam  = 1;

    return ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_canon_checkevent:
 * params:    PTPParams*
 * 
 * The camera has a FIFO stack, in which it accumulates events.
 * Partially these events are communicated also via the USB interrupt pipe
 * according to the PTP USB specification, partially not.
 * This operation returns from the device a block of data, empty,
 * if the event stack is empty, or filled with an event's data otherwise.
 * The event is removed from the stack in the latter case.
 * The Remote Capture app sends this command to the camera all the time
 * of connection, filling with it the gaps between other operations. 
 *
 * Return values: Some PTP_RC_* code.
 * Upon success : PTPUSBEventContainer* event   - filled with the event data
 *                                                if any
 *                int *isevent                  - returns 1 in case of event
 *                                                or 0 otherwise
 **/

uint16_t ptp_canon_checkevent( PTPParams* params, PTPUSBEventContainer* event,
                               int* isevent)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char            *evdata = NULL;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_CheckEvent;
    ptp.Nparam  = 0;
    *isevent    = 0;

    ret=ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &evdata);

    if (evdata != NULL) {
        if (ret == PTP_RC_OK) {
            ptp_unpack_EC( params, evdata, event);
            *isevent=1;
        }
        free( evdata);
    }

    return ret;
}

/***************************************************************************/
/**
 * ptp_canon_focuslock:
 *
 * This operation locks the focus. It is followed by the CANON_GetChanges(?)
 * operation in the log. 
 * It affects the CANON_MacroMode property. 
 *
 * params:  PTPParams*
 *
 * Return values: Some PTP_RC_* code.
 *
 **/

uint16_t ptp_canon_focuslock( PTPParams* params)
{
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_FocusLock;
    ptp.Nparam  = 0;

    return ptp_transaction (params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_canon_focusunlock:
 *
 * This operation unlocks the focus. It is followed by the CANON_GetChanges(?)
 * operation in the log. 
 * It sets the CANON_MacroMode property value to 1 (where it occurs in the log).
 * 
 * params:  PTPParams*
 *
 * Return values: Some PTP_RC_* code.
 *
 **/

uint16_t ptp_canon_focusunlock( PTPParams* params)
{
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_FocusUnlock;
    ptp.Nparam  = 0;

    return ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_canon_initiatecaptureinmemory:
 * 
 * This operation starts the image capture according to the current camera
 * settings. When the capture has happened, the camera emits a CaptureComplete
 * event via the interrupt pipe and pushes the CANON_RequestObjectTransfer,
 * CANON_DeviceInfoChanged and CaptureComplete events onto the event stack
 * (see operation CANON_CheckEvent). From the CANON_RequestObjectTransfer
 * event's parameter one can learn the just captured image's ObjectHandle.
 * The image is stored in the camera's own RAM.
 * On the next capture the image will be overwritten!
 *
 * params:  PTPParams*
 *
 * Return values: Some PTP_RC_* code.
 *
 **/

uint16_t ptp_canon_initiatecaptureinmemory( PTPParams* params)
{
    PTPContainer ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_InitiateCaptureInMemory;
    ptp.Nparam  = 0;

    return ptp_transaction( params, &ptp, PTP_DP_NODATA, 0, NULL);
}

/***************************************************************************/
/**
 * ptp_canon_getpartialobject:
 *
 * This operation is used to read from the device a data 
 * block of an object from a specified offset.
 *
 * params:  PTPParams*
 *          uint32_t handle     - the handle of the requested object
 *          uint32_t offset     - the offset in bytes from the beginning
 *                                of the object
 *          uint32_t size       - the requested size of data block to read
 *          uint32_t pos        - 1 for the first block,
 *                                2 for a block in the middle,
 *                                3 for the last block
 *
 * Return values: Some PTP_RC_* code.
 *      char **block        - the pointer to the block of data read
 *      uint32_t* readnum   - the number of bytes read
 *
 **/

uint16_t ptp_canon_getpartialobject( PTPParams* params, uint32_t handle, 
                                     uint32_t offset, uint32_t size,
                                     uint32_t pos, char** block, 
                                     uint32_t* readnum)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char *          data = NULL;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_GetPartialObject;
    ptp.Param1  = handle;
    ptp.Param2  = offset;
    ptp.Param3  = size;
    ptp.Param4  = pos;
    ptp.Nparam  = 4;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &data);
    if (ret == PTP_RC_OK) {
        *block   = data;
        *readnum = ptp.Param1;
    }

    return ret;
}

/***************************************************************************/
/**
 * ptp_canon_getviewfinderimage:
 *
 * This operation can be used to read the image which is currently
 * in the camera's viewfinder. The image size is 320x240, format is JPEG.
 * Of course, prior to calling this operation, one must turn the viewfinder
 * on with the CANON_ViewfinderOn command.
 * Invoking this operation many times, one can get live video from the camera!
 * 
 * params:    PTPParams*
 * 
 * Return values: Some PTP_RC_* code.
 *                  char **image    - the pointer to the read image
 *                  unit32_t *size  - the size of the image in bytes
 *
 **/

uint16_t ptp_canon_getviewfinderimage( PTPParams* params, char** image,
                                       uint32_t* size)
{
    uint16_t        ret;
    PTPContainer    ptp;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_GetViewfinderImage;
    ptp.Nparam  = 0;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, image);
    if (ret==PTP_RC_OK)
        *size = ptp.Param1;

    return ret;
}

/***************************************************************************/
/**
 * ptp_canon_getchanges:
 *
 * This is an interesting operation, about the effect of which I am not sure.
 * This command is called every time when a device property has been changed 
 * with the SetDevicePropValue operation, and after some other operations.
 * This operation reads the array of Device Properties which have been changed
 * by the previous operation.
 * Probably, this operation is even required to make those changes work.
 *
 * params:    PTPParams*
 * 
 * Return values: Some PTP_RC_* code.
 *      uint16_t** props - the pointer to the array of changed properties
 *      uint32_t* propnum - the number of elements in the *props array
 *
 **/

uint16_t ptp_canon_getchanges( PTPParams* params, uint16_t** props,
                               uint32_t* propnum)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char*           data = NULL;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_GetChanges;
    ptp.Nparam  = 0;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &data);
    if (ret == PTP_RC_OK)
        *propnum = ptp_unpack_uint16_t_array( params,data,0,props);

    free( data);
    return ret;
}

/***************************************************************************/
/**
 * ptp_canon_getfolderentries:
 *
 * This command reads a specified object's record in a device's filesystem,
 * or the records of all objects belonging to a specified folder (association).
 *  
 * params:  PTPParams*
 *          uint32_t store  - StorageID,
 *          uint32_t p2     - Yet unknown (0 value works OK)
 *          uint32_t parent - Parent Object Handle
 *                            If Parent Object Handle is 0xffffffff, 
 *                            the Parent Object is the top level folder.
 *          uint32_t handle - Object Handle
 *                            If Object Handle is 0, the records of all
 *                            objects belonging to the Parent Object are read.
 *                            If Object Handle is not 0, only the record of
 *                            2this Object is read.
 *
 * Return values: Some PTP_RC_* code.
 *      PTPCANONFolderEntry** entries   - the pointer to the folder entry array
 *      uint32_t* entnum                - the number of elements of the array
 *
 **/

uint16_t ptp_canon_getfolderentries( PTPParams* params, uint32_t store,
                                     uint32_t p2, uint32_t parent,
                                     uint32_t handle, 
                                     PTPCANONFolderEntry** entries,
                                     uint32_t* entnum)
{
    uint16_t        ret;
    PTPContainer    ptp;
    char *          data = NULL;
    
    PTP_CNT_INIT( ptp);
    ptp.Code    = PTP_OC_CANON_GetFolderEntries;
    ptp.Param1  = store;
    ptp.Param2  = p2;
    ptp.Param3  = parent;
    ptp.Param4  = handle;
    ptp.Nparam  = 4;

    ret = ptp_transaction( params, &ptp, PTP_DP_GETDATA, 0, &data);
    if (ret == PTP_RC_OK) {
        int i;
        *entnum = ptp.Param1;
        *entries = calloc( *entnum, sizeof(PTPCANONFolderEntry));

        if (*entries == NULL)
            ret = PTP_ERROR_IO;
        else
            for(i = 0; i < *entnum; i++)
                ptp_unpack_Canon_FE( params, data+i*PTP_CANON_FolderEntryLen,
                                     &((*entries)[i]));
    }

    free(data);
    return ret;
}

/***************************************************************************/
#endif
// unused functions
/***************************************************************************/

#pragma pack()

/***************************************************************************/

