/*************************************************************************
 *                                                                       *
 * exiftool.cpp                                                          *
 *                                                                       *
 * Compile with emx!                                                     *
 *                                                                       *
 *************************************************************************/


#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#ifdef __OS2__
#include <sys/ea.h>
#endif

#include "exif.hpp"
#include "string.hpp"

#include <exception>
#include <sstream>


/* Return values */

#define RET_OK 0
#define RET_ERROR 1
#define RET_HELP 2
#define RET_INVALID_ARG 3


/*************************************************************************
 *                                                                       *
 * help()                                                                *
 * Print help screen                                                     *
 *                                                                       *
 *************************************************************************/
static void help()
{  puts(
    "\n"
    "Exiftool 0.1\n"
    "  (C) by Marcel Mller, 2004\n"
    "Purpose:\n"
    "  Rename photo file(s) in EXIF format. Convert EXIF\n"
    "  info contained in the file(s) into OS/2 extended attributes.\n"
    "  EXIF up to version 2.2 is supported.\n"
    "  Exiftool is very similar to renamepics.\n"
    "Usage:\n"
    #ifdef __OS2__
    "  exiftool [/d] [/e] [/h] [File]\n"
    #else
    "  exiftool [/d] [/h] [File]\n"
    #endif
    "Parameters:\n"
    "  /d        Only print complete file information to standard output\n"
    "            [default: rename file(s) and generate EAs quietly]\n"
    #ifdef __OS2__
    "  /e        Write EXIF infos to EAs only (don't reaname pics)\n"
    #endif
    "  /h        Print this help screen, then abort [default]\n"
    "  File      File name, or template for file names (wildcards: * or ?)\n"
    "Examples:\n"
    "  exiftool /d 00000001.jpg > afile\n"
    "  exiftool *.jpg\n"
    "Note:\n"
    "  Run Exiftool before editing your photo files. Non EXIF aware graphics\n"
    "  software usually destroys the exposure info in EXIF files."
    );

   exit(RET_HELP);
}


/*************************************************************************
 *                                                                       *
 * storeIFDstring(destination,source)                                    *
 * store an ASCII value fron the IFD if the destination is still unset   *
 *                                                                       *
 *************************************************************************/
static void storeIFDstring(std::string& dst, const IFD::ValueList& src)
{  if (!dst.empty() || src.Count == 0)
      return;
   dst = trim(src.GetASCII());
}


/*************************************************************************
 *                                                                       *
 * string2multivalueEA(string)                                           *
 * convert a sting into a multi-value EA                                 *
 *                                                                       *
 *************************************************************************/
static std::string string2multivalueEA(const std::string& str)
{  std::string ret;
   ret.append("\xdf\xff\0\0\0\0", 6);
   int n = 0;
   std::string::size_type p = 0;
   while (p < str.size())
   {  std::string::size_type q = std::min(str.find('\n', p), str.size());
      std::string::size_type l = q - p;
      if (q && str[q-1] == '\r')
         --l;
      ret.append("\xfd\xff", 2);
      ret += (char)l; // lowbyte
      ret += (char)(l >> 8); // highbyte
      ret.append(str.data()+p, l);
      ++n;
      p = q+1;
   }
   // replace total number of items
   ret[4] = (char)n; // lowbyte
   ret[5] = (char)(n >> 8); // highbyte
   return ret;
}


/*************************************************************************
 *                                                                       *
 * main()                                                                *
 * Main procedure                                                        *
 *                                                                       *
 *************************************************************************/
int main (int argc, char* argv[])
{  try
   {  FILE *file = NULL;
      std::string filename;
      bool dump = false;
      #ifdef __OS2__
      bool eas = false;
      #endif
      bool ren = false;

      if( 1 == argc )
         help();
      for (int i = 1; i < argc; i += 1)
      {  char* op = argv[i];
         if (op[0] == '/' || op[0] == '-')
         {  if (stricmp("d", op+1) == 0)
               dump = true;
            #ifdef __OS2__
             else if (stricmp("e", op+1) == 0)
               eas = true;
            #endif
             else if (stricmp("h", op+1) == 0)
               help();
             else
               throw stringf("Invalid option %s", op);
      }  }

      /* Expand wildcards in command line arguments */
      _wildcard( &argc, &argv );

      /* default opeartion */
      if (!dump && !eas && !ren)
         eas = ren = true;

      /* Iterate through all command line arguments (assumed to be file names) */
      for(int i = 1; i < argc; i += 1 )
      {  filename = argv[i];
         if (filename[0] == '/' || filename[0] == '-')
            continue;
         try
         {  /* Rename the variable that holds the file name */
            std::cout << "File: " << filename << std::endl;

            /* Open file for reading */
            file = fopen(filename.c_str(), "rb" );
            if (NULL == file)
               throw stringf("Unable to open file %s", filename.c_str());

            static char buffer[70000];
            memset(buffer, 0, sizeof buffer);
            fread(buffer, 1, sizeof buffer, file);
            fclose(file);

            /* Check format and call appropriate function to process file */
            std::auto_ptr<Exif> exif(new Exif(buffer, buffer + sizeof buffer));

            exif->Parse();

            if (dump)
               std::cout << *exif;
            #ifdef __OS2__
            if (eas)
            {  // write exif info into .COMMENT EA
               std::ostringstream oss;
               oss << *exif;
               oss.str(string2multivalueEA(oss.str()));
               _ea EA = {0, oss.str().size(), (void*)oss.str().data()};
               if (_ea_put(&EA, filename.c_str(), 0, ".COMMENTS"))
                  throw stringf("*Error* Failed to write .COMMENTS EA of %s to %s: %s\n", filename.c_str(), EA.value, strerror(errno));
            }
            #endif
            if (ren)
            {  // fetch timestamp
               std::string datetime;
               storeIFDstring(datetime, (*exif)[Exif::TIFF_DateTime]);
               storeIFDstring(datetime, (*exif)[Exif::EXIF_DateTimeOriginal]);
               storeIFDstring(datetime, (*exif)[Exif::EXIF_DateTimeDigitized]);
               // create new filename
               replace(datetime.begin(), datetime.end(), ':', '-');
               replace(datetime.begin(), datetime.end(), ' ', '_');
               // keep extension
               std::string::size_type p = filename.find_last_of(".:\\/");
               if (p != std::string::npos && filename[p] == '.')
                  datetime.append(filename, p, filename.size() - p);
               // rename file
               if (!datetime.empty())
               {  if (rename(filename.c_str(), datetime.c_str()))
                     throw stringf("*Error* Failed to rename %s to %s: %s\n", filename.c_str(), datetime.c_str(), strerror(errno));
                  #ifdef __OS2__
                  // purge .LONGNAME EA
                  _ea_remove(datetime.c_str(), 0, ".LONGNAME");
                  #endif
               }
            }
         } catch (const std::string& err)
         {  fprintf(stderr, "*Error* %s File skipped.\n", err.c_str());
         }

         if (file != NULL)
            fclose(file);

      } // end for all files
   } catch (const std::string& err)
   {  fprintf(stderr, "*Error* %s.\n", err.c_str());
   } catch (const std::exception& ex)
   {  fprintf(stderr, "unhandled runtime exception: %s\n", ex.what());
      exit(-1);
   } catch (...)
   {  fprintf(stderr, "unhandled exception.\n");
      exit(-1);
   }
   return 0;
}



