/*
Konami Twin16 (preliminary)

Main processors are a pair of 68000 CPUs
Sounds are generated by a Z80, a Yamaha 2151 and 3012, a Konami custom IC and a UPD7759C

GX687	Dark Adventure / Devil World
GX785	Vulcan Venture / Gradius II
GX903	Cuebrick
GX870	Final Round

Known Issues:
	sprite system: zoom and shadows?
	Final Round: some sprite flicker (vblank issue?)
	video driver has great potential for speedup

	Vulcan Venture: no collisions(!)
	Devils World locks up (related problem to Vulcan Venture?)

	Cuebrick sound is messed up - it plays only a series of loud beeps.
	Note that it does this even if no sound commands are sent to it.
	Could it use a different sound chip?

68000 Memory Map for Konami Twin System

	CPUA										CPUB
	0x000000..0x03ffff	ROM						0x000000..0x03ffff
	0x040000..0x043fff	shared RAM				0x040000..0x043fff
	0x060000..0x063fff	working RAM				0x060000..0x063fff
	0x080000..0x080fff	palette
	0x080000..0x09ffff	ROM (extra tile data)
	0x0a0000..0x0a0001	IRQ control				0x0a0000..0x0a0001
	0x0a0008..0x0a0009	sound command
	0x0a0010..0xa00011	watchdog
	0x0c0000..0x0c0001	screenflip (+?)
	0x0c0002..0x0c000f	scroll registers

	0x100000..0x103fff	FIXRAM (text layer)
	0x120000..0x123fff	VIDRAM (tilemaps)		0x480000..0x483fff
	0x140000..0x143fff	OBJRAM (sprites)		0x400000..0x403fff
						ZIP RAM	(tiles)			0x500000..0x53ffff
						gfx ROM (banked)		0x600000..0x77ffff
						sprite gfx RAM			0x780000..0x79ffff
*/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "cpu/m68000/m68000.h"
#include "cpu/z80/z80.h"

static UINT8 twin16_soundlatch;
static UINT16 twin16_video_register[8];

static UINT16 twin16_CPUA_register, twin16_CPUB_register;
#define CPUA_IRQ_ENABLE (twin16_CPUA_register&0x20)
#define CPUB_IRQ_ENABLE (twin16_CPUB_register&0x02)

static UINT16 fround_gfx_bank;
static UINT16 twin16_sound_command;

static UINT16 *twin16_gfx_rom;
static unsigned char *sprite_gfx_ram;
static unsigned char *tile_gfx_ram;

static unsigned char *twin16_fixram; /* text layer */
static unsigned char *twin16_videoram; /* two 64x64 tilemaps */
static unsigned char *twin16_spriteram; /* two banks of 0x200 sprites */

static unsigned char *twin16_battery_backed_ram;
/* battery_backed_ram is used by CueBrick.  we aren't initializing it or saving it yet */

static int konami_twin_gfx_type;
/* in Vulcan Venture and Dark Adventure, video is arranged differently */

#define CREDITS "Phil Stroffolino"

enum {
	MEM_CPU_SOUND = 0,
	MEM_SOUND_SAMPLES1,
	MEM_SOUND_SAMPLES2,
	MEM_GFX_CHARS,
	MEM_GFX_SPRITES,
	MEM_CPU_A,
	MEM_CPU_B,
	MEM_GFX_TILES
};

/******************************************************************************************/

void fround_gfx_bank_w( int offset, int data ){
	fround_gfx_bank = COMBINE_WORD(fround_gfx_bank,data);
}

void twin16_video_register_w( int offset, int data ){
	COMBINE_WORD_MEM( &twin16_video_register[offset/2], data );
}

#define WORKRAM_CPUA_r	MRA_BANK1
#define WORKRAM_CPUA_w	MWA_BANK1

#define WORKRAM_CPUB_r	MRA_BANK2
#define WORKRAM_CPUB_w	MWA_BANK2

#define COMRAM_r		MRA_BANK3
#define COMRAM_w		MWA_BANK3

#define FIXRAM_CPUA_w	MWA_BANK4
#define FIXRAM_CPUA_r	MRA_BANK4, &twin16_fixram

#define VIDRAM_CPUA_w	MWA_BANK5
#define VIDRAM_CPUA_r	MRA_BANK5, &twin16_videoram

#define OBJRAM_CPUA_w	MWA_BANK6
#define OBJRAM_CPUA_r	MRA_BANK6, &twin16_spriteram

static int battery_backed_ram_r( int offset ){
	return READ_WORD(&twin16_battery_backed_ram[offset]);
}
static void battery_backed_ram_w( int offset, int data ){
	COMBINE_WORD_MEM(&twin16_battery_backed_ram[offset],data);
}

static int sprite_gfx_ram_r( int offset ){
	return READ_WORD(&sprite_gfx_ram[offset]);
}
static void sprite_gfx_ram_w( int offset, int data ){
	COMBINE_WORD_MEM(&sprite_gfx_ram[offset],data);
}

static int tile_gfx_ram_r( int offset ){
	return READ_WORD(&tile_gfx_ram[offset]);
}
static void tile_gfx_ram_w( int offset, int data ){
	COMBINE_WORD_MEM(&tile_gfx_ram[offset],data);
}

// VIDRAM, as seen by CPUB.  Not sure if correct.
int VIDRAM_CPUB_r( int offset ){
	offset ^= 0x2000;
	return READ_WORD(&twin16_videoram[offset]);
}

void VIDRAM_CPUB_w( int offset, int data ){
	offset ^= 0x2000;
	COMBINE_WORD_MEM(&twin16_videoram[offset],data);
}

// OBJRAM, as seen by CPUB.  Not sure if correct.
int OBJRAM_CPUB_r( int offset ){
	offset ^= 0x2000;
	return READ_WORD(&twin16_spriteram[offset]);
}

void OBJRAM_CPUB_w( int offset, int data ){
	offset ^= 0x2000;
	COMBINE_WORD_MEM(&twin16_spriteram[offset],data);
}

/******************************************************************************************/

struct osd_bitmap *bg_bitmap;

static int tile_rom_r( int offset ){
	return ((UINT16 *)Machine->memory_region[MEM_GFX_TILES])[offset/2];
}

static int twin16_gfx_rom1_r( int offset ){
	return twin16_gfx_rom[offset/2];
}

static int twin16_gfx_rom2_r( int offset ){
	return twin16_gfx_rom[offset/2 + 0x80000 + ((twin16_CPUB_register&0x04)?0x40000:0)];
}

static void twin16_paletteram_w(int offset,int data) // tmnt_paletteram_w
{
	int oldword = READ_WORD(&paletteram[offset]);
	int newword = COMBINE_WORD(oldword,data);
	WRITE_WORD(&paletteram[offset],newword);

	offset /= 4;
	{
		int palette = ((READ_WORD(&paletteram[offset * 4]) & 0x00ff) << 8)
				+ (READ_WORD(&paletteram[offset * 4 + 2]) & 0x00ff);
		int r = palette & 31;
		int g = (palette >> 5) & 31;
		int b = (palette >> 10) & 31;

		r = (r << 3) + (r >> 2);
		g = (g << 3) + (g >> 2);
		b = (b << 3) + (b >> 2);

		palette_change_color (offset,r,g,b);
	}
}

static int vh_start( void ){
	bg_bitmap = osd_new_bitmap( 64*8, 64*8, Machine->scrbitmap->depth );
	return bg_bitmap?0:1;
}

static void vh_stop( void ){
	osd_free_bitmap( bg_bitmap );
}

#define TRANSPARENT_COLOR 0x0000

static void draw_layer(
	struct osd_bitmap *bitmap,
	const UINT16 *source,
	const UINT16 *gfx_base,
	int scrollx, int scrolly,
	const int *bank_table,
	int palette,
	int opaque
){
	int i;
	for( i=0; i<64*64; i++ ){
		int sx = (i%64)*8;
		int sy = (i/64)*8;
		if( konami_twin_gfx_type ) sy = 63*8 - sy;

		{
			int code = source[i];
			const UINT16 *gfx_data = gfx_base + (code&0x7ff)*16 + bank_table[(code>>11)&0x3]*0x8000;
			int color = (code>>13);
			UINT16 *pal_data = Machine->pens + 16*(0x20+color+8*palette);

			{
				int y;
				UINT16 data;
				int pen;

				for( y=0; y<8; y++ ){
					UINT16 *dest = ((UINT16 *)bg_bitmap->line[sy+y])+sx;
					data = *gfx_data++;
					pen = (data>>4*3)&0xf; *dest++ = (opaque||pen)?pal_data[pen]:TRANSPARENT_COLOR;
					pen = (data>>4*2)&0xf; *dest++ = (opaque||pen)?pal_data[pen]:TRANSPARENT_COLOR;
					pen = (data>>4*1)&0xf; *dest++ = (opaque||pen)?pal_data[pen]:TRANSPARENT_COLOR;
					pen = (data>>4*0)&0xf; *dest++ = (opaque||pen)?pal_data[pen]:TRANSPARENT_COLOR;
					data = *gfx_data++;
					pen = (data>>4*3)&0xf; *dest++ = (opaque||pen)?pal_data[pen]:TRANSPARENT_COLOR;
					pen = (data>>4*2)&0xf; *dest++ = (opaque||pen)?pal_data[pen]:TRANSPARENT_COLOR;
					pen = (data>>4*1)&0xf; *dest++ = (opaque||pen)?pal_data[pen]:TRANSPARENT_COLOR;
					pen = (data>>4*0)&0xf; *dest++ = (opaque||pen)?pal_data[pen]:TRANSPARENT_COLOR;
				}
			}
		}
	}

	copyscrollbitmap(bitmap,bg_bitmap,
			1,&scrollx,1,&scrolly,
			&Machine->drv->visible_area,
			opaque?TRANSPARENCY_NONE:TRANSPARENCY_PEN,TRANSPARENT_COLOR);
}

static void draw_text( struct osd_bitmap *bitmap ){
	const struct rectangle *clip = &Machine->drv->visible_area;
	const UINT16 *source = (UINT16 *)twin16_fixram;
	int i;


	for( i=0; i<64*64; i++ ){
		int code = source[i];

		int sx = (i%64)*8;
		int sy = (i/64)*8;

		if( konami_twin_gfx_type ) sy = 31*8 - sy;

		drawgfx( bitmap, Machine->gfx[0],
			code&0x1ff,(code>>9)&0xf,
			0,0,
			sx,sy,
			clip,TRANSPARENCY_PEN,0);
	}
}

static void draw_sprite( /* slow slow slow, but it's good for now */
	struct osd_bitmap *bitmap,
	const UINT16 *pen_data,
	const UINT16 *pal_data,
	int xpos, int ypos,
	int width, int height,
	int flipx, int flipy ){

	int x, y;

	if( xpos+width>=0 && xpos<320 && ypos+height>=0 && ypos<256 )

	for( y=0; y<height; y++ ){
		int sy = (flipy)?(ypos+height-1-y):(ypos+y);
		if( sy>=0 && sy<256 ){
			UINT16 *dest = (UINT16 *)bitmap->line[sy];
			for( x=0; x<width; x++ ){
				int sx = (flipx)?(xpos+width-1-x):(xpos+x);
				if( sx>=0 && sy>=0 && sx<320 && sy<256 ){
					UINT16 pen = pen_data[x/4];
					switch( x%4 ){
						case 0: pen = pen>>12; break;
						case 1: pen = (pen>>8)&0xf; break;
						case 2: pen = (pen>>4)&0xf; break;
						case 3: pen = pen&0xf; break;
					}
					if( pen ) dest[sx] = pal_data[pen];
				}
			}
		}
		pen_data += width/4;
	}
}

/*
 * Sprite Format (similar to K051937)
 * ----------------------------------
 *
 * Word | Bit(s)           | Use
 * -----+-fedcba9876543210-+----------------
 *   0  | x--------------- | active (show this sprite)
 *   0  | ------xxxxxxxxxx | priority order
 *   1  | -------x-------- | unknown
 *   2  | xxxxxx---------- | zoom?
 *   2  | ------x--------- | flipy
 *   2  | -------x-------- | flipx
 *   2  | --------xx------ | height
 *   2  | ----------xx---- | width
 *   2  | ------------xxxx | color
 *   3  | --xx------------ | bank
 *   3  | ----xxxxxxxxxxxx | code
 *   4  | xxxxxxxxxxxxxxxx | y position (high bits)
 *   5  | xxxxxxxx-------- | y position (low 8 bits)
 *   6  | xxxxxxxxxxxxxxxx | x position (high bits)
 *   7  | xxxxxxxx-------- | x position (low 8 bits)
 */

static void draw_sprites( struct osd_bitmap *bitmap, int xscroll, const int *bank_table, int which ){
	const UINT16 *sprite_ptr[0x200];
	int i;
	{
		const UINT16 *source = (UINT16 *)twin16_spriteram;
		if( which ) source += 0x1000;

		for( i=0; i<0x200; i++ ) sprite_ptr[i] = NULL;

		for( i=0; i<0x200; i++ ){
			if( (*source&0x8000) && *source!=0xffff ) sprite_ptr[*source&0x1ff] = source;
			source+=8;
		}
	}

	for( i=0; i<0x200; i++ ){
		const UINT16 *source = sprite_ptr[i];
		if( source ){
			UINT16 priority = source[0];
			if( priority&0x8000 && priority!=0xffff ){ /* enable */
				UINT16 attributes = source[2];
				int color = (attributes&0xf);
				const UINT16 *pal_data = Machine->pens+(color+0x10)*16;
				int height	= 16<<((attributes>>6)&0x3);
				int width	= 16<<((attributes>>4)&0x3);

				UINT16 tile_number = source[3];

				INT32 xpos = (0x10000*source[4])|source[5];
				INT32 ypos = (0x10000*source[6])|source[7];

				const UINT16 *pen_data;

				int flipy = attributes&0x0200;
				int flipx = attributes&0x0100;

				xpos = (xpos>>8)-xscroll;
				ypos = (ypos>>8);

				if( bank_table ){
					pen_data = twin16_gfx_rom + 0x80000;
				}
				else {
					switch( (tile_number>>12)&0x3 ){ /* bank select */
						case 0:
						default:
						pen_data = twin16_gfx_rom;
						break;

						case 1:
						pen_data = twin16_gfx_rom + 0x40000;
						break;

						case 2:
						pen_data = twin16_gfx_rom + 0x80000;
						if( twin16_CPUB_register&0x04 ) pen_data += 0x40000;
						break;

						case 3:
						pen_data = (UINT16 *)sprite_gfx_ram;
						break;
					}
					tile_number &= 0xfff;
				}

				pen_data += tile_number*0x40;

				if( konami_twin_gfx_type ){
					ypos = 256-ypos-height;
					flipy = !flipy;
				}
				else {
					xpos = xpos&0x1ff;
					ypos = ypos&0x1ff;
				}

				draw_sprite( bitmap, pen_data, pal_data, xpos, ypos, width, height, flipx, flipy );

/*				{
					UINT16 data = source[k];
					int i;
					for( i=0; i<4; i++ ){
						int sy = ypos;
						if( tile_number&0x2000 ) sy += i;
						drawgfx( bitmap, Machine->uifont,
							"0123456789abcdef"[(data>>12)&0xf],0,
							0,0,
							xpos+i*6,sy,
							&Machine->drv->visible_area,
							TRANSPARENCY_NONE,0);
						data = data<<4;
					}
				}
*/
			}
		}
	}
}

/*
	if( keyboard_pressed(KEYCODE_N) ){
		k = (k-1);
		while( keyboard_pressed(KEYCODE_N) ){}
	}

	if( keyboard_pressed(KEYCODE_M) ){
		k = (k+1);
		while( keyboard_pressed(KEYCODE_M) ){}
	}
	{
		int i,n;
		for( i=0; i<8; i++ ){
			for( n=0; n<4; n++ ){
				drawgfx( bitmap, Machine->uifont,
					"0123456789abcdef"[(twin16_video_register[i]>>(12-4*n))&0xf],
					0,
					0,0,
					n*6+8,i*8+16,
					0,TRANSPARENCY_NONE,0);
			}
		}
	}
*/

static void twin16_vh_screenrefresh( struct osd_bitmap *bitmap, int fullrefresh ){
	const int bank_table[4] = { 0,1,2,3 };
	int scrollx, scrolly;
	int palette;

	palette_recalc();

	{ /* background */
		if( konami_twin_gfx_type ){
			scrollx = twin16_video_register[3];
			scrolly = 256-twin16_video_register[4];
			palette = 0;
		}
		else {
			scrollx = twin16_video_register[5];
			scrolly = twin16_video_register[6];
			palette = 1;
		}

		draw_layer(
			bitmap,
			0x1000 + (UINT16 *)twin16_videoram,
			(UINT16 *)tile_gfx_ram,
			-scrollx,-scrolly,
			bank_table,
			palette,
			1 /* opaque */
		);
	}

	{ /* foreground */
		if( konami_twin_gfx_type ){
			scrollx = twin16_video_register[5];
			scrolly = 256-twin16_video_register[6];
		}
		else {
			scrollx = twin16_video_register[3];
			scrolly = twin16_video_register[4];
		}
		palette = CPUB_IRQ_ENABLE?0:1; /* ? */

		draw_layer(
			bitmap,
			0x0000 + (UINT16 *)twin16_videoram,
			(UINT16 *)tile_gfx_ram,
			-scrollx,-scrolly,
			bank_table,
			palette,
			0 /* opaque */
		);
	}

	draw_sprites( bitmap, twin16_video_register[1], NULL, 0 );
	draw_sprites( bitmap, twin16_video_register[1], NULL, 1 );

	draw_text( bitmap );
}

void fround_screenrefresh( struct osd_bitmap *bitmap, int fullrefresh ){
	int bank_table[4];
	bank_table[0] = (fround_gfx_bank>>(4*0))&0xf;
	bank_table[1] = (fround_gfx_bank>>(4*1))&0xf;
	bank_table[2] = (fround_gfx_bank>>(4*2))&0xf;
	bank_table[3] = (fround_gfx_bank>>(4*3))&0xf;

	palette_recalc();

	draw_layer(
		bitmap,
		0x1000 + (UINT16 *)twin16_videoram,
		twin16_gfx_rom,
		-twin16_video_register[5],-twin16_video_register[6],
		bank_table,
		1, /* palette select */
		1 /* opaque */
	);

	draw_layer(
		bitmap,
		0x0000 + (UINT16 *)twin16_videoram,
		twin16_gfx_rom,
		-twin16_video_register[3],-twin16_video_register[4],
		bank_table,
		0, /* palette select */
		0 /* opaque */
	);

	draw_sprites( bitmap, 0, bank_table, 1 );
	draw_sprites( bitmap, 0, bank_table, 0 );

	draw_text( bitmap );
}

static struct GfxLayout alpha_layout = {
	8,8,
	0x200,
	4,
	{ 0,1,2,3 },
	{ 0*4,1*4,2*4,3*4,4*4,5*4,6*4,7*4 },
	{ 0*32,1*32,2*32,3*32,4*32,5*32,6*32,7*32 },
	8*32
};

static void gfx_decode( void ){
	int i;
	UINT16 *temp = (UINT16 *)malloc(0x200000);
	if( temp ){
		twin16_gfx_rom = (UINT16 *)Machine->memory_region[MEM_GFX_SPRITES];
		memcpy( temp, twin16_gfx_rom, 0x200000 );

		for( i=0; i<0x080000; i++ ){
			twin16_gfx_rom[i*2+0] = temp[i+0x080000];
			twin16_gfx_rom[i*2+1] = temp[i];
		}
		free( temp );
	}
}

static void cuebrick_decode( void ){
	gfx_decode();
	konami_twin_gfx_type = 0;
}

static void konami_twin16_decode( void ){
	gfx_decode();
	konami_twin_gfx_type = 1;
}

static struct GfxDecodeInfo gfxdecodeinfo[] = {
	{ MEM_GFX_CHARS, 0x00000, &alpha_layout, 0, 16 },
	{ -1 }
};


static int twin16_vblank_r( int offset ){ // probably vblank
	return cpu_getvblank();
}

/******************************************************************************************/

static int CPUA_interrupt( void ){
	return CPUA_IRQ_ENABLE? MC68000_IRQ_5:MC68000_INT_NONE;
}
static int CPUB_interrupt( void ){
	return CPUB_IRQ_ENABLE? MC68000_IRQ_5:MC68000_INT_NONE;
}

static void sound_command_w( int offset, int data ){
	twin16_sound_command = COMBINE_WORD( twin16_sound_command, data );
	soundlatch_w( 0, twin16_sound_command&0xff );
}

static void twin16_CPUA_register_w( int offset, int data ){
	/*
		7	6	5	4	3	2	1	0
			?							unknown (reset trigger?)
				X						IRQ5 enable (CPUA)
					X					trigger IRQ6 on CPUB
						X				trigger IRQ on sound CPU
	*/
	UINT16 old = twin16_CPUA_register;
	twin16_CPUA_register = COMBINE_WORD( old, data );
	if( twin16_CPUA_register!=old ){
		if( (old&0x08)==0 && (twin16_CPUA_register&0x08) )
			cpu_cause_interrupt( 0, 0xff ); // trigger IRQ on sound CPU

		if( (old&0x10)==0 && (twin16_CPUA_register&0x10) )
			cpu_cause_interrupt( 2, MC68000_IRQ_6 ); // trigger IRQ on CPUB
	}
}

static void twin16_CPUB_register_w( int offset, int data ){
	/*
		7	6	5	4	3	2	1	0
							X			gfx bank select
								X		CPUB IRQ5 enable (also palette bank??)
									X	trigger IRQ6 on CPUA
	*/
	UINT16 old = twin16_CPUB_register;
	twin16_CPUB_register = COMBINE_WORD( old, data );
	if( twin16_CPUB_register!=old ){
		if( (old&0x01)==0 && (twin16_CPUB_register&0x1) )
			cpu_cause_interrupt( 1, MC68000_IRQ_6 ); // trigger IRQ on CPUA
	}
}

static void fround_CPU_register_w( int offset, int data ){
	UINT16 old = twin16_CPUA_register;
	twin16_CPUA_register = COMBINE_WORD( old, data );
	if( twin16_CPUA_register!=old ){
		if( (old&0x08)==0 && (twin16_CPUA_register&0x08) )
			cpu_cause_interrupt( 0, 0xff ); // trigger IRQ on sound CPU
	}
}

/******************************************************************************************/

static int twin16_input_r (int offset){
	switch( offset ){
		case 0x00: return readinputport(0); // coin
		case 0x02: return readinputport(1); // p1
		case 0x04: return readinputport(2); // p2
		case 0x06: return readinputport(3); // p3 (Devils World)
		case 0x10: return readinputport(5); // DSW1
		case 0x12: return readinputport(4); // DSW2
		case 0x18: return readinputport(6); // DSW3
	}
	if (errorlog) fprintf(errorlog,"CPU #0 PC %06x: warning - read unmapped memory address %06x\n",cpu_get_pc(),0x0a0000+offset);
	return 0;
}

/******************************************************************************************/
/* sound system */

int twin16_sres_r(int offset){
	return twin16_soundlatch;
}

void twin16_sres_w(int offset,int data){
	/* bit 1 resets the UPD7795C sound chip */
	if ((data & 0x02) == 0){
		UPD7759_reset_w(0,(data & 0x02) >> 1);
	}
	twin16_soundlatch = data;
}

static struct YM2151interface ym2151_interface = {
	1,			/* 1 chip */
	7159160/2,	/* 3.58 MHZ ? */
	{ YM3012_VOL(50,MIXER_PAN_LEFT,50,MIXER_PAN_RIGHT) },
	{ 0 }
};

static void volume_callback(int v)
{
	K007232_set_volume(0,0,(v >> 4) * 0x11,0);
	K007232_set_volume(0,1,0,(v & 0x0f) * 0x11);
}

static struct K007232_interface k007232_interface =
{
	1,		/* number of chips */
	{MEM_SOUND_SAMPLES1,MEM_SOUND_SAMPLES1}, /* memory regions */
	{ K007232_VOL(20,MIXER_PAN_CENTER,20,MIXER_PAN_CENTER) },	/* volume */
	{ volume_callback }	/* external port callback */
};

static struct UPD7759_interface upd7759_interface =
{
	1,		/* number of chips */
	UPD7759_STANDARD_CLOCK,
	{ 60 }, /* volume */
	{ MEM_SOUND_SAMPLES2 }, /* memory region */
	UPD7759_STANDALONE_MODE, /* chip mode */
	{0}
};

static struct MemoryReadAddress readmem_sound[] = {
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x8000, 0x8fff, MRA_RAM },
	{ 0x9000, 0x9000, twin16_sres_r },
	{ 0xa000, 0xa000, soundlatch_r },
	{ 0xb000, 0xb00d, K007232_read_port_0_r },
	{ 0xc001, 0xc001, YM2151_status_port_0_r },
	{ 0xf000, 0xf000, UPD7759_busy_r },
	{ -1 }
};

static struct MemoryWriteAddress writemem_sound[] = {
	{ 0x0000, 0x7fff, MWA_ROM },
	{ 0x8000, 0x8fff, MWA_RAM },
	{ 0x9000, 0x9000, twin16_sres_w },
	{ 0xb000, 0xb00d, K007232_write_port_0_w  },
	{ 0xc000, 0xc000, YM2151_register_port_0_w },
	{ 0xc001, 0xc001, YM2151_data_port_0_w },
	{ 0xd000, 0xd000, UPD7759_message_w },
	{ 0xe000, 0xe000, UPD7759_start_w },
	{ -1 }
};

/******************************************************************************************/

static struct MemoryReadAddress readmem[] = {
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x040000, 0x043fff, COMRAM_r },
	{ 0x060000, 0x063fff, WORKRAM_CPUA_r },
	{ 0x080000, 0x080fff, paletteram_word_r, &paletteram },
	{ 0x0a0000, 0x0a001b, twin16_input_r },
	{ 0x0b0000, 0x0b3fff, battery_backed_ram_r, &twin16_battery_backed_ram }, /* cuebrick only */
	{ 0x0c000e, 0x0c000f, twin16_vblank_r },
	{ 0x100000, 0x103fff, FIXRAM_CPUA_r },
	{ 0x120000, 0x123fff, VIDRAM_CPUA_r },
	{ 0x140000, 0x143fff, OBJRAM_CPUA_r },
	{ -1 }
};

static struct MemoryWriteAddress writemem[] = {
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x040000, 0x043fff, COMRAM_w },		/* 10G,10E */
	{ 0x060000, 0x063fff, WORKRAM_CPUA_w },	/* 4S,6S */
	{ 0x080000, 0x080fff, twin16_paletteram_w },	/* 1U */
	{ 0x0a0000, 0x0a0001, twin16_CPUA_register_w },
	{ 0x0a0008, 0x0a0009, sound_command_w },
	{ 0x0a0010, 0x0a0011, MWA_NOP },		/* watchdog */
	{ 0x0b0000, 0x0b3fff, battery_backed_ram_w }, /* cuebrick only */
	{ 0x0c0000, 0x0c000f, twin16_video_register_w },
	{ 0x100000, 0x103fff, FIXRAM_CPUA_w },	/* 3A,3B */
	{ 0x120000, 0x123fff, VIDRAM_CPUA_w },	/* 8A,8B */
	{ 0x140000, 0x143fff, OBJRAM_CPUA_w },	/* 3L,3M */
	{ -1 }
};

static struct MemoryReadAddress readmem_sub[] = {
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x040000, 0x043fff, COMRAM_r },
	{ 0x060000, 0x063fff, WORKRAM_CPUB_r },
	{ 0x080000, 0x09ffff, tile_rom_r }, /* dark adv only */
	{ 0x400000, 0x403fff, OBJRAM_CPUB_r },
	{ 0x480000, 0x483fff, VIDRAM_CPUB_r },
	{ 0x500000, 0x53ffff, tile_gfx_ram_r, &tile_gfx_ram },
	{ 0x600000, 0x6fffff, twin16_gfx_rom1_r },
	{ 0x700000, 0x77ffff, twin16_gfx_rom2_r },
	{ 0x780000, 0x79ffff, sprite_gfx_ram_r, &sprite_gfx_ram },
	{ -1 }
};

static struct MemoryWriteAddress writemem_sub[] = {
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x040000, 0x043fff, COMRAM_w },			/* 10G,10E */
	{ 0x060000, 0x063fff, WORKRAM_CPUB_w },		/* 8U,10U */
	{ 0x0a0000, 0x0a0001, twin16_CPUB_register_w },
	{ 0x400000, 0x403fff, OBJRAM_CPUB_w },		/* 3L,3M */
	{ 0x480000, 0x483fff, VIDRAM_CPUB_w },		/* 8A,8B */
	{ 0x500000, 0x53ffff, tile_gfx_ram_w },		/* ZIP RAM */
	{ 0x780000, 0x79ffff, sprite_gfx_ram_w },	/* 8L,8M */
	{ -1 }
};

/******************************************************************************************/

static struct MemoryReadAddress fround_readmem[] = {
	{ 0x000000, 0x03ffff, MRA_ROM },
	{ 0x040000, 0x043fff, COMRAM_r },
	{ 0x060000, 0x063fff, WORKRAM_CPUA_r },
	{ 0x080000, 0x080fff, paletteram_word_r, &paletteram },
	{ 0x0a0000, 0x0a001b, twin16_input_r },
	{ 0x0c000e, 0x0c000f, twin16_vblank_r },
	{ 0x100000, 0x103fff, FIXRAM_CPUA_r },
	{ 0x120000, 0x123fff, VIDRAM_CPUA_r },
	{ 0x140000, 0x143fff, OBJRAM_CPUA_r },
	{ 0x500000, 0x6fffff, twin16_gfx_rom1_r },
	{ -1 }
};

static struct MemoryWriteAddress fround_writemem[] = {
	{ 0x000000, 0x03ffff, MWA_ROM },
	{ 0x040000, 0x043fff, COMRAM_w },
	{ 0x060000, 0x063fff, WORKRAM_CPUA_w },
	{ 0x080000, 0x080fff, twin16_paletteram_w },
	{ 0x0a0000, 0x0a0001, fround_CPU_register_w },
	{ 0x0a0008, 0x0a0009, sound_command_w },
	{ 0x0a0010, 0x0a0011, MWA_NOP }, /* watchdog */
	{ 0x0c0000, 0x0c000f, twin16_video_register_w },
	{ 0x0e0000, 0x0e0001, fround_gfx_bank_w },
	{ 0x100000, 0x103fff, FIXRAM_CPUA_w },
	{ 0x120000, 0x123fff, VIDRAM_CPUA_w },
	{ 0x140000, 0x143fff, OBJRAM_CPUA_w },
	{ -1 }
};

/******************************************************************************************/

static struct MachineDriver machine_driver =
{
	{
		{
			CPU_Z80 | CPU_AUDIO_CPU,
			4000000, /* ? */
			MEM_CPU_SOUND,
			readmem_sound,writemem_sound,0,0,
			ignore_interrupt,1
		},
		{
			CPU_M68000,
			10000000, /* ? */
			MEM_CPU_A,
			readmem,writemem,0,0,
			CPUA_interrupt,1
		},
		{
			CPU_M68000,
			10000000, /* ? */
			MEM_CPU_B,
			readmem_sub,writemem_sub,0,0,
			CPUB_interrupt,1
		}
	},
	60, DEFAULT_REAL_60HZ_VBLANK_DURATION,
	100, /* CPU slices */
	0, /* init machine */

	/* video hardware */
	320, 256, { 0, 319, 8, 255-8 },
	gfxdecodeinfo,
	0x800,0x800,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_16BIT,
	0,
	vh_start,
	vh_stop,
	twin16_vh_screenrefresh,

	/* sound hardware */
	0,0,0,0,
	{
		{
			SOUND_YM2151,
			&ym2151_interface
		},
		{
			SOUND_K007232,
			&k007232_interface,
		},
		{
			SOUND_UPD7759,
			&upd7759_interface
		}
	}
};

static struct MachineDriver fround_machine_driver =
{
	{
		{
			CPU_Z80 | CPU_AUDIO_CPU,
			4000000, /* ? */
			MEM_CPU_SOUND,
			readmem_sound,writemem_sound,0,0,
			ignore_interrupt,1
		},
		{
			CPU_M68000,
			10000000, /* ? */
			MEM_CPU_A,
			fround_readmem,fround_writemem,0,0,
			CPUA_interrupt,1
		},
	},
	60, DEFAULT_REAL_60HZ_VBLANK_DURATION,
	100, /* CPU slices */
	0, /* init machine */

	/* video hardware */
	320, 256, { 0, 319, 0, 255 },
	gfxdecodeinfo,
	0x800,0x800,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE | VIDEO_SUPPORTS_16BIT,
	0,
	vh_start,
	vh_stop,
	fround_screenrefresh,

	/* sound hardware */
	0,0,0,0,
	{
		{
			SOUND_YM2151,
			&ym2151_interface
		},
		{
			SOUND_K007232,
			&k007232_interface,
		},
		{
			SOUND_UPD7759,
			&upd7759_interface
		}
	}
};

ROM_START( devilw_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "dw-m03.rom",	0x00000,  0x8000, 0x7201983c )
	ROM_REGION(0x20000) /* samples */
	ROM_LOAD(	"dw-ic5a.rom",	0x00000, 0x20000, 0xd4992dfb )
	ROM_REGION(0x20000) /* samples */
	ROM_LOAD(	"dw-ic7c.rom",	0x00000, 0x20000, 0xe5947501 )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD( "dw-m14.rom",	0x00000,  0x4000, 0xd7338557 ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE_SWAP(	"dw-10p.rom",	0x000000, 0x80000, 0x66cb3923 )
	ROM_LOAD_WIDE_SWAP(	"dw-10r.rom",	0x080000, 0x80000, 0xa1c7d0db )
	ROM_LOAD_WIDE_SWAP(	"dw-10l.rom",	0x100000, 0x80000, 0xeec8c5b2 )
	ROM_LOAD_WIDE_SWAP(	"dw-10m.rom",	0x180000, 0x80000, 0x746cf48b )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"dw-t05.rom",	0x00000, 0x10000, 0x8ab7dc61 )
	ROM_LOAD_ODD(	"dw-t04.rom",	0x00000, 0x10000, 0xc69924da )
	ROM_LOAD_EVEN(	"dw-t09.rom",   0x20000, 0x10000, 0xfae97de0 )
	ROM_LOAD_ODD(	"dw-t08.rom",   0x20000, 0x10000, 0x8c898d67 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"dw-r07.rom",	0x00000, 0x10000, 0x53110c0b )
	ROM_LOAD_ODD(	"dw-r06.rom",	0x00000, 0x10000, 0x9c53a0c5 )
	ROM_LOAD_EVEN(	"dw-r13.rom",	0x20000, 0x10000, 0x36ae6014 )
	ROM_LOAD_ODD(	"dw-r12.rom",	0x20000, 0x10000, 0x6d012167 )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD_EVEN(	"dw-m11.rom",	0x00000, 0x10000, 0x399deee8 )
	ROM_LOAD_ODD(	"dw-m10.rom",	0x00000, 0x10000, 0x117c91ee )
ROM_END

ROM_START( darkadv_rom ) // this set is incomplete
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "n03.10a",	0x00000,  0x8000, 0xa24c682f )
	ROM_REGION(0x20000) /* samples */
	ROM_LOAD(	"dw-ic5a.rom",	0x00000, 0x20000, 0 )
	ROM_REGION(0x20000) /* samples */
	ROM_LOAD(	"dw-ic7c.rom",	0x00000, 0x20000, 0 )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD( "n14.3f",	0x0000,  0x4000, 0xc76ac6d2 ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE_SWAP(	"dw-10p.rom",	0x000000, 0x80000, 0x66cb3923 )
	ROM_LOAD_WIDE_SWAP(	"dw-10r.rom",	0x080000, 0x80000, 0xa1c7d0db )
	ROM_LOAD_WIDE_SWAP(	"dw-10l.rom",	0x100000, 0x80000, 0xeec8c5b2 )
	ROM_LOAD_WIDE_SWAP(	"dw-10m.rom",	0x180000, 0x80000, 0x746cf48b )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"n05.6n",	0x00000, 0x10000, 0xa9195b0b )
	ROM_LOAD_ODD(	"n04.4n",	0x00000, 0x10000, 0x65b55105 )
	ROM_LOAD_EVEN(	"n09",		0x20000, 0x10000, 0 ) /* missing! */
	ROM_LOAD_ODD(   "n08.4r",   0x20000, 0x10000, 0xa9603196 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"n07.10n",	0x00000, 0x10000, 0x6154322a )
	ROM_LOAD_ODD(	"n06.8n",	0x00000, 0x10000, 0x37a72e8b )
	ROM_LOAD_EVEN(	"n13.10s",	0x20000, 0x10000, 0xf1c252af )
	ROM_LOAD_ODD(	"n12.8s",	0x20000, 0x10000, 0xda221944 )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD_EVEN(	"dw-m11.rom",	0x00000, 0x10000, 0x399deee8 )
	ROM_LOAD_ODD(	"dw-m10.rom",	0x00000, 0x10000, 0x117c91ee ) /* tiles */
ROM_END

ROM_START( cuebrick_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "903-d03.10a",	0x00000,  0x4000, 0x0a404411 )

	ROM_REGION( 0x20000 ) // missing?
	ROM_REGION( 0x20000 ) // missing?

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD(	"903-e14.d8",	0x0000, 0x4000, 0xddbebbd5 ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"903-e05.6n",	0x00000, 0x10000, 0x8b556220 )
	ROM_LOAD_ODD(	"903-e04.4n",	0x00000, 0x10000, 0xbf9c7927 )
	ROM_LOAD_EVEN(	"903-e09.6r",	0x20000, 0x10000, 0x2a77554d )
	ROM_LOAD_ODD(	"903-e08.4r",	0x20000, 0x10000, 0xc0a430c1 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"903-d07.10n",	0x00000, 0x10000, 0xfc0edce7 )
	ROM_LOAD_ODD(	"903-d06.8n",	0x00000, 0x10000, 0xb2cef6fe )
	ROM_LOAD_EVEN(	"903-e13.10s",	0x20000, 0x10000, 0x4fb5fb80 )
	ROM_LOAD_ODD(	"903-e12.8s",	0x20000, 0x10000, 0x883e3097 )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD_EVEN(	"903-e11.10r",	0x00000, 0x10000, 0x5c41faf8 )
	ROM_LOAD_ODD(	"903-e10.8r",	0x00000, 0x10000, 0x417576d4 )
ROM_END

ROM_START( gradius2_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "vulcan.g03",	0x00000,  0x8000, 0x67a3b50d )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"vulcan.f01",	0x00000, 0x20000, 0xa0d8d69e )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"vulcan.f02",	0x00000, 0x20000, 0xc39f5ca4 )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD(	"gradius2.g14",	0x0000, 0x4000, 0x9dcdad9d ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE(	"vulcan.f17",	0x000000, 0x80000, 0x8fbec1a4 )
	ROM_LOAD_WIDE(	"vulcan.f18",	0x080000, 0x80000, 0x50d61e38 )
	ROM_LOAD_WIDE(	"vulcan.f15",	0x100000, 0x80000, 0xaf96aef3 )
	ROM_LOAD_WIDE(	"vulcan.f16",	0x180000, 0x80000, 0xb858df1f )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"gradius2.p05", 0x00000, 0x10000, 0x4db0e736 )
	ROM_LOAD_ODD(	"gradius2.p04", 0x00000, 0x10000, 0x765b99e6 )
	ROM_LOAD_EVEN(	"gradius2.j09", 0x20000, 0x10000, 0x6d96a7e3 )
	ROM_LOAD_ODD(	"gradius2.j08", 0x20000, 0x10000, 0x2b1c9108 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"vulcan.p07", 0x00000, 0x10000, 0x686d549d )
	ROM_LOAD_ODD(	"vulcan.p06", 0x00000, 0x10000, 0x70c94bee )
	ROM_LOAD_EVEN(	"vulcan.p13", 0x20000, 0x10000, 0x478fdb0a )
	ROM_LOAD_ODD(	"vulcan.p12", 0x20000, 0x10000, 0x38ea402a )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
ROM_END

ROM_START( vulcan_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "vulcan.g03",	0x00000,  0x8000, 0x67a3b50d )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"vulcan.f01",	0x00000, 0x20000, 0xa0d8d69e )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"vulcan.f02",	0x00000, 0x20000, 0xc39f5ca4 )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD(	"vulcan.h14",	0x0000, 0x4000, 0x02f4b16f ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE(	"vulcan.f17",	0x000000, 0x80000, 0x8fbec1a4 )
	ROM_LOAD_WIDE(	"vulcan.f18",	0x080000, 0x80000, 0x50d61e38 )
	ROM_LOAD_WIDE(	"vulcan.f15",	0x100000, 0x80000, 0xaf96aef3 )
	ROM_LOAD_WIDE(	"vulcan.f16",	0x180000, 0x80000, 0xb858df1f )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"vulcan.w05", 0x00000, 0x10000, 0x6e0e99cd )
	ROM_LOAD_ODD(	"vulcan.w04", 0x00000, 0x10000, 0x23ec74ca )
	ROM_LOAD_EVEN(	"vulcan.w09", 0x20000, 0x10000, 0x377e4f28 )
	ROM_LOAD_ODD(	"vulcan.w08", 0x20000, 0x10000, 0x813d41ea )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU B) */
	ROM_LOAD_EVEN(	"vulcan.p07", 0x00000, 0x10000, 0x686d549d )
	ROM_LOAD_ODD(	"vulcan.p06", 0x00000, 0x10000, 0x70c94bee )
	ROM_LOAD_EVEN(	"vulcan.p13", 0x20000, 0x10000, 0x478fdb0a )
	ROM_LOAD_ODD(	"vulcan.p12", 0x20000, 0x10000, 0x38ea402a )

	ROM_REGION( 0x20000 ) /* tile data; mapped at 0x80000 on CPUB */
ROM_END

ROM_START( fround_rom )
	ROM_REGION( 0x10000 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "frf03.bin",	0x00000,  0x8000, 0xa645c727 )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"frc01.bin",	0x00000, 0x20000, 0x6af96546 )
	ROM_REGION( 0x20000 )
	ROM_LOAD(	"frc02.bin",	0x00000, 0x20000, 0x54e12c6d )

	ROM_REGION_DISPOSE( 0x4000 )
	ROM_LOAD(	"frp17.bin",	0x0000, 0x4000, 0xc9b46615 ) /* characters */

	ROM_REGION( 0x200000 ) /* gfx data */
	ROM_LOAD_WIDE_SWAP(	"frc15.bin",	0x180000, 0x80000, 0x8c9281df )
	ROM_LOAD_WIDE_SWAP(	"frc16.bin",	0x100000, 0x80000, 0x41df6a1b )
	ROM_LOAD_WIDE_SWAP(	"frc17.bin",	0x080000, 0x80000, 0x2bc99ff8 )
	ROM_LOAD_WIDE_SWAP(	"frc18.bin",	0x000000, 0x80000, 0x07927fe8 )

	ROM_REGION( 0x40000 ) /* 68000 code (CPU A) */
	ROM_LOAD_EVEN(	"frl21.bin", 0x00000, 0x20000, 0xe21a3a19 )
	ROM_LOAD_ODD(	"frl20.bin", 0x00000, 0x20000, 0x0ce9786f )
ROM_END

#define KONAMI_TWIN_COINAGE PORT_START \
	PORT_DIPNAME( 0x0f, 0x0f, DEF_STR( Coin_A ) ) \
	PORT_DIPSETTING(    0x00, "freeplay" ) \
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x05, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x04, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0x0f, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x03, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_5C ) ) \
	PORT_DIPSETTING(    0x0e, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0x0d, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_4C ) ) \
	PORT_DIPSETTING(    0x0b, DEF_STR( 1C_5C ) ) \
	PORT_DIPSETTING(    0x0a, DEF_STR( 1C_6C ) ) \
	PORT_DIPSETTING(    0x09, DEF_STR( 1C_7C ) ) \
	PORT_DIPNAME( 0xf0, 0xf0, DEF_STR( Coin_B ) ) \
	PORT_DIPSETTING(    0x20, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x50, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x80, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x40, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x10, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0xf0, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x30, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x70, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0x60, DEF_STR( 2C_5C ) ) \
	PORT_DIPSETTING(    0xe0, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0xd0, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0xc0, DEF_STR( 1C_4C ) ) \
	PORT_DIPSETTING(    0xb0, DEF_STR( 1C_5C ) ) \
	PORT_DIPSETTING(    0xa0, DEF_STR( 1C_6C ) ) \
	PORT_DIPSETTING(    0x90, DEF_STR( 1C_7C ) )

INPUT_PORTS_START( twin16_input_ports ) /* generic; currently used by fround and cuebrick */
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( fround_input_ports )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, 0xff, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x03, "Player's Basic Power" )
	PORT_DIPSETTING(    0x03, "18" )
	PORT_DIPSETTING(    0x02, "20" )
	PORT_DIPSETTING(    0x01, "22" )
	PORT_DIPSETTING(    0x00, "24" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x60, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( gradius2_input_ports )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, 0xff, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x18, "Bonus" )
	PORT_DIPSETTING(    0x00, "70K" )
	PORT_DIPSETTING(    0x08, "20K" )
	PORT_DIPSETTING(    0x10, "30K/80K" )
	PORT_DIPSETTING(    0x18, "20K/70K" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Controls" )
	PORT_DIPSETTING(    0x02, "Single" )
	PORT_DIPSETTING(    0x00, "Dual" )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( devilw_input_ports )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON3 ) /* "map" button (also advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 ) /* also used in-game */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 ) /* also used in-game */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START3 ) /* also used in-game */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Controls" )
	PORT_DIPSETTING(    0x02, "Single" )
	PORT_DIPSETTING(    0x00, "Dual" )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

struct GameDriver devilw_driver =  {
	__FILE__,
	0,
	"devilw",
	"Devil's World",
	"1987",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING,
	&machine_driver,
	0,
	devilw_rom,
	konami_twin16_decode,
	0,
	0,
	0,	/* sound_prom */

	devilw_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver darkadv_driver =  {
	__FILE__,
	&devilw_driver,
	"darkadv",
	"Dark Adventure",
	"1987",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING,
	&machine_driver,
	0,
	darkadv_rom,
	konami_twin16_decode,
	0,
	0,
	0,	/* sound_prom */

	devilw_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver vulcan_driver =  {
	__FILE__,
	0,
	"vulcan",
	"Vulcan Venture",
	"1988",
	"Konami",
	CREDITS,
	0,
	&machine_driver,
	0,
	vulcan_rom,
	konami_twin16_decode,
	0,
	0,
	0,	/* sound_prom */

	gradius2_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver gradius2_driver =  {
	__FILE__,
	&vulcan_driver,
	"gradius2",
	"Gradius II - Gofer no Yabou (Japan)",
	"1988",
	"Konami",
	CREDITS,
	0,
	&machine_driver,
	0,
	gradius2_rom,
	konami_twin16_decode,
	0,
	0,
	0,	/* sound_prom */

	gradius2_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver cuebrick_driver =  {
	__FILE__,
	0,
	"cuebrick",
	"Cuebrick",
	"1989",
	"Konami",
	CREDITS,
	GAME_NOT_WORKING | GAME_IMPERFECT_SOUND,
	&machine_driver,
	0,
	cuebrick_rom,
	cuebrick_decode,
	0,
	0,
	0,	/* sound_prom */

	twin16_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};

struct GameDriver fround_driver =  {
	__FILE__,
	0,
	"fround",
	"Final Round",
	"1988",
	"Konami",
	CREDITS,
	0,
	&fround_machine_driver,
	0,
	fround_rom,
	cuebrick_decode, 0,
	0,
	0,	/* sound_prom */

	fround_input_ports,

	0, 0, 0, /* colors, palette, colortable */
	ORIENTATION_DEFAULT,
	0,0
};
