/***************************************************************************

	Midway MCR-2 system

	Currently implemented:
		* Satan's Hollow
		* Tron
		* Kozmik Krooz'r
		* Domino Man
		* Wacko
		* Two Tigers
		* Journey

****************************************************************************

	Memory map

****************************************************************************

	========================================================================
	CPU #1
	========================================================================
	0000-BFFF   R     xxxxxxxx    Program ROM
	C000-C7FF   R/W   xxxxxxxx    NVRAM
	F000-F1FF   R/W   xxxxxxxx    Sprite RAM
	F800-FF7F   R/W   xxxxxxxx    Background video RAM
	FF80-FFFF     W   xxxxxxxx    Palette RAM
	========================================================================
	0000        R     x-xxxxxx    Input ports
	            R     x-------    Service switch (active low)
	            R     --x-----    Tilt
	            R     ---xxx--    External inputs
	            R     ------x-    Right coin
	            R     -------x    Left coin
	0000        W     xxxxxxxx    Data latch OP0 (coin meters, 2 led's and cocktail 'flip')
	0001        R     xxxxxxxx    External inputs
	0002        R     xxxxxxxx    External inputs
	0003        R     xxxxxxxx    DIP switches
	0004        R     xxxxxxxx    External inputs
	0004        W     xxxxxxxx    Data latch OP4 (comm. with external hardware)
	0007        R     xxxxxxxx    Audio status
	001C-001F   W     xxxxxxxx    Audio latches 1-4
	00E0        W     --------    Watchdog reset
	00E8        W     xxxxxxxx    Unknown (written at initialization time)
	00F0-00F3   W     xxxxxxxx    CTC communications
	========================================================================
	Interrupts:
		NMI ???
		INT generated by CTC
	========================================================================


	========================================================================
	CPU #2 (Super Sound I/O)
	========================================================================
	0000-3FFF   R     xxxxxxxx    Program ROM
	8000-83FF   R/W   xxxxxxxx    Program RAM
	9000-9003   R     xxxxxxxx    Audio latches 1-4
	A000          W   xxxxxxxx    AY-8910 #1 control
	A001        R     xxxxxxxx    AY-8910 #1 status
	A002          W   xxxxxxxx    AY-8910 #1 data
	B000          W   xxxxxxxx    AY-8910 #2 control
	B001        R     xxxxxxxx    AY-8910 #2 status
	B002          W   xxxxxxxx    AY-8910 #2 data
	C000          W   xxxxxxxx    Audio status
	E000          W   xxxxxxxx    Unknown
	F000        R     xxxxxxxx    Audio board switches
	========================================================================
	Interrupts:
		NMI ???
		INT generated by external circuitry 780 times/second
	========================================================================

***************************************************************************/


#include "driver.h"
#include "machine/mcr.h"
#include "machine/z80fmly.h"
#include "sndhrdw/mcr.h"
#include "vidhrdw/generic.h"


/* external video code and data */
extern UINT8 mcr2_sprite_color;
extern UINT16 mcr3_sprite_code_mask;

void mcr2_videoram_w(int offset, int data);
void mcr2_paletteram_w(int offset, int data);
void mcr2_vh_screenrefresh(struct osd_bitmap *bitmap, int full_refresh);

void journey_vh_screenrefresh(struct osd_bitmap *bitmap, int full_refresh);



/*************************************
 *
 *	Kozmik Krooz'r input ports
 *
 *************************************/

static int kroozr_dial_r(int offset)
{
	int dial = readinputport(7);
	int val = readinputport(1);

	val |= (dial & 0x80) >> 1;
	val |= (dial & 0x70) >> 4;

	return val;
}


static int kroozr_trakball_x_r(int data)
{
	int val = readinputport(6);

	if (val & 0x02)		/* left */
		return 0x64 - 0x34;
	if (val & 0x01)		/* right */
		return 0x64 + 0x34;
	return 0x64;
}


static int kroozr_trakball_y_r(int data)
{
	int val = readinputport(6);

	if (val & 0x08)		/* up */
		return 0x64 - 0x34;
	if (val & 0x04)		/* down */
		return 0x64 + 0x34;
	return 0x64;
}



/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

static struct MemoryReadAddress readmem[] =
{
	{ 0x0000, 0xbfff, MRA_ROM },
	{ 0xc000, 0xc7ff, MRA_RAM },
	{ 0xf000, 0xf1ff, MRA_RAM },
	{ 0xf800, 0xff7f, MRA_RAM },
	{ -1 }  /* end of table */
};


static struct MemoryWriteAddress writemem[] =
{
	{ 0x0000, 0xbfff, MWA_ROM },
	{ 0xc000, 0xc7ff, MWA_RAM },
	{ 0xf000, 0xf1ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0xf800, 0xff7f, mcr2_videoram_w, &videoram, &videoram_size },
	{ 0xff80, 0xffff, mcr2_paletteram_w, &paletteram },
	{ -1 }  /* end of table */
};


static struct IOReadPort readport[] =
{
	{ 0x00, 0x04, mcr_port_04_dispatch_r },
	{ 0x07, 0x07, mcr_sound_status_r },
	{ 0x10, 0x10, mcr_port_04_dispatch_r },
	{ 0xf0, 0xf3, z80ctc_0_r },
	{ -1 }
};


static struct IOWritePort writeport[] =
{
	{ 0x00, 0x01, mcr_port_01_w },
	{ 0x04, 0x07, mcr_port_47_dispatch_w },
	{ 0x1c, 0x1f, ssio_data_w },
	{ 0xe0, 0xe0, watchdog_reset_w },
	{ 0xe8, 0xe8, mcr_unknown_w },
	{ 0xf0, 0xf3, z80ctc_0_w },
	{ -1 }	/* end of table */
};



/*************************************
 *
 *	Port definitions
 *
 *************************************/

INPUT_PORTS_START( shollow_input_ports )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_2WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )

	PORT_START	/* IN2 unused */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN3 -- dipswitches */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Cocktail ) )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* IN4 unused */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* AIN0 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( tron_input_ports )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* IN1 -- controls spinner */
	PORT_ANALOGX( 0xff, 0x00, IPT_DIAL | IPF_REVERSE, 50, 10, 0, 0, 0, KEYCODE_Z, KEYCODE_X, 0, 0 )

	PORT_START	/* IN2 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_COCKTAIL )

	PORT_START	/* IN3 -- dipswitches */
	PORT_DIPNAME( 0x01, 0x00, "Coin Meters" )
	PORT_DIPSETTING(    0x01, "1" )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x04, 0x00, "Allow Continue" )
	PORT_DIPSETTING(    0x04, DEF_STR( No ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Yes ) )
	PORT_BIT( 0xf8, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN4 */
	PORT_ANALOG( 0xff, 0x00, IPT_DIAL | IPF_REVERSE | IPF_COCKTAIL, 50, 10, 0, 0, 0 )

	PORT_START	/* AIN0 */
	PORT_BIT( 0x0f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0xe0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( kroozr_input_ports )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* IN1 -- controls firing spinner */
	PORT_BIT( 0x07, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 )

	PORT_START	/* IN2 -- controls joystick x-axis */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START	/* IN3 -- dipswitches */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Cocktail ) )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* IN4 -- controls joystick y-axis */
	PORT_BIT( 0xff, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START	/* AIN0 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* dummy extra port for keyboard movement */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START	/* dummy extra port for dial control */
	PORT_ANALOGX( 0xff, 0x00, IPT_DIAL | IPF_REVERSE, 40, 10, 0, 0, 0, KEYCODE_Z, KEYCODE_X, 0, 0 )
INPUT_PORTS_END


INPUT_PORTS_START( domino_input_ports )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY )

	PORT_START	/* IN2 unused */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )

	PORT_START	/* IN3 -- dipswitches */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x02, 0x02, "Skin Color" )
	PORT_DIPSETTING(    0x02, "Light" )
	PORT_DIPSETTING(    0x00, "Dark" )
	PORT_BIT( 0x3c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x80, 0x00, "Coin Meters" )
	PORT_DIPSETTING(    0x80, "1" )
	PORT_DIPSETTING(    0x00, "2" )

	PORT_START	/* IN4 unused */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* AIN0 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( wacko_input_ports )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* IN1 -- controls joystick x-axis */
	PORT_ANALOG ( 0xff, 0x00, IPT_TRACKBALL_X, 50, 10, 0, 0, 0 )

	PORT_START	/* IN2 -- controls joystick y-axis */
	PORT_ANALOG ( 0xff, 0x00, IPT_TRACKBALL_Y | IPF_REVERSE, 50, 10, 0, 0, 0 )

	PORT_START	/* IN3 -- dipswitches */
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x80, 0x00, "Coin Meters" )
	PORT_DIPSETTING(    0x80, "1" )
	PORT_DIPSETTING(    0x00, "2" )

	PORT_START	/* IN4 -- 4-way firing joystick */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP | IPF_4WAY )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* AIN0 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( twotiger_input_ports )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BITX(0x10, IP_ACTIVE_LOW, IPT_START1, "Dogfight Start", KEYCODE_6, IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	/* IN1 -- player 1 spinner */
	PORT_ANALOG( 0xff, 0x00, IPT_DIAL | IPF_REVERSE, 10, 10, 0, 0, 0 )

	PORT_START	/* IN2 -- buttons for player 1 & player 2 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN3 -- dipswitches */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN4 -- player 2 spinner */
	PORT_ANALOG( 0xff, 0x00, IPT_DIAL | IPF_REVERSE | IPF_PLAYER2, 10, 10, 0, 0, 0 )

	PORT_START	/* AIN0 */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END



/*************************************
 *
 *	Graphics definitions
 *
 *************************************/

MCR_CHAR_LAYOUT(charlayout, 512);
MCR_SPRITE_LAYOUT(spritelayout, 128);

static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,    0, 4 },	/* colors 0-63 */
	{ 1, 0x4000, &spritelayout,  0, 4 },	/* colors 0-63 */
	{ -1 } /* end of array */
};



/*************************************
 *
 *	Machine driver
 *
 *************************************/

static struct MachineDriver machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_Z80,
			2500000,	/* 2.5 Mhz */
			0,
			readmem,writemem,readport,writeport,
			mcr_interrupt,1,
			0,0,mcr_daisy_chain
		},
		SOUND_CPU_SSIO(2)
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	mcr_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	gfxdecodeinfo,
	64,64,
	0,

	VIDEO_TYPE_RASTER | VIDEO_SUPPORTS_DIRTY | VIDEO_MODIFIES_PALETTE | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	mcr2_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_SSIO
	}
};


static struct MachineDriver journey_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_Z80,
			7500000,	/* Looks like it runs at 7.5 Mhz rather than 5 or 2.5 */
			0,
			readmem,writemem,readport,writeport,
			mcr_interrupt,1,
			0,0,mcr_daisy_chain
		},
		SOUND_CPU_SSIO(2)
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	mcr_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	gfxdecodeinfo,
	64,64,
	0,

	VIDEO_TYPE_RASTER | VIDEO_SUPPORTS_DIRTY | VIDEO_MODIFIES_PALETTE | VIDEO_UPDATE_BEFORE_VBLANK,
	0,
	generic_vh_start,
	generic_vh_stop,
	journey_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_SSIO
	}
};



/*************************************
 *
 *	Driver initialization
 *
 *************************************/

static void shollow_init(void)
{
	MCR_CONFIGURE_HISCORE(0xc000, 0x800, NULL);
	MCR_CONFIGURE_SOUND(MCR_SSIO);
	MCR_CONFIGURE_DEFAULT_PORTS;

	mcr2_sprite_color = 1;
}


static void tron_init(void)
{
	MCR_CONFIGURE_HISCORE(0xc000, 0x800, NULL);
	MCR_CONFIGURE_SOUND(MCR_SSIO);
	MCR_CONFIGURE_DEFAULT_PORTS;

	mcr2_sprite_color = 1;
}


static void kroozr_init(void)
{
	MCR_CONFIGURE_HISCORE(0xc000, 0x800, NULL);
	MCR_CONFIGURE_SOUND(MCR_SSIO);
	MCR_CONFIGURE_PORT_04_READS(NULL, kroozr_dial_r, kroozr_trakball_x_r, NULL, kroozr_trakball_y_r);
	MCR_CONFIGURE_PORT_47_WRITES(NULL, NULL, NULL, NULL);

	mcr2_sprite_color = 1;
}


static void domino_init(void)
{
	MCR_CONFIGURE_HISCORE(0xc000, 0x800, NULL);
	MCR_CONFIGURE_SOUND(MCR_SSIO);
	MCR_CONFIGURE_DEFAULT_PORTS;

	mcr2_sprite_color = 1;
}


static void wacko_init(void)
{
	MCR_CONFIGURE_HISCORE(0xc000, 0x800, NULL);
	MCR_CONFIGURE_SOUND(MCR_SSIO);
	MCR_CONFIGURE_DEFAULT_PORTS;

	mcr2_sprite_color = 0;
}


static void twotiger_init(void)
{
	MCR_CONFIGURE_HISCORE(0xc000, 0x800, NULL);
	MCR_CONFIGURE_SOUND(MCR_SSIO);
	MCR_CONFIGURE_DEFAULT_PORTS;

	mcr2_sprite_color = 1;
}


static void journey_init(void)
{
	MCR_CONFIGURE_HISCORE(0xc000, 0x800, NULL);
	MCR_CONFIGURE_SOUND(MCR_SSIO);
	MCR_CONFIGURE_DEFAULT_PORTS;

	mcr3_sprite_code_mask = 0x07f;
}



/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( shollow_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "sh-pro.00",    0x0000, 0x2000, 0x95e2b800 )
	ROM_LOAD( "sh-pro.01",    0x2000, 0x2000, 0xb99f6ff8 )
	ROM_LOAD( "sh-pro.02",    0x4000, 0x2000, 0x1202c7b2 )
	ROM_LOAD( "sh-pro.03",    0x6000, 0x2000, 0x0a64afb9 )
	ROM_LOAD( "sh-pro.04",    0x8000, 0x2000, 0x22fa9175 )
	ROM_LOAD( "sh-pro.05",    0xa000, 0x2000, 0x1716e2bb )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "sh-bg.00",     0x00000, 0x2000, 0x3e2b333c )
	ROM_LOAD( "sh-bg.01",     0x02000, 0x2000, 0xd1d70cc4 )
	ROM_LOAD( "sh-fg.00",     0x04000, 0x2000, 0x33f4554e )
	ROM_LOAD( "sh-fg.01",     0x08000, 0x2000, 0xba1a38b4 )
	ROM_LOAD( "sh-fg.02",     0x0c000, 0x2000, 0x6b57f6da )
	ROM_LOAD( "sh-fg.03",     0x10000, 0x2000, 0x37ea9d07 )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "sh-snd.01",    0x0000, 0x1000, 0x55a297cc )
	ROM_LOAD( "sh-snd.02",    0x1000, 0x1000, 0x46fc31f6 )
	ROM_LOAD( "sh-snd.03",    0x2000, 0x1000, 0xb1f4a6a8 )
ROM_END


ROM_START( shollow2_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "sh-pro.00",    0x0000, 0x2000, 0x95e2b800 )
	ROM_LOAD( "sh-pro.01",    0x2000, 0x2000, 0xb99f6ff8 )
	ROM_LOAD( "sh-pro.02",    0x4000, 0x2000, 0x1202c7b2 )
	ROM_LOAD( "sh-pro.03",    0x6000, 0x2000, 0x0a64afb9 )
	ROM_LOAD( "sh-pro.04",    0x8000, 0x2000, 0x22fa9175 )
	ROM_LOAD( "sh-pro.05",    0xa000, 0x2000, 0x1716e2bb )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "sh-bg.00",     0x00000, 0x2000, 0x3e2b333c )
	ROM_LOAD( "sh-bg.01",     0x02000, 0x2000, 0xd1d70cc4 )
	ROM_LOAD( "sh-fg.00",     0x04000, 0x2000, 0x33f4554e )
	ROM_LOAD( "sh-fg.01",     0x08000, 0x2000, 0xba1a38b4 )
	ROM_LOAD( "sh-fg.02",     0x0c000, 0x2000, 0x6b57f6da )
	ROM_LOAD( "sh-fg.03",     0x10000, 0x2000, 0x37ea9d07 )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "snd-0.a7",     0x0000, 0x1000, 0x9d815bb3 )
	ROM_LOAD( "snd-1.a8",     0x1000, 0x1000, 0x9f253412 )
	ROM_LOAD( "snd-2.a9",     0x2000, 0x1000, 0x7783d6c6 )
ROM_END


ROM_START( tron_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "scpu_pga.bin", 0x0000, 0x2000, 0x5151770b )
	ROM_LOAD( "scpu_pgb.bin", 0x2000, 0x2000, 0x8ddf8717 )
	ROM_LOAD( "scpu_pgc.bin", 0x4000, 0x2000, 0x4241e3a0 )
	ROM_LOAD( "scpu_pgd.bin", 0x6000, 0x2000, 0x035d2fe7 )
	ROM_LOAD( "scpu_pge.bin", 0x8000, 0x2000, 0x24c185d8 )
	ROM_LOAD( "scpu_pgf.bin", 0xA000, 0x2000, 0x38c4bbaf )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "scpu_bgg.bin", 0x00000, 0x2000, 0x1a9ed2f5 )
	ROM_LOAD( "scpu_bgh.bin", 0x02000, 0x2000, 0x3220f974 )
	ROM_LOAD( "vg_3.bin",     0x04000, 0x2000, 0xbc036d1d )
	ROM_LOAD( "vg_2.bin",     0x08000, 0x2000, 0x58ee14d3 )
	ROM_LOAD( "vg_1.bin",     0x0c000, 0x2000, 0x3329f9d4 )
	ROM_LOAD( "vg_0.bin",     0x10000, 0x2000, 0x9743f873 )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "ssi_0a.bin",   0x0000, 0x1000, 0x765e6eba )
	ROM_LOAD( "ssi_0b.bin",   0x1000, 0x1000, 0x1b90ccdd )
	ROM_LOAD( "ssi_0c.bin",   0x2000, 0x1000, 0x3a4bc629 )
ROM_END


ROM_START( tron2_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "pro0.d2",      0x0000, 0x2000, 0x0de0471a )
	ROM_LOAD( "scpu_pgb.bin", 0x2000, 0x2000, 0x8ddf8717 )
	ROM_LOAD( "scpu_pgc.bin", 0x4000, 0x2000, 0x4241e3a0 )
	ROM_LOAD( "scpu_pgd.bin", 0x6000, 0x2000, 0x035d2fe7 )
	ROM_LOAD( "scpu_pge.bin", 0x8000, 0x2000, 0x24c185d8 )
	ROM_LOAD( "scpu_pgf.bin", 0xA000, 0x2000, 0x38c4bbaf )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "scpu_bgg.bin", 0x00000, 0x2000, 0x1a9ed2f5 )
	ROM_LOAD( "scpu_bgh.bin", 0x02000, 0x2000, 0x3220f974 )
	ROM_LOAD( "vg_3.bin",     0x04000, 0x2000, 0xbc036d1d )
	ROM_LOAD( "vg_2.bin",     0x08000, 0x2000, 0x58ee14d3 )
	ROM_LOAD( "vg_1.bin",     0x0c000, 0x2000, 0x3329f9d4 )
	ROM_LOAD( "vg_0.bin",     0x10000, 0x2000, 0x9743f873 )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "ssi_0a.bin",   0x0000, 0x1000, 0x765e6eba )
	ROM_LOAD( "ssi_0b.bin",   0x1000, 0x1000, 0x1b90ccdd )
	ROM_LOAD( "ssi_0c.bin",   0x2000, 0x1000, 0x3a4bc629 )
ROM_END


ROM_START( kroozr_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "kozmkcpu.2d",  0x0000, 0x2000, 0x61e02045 )
	ROM_LOAD( "kozmkcpu.3d",  0x2000, 0x2000, 0xcaabed57 )
	ROM_LOAD( "kozmkcpu.4d",  0x4000, 0x2000, 0x2bc83fc7 )
	ROM_LOAD( "kozmkcpu.5d",  0x6000, 0x2000, 0xa0ec38c1 )
	ROM_LOAD( "kozmkcpu.6d",  0x8000, 0x2000, 0x7044f2b6 )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "kozmkcpu.3g",  0x00000, 0x2000, 0xeda6ed2d )
	ROM_LOAD( "kozmkcpu.4g",  0x02000, 0x2000, 0xddde894b )
	ROM_LOAD( "kozmkvid.1e",  0x04000, 0x2000, 0xca60e2cc )
	ROM_LOAD( "kozmkvid.1d",  0x08000, 0x2000, 0x4e23b35b )
	ROM_LOAD( "kozmkvid.1b",  0x0c000, 0x2000, 0xc6041ba7 )
	ROM_LOAD( "kozmkvid.1a",  0x10000, 0x2000, 0xb57fb0ff )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "kozmksnd.7a",  0x0000, 0x1000, 0x6736e433 )
	ROM_LOAD( "kozmksnd.8a",  0x1000, 0x1000, 0xea9cd919 )
	ROM_LOAD( "kozmksnd.9a",  0x2000, 0x1000, 0x9dfa7994 )
ROM_END


ROM_START( domino_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "dmanpg0.bin",  0x0000, 0x2000, 0x3bf3bb1c )
	ROM_LOAD( "dmanpg1.bin",  0x2000, 0x2000, 0x85cf1d69 )
	ROM_LOAD( "dmanpg2.bin",  0x4000, 0x2000, 0x7dd2177a )
	ROM_LOAD( "dmanpg3.bin",  0x6000, 0x2000, 0xf2e0aa44 )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "dmanbg0.bin",  0x00000, 0x2000, 0x9163007f )
	ROM_LOAD( "dmanbg1.bin",  0x02000, 0x2000, 0x28615c56 )
	ROM_LOAD( "dmanfg0.bin",  0x04000, 0x2000, 0x0b1f9f9e )
	ROM_LOAD( "dmanfg1.bin",  0x08000, 0x2000, 0x16aa4b9b )
	ROM_LOAD( "dmanfg2.bin",  0x0c000, 0x2000, 0x4a8e76b8 )
	ROM_LOAD( "dmanfg3.bin",  0x10000, 0x2000, 0x1f39257e )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "dm-a7.snd",    0x0000, 0x1000, 0xfa982dcc )
	ROM_LOAD( "dm-a8.snd",    0x1000, 0x1000, 0x72839019 )
	ROM_LOAD( "dm-a9.snd",    0x2000, 0x1000, 0xad760da7 )
	ROM_LOAD( "dm-a10.snd",   0x3000, 0x1000, 0x958c7287 )
ROM_END


ROM_START( wacko_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "wackocpu.2d",  0x0000, 0x2000, 0xc98e29b6 )
	ROM_LOAD( "wackocpu.3d",  0x2000, 0x2000, 0x90b89774 )
	ROM_LOAD( "wackocpu.4d",  0x4000, 0x2000, 0x515edff7 )
	ROM_LOAD( "wackocpu.5d",  0x6000, 0x2000, 0x9b01bf32 )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "wackocpu.3g",  0x00000, 0x2000, 0x33160eb1 )
	ROM_LOAD( "wackocpu.4g",  0x02000, 0x2000, 0xdaf37d7c )
	ROM_LOAD( "wackovid.1e",  0x04000, 0x2000, 0xdca59be7 )
	ROM_LOAD( "wackovid.1d",  0x08000, 0x2000, 0xa02f1672 )
	ROM_LOAD( "wackovid.1b",  0x0c000, 0x2000, 0x7d899790 )
	ROM_LOAD( "wackovid.1a",  0x10000, 0x2000, 0x080be3ad )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "wackosnd.7a",  0x0000, 0x1000, 0x1a58763f )
	ROM_LOAD( "wackosnd.8a",  0x1000, 0x1000, 0xa4e3c771 )
	ROM_LOAD( "wackosnd.9a",  0x2000, 0x1000, 0x155ba3dd )
ROM_END


ROM_START( twotiger_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "2tgrpg0.bin",  0x0000, 0x2000, 0xe77a924b )
	ROM_LOAD( "2tgrpg1.bin",  0x2000, 0x2000, 0x2699ebdc )
	ROM_LOAD( "2tgrpg2.bin",  0x4000, 0x2000, 0xb5ca3f17 )
	ROM_LOAD( "2tgrpg3.bin",  0x6000, 0x2000, 0x8aa82049 )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "2tgrbg0.bin",  0x00000, 0x2000, 0x52f69068 )
	ROM_LOAD( "2tgrbg1.bin",  0x02000, 0x2000, 0x758d4f7d )
	ROM_LOAD( "2tgrfg0.bin",  0x04000, 0x2000, 0x4abf3ca0 )
	ROM_LOAD( "2tgrfg1.bin",  0x08000, 0x2000, 0xfbcaffa5 )
	ROM_LOAD( "2tgrfg2.bin",  0x0c000, 0x2000, 0x08e3e1a6 )
	ROM_LOAD( "2tgrfg3.bin",  0x10000, 0x2000, 0x9b22697b )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "2tgra7.bin",   0x0000, 0x1000, 0x4620d970 )
	ROM_LOAD( "2tgra8.bin",   0x1000, 0x1000, 0xe95d8cfe )
	ROM_LOAD( "2tgra9.bin",   0x2000, 0x1000, 0x81e6ce0e )
ROM_END


ROM_START( journey_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "d2",           0x0000, 0x2000, 0xf2618913 )
	ROM_LOAD( "d3",           0x2000, 0x2000, 0x2f290d2e )
	ROM_LOAD( "d4",           0x4000, 0x2000, 0xcc6c0150 )
	ROM_LOAD( "d5",           0x6000, 0x2000, 0xc3023931 )
	ROM_LOAD( "d6",           0x8000, 0x2000, 0x5d445c99 )

	ROM_REGION_DISPOSE(0x14000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "g3",           0x00000, 0x2000, 0xc14558de )
	ROM_LOAD( "g4",           0x02000, 0x2000, 0x9104c1d0 )
	ROM_LOAD( "a7",           0x04000, 0x2000, 0x4ca2bb2d )
	ROM_LOAD( "a8",           0x06000, 0x2000, 0x4fb7925d )
	ROM_LOAD( "a5",           0x08000, 0x2000, 0x560c474f )
	ROM_LOAD( "a6",           0x0a000, 0x2000, 0xb1f31583 )
	ROM_LOAD( "a3",           0x0c000, 0x2000, 0xf295afda )
	ROM_LOAD( "a4",           0x0e000, 0x2000, 0x765876a7 )
	ROM_LOAD( "a1",           0x10000, 0x2000, 0x4af986f8 )
	ROM_LOAD( "a2",           0x12000, 0x2000, 0xb30cd2a7 )

	ROM_REGION(0x10000)	/* 64k for the audio CPU */
	ROM_LOAD( "a",            0x0000, 0x1000, 0x2524a2aa )
	ROM_LOAD( "b",            0x1000, 0x1000, 0xb8e35814 )
	ROM_LOAD( "c",            0x2000, 0x1000, 0x09c488cf )
	ROM_LOAD( "d",            0x3000, 0x1000, 0x3d627bee )
ROM_END



/*************************************
 *
 *	Game drivers
 *
 *************************************/

#define MCR2_DRIVER(name,year,rotate,fullname) 		\
	struct GameDriver name##_driver =				\
	{												\
		__FILE__,									\
		0,											\
		#name,										\
		fullname,									\
		#year,										\
		"Bally Midway",								\
		"Christopher Kirmse\nAaron Giles\n"			\
		"Nicola Salmoria\nBrad Oliver",				\
		0,											\
		&machine_driver,							\
		name##_init,								\
													\
		name##_rom,									\
		0, 0,										\
		0,											\
		0,	/* sound_prom */						\
													\
		name##_input_ports,							\
													\
		0, 0,0,										\
		rotate,										\
													\
		mcr_hiload,mcr_hisave						\
	};

#define MCR2_CLONE_DRIVER(name,year,rotate,fullname,cloneof) \
	struct GameDriver name##_driver =				\
	{												\
		__FILE__,									\
		&cloneof##_driver,							\
		#name,										\
		fullname,									\
		#year,										\
		"Bally Midway",								\
		"Christopher Kirmse\nAaron Giles\n"			\
		"Nicola Salmoria\nBrad Oliver",				\
		0,											\
		&machine_driver,							\
		cloneof##_init,								\
													\
		name##_rom,									\
		0, 0,										\
		0,											\
		0,	/* sound_prom */						\
													\
		cloneof##_input_ports,						\
													\
		0, 0,0,										\
		rotate,										\
													\
		mcr_hiload,mcr_hisave						\
	};


MCR2_DRIVER      (shollow,  1981, ORIENTATION_ROTATE_90, "Satan's Hollow (set 1)")
MCR2_CLONE_DRIVER(shollow2, 1981, ORIENTATION_ROTATE_90, "Satan's Hollow (set 2)", shollow)
MCR2_DRIVER      (tron,     1982, ORIENTATION_ROTATE_90, "Tron (set 1)")
MCR2_CLONE_DRIVER(tron2,    1982, ORIENTATION_ROTATE_90, "Tron (set 2)", tron)
MCR2_DRIVER      (kroozr,   1982, ORIENTATION_DEFAULT,   "Kozmik Kroozr")
MCR2_DRIVER      (domino,   1982, ORIENTATION_DEFAULT,   "Domino Man")
MCR2_DRIVER      (wacko,    1982, ORIENTATION_DEFAULT,   "Wacko")
MCR2_DRIVER      (twotiger, 1984, ORIENTATION_DEFAULT,   "Two Tigers")

struct GameDriver journey_driver =
{
	__FILE__,
	0,
	"journey",
	"Journey",
	"1983",
	"Bally Midway",
	"Christopher Kirmse\nAaron Giles\n"
	"Nicola Salmoria\nBrad Oliver",
	0,
	&journey_machine_driver,
	journey_init,

	journey_rom,
	0, 0,
	0,
	0,	/* sound_prom */

	domino_input_ports,

	0,0,0,
	ORIENTATION_ROTATE_90,

	mcr_hiload,mcr_hisave
};
