/* IMAPADD.CMD
 *
 * Script to batch-move messages to the Inbox of an IMAP mail account.
 * Takes one parameter: the message(s) to be copied.  This may be a single
 * filespec, a wildcard pattern, or the name of a directory containing
 * messages to be sent.  All files so specified MUST be single, complete
 * messages in valid RFC2822 format.  No validation is performed by this
 * script.  Messages are DELETED immediately after being sent.
 *
 * This script assumes that the IMAP server is connected via SSL using
 * stunnel on localhost port 993.  Servers requiring STARTTLS or SASL
 * authentication are not supported.
 *
 * Knowledgeable persons may realize that it is trivially easy to modify
 * this script to connect directly to a remote IMAP server without SSL.
 * Do NOT do this.  Most IMAP servers will reject unencrypted connections,
 * and all you are likely to accomplish is transmitting your username and
 * password in cleartext across the Internet.
 *
 */
SIGNAL ON NOVALUE

/***** MODIFY THE NEXT TWO VALUES FOR YOUR OWN IMAP MAIL ACCOUNT *****/

globals.!userid   = 'my_userid'
globals.!password = 'my_password'

/***** NO MODIFICATIONS REQUIRED BEYOND THIS POINT *****/


PARSE ARG message_file
IF message_file == '' THEN RETURN 1

/* Initialize the required REXX library functions.
 */
CALL RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
CALL SysLoadFuncs
CALL RxFuncAdd 'SockLoadFuncs', 'RXSOCK', 'SockLoadFuncs'
CALL SockLoadFuncs

/* Prepare the log file.
 */
logs = VALUE('LOGFILES',,'OS2ENVIRONMENT')
IF logs == '' THEN DO
    PARSE SOURCE . . me .
    mydir = FILESPEC('DRIVE', me ) || FILESPEC('PATH', me )
    logs = STRIP( mydir ), 'T', '\')
END
globals.!logfile = logs'\IMAPADD.LOG'
logsize = STREAM( globals.!logfile, 'C', 'QUERY SIZE')
IF logsize <> '' THEN DO
    IF logsize >= 100000 THEN DO
        '@copy' globals.!logfile logs'\IMAPADD.BAK'
        CALL SysFileDelete globals.!logfile
    END
END
CALL LINEOUT globals.!logfile, '--[' DATE() TIME() ']--------------------------------------------------'

/* Get the messages to be copied.
 */
ok = SysFileTree( message_file, 'dirs.', 'DO')
IF dirs.0 > 0 THEN
    message_file = STRIP( message_file, 'T', '\') || '\'
ok = SysFileTree( message_file, 'messages.', 'FSL')
IF ( ok <> 0 ) | ( messages.0 < 1 ) THEN DO
    CALL SayLog 'No messages found.'
    CALL LINEOUT globals.!logfile
    RETURN 0
END

/* Try to connect to the server via stunnel on localhost:993
 */
globals.!online = 0
CALL Connect 'localhost:993'
CALL WaitForIncoming

IF globals.!online = 1 THEN DO i = 1 TO messages.0
    /* We are connected and authenticated, so proceed. */
    PARSE VAR messages.i . . _msize . _mfile
    _mfile = STRIP( _mfile )
    CALL SayLog 'Processing "'_mfile'"'
    ok = AppendMessageToInbox( _msize, _mfile )
    /* 0 means success confirmed
     * 1 means server response timed out, which is PROBABLY ok
     * -1 means a socket error occurred, meaning the connection probably died
     */
    IF ( ok == 0 ) | ( ok == 1 ) THEN CALL SysFileDelete _mfile
    ELSE LEAVE
END

CALL Disconnect

CALL LINEOUT globals.!logfile
RETURN 0


/***************************************************************************
 * Connect
 *
 */
Connect: PROCEDURE EXPOSE globals.
    PARSE ARG useraddress':'userport

    IF useraddress == '' THEN RETURN
    IF userport == '' THEN userport = 143

    CALL SayLog 'Locating host' useraddress '...'
    rc = SockGetHostByName( useraddress, 'host.!')
    IF rc == 0 THEN hostaddress = useraddress
    ELSE hostaddress = host.!addr

    imapsrv.!family = 'AF_INET'
    imapsrv.!addr   = hostaddress
    imapsrv.!port   = userport

    socket = SockSocket('AF_INET', 'SOCK_STREAM', 'IPPROTO_TCP')

    CALL SayLog 'Connecting to' imapsrv.!addr '(max. timeout 75 secs) ...'
    rc = SockConnect( socket, 'imapsrv.!')

    IF rc == -1 THEN DO
        CALL SayLog 'Failed to connect to' imapsrv.!addr '-->' GetSocketErrorMsg( SockSock_Errno() )
        RETURN
    END

    globals.!servname       = useraddress
    globals.!online         = 2             /* 2 = connect pending */
    globals.!socket         = socket
    globals.!tag            = '0000'
    globals.!selected       = 0
    globals.!receiveliteral = 0
    globals.!sendliteral    = 0
    globals.!mbox.0         = 0
    globals.!pending.0      = 0

RETURN socket


/***************************************************************************
 * ConnectFinish
 *
 * Finish up the connection process once the server responds .
 */
ConnectFinish: PROCEDURE EXPOSE globals.

    globals.!online = 1

/* TODO:
 * should do a CAPABILITY here, and disconnect if 'IMAP4rev1' is not returned.
    CALL Transmit 'CAPABILITY', 'CAPABILITY'
 */

    CALL Login globals.!userid globals.!password
    CALL WaitForIncoming

    CALL SayLog 'Connection successful.'

RETURN


/***************************************************************************
 * Disconnect
 */
Disconnect: PROCEDURE EXPOSE globals.

    CALL Transmit 'LOGOUT', 'LOGOUT'

RETURN


/***************************************************************************
 * DisconnectFinish
 */
DisconnectFinish: PROCEDURE EXPOSE globals.

    CALL SockClose( globals.!socket )

    DROP globals.!tid
    DROP globals.!socket
    DROP globals.!servname

    DO i = 1 TO globals.!mbox.0
        DROP globals.!mbox.i.!msglist.
        name = globals.!mbox.i.!name
        DROP globals.!mbox.name
        DROP globals.!mbox.i.
    END
    DROP globals.!mbox.

    /*
     * If this flag is not set to 'pending disconnect', it means the connection
     * died without us receiving a 'BYE' from the server.  This should be
     * considered an error condition, and the user notified accordingly.
     */
    IF globals.!online > 0 THEN DO
        CALL SayLog 'The connection to the server has been unexpectedly lost.'
    END

    globals.!online   = 0
    globals.!selected = ''

    CALL SayLog 'The connection has been closed.'

RETURN


/***************************************************************************
 * Login
 */
Login: PROCEDURE EXPOSE globals.
    PARSE ARG userid password
    IF userid == '' THEN DO
        CALL Transmit 'LOGOUT', 'LOGOUT'
        RETURN
    END
    CALL Transmit 'LOGIN', 'LOGIN' userid password
RETURN


/***************************************************************************
 * LoginFinish
 */
LoginFinish: PROCEDURE EXPOSE globals.
    ARG flag

    IF flag == 0 THEN DO
        CALL SayLog 'The login attempt was unsuccessful.  The connection will be closed.'
        CALL DisconnectFinish
        RETURN
    END
    globals.!mbox.!scanned = 0

RETURN

/***************************************************************************
 * AppendMessageToInbox
 *
 * Appends the specified message (as-is) into the remote INBOX.
 */
AppendMessageToInbox: PROCEDURE EXPOSE globals.
    PARSE ARG message_size, message_file

    retval = 0
    CALL Transmit 'APPEND', 'APPEND INBOX {'message_size'}'
    CALL WaitForIncoming
    IF globals.!sendliteral == 1 THEN DO
        _data = CHARIN( message_file, 1, message_size )
        CALL CHAROUT, 'Sending' message_size 'literal bytes to server... '
        CALL CHAROUT globals.!logfile, 'Sending' message_size 'literal bytes to server... '
        rc = SockSend( globals.!socket, _data || '0d0a'x )
        CALL SayLog 'done.'
        CALL STREAM message_file, 'C', 'CLOSE'
        IF rc == -1 THEN DO
            /* Send failed; determine the error and display a message */
            CALL SayLog 'Socket operation failed -->' GetSocketErrorMsg( SockSock_Errno() )
            retval = -1
        END
        ELSE DO
            retval = WaitForIncoming()
        END
    END
    globals.!sendliteral = 0
RETURN retval


/***************************************************************************
 * WaitForIncoming
 *
 * Wait for and read response data from the server. Once a complete line
 * ending in crlf is received, pass it to the Incoming routine. Then
 * return, unless we're in the middle of a data stream in which case
 * we continue.
 *
 * This routine will wait up to 20 seconds before giving up.
 */
WaitForIncoming: PROCEDURE EXPOSE globals.
    buffer = ''
    DO FOREVER
        sr.0  = 1
        sr.1  = globals.!socket
        ready = SockSelect('sr.',,,20)
        IF ready > 0 THEN DO
            CALL SockIoctl globals.!socket, 'FIONREAD', 'bytes'
            IF bytes == 0 THEN LEAVE
            CALL SockRecv globals.!socket, 'data', bytes
            buffer = buffer || data
            IF RIGHT( data, 2 ) == '0d0a'x THEN DO
                CALL Incoming buffer
                IF globals.!receiveliteral == 0 THEN LEAVE
            END
        END
        ELSE DO
            CALL SayLog 'Timed out waiting for response from server.'
            RETURN 1
        END
    END
RETURN 0


/***************************************************************************
 * Incoming
 *
 * This method checks incoming data from the server.  It parses the data
 * into its CRLF-separated sections, and then decides how to deal with it.
 */
Incoming: PROCEDURE EXPOSE globals.
    PARSE ARG data

    /* Break data up into individual lines */
    datalines.0 = 0
    buffer      = ''
    j           = 0
    DO i = 1 TO LENGTH( data )
        IF SUBSTR( data, i, 2 ) == '0D0A'x THEN DO
            i = i + 1
            j = j + 1
            datalines.j = buffer
            buffer = ''
        END
        ELSE
            buffer = buffer || SUBSTR( data, i, 1 )
    END
    datalines.0 = j


    DO i = 1 TO datalines.0

        CALL SayLog '<<<' datalines.i

        /* See if the command indicates the start of a 'literal' bytestream.  If
         * so, we set !receiveliteral to indicate that we're receiving literal
         * data.  This and all subsequent datalines will be concatenated into a
         * buffer (!literaltext) until the indicated number of bytes is reached.
         */
        IF globals.!receiveliteral == 0 THEN DO
            PARSE VAR datalines.i . '{'count'}' .
            IF ( count \= '') & ( VERIFY( count, '0123456789') == 0 ) THEN DO
                globals.!receiveliteral = count + LENGTH( datalines.i ) + 2
                globals.!literaltext    = ''
            END
        END

        /* Literal bytestream in progress. Append to buffer. */
        IF globals.!receiveliteral > 0 THEN DO
            globals.!literaltext    = globals.!literaltext || datalines.i || '0D0A'x
            IF ( LENGTH( globals.!literaltext ) > globals.!receiveliteral ) THEN DO
                CALL InterpretResponse globals.!literaltext
                globals.!literaltext    = ''
                globals.!receiveliteral = 0
            END
        END

        /* Otherwise this is a standalone response; interpret it directly. */
        ELSE
            CALL InterpretResponse datalines.i
    END

RETURN


/***************************************************************************
 * InterpretResponse
 *
 * This method performs some checks on incoming data and performs any
 * necessary reformatting before calling HandleResponse()
 */
InterpretResponse: PROCEDURE EXPOSE globals.
    PARSE ARG responsetext

    /* Command completion request */
    IF LEFT( responsetext, 1 ) == '+' THEN DO
        globals.!sendliteral = 1
    END

    /* Untagged response */
    ELSE IF LEFT( responsetext, 1 ) == '*' THEN DO
        PARSE VAR responsetext '*' response value

        /*
         * Annoyingly, a few IMAP responses put a value before the
         * response keyword, so we have to handle these specially...
         */
        vw = WORD( value, 1 )
        IF ( vw == 'EXISTS') | ( vw == 'RECENT') | ( vw == 'EXPUNGE') | ( vw == 'FETCH') THEN DO
            len = LENGTH( vw )
            IF len < LENGTH( value ) THEN
                newvalue = response STRIP( SUBSTR( value, len+1 ))
            ELSE
                newvalue = response
            key   = vw
            parms = newvalue
        END
        ELSE DO
            key   = response
            parms = value
        END

        CALL HandleResponse , key, parms

    END

    /* Tagged response */
    ELSE DO
        PARSE VAR responsetext tag response command value
        /* Remove this tag's command from the unresolved list */
        IF SYMBOL('globals.!pending.'tag'.!command') == 'VAR' THEN DO
            CALL HandleResponse tag, response, value
            DROP globals.!pending.tag.
        END
        ELSE DO
            CALL SayLog '*** Received unexpected tag' tag ' in response'
        END
    END

RETURN


/***************************************************************************
 * HandleResponse
 *
 * Very important method which determines what action to take when any
 * response is received from the server.
 *
 * Arguments:
 *      tag         The IMAP 'tag' associated with the response.  (Only applies
 *                  to so-called 'tagged responses'.)
 *      keyword     The response keyword.
 *      parameters  Any parameters returned along with the keyword.
 */
HandleResponse: PROCEDURE EXPOSE globals.
    PARSE ARG tag, keyword, parameters

    keyword = TRANSLATE( keyword )
    action  = ''
    origcmd = ''
    origkey = ''

    IF tag \= '' THEN DO
        action  = TRANSLATE( globals.!pending.tag.!action )
        origcmd = globals.!pending.tag.!command
        origkey = TRANSLATE( WORD( origcmd, 1 ))
    END

    SELECT

        /*
         * Untagged Responses (either unsolicited or from commands in-progress)
         */
        WHEN keyword == 'BYE' THEN DO
            globals.!online = -2    /* Set disconnect pending */
        END

        WHEN keyword == 'EXISTS' THEN DO
            mailbox = globals.!selected
            IF mailbox \= 0 THEN globals.!mbox.mailbox.!exists = STRIP( parameters )
        END

/*
        WHEN keyword == 'FETCH' THEN DO
            CALL ParseFetchResponse parameters
            SELECT
                WHEN message.!contents \= '' THEN CALL DisplayMessage
                WHEN message.!from     \= '' THEN CALL AddMessage
                WHEN message.!flags    \= '' THEN CALL UpdateMessageFlags
                OTHERWISE NOP
            END
            DROP message.
        END
*/

        WHEN keyword == 'FLAGS' THEN DO
            mailbox = globals.!selected
            IF mailbox \= 0 THEN DO
                IF POS('\Answered', parameters ) > 0 THEN globals.!mbox.mailbox.!answered = 1
                IF POS('\Flagged',  parameters ) > 0 THEN globals.!mbox.mailbox.!flagged  = 1
                IF POS('\Deleted',  parameters ) > 0 THEN globals.!mbox.mailbox.!deleted  = 1
                IF POS('\Seen',     parameters ) > 0 THEN globals.!mbox.mailbox.!seen     = 1
                IF POS('\Draft',    parameters ) > 0 THEN globals.!mbox.mailbox.!draft    = 1
            END
        END

/*
        WHEN keyword == 'LIST' THEN DO
            CALL ListFolder parameters
        END
*/

        WHEN ( keyword == 'OK') & ( tag == '') THEN DO
            PARSE VAR parameters '[' remark val ']' .
            SELECT
                WHEN globals.!online == 2 THEN
                    CALL ConnectFinish
                WHEN remark == 'UIDVALIDITY' THEN DO
                    mailbox = globals.!selected
                    IF mailbox \= 0 THEN globals.!mbox.mailbox.!uidvalidity = STRIP( val )
                END
                WHEN remark == 'UIDNEXT' THEN DO
                    mailbox = globals.!selected
                    IF mailbox \= 0 THEN globals.!mbox.mailbox.!uidnext = STRIP( val )
                END
                OTHERWISE NOP
            END
        END

        WHEN keyword == 'RECENT' THEN DO
            mailbox = globals.!selected
            IF mailbox \= 0 THEN globals.!mbox.mailbox.!recent = STRIP( parameters )
        END


        /*
         * Tagged Responses (from completed commands)
         */

        WHEN ( keyword == 'BAD') & ( tag \= '') THEN DO
            SELECT
                WHEN action == 'SELECTMAILBOX' THEN globals.!selected == ''
                OTHERWISE NOP
            END
            CALL SayLog 'BAD response received:' parameters
        END

        WHEN ( keyword == 'NO') & ( tag \= '') THEN DO
            SELECT
                WHEN action == 'LOGIN' THEN CALL LoginFinish 0
                WHEN action == 'SELECTMAILBOX' THEN DO
                    CALL SayLog 'Unable to select mailbox:' parameters
                    globals.!selected = 0
                END
                OTHERWISE NOP
            END
        END

        WHEN ( keyword == 'OK') & ( tag \= '') THEN DO
            SELECT
/*
                WHEN origkey == 'CLOSE'   THEN CALL CloseMailboxFinish action
                WHEN origkey == 'COPY'    THEN CALL CopyMessageFinish action, origcmd
                WHEN origkey == 'FETCH'   THEN DO
                    SELECT
                        WHEN action == 'FETCHLIST'   THEN CALL FetchBatchFinish origcmd, parameters
                        WHEN action == 'SHOWMESSAGE' THEN CALL FetchMessageFinish
                        OTHERWISE NOP
                    END
                END
                WHEN origkey == 'LIST'    THEN CALL ListFoldersFinish
*/
                WHEN origkey == 'LOGIN'   THEN CALL LoginFinish 1
                WHEN origkey == 'SELECT'  THEN CALL SelectMailboxFinish
/*
                WHEN origkey == 'STORE' THEN DO
                    SELECT
                        WHEN action == 'DELETEMESSAGE' THEN
                            CALL DeleteMessageFinish
                        WHEN action == 'UNDELETEMESSAGE' THEN
                            CALL UndeleteMessageFinish
                        OTHERWISE NOP
                    END
                END
*/

                OTHERWISE NOP
            END
        END

        OTHERWISE NOP

    END

RETURN


/***************************************************************************
 * Transmit
 *
 * Transmits the specified command to the server. The 'purpose' arg is a
 * keyword which identifies the command being sent; normally it is simply
 * the IMAP command word being used.
 */
Transmit: PROCEDURE EXPOSE globals.
    PARSE ARG purpose, string

    IF globals.!online == 0 THEN RETURN

    /*
     * Generate an IMAP tag prefix
     */
    tag         = NextTag( globals.!tag )
    imapcommand = tag string

    IF WORD( string, 1 ) == 'LOGIN' THEN
        printcommand = tag 'LOGIN' WORD( string, 2 ) '<hidden>'
    ELSE
        printcommand = imapcommand

    CALL SayLog '>>>' printcommand

    rc = SockSend( globals.!socket, imapcommand || '0D0A'x )
    IF rc == -1 THEN DO
        /* Transmit failed; determine the error and display a message */
        CALL SayLog 'Socket operation failed -->' GetSocketErrorMsg( SockSock_Errno() )
        RETURN
    END

    /* Add this command to the 'pending commands' dictionary
     */
    globals.!pending.tag.!command = string
    globals.!pending.tag.!action  = purpose
    globals.!tag                  = tag

RETURN


/***************************************************************************/
NextTag: PROCEDURE EXPOSE globals.
    ARG tag
    IF tag == '' THEN tag = '0000'

    work  = REVERSE( tag )
    final = ''
    DO i = 1 TO LENGTH( work )
        next = 0
        c = SUBSTR( work, i, 1 )
        IF VERIFY( c, '0123456789') == 0 THEN DO
            IF c == '9' THEN n = 'A'
            ELSE             n = c + 1
        END
        ELSE IF VERIFY( c, 'ABCDEFGHIJKLMNOPQRSTUVWXYZ') == 0 THEN DO
            IF c == 'Z' THEN DO
                n    = '0'
                next = 1
            END
            ELSE n = D2C( C2D( c ) + 1 )
        END
        work = OVERLAY( n, work, i )
        IF next == 0 THEN LEAVE
    END

    IF ( i > LENGTH( work )) & ( next == 1 ) THEN
        work = work || 'A'

    tag = REVERSE( work )

RETURN tag


/***************************************************************************/
SayLog: PROCEDURE EXPOSE globals.
    PARSE ARG message
    SAY message
    CALL LINEOUT globals.!logfile, message
RETURN 0


/***************************************************************************/
NOVALUE:
    SAY 'NOVALUE condition on line' sigl':'
    SAY SOURCELINE( sigl )
EXIT sigl
