/***********************************************************/
/*    													   */
/*    tabs.cr											   */
/*    													   */
/*    Macros for tab handling							   */
/*    													   */
/***********************************************************/

# include "crisp.h"

string
	spacefill;

void
main()
{
	int width = 132;
	spacefill = " ";
	while (width--)
		spacefill += " ";
}


	/*****************************************/
	/*	 detab a single line Fills tabs 	 */
	/*	 with spaces at current tab 		 */
	/*	 setting							 */
	/*****************************************/

string
detab (string line)
{
	int pos;

	/*****************************************/
	/*	  Process all tabs					 */
	/*****************************************/
	while ((pos = index(line, "\t")) > 0)
	{
		int expand = distance_to_tab(pos);
		line =	substr (line, 1, pos - 1) +
				substr (spacefill, 1, expand) +
				substr (line, pos + 1);
	}
	return line;
}


	/*****************************************/
	/*	  Backfill all whitespace			 */
	/*****************************************/

string
backfill (int atcol, int fillto)
{
	int tabinc;
	string result = "";

	if (atcol != fillto)
	{
		tabinc = distance_to_tab(atcol);
		while ((atcol + tabinc) <= fillto)
		{
			result += (tabinc == 1) ? " " : "\t";
			atcol  += tabinc;
			tabinc	= distance_to_tab(atcol);
		}
		result += substr(spacefill, 1, fillto - atcol);
		atcol = fillto;
	}
	return result;
}

	/*****************************************/
	/*	  entab a single line. Fills > 1 	 */
	/*	  consecutive spaces with tabs		 */
	/*	  at current tab setting			 */
	/*****************************************/

string
entab (string line)
{
	int atcol, col, pos, tabinc;
	string result = line;

	result = "";
	atcol = col = 1;
	while ((pos = search_string("[ \t]", line)) > 0)
	{											  // Find next whitespace
		if (pos > 1)
		{
			result += backfill(atcol, col) + substr (line, 1, pos - 1);
			col    += pos - 1;
			atcol	= col;
		}
		tabinc = (substr (line, pos, 1) == " ") ? 1
												: distance_to_tab(col);
		col += tabinc;
		line = substr (line, pos + 1);
	}
	return result + backfill (atcol, col) + line;
}

	/*****************************************/
	/*    Process a group of lines in		 */
	/*    the current buffer with a			 */
	/*    given macro						 */
	/*****************************************/

void
process_lines(string macroname, int startline, int endline)
{
	int 	line,
			lines,
			save_line,
			save_col,
			top_line,
			top_col;
	string	temp;

	inq_top_left(top_line, top_col);
	inq_position(save_line, save_col);
	top_of_buffer();
	lines = endline - startline + 1;
	for (line = 0; line++ < lines; )
	{
		int percent = line * 100 / lines;
		if ((percent % 10) == 0)
			message("%s %d%% done", macroname, percent);
		move_abs(line + startline - 1, 1);
		temp = read();
		delete_line();
		temp = execute_macro(macroname, temp);
		insert(trim(temp) + "\n");
		down();
	}
	set_top_left(top_line, top_col);
	move_abs(save_line, save_col);
}


	/*****************************************/
	/*	  Process all lines in the			 */
	/*	  current buffer with a given		 */
	/*	  macro 							 */
	/*****************************************/

void
process_all_lines(string macroname)
{
	process_lines(macroname, 1, inq_lines());
}


	/*****************************************/
	/*	  Process all marked lines			 */
	/*	  within a marked region			 */
	/*****************************************/

void
process_marked_lines(string macroname)
{
	int 	start_mark_line,
			start_mark_col,
			end_mark_line,
			end_mark_col,
			mark_type;

	mark_type = inq_marked(start_mark_line, start_mark_col,
							end_mark_line, end_mark_col);
	if (mark_type == MK_LINE)
		process_lines(macroname, start_mark_line, end_mark_line);
	else
	{
		beep();
		if (mark_type == MK_NONE)
			message("No marked region.");
		else
			message("Only line <Alt-L> marks supported here.");
	}
}

	/*****************************************/
	/*	  Detab all lines in current		 */
	/*	  buffer							 */
	/*****************************************/

void
detab_buffer ()
{
	process_all_lines("detab");
}

	/*****************************************/
	/*	  Detab marked lines in current 	 */
	/*	  buffer							 */
	/*****************************************/

void
detab_region ()
{
	process_marked_lines("detab");
}

	/*****************************************/
	/*	  Entab all lines in current		 */
	/*	  buffer							 */
	/*****************************************/

void
entab_buffer ()
{
	process_all_lines("entab");
}


void
entab_region ()
{
	process_marked_lines("entab");
}


/***********************************************************/
/*    This macro was written for testing the new form	   */
/*    of distance_to_tab()								   */
/***********************************************************/
void
print_tabs()
{
	int		i = 0;

	save_position();
	for (i =0; i < 3; ++i)
	{
		insert("\n");
		beginning_of_line();
		up();
	}
	beginning_of_line();
	for (i= 0; i++ < 80; )
	{
		int t = distance_to_tab(i);
		int x = 0;
		while (x++ < 3)
		{
			string yy;
			switch (x)
			{
			case 1:	yy = (i / 10);	break;
			case 2:	yy = (i % 10);	break;
			case 3:	yy = t;			break;
			}
			insert(yy);
			left();
			down();
		}
		up(5);
		right();
	}
	restore_position();
}

