/*-------------------------------------------------------------------------
 *
 * pipe.c
 *	  pipe()
 *
 * Copyright (c) 1996-2005, PostgreSQL Global Development Group
 *
 *	This is a replacement version of pipe for Win32 which allows
 *	returned handles to be used in select(). Note that read/write calls
 *	must be replaced with recv/send.
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/src/port/pipe.c,v 1.10 2004/12/31 22:03:53 pgsql Exp $
 *
 *-------------------------------------------------------------------------
 */

#include "postgres.h"

#ifdef WIN32
int
pgpipe(int handles[2])
{
	SOCKET		s;
	struct sockaddr_in serv_addr;
	int			len = sizeof(serv_addr);

	handles[0] = handles[1] = INVALID_SOCKET;

	if ((s = socket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET)
	{
		ereport(LOG, (errmsg_internal("pgpipe failed to create socket: %ui", WSAGetLastError())));
		return -1;
	}

	memset((void *) &serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_port = htons(0);
	serv_addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	if (bind(s, (SOCKADDR *) & serv_addr, len) == SOCKET_ERROR)
	{
		ereport(LOG, (errmsg_internal("pgpipe failed to bind: %ui", WSAGetLastError())));
		closesocket(s);
		return -1;
	}
	if (listen(s, 1) == SOCKET_ERROR)
	{
		ereport(LOG, (errmsg_internal("pgpipe failed to listen: %ui", WSAGetLastError())));
		closesocket(s);
		return -1;
	}
	if (getsockname(s, (SOCKADDR *) & serv_addr, &len) == SOCKET_ERROR)
	{
		ereport(LOG, (errmsg_internal("pgpipe failed to getsockname: %ui", WSAGetLastError())));
		closesocket(s);
		return -1;
	}
	if ((handles[1] = socket(PF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET)
	{
		ereport(LOG, (errmsg_internal("pgpipe failed to create socket 2: %ui", WSAGetLastError())));
		closesocket(s);
		return -1;
	}

	if (connect(handles[1], (SOCKADDR *) & serv_addr, len) == SOCKET_ERROR)
	{
		ereport(LOG, (errmsg_internal("pgpipe failed to connect socket: %ui", WSAGetLastError())));
		closesocket(s);
		return -1;
	}
	if ((handles[0] = accept(s, (SOCKADDR *) & serv_addr, &len)) == INVALID_SOCKET)
	{
		ereport(LOG, (errmsg_internal("pgpipe failed to accept socket: %ui", WSAGetLastError())));
		closesocket(handles[1]);
		handles[1] = INVALID_SOCKET;
		closesocket(s);
		return -1;
	}
	closesocket(s);
	return 0;
}


int
piperead(int s, char *buf, int len)
{
	int			ret = recv(s, buf, len, 0);

	if (ret < 0 && WSAGetLastError() == WSAECONNRESET)
		/* EOF on the pipe! (win32 socket based implementation) */
		ret = 0;
	return ret;
}

#endif

#ifdef __OS2__
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

int
pgpipe(int handles[2])
{
	int		s;
	struct sockaddr_in serv_addr;
	int			len = sizeof(serv_addr);
	handles[0] = handles[1] = -1;
	if ((s = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	{
		ereport(LOG,
				(errcode_for_socket_access(),
				 errmsg("pgpipe failed to create socket: %m")));
		return -1;
	}
	memset((void *) &serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_len = sizeof(serv_addr);
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_port = htons(0);
	serv_addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	if (bind(s,  (struct sockaddr *) &serv_addr, len) < 0)
	{
		ereport(LOG,
				(errcode_for_socket_access(),
				 errmsg("pgpipe failed to bind: %m")));
		closesocket(s);
		return -1;
	}
	if (listen(s, 1) < 0)
	{
		ereport(LOG,
				(errcode_for_socket_access(),
				 errmsg("pgpipe failed to listen: %m")));
		closesocket(s);
		return -1;
	}
	if (getsockname(s,  (struct sockaddr *) &serv_addr, &len) < 0)
	{
		ereport(LOG,
				(errcode_for_socket_access(),
				 errmsg("pgpipe failed to getsockname: %m")));
		closesocket(s);
		return -1;
	}
	if ((handles[1] = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	{
		ereport(LOG,
				(errcode_for_socket_access(),
				 errmsg("pgpipe failed to create socket 2: %m")));
		closesocket(s);
		return -1;
	}
	if (connect(handles[1],  (struct sockaddr *) &serv_addr, len) < 0)
	{
		ereport(LOG,
				(errcode_for_socket_access(),
				 errmsg("pgpipe failed to connect socket: %m")));
		closesocket(s);
		return -1;
	}
	if ((handles[0] = accept(s,  (struct sockaddr *) &serv_addr, &len)) < 0)
	{
		ereport(LOG,
				(errcode_for_socket_access(),
				 errmsg("pgpipe failed to accept socket: %m")));
		closesocket(handles[1]);
		handles[1] = -1;
		closesocket(s);
		return -1;
	}
	closesocket(s);
	return 0;
}

int
piperead(int s, char *buf, int len)
{
	int			ret = recv(s, buf, len, 0);
	if (ret < 0 )
		ret = 0;
	return ret;
}
#endif
