/*-------------------------------------------------------------------------
 *
 * sema.c
 *	  IBM OS/2 Semaphores Emulation
 *
 * Portions Copyright (c) 1996-2004, PostgreSQL Global Development Group
 * Portions Copyright (c) 2004, Lorne R Sunley
 * adapted from win32 semaphore emulation code
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/src/backend/port/ibmos2/sema.c,v 1.9 2004/09/07 14:31:42 tgl Exp $
 *
 *-------------------------------------------------------------------------
 */
#define INCL_EXAPIS
#include "postgres.h"
#include <sys/sem.h>
#include "storage/shmem.h"
#include <errno.h>
#include <unistd.h>
typedef struct
{
	int			m_numSems;
	char		set_semname[32];
	char		set_mtxname[32];
	off_t		m_semaphoreHandles;
	/* offset from beginning of header */
	off_t		m_semaphoreCounts;
	/* offset from beginning of header */
	off_t		m_semaphoreMutex;
	/* offset from beginning of header */
	pid_t		m_semaphorePID;
	/* offset from beginning of header */
}	os2_sem_set_hdr;
/* Control of a semaphore pool. The pool is an area in which we stored all
** the semIds of the pool. The first long is the number of semaphore
** allocated in the pool followed by semaphore handles
*/

int
semctl(int semId, int semNum, int flag, union semun semun)
{
	APIRET rc;
	os2_sem_set_hdr *the_set = (os2_sem_set_hdr *) MAKE_PTR(semId);
	if (semNum < 0 || semNum > the_set->m_numSems)
	{
		errno = EINVAL;
		return -1;
	}

	/* semNum might be 0 */
	/* semun.array contains the sem initial values */
	int		   *sem_counts = (int *) ((off_t) the_set + the_set->m_semaphoreCounts);
	int		   *sem_pids = (int *) ((off_t) the_set + the_set->m_semaphorePID);
	/* Fix the count of all sem of the pool to semun.array */
	if (flag == SETALL)
	{
		int			i;
		struct sembuf sops;
		sops.sem_flg = IPC_NOWAIT;
		for (i = 0; i < the_set->m_numSems; ++i)
		{
			if (semun.array[i] == sem_counts[i])
				continue;		/* Nothing to do */
			if (semun.array[i] < sem_counts[i])
				sops.sem_op = -1;
			else
				sops.sem_op = 1;
			sops.sem_num = i;
			/* Quickly lock/unlock the semaphore (if we can) */
//			if (semop(semId, &sops, 1) < 0)
//				return -1;
		}
		return 0;
	}
	/* Fix the count of one semaphore to semun.val */
	else if (flag == SETVAL)
	{
		if (semun.val != sem_counts[semNum])
		{
			sem_counts[semNum] = semun.val;
			struct sembuf sops;
			sops.sem_flg = IPC_NOWAIT;
			sops.sem_num = semNum;
			if (semun.val < sem_counts[semNum])
				sops.sem_op = -1;
			else
				sops.sem_op = 1;
			/* Quickly lock/unlock the semaphore (if we can) */
//			if (semop(semId, &sops, 1) < 0)
//				return -1;
		}
		return 0;
	}
	/* Delete the pool */
	else if (flag == IPC_RMID)
	{
		int			i;
		HEV	   *sem_handles = (HEV *) ((off_t) the_set + the_set->m_semaphoreHandles);
		/* Loop over all semaphore to delete them */
		for (i = 0; i < (the_set->m_numSems - 1); ++i)
			rc = DosCloseEventSem(sem_handles[i]);
		return 0;
	}
	/* Get the current semaphore count */
	else if (flag == GETNCNT)
		return the_set->m_numSems;
	/* Get the current semaphore count of the first semaphore in the pool */
	else if (flag == GETVAL)
	{
		return sem_counts[semNum];
	}
	else if (flag == GETPID)
	{
		return sem_pids[semNum];
	}
	/* Other commands not yet supported */
	else
	{
		errno = EINVAL;
		return -1;
	}
}
/* Find a pool id based on IPC key */
int
semget(key_t semKey, int semNum, int flags)
{
	char		semname[32];
	char		mtxname[32];
//	char		cur_num[20];
//	ULONG		last_error;
//	char	   *num_part;
	bool		ans = true;
	HEV		cur_handle;
	HMTX		mtx_handle;
	bool		found = false;
	Size		sem_set_size = sizeof(os2_sem_set_hdr) + (semNum * (sizeof(HEV) +  sizeof(HMTX) + sizeof(int) + sizeof(pid_t) ) );
	HEV	   *sem_handles = NULL;
	int		   *sem_counts = NULL;
	HMTX	   *mtx_handles = NULL;
	pid_t      *sem_pids = NULL;
	int			i;
	APIRET		ulRc;

	sprintf(semname, "\\SEM32\\PS.%d", (int)semKey);
	sprintf(mtxname, "\\SEM32\\MS.%d", (int)semKey);


	os2_sem_set_hdr *new_set = (os2_sem_set_hdr *) ShmemInitStruct(semname, sem_set_size, &found);
	if (found)
	{
		/* This should *never* happen */
		errno = EEXIST;
		return -1;
	}
	sprintf(new_set->set_semname, "\\SEM32\\PS.%d", (int)semKey);
	sprintf(new_set->set_mtxname, "\\SEM32\\MS.%d", (int)semKey);
	new_set->m_numSems = semNum;
	new_set->m_semaphoreHandles = sizeof(os2_sem_set_hdr);
	/* array starts after header */
	new_set->m_semaphoreCounts = new_set->m_semaphoreHandles + (sizeof(HEV) * semNum);
	new_set->m_semaphoreMutex = new_set->m_semaphoreCounts + (sizeof(int) * semNum);
	new_set->m_semaphorePID = new_set->m_semaphoreMutex + (sizeof(HMTX) * semNum);
	sem_handles = (HEV *) ((off_t) new_set + new_set->m_semaphoreHandles);
	sem_counts = (int *) ((off_t) new_set + new_set->m_semaphoreCounts);
	mtx_handles = (HMTX *) ((off_t) new_set + new_set->m_semaphoreMutex);
	sem_pids = (pid_t *) ((off_t) new_set + new_set->m_semaphorePID);
	for (i = 0; i < semNum && ans; ++i)
	{
		sprintf(semname, "\\SEM32\\PS.%d.%d", (int)semKey, i );
		if (flags & IPC_CREAT)
		{
			ulRc = DosCreateEventSemEx((PSZ)&semname, &cur_handle, DC_SEM_SHARED, FALSE );
		}
		else
		{
			ulRc = DosOpenEventSemEx((PSZ)&semname, &cur_handle);
		}
		sem_handles[i] = cur_handle;

		if (!cur_handle || ulRc != NO_ERROR)
		{
			if (!cur_handle)
			{
				errno = EACCES;
				ans = false;
			}
			if (ulRc == ERROR_DUPLICATE_NAME && (flags & (IPC_CREAT | IPC_EXCL)))
			{
				errno = EEXIST;
				ans = false;
			}
			if (ulRc != NO_ERROR)
			{
				errno = EACCES;
				ans = false;
			}
		}
		sprintf(mtxname, "\\SEM32\\MS.%d.%d", (int)semKey, i );
		if (flags & IPC_CREAT)
		{
			ulRc = DosCreateMutexSemEx((PSZ)&mtxname, &mtx_handle, DC_SEM_SHARED, FALSE );
		}
		else
		{
			ulRc = DosOpenMutexSemEx((PSZ)&mtxname, &mtx_handle);
		}
		mtx_handles[i] = mtx_handle;

		if (!mtx_handle || ulRc != NO_ERROR)
		{
			if (!mtx_handle)
			{
				errno = EACCES;
				ans = false;
			}
			if (ulRc == ERROR_DUPLICATE_NAME && (flags & (IPC_CREAT | IPC_EXCL)))
			{
				errno = EEXIST;
				ans = false;
			}
			if (ulRc != NO_ERROR)
			{
				errno = EACCES;
				ans = false;
			}
		}

		sem_pids[i] = getpid();

	}
	if (ans)
	{
		return MAKE_OFFSET(new_set);
	}
	else
	{
		int			i;
		/* Blow away what we've got right now... */
		for (i = 0; i < semNum; ++i)
		{
			if (sem_handles[i])
				ulRc = DosCloseEventSemEx(sem_handles[i]);
			else
				break;
		}
		for (i = 0; i < semNum; ++i)
		{
			if (mtx_handles[i])
				ulRc = DosCloseMutexSemEx(mtx_handles[i]);
			else
				break;
		}
		for (i = 0; i < semNum; i++)
			sem_pids[i] = NULL;

		return -1;
	}
}
/* Acquire or release in the semaphore pool */
int
semop(int semId, struct sembuf * sops, size_t nsops)
{
	os2_sem_set_hdr *the_set = (os2_sem_set_hdr *) MAKE_PTR(semId);
	HEV	   *sem_handles = (HEV *) ((off_t) the_set + the_set->m_semaphoreHandles);
	int		   *sem_counts = (int *) ((off_t) the_set + the_set->m_semaphoreCounts);
	HMTX	   *mtx_handles = (HMTX *) ((off_t) the_set + the_set->m_semaphoreMutex);
	pid_t	   *sem_pids = (pid_t *) ((off_t) the_set + the_set->m_semaphorePID);
	HEV		cur_handle;
	HMTX		mtx_handle;
	APIRET rc;
//	ULONG		postcount;
	char		sem_name[32];
	char		mtx_name[32];
	char		num_value[30];
	ULONG		postCount;
	ULONG		ulPostCount;
	PID     pidOwner = 0;
	TID     tidOwner = 0;


	if (nsops != 1)
	{
		/*
		 * Not supported (we return on 1st success, and don't cancel
		 * earlier ops)
		 */
		errno = E2BIG;
		return -1;
	}
	cur_handle = sem_handles[sops[0].sem_num];
	mtx_handle = mtx_handles[sops[0].sem_num];
	strcpy(mtx_name, the_set->set_mtxname);
	strcat(mtx_name, ".");
	strcat(mtx_name, _itoa(sops[0].sem_num, num_value, 10));
	strcpy(sem_name, the_set->set_semname);
	strcat(sem_name, ".");
	strcat(sem_name, _itoa(sops[0].sem_num, num_value, 10));

	rc = DosQueryMutexSem(mtx_handle, &pidOwner, &tidOwner, &postCount);
	if (rc != NO_ERROR)
	{
		rc = DosOpenMutexSemEx(mtx_name, &mtx_handle);
	}

	rc = DosRequestMutexSem(mtx_handle, (ULONG)SEM_INDEFINITE_WAIT);
	if (rc != NO_ERROR)
	{
		rc = DosOpenMutexSem(mtx_name, &mtx_handle);
		if (rc != NO_ERROR)
		{
			errno = EACCES;
			return -1;
		}
		rc = DosRequestMutexSem(mtx_handle, (ULONG)SEM_INDEFINITE_WAIT);
		if (rc != NO_ERROR)
		{
			errno = EACCES;
			return -1;
		}
	}
	rc = DosQueryEventSem(cur_handle, &postCount);
	if (rc != NO_ERROR)
		rc = DosOpenEventSem(sem_name, &cur_handle);

	sem_pids[sops[0].sem_num] = getpid();

	if (sops[0].sem_op == -1)
	{
		if (sem_counts[sops[0].sem_num] < 1)
		{
			if (sops[0].sem_flg & IPC_NOWAIT)
			{
				DosReleaseMutexSem(mtx_handle);
				errno = EAGAIN;
				return -1;
			}
			else
			{
				DosReleaseMutexSem(mtx_handle);
				rc = DosWaitEventSem(cur_handle, (sops[0].sem_flg & IPC_NOWAIT) ? SEM_IMMEDIATE_RETURN : SEM_INDEFINITE_WAIT );
				if (rc == NO_ERROR)
				{
					rc = DosRequestMutexSem(mtx_handle, (ULONG)SEM_INDEFINITE_WAIT);
					rc = DosResetEventSem(cur_handle, &ulPostCount);
					sem_counts[sops[0].sem_num]--;
					DosReleaseMutexSem(mtx_handle);
					return 0;
				}
				else if (rc == ERROR_TIMEOUT)
				{
					/* Couldn't get it */
					errno = EAGAIN;
				}
				else if (rc == ERROR_INTERRUPT)
				{
//					rc = DosResetEventSem(cur_handle, &ulPostCount);
					errno = EINTR;
				}
				else
				{
					errno = EIDRM;
				}
				return -1;
			}
		}
		else
		{
			sem_counts[sops[0].sem_num]--;
			DosResetEventSem(cur_handle, &ulPostCount);
			DosReleaseMutexSem(mtx_handle);
			return 0;
		}
	}
	if (sops[0].sem_op == 1)
	{
		rc = DosQueryEventSem(cur_handle, &postCount);
		if (postCount == 0)
		{
			sem_counts[sops[0].sem_num]++;
			rc = DosPostEventSem(cur_handle);
		}
		DosReleaseMutexSem(mtx_handle);
		return 0;
	}
	if (sops[0].sem_op == 0)
	{
		if (sem_counts[sops[0].sem_num] != 0)
		{
			if (sops[0].sem_flg & IPC_NOWAIT)
			{
				DosReleaseMutexSem(mtx_handle);
				errno = EAGAIN;
				return -1;
			}
			else
			{
				DosReleaseMutexSem(mtx_handle);
				rc = DosWaitEventSem(cur_handle, (sops[0].sem_flg & IPC_NOWAIT) ? SEM_IMMEDIATE_RETURN : SEM_INDEFINITE_WAIT );
				if (rc == NO_ERROR)
				{
					rc = DosResetEventSem(cur_handle, &ulPostCount);
					return 0;
				}
				else if (rc == ERROR_TIMEOUT)
				{
					/* Couldn't get it */
					errno = EAGAIN;
				}
				else if (rc == ERROR_INTERRUPT)
				{
//					rc = DosResetEventSem(cur_handle, &ulPostCount);
					errno = EINTR;
				}
				else
				{
					errno = EIDRM;
				}
				return -1;
			}
		}
		else
		{
			return 0;
		}
	}


	/* If we get down here, then something is wrong */
	return -1;
}
