IMPLEMENTATION MODULE TZmonCalcs;

        (********************************************************)
        (*                                                      *)
        (*       Manipulation of dates and times in order to    *)
        (*            deal with summer time adjustments.        *)
        (*                                                      *)
        (*   This is a specialised version for use by TZmon,    *)
        (*  pruned to where we can keep the software compact.   *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            22 September 2020               *)
        (*  Last edited:        30 September 2020               *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)


FROM SYSTEM IMPORT LOC;

IMPORT OS2;

FROM OS2 IMPORT
    (* type *)  PCSZ, DATETIME, APIRET,
    (* proc *)  DosScanEnv, DosGetDateTime, DosSetDateTime;

FROM INIData IMPORT
    (* type *)  HINI,
    (* proc *)  OpenINIFile, INIGet, CloseINIFile;

<* IF TZScreen THEN *>
    FROM STextIO IMPORT
        (* proc *)  WriteChar, WriteString, WriteLn;
<*END*>

(************************************************************************)

TYPE
    (* Encoded versions of some of the TZ parameters.  The values are   *)
    (*  0   month                                                       *)
    (*  1   week within month, where negative values mean counting back *)
    (*      from the end of the month.                                  *)
    (*  2   day of week, or day of month if value[1]=0.                 *)
    (*  3   seconds after midnight.                                     *)

    (* A variable of type RawDataType specifies a date/time when summer *)
    (* time starts or ends.                                             *)

    RawDataType = ARRAY [0..3] OF INTEGER;

    MonthData = ARRAY [1..13] OF CARDINAL;

CONST
    MinutesPerDay = 24*60;

    (* Days since beginning of year, for the 1st of each month.  In a   *)
    (* leap year you need an extra correction.                          *)

    FirstDayInMonth = MonthData {  0,  31,  59,  90, 120, 151,
                                 181, 212, 243, 273, 304, 334, 365};

    Feb28 = 31 + 27;   (* not 28, because we use zero-based numbering.  *)

VAR
    (* Raw TZ data. *)

    StartRaw, EndRaw: RawDataType;
    BaseTimeZone, SummerAdjust: INTEGER;

(************************************************************************)
(*                          SCREEN OUTPUT                               *)
(************************************************************************)

<* IF TZScreen THEN *>

PROCEDURE WriteCard1 (N: CARDINAL);

    (* Writes a one-digit number in decimal. *)

    BEGIN
        WriteChar (CHR(ORD('0')+N));
    END WriteCard1;

(************************************************************************)

PROCEDURE WriteCard2 (value: CARDINAL);

    (* Writes a two-digit number in decimal. *)

    BEGIN
        WriteCard1 (value DIV 10);
        WriteCard1 (value MOD 10);
    END WriteCard2;

(********************************************************************************)

PROCEDURE WriteCard (N: CARDINAL);

    (* Writes a number in decimal. *)

    BEGIN
        IF N > 9 THEN
            WriteCard (N DIV 10);
            N := N MOD 10;
        END (*IF*);
        WriteCard1 (N);
    END WriteCard;

(********************************************************************************)

PROCEDURE WriteTimeZone (value: INTEGER);

    (* Takes a value in minutes, writes it in +hhmm format.  Note that the      *)
    (* value we report here is actually the negative of the value that is used  *)
    (* by DosGetDateTime or by the TZ environment variable.                     *)

    BEGIN
        IF value < 0 THEN
            WriteChar ('+');  value := -value;
        ELSE
            WriteChar ('-');
        END (*IF*);
        WriteCard2 (value DIV 60);
        WriteCard2 (value MOD 60);
    END WriteTimeZone;

(************************************************************************)

PROCEDURE WriteDate (d: IDateTime);

    (* Writes the date to the screen.  The argument is in internal      *)
    (* date format, but the output is in terms of months and days.      *)

    BEGIN
        WriteCard (d.year);
        WriteString (" day ");
        WriteCard (d.day);
        WriteString (" minute ");
        WriteCard (d.minute);
    END WriteDate;

<*END*>

(************************************************************************)
(*                                                                      *)
(*                RAW DATA DERIVED FROM THE TZ STRING                   *)
(*                                                                      *)
(* We do not analyse the TZ string in this module.  Instead, we load    *)
(* values that another program has stored in an INI file.               *)
(*                                                                      *)
(************************************************************************)

PROCEDURE EVAL (f: ARRAY OF LOC);

    (* A do-nothing procedure - we use it for evaluating a function and *)
    (* ignoring the result.                                             *)

    BEGIN
    END EVAL;

(************************************************************************)

PROCEDURE LoadRawData;

    CONST INIFile = "CLOCK.INI";

    VAR hini: HINI;
        filename: ARRAY [0..15] OF CHAR;
        app: ARRAY [0..5] OF CHAR;

    BEGIN
        filename := INIFile;
        hini := OpenINIFile (filename);
        app := "TZRaw";
        EVAL (INIGet (hini, app, "StartRaw", StartRaw));
        EVAL (INIGet (hini, app, "EndRaw", EndRaw));
        EVAL (INIGet (hini, app, "BaseTimeZone", BaseTimeZone));
        EVAL (INIGet (hini, app, "SummerAdjust", SummerAdjust));
        CloseINIFile (hini);
    END LoadRawData;

(************************************************************************)
(*                      TIME ZONE AND ADJUSTMENT                        *)
(************************************************************************)

PROCEDURE ZoneInfo (VAR (*OUT*) BaseZone, Adjust: INTEGER);

    (* Returns the base time zone and summer adjustment in minutes. *)

    BEGIN
        BaseZone := BaseTimeZone;
        Adjust := SummerAdjust;
    END ZoneInfo;

(************************************************************************)
(*                   THE TIME ZONE AS STORED BY OS/2                    *)
(************************************************************************)

PROCEDURE UpdateTimeZone (value: INTEGER;  AdjustClock, EnteringSummer: BOOLEAN);

    (* Stores an updated value in the system's copy of the time zone,   *)
    (* and puts the clock forward or back as appropriate. Assumption:   *)
    (* this will never take us to a different day.                      *)

    (* We use two kinds of critical section protection here.  The       *)
    (* priority setting reduces the likelihood of clock changes by      *)
    (* some other program.  The "must complete" protection ensures that *)
    (* we don't get interrupted by a process shutdown.                  *)

    VAR dt: DATETIME;  rc: APIRET;  nesting: CARDINAL;
        newval: INTEGER;

    BEGIN
        rc := OS2.DosEnterMustComplete (nesting);
        rc := OS2.DosSetPriority (OS2.PRTYS_THREAD,
                                    OS2. PRTYC_TIMECRITICAL, 0, 0);
        OS2.DosGetDateTime (dt);
        IF (dt.timezone <> value) OR AdjustClock THEN
            dt.timezone := value;
            IF AdjustClock THEN
                <* IF TZScreen THEN *>
                    WriteString ("Adjusting clock");  WriteLn;
                <* END *>
                newval := VAL(INTEGER,dt.minutes);
                IF EnteringSummer THEN
                    INC (newval, SummerAdjust);
                ELSE
                    DEC (newval, SummerAdjust);
                END (*IF*);
                IF newval >= 60 THEN
                    INC (dt.hours);
                    DEC (newval, 60);
                ELSIF newval <= -60 THEN
                    DEC (dt.hours);
                    INC (newval, 60);
                END (*IF*);
                dt.minutes := CHR(newval);
            END (*IF*);
            OS2.DosSetDateTime (dt);
        END (*IF*);
        rc := OS2.DosSetPriority (OS2.PRTYS_THREAD,
                                    OS2. PRTYC_REGULAR, 0, 0);
        rc := OS2.DosExitMustComplete (nesting);
    END UpdateTimeZone;

(************************************************************************)

PROCEDURE DTtoIDT (dt: DATETIME): IDateTime;

    (* Converts from the OS/2 representation to our internal    *)
    (* representation.  Seconds are ignored.                    *)

    VAR result: IDateTime;  LeapYear: BOOLEAN;

    BEGIN
        result.year := dt.year;
        result.day := FirstDayInMonth[ORD(dt.month)] + ORD(dt.day) - 1;
        result.minute := 60*ORD(dt.hours) + ORD(dt.minutes);
        LeapYear := (dt.year MOD 4) = 0;
        IF LeapYear AND (ORD(dt.month) > 2) THEN
            INC (result.day);
        END (*IF*);
        result.weekday := ORD(dt.weekday);
        RETURN result;
    END DTtoIDT;

(************************************************************************)

PROCEDURE CurrentDateTime(): IDateTime;

    (* Returns today's date and time.  *)

    VAR dt: DATETIME;

    BEGIN
        DosGetDateTime (dt);
        RETURN DTtoIDT (dt);
    END CurrentDateTime;

(************************************************************************)
(*           DATES AND TIMES IN OUR INTERNAL REPRESENTATION             *)
(************************************************************************)

PROCEDURE SetDayOfWeek (VAR (*INOUT*) d: IDateTime);

    (* Calculates d.weekday, on the assumption that d.year and d.day    *)
    (* are already set correctly.                                       *)

    CONST BaseDay = 6;

    VAR FirstDayOfYear: DayOfWeek;

    BEGIN
        (* Every group of four years has 4*365+1 = 1461 days, and       *)
        (* 1461 MOD 7 = 5.  This means that the DayOfWeek changes by    *)
        (* 5 days per 4 years.                                          *)

        FirstDayOfYear := (BaseDay + 5*(d.year DIV 4)) MOD 7;

        (* Thereafter, it changes by 2 days in the first year, and one  *)
        (* day per year after that.                                     *)

        IF d.year MOD 4 <> 0 THEN
            FirstDayOfYear := (FirstDayOfYear + (d.year MOD 4) + 1) MOD 7;
        END (*IF*);

        d.weekday := (FirstDayOfYear + d.day - 1) MOD 7;

    END SetDayOfWeek;

(************************************************************************)

PROCEDURE IsEqual (d1, d2: IDateTime): BOOLEAN;

    (* Returns TRUE iff d1 and d2 have the same year/day/minute values. *)

    BEGIN
        RETURN (d1.year = d2.year) AND (d1.day = d2.day)
                        AND (d1.minute = d2.minute);
    END IsEqual;

(************************************************************************)

PROCEDURE IsLater (d1, d2: IDateTime): BOOLEAN;

    (* Returns TRUE iff d1 > d2. *)

    VAR result: BOOLEAN;

    BEGIN
        IF d1.year > d2.year THEN result := TRUE
        ELSIF d1.year < d2.year THEN result := FALSE
        ELSIF d1.day > d2.day THEN result := TRUE
        ELSIF d1.day < d2.day THEN result := FALSE
        ELSIF d1.minute > d2.minute THEN result := TRUE
        ELSE result := FALSE;
        END (*IF*);
        RETURN result;
    END IsLater;

(************************************************************************)
(*                                                                      *)
(*            CONVERSION FROM RAW DATA TO IDateTime FORMAT              *)
(*                                                                      *)
(*   This is where we calculate a time where summer starts or ends.     *)
(*                                                                      *)
(************************************************************************)

PROCEDURE SetSwitchTime (thisyear: CARDINAL;
                            raw: RawDataType;
                             VAR (*OUT*) result: IDateTime);

    (* Converts a date/time in RawData form to IDateTime form.  *)

    VAR TargetDay: DayOfWeek;  val, weekcode: INTEGER;  IsLeapYear: BOOLEAN;

    BEGIN
        (* Set the year in the result. *)

        result.year := thisyear;
        IsLeapYear := (thisyear MOD 4) = 0;

        (* The first of the four numbers is a month. *)

        result.day := FirstDayInMonth[raw[0]];
        IF IsLeapYear AND (raw[0] > 2) THEN
            INC (result.day);
        END (*IF*);

        (* The second and third numbers give the day in the month,  *)
        (* but it's in a non-obvious format.  The third number is a *)
        (* day of the week (0 = Sunday).  The second, if it's       *)
        (* positive, says whether it's the first, second, etc.      *)
        (* occurrence of that day in the month.  If it's negative,  *)
        (* we count backwards from the end of the month.  If it's   *)
        (* zero, then the third number is an actual day of the      *)
        (* month, i.e. it's not a day-within-week.                  *)

        weekcode := raw[1];
        IF weekcode = 0 THEN
            val := raw[2]; (* supposed to be an actual day of month *)
            IF val < 1 THEN

                (* Should never happen, but it pays to be paranoid  *)
                (* given the possibility of malformed TZ strings.   *)

                IF VAL (INTEGER, result.day) + val < 1 THEN
                    result.day := 0;
                ELSE
                    DEC (result.day, 1 - val);
                END (*IF*);

            ELSIF val > 1 THEN
                INC (result.day, val - 1);
            END (*IF*);
            SetDayOfWeek (result);
        ELSE
            SetDayOfWeek (result);
            TargetDay := ABS(raw[2]);
            IF weekcode > 0 THEN

                (* Counting from first day of month. *)

                WHILE result.weekday <> TargetDay DO
                    INC (result.day);
                    result.weekday := (result.weekday + 1) MOD 7;
                END (*WHILE*);
                DEC (weekcode);
                INC (result.day, 7*ABS(weekcode));
            ELSE

                (* Counting from last day of month. *)

                INC (result.day, FirstDayInMonth[raw[0]+1]
                           - FirstDayInMonth[raw[0]] - 1);
                IF IsLeapYear AND (raw[0] = 2) THEN
                    INC (result.day);
                END (*IF*);
                SetDayOfWeek (result);

                WHILE result.weekday <> TargetDay DO
                    DEC (result.day);
                    IF result.weekday = 0 THEN result.weekday := 6
                    ELSE DEC(result.weekday)
                    END (*IF*);
                END (*WHILE*);
                INC (weekcode);
                DEC (result.day, 7*ABS(weekcode));

            END (*IF*);
        END (*IF*);

        (* raw[3] should never be negative, but a subsidiary module *)
        (* has already checked this.                                *)

        result.minute := (raw[3] + 30) DIV 60;
        IF result.minute >= MinutesPerDay THEN
            result.minute := MinutesPerDay - 1;
        END (*IF*);

    END SetSwitchTime;

(************************************************************************)
(*                          MAIN CALCULATION                            *)
(************************************************************************)

PROCEDURE FindSummerStartAndEnd (VAR (*OUT*) SummerStart, SummerEnd: IDateTime;
                                      VAR (*OUT*) InSummerTime: BOOLEAN);

    (* Sets the values of the variables that define the summer start    *)
    (* and end date.                                                    *)

    VAR Now: IDateTime;
        newTZ: INTEGER;
        SummerHasStarted, SummerIsNotOver: BOOLEAN;

    BEGIN
        (* Report what OS/2 currently has recorded. *)

        Now := CurrentDateTime();
        <* IF TZScreen THEN *>
            WriteString ("It is now ");  WriteDate (Now);  WriteLn;
            WriteString ("Your base time zone is ");
            WriteTimeZone (BaseTimeZone);  WriteLn;
        <* END *>

        SetSwitchTime (Now.year, StartRaw, SummerStart);
        SetSwitchTime (Now.year, EndRaw, SummerEnd);

        IF IsEqual (SummerStart, SummerEnd) THEN
            <* IF TZScreen THEN *>
                WriteString ("Summer time adjustments are disabled.");
                WriteLn;
            <* END *>
            SummerAdjust := 0;
        ELSE

            SummerHasStarted := NOT IsLater (SummerStart, Now);
            SummerIsNotOver := IsLater (SummerEnd, Now);

            IF IsLater (SummerStart, SummerEnd) THEN

                (* Southern hemisphere *)

                InSummerTime := SummerHasStarted OR SummerIsNotOver;

            ELSE

                (* Northern hemisphere *)

                InSummerTime := SummerHasStarted AND SummerIsNotOver;

            END (*IF*);

            <* IF TZScreen THEN *>
                WriteString ("You are now on ");
                IF InSummerTime THEN
                    WriteString ("summer");
                ELSE
                    WriteString ("winter");
                END (*IF*);
                WriteString (" time");  WriteLn;
            <* END *>

            (* Make the summer time correction, if applicable. *)

            newTZ := BaseTimeZone;
            IF InSummerTime THEN
                DEC (newTZ, SummerAdjust);
            END (*IF*);
            UpdateTimeZone (newTZ, FALSE, InSummerTime);

            (* Report the final time zone setting. *)

            <* IF TZScreen THEN *>
                WriteString ("Time zone has been set to ");
                WriteTimeZone (newTZ);  WriteLn;
            <* END *>

            (* Up to this point, SummerTimeStart and SummerTimeEnd are  *)
            (* dates in this year.  Often they need to be in the next   *)
            (* year, in which case we need to redo the calculation.     *)
            (* It is not good enough to simply add a year, because for  *)
            (* example the last Sunday in a month can be on a very      *)
            (* different date from year to year.                        *)

            IF SummerHasStarted THEN
                SetSwitchTime (Now.year+1, StartRaw, SummerStart);
            END (*IF*);
            IF NOT SummerIsNotOver THEN
                SetSwitchTime (Now.year+1, EndRaw, SummerEnd);
            END (*IF*);

            <* IF TZScreen THEN *>
                IF InSummerTime THEN
                    WriteString ("Summer time  ends  at ");
                    WriteDate (SummerEnd);  WriteLn;
                END (*IF*);
                WriteString ("Summer time starts at ");
                WriteDate (SummerStart);  WriteLn;
                IF NOT InSummerTime THEN
                    WriteString ("Summer time  ends  at ");
                    WriteDate (SummerEnd);  WriteLn;
                END (*IF*);
            <* END *>

        END (*IF*);

    END FindSummerStartAndEnd;

(************************************************************************)

BEGIN
    LoadRawData;
END TZmonCalcs.

