MODULE TZmon;

        (********************************************************)
        (*                                                      *)
        (*       Program to update the OS/2 system clock        *)
        (*         on a switch to or from summer time.          *)
        (*                                                      *)
        (*   Most of the time this program is doing nothing,    *)
        (*   but it wakes up every so often to see whether a    *)
        (*   change in time zone offset is needed.              *)
        (*                                                      *)
        (*   NOTE: the leap year calculations in this module    *)
        (*   are valid only until the year 2099.                *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        1 October 2020                  *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)


FROM SYSTEM IMPORT CAST, LOC, ADDRESS;

IMPORT OS2;

<* IF TZScreen THEN *>
    FROM STextIO IMPORT
        (* proc *)  WriteChar, WriteString, WriteLn;
<* END *>

FROM TZmonCalcs IMPORT
    (* type *)  IDateTime,
    (* proc *)  DTtoIDT, ZoneInfo, FindSummerStartAndEnd, UpdateTimeZone;

(************************************************************************)
(*                               CONSTANTS                              *)
(************************************************************************)

CONST
    version = "01";

    STACKSIZE = 8192;

    Feb28 = 31 + 27;   (* not 28, because we use zero-based numbering.  *)

(************************************************************************)
(*                           GLOBAL VARIABLES                           *)
(************************************************************************)

VAR
    (* When summer time next starts and ends. *)

    SummerTimeStart, SummerTimeEnd: IDateTime;

    (* An event semaphore that controls the timing of the main thread.  *)

    MainSem: OS2.HEV;

    (* A flag to tell the main thread to exit. *)

    Shutdown: BOOLEAN;

(************************************************************************)
(*                              SCREEN OUTPUT                           *)
(*                  which will ultimately be redundant                  *)
(************************************************************************)

<* IF TZScreen THEN *>

PROCEDURE WriteCard (N: CARDINAL);

    BEGIN
        IF N > 9 THEN
            WriteCard (N DIV 10);
            N := N MOD 10;
        END (*IF*);
        WriteChar (CHR(ORD('0')+N));
    END WriteCard;

PROCEDURE WriteInt (x: INTEGER);

    BEGIN
        IF x < 0 THEN
            WriteChar ('-');
            x := -x;
        END (*IF*);
        WriteCard (x);
    END WriteInt;

<* END *>

(************************************************************************)
(*                    MISCELLANEOUS USEFUL FUNCTIONS                    *)
(************************************************************************)

PROCEDURE EVAL (f: ARRAY OF LOC);

    (* A do-nothing procedure - we use it for evaluating a function and *)
    (* ignoring the result.                                             *)

    BEGIN
        f[0] := f[0];           (* to suppress a compiler warning *)
    END EVAL;

(************************************************************************)

PROCEDURE DiffMinutes (VAR (*IN*) from, to: IDateTime): INTEGER;

    (* Returns (to - from), in minutes.  Assumption: the dates are      *)
    (* either in the same year, or to is in the year after from.        *)

    VAR result: INTEGER;

    BEGIN
        result := 0;
        IF to.year > from.year THEN
            result := 365;

            (* Increment the result if either from is a leap year and   *)
            (* from.day is before Feb 29, or to is a leap year and      *)
            (* to.day is after Feb 28.  Because we assuming at most a   *)
            (* one-year gap, only one of these conditions can hold.     *)

            IF ((from.year MOD 4 = 0) AND (from.day <= Feb28))
                       OR ((to.year MOD 4 = 0) AND (to.day > Feb28)) THEN
                result := 366;
            END (*IF*);

        END (*IF*);
        result := result + CAST(INTEGER,to.day) - CAST(INTEGER,from.day);
        result := 60*24*result
                  + CAST(INTEGER,to.minute) - CAST(INTEGER,from.minute);
        RETURN result;
    END DiffMinutes;

(************************************************************************)

PROCEDURE DiffSeconds (VAR (*IN*) from: OS2.DATETIME;  to: IDateTime): INTEGER;

    (* Returns (to - from), in seconds.  Assumption: the dates are      *)
    (* either in the same year, or to is in the year after from.        *)

    VAR start: IDateTime;

    BEGIN
        start := DTtoIDT (from);
        RETURN 60*DiffMinutes (start, to) - VAL(INTEGER, from.seconds);
    END DiffSeconds;

(************************************************************************)

PROCEDURE SecsToNext (WeAreInSummerTime: BOOLEAN): CARDINAL;

    (* Returns the time until the next correction is due, in seconds. *)

    VAR delay: INTEGER;
        SecsToGo: CARDINAL;
        dt: OS2.DATETIME;
        Now: IDateTime;
        <* IF TZScreen THEN *>
            val: CARDINAL;
        <* END *>

    BEGIN
        OS2.DosGetDateTime (dt);
        Now := DTtoIDT (dt);
        IF WeAreInSummerTime THEN
            delay := DiffSeconds (dt, SummerTimeEnd);
        ELSE
            delay := DiffSeconds (dt, SummerTimeStart);
        END (*IF*);
        IF delay < 0 THEN
            delay := 0;
        END (*IF*);
        SecsToGo := ABS(delay);
        <* IF TZScreen THEN *>
            val := SecsToGo;
            IF val < 60 THEN
                WriteCard (val);
                WriteString (" seconds");
            ELSE
                val := (val + 30) DIV 60;
                IF val < 180 THEN
                    WriteCard (val);
                    WriteString (" minutes");
                ELSE
                    val := (val + 30) DIV 60;
                    IF val < 72 THEN
                        WriteCard (val);
                        WriteString (" hours");
                    ELSE
                        val := (val + 12) DIV 24;
                        WriteCard (val);
                        WriteString (" days");
                    END (*IF*);
                END (*IF*);
            END (*IF*);
            WriteString (" until next time zone change.");  WriteLn;
        <* END *>

        RETURN SecsToGo;

    END SecsToNext;

(************************************************************************)

PROCEDURE SecsToms (seconds: INTEGER): INTEGER;

    (* Converts seconds to milliseconds, with saturation if this would  *)
    (* cause arithmetic overflow.                                       *)

    VAR result: INTEGER;

    BEGIN
        IF seconds > MAX(INTEGER)/1000 THEN
            result := MAX(INTEGER);
        ELSIF seconds < MIN(INTEGER)/1000 THEN
            result := MIN(INTEGER);
        ELSE
            result := 1000*seconds;
        END (*IF*);

        RETURN result;

     END SecsToms;

(************************************************************************)

PROCEDURE DoSetting (BaseTimeZone, SummerAdjust: INTEGER;
                            transition, WeAreInSummerTime: BOOLEAN): INTEGER;

    (* Sets the time zone, returns the seconds to the next adjustment. *)
    (* If transition=TRUE, this is a switch between summer and winter time. *)

    BEGIN
        <* IF TZScreen THEN *>
            WriteString ("Entering DoSetting");  WriteLn;
        <* END *>
        IF WeAreInSummerTime THEN
            UpdateTimeZone (BaseTimeZone - SummerAdjust, transition, WeAreInSummerTime);
        ELSE
            UpdateTimeZone (BaseTimeZone, transition, WeAreInSummerTime);
        END (*IF*);

        RETURN SecsToNext(WeAreInSummerTime);

    END DoSetting;

(************************************************************************)
(*                 PROVISION TO SHUT DOWN THE PROGRAM                   *)
(************************************************************************)

PROCEDURE ["SysCall"] BreakHandler (p1: OS2.PEXCEPTIONREPORTRECORD;
                                     p2: OS2.PEXCEPTIONREGISTRATIONRECORD;
                                     bbbp3: OS2.PCONTEXTRECORD;
                                     p4: ADDRESS): CARDINAL;

    (* This handler shuts down the program when an XCPT_SIGNAL is received.  *)

    BEGIN
        EVAL(p2);
        EVAL (bbbp3);           (* to remove a compiler warning *)
        EVAL (p4);              (* to remove a compiler warning *)
        IF p1^.ExceptionNum = OS2.XCPT_SIGNAL THEN
            <* IF TZScreen THEN *>
                WriteString ("Shutting down.");
                WriteLn;
            <* END *>
            Shutdown := TRUE;
            OS2.DosPostEventSem (MainSem);
            RETURN OS2.XCPT_CONTINUE_EXECUTION;
        ELSE
            RETURN OS2.XCPT_CONTINUE_SEARCH;
        END (*IF*);
    END BreakHandler;

(************************************************************************)

PROCEDURE [OS2.APIENTRY] TaskKiller (dummy: CARDINAL);

    (* A separate thread, to shut down this program if requested.  The  *)
    (* goal is to prevent having two copies of the program running.     *)

    VAR semName: ARRAY [0..127] OF CHAR;
        ShutdownSignal: OS2.HEV;

    BEGIN
        IF dummy = 0 THEN END(*IF*);  (* to suppress a compiler warning *)

        semName := "\SEM32\TZSET_TERMINATE";
        ShutdownSignal := 0;
        WHILE OS2.DosOpenEventSem (semName, ShutdownSignal)
                                            <> OS2.ERROR_SEM_NOT_FOUND DO

            (* Another instance of this program must be running.  Kill  *)
            (* it, and don't proceed until we know that that other      *)
            (* instance has closed the semaphore.                       *)

            OS2.DosPostEventSem (ShutdownSignal);
            OS2.DosSleep (500);
        END (*WHILE*);

        OS2.DosCreateEventSem (semName, ShutdownSignal, 0, FALSE);

        (* Wait for a signal on the shutdown semaphore.    *)

        WHILE NOT Shutdown DO
            OS2.DosWaitEventSem (ShutdownSignal, OS2.SEM_INDEFINITE_WAIT);

            <* IF TZScreen THEN *>
                WriteString ("Program terminated by another instance of TZmon");
                WriteLn;
            <* END *>

            Shutdown := TRUE;
            OS2.DosPostEventSem (MainSem);
        END (*WHILE*);

        OS2.DosResetEventSem (ShutdownSignal, dummy);
        OS2.DosCloseEventSem (ShutdownSignal);

    END TaskKiller;

(************************************************************************)
(*                             MAIN PROGRAM                             *)
(************************************************************************)

PROCEDURE RunTheOperation;

    (* Parses the TZ environment variable, calculates summer time start *)
    (* and end, sets the timezone field in the system DATETIME record.  *)
    (* If there is no summer time, exits.  Otherwise, loops forever,    *)
    (* asleep most of the time, but waking up each time there is a      *)
    (* time zone change.                                                *)

    VAR RegRec: OS2.EXCEPTIONREGISTRATIONRECORD;
        rc: OS2.APIRET;
        delaytime: INTEGER;
        BaseTimeZone, SummerAdjust: INTEGER;
        WeAreInSummerTime: BOOLEAN;

    BEGIN
        (* Initial setting. *)

        ZoneInfo (BaseTimeZone, SummerAdjust);
        IF SummerAdjust = 0 THEN
            <* IF TZScreen THEN *>
                WriteString ("No further adjustments needed, exiting program.");
                WriteLn;
            <* END *>
            RETURN;
        END (*IF*);

        FindSummerStartAndEnd (SummerTimeStart, SummerTimeEnd,
                                               WeAreInSummerTime);

        <* IF TZScreen THEN *>
            WriteString ("Doing initial setting.");
            WriteLn;
        <* END *>

        delaytime := SecsToms(DoSetting(BaseTimeZone, SummerAdjust, FALSE, WeAreInSummerTime));

        (* Add our break handler to this thread's exception handler chain. *)

        RegRec.prev_structure := NIL;
        RegRec.ExceptionHandler := BreakHandler;
        rc := OS2.DosSetExceptionHandler (RegRec);

        <* IF TZScreen THEN *>
            WriteString ("Going to sleep until next time zone update needed.");
            WriteLn;
        <* END *>

        (* Go into an almost-never-ending loop, sleeping until the  *)
        (* next change is due.                                      *)

        WHILE NOT Shutdown DO
            rc := OS2.DosWaitEventSem (MainSem, delaytime);
            IF NOT Shutdown THEN
                delaytime := SecsToms (SecsToNext (WeAreInSummerTime));
                <* IF TZScreen THEN *>
                    WriteString ("delaytime = ");  WriteInt (delaytime);
                    WriteLn;
                <* END *>
                IF delaytime <= 0 THEN

                    <* IF TZScreen THEN *>
                        WriteString ("Zero or negative delay time.");
                        WriteLn;
                    <* END *>

                    (* It is time to switch from winter to summer   *)
                    (* time, or vice versa.                         *)

                    WeAreInSummerTime := NOT WeAreInSummerTime;
                    delaytime := SecsToms(DoSetting(BaseTimeZone, SummerAdjust,
                                                TRUE, WeAreInSummerTime));

                END (*IF*);

            END (*IF*);

        END (*WHILE*);

     END RunTheOperation;

(************************************************************************)

VAR tid: OS2.TID;
    rc: OS2.APIRET;

BEGIN
    Shutdown := FALSE;
    <* IF TZScreen THEN *>
        WriteString ("TZmon version ");  WriteString (version);  WriteLn;
    <* END *>
    rc := OS2.DosCreateEventSem (NIL, MainSem, 0, FALSE);
    rc := OS2.DosCreateThread (tid, TaskKiller, 0, 0, STACKSIZE);
    RunTheOperation;
FINALLY
    rc := OS2.DosCloseEventSem (MainSem);
END TZmon.

