/* Icon Plus
 * Copyright (C) 2001 Dmitry A.Steklenev
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR CONTRIBUTORS "AS IS"
 * AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR OR THE CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ic_palette.h,v 1.4 2001/05/31 05:39:08 glass Exp $
 */

#ifndef IC_PALETTE_H
#define IC_PALETTE_H

#include "ic_config.h"

/**
 * Icon Plus generic RGB color class
 * @author Dmitry A Steklenev
 */

class ICRGB {

  public:
    /* Constructs the black RGB color */
    ICRGB();
    /* Constructs the RGB color */
    ICRGB( BYTE r, BYTE g, BYTE b );

    /* Converts the RGB color to unsigned long */
    operator unsigned long() const;

    /** Compares this color to another color object for equality   */
    int operator==( const ICRGB& ) const;
    /** Compares this color to another color object for inequality */
    int operator!=( const ICRGB& ) const;

  public:
    static const ICRGB white;
    static const ICRGB black;

  public:
    BYTE blue;
    BYTE green;
    BYTE red;
};

/*------------------------------------------------------------------
 * Constructs the black RGB color
 *------------------------------------------------------------------*/
inline ICRGB::ICRGB()

: red  (0),
  green(0),
  blue (0)
{}

/*------------------------------------------------------------------
 * Constructs the RGB color
 *------------------------------------------------------------------*/
inline ICRGB::ICRGB( BYTE r, BYTE g, BYTE b )

: red  (r),
  green(g),
  blue (b)
{}

/*------------------------------------------------------------------
 * Converts the RGB color to unsigned long
 *------------------------------------------------------------------*/
inline ICRGB::operator unsigned long() const
{
  return blue | (green << 8) | (red << 16);
}

/*------------------------------------------------------------------
 * Compares this color to another color object for equality
 *------------------------------------------------------------------*/
inline int ICRGB::operator==( const ICRGB& color ) const
{
  return red   == color.red   &&
         green == color.green &&
         blue  == color.blue;
}

/*------------------------------------------------------------------
 * Compares this color to another color object for inequality
 *------------------------------------------------------------------*/
inline int ICRGB::operator!=( const ICRGB& color ) const
{
  return red   != color.red   ||
         green != color.green ||
         blue  != color.blue;
}

/**
 * Icon Plus generic color palette class
 *
 * The ICPalette class creates and manages a RGB color palette.
 * @author Dmitry A Steklenev
 */

class ICPalette
{
  public:
    /** An unsigned integral type */
    typedef unsigned long size_type;

    /** The largest possible value of type size_type. */
    static const size_type npos;

  public:
    /** Constructs the empty color palette */
    ICPalette();
    /** Constructs the color palette */
    ICPalette( size_type size   );
    /** Constructs the color palette from another palette object */
    ICPalette( const ICPalette& );
    /** Destructs the color palette */
   ~ICPalette();

  public:
    /** Returns the number of RGB colors stored in the palette */
    size_type count() const;
    /** Returns the RGB color index */
    size_type find( const ICRGB& ) const;
    /** Returns true (1) if palette is in truecolor mode */
    int is_truecolor() const;

  public:
    /** Assignment operator */
    ICPalette& operator=( const ICPalette& );
    /** Returns a reference to the RGB color */
    ICRGB& operator[]( size_type pos );
    /** Returns a constant reference to the RGB color */
    const ICRGB& operator[]( size_type pos ) const;
    /** Compares this palette to another palette object for equality   */
    int operator==( const ICPalette& ) const;
    /** Compares this palette to another palette object for inequality */
    int operator!=( const ICPalette& ) const;

  public:
    /** Returns standard black and white color palette */
    static ICPalette default_bw();

  private:
    ICRGB*    pal_list;
    size_type pal_size;
    int       pal_true;
};

/*------------------------------------------------------------------
 * Returns the number of RGB colors
 *------------------------------------------------------------------*/
inline ICPalette::size_type ICPalette::count() const
{
  return pal_size;
}

/*------------------------------------------------------------------
 * Returns true if the palette is in truecolor mode
 *------------------------------------------------------------------*/
inline int ICPalette::is_truecolor() const
{
  return pal_true;
}

/*------------------------------------------------------------------
 * Returns a reference to the RGB color
 *------------------------------------------------------------------*/
inline ICRGB& ICPalette::operator[]( size_type pos )
{
  return pal_list[pos];
}

/*------------------------------------------------------------------
 * Returns a constant reference to the RGB color
 *------------------------------------------------------------------*/
inline const ICRGB& ICPalette::operator[]( size_type pos ) const
{
  return pal_list[pos];
}

#endif
