/* Icon Plus
 * Copyright (C) 2001 Dmitry A.Steklenev
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Dmitry A.Steklenev".
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR OR CONTRIBUTORS "AS IS"
 * AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * AUTHOR OR THE CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ic_icon.h,v 1.3 2001/05/30 10:04:35 glass Exp $
 */

#ifndef IC_ICON_H
#define IC_ICON_H

#include "ic_palette.h"
#include "ic_bitmap.h"

/**
 * Icon Plus generic icon class
 *
 * The ICIcon class creates and manages a generic icon.
 * <p>
 * The following table shows how these bitmaps are used for a
 * color icon or pointer:
 * <pre>
 *   AND XOR COLOR
 *    1   1    x   Invert screen
 *    1   0    x   Transparency
 *    0   0    x   Use color x
 *    0   1    x   Use color x

 * </pre>
 *
 * @author Dmitry A Steklenev
 */

class ICIcon
{
  public:
    /** An unsigned integral type */
    typedef UINT32 size_type;

  public:
    /** Constructs the empty generic icon */
    ICIcon();
    /** Constructs the generic icon */
    ICIcon( size_type width, size_type height, size_type bits_per_pixel );
    /** Constructs the generic icon from another icon object */
    ICIcon( const ICIcon& );
    /** Destructs the generic icon */
   ~ICIcon();

  public:
    /** Returns the icon width in pixels */
    size_type width() const;
    /** Returns the icon height in pixels */
    size_type height() const;
    /** Returns the icon of bits per pixel */
    size_type bits_per_pixel() const;

    /** Returns a reference to the icon bitmap */
    ICBitmap&  bitmap () { return ico_clr_map; }
    /** Returns a reference to the icon bitmap palette */
    ICPalette& palette() { return ico_clr_pal; }
    /** Returns a reference to the icon "and" mask */
    ICBitMask& andmap () { return ico_and_map; }
    /** Returns a reference to the icon "xor" mask */
    ICBitMask& xormap () { return ico_xor_map; }

    /** Returns the width of icon hotspot */
    size_type  xhotspot() const;
    /** Returns the height of icon hotspot */
    size_type  yhotspot() const;
    /** Sets the icon hotspot */
    void move_hotspot( size_type x, size_type y );

    /** Returns true if this icon is black and white */
    int is_bw() const;

  public:
    /** Assignment operator */
    ICIcon& operator=( const ICIcon& );

  private:
    ICBitmap  ico_clr_map;
    ICPalette ico_clr_pal;
    ICBitMask ico_xor_map;
    ICBitMask ico_and_map;

    size_type ico_xhotspot;
    size_type ico_yhotspot;
};


/*------------------------------------------------------------------
 * Returns the icon width in pixels
 *------------------------------------------------------------------*/
inline ICIcon::size_type ICIcon::width() const
{
  return ico_clr_map.width();
}

/*------------------------------------------------------------------
 * Returns the icon height in pixels
 *------------------------------------------------------------------*/
inline ICIcon::size_type ICIcon::height() const
{
  return ico_clr_map.height();
}

/*------------------------------------------------------------------
 * Returns the number of bits per pixel
 *------------------------------------------------------------------*/
inline ICIcon::size_type ICIcon::bits_per_pixel() const
{
  return ico_clr_map.bits_per_pixel();
}

/*------------------------------------------------------------------
 * Returns the width of icon hotspot
 *------------------------------------------------------------------*/
inline ICIcon::size_type ICIcon::xhotspot() const
{
  return ico_xhotspot;
}

/*------------------------------------------------------------------
 * Returns the height of icon hotspot
 *------------------------------------------------------------------*/
inline ICIcon::size_type ICIcon::yhotspot() const
{
  return ico_yhotspot;
}

/*------------------------------------------------------------------
 * Sets the icon hotspot
 *------------------------------------------------------------------*/
inline void ICIcon::move_hotspot( size_type x, size_type y )
{
  ico_xhotspot = x;
  ico_yhotspot = y;
}

/*------------------------------------------------------------------
 * Returns true if this icon is black and white
 *------------------------------------------------------------------*/
inline int ICIcon::is_bw() const
{
  return ico_clr_pal == ICPalette::default_bw();
}

/**
 * Icon Plus generic icons list
 *
 * The ICIconList class creates and manages a generic icons list.
 * @author Dmitry A Steklenev
 */

class ICIconList
{
  public:
    /** An unsigned integral type */
    typedef unsigned long size_type;

  public:
    /** Constructs the icons list */
    ICIconList();
    /** Destructs the icons list */
   ~ICIconList();

  public:
    static ICIconList icon_list;

  public:
    /** Iterator used to iterate over the icons. */
    typedef ICIcon* iterator;
    /** Returns an iterator pointing to the beginning of the icons list */
    iterator  begin() const;
    /** Returns an iterator pointing to the end of the icons list */
    iterator  end  () const;
    /** Returns the number of icons stored in the list */
    size_type count() const;

    /** Inserts the icon as the last item in the list */
    void push_back( const ICIcon& icon );
    /** Removes all icons from the list */
    void clear();

  private:
    ICIcon*   lst_icons;
    size_type lst_size;
};

/*------------------------------------------------------------------
 * Returns an iterator pointing to the beginning of the icons list
 *------------------------------------------------------------------*/
inline ICIconList::iterator ICIconList::begin() const
{
  return lst_icons;
}

/*------------------------------------------------------------------
 * Returns an iterator pointing to the end of the icons list
 *------------------------------------------------------------------*/
inline ICIconList::iterator ICIconList::end() const
{
  return lst_icons + lst_size;
}

/*------------------------------------------------------------------
 * Returns the number of icons stored in the list
 *------------------------------------------------------------------*/
inline ICIconList::size_type ICIconList::count() const
{
  return lst_size;
}

#endif
