/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import info.clearthought.layout.TableLayout;
import info.clearthought.layout.TableLayoutConstraints;
import java.awt.*;
import com.jformdesigner.model.FormLayoutConstraints;
import com.jformdesigner.model.FormLayoutManager;

/**
 * @author Karl Tauber
 */
public class TableLayoutCreator
	extends AbstractLayoutCreator
{
	private static final String PROP_COLUMN_SPECS = "$columnSpecs";
	private static final String PROP_ROW_SPECS = "$rowSpecs";

	private TableSpec[] columnSpecs;
	private TableSpec[] rowSpecs;

	public LayoutManager createLayoutManager( Container container, FormLayoutManager formLayout )
		throws InstantiationException, IllegalAccessException
	{
		String encodedColumnSpecs = formLayout.getPropertyString( PROP_COLUMN_SPECS, "" );
		String encodedRowSpecs = formLayout.getPropertyString( PROP_ROW_SPECS, "" );
		int hGap = formLayout.getPropertyInt( "hGap", 0 );
		int vGap = formLayout.getPropertyInt( "vGap", 0 );

		columnSpecs = TableSpec.decodeSpecs( encodedColumnSpecs );
		rowSpecs = TableSpec.decodeSpecs( encodedRowSpecs );

		double[] col = toSizes( columnSpecs );
		double[] row = toSizes( rowSpecs );

		TableLayout layout = new TableLayout( col, row );
		layout.setHGap( hGap );
		layout.setVGap( vGap );
		return layout;
	}

	private double[] toSizes( TableSpec[] specs ) {
		double[] sizes = new double[specs.length];
		for( int i = 0; i < specs.length; i++ )
			sizes[i] = specs[i].getSize();
		return sizes;
	}

	@Override
	public void addComponentToContainer( Container container, Component component,
		Object constraints, int index, FormLayoutConstraints formConstraints )
	{
		constraints = updateConstraints( constraints, columnSpecs, rowSpecs );
		container.add( component, constraints, index );
	}

	public static Object updateConstraints( Object constraints,
		TableSpec[] columnSpecs, TableSpec[] rowSpecs )
	{
		if( !(constraints instanceof TableConstraints) )
			return constraints;

		TableConstraints cons = (TableConstraints) constraints;
		int hAlign = cons.getHAlign();
		int vAlign = cons.getVAlign();

		if( hAlign != TableConstraints.DEFAULT &&
			vAlign != TableConstraints.DEFAULT )
		  return cons;

		// clone constraints
		cons = new TableConstraints( cons );

		if( hAlign == TableConstraints.DEFAULT ) {
			int align = TableLayoutConstraints.FULL;
			if( cons.col1 == cons.col2 && cons.col1 >= 0 && cons.col1 < columnSpecs.length )
				align = updateHAlign( hAlign, cons.col1, columnSpecs );
			cons.setHAlign( align );
		}
		if( vAlign == TableConstraints.DEFAULT ) {
			int align = TableLayoutConstraints.FULL;
			if( cons.row1 == cons.row2 && cons.row1 >= 0 && cons.row1 < rowSpecs.length )
				align = updateVAlign( vAlign, cons.row1, rowSpecs );
			cons.setVAlign( align );
		}

		return cons;
	}

	public static int updateHAlign( int hAlign, int columnIndex, TableSpec[] columnSpecs ) {
		int align = TableLayoutConstraints.FULL;
		switch( columnSpecs[columnIndex].getDefaultAlignment() ) {
			case TableSpec.LEFT:	align = TableLayoutConstraints.LEFT;   break;
			case TableSpec.CENTER:	align = TableLayoutConstraints.CENTER; break;
			case TableSpec.RIGHT:	align = TableLayoutConstraints.RIGHT;  break;
			case TableSpec.FILL:	align = TableLayoutConstraints.FULL;   break;
		}
		return align;
	}

	public static int updateVAlign( int vAlign, int rowIndex, TableSpec[] rowSpecs ) {
		int align = TableLayoutConstraints.FULL;
		switch( rowSpecs[rowIndex].getDefaultAlignment() ) {
			case TableSpec.TOP:		align = TableLayoutConstraints.TOP;    break;
			case TableSpec.CENTER:	align = TableLayoutConstraints.CENTER; break;
			case TableSpec.BOTTOM:	align = TableLayoutConstraints.BOTTOM; break;
			case TableSpec.FILL:	align = TableLayoutConstraints.FULL;   break;
		}
		return align;
	}
}
