/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.*;
import java.util.*;

/**
 * @author Karl Tauber
 */
public class NullLayout
	implements LayoutManager2
{
	private boolean autoSize = true;
	private final HashMap<Component, NullConstraints> consMap = new HashMap<Component, NullConstraints>();

	public NullLayout() {
	}

	public NullLayout( boolean autoSize ) {
		this.autoSize = autoSize;
	}

	public boolean isAutoSize() {
		return autoSize;
	}

	public void setAutoSize( boolean autoSize ) {
		this.autoSize = autoSize;
	}

	public NullConstraints getConstraints( Component comp ) {
		return consMap.get( comp );
	}

	public void addLayoutComponent( String name, Component comp ) {
		throw new IllegalArgumentException();
	}

	public void addLayoutComponent( Component comp, Object constraints ) {
		if( !(constraints instanceof NullConstraints) )
			throw new IllegalArgumentException();
		consMap.put( comp, (NullConstraints) constraints );
	}

	public void removeLayoutComponent( Component comp ) {
		consMap.remove( comp );
	}

	public Dimension preferredLayoutSize( Container parent ) {
		if( !autoSize )
			return new Dimension( 0, 0 );

		Dimension preferredSize = new Dimension();
		for( Map.Entry<Component, NullConstraints> e : consMap.entrySet() ) {
			Component c = e.getKey();
			NullConstraints cons = e.getValue();
			Rectangle bounds = componentBounds( c, cons );
			preferredSize.width = Math.max( bounds.x + bounds.width, preferredSize.width );
			preferredSize.height = Math.max( bounds.y + bounds.height, preferredSize.height );
		}

		Insets insets = parent.getInsets();
		preferredSize.width += insets.right;
		preferredSize.height += insets.bottom;

		return preferredSize;
	}

	public Dimension minimumLayoutSize( Container parent ) {
		return preferredLayoutSize( parent );
	}

	public Dimension maximumLayoutSize( Container target ) {
		return new Dimension( Short.MAX_VALUE, Short.MAX_VALUE );
	}

	public void layoutContainer( Container parent ) {
		for( Map.Entry<Component, NullConstraints> e : consMap.entrySet() ) {
			Component c = e.getKey();
			NullConstraints cons = e.getValue();
			c.setBounds( componentBounds( c, cons ) );
		}
	}

	public void invalidateLayout( Container target ) {
	}

	public float getLayoutAlignmentX( Container target ) {
		return 0.5f;
	}

	public float getLayoutAlignmentY( Container target ) {
		return 0.5f;
	}

	private Rectangle componentBounds( Component c, NullConstraints cons ) {
		Rectangle bounds = new Rectangle( cons.x, cons.y, cons.width, cons.height );
		if( bounds.width < 0 || bounds.height < 0 ) {
			Dimension size = c.getPreferredSize();
			if( bounds.width < 0 )
				bounds.width = size.width;
			if( bounds.height < 0 )
				bounds.height = size.height;
		}
		return bounds;
	}
}
