/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.*;
import com.jformdesigner.model.FormLayoutConstraints;
import com.jformdesigner.model.FormLayoutManager;

/**
 * @author Karl Tauber
 */
public class GridBagLayoutCreator
	extends AbstractLayoutCreator
{
	private static final String PROP_COLUMN_SPECS = "$columnSpecs";
	private static final String PROP_ROW_SPECS = "$rowSpecs";
	private static final String PROP_HGAP = "$hGap";
	private static final String PROP_VGAP = "$vGap";
	private static final String PROP_ALIGN_LEFT = "$alignLeft";
	private static final String PROP_ALIGN_TOP = "$alignTop";

	private static final boolean DEF_ALIGN_LEFT = false;
	private static final boolean DEF_ALIGN_TOP = false;

	private GridBagSpec[] columnSpecs;
	private GridBagSpec[] rowSpecs;
	private int hGap;
	private int vGap;

	public LayoutManager createLayoutManager( Container container, FormLayoutManager formLayout )
		throws InstantiationException, IllegalAccessException
	{
		String encodedColumnSpecs = formLayout.getPropertyString( PROP_COLUMN_SPECS, "" );
		String encodedRowSpecs = formLayout.getPropertyString( PROP_ROW_SPECS, "" );
		hGap = formLayout.getPropertyInt( PROP_HGAP, 0 );
		vGap = formLayout.getPropertyInt( PROP_VGAP, 0 );
		boolean alignLeft = formLayout.getPropertyBoolean( PROP_ALIGN_LEFT, DEF_ALIGN_LEFT );
		boolean alignTop = formLayout.getPropertyBoolean( PROP_ALIGN_TOP, DEF_ALIGN_TOP );

		columnSpecs = GridBagSpec.decodeSpecs( encodedColumnSpecs );
		rowSpecs = GridBagSpec.decodeSpecs( encodedRowSpecs );

		GridBagLayout layout = new GridBagLayout();
		layout.columnWidths = computeMinSize( columnSpecs, hGap, alignLeft );
		layout.rowHeights = computeMinSize( rowSpecs, vGap, alignTop );
		layout.columnWeights = computeWeights( columnSpecs, alignLeft );
		layout.rowWeights = computeWeights( rowSpecs, alignTop );
		return layout;
	}

	@Override
	public void addComponentToContainer( Container container, Component component,
		Object constraints, int index, FormLayoutConstraints formConstraints )
	{
		constraints = updateConstraints( constraints, columnSpecs, rowSpecs, hGap, vGap );
		container.add( component, constraints, index );
	}

	public static int[] computeMinSize( GridBagSpec[] specs, int gap, boolean align ) {
		if( !align ) {
			boolean empty = true;
			for( int i = 0; i < specs.length; i++ ) {
				if( specs[i].getMinimumSize() != 0 ) {
					empty = false;
					break;
				}
			}
			if( empty )
				return null;
		}

		int[] minSizes = new int[specs.length + (align ? 1 : 0)];
		for( int i = 0; i < specs.length; i++ ) {
			minSizes[i] = specs[i].getMinimumSize();
			if( gap > 0 && minSizes[i] > 0 && i < (specs.length - 1) )
				minSizes[i] += gap;
		}
		if( align )
			minSizes[specs.length] = 0;
		return minSizes;
	}

	public static double[] computeWeights( GridBagSpec[] specs, boolean align ) {
		if( !align ) {
			boolean empty = true;
			for( int i = 0; i < specs.length; i++ ) {
				if( specs[i].getResizeWeight() != 0.0 ) {
					empty = false;
					break;
				}
			}
			if( empty )
				return null;
		}

		double[] weights = new double[specs.length + (align ? 1 : 0)];
		for( int i = 0; i < specs.length; i++ )
			weights[i] = specs[i].getResizeWeight();
		if( align )
			weights[specs.length] = 0.0001;
		return weights;
	}

	public static Object updateConstraints( Object constraints,
		GridBagSpec[] columnSpecs, GridBagSpec[] rowSpecs, int hGap, int vGap )
	{
		if( !(constraints instanceof GridBagConstraintsEx) )
			return constraints;

		GridBagConstraintsEx gbc = (GridBagConstraintsEx) constraints;
		int hAlign = gbc.getHAlign();
		int vAlign = gbc.getVAlign();

		if( hGap == 0 && vGap == 0 &&
			hAlign != GridBagConstraintsEx.DEFAULT &&
			vAlign != GridBagConstraintsEx.DEFAULT )
		  return gbc;

		// clone constraints
		gbc = (GridBagConstraintsEx) gbc.clone();

		if( hAlign == GridBagConstraintsEx.DEFAULT && gbc.gridwidth == 1 &&
				gbc.gridx >= 0 && gbc.gridx < columnSpecs.length )
			gbc.setHAlign( columnSpecs[gbc.gridx].getDefaultAlignment() );
		if( vAlign == GridBagConstraintsEx.DEFAULT && gbc.gridheight == 1 &&
				gbc.gridy >= 0 && gbc.gridy < rowSpecs.length )
			gbc.setVAlign( rowSpecs[gbc.gridy].getDefaultAlignment() );

		if( hGap != 0 && gbc.gridx + gbc.gridwidth < columnSpecs.length )
			gbc.insets.right += hGap;
		if( vGap != 0 && gbc.gridy + gbc.gridheight < rowSpecs.length )
			gbc.insets.bottom += vGap;

		return gbc;
	}
}
